/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// ECOTRX.C
//
// (C) Mark Usher
// marku@magnet.at
// release 1.0  30.06.1999                                                       
*/

#include <go32.h>
#include <dpmi.h>
#include <sys/farptr.h>
#include <sys/types.h>
#include <sys/movedata.h>

#include <dos.h>
#include <stdio.h>
#include <string.h>

#include "ecotrx.h"
#include "econet.h"
#include "ecolink.h"

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Main program globals
*/

#define debug	0x00

#include "initcard.c"

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Function prototypes
*/
void Transmit(unsigned short Station);
byte Transmit_buffer(unsigned short Station,
                     byte Port,
                     byte length,
                     char *txbuffer);
void Receive(unsigned short Station);
byte ReceiveBuffer(unsigned short Station, byte Port);

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Main program   ECOTRX.C
*/

char ECONET_ERRORS[5][27] =
	{ "Line Jammed.",
      "Net Error.",
      "Not Listening.",
      "No Clock.",
      "Bad TRANSMIT control block." };

char *help_string=
	"\nEcolink Transmit and Receive. v1.00 by Mark Usher 1999\n\n" \
	"Syntax: ECOTRX [/t /r] \n\n" \
	"/t : Transmit to a station\n" \
	"/r : Receive from a station\n";

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
*/
int main(int argc,char *argv[])
{
    byte handle;
    unsigned short Station;
    _go32_dpmi_seginfo old_timer_handler, new_timer_handler;
    _go32_dpmi_seginfo old_card_handler, new_card_handler;


	if (argc == 1 )													/* no arguemnts have been supplied */
	{
		printf(help_string);										/* display the default help screen */
		exit(0);
	}

		
	if ( strcmp (argv[1], "/?" ) == 0)								/* display syntax only */
	{
		printf(help_string);										/* display the default help screen */
		exit(0);
	}

        if (( strcmp (argv[1], "/t" ) != 0) &&  ( strcmp (argv[1], "/r" ) != 0))
	{
		printf(help_string);										/* display the default help screen */
		exit(0);
	}


    /* get the current handler saving it in old_handler for later */
    _go32_dpmi_get_protected_mode_interrupt_vector( TIMER_INT, &old_timer_handler);
    _go32_dpmi_get_protected_mode_interrupt_vector( MASTER_FIRST_VECTOR + CARD_INTERRUPT, &old_card_handler);

    /* lock all functions and variables used so they are not paged out */
    _go32_dpmi_lock_data(&counter_150, (long)sizeof(counter_150));
    _go32_dpmi_lock_data(&Allow_reentry, (long)sizeof(Allow_reentry));
    _go32_dpmi_lock_code(my_timer_handler, (long)end_my_timer_handler - (long)my_timer_handler);
    _go32_dpmi_lock_code(card_handler, (long)end_card_handler - (long)card_handler);
    _go32_dpmi_lock_code(AddHandleToList, (long)end_AddHandleToList - (long)AddHandleToList);
    _go32_dpmi_lock_code(RemoveHandleFromList, (long)end_RemoveHandleFromList - (long)RemoveHandleFromList);
    _go32_dpmi_lock_code(default_event_handler, (long)end_default_event_handler - (long)default_event_handler);
    _go32_dpmi_lock_code(SetLatch, (long)end_SetLatch - (long)SetLatch);
    _go32_dpmi_lock_code(EventChain, (long)end_EventChain - (long)EventChain);

    /* get the address of the function to execute when the timer INT occurs */
    new_timer_handler.pm_offset = (int)my_timer_handler;
    new_timer_handler.pm_selector = _go32_my_cs();

    /* get the address of the function to execute when the card INT occurs */
    new_card_handler.pm_offset = (int)card_handler;
    new_card_handler.pm_selector = _go32_my_cs();

    /* add handlers to the chain */
    _go32_dpmi_chain_protected_mode_interrupt_vector( TIMER_INT, &new_timer_handler);
    _go32_dpmi_chain_protected_mode_interrupt_vector( MASTER_FIRST_VECTOR + CARD_INTERRUPT, &new_card_handler);

	/* display the labels for a grid showing the contents of the lists */
	#if debug
                ScreenClear();
		ScreenPutString("Unused",        (BLACK << 4) | WHITE, 0, 1);
		ScreenPutString("Open Tx",       (BLACK << 4) | WHITE, 0, 2);
		ScreenPutString("Open Tx Chain", (BLACK << 4) | WHITE, 0, 3);
		ScreenPutString("Open Rx",       (BLACK << 4) | WHITE, 0, 4);
		ScreenPutString("Completed",     (BLACK << 4) | WHITE, 0, 5);
		ScreenPutString("Enabled",       (BLACK << 4) | WHITE, 0, 6);
		ScreenPutString("Stored",        (BLACK << 4) | WHITE, 0, 7);

		ScreenSetCursor(9,0);
	#endif

    counter_150 = 0;
    if (InitialiseCard() == ERROR)
       exit(0);

    do																/* get the station to communicate with */
    {
        printf("Station [1-254]:");
        scanf("%d", &Station);
    }
    while ((Station <= 0) && (Station >= 0xFF));

    if ( strcmp (argv[1], "/t" ) == 0)								/* transmit */
        Transmit(Station);

    if ( strcmp (argv[1], "/r" ) == 0)								/* receive */
        Receive(Station);

     /* Remove the IRQ event from the chain */
    _go32_dpmi_set_protected_mode_interrupt_vector(MASTER_FIRST_VECTOR + CARD_INTERRUPT, &old_card_handler);
    _go32_dpmi_set_protected_mode_interrupt_vector(TIMER_INT, &old_timer_handler);

    /* Flag the interrupt in the PIC as no longer being used */
    handle =inportb(0x21);
    handle = handle & (0x1 << CARD_INTERRUPT);  /* Set the mask for the PIC command */
    outportb(0x21, handle);

    return OK;

}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
*/
void Transmit(unsigned short Station)
{

    byte rc;
    char txbuffer[MAX_TX_LENGTH];
    char temp;
    char key[2] = {0,0};

    rc=0;
    printf("ESC to quit.\n");

    do
    {
        /* Get a key into the transmit buffer */
        strcpy(txbuffer, "\0");
        *key = getch();
        strcat(txbuffer,strupr(key));
        printf("%s",txbuffer);
        rc = Transmit_buffer(Station, TX_PORT, strlen(txbuffer), &txbuffer);
    }
    /* until ESC is pressed or an error occurs */
    while ((*key != 0x1B) && (rc == 0));
    printf("\n");

    /* an error has occured */
    if (rc != 0)
    {
        rc = rc - ERR_LINE_JAMMED;      /* offset */
        printf("%s\n",ECONET_ERRORS[rc]);
    }

    return;
}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
*/
byte Transmit_buffer(unsigned short Station,
                     byte Port,
                     byte length,
                     char *txbuffer)
{
    byte handle;
    struct ReplyBlock_PollCB PollReplyBlock;
    int tries;
    bool NonFatal;

    tries = 10;

    handle = OpenTxCBH( 0x80,  // Control byte
                        Port,  // Port
                     Station,  // Station
                          10,  // Retry Count
                      length,  // Data Length
                           0,  // Remote Address
                    txbuffer); // pointer to data

    /* if a handle has been assigned then poll the control block
       until completion
    */
    if (validHandle(handle))
    {
       do
       {
           do
           {
               PollCBH(handle, &PollReplyBlock);
           }
           /* until the top bit is no longer set */
           while ((PollReplyBlock.ControlByte & 0x80) != 0);

           /* check the returned code for a non fatal error */
           if ((PollReplyBlock.ControlByte == ERR_NET_ERROR) ||
               (PollReplyBlock.ControlByte == ERR_NOT_LISTENING))
                NonFatal = TRUE;
           else
                NonFatal = FALSE;

         if (NonFatal)
         {
             tries --;
             sleep(1);
         }

       }

       /* continue until either a fatal error occurs,
          a successful transmission
          or all retries have failed
       */
       while ((tries > 0) && NonFatal);

    KillCBH(handle,KILL_ALL);
    }

    return PollReplyBlock.ControlByte;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
*/
void Receive(unsigned short Station)
{
    byte rc;

    rc=0;

	printf("Send Z to exit.\n");				
    do
    {
        rc = ReceiveBuffer(Station, RX_PORT);
    }
    while (rc == 0);

    return;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Set up the receive block and wait for some data to arrive from
// <station> on <port>
//
*/
byte ReceiveBuffer(unsigned short Station,
                   byte Port)
{
    byte handle;
    int ints;
    struct ReplyBlock_PollCB PollReplyBlock;
    char readbuffer[0x500];

    handle = OpenRxCBH(   Port,  // Port
                       Station,  // Station
                  RPC_DISABLED,  // RPC Flag
                       RPC_ALL,  // RPC Number
                             0); // Timeout

    if (validHandle(handle))
    {
		/* keep checking the state of the handle until the top bit of the control byte is set */
	    do
        {
            PollCBH(handle, &PollReplyBlock);
        }
        while ((PollReplyBlock.ControlByte & 0x80) == 0);

		/* read the data recieved by the handle */
        ReadCBH(handle, 0, PollReplyBlock.Length, &readbuffer);
        KillCBH(handle,KILL_ALL);
    }

    readbuffer[PollReplyBlock.Length] = 0;
    printf("%s", readbuffer);

	/* generate a sound on the PC speaker according to the character received */
	sound(*readbuffer);
	sleep(1);
	nosound();

        /* check if Z has been sent */
	if (*readbuffer == 90)
   	    return 1;
	else
	    return 0;

}
