/*
	io_str.c, input/output and string handling for `read'.
	Copyright (c) 2004 Kalle Risnen.


	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/time.h>
#include "read.h"

#define is_delim(d,c) strchr((d), (c))
#define is_space(c)   is_delim(" \t", (c))


void parse_escapes(char *s);


#ifdef AROS
#	define set_raw(r) SetMode(Input(), r)
#else /* !AROS */
#	define get_env getenv

#	ifdef HAVE_TERMIOS_H
#		include <unistd.h>
#		include <termios.h>

		static struct termios stored_settings;

		void set_raw(int r)
		{
			struct termios new_settings;

			if(r)
				{
				tcgetattr(0, &stored_settings);
				new_settings = stored_settings;
				new_settings.c_lflag &= ~(ICANON | ECHO);
				new_settings.c_cc[VTIME] = 0;
				new_settings.c_cc[VMIN]  = 1;

				tcsetattr(0, TCSANOW, &new_settings);
				}
			else
				tcsetattr(0, TCSANOW, &stored_settings);
		}
#	else  /* !HAVE_TERMIOS_H */
#		define set_raw(r)
#	endif /* HAVE_TERMIOS_H  */
#endif /* AROS */


long get_secs(void)
{
	struct timeval tv;

	gettimeofday(&tv, NULL);

	return tv.tv_sec;
}


void parse_escapes(char *s)
{
	int i, j;
	char *out;

	out = xmalloc(strlen(s));

	for(i = j = 0; i < strlen(s); i++)
		{
		if(s[i] == '\\' && s[i+1])
			{
			int tmp = 0;
			switch(s[++i])
				{
				case 'a':  out[j++] = '\a';   break;
				case 'b':  out[j++] = '\b';   break;
				case 'e':  out[j++] = '\033'; break;
				case 'f':  out[j++] = '\f';   break;
				case 'n':  out[j++] = '\n';   break;
				case 'r':  out[j++] = '\r';   break;
				case 't':  out[j++] = '\t';   break;
				case 'v':  out[j++] = '\v';   break;
				case '\\': out[j++] = '\\';   break;
				case 'x': case 'X':
					while(s[++i] && isxdigit(s[i]))
						tmp = (tmp * 0x10) + (isdigit(s[i]) ? s[i] - '0' 
						                                    : tolower(s[i]) - 'a' + 0xa);
					out[j++] = tmp;
					i--;
					break;
				case '0':
					while(s[++i] && '0' <= s[i] && s[i] <= '7')
						tmp = (tmp * 010) + s[i] - '0';
					out[j++] = tmp;
					i--; 
					break;
				default:
					out[j++] = '\\';
					out[j++] = s[i];
					break;
				}
			}
		else
			out[j++] = s[i];
		}
	out[j] = 0;
	strcpy(s, out);

	free(out);
}


int get_str(FILE *in, char *buff, int max_len, int escape, int timeout, int s, char *delim)
{
	long start_time = 0, new_time = 0;
	int i = 0, c = 0, prev = 0;

	set_raw(1);
	if(timeout) 
		start_time = get_secs();

	while(
	      (i < max_len) && 
	      ( ( !is_delim( delim, (c = fgetc(in)) ) ) ||
	      ( prev == '\\' && escape ) )
	     ) 
		{
		if(timeout)
			{
			new_time = get_secs();
			if(new_time - start_time > timeout)
				{
				set_raw(0);
				return -1;
				}
			}

		if(c == '\r') c = '\n';
		if(!s) 
			{
			putchar(c);
			if(c == '\b') 
				{
				printf(" \b");
				if(i) i--;
				}
			fflush(stdout);
			}
		if(c != '\b')
			buff[i++] = c;
		prev = c;
	}
	
	buff[i] = 0;

	if(c != '\n' && in == stdin) printf("\n");

	set_raw(0);

	if(escape)
		parse_escapes(buff);

	return i;
}


void put_str(FILE *op, char *s, int escape)
{
	if(escape)
		parse_escapes(s);

	fprintf(op, s);

	fflush(op);
}


int str_to_vars(char *str, char **vars, int num_vars, char *delim, int escape)
{
	char out[MAXLEN];
	int i, pos;

	if(!strlen(str)) return 0;

	if(escape)
		parse_escapes(delim);

	for(i = pos = 0; i < num_vars; i++)
		{
		if(i == num_vars - 1 && pos < strlen(str))
			strncpy(out, str + pos, MAXLEN);
		else if((pos = get_word(str, out, pos, delim)) == -1)
			return i;
		if(strlen(out))
			set_env(vars[i], out);

		}
	return i;
}


int get_word(char *str, char *out, int start, char *delim)
{
	int i, j, in_quote = 0;

	for(i = start; str[i] && (is_space(str[i]) || is_delim(delim, str[i])); i++)
		;

	if(i >= strlen(str) - 1)
		return -1;

	for(j = 0; i < strlen(str) && (!is_delim(delim, str[i]) || in_quote); i++)
		{
		if(str[i] == '"')
			in_quote ^= 1;
		else
			out[j++] = str[i];
		}

	out[j] = 0;

	return i + 1;
}
