/*         
 * asciitable - asciitable function
 * Copyright 2007 Rupert Hausberger <rupert_hasuberger@gmx.net>
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//-----------------------------------------------------------------------------

#include <exec/errors.h>
#include <exec/execbase.h>
#ifndef __AROS__
#include <exec/rawfmt.h>
#endif

#include <proto/dos.h>
#include <proto/exec.h>

#include "asciitable.h"

//-----------------------------------------------------------------------------

#ifdef __MORPHOS__
const int __initlibraries	= 0; /* no auto-libinit */
const int __nocommandline	= 1; /* no argc, argv   */
const int __abox__ 			= 1; /* */
#elif defined(__AROS__)
#include <aros/symbolsets.h>
#include <stdio.h>
#include <string.h>
const int __nocommandline = 1;
#endif

#define NAME "asciitable"

#define ARG_TEMPLATE "FORM/N,TO/N,DECIMAL=DEC/S,HEXADECIMAL=HEX/S,OCTAL=OCT/S,BINARY=BIN/S,CHARACTER=CHR/S,ALL/S"

static const char version[] = "\0$VER: "NAME" 1.1 ("__AMIGADATE__") "__YEAR__" Rupert Hausberger\0";
static const char exthelp[] =
	NAME" : Print out the ascii-table\n"
	"\tFORM/N            - Start from (0-255, default 0)\n"
	"\tTO/N              - Stop at (0-255, default 127)\n"
	"\tDECIMAL=DEC/S     - Print decimals\n"
	"\tHEXADECIMAL=HEX/S - Print hexadecimals\n"
	"\tOCTAL=OCT/S       - Print octals\n"
	"\tBINARY=BIN/S      - Print binaries\n"
	"\tCHARACTER=CHR/S   - Print characters\n"
	"\tALL/S             - Print all\n";

enum { ARG_FROM, ARG_TO, ARG_DEC, ARG_HEX, ARG_OCT, ARG_BIN, ARG_CHR, ARG_ALL, ARG_END };

struct ExecBase *SysBase;
struct DosLibrary *DOSBase;

//-----------------------------------------------------------------------------

#ifndef __AROS__
static ULONG strlen(const UBYTE *str);
static void vsprintf(UBYTE *to, UBYTE *fmt, va_list args);
static void sprintf(UBYTE *to, UBYTE *fmt, ...);
#endif

static LONG acsiitable(ULONG *args);

//-----------------------------------------------------------------------------

#if defined(__AROS__)

	int realmain(struct ExecBase *SysBase);

	AROS_UFH3(__startup static int, Start,
	  AROS_UFHA(char *, argstr, A0),
	  AROS_UFHA(ULONG, argsize, D0),
	  AROS_UFHA(struct ExecBase *, sBase, A6))
	{
		AROS_USERFUNC_INIT
		return realmain(sBase);
		AROS_USERFUNC_EXIT
	}
#else
	// mazze: I haven't tested if it still compiles for other systems.
	typedef ULONG IPTR;
#endif

//-----------------------------------------------------------------------------

#ifdef __AROS__
int realmain(struct ExecBase *SysBase)
{
#else
int realmain(void)
{
	SysBase = *((struct ExecBase **)4l);
#endif

	int result = RETURN_FAIL;

	if ((DOSBase = (struct DosLibrary *)OpenLibrary("dos.library", 37)))
	{
		struct RDArgs *rda;
		LONG err = 0l;

		if ((rda = AllocDosObject(DOS_RDARGS, NULL)))
		{
			static IPTR args[ARG_END] = { 0ul, 0ul, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE };
			struct RDArgs *rd;

			rda->RDA_ExtHelp = (UBYTE *)exthelp;
			if ((rd = ReadArgs(ARG_TEMPLATE, args, rda)))
			{
				if (!(err = acsiitable(&args[0])))
					result = RETURN_OK;

				FreeArgs(rd);
			} else
				err = IoErr();

			FreeDosObject(DOS_RDARGS, rda);
		} else
			err = ERROR_NO_FREE_STORE;

		if (err) {
			PrintFault(err, NAME);
			SetIoErr(err);
		}
		CloseLibrary((struct Library *)DOSBase);
	}
	return result;
}

//-----------------------------------------------------------------------------

#ifndef __AROS__
static ULONG strlen(const UBYTE *str)
{
	const UBYTE *s;

	for (s = str; *s; ++s);
	return ((ULONG)(s - str));
}

static void vsprintf(UBYTE *to, UBYTE *fmt, va_list args)
{
	VNewRawDoFmt(fmt, (APTR)RAWFMTFUNC_STRING, to, args);
}

static void sprintf(UBYTE *to, UBYTE *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsprintf(to, fmt, args);
	va_end(args);
}
#endif

//-----------------------------------------------------------------------------

#define isset(var, bit) (((var) & (1l << (bit))) != 0)

static UBYTE *npc_lower[33] = { //non-printable characters
	"NUL",
	"SOH",
	"STX",
	"ETX",
	"EOT",
	"ENQ",
	"ACK",
	"BEL",
	"BS ",
	"HT ",
	"LF ",
	"VT ",
	"FF ",
	"CR ",
	"SO ",
	"SI ",
	"DLE",
	"DC1",
	"DC2",
	"DC3",
	"DC4",
	"NAK",
	"SYN",
	"ETB",
	"CAN",
	"EM ",
	"SUB",
	"ESC",
	"FS ",
	"GS ",
	"RS ",
	"US ",
	"SP "
};

static UBYTE *npc_upper = "DEL";

static LONG acsiitable(ULONG *args)
{
	IPTR from_addr = args[ARG_FROM];
	IPTR to_addr = args[ARG_TO];
	BOOL dec = (BOOL)args[ARG_DEC];
	BOOL hex = (BOOL)args[ARG_HEX];
	BOOL oct = (BOOL)args[ARG_OCT];
	BOOL bin = (BOOL)args[ARG_BIN];
	BOOL chr = (BOOL)args[ARG_CHR];
	BOOL all = (BOOL)args[ARG_ALL];
	LONG i, from, to;

	from = (from_addr) ? *((LONG *)args[ARG_FROM]) : 0l;
	to = (to_addr) ? *((LONG *)args[ARG_TO]) : 127l;

   if (from < 0 || from > 255 || to < 0 || to > 255 || from > to)
		return ERROR_BAD_NUMBER;
	else if (!(dec || hex || oct || bin || chr || all))
		return ERROR_BAD_TEMPLATE;

	if (all)
		dec = hex = oct = bin = chr = TRUE;

   for (i = from; i <= to; i++)
	{
		UBYTE buf[32] = { '\0' };

		if (dec) {
			sprintf(&buf[strlen(buf)], "%03ld ", i);
		}
		if (hex) {
			sprintf(&buf[strlen(buf)], "%02lX ", i);
		}
		if (oct) {
			LONG quo, rem, a = i, b = 8l, j = 1l, oct = 0l;

			do {
				rem = a % b;
				quo = a / b;
				oct = oct + rem * j;
				j = j * 10;
				a = quo;
			} while (quo != 0);

			sprintf(&buf[strlen(buf)], "%03ld ", oct);
		}
		if (bin) {
			LONG j;

			for (j = 7; j >= 0; j--)
				sprintf(&buf[strlen(buf)], "%c", (isset(i, j)) ? '1' : '0');
			sprintf(&buf[strlen(buf)], " ");
		}
		if (chr) {
			if (i <= 32)
				sprintf(&buf[strlen(buf)], "%s ", npc_lower[i]);
			else if (i == 127)
				sprintf(&buf[strlen(buf)], "%s ", npc_upper);
			else
				sprintf(&buf[strlen(buf)], "%c ", (BYTE)i);
		}
		sprintf(&buf[strlen(buf)], "\n");

		PutStr(buf);
	}

	return 0l;
}

//-----------------------------------------------------------------------------
