/*
    Copyright 2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
    #include <direct.h>
#else
    #include <unistd.h>
    #define stricmp strcasecmp
#endif

typedef unsigned char   u_char;
typedef unsigned int   u_int;

#include "endian.h"

#define VER         "0.1"
#define AFSSIGN     *(u_int *)"AFS\0"
#define MYFSEEK(x)  if(fseek(fd, x, SEEK_SET) < 0) std_err();



void read_file(FILE *fd, void *buff, int len);
void getfile(FILE *fdin, char *fname, u_int size);
void read_err(void);
void write_err(void);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd;
    u_int  i,
            num,
            sign,
            idxoff,
            nameoff,
            list        = 0;
    char    filename[33],
            *outdir     = NULL;

    struct {
        u_int  off;
        u_int  len;
    } afs;


    setbuf(stdout, NULL);

    fputs("\n"
        "AFS files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file.AFS>\n"
            "\n"
            "Options:\n"
            "-l      lists the offsets and the length of the files without extracting them\n"
            "-d DIR  output directory where extracting the files\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'l': list = 1; break;
            case 'd': outdir = argv[++i]; break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                }
        }
    }

    printf("- open file         %s\n", argv[argc]);
    fd = fopen(argv[argc], "rb");
    if(!fd) std_err();

    if(outdir) {
        printf("- change directory  %s\n", outdir);
        if(chdir(outdir) < 0) std_err();
    }

    read_file(fd, &sign, 4);
    if(sign != AFSSIGN) {
        printf("\n"
            "Alert: the sign is invalid, is \"%.4s\" while should be \"%.4s\"\n\n",
            (char *)&sign, (char *)&AFSSIGN);
        exit(1);
    }

    read_file(fd, &num, 4);
    num = h2le32(num);
    printf("- %u files\n", num);

    nameoff = 0xffffffff;
    for(i = 0; i < num; i++) {
        read_file(fd, &afs, sizeof(afs));
        afs.off = h2le32(afs.off);
        if(afs.off < nameoff) nameoff = afs.off;
    }   // useless? probably, but this should work with any possible file structure
    MYFSEEK(nameoff - 8);
    read_file(fd, &nameoff, 4);
    nameoff = h2le32(nameoff);

    idxoff = 8;
    filename[sizeof(filename) - 1] = 0;

    fputs("\n"
        "  Offset      Size        Filename\n"
        "=======================================\n", stdout);
    for(i = 0; i < num; i++) {
        MYFSEEK(nameoff);
        read_file(fd, filename, 32);
        nameoff += 48;

        MYFSEEK(idxoff);
        read_file(fd, &afs, sizeof(afs));
        afs.off = h2le32(afs.off);
        afs.len = h2le32(afs.len);
        idxoff += sizeof(afs);

        printf("  0x%08x  %-10u  %s\n", afs.off, afs.len, filename);

        if(list) continue;

        if(!*filename) {
            sprintf(filename, "%08x.dat", afs.off);
            printf("                 renamed: %s\n", filename);
        }

        MYFSEEK(afs.off);
        getfile(fd, filename, afs.len);
    }

    fclose(fd);
    fputs("\nFinished\n\n", stdout);
    return(0);
}



void read_file(FILE *fd, void *buff, int len) {
    if(fread(buff, len, 1, fd) != 1) read_err();
}



void getfile(FILE *fdin, char *fname, u_int size) {
    FILE    *fdout;
    int     len;
    u_char  buff[16384];

    fdout = fopen(fname, "wb");
    if(!fdout) std_err();

    for(len = sizeof(buff) - 1; size; size -= len) {
        if(len > size) len = size;
        if(fread(buff, len, 1, fdin) != 1) read_err();
        if(fwrite(buff, len, 1, fdout) != 1) write_err();
    }

    fclose(fdout);
}



void read_err(void) {
    fputs("\nError: the file contains unexpected data\n\n", stdout);
    exit(1);
}



void write_err(void) {
    fputs("\nError: impossible to write the output file, probably your disk space is finished\n\n", stdout);
    exit(1);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


