;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; led 20210318 - a simple line-oriented text editor ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Copyright (C) 2021 Pasquale Frega
; All rights reserved
;
; Released under the simplified BSD license
;
; last edit: 21-Mar-2021
;
; begin
;
; Apply d, j, o, r, s, t commands (see below) from SCRIPT_FILE to files listed in LIST_FILE
;
(import io (read-lines! write-lines!))
(import util (file-exists?))
(when (and (eq? (car *arguments*) "-a") (cadr *arguments*) (caddr *arguments*))
 (if-let [(_script (read-lines! (cadr *arguments*))) (_files (read-lines! (caddr *arguments*)))]
  (for-each _file _files
   (when _file
    (print! (.. "Processing \"" _file "\"..."))
    (if-with (_buffer (read-lines! _file))
     (progn
      (for-each _got_string _script
       (when _got_string
        (with (_got_list (string/split _got_string ","))
         (for _index 1 (n _got_list) 1
          (when (string/match (nth _got_list _index) "|cM")
           (setq! (nth _got_list _index) (string/gsub (nth _got_list _index) "|cM" ","))
          )
         )
         (cond
          ((eq? (car _got_list) "d")
           (cond
            ((= (n _got_list) 1)
             (set! _buffer '())
            )
            ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
             (remove-nth! _buffer (string->number (cadr _got_list)))
            )
            ((and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
             (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
              (for _line (string->number (cadr _got_list)) (string->number (caddr _got_list)) 1
               (remove-nth! _buffer (string->number (cadr _got_list)))
              )
             )
            )
            (true)
           )
          )
          ((eq? (car _got_list) "j")
           (when (and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
            (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
             (setq! (nth _buffer (string->number (cadr _got_list))) (sprintf "%s%s" (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list)))))
            )
           )
          )
          ((eq? (car _got_list) "o")
           (with (_old_buffer _buffer)
            (set! _buffer '())
            (for-each _string _old_buffer
             (if (string/ends-with? _string "\r")
              (push! _buffer _string)
              (push! _buffer (sprintf "%s\r" _string))
             )
            )
           )
          )
          ((eq? (car _got_list) "r")
           (when (and (> (n _got_list) 1) (not (empty? (string/trim (cadr _got_list)))))
            (with (_read_file (string/trim (cadr _got_list)))
             (if (file-exists? _read_file)
              (with (_content (read-lines! _read_file))
               (if (and (string->number (caddr _got_list)) (nth _buffer (string->number (caddr _got_list))))
                (for-each _string (reverse _content)
                 (insert-nth! _buffer (string->number (caddr _got_list)) _string)
                )
                (set! _buffer (append _buffer _content))
               )
               (print! (.. "File \"" _read_file "\" read successful."))
              )
              (print! (.. "File \"" _read_file "\" read failed."))
             )
            )
           )
          )
          ((eq? (car _got_list) "s")
           (cond
            ((= (n _got_list) 3)
             (for _line 1 (n _buffer) 1
              (when (and (pcall string/match (nth _buffer _line) (cadr _got_list)) (string/match (nth _buffer _line) (cadr _got_list)))
               (setq! (nth _buffer _line) (string/gsub (nth _buffer _line) (cadr _got_list) (caddr _got_list)))
              )
             )
            )
            ((= (n _got_list) 4)
             (when (and (string->number (cadddr _got_list)) (nth _buffer (string->number (cadddr _got_list))))
              (when (and (pcall string/match (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list)) (string/match (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list)))
               (setq! (nth _buffer (string->number (cadddr _got_list))) (string/gsub (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list) (caddr _got_list)))
              )
             )
            )
            ((> (n _got_list) 4)
             (when (and (string->number (cadddr _got_list)) (string->number (nth _got_list 5)))
              (when (and (nth _buffer (string->number (cadddr _got_list))) (nth _buffer (string->number (nth _got_list 5))))
               (for _line (string->number (cadddr _got_list)) (string->number (nth _got_list 5)) 1
                (when (and (pcall string/match (nth _buffer _line) (cadr _got_list)) (string/match (nth _buffer _line) (cadr _got_list)))
                 (setq! (nth _buffer _line) (string/gsub (nth _buffer _line) (cadr _got_list) (caddr _got_list)))
                )
               )
              )
             )
            )
            (true)
           )
          )
          ((eq? (car _got_list) "t")
           (let [(_old_buffer _buffer) (_times (string->number (cadr _got_list))) (_spaces nil)]
            (set! _buffer '())
            (when (not _times)
             (set! _times 1)
            )
            (set! _spaces (string/rep " " _times))
            (for-each _string _old_buffer
             (push! _buffer (car (list (string/gsub _string "\t" _spaces))))
            )
           )
          )
          (true)
         )
        )
       )
      )
      (if (write-lines! _file _buffer)
       (print! "File written.")
       (print! "Unable to write file.")
      )
     )
     (print! "Unable to open file.")
    )
   )
  )
  (print! "Unable to read file(s).")
 )
 (exit!)
)
;
; Scroll _list from _start by _step, typing s stop this, return _start _step and _line_number
;
(defun scroll (_list _start _step _line_number)
 (let [(_got_input nil) (_current_page (math/ceil (/ _start _step))) (_total_pages (math/ceil (/ (n _list) _step)))]
  (set! _list (drop _list (- _start 1)))
  (if (not _line_number)
   (for-each _list (groups-of _list _step)
    (when (neq? _got_input "s")
     (for-each _string _list
      (print! _string)
      (inc! _start)
     )
     (io/write (sprintf "< %d / %d >" _current_page _total_pages))
     (io/flush)
     (set! _got_input (io/read))
     (inc! _current_page)
    )
   )
   (for-each _list (groups-of _list _step)
    (when (neq? _got_input "s")
     (for-each _string _list
      (printf "%d\t%s" _start _string)
      (inc! _start)
     )
     (io/write (sprintf "< %d / %d >" _current_page _total_pages))
     (io/flush)
     (set! _got_input (io/read))
     (inc! _current_page)
    )
   )
  )
 )
 (list _start _step _line_number)
)
;
; A file name is mandatory
;
(when (= (n *arguments*) 0)
 (exit! "Please specify a file name." 3)
)
;
(import io (append-all!))
(import util (OS?))
(import lua/os (getenv execute tmpname remove))
(let* [(_file_name (car *arguments*)) (_buffer (read-lines! _file_name)) (_OS (OS?)) (_got_list '()) (_copied '()) (_from 0) (_to 0) (_to_insert nil) (_record_scroll '(1 40)) (_old_buffer _buffer) (_shell_command "")]
 (if (not _buffer)
  (progn
   (print! "New file.")
   (set! _buffer '())
  )
  (print! "File open successful.")
 )
 (when (and (eq? _OS "UNIX") (eq? (getenv "TERM") "xterm") (= (execute "which xsel >/dev/null") 0))
  (set! _OS "UNIX+XTerm&XSel")
 )
 (while true
  (io/write "led>")
  (io/flush)
  (when-with (_got_input (io/read))
   (set! _got_list (string/split _got_input ","))
   ;
   ; Check for any comma
   ;
   (for _index 1 (n _got_list) 1
    (when (string/match (nth _got_list _index) "|cM")
     (setq! (nth _got_list _index) (string/gsub (nth _got_list _index) "|cM" ","))
    )
   )
   ;
   (cond
    ;
    ; List the 1st lines (default 10), with n show line number
    ;
    ((eq? (car _got_list) "1")
     (with (_max_lines 10)
      (when (> _max_lines (n _buffer))
       (set! _max_lines (n _buffer))
      )
      (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _max_lines (string->number (cadr _got_list)))
      )
      (if (< (n _got_list) 3)
       (for _line 1 _max_lines 1
        (print! (nth _buffer _line))
       )
       (for _line 1 _max_lines 1
        (printf "%d\t%s" _line (nth _buffer _line))
       )
      )
     )
    )
    ;
    ; List the previous and/or following lines (default 5) of any line, with (a) or without (z) line number
    ;
    ((or (eq? (car _got_list) "a") (eq? (car _got_list) "z"))
     (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
      (let [(_selected_line (string->number (cadr _got_list))) (_half 5) (_start 1) (_end (n _buffer))]
       (when (and (string->number (caddr _got_list)) (nth _buffer (string->number (caddr _got_list))))
        (set! _half (string->number (caddr _got_list)))
       )
       (when (nth _buffer (- _selected_line _half))
        (set! _start (- _selected_line _half))
       )
       (when (nth _buffer (+ _selected_line _half))
        (set! _end (+ _selected_line _half))
       )
       (when (> (n _got_list) 3)
        (if (eq? (cadddr _got_list) "f")
         (set! _start _selected_line)
         (set! _end _selected_line)
        )
       )
       (if (eq? (car _got_list) "a")
        (for _line _start _end 1
         (printf "%d\t%s" _line (nth _buffer _line))
        )
        (for _line _start _end 1
         (print! (nth _buffer _line))
        )
       )
      )
     )
    )
    ;
    ; List the bottom lines (default 10), with n show line number
    ;
    ((eq? (car _got_list) "b")
     (with (_max_lines 10)
      (when (> _max_lines (n _buffer))
       (set! _max_lines (n _buffer))
      )
      (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _max_lines (string->number (cadr _got_list)))
      )
      (dec! _max_lines)
      (if (< (n _got_list) 3)
       (for _line (- (n _buffer) _max_lines) (n _buffer) 1
        (print! (nth _buffer _line))
       )
       (for _line (- (n _buffer) _max_lines) (n _buffer) 1
        (printf "%d\t%s" _line (nth _buffer _line))
       )
      )
     )
    )
    ;
    ; Copy all or some or only one line
    ;
    ((eq? (car _got_list) "c")
     (set! _copied '())
     (cond
      ((= (n _got_list) 1)
       (set! _copied _buffer)
      )
      ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _copied (list (nth _buffer (string->number (cadr _got_list)))))
      )
      ((and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
       (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
        (for _line (string->number (cadr _got_list)) (string->number (caddr _got_list)) 1
         (set! _copied (snoc _copied (nth _buffer _line)))
        )
       )
      )
      (true)
     )
     (cond
     ;
     ; Copy selected lines into clipboard, but this feature does not work well on AROS (untested on AmigaOS and MorphOS)
     ;
     ; ((eq? _OS "AmigaOS")
     ;  (with (_new_string "")
     ;   (for-each _string _copied
     ;    (for-each _char (string/string->chars _string)
     ;     (if (string/match _char "%a")
     ;      (set! _new_string (sprintf "%s%s" _new_string _char))
     ;      (set! _new_string (sprintf "%s*%s" _new_string _char))
     ;     )
     ;    )
     ;    (set! _new_string (sprintf "%s*n" _new_string))
     ;   )
     ;   (execute (sprintf "C:Run >NIL: C:Clip set \"%s\"" _new_string))
     ;  )
     ; )
     ;
      ((eq? _OS "UNIX+XTerm&XSel")
       (with (_temp_file (tmpname))
        (append-all! _temp_file (concat _copied "\n"))
        (execute (sprintf "xsel -cb && xsel -bl /tmp/.xsel.log <%s 2>/dev/null" _temp_file))
        (remove _temp_file)
       )
      )
      (true)
     )
    )
    ;
    ; Delete all or some or only one line
    ;
    ((eq? (car _got_list) "d")
     (cond
      ((= (n _got_list) 1)
       (set! _old_buffer _buffer)
       (set! _buffer '())
      )
      ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _old_buffer (snoc _buffer))
       (remove-nth! _buffer (string->number (cadr _got_list)))
      )
      ((and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
       (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
        (set! _old_buffer (snoc _buffer))
        (for _line (string->number (cadr _got_list)) (string->number (caddr _got_list)) 1
         (remove-nth! _buffer (string->number (cadr _got_list)))
        )
       )
      )
      (true)
     )
    )
    ;
    ; Edit a line (get it from clipboard), with i start to insert below that line, type a dot to return
    ;
    ((eq? (car _got_list) "e")
     (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
      (set! _old_buffer (snoc _buffer))
      (cond
       ((eq? _OS "AmigaOS")
        (let [(_old_string (nth _buffer (string->number (cadr _got_list)))) (_new_string "")]
         (for-each _char (string/string->chars _old_string)
          (if (string/match _char "%a")
           (set! _new_string (sprintf "%s%s" _new_string _char))
           (set! _new_string (sprintf "%s*%s" _new_string _char))
          )
         )
         (execute (sprintf "C:Run >NIL: C:Clip set \"%s\"" _new_string))
        )
       )
       ((eq? _OS "UNIX+XTerm&XSel")
        (with (_temp_file (tmpname))
         (append-all! _temp_file (nth _buffer (string->number (cadr _got_list))))
         (execute (sprintf "xsel -c && xsel -l /tmp/.xsel.log <%s 2>/dev/null" _temp_file))
         (remove _temp_file)
        )
       )
       (true
        (if-with (_selected_line (nth _buffer (string->number (cadr _got_list))))
         (printf "-\n%s\n+" _selected_line)
         (print! "-\n+")
        )
       )
      )
      (if (= (n _got_list) 2)
       (setq! (nth _buffer (string->number (cadr _got_list))) (io/read))
       (progn
        (remove-nth! _buffer (string->number (cadr _got_list)))
        (set! _to_insert '())
        (while (neq? (last _to_insert) ".")
         (set! _to_insert (snoc _to_insert (io/read)))
        )
        (pop-last! _to_insert)
        (for-each _string (reverse _to_insert)
         (insert-nth! _buffer (string->number (cadr _got_list)) _string)
        )
       )
      )
     )
    )
    ;
    ; Show file info or change file name
    ;
    ((eq? (car _got_list) "f")
     (if (= (n _got_list) 1)
      (if (empty? _buffer)
       (printf "Name: %s  Lines: 0  Bytes: 0  Format:" _file_name)
       (let [(_bytes (n _buffer)) (_format "Unix")]
        (for-each _string _buffer
         (set! _bytes (+ _bytes (string/len _string)))
        )
        (dec! _bytes)
        (when (string/ends-with? (car _buffer) "\r")
         (set! _format "DOS/Mac")
        )
        (printf "Name: %s  Lines: %d  Bytes: %d  Format: %s" _file_name (n _buffer) _bytes _format)
       )
      )
      (if (file-exists? (string/trim (cadr _got_list)))
       (progn
        (io/write "This file exists, do you want to proceed? <y/N>")
        (io/flush)
        (when (eq? (io/read) "y")
         (set! _file_name (string/trim (cadr _got_list)))
        )
       )
       (set! _file_name (string/trim (cadr _got_list)))
      )
     )
    )
    ;
    ; Mark a range of lines, list that range with line number
    ;
    ((eq? (car _got_list) "g")
     (if (= (n _got_list) 1)
      (for _line _from _to 1
       (when-with (_string (nth _buffer _line))
        (printf "%d\t%s" _line _string)
       )
      )
      (when (and (string->number (cadr _got_list)) (string->number (caddr _got_list)) (> (string->number (caddr _got_list)) (string->number (cadr _got_list))))
       (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
        (set! _from (string->number (cadr _got_list)))
        (set! _to (string->number (caddr _got_list)))
       )
      )
     )
    )
    ;
    ; Show help
    ;
    ((eq? (car _got_list) "h")
     (let* [(_help '("Usage: lua led.lua | FILE | -a SCRIPT_FILE LIST_FILE\n"
     "Commands are:"
     " 1[,#,n]       List the 1st lines (default 10), with n show line number"
     " a|z,#[,#,p|f] List the previous and/or following lines (default 5) of any line, with (a) or without (z) line number"
     " b[,#,n]       List the bottom lines (default 10), with n show line number"
     " c[,#,#]       Copy all or some or only one line"
     " d[,#,#]       Delete all or some or only one line"
     " e,#[,i]       Edit a line (get it from clipboard), with i start to insert below that line, type a dot to return"
     " f[,$]         Show file info or change file name"
     " g[,#,#]       Mark a range of lines, list that range with line number"
     " h[,#]         Show this help (default all)"
     " i[,#]         Insert into buffer given text below last or from any line, type a dot to return"
     " j,#,#         Join a line with another"
     " k             List marked range without line number"
     " l|n[,#,#]     List all or some or only one line, without (l) or with (n) line number"
     " m[,#,n]       Press Return to scroll # lines every time (default 40), with n show line number, type s to stop, 'm,' to continue"
     " o             Switch to DOS/Mac format"
     " p[,#]         Paste below last or from any line"
     " q             Ask to quit"
     " r,$[,#]       Read a file and insert into buffer content below last or from any line"
     " s,$[,$,#,#]   Search a string or substitute it with another one (support Lua patterns) in all or some or only one line, for a comma type |cM"
     " t[,#]         Replace tab with # spaces (default 1)"
     " u             Undo<->Redo"
     " v             Show version and license"
     " w             Write out"
     " x[,$,r]       Write and execute file or a shell command + file, execute a shell command, with r record it\n"
     "With -a apply d, j, o, r, s, t commands from SCRIPT_FILE to files listed in LIST_FILE")) (_show (n _help))]
      (when (and (> (n _got_list) 1) (string->number (cadr _got_list)) (nth _help (string->number (cadr _got_list))))
       (set! _show (string->number (cadr _got_list)))
      )
      (for _index 1 _show 1
       (print! (nth _help _index))
      )
     )
    )
    ;
    ; Insert into buffer given text below last or from any line, type a dot to return
    ;
    ((eq? (car _got_list) "i")
     (set! _to_insert '())
     (if (= (n _got_list) 1)
      (when-with (_string (last _buffer))
       (printf "%s" _string)
      )
      (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (when-with (_string (nth _buffer (- (string->number (cadr _got_list)) 1)))
        (printf "%s" _string)
       )
      )
     )
     (while (neq? (last _to_insert) ".")
      (set! _to_insert (snoc _to_insert (io/read)))
     )
     (pop-last! _to_insert)
     (if (= (n _got_list) 1)
      (progn
       (set! _old_buffer _buffer)
       (set! _buffer (append _buffer _to_insert))
      )
      (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _old_buffer (snoc _buffer))
       (for-each _string (reverse _to_insert)
        (insert-nth! _buffer (string->number (cadr _got_list)) _string)
       )
      )
     )
    )
    ;
    ; Join a line with another
    ;
    ((eq? (car _got_list) "j")
     (when (and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
      (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
       (set! _old_buffer (snoc _buffer))
       (setq! (nth _buffer (string->number (cadr _got_list))) (sprintf "%s%s" (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list)))))
      )
     )
    )
    ;
    ; List marked range without line number
    ;
    ((eq? (car _got_list) "k")
     (for _line _from _to 1
      (when-with (_string (nth _buffer _line))
       (printf "%s" _string)
      )
     )
    )
    ;
    ; List all or some or only one line, without line number
    ;
    ((eq? (car _got_list) "l")
     ;
     ; Check for the bottom line
     ;
     (for _index 1 (n _got_list) 1
      (when (string/match (nth _got_list _index) "^$")
       (setq! (nth _got_list _index) (string/gsub (nth _got_list _index) "^$" (sprintf "%d" (n _buffer))))
      )
     )
     ;
     (cond
      ((= (n _got_list) 1)
       (for _line 1 (n _buffer) 1
        (print! (nth _buffer _line))
       )
      )
      ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (print! (nth _buffer (string->number (cadr _got_list))))
      )
      ((and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
       (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
        (for _line (string->number (cadr _got_list)) (string->number (caddr _got_list)) 1
         (print! (nth _buffer _line))
        )
       )
      )
      (true)
     )
    )
    ;
    ; Press Return to scroll # lines every time (default 40), with n show line number, type s to stop, 'm,' to continue
    ;
    ((eq? (car _got_list) "m")
     (cond
      ((= (n _got_list) 1)
       (set! _record_scroll (scroll _buffer 1 40 false))
      )
      ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _record_scroll (scroll _buffer 1 (string->number (cadr _got_list)) false))
      )
      ((= (n _got_list) 3)
       ;
       ; Check for the default value
       ;
       (when (string/match (cadr _got_list) "^$")
        (setq! (cadr _got_list) (string/gsub (cadr _got_list) "^$" "40"))
       )
       ;
       (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
        (set! _record_scroll (scroll _buffer 1 (string->number (cadr _got_list)) true))
       )
      )
      (true
       (set! _record_scroll (scroll _buffer (car _record_scroll) (cadr _record_scroll) (caddr _record_scroll)))
      )
     )
    )
    ;
    ; List all or some or only one line, with line number
    ;
    ((eq? (car _got_list) "n")
     ;
     ; Check for the bottom line
     ;
     (for _index 1 (n _got_list) 1
      (when (string/match (nth _got_list _index) "^$")
       (setq! (nth _got_list _index) (string/gsub (nth _got_list _index) "^$" (sprintf "%d" (n _buffer))))
      )
     )
     ;
     (cond
      ((= (n _got_list) 1)
       (for _line 1 (n _buffer) 1
        (printf "%d\t%s" _line (nth _buffer _line))
       )
      )
      ((and (= (n _got_list) 2) (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (printf "%d\t%s" (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
      )
      ((and (> (n _got_list) 2) (string->number (cadr _got_list)) (string->number (caddr _got_list)))
       (when (and (nth _buffer (string->number (cadr _got_list))) (nth _buffer (string->number (caddr _got_list))))
        (for _line (string->number (cadr _got_list)) (string->number (caddr _got_list)) 1
         (printf "%d\t%s" _line (nth _buffer _line))
        )
       )
      )
      (true)
     )
    )
    ;
    ; Switch to DOS/Mac format
    ;
    ((eq? (car _got_list) "o")
     (set! _old_buffer _buffer)
     (set! _buffer '())
     (for-each _string _old_buffer
      (if (string/ends-with? _string "\r")
       (push! _buffer _string)
       (push! _buffer (sprintf "%s\r" _string))
      )
     )
    )
    ;
    ; Paste below last or from any line
    ;
    ((eq? (car _got_list) "p")
     (if (= (n _got_list) 1)
      (progn
       (set! _old_buffer _buffer)
       (set! _buffer (append _buffer _copied))
      )
      (when (and (string->number (cadr _got_list)) (nth _buffer (string->number (cadr _got_list))))
       (set! _old_buffer (snoc _buffer))
       (for-each _string (reverse _copied)
        (insert-nth! _buffer (string->number (cadr _got_list)) _string)
       )
      )
     )
    )
    ;
    ; Ask to quit
    ;
    ((eq? (car _got_list) "q")
     (io/write "Do you really want to quit now? <y/N>")
     (io/flush)
     (when (eq? (io/read) "y")
      (exit!)
     )
    )
    ;
    ; Read a file and insert into buffer content below last or from any line
    ;
    ((eq? (car _got_list) "r")
     (when (and (> (n _got_list) 1) (not (empty? (string/trim (cadr _got_list)))))
      (if (file-exists? (string/trim (cadr _got_list)))
       (with (_content (read-lines! (string/trim (cadr _got_list))))
        (set! _old_buffer _buffer)
        (if (and (string->number (caddr _got_list)) (nth _buffer (string->number (caddr _got_list))))
         (for-each _string (reverse _content)
          (insert-nth! _buffer (string->number (caddr _got_list)) _string)
         )
         (set! _buffer (append _buffer _content))
        )
        (print! "File read successful.")
       )
       (print! "File read failed.")
      )
     )
    )
    ;
    ; Search a string or substitute it with another one (support Lua patterns) in all or some or only one line
    ;
    ((eq? (car _got_list) "s")
     (cond
      ((= (n _got_list) 2)
       (let [(_pattern (cadr _got_list)) (_total_found 0)]
        (for _line 1 (n _buffer) 1
         (with (_string (nth _buffer _line))
          (when (and (pcall string/match _string _pattern) (string/match _string _pattern))
           (printf "%d\t%s" _line (nth _buffer _line))
           (set! _total_found (+ _total_found (cadr (list (string/gsub _string _pattern "")))))
          )
         )
        )
        (printf "Total found: %d" _total_found)
       )
      )
      ((= (n _got_list) 3)
       (set! _old_buffer (snoc _buffer))
       (for _line 1 (n _buffer) 1
        (when (and (pcall string/match (nth _buffer _line) (cadr _got_list)) (string/match (nth _buffer _line) (cadr _got_list)))
         (setq! (nth _buffer _line) (string/gsub (nth _buffer _line) (cadr _got_list) (caddr _got_list)))
        )
       )
      )
      ((= (n _got_list) 4)
       (when (and (string->number (cadddr _got_list)) (nth _buffer (string->number (cadddr _got_list))))
        (when (and (pcall string/match (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list)) (string/match (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list)))
         (set! _old_buffer (snoc _buffer))
         (setq! (nth _buffer (string->number (cadddr _got_list))) (string/gsub (nth _buffer (string->number (cadddr _got_list))) (cadr _got_list) (caddr _got_list)))
        )
       )
      )
      ((> (n _got_list) 4)
       (when (and (string->number (cadddr _got_list)) (string->number (nth _got_list 5)))
        (when (and (nth _buffer (string->number (cadddr _got_list))) (nth _buffer (string->number (nth _got_list 5))))
         (set! _old_buffer (snoc _buffer))
         (for _line (string->number (cadddr _got_list)) (string->number (nth _got_list 5)) 1
          (when (and (pcall string/match (nth _buffer _line) (cadr _got_list)) (string/match (nth _buffer _line) (cadr _got_list)))
           (setq! (nth _buffer _line) (string/gsub (nth _buffer _line) (cadr _got_list) (caddr _got_list)))
          )
         )
        )
       )
      )
      (true)
     )
    )
    ;
    ; Show version and license
    ;
    ((eq? (car _got_list) "v")
     (print! "led version 20210318\n\nCopyright (c) 2021, Pasquale Frega\nAll rights reserved.\nhttp://pasqualefrega.antiblog.com/\n\nReleased under the simplified BSD license.")
    )
    ;
    ; Replace tab with # spaces (default 1)
    ;
    ((eq? (car _got_list) "t")
     (set! _old_buffer _buffer)
     (set! _buffer '())
     (let [(_times (string->number (cadr _got_list))) (_spaces nil)]
      (when (not _times)
       (set! _times 1)
      )
      (set! _spaces (string/rep " " _times))
      (for-each _string _old_buffer
       (push! _buffer (car (list (string/gsub _string "\t" _spaces))))
      )
     )
    )
    ;
    ; Undo<->Redo
    ;
    ((eq? (car _got_list) "u")
     (with (_exchanger _buffer)
      (set! _buffer _old_buffer)
      (set! _old_buffer _exchanger)
     )
    )
    ;
    ; Write out
    ;
    ((eq? (car _got_list) "w")
     (if (write-lines! _file_name _buffer)
      (print! "File written.")
      (print! "Unable to write file.")
     )
    )
    ;
    ; Write and execute file or a shell command + file, execute a shell command, with r record it
    ;
    ((eq? (car _got_list) "x")
     (cond
      ((= (n _got_list) 1)
       (if (write-lines! _file_name _buffer)
        (execute (string/trim (sprintf "%s %s" _shell_command _file_name)))
        (print! "Unable to write file.")
       )
      )
      ((and (= (n _got_list) 2) (not (empty? (string/trim (cadr _got_list)))))
       (execute (string/trim (cadr _got_list)))
      )
      (true
       (set! _shell_command (string/trim (cadr _got_list)))
      )
     )
    )
    ;
    (true)
   )
  )
 )
)
;
; end