/* Copyright 2010 Fredrik Wikstrom. All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
**
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
**
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS `AS IS'
** AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
** LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
** CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
** SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
** CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
** POSSIBILITY OF SUCH DAMAGE.
*/

#include <exec/exec.h>
#include <devices/ahi.h>
#include <proto/exec.h>
#include "ahi.h"
#include "os4types.h"
#include "endian.h"

static struct MsgPort *mp;
static struct AHIRequest *io, *io2, *prev_io;
static APTR buf, buf2, prev_buf;

ULONG OpenAHIDevice (ULONG bufsize) {
	mp = CreateMsgPort();
	io = (struct AHIRequest *)CreateIORequest(mp, sizeof(*io));
	io2 = (struct AHIRequest *)CreateIORequest(mp, sizeof(*io));
	if (!io || !io2) return FALSE;
	io->ahir_Version = 4;
	if (OpenDevice("ahi.device", 0, (struct IORequest *)io, 0)) {
		io->ahir_Std.io_Device = NULL;
		return FALSE;
	}
	CopyMem(io, io2, sizeof(*io));
	buf = AllocVec(bufsize, MEMF_PUBLIC);
	buf2 = AllocVec(bufsize, MEMF_PUBLIC);
	if (!buf || !buf2) {
		return FALSE;
	}
	return 1UL << mp->mp_SigBit;
}

void CloseAHIDevice (void) {
	if (buf || buf2) {
		FreeVec(buf);
		FreeVec(buf2);
		buf = buf2 = NULL;
	}
	if (io && io->ahir_Std.io_Device) {
		if (prev_io) {
			AbortIO((struct IORequest *)prev_io);
			WaitIO((struct IORequest *)prev_io);
		}
		CloseDevice((struct IORequest *)io);
	}
	DeleteIORequest((struct IORequest *)io);
	DeleteIORequest((struct IORequest *)io2);
	io = io2 = NULL;
	DeleteMsgPort(mp);
	mp = NULL;
}

static void swab(APTR source, APTR destination, ULONG size) {
	UWORD *src = source;
	UWORD *dst = destination;
	size >>= 1;
	switch (size & 3) {
		case 3: wle16(dst++, *src++);
		case 2: wle16(dst++, *src++);
		case 1: wle16(dst++, *src++);
	}
	size >>= 2;
	while (size--) {
		wle16(dst++, *src++);
		wle16(dst++, *src++);
		wle16(dst++, *src++);
		wle16(dst++, *src++);
	}
}

void PlayCDDA (APTR cdda, ULONG cdda_size, ULONG volume) {
	swab(cdda, buf, cdda_size);
	io->ahir_Std.io_Command = CMD_WRITE;
	io->ahir_Std.io_Offset = 0;
	io->ahir_Std.io_Data = buf;
	io->ahir_Std.io_Length = cdda_size;
	io->ahir_Type = AHIST_S16S;
	io->ahir_Frequency = 44100;
	io->ahir_Volume = volume;
	io->ahir_Position = 0x08000;
	io->ahir_Link = prev_io;
	SendIO((struct IORequest *)io);
	if (prev_io) {
		WaitIO((struct IORequest *)prev_io);
	}
	prev_io = io; io = io2; io2 = prev_io;
	prev_buf = buf; buf = buf2; buf2 = prev_buf;
}
