/* AMOS Sprite and Icon Bank Viewer for AROS/AmigaOS */

/* AROS/AmigaOS includes: */
#include <dos/dos.h>
#include <intuition/intuition.h>
#include <graphics/gfx.h>
#include <proto/dos.h>
#include <proto/graphics.h>
#include <proto/intuition.h>


/* Patch for using C++ compiler: */
/*#define CPP 1*/

#ifdef CPP
#define MYCHAR char
#else
#define MYCHAR unsigned char
#endif

/* Set the scale: */
ULONG scale = 2;


/* Added for portability: */
#ifndef __AROS__
#   ifndef __MORPHOS__
#	  define IPTR unsigned long
#   endif
#endif


/* Cybergraphics includes: */
/*#define CGX_CGX_H <cybergraphics/cybergraphics.h>*/ /* enable for phase5 change */
/*#define CGX_CGX_H <libraries/cybergraphics.h>*/
#define CGX_CGX_H <cybergraphx/cybergraphics.h>

/*#ifdef HAVE_LIBRARIES_CYBERGRAPHICS_H
# define CGX_CGX_H <libraries/cybergraphics.h>
#else
# ifdef HAVE_CYBERGRAPHX_CYBERGRAPHICS_H
#  define CGX_CGX_H <cybergraphx/cybergraphics.h>
# endif
#endif*/

#include CGX_CGX_H
#include <proto/cybergraphics.h>

/*#if defined __MORPHOS__ || defined __AROS__ || defined __amigaos4__
# define USE_CYBERGFX_V41
#endif */


/* Endianness checking: */
#ifndef __AROS__
#   define USING_BIG_ENDIAN 1
#else
#   if AROS_BIG_ENDIAN
#       define USING_BIG_ENDIAN 1
#   endif
#endif


/* AROS/AmigaOS file selector includes: */
#include <exec/types.h>
#include <exec/libraries.h>
#include <libraries/asl.h>
#ifndef __AROS__
#   include <clib/exec_protos.h>
#   include <clib/asl_protos.h>
#else
#   include <proto/exec.h>
#   include <proto/asl.h>
#endif


/* Other includes: */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* For file selector: */
#ifdef LATTICE
int CXBRK(void)     { return(0); }  /* Disable Lattice CTRL/C handling */
void chkabort(void) { return; }     /* really */
#endif

UBYTE *vers = (UBYTE*)"$VER: filereq 37.0";

#define MYLEFTEDGE 0
#define MYTOPEDGE  0

/* To do - patch for window border - added: reduce by 11x11 for MorphOS skinnier borders */
#ifndef __MORPHOS__
#	define WBORDER 30
#	define HBORDER 35
#else
#	define WBORDER 19
#	define HBORDER 24
#endif

struct Library *AslBase = NULL;

struct TextAttr mytextattr =
{
	(MYCHAR*)"arial.font", 13, 0, 0
};

struct TagItem frtags[] =
{
	{ ASLFR_TextAttr, (IPTR)&mytextattr},
	{ ASLFR_TitleText,	        (IPTR)"Select an AMOS Sprite or Icon bank" },
	{ ASLFR_PositiveText,       (IPTR)"Load File" },
	{ ASLFR_NegativeText,       (IPTR)"Forget it" },
	{ ASLFR_InitialShowVolumes, FALSE		  },
	{ ASLFR_SetSortBy,		ASLFRSORTBY_Size  },
	{ ASLFR_SetSortOrder,	ASLFRSORTORDER_Descend },
	{ ASLFR_SetSortDrawers,	ASLFRSORTDRAWERS_Mix	},
	{ TAG_DONE,       	        0 }
};



/* Screen/window definitions: */
#define INITWIDTH 640
#define INITHEIGHT 480
#define SCREENWIDTH  1920
#define SCREENHEIGHT 1080
#define SCREENCY (SCREENHEIGHT / 2)

char* filename = NULL;
char* simplefilename = NULL;

/* File requester procedure: */
static void showrequester(char *msg, struct TagItem *tags)
{
	struct FileRequester *fr;

	/*printf("\n%s:\n",msg ? msg : "");*/

	if ((fr = (struct FileRequester *)AllocAslRequest(ASL_FileRequest, tags)))
	{
		if (AslRequest(fr, NULL))
		{
			/*printf("\n-------------------------------------------------------\n\n");*/
			/*printf("PATH=\"%s\"  FILE=\"%s\"\n", fr->rf_Dir, fr->rf_File ? fr->rf_File : (STRPTR)"<NOFILE>"); */
			/*printf("To combine the path and filename, copy the path\n");
			printf("to a buffer, add the filename with Dos AddPart().\n\n");*/

			if (fr->rf_File == NULL)
			{
				/*printf("Error loading file!\n");*/
				exit(0);
			}
			else
			{
				filename = (char*)malloc(sizeof(char) * (3 + strlen((char*)(fr->rf_Dir)) + strlen((char*)(fr->rf_File))) );
				simplefilename = (char*)malloc(sizeof(char) *( 3 + strlen((char*)(fr->rf_File))) );
				strcpy(simplefilename, (char*)(fr->rf_File));
				strcpy(filename, (char*)(fr->rf_Dir));
				strcat(filename, (char*)(fr->rf_File));
			}

			if(fr->fr_NumArgs > 0)
			{
				struct WBArg *wbarg = fr->fr_ArgList;
				WORD i;

				/*printf("MULTI SELECTION:\n"
				"----------------\n");*/

				for(i = 1; i <= fr->fr_NumArgs; i++)
				{
					/*printf("%3ld: %s\n", (long)i, wbarg->wa_Name);*/
					wbarg++;
				}
			}
		} else exit(0); /*printf("\nRequester was aborted\n"); */
		FreeAslRequest(fr);
	}
	else exit(0); /*printf("Could not alloc FileRequester\n");*/
}

/***********************************************************************************/

struct IntuitionBase *IntuitionBase;
struct GfxBase *GfxBase;
struct Library *CyberGfxBase;
struct Screen *scr;
struct Window *win;
struct RastPort *rp;

ULONG cgfx_coltab[256];
UBYTE Keys[128];

/***********************************************************************************/

char *buffer;
ULONG ***image;
ULONG ***image32;
ULONG *xsize;
ULONG *ysize;
ULONG *bitplanes;
LONG *xhandle;
LONG *yhandle;
ULONG *palette;
ULONG *colorpalette;

ULONG bufferpos = 0;
int usehalfbright = 1;
int usefilebuffer = 1;
int spriteloaded = 0;
int icon = 0;
unsigned int numobjects = 0;
ULONG bitplane = 0;


/* Read file: */
void ReadFile(char *name)
{
	FILE *file;
	unsigned long fileLen;

	/* Open file */
	file = fopen(name, "rb");
	if (!file)
	{
		/*fprintf(stderr, "Unable to open file %s\n", name);*/
		exit(0);
		return;
	}

	/* Get file length */
	fseek(file, 0, SEEK_END);
	fileLen=ftell(file);
	fseek(file, 0, SEEK_SET);

	/* Allocate memory */
	buffer=(char *)malloc(fileLen+1);
	if (!buffer)
	{
		/*fprintf(stderr, "Memory error!\n");*/
		fclose(file);
		exit(0);
		return;
	}

	/* Read file contents into buffer */
	fread(buffer, fileLen, 1, file);
	fclose(file);
}

UBYTE getnextbyte()
{
	UBYTE out = buffer[bufferpos];
	bufferpos++;
	return out;
}

ULONG getcolor(unsigned int r, unsigned int g, unsigned int b)
{
	ULONG out = 0;

#ifdef USING_BIG_ENDIAN
	out |= b & 0xff;
	out |= ((g & 0xff) << 8);
	out |= ((r & 0xff) << 16);
#else
	out |= ((b & 0xff) << 24);
	out |= ((g & 0xff) << 16);
	out |= ((r & 0xff) << 8);
#endif

	return out;
}

void decode()
{
	char charfield = 0;
	char str[10];

	unsigned int foundbank = 0;
	unsigned int a, obj, bitmap, x, y, bit, bitnum;
	ULONG entry;
	unsigned int red, green, blue;
	unsigned int index;

	/* First check the first 4 characters: */
	for (a=0; a<4; a++)
	{
		charfield = getnextbyte();
		str[a] = charfield;
	}
	str[4]=0;

	if (strlen(str)==4 && strcmp(str, "AmSp")==0)
	{
		/*printf("AMOS Sprite Bank\n");*/
		foundbank = 1;
		icon = 0;
	}
	else if (strlen(str)==4 && strcmp(str, "AmIc")==0)
	{
		/*printf("AMOS Icon Bank\n");*/
		foundbank = 1;
		icon = 1;
	}
	else if (strlen(str)==4)
	{
		if (strlen(str)==4 && strcmp(str, "AmBk")==0)
		{
			/*printf("Wrong type of AMOS Bank. Only Sprite/Object/Bob and Icon type are supported. Quitting.\n");*/
			exit(0);
		}
		else
		{
			/*printf("Not an AMOS bank. Quitting.\n");*/
			exit(0);
		}
		foundbank = 0;
		/*parent.abkfile.closebinaryfile(parent.abkfile.in);*/
		return;
	}

	/* Exit if bank not found: */
	if (!foundbank)
		;/*(parent.abkfile).closebinaryfile((parent.abkfile).in);*/

	str[0] = 0;

	/* Now try to find the number of objects: */
	numobjects = (getnextbyte()<<8)|(getnextbyte());
	/*printf("Number of Objects: %d\n", numobjects);*/

	image = (ULONG***)malloc(sizeof(ULONG**) * numobjects);
	xsize = (ULONG*)malloc(sizeof(ULONG) * numobjects);
	ysize = (ULONG*)malloc(sizeof(ULONG) * numobjects);
	bitplanes = (ULONG*)malloc(sizeof(ULONG) * numobjects);
	xhandle = (LONG*)malloc(sizeof(LONG) * numobjects);
	yhandle = (LONG*)malloc(sizeof(LONG) * numobjects);

	/* Now parse each object in turn to get its details: */
	for (obj=0; obj<numobjects; obj++)
	{
		xsize[obj] = (getnextbyte()<<8)|(getnextbyte()) << 4;
		ysize[obj] = (getnextbyte()<<8)|(getnextbyte());
		bitplanes[obj] = (getnextbyte()<<8)|(getnextbyte());
		xhandle[obj] = (getnextbyte()<<8)|(getnextbyte());
		yhandle[obj] = (getnextbyte()<<8)|(getnextbyte());

		/*printf("For Object %d: X_Size=%d, Y_Size=%d, Bitplanes=%d, X_Hot_Spot=%d, Y_Hot_Spot=%d\n",
		obj, xsize[obj], ysize[obj], bitplanes[obj], xhandle[obj], yhandle[obj]);*/


		/* Allocate an array to hold the converted bitmap: */
		image[obj] = NULL;
		image[obj] = (ULONG**) malloc(sizeof(ULONG*) * xsize[obj]);
		for (x=0; x<xsize[obj]; x++)
			image[obj][x] = (ULONG*) malloc(sizeof(ULONG) * ysize[obj]);

		/*printf("Image size: %d, %d\n", xsize[obj], ysize[obj]);*/

		/* Clear the buffer: */
		for (x=0; x<xsize[obj]; x++)
			for (y=0; y<ysize[obj]; y++)
				image[obj][x][y] = 0;

		/* Now parse the bitmap itself: */
		for (bitplane=0; bitplane<bitplanes[obj]; bitplane++)
		{
			/* Clear the bitmap - add each bit on in turn: */
			bitmap = 0;

			/* Parse the X and Y here: */
			for (y=0; y<ysize[obj]; y++)
			{
				for (x=0; x<xsize[obj]; x+=16)
				{
					bit=0;

					/* Grab the 16-bit bitmap segment here: */
					bitmap = ((unsigned)getnextbyte() << 8)|((unsigned)getnextbyte());

					/* Now parse and manipulate the bits and convert to chunky byte format: */
					for (bitnum=0; bitnum<16; bitnum++)
					{
						/* Get least significant bit: */
						bit = bitmap & 1;

						/* Shift down: */
						bitmap = ((unsigned)bitmap) >> 1;

						/* Now add to the chunky paletted buffer: */
						image[obj][x+(15 - bitnum)][y] |= (bit << bitplane);
					}
				}
			}
		}
	}

	/* Finally after all objects are processed, get the sprite palette details. */
	/* This is always 32 colours: */

	/* Create extra indexes if using half bright mode: */
	if (usehalfbright)
	{
		palette = (ULONG*)malloc(sizeof(ULONG) * 64);
		colorpalette = (ULONG*)malloc(sizeof(ULONG) * 64);
	}
	else
	{
		palette = (ULONG*)malloc(sizeof(ULONG) * 32);
		colorpalette = (ULONG*)malloc(sizeof(ULONG) * 32);
	}

	/* Note: Always uses 32 colours for the sprite palette. */
	for (index=0; index<32; index++)
	{
		entry = ((getnextbyte() << 8 ) + getnextbyte());
		palette[index]=entry;

		/*printf("Colour %d: %x\n", index, entry);*/

		/* Now get the red, green and blue values */
		/* Bitwise AND to separate colour entries: */
		red   = (unsigned)(entry & 0xF00) >> 8;
		green = (unsigned)(entry & 0x0F0) >> 4;
		blue  = (unsigned)(entry & 0x00F);

		/* Added: For 64 colour mode - 6 bitplanes on sprite images: Do the old-skool palette half-bright (may incur rounding): */
		/* If the 24-bit/32-bit buffer need be dropped, this may work as a fall-back - but then implement EHB directly (TO DO): */
		if (usehalfbright)
			palette[index + 32] = (((unsigned)red>>1) * 0x100) + (((unsigned)green>>1) * 0x10) + ((unsigned)blue>>1);

		/* Multiply by 0x11 (17) to scale 0xFFF to 0xFFFFFF */
		/* Multiply by 0x11 (17) to convert 0xF (15) into 0xFF (255): */
		red   *= 0x11;
		green *= 0x11;
		blue  *= 0x11;
		colorpalette[index]=getcolor(red, green, blue);

		/* Added: For 64 colour mode: Do the half-bright 24-bit/32-bit palette index (if appropriate): */
		/* Note that this uses true 24-bit brightness halving. */
		if (usehalfbright)
			colorpalette[index + 32]=getcolor(((unsigned)red)>>1, ((unsigned)green)>>1, ((unsigned)blue)>>1);
	}

	/* The file can be closed now if not using a buffer (otherwise it is closed earlier): */
	if (!usefilebuffer)
		; /*(abkfile).closebinaryfile(abkfile.in); */

	/* Before drawing the image, create a 32-bit 2D array: */
	image32 = (ULONG***)malloc(sizeof(ULONG**) * numobjects);
	for (obj=0; obj<numobjects; obj++)
	{
		image32[obj] = NULL;
		image32[obj] = (ULONG**) malloc(sizeof(ULONG*) * xsize[obj]);
		for (x=0; x<xsize[obj]; x++)
		{
			image32[obj][x] = (ULONG*) malloc(sizeof(ULONG) * ysize[obj]);
		}
	}

	/* Create a 24-bit/32-bit Color image array: */
	for (obj=0; obj<numobjects; obj++)
		for (y=0; y<ysize[obj]; y++)
			for (x=0; x<xsize[obj]; x++)
				image32[obj][x][y] = colorpalette[image[obj][x][y]];

	/* Close the raw file buffer here: */
	free(buffer);

	spriteloaded = 1;
}

static void cleanup(char *msg)
{
	if (msg)
	{
		/*printf("WritePixelArray: %s\n",msg);*/
	}

	if (win) CloseWindow(win);

	if (scr) UnlockPubScreen(0, scr);

	if (CyberGfxBase) CloseLibrary(CyberGfxBase);
	if (GfxBase) CloseLibrary((struct Library *)GfxBase);
	if (IntuitionBase) CloseLibrary((struct Library *)IntuitionBase);

	exit(0);
}

/***********************************************************************************/

static void openlibs(void)
{
	if (!(IntuitionBase = (struct IntuitionBase *)OpenLibrary((MYCHAR*)"intuition.library", 39)))
	{
		cleanup((char*)"Can't open intuition.library V39!");
	}

	if (!(GfxBase = (struct GfxBase *)OpenLibrary((MYCHAR*)"graphics.library", 39)))
	{
		cleanup((char*)"Can't open graphics.library V39!");
	}

	if (!(CyberGfxBase = OpenLibrary((MYCHAR*)"cybergraphics.library",0)))
	{
		cleanup((char*)"Can't open cybergraphics.library!");
	}
}

/***********************************************************************************/

static void getvisual(void)
{
	if (!(scr = LockPubScreen(NULL)))
	{
		cleanup((char*)"Can't lock pub screen!");
	}

	if (GetBitMapAttr(scr->RastPort.BitMap, BMA_DEPTH) <= 8)
	{
		cleanup((char*)"Need hi or true color screen!");
	}
}

/***********************************************************************************/

static void makewin(void)
{
	char* title;
	struct NewWindow winlayout =
	{
		100, 100, INITWIDTH, INITHEIGHT, 0, 1, IDCMP_CLOSEWINDOW | IDCMP_REFRESHWINDOW,
		WFLG_SIZEGADGET | WFLG_CLOSEGADGET | WFLG_DEPTHGADGET | WFLG_DRAGBAR | WFLG_ACTIVATE,
		NULL, NULL,
		NULL,
		NULL, NULL,
		40, 40,
		1920, 1080,
		WBENCHSCREEN
	};

	title = (char*) malloc(sizeof(char) * 256);

	win = OpenWindow(&winlayout);

	if (!win)
	{
		cleanup((char*)"Can't open window");
		return;
	}

	/* Set window title: */
	if (icon)
		strcpy(title, "AMOS Icon Bank Viewer");
	else
		strcpy(title, "AMOS Sprite Bank Viewer");

	strcat(title, " - ");
	strcat(title, simplefilename);

	SetWindowTitles(win, (MYCHAR*)title, (MYCHAR*)"AbkViewer");

	ScreenToFront(win->WScreen);
	SizeWindow(win, 0, 0);
	WindowToFront(win);

	rp = win->RPort;
}

/***********************************************************************************/

#define KC_LEFT         0x4F
#define KC_RIGHT     	0x4E
#define KC_UP        	0x4C
#define KC_DOWN      	0x4D
#define KC_ESC       	0x45

/***********************************************************************************/

static void getevents(void)
{
	struct IntuiMessage *msg;

	while ((msg = (struct IntuiMessage *)GetMsg(win->UserPort)))
	{
		switch(msg->Class)
		{
		case IDCMP_CLOSEWINDOW:
			Keys[KC_ESC] = 1;
			break;

		case IDCMP_RAWKEY:
			{
				WORD code = msg->Code & ~IECODE_UP_PREFIX;

				Keys[code] = (code == msg->Code) ? 1 : 0;

			}
			break;

		}
		ReplyMsg((struct Message *)msg);
	}

}

/***********************************************************************************/

static void redraw(void)
{
	static LONG tab[SCREENWIDTH * SCREENHEIGHT];
	ULONG x, y, xx, yy, obj;
	ULONG lastx = 0;
	ULONG lasty = 0;
	ULONG maxy = 0;
	int drawn = 0;
	ULONG width = SCREENWIDTH - WBORDER;
	ULONG height = SCREENHEIGHT - HBORDER;
	ULONG lastwidth = width;
	ULONG lastheight = height;

	/* The top-left border: */
	ULONG xinitialposition = 0;
	ULONG yinitialposition = 0;

	/* Gap between sprites: */
	ULONG xspace = 10;
	ULONG yspace = 10;

	/* Add small offset to top-left border: */
	xinitialposition += 2;
	yinitialposition += 2;

	while(!Keys[KC_ESC])
	{
		/*x = scr->MouseX;*/
		/*if (x < 0) x = 0; else if (x >= scr->Width) x = scr->Width - 1;*/

		width = win -> Width - WBORDER;
		height = win -> Height - HBORDER;
		if (width >  (SCREENWIDTH - WBORDER))
			width=SCREENWIDTH - WBORDER;
		if (height > (SCREENHEIGHT - WBORDER))
			height=SCREENHEIGHT - HBORDER;

		if ((lastwidth != width) || (lastheight != height))
			drawn = 0;

		lastwidth = width; lastheight = height;

		if (!drawn)
		{
			/* Clear buffer: */
			for(y = 0; y < height; y ++)
				for(x = 0; x < width; x++)
					tab[y * SCREENWIDTH + x] = colorpalette[0];

			/* Try to draw the sprites! */
			lastx = xinitialposition; lasty = yinitialposition;
			maxy = 0;
			for (obj=0; obj<numobjects; obj++)
			{
				if (((lastx+xsize[obj])>=((width - xinitialposition)/scale)) && (obj > 0)) /*remove obj>0?*/
				{
					lastx = xinitialposition;
					lasty += maxy + yspace;
					maxy = 0;
				}
				for(y = 0; y < ysize[obj]; y ++)
				{
					for(x = 0; x < xsize[obj]; x++)
					{
						/* These inner loops are used for scaling the images: */
						for (yy = 0; yy < scale; yy++)
						{
							for (xx = 0; xx < scale; xx++)
							{
								if ((((x+lastx)*scale+xx) < width) && (((y+lasty)*scale+yy) < height)
									&& (image[obj][x][y] != 0))
									tab[((y+lasty)*scale+yy) * SCREENWIDTH + ((x+lastx)*scale+xx)]
								= image32[obj][x][y];
							}
						}
					}
				}
				lastx += xsize[obj] + xspace;
				if (ysize[obj] > maxy)
					maxy = ysize[obj];
			}

			WritePixelArray(tab, 0, 0, SCREENWIDTH * sizeof(LONG),
				win->RPort, win->BorderLeft, win->BorderTop, width, height,
				RECTFMT_ARGB);

			drawn = 1;
		}

		getevents();
	}
}


/***********************************************************************************/

int main(int argc, char **argv)
{
	openlibs();
	getvisual();

	if (argc <= 1)
	{
		/* Load file selector here: */
		if ((AslBase = OpenLibrary((MYCHAR*)"asl.library", 37L)))
		{
			showrequester(NULL, frtags);
			CloseLibrary(AslBase);
		}
		else
		{
			puts("Could not open asl.library!\n");
			exit(0);
		}
	}
	else
	{
		filename = argv[1];
		simplefilename = argv[1]; /* fix this */
	}

	ReadFile(filename);
	decode();
	makewin();

	if (spriteloaded)
		redraw();

	cleanup(0);
	return 0;
}

/***********************************************************************************/
