/*------------------------------------------------------------------------*\
 |                "Fractal Mountains" landscape generator.                |
 *------------------------------------------------------------------------*
 |              Original algorithm and BASIC implementation               |
 |                    1987 Compute Publications, Inc.                    |
 |                                                                        |
 |                 KickPascal conversion by W. Nker 1992                 |
 |           C conversion and interactive GUI by W. Nker 1999            |
 *------------------------------------------------------------------------*
 |                  Wilhelm Nker <wnoeker@t-online.de>                   |
\*------------------------------------------------------------------------*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <proto/exec.h>
#include <proto/intuition.h>
#include <proto/gadtools.h>
#include <proto/graphics.h>
#undef GetOutlinePen    /* suppress a warning for the gcc include files */
#include <graphics/gfxmacros.h>
#include "gui.h"

typedef double REAL;
#define MAPSIZE 64

REAL raMap[ MAPSIZE+1 ][ MAPSIZE+1 ];
REAL rPeak, rSnowLine;
BOOL fBorder = FALSE;
WORD wHotGadget = IDC_NONE;

BOOL CheckAbort();


/*------------------------------------------------------------------------*\

  Generate a random number 0.0 < r < 1.0.
  Calling srand( time( NULL ) ) is recommended to seed the generator.

\*------------------------------------------------------------------------*/

REAL Random()
    {
    return rand() / (RAND_MAX + 1.0);
    }



/*------------------------------------------------------------------------*\

  Recalculate map levels * using some of their neighbours +.
  The following patterns are applied:

        DoColumns()     DoRows()        DoCentres()

        + * + * +       +  +  +        +  + 
                   *  *  *       + * + * +
        + * + * +       +  +  +        +  + 
                   *  *  *       + * + * +
        + * + * +       +  +  +        +  + 

  The wGrid parameter is the granularity (distance between two *'s) and
  is decreased with each iteration, starting at MAPSIZE, ending at 2.
  The above diagrams show the 2nd iteration (i.e. wGrid == MAPSIZE / 2).

\*------------------------------------------------------------------------*/

VOID DoColumns( WORD wGrid, REAL rScale )
    {
    WORD wX, wY, nHalf = wGrid / 2;

    for( wY = 0; wY <= MAPSIZE; wY += wGrid )
        for( wX = nHalf; wX <= MAPSIZE; wX += wGrid )
            {
            raMap[ wX ][ wY ] =
                (raMap[ wX-nHalf ][ wY ] + raMap[ wX+nHalf ][ wY ]) / 2
                + (Random() - 0.5) * wGrid * rScale;
            }
    }


VOID DoRows( WORD wGrid, REAL rScale )
    {
    WORD wX, wY, nHalf = wGrid / 2;

    for( wX = 0; wX <= MAPSIZE; wX += wGrid )
        for( wY = nHalf; wY <= MAPSIZE; wY += wGrid )
            {
            raMap[ wX ][ wY ] =
                (raMap[ wX ][ wY-nHalf ] + raMap[ wX ][ wY+nHalf ]) / 2
                + (Random() - 0.5) * wGrid * rScale;
            }
    }


VOID DoCentres( WORD wGrid, REAL rScale )
    {
    WORD wX, wY, nHalf = wGrid / 2;

    for( wX = nHalf; wX <= MAPSIZE; wX += wGrid )
        for( wY = nHalf; wY <= MAPSIZE; wY += wGrid )
            {
            raMap[ wX ][ wY ] =
                (raMap[ wX-nHalf ][ wY ] + raMap[ wX+nHalf ][ wY ]
               + raMap[ wX ][ wY-nHalf ] + raMap[ wX ][ wY+nHalf ]) / 4
                + (Random() - 0.5) * wGrid * rScale;
            }
    }



/*------------------------------------------------------------------------*\

  Generate a random map. The larger the rScale parameter, the "craggier"
  it will look.

\*------------------------------------------------------------------------*/

VOID MakeMountain( REAL rScale )
    {
    WORD wX, wY, wGrid;
    WORD wArea, wOcean, wIce;
    static char caTitle[ 80 ];

    srand( time( NULL ) );
    for( wX = 0; wX <= MAPSIZE; wX++ )
        for( wY = 0; wY <= MAPSIZE; wY++ )
            raMap[ wX ][ wY ] = 0;
    for( wGrid = MAPSIZE; wGrid >= 2; wGrid /= 2 )
        {
        sprintf( caTitle, "Building landscape (%02d)...", wGrid );
        SetWindowTitles( pwinMain, caTitle, (STRPTR)-1 );
        DoColumns( wGrid, rScale );
        DoRows( wGrid, rScale );
        DoCentres( wGrid, rScale );
        if( CheckAbort() )
            break;
        }

    /* Determine the peak height. */
    rPeak = 0;
    for( wX = 0; wX <= MAPSIZE; wX++ )
        for( wY = 0; wY <= MAPSIZE; wY++ )
            if( rPeak < raMap[ wX ][ wY ] )
                rPeak = raMap[ wX ][ wY ];
    rSnowLine = 0.75 * rPeak;

    /* Calculate (and display) some statistics. */
    wArea = wOcean = wIce = 0;
    for( wX = 0; wX <= MAPSIZE; wX++ )
        for( wY = 0; wY <= MAPSIZE; wY++ )
            {
            wArea++;
            if( raMap[ wX ][ wY ] < 0 )
                wOcean++;
            if( raMap[ wX ][ wY ] > rSnowLine )
                wIce++;
            }
    if( wGrid > 2 )
        strcat( caTitle, " aborted" );
    else
        sprintf( caTitle, "Ocean: %d %%, Ice: %d %%",
            (100 * wOcean) / wArea, (100 * wIce) / wArea );
    SetWindowTitles( pwinMain, caTitle, (STRPTR)-1 );
    }



/*------------------------------------------------------------------------*\

  Calculate the color for one triangle in the fractal surface, the steeper
  the darker. To be precise: the illumination is proportional to the
  z-component of the normalized cross-product of two of the triangle's edge
  vectors. And because of the regular layout of the x/y-coordinates, which
  always mark an isosceles, right-angled triangle, for example:

        ^
        |
    y+1 +   z1 *-----*
        |      |\   /|
        |      | \ z0|
        |      |  *  |
        |      | / \ |
        |      |/   \|
     y  +   z2 *-----*
        |
        +------+-----+----->
               x    x+1

  everything boils down to the rather boring formula that you will find
  below.

  Additional rules: If all z-values are above the snowline, the color will
  be gray instead of brown, if they are below zero, the color will be blue.

  The returned pen number is intended for the following color map:
         0: black
     1..15: brown (dark to light)
        16: blue
    17..31: gray (dark gray to white)

\*------------------------------------------------------------------------*/

WORD GetShadePen( REAL rZ0, REAL rZ1, REAL rZ2, REAL rSnowLine )
    {
    REAL rLight;
    WORD wPen = 1;

    if( rZ0 <= 0 && rZ1 <= 0 && rZ2 <= 0 )
        return 16;              /* deep blue sea */
    if( rZ0 > rSnowLine && rZ1 > rSnowLine && rZ2 > rSnowLine )
        wPen += 16;             /* snow peaked mountains */
    rZ1 -= rZ0;
    rZ2 -= rZ0;
    rLight = 1 / (1 + 2*rZ1*rZ1 + 2*rZ2*rZ2);
    return wPen + (WORD)(14 * rLight + 0.5);
    }



/*------------------------------------------------------------------------*\

  Transform map coordinates into window coordinates. This will look
  somehow like this:
     _________________________________________________
    ||____________________________________________|__|
    ||                                              | |
    ||  \ y                                         | |
    ||                                              | |
    ||    +-MAPSIZE--------------+                  | |
    ||     \                       \                | |
    ||      \                        \              | |
    ||       \                         \            | |
    ||        \                          \          | |
    ||       0 +--------------------------+  -> x   | |
    ||         0                       MAPSIZE      | |
    ||                                              |_|
    ++==============================================|/|

\*------------------------------------------------------------------------*/

typedef struct
    {
    REAL x;
    REAL y;
    REAL z;
    } VECTOR;

struct                  /* Vectors for projection */
    {
    VECTOR f;           /* the point we focus on */
    VECTOR o;           /* the observer */
    VECTOR l;           /* the line of sight */
    VECTOR h;           /* horizon on our silver screen */
    VECTOR v;           /* vertical line */
    REAL rXScale;
    REAL rYScale;
    WORD wXOff;
    WORD wYOff;
    } prj;


/*------------------------------------------------------------------------*\

  Set a new postion for the observer, in spherical coordinates around
  the center of our landscape. Parameters:

    rR - radius, in multiples of the landscape's diameter
    rPhi - angle around the z-axis, 0..360 degrees
    rTheta - angle against the z-axis, 0..180 degrees

\*------------------------------------------------------------------------*/

VOID SetObserver( REAL rR, REAL rPhi, REAL rTheta )
    {
    REAL rDeg2Rad = 3.14159 / 180;

    rR *= sqrt( 2 * MAPSIZE*MAPSIZE + rPeak*rPeak );
    rPhi *= rDeg2Rad;
    rTheta *= rDeg2Rad;
    /* Focus shall be in the middle of the chunk of landscape. */
    prj.f.x = MAPSIZE / 2;
    prj.f.y = MAPSIZE / 2;
    prj.f.z = rPeak / 2;
    /* The observer's line of sight. */
    prj.l.x = -cos( rPhi )*sin( rTheta );
    prj.l.y = -sin( rPhi )*sin( rTheta );
    prj.l.z = -cos( rTheta );
    /* Their position in absolute x, y, z. */
    prj.o.x = prj.f.x - rR * prj.l.x;
    prj.o.y = prj.f.y - rR * prj.l.y;
    prj.o.z = prj.f.z - rR * prj.l.z;
    /* Base vectors of the projection plane. */
    prj.h.x = -sin( rPhi );
    prj.h.y =  cos( rPhi );
    prj.h.z =  0;
    prj.v.x =  cos( rTheta )*cos( rPhi );
    prj.v.y =  cos( rTheta )*sin( rPhi );
    prj.v.z = -sin( rTheta );
    }

VOID Transform( REAL rX, REAL rY, REAL rZ, WORD *pwX, WORD *pwY )
    {
    VECTOR v;
    REAL r;
    if( rZ < 0)
        rZ = 0;
    /* Calculate a vector from observer to target, */
    /* normalize it with respect to the observer's line of sight. */
    r = (rX - prj.o.x)*prj.l.x + (rY - prj.o.y)*prj.l.y + (rZ - prj.o.z)*prj.l.z;
    v.x = (rX - prj.o.x) / r;
    v.y = (rY - prj.o.y) / r;
    v.z = (rZ - prj.o.z) / r;
    /* Do the projection and adjust for screen display. */
    *pwX = prj.wXOff + prj.rXScale *
            (v.x*prj.h.x + v.y*prj.h.y + v.z*prj.h.z);
    *pwY = prj.wYOff + prj.rYScale *
            (v.x*prj.v.x + v.y*prj.v.y + v.z*prj.v.z);
    }



/*------------------------------------------------------------------------*\

  Adjust projection scaling and offsets, so that the eight vertices of
  the cuboid surrounding our landscape best fit into the available drawing
  area.

\*------------------------------------------------------------------------*/

VOID MakeScales( WORD wXMin, WORD wXMax, WORD wYMin, WORD wYMax )
    {
    REAL rMag, rTmp;
    REAL rX, rY, rZ;
    WORD wX, wY;
    WORD wX0, wX1, wY0, wY1;
    WORD wIdx, wRun;
    WORD wXSize = wXMax - wXMin, wYSize = wYMax - wYMin;

    /* We calculate the required magnification by iteration. */
    /* The reason to do so is a little pathetic: Although the formula */
    /* itself is precise, we have to deal with rounding errors, as we */
    /* chose to let the coordinate transformation return integer values. */
    wX0 = wX1 = wY0 = wY1 = 0;  /* suppress optimizer warning */

    /* "First guess" for scaling and offsets. */
    /* If a skewed x/y scaling ratio is required, include it here. */
    prj.rXScale =
    prj.rYScale = wXSize;
    prj.wXOff = 0;
    prj.wYOff = 0;
    for( wRun = 0; wRun < 4; wRun++ )
        {
        /* Make projections of the eight vertices. */
        rX = rY = rZ = 0;
        for( wIdx = 0; wIdx < 8; wIdx++ )
            {
            Transform( rX, rY, rZ, &wX, &wY );
            if( wX0 > wX || wIdx == 0 )
                wX0 = wX;
            if( wY0 > wY || wIdx == 0 )
                wY0 = wY;
            if( wX1 < wX || wIdx == 0 )
                wX1 = wX;
            if( wY1 < wY || wIdx == 0 )
                wY1 = wY;
            switch( wIdx )
                {
                case 0:
                    rX += MAPSIZE+1;
                    break;
                case 1:
                    rY += MAPSIZE+1;
                    break;
                case 2:
                    rX -= MAPSIZE+1;
                    break;
                case 3:
                    rZ += rPeak;
                    break;
                case 4:
                    rX += MAPSIZE+1;
                    break;
                case 5:
                    rY -= MAPSIZE+1;
                    break;
                case 6:
                    rX -= MAPSIZE+1;
                    break;
                }
            }
        /*printf( "(%d|%d) : (%d|%d)\n", wX0, wY0, wX1, wY1 );*/
        rMag = wXSize / (REAL)(wX1 - wX0 + 1);
        rTmp = wYSize / (REAL)(wY1 - wY0 + 1);
        if( rMag > rTmp )
            rMag = rTmp;
        prj.rXScale *= rMag;
        prj.rYScale *= rMag;
        prj.wXOff -= rMag * wX0;
        prj.wYOff -= rMag * wY0;
        /*printf( "+ (%d|%d), * %f\n", prj.wXOff, prj.wYOff, rMag );*/
        }

    prj.wXOff += wXMin + 1;
    prj.wYOff += wYMin + 1;
    }


/*------------------------------------------------------------------------*\

  Actually draw a sketch of the cuboid for which we made room in
  MakeScales().

\*------------------------------------------------------------------------*/

VOID DrawBox()
    {
    REAL rX, rY, rZ;
    WORD wX0, wX1=0, wY0, wY1=0;
    WORD wIdx;

    SetAPen( pwinMain->RPort, 0 );
    RectFill( pwinMain->RPort,
        pwinMain->BorderLeft,
        pwinMain->BorderTop,
        pwinMain->Width - pwinMain->BorderRight - 1,
        pwinMain->Height - pwinMain->BorderBottom - 1 );
    SetAPen( pwinMain->RPort, 1 );
    SetDrMd( pwinMain->RPort, JAM1 );
    /* Make projections of the eight vertices. */
    rX = rY = rZ = 0;
    for( wIdx = 0; wIdx < 8; wIdx++ )
        {
        wX0 = wX1;
        wY0 = wY1;
        Transform( rX, rY, rZ, &wX1, &wY1 );
        if( wIdx == 0 )
            Move( pwinMain->RPort, wX1, wY1 );
        else
            Draw( pwinMain->RPort, wX1, wY1 );
        switch( wIdx )
            {
            case 0:
                rX += MAPSIZE+1;
                break;
            case 1:
                rY += MAPSIZE+1;
                break;
            case 2:
                rX -= MAPSIZE+1;
                break;
            case 3:
                rZ += rPeak;
                break;
            case 4:
                rX += MAPSIZE+1;
                break;
            case 5:
                rY -= MAPSIZE+1;
                break;
            case 6:
                rX -= MAPSIZE+1;
                break;
            }
        }
    }


/*------------------------------------------------------------------------*\

  Draw an image of the the previously calculated landscape.

  You must have called SetObserver() and MakeScales() already!

\*------------------------------------------------------------------------*/

VOID DrawMountain( REAL rSnowLine )
    {
    WORD wX0, wY0, wXStep, wYStep;
    WORD wI, wJ, wX, wY, wIdx, wSwap, wPen;
    WORD waX[ 5 ], waY[ 5 ];
    REAL raX[ 5 ], raY[ 5 ], raZ[ 5 ];
    REAL rZ0, rZ1;
#define POINT( n, i, j ) raX[ n ] = i; raY[ n ] = j; raZ[ n ] = raMap[ i ][ j ]

    SetRast( pwinMain->RPort, 0 );
    RefreshWindowFrame( pwinMain );
    SetDrMd( pwinMain->RPort, JAM1 );

    /* Speed optimization: start by drawing a rectangle of blue ocean. */
    SetAPen( pwinMain->RPort, laPens[ 16 ] );
    if( fBorder )
        SetOPen( pwinMain->RPort, laPens[ 0 ] );
    Transform( 0, 0, 0, &wX, &wY );
    AreaMove( pwinMain->RPort, wX, wY );
    Transform( 0, MAPSIZE, 0, &wX, &wY );
    AreaDraw( pwinMain->RPort, wX, wY );
    Transform( MAPSIZE, MAPSIZE, 0, &wX, &wY );
    AreaDraw( pwinMain->RPort, wX, wY );
    Transform( MAPSIZE, 0, 0, &wX, &wY );
    AreaDraw( pwinMain->RPort, wX, wY );
    AreaEnd( pwinMain->RPort );
    BNDRYOFF( pwinMain->RPort );

    /* For the hidden-line drawing to work correctly, we have to start at */
    /* the rear (from the observer's pont of view) of the map. */
    /* Also we may have to swap the order of the x- and y-loops. */
    wSwap = ( prj.o.x * prj.o.x > prj.o.y * prj.o.y ) ? 1 : 0;
    wXStep = ( prj.o.x > 0 ) ? 1 : -1;
    wYStep = ( prj.o.y > 0 ) ? 1 : -1;
    wX0 = ( wXStep > 0 ) ? 1 : MAPSIZE;
    wY0 = ( wYStep > 0 ) ? 1 : MAPSIZE;

    wI = wJ = 0;
    for( wI *= (1-wSwap), wJ *= wSwap; wI < MAPSIZE && wJ < MAPSIZE ; wI += wSwap, wJ += (1-wSwap) )
        {
        for( wI *= wSwap, wJ *= (1-wSwap); wI < MAPSIZE && wJ < MAPSIZE ; wI += (1-wSwap), wJ += wSwap )
            {
            if( CheckAbort() )
                goto quit;
            wX = wX0 + wXStep * wI;
            wY = wY0 + wYStep * wJ;
            POINT( 0, wX,   wY   );     /* y ^     1  0   */
            POINT( 1, wX-1, wY   );     /*   |            */
            POINT( 2, wX-1, wY-1 );     /*   +->   2  3   */
            POINT( 3, wX,   wY-1 );     /*     x          */
            rZ0 = rPeak;
            rZ1 = 0;
            for( wIdx = 0; wIdx < 4; wIdx++ )
                {
                if( rZ0 > raZ[ wIdx ] )
                    rZ0 = raZ[ wIdx ];
                if( rZ1 < raZ[ wIdx ] )
                    rZ1 = raZ[ wIdx ];
                }
            /* The fifth point is the center of the current square. */
            raX[ 4 ] = wX - 0.5;
            raY[ 4 ] = wY - 0.5;
            raZ[ 4 ] = rZ0 + (0.25 + 0.5*Random()) * (rZ1 - rZ0);
            for( wIdx = 0; wIdx < 5; wIdx++ )
                Transform( raX[ wIdx ], raY[ wIdx ], raZ[ wIdx ], &waX[ wIdx ], &waY[ wIdx ] );
            /* Paint the surface of this square, as four triangles. */
            for( wIdx = 0; wIdx < 4; wIdx++ )
                {
                wPen = GetShadePen( raZ[ 4 ], raZ[ wIdx ], raZ[ (wIdx+1) % 4 ],
                            rSnowLine );
                if( wPen == 16 )
                    continue;   /* optimization for the ocean, see above */
                SetAPen( pwinMain->RPort, laPens[ wPen ] );
                AreaMove( pwinMain->RPort, waX[ 4 ], waY[ 4 ] );
                AreaDraw( pwinMain->RPort, waX[ wIdx ], waY[ wIdx ] );
                AreaDraw( pwinMain->RPort, waX[ (wIdx+1) % 4 ], waY[ (wIdx+1) % 4 ] );
                AreaEnd( pwinMain->RPort );
                }
            /* Skip the following black border stuff if this was a square */
            /* full of ocean. Even if we have a thin black line around the */
            /* whole map this test is important, because otherwise that line */
            /* tends to look not so thin any more (rather a little lumpy). */
            if( raZ[ 0 ] <= 0 && raZ[ 1 ] <= 0 && raZ[ 2 ] <= 0 && raZ[ 3 ] <= 0 )
                continue;
            /* Close the end of each row with a black vertical wall. */
            /* Note that we can reuse two of the transformed corners of */
            /* the surface that we have just drawn, but which two depends */
            /* on whether we were going backwards or forward. */
            if( wI == MAPSIZE - 1 )
                {
                if( wX == 1 )   /* keep points 1 and 2 */
                    {
                    Transform( raX[ 1 ], raY[ 1 ], 0, &waX[ 0 ], &waY[ 0 ] );
                    Transform( raX[ 2 ], raY[ 2 ], 0, &waX[ 3 ], &waY[ 3 ] );
                    }
                else            /* keep points 0 and 3 */
                    {
                    Transform( raX[ 0 ], raY[ 0 ], 0, &waX[ 1 ], &waY[ 1 ] );
                    Transform( raX[ 3 ], raY[ 3 ], 0, &waX[ 2 ], &waY[ 2 ] );
                    }
                SetAPen( pwinMain->RPort, laPens[ 0 ] );
                AreaMove( pwinMain->RPort, waX[ 0 ], waY[ 0 ] );
                for( wIdx = 1; wIdx < 4; wIdx++ )
                    AreaDraw( pwinMain->RPort, waX[ wIdx ], waY[ wIdx ] );
                AreaEnd( pwinMain->RPort );
                }
            /* The same for the other (i.e. Y-) direction. */
            /* Oh, one more thing: If both if()-blocks are TRUE (which */
            /* happens on the frontmost corner of the map), we will first */
            /* have to recalculate the transformed corners of the surface. */
            if( wI == MAPSIZE - 1 && wJ == MAPSIZE - 1 )
                for( wIdx = 0; wIdx < 4; wIdx++ )
                    Transform( raX[ wIdx ], raY[ wIdx ], raZ[ wIdx ], &waX[ wIdx ], &waY[ wIdx ] );
            if( wJ == MAPSIZE - 1 )
                {
                if( wY == 1 )   /* keep points 2 and 3 */
                    {
                    Transform( raX[ 2 ], raY[ 2 ], 0, &waX[ 1 ], &waY[ 1 ] );
                    Transform( raX[ 3 ], raY[ 3 ], 0, &waX[ 0 ], &waY[ 0 ] );
                    }
                else            /* keep points 0 and 1 */
                    {
                    Transform( raX[ 1 ], raY[ 1 ], 0, &waX[ 2 ], &waY[ 2 ] );
                    Transform( raX[ 0 ], raY[ 0 ], 0, &waX[ 3 ], &waY[ 3 ] );
                    }
                SetAPen( pwinMain->RPort, laPens[ 0 ] );
                AreaMove( pwinMain->RPort, waX[ 0 ], waY[ 0 ] );
                for( wIdx = 1; wIdx < 4; wIdx++ )
                    AreaDraw( pwinMain->RPort, waX[ wIdx ], waY[ wIdx ] );
                AreaEnd( pwinMain->RPort );
                }
            }
        wI *= wSwap;
        wJ *= (1-wSwap);
        }
quit:
    RefreshWindowFrame( pwinMain );
    }



/*------------------------------------------------------------------------*\

  Call this during lengthy operations.

  Will return TRUE if the user has done something that should make us
  abort (which can be ESC, the cursor keys, or any gadget or menu
  action).

\*------------------------------------------------------------------------*/

BOOL CheckAbort()
    {
    struct IntuiMessage *intuiMsg;
    BOOL fResult = FALSE;

    intuiMsg = GT_GetIMsg( pwinMain->UserPort );
    if( intuiMsg == NULL )
        return FALSE;

    switch( intuiMsg->Class )
        {
        case IDCMP_MENUPICK:
        case IDCMP_NEWSIZE:
        case IDCMP_CLOSEWINDOW:
            fResult = TRUE;
            break;
        case IDCMP_GADGETDOWN:
            wHotGadget = ((struct Gadget *)(intuiMsg->IAddress))->GadgetID;
            fResult = TRUE;
            break;
        case IDCMP_GADGETUP:
            wHotGadget = IDC_NONE;
            fResult = TRUE;
            break;
        case IDCMP_VANILLAKEY:
            if( intuiMsg->Code == 27 )
                fResult = TRUE;
            break;
        case IDCMP_RAWKEY:
            switch( intuiMsg->Code )
                {
                case CURSORUP:
                case CURSORDOWN:
                case CURSORLEFT:
                case CURSORRIGHT:
                    fResult = TRUE;
                    break;
                }
            break;
        }
    GT_ReplyIMsg( intuiMsg );

    return fResult;
    }



/*------------------------------------------------------------------------*\

  Processing of input events.

\*------------------------------------------------------------------------*/

BOOL HandleGui()
    {
    REAL rScale = 0.5;
    REAL rR = 2.0, rPhi, rTheta;
    struct IntuiMessage *intuiMsg, imCopy;
    struct MenuItem *item;
    LONG lMenuID;
    BOOL fShift, fAutoDraw = FALSE, fQuit = FALSE;
    WORD wAutoCount = -1;
    WORD wRedraw;       /* 0: don't redraw, 1: sketch, 2: full */
    UWORD wOld = 0;     /* used to determine actual movement */

    MakeMountain( rScale );
    rTheta = (90.0 * piRise.VertPot) / MAXPOT;
    rPhi = (360.0 * piSpin.HorizPot) / MAXPOT;
    wRedraw = 1;
    while( !fQuit )
        {
        if( wRedraw )
            {
            SetObserver( rR, rPhi, rTheta );
            MakeScales( pwinMain->BorderLeft,
                        pwinMain->Width  - pwinMain->BorderRight - 1,
                        pwinMain->BorderTop,
                        pwinMain->Height - pwinMain->BorderBottom - 1 );
            if( wRedraw > 1 )
                {
                DrawMountain( rSnowLine );
                wAutoCount = -1;
                }
            else
                {
                DrawBox();
                if( fAutoDraw )
                    wAutoCount = 10;    /* redraw at 1 sec from now */
                }
            wRedraw = 0;
            }

        intuiMsg = GT_GetIMsg( pwinMain->UserPort );
        if( intuiMsg == NULL )
            {
            Wait( -1 );
            continue;
            }

        /* We've got a message. Reply to it ASAP, then process it. */
        imCopy = *intuiMsg;
        GT_ReplyIMsg( intuiMsg );
        fShift = imCopy.Qualifier & (IEQUALIFIER_LSHIFT | IEQUALIFIER_RSHIFT);
        switch( imCopy.Class )
            {
            case IDCMP_GADGETDOWN:
                wHotGadget = ((struct Gadget *)(imCopy.IAddress))->GadgetID;
                break;
            case IDCMP_INTUITICKS:
                /* At each tick check the prop gadgets. */
                if( wHotGadget == IDC_HEIGHT )
                    {
                    if( wOld != piRise.VertPot )
                        {
                        wOld = piRise.VertPot;
                        rTheta = (90.0 * wOld) / MAXPOT;
                        wRedraw = 1;
                        }
                    }
                else if( wHotGadget == IDC_ROTATION )
                    {
                    if( wOld != piSpin.HorizPot )
                        {
                        wOld = piSpin.HorizPot;
                        rPhi = (360.0 * wOld) / MAXPOT;
                        wRedraw = 1;
                        }
                    }
                else
                /* If the prop gadgets are free, count down for auto redraw. */
                    {
                    if( wAutoCount == 0 )
                        wRedraw = 2;
                    if( wAutoCount >= 0 )
                        wAutoCount--;
                    }
                break;
            case IDCMP_GADGETUP:
                switch( ((struct Gadget *)(imCopy.IAddress))->GadgetID )
                    {
                    case IDC_ROTATION:
                    case IDC_HEIGHT:
                        rTheta = (90.0 * piRise.VertPot) / MAXPOT;
                        rPhi = (360.0 * piSpin.HorizPot) / MAXPOT;
                        wRedraw = 1;
                        break;
                    case IDC_STOP_GO:
                        wRedraw = 2;
                        break;
                    }
                wHotGadget = IDC_NONE;
                break;
            case IDCMP_VANILLAKEY:
                switch( imCopy.Code )
                    {
                    case 13:
                        wRedraw = 2;
                        break;
                    }
                break;
            case IDCMP_RAWKEY:
                switch( imCopy.Code )
                    {
                    case CURSORUP:
                        rTheta -= fShift ? 30 : 5;
                        wRedraw = 1;
                        break;
                    case CURSORDOWN:
                        rTheta += fShift ? 30 : 5;
                        wRedraw = 1;
                        break;
                    case CURSORLEFT:
                        rPhi -= fShift ? 30 : 5;
                        wRedraw = 1;
                        break;
                    case CURSORRIGHT:
                        rPhi += fShift ? 30 : 5;
                        wRedraw = 1;
                        break;
                    }
                if( rTheta < 0 )
                    rTheta = 0;
                if( rTheta > 90 )
                    rTheta = 90;
                if( rPhi < 0 )
                    rPhi = 0;
                if( rPhi > 360 )
                    rPhi = 360;
                NewModifyProp( &gadRise, pwinMain, NULL,
                    piRise.Flags, 0, (rTheta / 90) * MAXPOT, 0,
                    piRise.VertBody, 1 );
                NewModifyProp( &gadSpin, pwinMain, NULL,
                    piSpin.Flags, (rPhi / 360) * MAXPOT, 0,
                    piSpin.HorizBody, 0, 1 );
                break;
            case IDCMP_MENUPICK:
                while( imCopy.Code != MENUNULL )
                    {
                	item = ItemAddress( pwinMain->MenuStrip, imCopy.Code );
                	lMenuID = (LONG)GTMENUITEM_USERDATA( item );
                    switch( lMenuID )
                        {
                        case IDM_NEW:
                            MakeMountain( rScale );
                            wRedraw = 1;
                            break;
                        case IDM_BORDER:
                            fBorder = item->Flags & CHECKED;
                            break;
                        case IDM_AUTODRAW:
                            fAutoDraw = item->Flags & CHECKED;
                            break;
                        case IDM_DRAW:
                            wRedraw = 2;
                            break;
                        case IDM_QUIT:
                            fQuit = TRUE;
                            break;
                        case IDM_MAX:
                            {
                            struct Screen *scr = pwinMain->WScreen;
                            ChangeWindowBox( pwinMain, 0, scr->BarHeight + 1,
                                scr->Width, scr->Height - scr->BarHeight - 1 );
                            break;
                            }
                        case IDM_CENTER:
                            {
                            struct Screen *scr = pwinMain->WScreen;
                            ChangeWindowBox( pwinMain,
                                scr->Width / 4, scr->Height / 4,
                                scr->Width / 2, scr->Height / 2 );
                            break;
                            }
                        case IDM_ZOOM:
                            ZipWindow( pwinMain );
                            break;
                        }
		            imCopy.Code = item->NextSelect;
                    }
                break;
            case IDCMP_NEWSIZE:
                wRedraw = 1;
                break;
            case IDCMP_CLOSEWINDOW:
                fQuit = TRUE;
                break;
            }
        }
    return TRUE;
    }



/*------------------------------------------------------------------------*\

  Main entry point.

\*------------------------------------------------------------------------*/

int main( int argc, char *argv[] )
    {
    atexit( GetMeOutOfHere );
    if( !StartMeUp() )
        return 10;
    if( argc > 1 )
        fGfxV39 = FALSE;     /* FIXME */
    atexit( DestroyGui );
    if( !BuildGui( TRUE ) )
        return 10;
    HandleGui();
    return 0;
    }

