require "muidefs"
require "muifuncs"
require "muiasl"
require "lfs"

Pages = strarray.new("Autotrace","Potrace")

-- Autotrace options
--background-color <hexadecimal>: the color of the background that should be ignored, for example FFFFFF; default is no background color.
--centerline: trace a character's centerline, rather than its outline.
--color-count <unsigned>: number of colors a color bitmap is reduced to, it does not work on gray scale, allowed are 1..256; default is 0, that means not color reduction is done.
--corner-always-threshold <angle-in-degrees>: if the angle at a pixel is less than this, it is considered a corner, even if it is within `corner-surround' pixels of another corner; default is 60.
--corner-surround <unsigned>: number of pixels on either side of a point to consider when determining if that point is a corner; default is 4.
--corner-threshold <angle-in-degrees>: if a pixel, its predecessor(s), and its successor(s) meet at an angle smaller than this, it's a corner; default is 100.
--despeckle-level <unsigned>: 0..20; default is no despeckling.
--despeckle-tightness <real>: 0.0..8.0; default is 2.0.
--dpi <unsigned>: The dots per inch value in the input image, affects scaling of mif output image
--error-threshold <real>: subdivide fitted curves that are off by more pixels than this; default is 2.0.
--filter-iterations <unsigned>: smooth the curve this many times before fitting; default is 4.
--input-format:  TGA, PBM, PNM, PGM, PPM or BMP.
--line-reversion-threshold <real>: if a spline is closer to a straight line than this, weighted by the square of the curve length, keep it a straight line even if it is a list with curves; default is .01.
--line-threshold <real>: if the spline is not more than this far away from the straight line defined by its endpoints, then output a straight line; default is 1.
--output-file <filename>: write to <filename>
--output-format <format>: use format <format> for the output file eps, ai, p2e, sk, svg, fig, swf, emf, mif, er, dxf, epd, pdf, cgm or dr2d can be used.
AT_ExportList	= strarray.new("eps","ai","p2e","sk","svg","fig","swf","emf","mif","er","dxf","epd","pdf","cgm","dr2d")
--preserve-width: whether to preserve line width prior to thinning.\n\
--remove-adjacent-corners: remove corners that are adjacent.
--tangent-surround <unsigned>: number of points on either side of a point to consider when computing the tangent at that point; default is 3.
--width-weight-factor: weight factor for fitting the line width.


-- mkbitmap options
-- -o, --output <file>  - output to file
-- -i, --invert         - invert the input (undo 'blackboard' effect)
-- -f, --filter <n>     - apply highpass filter with radius n (default 4)
-- -n, --nofilter       - no highpass filtering
-- -s, --scale <n>      - scale by integer factor n (default 2)
-- -1, --linear         - use linear interpolation
-- -3, --cubic          - use cubic interpolation (default)
MK_InterpList	= strarray.new("linear","cubic")
-- -t, --threshold <n>  - set threshold for bilevel conversion (default 0.45)
-- -g, --grey           - no bilevel conversion, output a greymap


--potrace backend options (-b name)
PO_ExportList	= strarray.new("eps","postscript","pdf","svg","pgm","gimppath","xfig")
PO_ExportListExt = strarray.new("eps","ps","pdf","svg","pgm","gimppath","fig")

--potrace Algorithm options
-- -z, --turnpolicy <policy>  - how to resolve ambiguities in path decomposition
PO_PolicyList =   strarray.new("black","white","left","right","minority","majority","random")
-- -t, --turdsize <n>         - suppress speckles of up to this size (default 2)
-- -a, --alphamax <n>         - corner threshold parameter (default 1)
-- -n, --longcurve            - turn off curve optimization
-- -O, --opttolerance <n>     - curve optimization tolerance (default 0.2)
-- -u, --unit <n>             - quantize output to 1/unit pixels (default 10)
-- -d, --debug <n>            - produce debugging output of type n (n=1,2,3)

--potrace Scaling and placement options
-- -W, --width <dim>          - width of output image
-- -H, --height <dim>         - height of output image
-- -r, --resolution <n>[x<n>] - resolution (in dpi)
-- -x, --scale <n>[x<n>]      - scaling factor (pgm backend)
-- -S, --stretch <n>          - yresolution/xresolution
-- -A, --rotate <angle>       - rotate counterclockwise by angle
-- -M, --margin <dim>         - margin
-- -L, --leftmargin <dim>     - left margin
-- -R, --rightmargin <dim>    - right margin
-- -T, --topmargin <dim>      - top margin
-- -B, --bottommargin <dim>   - bottom margin

--potrace Output options
-- -C, --color #rrggbb        - set line color (default black)
-- --fillcolor #rrggbb        - set fill color (default transparent)
-- --opaque                   - make white shapes opaque
-- --group                    - group related paths together
  
--potrace Postscript/EPS options
-- -P, --pagesize <format>    - page size (default is "DEFAULT_PAPERFORMAT")
-- -c, --cleartext            - do not compress the output
-- -2, --level2               - use postscript level 2 compression (default)
-- -3, --level3               - use postscript level 3 compression
-- -q, --longcoding           - do not optimize for file size
  
--potrace PGM options
-- -G, --gamma <n>            - gamma value for anti-aliasing (default 2.2)

--potrace Frontend options
-- -k, --blacklevel <n>       - black/white cutoff in input file (default 0.5)
-- -i, --invert               - invert bitmap

UnitList	= strarray.new("cm","in","pt")

INFILE_ID		= 10
				   
PO_EXPORT_ID 	= 100
PO_PREVIEW_ID	= 110

AT_EXPORT_ID 	= 200
AT_PREVIEW_ID	= 210

PR_CLOSE_ID		= 300


function Get_File_Path(file)
-- returns separately File and Path from file string
  local l = string.len(file)
  local pos = string.find(string.reverse(file),'/')
  if pos==nil then
	pos = string.find(string.reverse(file),':')
  end
  if pos==nil then
    return "",file
  else
    return string.sub(file,1,l-pos+1),string.sub(file,-pos+1) 
  end
end

function basename(file)
	local l = string.len(file)
	local pos = string.find(file,'%.')
	if pos==nil then
		return file
	else
		return string.sub(file,1,pos-1)
	end
end

function AddCmd(command, opt, val)
	if (val == "") then
		return command
	elseif (val == nil) then
		return command
	else
		return (command .. " " .. opt .. " " .. val)
	end
end

function CheckMark(label, state)
  local obj = mui.make(mui.MUIO_Checkmark, label)
  obj:setcheckmark(state)
  return obj
end

function GenerateAutotraceCommandLine()
	-- Initialise command string
	local command = "autotrace/autotrace"
	return command
end

function GenerateMkbitmapCommandLine()
	-- Initialise command string
	local command = "potrace/mkbitmap"
	
	-- Add options
	if (mui.getbool(MK_invert,mui.MUIA_Selected)) then
		command = command .. " -i"
	end
	command = AddCmd(command,"-s",mui.getint(MK_scale,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-t",mui.getint(MK_threshold,mui.MUIA_Numeric_Value)/255)
	if (mui.getint(MK_filter,mui.MUIA_Numeric_Value) > 0) then
		command = AddCmd(command,"-f",mui.getint(MK_filter,mui.MUIA_Numeric_Value))
	else
		command = command .. " -n"
	end
	if (mui.getint(MK_interp,mui.MUIA_Cycle_Active) == 0) then
		command = command .. " -1"
	else
		command = command .. " -3"
	end
	
	command = AddCmd(command,"-o","T:mkbitmaptmp.pbm")
	
	-- Add Input file name
	command = AddCmd(command,"",mui.getstr(InputFile, mui.MUIA_String_Contents))
	
	return command
end

function GeneratePotraceCommandLine()
	-- Initialise command string
	local command = "potrace/potrace"

	-- Add export format
	command = AddCmd(command,"-b",strarray.get(PO_ExportList,mui.getint(PO_Format, mui.MUIA_List_Active)+1))

	-- Get export filename
	if (exportfile == nil) then
		ipath, ifile = Get_File_Path(mui.getstr(InputFile, mui.MUIA_String_Contents))
		exportfile = (basename(ifile) .. "." .. strarray.get(PO_ExportListExt,mui.getint(PO_Format, mui.MUIA_List_Active)+1))
		if (OutputDir~=nil) then
			ipath = OutputDir
		else
			ipath = lfs.currentdir ()
		end
		exportfile = mui.filerequest(	mui.ASLFR_PubScreenName, scr,
										mui.ASLFR_TitleText,       "Save as...",
										mui.ASLFR_InitialDrawer,   ipath,
										mui.ASLFR_InitialFile,     exportfile)
	end

	-- Add Export file name
	command = AddCmd(command,"-o",exportfile)
	
	-- Add Options
	command = AddCmd(command,"-t",mui.getint(PO_Turdsize,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-a",mui.getint(PO_AlphaMax,mui.MUIA_Numeric_Value)/10)
	command = AddCmd(command,"-O",mui.getint(PO_Tolerance,mui.MUIA_Numeric_Value)/100)
	command = AddCmd(command,"-u",mui.getint(PO_Unit,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-z",strarray.get(PO_PolicyList,mui.getint(PO_Policy,mui.MUIA_Cycle_Active)+1))
	if (mui.getbool(PO_LongCurve,mui.MUIA_Selected)) then
		command = command .. " -n"
	end
	
	-- Add Input file name
	command = AddCmd(command,"","T:mkbitmaptmp.pbm")
	
	return command
end

function GenerateATCommandLine()
	-- Initialise command string
	local command = "autotrace/autotrace"

	-- Add export format
	command = AddCmd(command,"-output-format",strarray.get(AT_ExportList,mui.getint(AT_Format, mui.MUIA_List_Active)+1))

	-- Get export filename
	if (exportfile == nil) then
		ipath, ifile = Get_File_Path(mui.getstr(InputFile, mui.MUIA_String_Contents))
		exportfile = (basename(ifile) .. "." .. strarray.get(PO_ExportListExt,mui.getint(PO_Format, mui.MUIA_List_Active)+1))
		if (OutputDir~=nil) then
			ipath = OutputDir
		else
			ipath = lfs.currentdir ()
		end
		exportfile = mui.filerequest(	mui.ASLFR_PubScreenName, scr,
										mui.ASLFR_TitleText,       "Save as...",
										mui.ASLFR_InitialDrawer,   ipath,
										mui.ASLFR_InitialFile,     exportfile)
	end

	-- Add Export file name
	command = AddCmd(command,"-output-file",exportfile)
	
	-- Add Options
	if (mui.getbool(AT_centerline,mui.MUIA_Selected)) then
		command = command .. " -centerline"
	end
	if (mui.getbool(AT_preserve_width,mui.MUIA_Selected)) then
		command = command .. " -preserve-width"
	end
	if (mui.getbool(AT_remove_adjacent_corners,mui.MUIA_Selected)) then
		command = command .. " -remove-adjacent-corners"
	end
	command = AddCmd(command,"-color-count",mui.getint(AT_colorcount,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-corner-always-threshold",mui.getint(AT_corner_always_threshold,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-corner-surround",mui.getint(AT_corner_surround,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-corner-threshold",mui.getint(AT_corner_threshold,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-despeckle-level",mui.getint(AT_despeckle_level,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-despeckle-tightness",mui.getint(AT_despeckle_tightness,mui.MUIA_Numeric_Value)/10)
	command = AddCmd(command,"-dpi",mui.getint(AT_dpi,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-error-threshold",mui.getint(AT_error_threshold,mui.MUIA_Numeric_Value)/10)
	command = AddCmd(command,"-filter-iterations",mui.getint(AT_filter_iterations,mui.MUIA_Numeric_Value))
	command = AddCmd(command,"-line-reversion-threshold",mui.getint(AT_line_reversion_threshold,mui.MUIA_Numeric_Value)/1000)
	command = AddCmd(command,"-line-threshold",mui.getint(AT_line_threshold,mui.MUIA_Numeric_Value)/10)
	command = AddCmd(command,"-tangent-surround",mui.getint(AT_tangent_surround,mui.MUIA_Numeric_Value))
	if (mui.getint(AT_width_weight_factor,mui.MUIA_Numeric_Value)>0) then
		command = AddCmd(command,"-width-weight-factor",mui.getint(AT_width_weight_factor,mui.MUIA_Numeric_Value))
	end
	
	if (mui.getbool(AT_UseBackground,mui.MUIA_Selected)) then
		R = string.format ("%08X",mui.getint(AT_background, mui.MUIA_Coloradjust_Red))
		G = string.format ("%08X",mui.getint(AT_background, mui.MUIA_Coloradjust_Green))
		B = string.format ("%08X",mui.getint(AT_background, mui.MUIA_Coloradjust_Blue))
		RGB = string.sub(R,1,2) .. string.sub(G,1,2) .. string.sub(B,1,2)
		command = AddCmd(command,"-background",RGB)
	end
	
	-- Add Input file name
	command = AddCmd(command,"",mui.getstr(InputFile, mui.MUIA_String_Contents))
	
	return command
end

function Slider(min, max, value)
  return mui.SliderObject(
    mui.MUIA_Numeric_Min, min,
    mui.MUIA_Numeric_Max, max,
    mui.MUIA_Numeric_Value, value)
end

function NumBut(min, max, value)
  return mui.NumericbuttonObject(
    mui.MUIA_Numeric_Min, min,
    mui.MUIA_Numeric_Max, max,
    mui.MUIA_Numeric_Value, value)
end

function main()

	exportfile = nil
	scr = "Workbench"
	
	-- get input argument
	if (arg ~= nil) then
		i = 1;
		while (arg[i] ~= nil) do
			if (arg[i] == "-o") then
				if (arg[i+1] ~= nil) then 
					exportfile = arg[i+1] 
					i = i+2
				end
			
			elseif (arg[i] == "-screen") then
				if (arg[i+1] ~= nil) then 
					scr = arg[i+1] 
					i = i+2
				end
				
			else
				mui.set(InputFile, mui.MUIA_String_Contents,  arg[i])
				i = i+1;
			end
		end
	end

	
	-- autotrace options
	AT_background				= mui.ColoradjustObject()
	AT_UseBackground			= CheckMark("", false)
	AT_centerline				= CheckMark("", false)
	AT_colorcount				= NumBut(0,256,0)
	AT_corner_always_threshold 	= Slider(0,180,60)
	AT_corner_surround			= Slider(1,20,4)
	AT_corner_threshold			= Slider(0,180,100)
	AT_despeckle_level			= Slider(0,20,0)
	AT_despeckle_tightness		= Slider(0,80,20) -- x10
	AT_dpi						= NumBut(0,600,80)
	AT_error_threshold			= Slider(0,200,20) -- x10
	AT_filter_iterations		= Slider(0,10,4)
	AT_line_reversion_threshold	= Slider(0,100,10) -- x1000
	AT_line_threshold			= Slider(0,100,10) -- x10
	AT_Format 					= mui.ListObject()
	AT_preserve_width			= CheckMark("", false)
	AT_remove_adjacent_corners	= CheckMark("", false)
	AT_tangent_surround			= Slider(0,20,3)
	AT_width_weight_factor		= Slider(0,100,0)

	-- potrace options
	PO_Format 		= mui.ListObject()
	PO_Policy		= mui.CycleObject(	mui.MUIA_Cycle_Entries, PO_PolicyList,
										mui.MUIA_Cycle_Active, 4)
	PO_Turdsize		= Slider(1,20,2)
	PO_AlphaMax		= Slider(0,100,10) -- x10
	PO_LongCurve	= CheckMark("", false)
	PO_Tolerance	= Slider(0,100,20) -- x100
	PO_Unit			= Slider(1,100,10)
	
	-- mkbimap options
	MK_invert		= CheckMark("", false)
	MK_scale		= Slider(1,10,2)
	MK_threshold	= Slider(0,255,115) -- x255
	MK_filter		= Slider(0,100,4)
	MK_interp		= mui.CycleObject(	mui.MUIA_Cycle_Entries, MK_InterpList,
									mui.MUIA_Cycle_Active, 1)
			
	-- general options
	InputFile	= mui.StringObject( mui.MUIA_Frame, mui.MUIV_Frame_String,
									mui.MUIA_Weight, 400,
									mui.MUIA_String_Contents, "",
									mui.MUIA_String_MaxLen, 256)
	
	-- buttons
	PO_EXPORT 	= mui.SimpleButton("_Start")
	PO_PREVIEW = mui.SimpleButton("Preview")
	PO_CANCEL 	= mui.SimpleButton("_Cancel")
	
	AT_EXPORT 	= mui.SimpleButton("_Start")
	AT_PREVIEW = mui.SimpleButton("Preview")
	AT_CANCEL 	= mui.SimpleButton("_Cancel")
	
	INFILE  = mui.SimpleButton("Select")
	
	CLOSE	= mui.SimpleButton("Close")
	
	PreviewImage = mui.ImageObject()

	previewwindow = mui.WindowObject(
		mui.MUIA_Window_Title, "RAS2VEC Preview",
		mui.MUIA_Window_PublicScreen, scr,

		mui.WindowContents, mui.VGroup(
			mui.MUIA_Group_SameWidth, true,
			mui.MUIA_Frame, mui.MUIV_Frame_Group,
			
			mui.Child, PreviewImage,
			
			mui.Child, mui.HGroup(
				mui.MUIA_Frame, mui.MUIV_Frame_Group,
				mui.Child, mui.RectangleObject(),
				mui.Child, CLOSE
			)
		)
	)
	
	mainwindow = mui.WindowObject(
		mui.MUIA_Window_Title, "Raster 2 Vector",
		mui.MUIA_Window_PublicScreen, scr,
		mui.MUIA_Window_ID   , mui.makeid("POTG"),
		mui.MUIA_Window_Width , mui.MUIV_Window_Width_Screen(40),
		mui.MUIA_Window_Height, mui.MUIV_Window_Height_Screen(60),
		mui.MUIA_Window_LeftEdge, 0,
		mui.MUIA_Window_TopEdge, mui.MUIV_Window_TopEdge_Delta(0),
		mui.WindowContents, mui.VGroup(
			mui.MUIA_Group_SameWidth, true,
			
			mui.Child, mui.HGroup(
				mui.MUIA_Frame, mui.MUIV_Frame_Group,
				mui.MUIA_FrameTitle, "Input File",
				mui.Child, InputFile,
				mui.Child, mui.RectangleObject(),
				mui.Child, INFILE
			),
			
			mui.Child, mui.VSpace(2),						

			mui.Child, mui.RegisterGroup(
				Pages,
				mui.MUIA_Register_Frame, true,
				mui.MUIA_Group_SameSize, true,
				
				-- Autotrace options
				mui.Child, mui.VGroup(
					mui.Child, mui.VGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "Input/Output",
						
						mui.Child, mui.HGroup(
						
							mui.Child, mui.VGroup(
								mui.MUIA_Weight, 200,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
														
								mui.Child, mui.VGroup(							
									mui.Child, AT_background
								),
								
								mui.Child, mui.ColGroup(3,
									mui.Child, mui.Label("Use Backgroung"),
									mui.Child, mui.RectangleObject(),
									mui.Child, AT_UseBackground
								)
							),
							
							mui.Child, mui.VGroup(
								mui.MUIA_Weight, 100,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.Child, mui.ListviewObject(
									mui.MUIA_Listview_List,AT_Format
								)
							)
						),
						
						mui.Child, mui.ColGroup(5,
							mui.Child, mui.Label("Reduce color count"), 
							mui.Child, AT_colorcount,
							mui.Child, mui.RectangleObject(),
							mui.Child, mui.Label("DPI"), 
							mui.Child, AT_dpi
						)
					),

					mui.Child, mui.ColGroup(2,
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "Curves settings",
						mui.Child, mui.Label("Error Threshold"), mui.Child, AT_error_threshold,
						mui.Child, mui.Label("Filter Iteration"), mui.Child, AT_filter_iterations,
						mui.Child, mui.Label("Tangent Surround"), mui.Child, AT_tangent_surround			
					),	

					mui.Child, mui.VGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "Lines settings",
						mui.Child, mui.ColGroup(5,
							mui.Child, mui.Label("Preserve Width"),
							mui.Child, AT_preserve_width,
							mui.Child, mui.RectangleObject(),
							mui.Child, mui.Label("Centerline"),
							mui.Child, AT_centerline
						),
						mui.Child, mui.ColGroup(2,
							mui.Child, mui.Label("Reversion Threshold"), mui.Child, AT_line_reversion_threshold,		
							mui.Child, mui.Label("Threshold"), mui.Child, AT_line_threshold,
							mui.Child, mui.Label("Width/Weidth factor"), mui.Child, AT_width_weight_factor
						)
					),
					mui.Child, mui.VGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "Corner settings",
						mui.Child, mui.ColGroup(2,
							mui.Child, mui.Label("Threshold"), mui.Child, AT_corner_threshold,
							mui.Child, mui.Label("Always Threshold"), mui.Child, AT_corner_always_threshold,
							mui.Child, mui.Label("Surround"), mui.Child, AT_corner_surround
						),
						mui.Child, mui.ColGroup(3,
							mui.Child, mui.Label("Remove Adj. Corners"), mui.Child, mui.RectangleObject(), mui.Child, AT_remove_adjacent_corners
						)
					),					

					mui.Child, mui.ColGroup(2,
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "Despeckle",
						mui.Child, mui.Label("Level"), mui.Child, AT_despeckle_level,
						mui.Child, mui.Label("Tightness"), mui.Child, AT_despeckle_tightness
					),

					mui.Child, mui.VSpace(2),
			
					mui.Child, mui.HGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.Child, mui.RectangleObject(mui.MUIA_Weight, 20),
						mui.Child, AT_EXPORT,
						mui.Child, mui.RectangleObject(),
						mui.Child, AT_PREVIEW,
						mui.Child, mui.RectangleObject(),
						mui.Child, AT_CANCEL,
						mui.Child, mui.RectangleObject(mui.MUIA_Weight, 20)
					)					
				),
				
				mui.Child, mui.VGroup(
					-- MK Bitmap options
					mui.Child, mui.VGroup(
						mui.MUIA_Group_SameWidth, true,
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "MK Bitmap",
						
						mui.Child, mui.ColGroup(2,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "High Pass filter",
							mui.Child, mui.Label("Radius"), mui.Child, MK_filter
						),

						mui.Child, mui.ColGroup(4,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "Interpolation",
							mui.Child, mui.Label("Type"), mui.Child, mui.RectangleObject(), mui.Child, MK_interp, mui.Child, mui.RectangleObject()
						),
						
						mui.Child, mui.ColGroup(2,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "Scale output",
							mui.Child, mui.Label("Factor"), mui.Child, MK_scale
						),

						mui.Child, mui.ColGroup(3,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "Negative",
							mui.Child, mui.Label("Invert"), mui.Child, mui.RectangleObject(), mui.Child, MK_invert
						),

						mui.Child, mui.ColGroup(2,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "Threshold",
							mui.Child, mui.Label("Value"), mui.Child, MK_threshold
						)						

					),

					-- Potrace options
					mui.Child, mui.HGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.MUIA_FrameTitle, "POTRACE options",
						mui.Child, mui.VGroup(
							mui.MUIA_Group_SameWidth, true,
							mui.MUIA_Frame, mui.MUIV_Frame_Group,
							mui.MUIA_FrameTitle, "Export Format",
							mui.Child, mui.ListviewObject(
								mui.MUIA_Weight, 100,
								mui.MUIA_Listview_List,PO_Format
							)
						),
						
						mui.Child, mui.VGroup(
							mui.MUIA_Weight, 200,
							mui.Child, mui.ColGroup(4,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "Policy",
								mui.Child, mui.Label("Type"), mui.Child, mui.RectangleObject(), mui.Child, PO_Policy, mui.Child, mui.RectangleObject()
							),
							
							mui.Child, mui.ColGroup(2,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "TurdSize",
								mui.Child, mui.Label("Value"), mui.Child, PO_Turdsize
							),

							mui.Child, mui.ColGroup(2,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "Alpha Max",
								mui.Child, mui.Label("Value"), mui.Child, PO_AlphaMax
							),				

							mui.Child, mui.ColGroup(3,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "Long Curve",
								mui.Child, mui.Label("No optimisation"), mui.Child, mui.RectangleObject(), mui.Child, PO_LongCurve
							),				

							mui.Child, mui.ColGroup(2,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "Curve Optimisation Tolerance",
								mui.Child, mui.Label("Value"), mui.Child, PO_Tolerance
							),				

							mui.Child, mui.ColGroup(2,
								mui.MUIA_Frame, mui.MUIV_Frame_Group,
								mui.MUIA_FrameTitle, "Output Scale",
								mui.Child, mui.Label("Pt (1/72in) / Pixel"), mui.Child, PO_Unit
							)
						)
					),
					
					mui.Child, mui.VSpace(2),
			
					mui.Child, mui.HGroup(
						mui.MUIA_Frame, mui.MUIV_Frame_Group,
						mui.Child, mui.RectangleObject(mui.MUIA_Weight, 20),
						mui.Child, PO_EXPORT,
						mui.Child, mui.RectangleObject(),
						mui.Child, PO_PREVIEW,
						mui.Child, mui.RectangleObject(),
						mui.Child, PO_CANCEL,
						mui.Child, mui.RectangleObject(mui.MUIA_Weight, 20)
					)
				)
			)
		)
	)
	
	app = mui.ApplicationObject(
		mui.MUIA_Application_Title      , "RAS2VEC GUI",
		mui.MUIA_Application_Version    , "$VER: RAS2VEC GUI v1.0 (July 2010)",
		mui.MUIA_Application_Author     , "Yannick Erb",
		mui.MUIA_Application_Description, "RAS2VEC Graphical User Interface",
		mui.MUIA_Application_Base       , "RAS2VECGUI",
		mui.SubWindow					, mainwindow,
		mui.SubWindow					, previewwindow
	)

	assert(app:check(), "Failed to create Application.")

	collectgarbage("collect")
	
	-- set up button actions
	mainwindow:doint(mui.MUIM_Notify, mui.MUIA_Window_CloseRequest, true,
    app, 2, mui.MUIM_Application_ReturnID, mui.MUIV_Application_ReturnID_Quit)

	AT_CANCEL:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, mui.MUIV_Application_ReturnID_Quit)

	AT_EXPORT:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, AT_EXPORT_ID)

	AT_PREVIEW:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, AT_PREVIEW_ID)

	PO_CANCEL:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, mui.MUIV_Application_ReturnID_Quit)

	PO_EXPORT:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, PO_EXPORT_ID)

	PO_PREVIEW:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, PO_PREVIEW_ID)
	
	INFILE:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, INFILE_ID)

	CLOSE:doint(mui.MUIM_Notify, mui.MUIA_Pressed, false,
    app, 2, mui.MUIM_Application_ReturnID, PR_CLOSE_ID)
	
	previewwindow:doint(mui.MUIM_Notify, mui.MUIA_Window_CloseRequest, true,
    app, 2, mui.MUIM_Application_ReturnID, PR_CLOSE_ID)
	
	-- Open Window
	mainwindow:set(mui.MUIA_Window_Open, true)
	
	-- fill in list with export format entries
	PO_Format:doint(mui.MUIM_List_Insert,PO_ExportList,-1,mui.MUIV_List_Insert_Bottom)
	mui.set(PO_Format,mui.MUIA_List_Active,6)

	AT_Format:doint(mui.MUIM_List_Insert,AT_ExportList,-1,mui.MUIV_List_Insert_Bottom)
	mui.set(AT_Format,mui.MUIA_List_Active,5)
	
	running = true

	-- Main loop
	while running do
		id, signals = app:input()
		if id == mui.MUIV_Application_ReturnID_Quit then
			-- Exit GUI
			running = false
			
		elseif id == INFILE_ID then
			ipath, ifile = Get_File_Path(mui.getstr(InputFile, mui.MUIA_String_Contents))
			if ipath=="" then ipath = lfs.currentdir() end
			tmp = mui.filerequest(	mui.ASLFR_PubScreenName,	scr,
									mui.ASLFR_InitialFile,		ifile,
									mui.ASLFR_InitialDrawer,	ipath)             
			if tmp ~= nil then
				mui.set(InputFile, mui.MUIA_String_Contents,  tmp)
			end	
			
		elseif id == AT_PREVIEW_ID then
			if (mui.getstr(InputFile, mui.MUIA_String_Contents) == "") then
				app:request(mainwindow, 0 , "RAS2VEC error", "OK", "Input file is missing")
			else
				previewwindow:set(mui.MUIA_Window_Open, false)
				mui.set(PreviewImage,mui.MUIA_Image_Spec,"5:"..mui.getstr(InputFile, mui.MUIA_String_Contents))
				leftedge = mui.getint(mainwindow, mui.MUIA_Window_LeftEdge) + mui.getint(mainwindow,mui.MUIA_Window_Width) + 20
				topedge = mui.getint(mainwindow, mui.MUIA_Window_TopEdge)
				previewwindow:set(mui.MUIA_Window_LeftEdge, leftedge)
				previewwindow:set(mui.MUIA_Window_TopEdge, topedge)
				previewwindow:set(mui.MUIA_Window_Open, true)	
			end		
		
		elseif id == AT_EXPORT_ID then
			if (mui.getstr(InputFile, mui.MUIA_String_Contents) == "") then
				app:request(mainwindow, 0 , "AutoTrace error", "OK", "Input file is missing")
			else
				-- Generate command Lines
				ATcommand = GenerateATCommandLine()
				
				-- Start external application
				os.execute(ATcommand)
				running = false
			end
			
		elseif id == PO_PREVIEW_ID then
			if (mui.getstr(InputFile, mui.MUIA_String_Contents) == "") then
				app:request(mainwindow, 0 , "Potrace error", "OK", "Input file is missing")
			else
				previewwindow:set(mui.MUIA_Window_Open, false)
				MKcommand = GenerateMkbitmapCommandLine()
				os.execute(MKcommand)
				mui.set(PreviewImage,mui.MUIA_Image_Spec,"5:T:mkbitmaptmp.pbm")
				leftedge = mui.getint(mainwindow, mui.MUIA_Window_LeftEdge) + mui.getint(mainwindow,mui.MUIA_Window_Width) + 20
				topedge = mui.getint(mainwindow, mui.MUIA_Window_TopEdge)
				previewwindow:set(mui.MUIA_Window_LeftEdge, leftedge)
				previewwindow:set(mui.MUIA_Window_TopEdge, topedge)
				previewwindow:set(mui.MUIA_Window_Open, true)	
			end
			
		
		elseif id == PO_EXPORT_ID then
			if (mui.getstr(InputFile, mui.MUIA_String_Contents) == "") then
				app:request(mainwindow, 0 , "Potrace error", "OK", "Input file is missing")
			else
				-- Generate command Lines
				MKcommand = GenerateMkbitmapCommandLine()
				POcommand = GeneratePotraceCommandLine()
						
				-- Start external application
				os.execute(MKcommand)
				os.execute(POcommand)
				running = false
			end
			
		elseif id == PR_CLOSE_ID then
			previewwindow:set(mui.MUIA_Window_Open, false)
		end
      
		if running then mui.wait(signals) end
	end
	
	tmp = io.open("T:mkbitmaptmp.pbm")
	if ( tmp ~= nil) then
		io.close(tmp)
		os.execute("delete T:mkbitmaptmp.pbm QUIET")
	end

end

--main()
_, err = pcall(main)
if err then print("Error: " .. err) end
if app then app:dispose() end
PO_ExportList:dispose()
PO_ExportListExt:dispose()
AT_ExportList:dispose()
MK_InterpList:dispose()
UnitList:dispose()
PO_PolicyList:dispose()
Pages:dispose()
