/*
 * Copyright (C) 2012 Szilard Biro
 * Copyright (C) 2002 Jarmo Laakkonen and Hans-Joerg Frieden
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 * =======================================================================
 *
 * AHI sound driver
 *
 * =======================================================================
 */

#include <exec/exec.h>
#include <devices/ahi.h>
#include <proto/exec.h>
#include <proto/ahi.h>
#include <utility/hooks.h>

#include "../client/header/client.h"
#include "../client/sound/header/local.h"
#include "../client/sound/header/snddll.h"

struct ChannelInfo
{
	struct AHIEffChannelInfo aeci;
	ULONG offset;
};

struct Library *AHIBase = NULL;
static struct MsgPort *AHImp = NULL;
static struct AHIRequest *AHIio = NULL;
static BYTE AHIDevice = -1;
static struct AHIAudioCtrl *actrl = NULL;
static ULONG rc = 1;
static struct ChannelInfo info;

static int speed;
static UBYTE *dmabuf = NULL;
static int buflen;
static dma_t *dmabackend;

#define MINBUFFERSIZE 16384

sndimport_t si;

struct Hook EffHook;

#if defined __AROS__
AROS_UFH3(ULONG, EffFunc,
			AROS_UFHA(struct Hook *, hook, A0),
			AROS_UFHA(struct AHIAudioCtrl *, actrl, A2),
			AROS_UFHA(struct AHIEffChannelInfo *, info, A1))
{
	AROS_USERFUNC_INIT

	hook->h_Data = (APTR)(info->ahieci_Offset[0]);
	return 0;

	AROS_USERFUNC_EXIT
}
#else
ULONG EffectFunc()
{
	struct Hook *hook = (struct Hook *)REG_A0;
	struct AHIEffChannelInfo *info = (struct AHIEffChannelInfo *)REG_A1;

	hook->h_Data = (APTR)(info->ahieci_Offset[0]);
	return 0;
}

static struct EmulLibEntry EffFunc =
{
	TRAP_LIB, 0, (void (*)(void))EffectFunc
};
#endif

void AHISND_Shutdown(void)
{
	if (actrl) {
		info.aeci.ahie_Effect = AHIET_CHANNELINFO | AHIET_CANCEL;
		AHI_SetEffect(&info, actrl);
		AHI_ControlAudio(actrl, AHIC_Play, FALSE, TAG_END);
	}

	if (rc == 0) {
		AHI_UnloadSound(0, actrl);
		rc = 1;
	}

	if (dmabuf) {
		FreeVec(dmabuf);
		dmabuf = NULL;
	}

	if (actrl) {
		AHI_FreeAudio(actrl);
		actrl = NULL;
	}

	if (AHIDevice == 0) {
		CloseDevice((struct IORequest *)AHIio);
		AHIDevice = -1;
	}

	if (AHIio) {
		DeleteIORequest((struct IORequest *)AHIio);
		AHIio = NULL;
	}

	if (AHImp) {
		DeleteMsgPort(AHImp);
		AHImp = NULL;
	}
}

qboolean AHISND_Init(dma_t *dma)
{
	struct AHISampleInfo sample;
	ULONG mixfreq, playsamples;
	UBYTE name[256];
	ULONG mode;
	ULONG type;
	int ahichannels;
	int ahibits;
	cvar_t *cv;

	info.aeci.ahieci_Channels = 1;
	info.aeci.ahieci_Func = &EffHook;
	info.aeci.ahie_Effect = AHIET_CHANNELINFO;
	EffHook.h_Data = 0;
	EffHook.h_Entry = (void *)&EffFunc;

	cv = si.Cvar_Get("s_khz", "44", CVAR_ARCHIVE);
	if (cv->value == 44) 
		speed = 44100;
	else if (cv->value == 22) 
		speed = 22050;
	else if (cv->value == 11) 
		speed = 11025;
	else 
		speed = 44100;

	cv = si.Cvar_Get("sndchannels", "2", CVAR_ARCHIVE);
	if (cv->value == 2) 
		ahichannels = 2;
	else if (cv->value == 1)
		ahichannels = 1;
	else 
		ahichannels = 2;

	cv = si.Cvar_Get("sndbits", "16", CVAR_ARCHIVE);
	if (cv->value == 16)
		ahibits = 16;
	else if (cv->value == 8)
		ahibits = 8;
	else
		ahibits = 16;

	if (ahichannels == 1) {
		if (ahibits == 16) 
			type = AHIST_M16S;
		else
			type = AHIST_M8S;
	} else {
		if (ahibits == 16)
			type = AHIST_S16S;
		else
			type = AHIST_S8S;
	}

	if ((AHImp = CreateMsgPort()) == NULL) {
		AHISND_Shutdown();
		si.Com_Printf("ERROR: Can't create AHI message port\n");
		return false;
	}

	if ((AHIio = (struct AHIRequest *) CreateIORequest(AHImp, sizeof(struct AHIRequest))) == NULL) {
		AHISND_Shutdown();
		si.Com_Printf("ERROR: Can't create AHI io request\n");
		return false;
	}

	AHIio->ahir_Version = 4;

	if ((AHIDevice = OpenDevice("ahi.device", AHI_NO_UNIT, (struct IORequest *)AHIio, 0)) != 0) {
		AHISND_Shutdown();
		si.Com_Printf("Can't open ahi.device version 4\n");
		return false;
	}

	AHIBase = (struct Library *) AHIio->ahir_Std.io_Device;

	if ((actrl = AHI_AllocAudio(AHIA_AudioID, AHI_DEFAULT_ID,
						AHIA_MixFreq, speed,
						AHIA_Channels, 1,
						AHIA_Sounds, 1,
						TAG_END)) == NULL) {
		AHISND_Shutdown();
		si.Com_Printf("Can't allocate audio\n");
		return false;
	}

	AHI_GetAudioAttrs(AHI_INVALID_ID, actrl, AHIDB_MaxPlaySamples, &playsamples,
				AHIDB_BufferLen, 256, AHIDB_Name, (ULONG)&name,
				AHIDB_AudioID, &mode, TAG_END);
	AHI_ControlAudio(actrl, AHIC_MixFreq_Query, &mixfreq, TAG_END);
	buflen = playsamples * speed / mixfreq;

	if (buflen < MINBUFFERSIZE) 
		buflen = MINBUFFERSIZE;

	if ((dmabuf = AllocVec(buflen, MEMF_ANY|MEMF_PUBLIC|MEMF_CLEAR)) == NULL) {
		AHISND_Shutdown();
		si.Com_Printf("Can't allocate AHI dma buffer\n");
		return false;
	}
	
	dmabackend = dma;

	dmabackend->buffer = (unsigned char *)dmabuf;
	dmabackend->channels = ahichannels;
	dmabackend->speed = speed;
	dmabackend->samplebits = ahibits;
	dmabackend->samples = buflen / (dmabackend->samplebits / 8);
	dmabackend->submission_chunk = 1;
	dmabackend->samplepos = 0;

	sample.ahisi_Type = type;
	sample.ahisi_Address = (APTR)dmabuf;
	sample.ahisi_Length = buflen / AHI_SampleFrameSize(type);

	if ((rc = AHI_LoadSound(0, AHIST_DYNAMICSAMPLE, &sample, actrl)) != 0) {
		AHISND_Shutdown();
		si.Com_Printf("Can't load sound\n");
		return false;
	}
 
	if (AHI_ControlAudio(actrl, AHIC_Play, TRUE, TAG_END) != 0) {
		AHISND_Shutdown();
		si.Com_Printf("Can't start playback\n");
		return false;
	}

	si.Com_Printf("AHI audio initialized\n");
	//si.Com_Printf("AHI mode: %s (%08x)\n", name, mode);
	si.Com_Printf("AHI mode: %s\n", name);
	//si.Com_Printf("Output: %ibit %s\n", ahibits, ahichannels == 2 ? "stereo" : "mono");
	//si.Com_Printf("snd_ahi.so\nwritten by Jarmo Laakkonen, Hans-Joerg Frieden and Szilard Biro\n");

	AHI_Play(actrl, AHIP_BeginChannel, 0,
			AHIP_Freq, speed,
			AHIP_Vol, 0x10000,
			AHIP_Pan, 0x8000,
			AHIP_Sound, 0,
			AHIP_EndChannel, NULL,
			TAG_END);
	
	AHI_SetEffect(&info, actrl);
	
	return true;
}

int AHISND_GetDMAPos(void)
{
	return (dmabackend->samplepos = (int)(EffHook.h_Data) * dmabackend->channels);
}

void AHISND_Submit(void)
{
	// unused
}

void AHISND_BeginPainting(void)
{
	// unused
}

/* ========== EXPORTED FUNCTIONS ================================================ */

sndexport_t GetSndAPI(sndimport_t simp)
{
	sndexport_t se;

	si = simp;

	se.api_version		= SND_API_VERSION;
//	se.library_type		= SS_DMA;		
	se.Init		 		= AHISND_Init;
	se.Shutdown			= AHISND_Shutdown;
	se.GetDMAPos		= AHISND_GetDMAPos;
	se.BeginPainting	= AHISND_BeginPainting;
	se.Submit			= AHISND_Submit;

	return se;
}

#ifdef __AROS__
#include <dll.h>

void* dllFindResource(int id, char *pType)
{
	return NULL;
}

void* dllLoadResource(void *pHandle)
{
	return NULL;
}

void dllFreeResource(void *pHandle)
{
	return;
}


dll_tExportSymbol DLL_ExportSymbols[] =
{
	{dllFindResource,"dllFindResource"},
	{dllLoadResource,"dllLoadResource"},
	{dllFreeResource,"dllFreeResource"},
	{GetSndAPI,"GetSndAPI"},
	{0, 0},
};

dll_tImportSymbol DLL_ImportSymbols[] =
{
	{0, 0, 0, 0}
};

int DLL_Init(void)
{
	return 1;
}

void DLL_DeInit(void)
{
}

#endif
