/*
** RW_SDL.C
**
** This file contains ALL Linux specific stuff having to do with the
** software refresh.  When a port is being made the following functions
** must be implemented by the port:
**
** SWimp_EndFrame
** SWimp_Init
** SWimp_InitGraphics
** SWimp_SetPalette
** SWimp_Shutdown
** SWimp_SwitchFullscreen
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "SDL.h"

/* The window icon */
#include "icon/q2icon.xbm"

#include "../ref_soft/r_local.h"

/*****************************************************************************/

//static qboolean                 X11_active = false;

SDL_Surface *surface;

static unsigned int sdl_palettemode;

/*****************************************************************************/

/*
** SWimp_Init
**
** This routine is responsible for initializing the implementation
** specific stuff in a software rendering subsystem.
*/
int
SWimp_Init(void *hInstance, void *wndProc)
{
	if (!SDL_WasInit(SDL_INIT_VIDEO))
	{
		char driverName[64];

		if (SDL_Init(SDL_INIT_VIDEO) == -1)
		{
			ri.Con_Printf(PRINT_ALL, "Couldn't init SDL video: %s.\n",
					SDL_GetError());
			return false;
		}

		SDL_VideoDriverName(driverName, sizeof(driverName) - 1);
		ri.Con_Printf(PRINT_ALL, "SDL video driver is \"%s\".\n", driverName);	
	}

	/*if (SDL_WasInit(SDL_INIT_AUDIO|SDL_INIT_CDROM|SDL_INIT_VIDEO) == 0) {
		if (SDL_Init(SDL_INIT_VIDEO) < 0) {
		  Sys_Error("SDL Init failed: %s\n", SDL_GetError());
		  return false;
		}
	  } else if (SDL_WasInit(SDL_INIT_VIDEO) == 0) {
		if (SDL_InitSubSystem(SDL_INIT_VIDEO) < 0) {
		  Sys_Error("SDL Init failed: %s\n", SDL_GetError());
		  return false;
		}
	  }*/

	return true;
}

/*
 * Sets the window icon
 */
static void
SetSDLIcon()
{
	SDL_Surface *icon;
	SDL_Color color;
	Uint8 *ptr;
	int i;
	int mask;

	icon = SDL_CreateRGBSurface(SDL_SWSURFACE,
			q2icon_width, q2icon_height, 8,
			0, 0, 0, 0);

	if (icon == NULL)
	{
		return;
	}

	SDL_SetColorKey(icon, SDL_SRCCOLORKEY, 0);

	color.r = 255;
	color.g = 255;
	color.b = 255;

	SDL_SetColors(icon, &color, 0, 1);

	color.r = 0;
	color.g = 16;
	color.b = 0;

	SDL_SetColors(icon, &color, 1, 1);

	ptr = (Uint8 *)icon->pixels;

	for (i = 0; i < sizeof(q2icon_bits); i++)
	{
		for (mask = 1; mask != 0x100; mask <<= 1)
		{
			*ptr = (q2icon_bits[i] & mask) ? 1 : 0;
			ptr++;
		}
	}

	SDL_WM_SetIcon(icon, NULL);
	SDL_FreeSurface(icon);
}

/*
** SWimp_InitGraphics
**
** This initializes the software refresh's implementation specific
** graphics subsystem.  In the case of Windows it creates DIB or
** DDRAW surfaces.
**
** The necessary width and height parameters are grabbed from
** vid.width and vid.height.
*/
static qboolean SWimp_InitGraphics( qboolean fullscreen )
{
	const SDL_VideoInfo *vinfo;
	int flags;
	char title[24];	

	if (surface && (surface->w == vid.width) && (surface->h == vid.height))
	{
		/* Are we running fullscreen? */
		int isfullscreen = (surface->flags & SDL_FULLSCREEN) ? 1 : 0;

		/* We should, but we don't */
		if (fullscreen != isfullscreen)
		{
			SDL_WM_ToggleFullScreen(surface);
		}

		/* Do we now? */
		isfullscreen = (surface->flags & SDL_FULLSCREEN) ? 1 : 0;

		if (fullscreen == isfullscreen)
		{
			return true;
		}
	}
	
	/* Is the surface used? */
	if (surface)
	{
		SDL_FreeSurface(surface);
	}
	
	/* Create the window */
	ri.Vid_NewWindow(vid.width, vid.height);

/* 
	Okay, I am going to query SDL for the "best" pixel format.
	If the depth is not 8, use SetPalette with logical pal, 
	else use SetColors.
	
	Hopefully this works all the time.
*/
	vinfo = SDL_GetVideoInfo();
	/*sdl_palettemode = (vinfo->vfmt->BitsPerPixel == 8) ? (SDL_PHYSPAL|SDL_LOGPAL) : SDL_LOGPAL;*/
	sdl_palettemode = SDL_LOGPAL;
//#ifndef __MORPHOS__	
	sdl_palettemode |= (vinfo->vfmt->BitsPerPixel == 8) ? SDL_PHYSPAL : 0;
//#endif	
	flags = /*SDL_DOUBLEBUF|*/SDL_SWSURFACE|SDL_HWPALETTE;

	if (fullscreen)
	{
		flags |= SDL_FULLSCREEN;
	}
	
	SetSDLIcon(); /* currently uses q2icon.xbm data */
	
	if ((surface = SDL_SetVideoMode(vid.width, vid.height, 8, flags)) == NULL)
	{
		Sys_Error("(SOFTSDL) SDL SetVideoMode failed: %s\n", SDL_GetError());
		return false;
	}

	snprintf(title, sizeof(title), "Yamagi Quake II %s", VERSION);	
	SDL_WM_SetCaption(title, title);

	SDL_ShowCursor(0);

	vid.rowbytes = surface->pitch;
	vid.buffer = surface->pixels;

//	X11_active = true;

	return true;
}

/*
** SWimp_EndFrame
**
** This does an implementation specific copy from the backbuffer to the
** front buffer.  In the Win32 case it uses BitBlt or BltFast depending
** on whether we're using DIB sections/GDI or DDRAW.
*/

void SWimp_EndFrame (void)
{
	/* SDL_Flip(surface); */
	SDL_UpdateRect(surface, 0, 0, 0, 0);
}

/*
** SWimp_SetMode
*/
rserr_t
SWimp_SetMode(int *pwidth, int *pheight, int mode, qboolean fullscreen)
{
	ri.Con_Printf(PRINT_ALL, "setting mode %d:", mode);

	/* mode -1 is not in the vid mode table - so we keep the values in pwidth
	   and pheight and don't even try to look up the mode info */
	if ((mode != -1) && !ri.Vid_GetModeInfo(pwidth, pheight, mode))
	{
		ri.Con_Printf(PRINT_ALL, " invalid mode\n");
		return rserr_invalid_mode;
	}

	ri.Con_Printf(PRINT_ALL, " %d %d\n", *pwidth, *pheight);

	if (!SWimp_InitGraphics(fullscreen))
	{
		return rserr_invalid_mode;
	}

	R_GammaCorrectAndSetPalette((const unsigned char *) d_8to24table);

	return rserr_ok;
}

/*
** SWimp_SetPalette
**
** System specific palette setting routine.  A NULL palette means
** to use the existing palette.  The palette is expected to be in
** a padded 4-byte xRGB format.
*/
void
SWimp_SetPalette(const unsigned char *palette)
{
	SDL_Color colors[256];
	
	int i;

	/*if (!X11_active)
		return;*/

	if (!palette)
	{
		palette = (const unsigned char *) sw_state.currentpalette;
	}
 
	for (i = 0; i < 256; i++)
	{
		colors[i].r = palette[i*4+0];
		colors[i].g = palette[i*4+1];
		colors[i].b = palette[i*4+2];
	}

	SDL_SetPalette(surface, sdl_palettemode, colors, 0, 256);
}

/*
** SWimp_Shutdown
**
** System specific graphics subsystem shutdown routine.  Destroys
** DIBs or DDRAW surfaces as appropriate.
*/

void
SWimp_Shutdown(void)
{
	if (surface)
	{
		SDL_FreeSurface(surface);
	}
	
	surface = NULL;
	
	if (SDL_WasInit(SDL_INIT_EVERYTHING) == SDL_INIT_VIDEO)
	{
		SDL_Quit();
	}
	else
	{
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
	}
		
	//X11_active = false;
}

/*
** SWimp_AppActivate
*/
void SWimp_AppActivate( qboolean active )
{
}
