/*
 * Copyright (C) 2012 Szilard Biro
 * Copyright (C) 1997-2001 Id Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * =======================================================================
 *
 * This is the Quake II input system, written for Amiga Intuition.
 *
 * =======================================================================
 */
 
#include "../refresh/header/local.h"
#include "../client/header/keyboard.h"
#include "../unix/header/unix.h"

#include <devices/input.h>
#include <devices/rawkeycodes.h>
#include <intuition/intuition.h>
#include <proto/intuition.h>
#include <proto/exec.h>

#ifdef __MORPHOS__
#include <intuition/intuitionbase.h>
#endif

#define MOUSE_MAX 3000
#define MOUSE_MIN 40

extern struct Window *g_pWindow;
static void *g_pCursor;

static struct Interrupt InputHandler;
static struct MsgPort *inputport;
static struct IOStdReq *inputreq;

static qboolean mouse_grabbed;
static cvar_t *windowed_mouse;
static cvar_t *in_grab;
static int mouse_x, mouse_y;
static int old_mouse_x, old_mouse_y;
/*static ULONG screeninfront;*/

struct
{
	int key;
	int down;
} keyq[128];

int keyq_head = 0;
int keyq_tail = 0;
int mx;
int my;

Key_Event_fp_t Key_Event_fp;

static in_state_t *in_state;
static qboolean mlooking;

static cvar_t *sensitivity;
static cvar_t *exponential_speedup;
static cvar_t *lookstrafe;
static cvar_t *m_side;
static cvar_t *m_yaw;
static cvar_t *m_pitch;
static cvar_t *m_forward;
static cvar_t *freelook;
static cvar_t *m_filter;
static cvar_t *in_mouse;
/*static*/ cvar_t *vid_fullscreen;

#ifdef AROS
static AROS_UFP1(struct InputEvent *, myinputhandler, AROS_UFHA(struct InputEvent *, moo, A0));
#else
static struct EmulLibEntry myinputhandler;
#endif

byte scantokey[128] =
	{
	'`','1','2','3','4','5','6','7',									// 7  -  7 
	'8','9','0','-','=','\\',0,K_KP_INS,								// f  -  15
	'q','w','e','r','t','y','u','i',									// 17 -  23
	'o','p','[',']',0,K_KP_END,K_KP_DOWNARROW,K_KP_PGDN,				// 1f -  31
	'a','s','d','f','g','h','j','k',									// 27 -  39
	'l',';','\'','\\',0,K_KP_LEFTARROW,K_KP_5,K_KP_RIGHTARROW,			// 2f -  47
	'<','z','x','c','v','b','n','m',									// 37 -  55
	',','.','\\',0,K_KP_DEL,K_KP_HOME,K_KP_UPARROW,K_KP_PGUP,			// 3f -  63
	K_SPACE,K_BACKSPACE,K_TAB,K_KP_ENTER,K_ENTER,K_ESCAPE,K_DEL,K_INS,	// 47 -  71
	K_PGUP,K_PGDN,0,0,K_UPARROW,K_DOWNARROW,K_RIGHTARROW,K_LEFTARROW,	// 4f -  79
	K_F1,K_F2,K_F3,K_F4,K_F5,K_F6,K_F7,K_F8,							// 57 -  87
	K_F9,K_F10,0,K_KP_SLASH,K_KP_STAR,K_KP_MINUS,K_KP_PLUS,K_F11,		// 5f -  95
	K_SHIFT,K_SHIFT,0,K_CTRL,K_ALT,K_ALT,0,K_F12,						// 67 -  103
	0,0,0,0,0,0,K_PAUSE,0,												// 6f -  111
	K_HOME,K_END,0,0,0,0,0,0,											// 77 -  119
	0,0,0,0,0,0,0,0														// 7f -  127
	};

void IN_GrabInput(qboolean grab)
{
	if (grab)
	{
	    if (g_pCursor)
			SetPointer(g_pWindow, g_pCursor, 1, 16, 0, 0);
	}
	else
	{	
		ClearPointer(g_pWindow);
	}
}

/*
 * Updates the state of the input queue
 */
void
IN_Update(void)
{
	static int IN_Update_Flag;
    struct IntuiMessage *imsg;

	/* Protection against multiple calls */
	if (IN_Update_Flag == 1)
	{
		return;
	}

	IN_Update_Flag = 1;

    while ((imsg = (struct IntuiMessage *) GetMsg(g_pWindow->UserPort)))
    {
		switch (imsg->Class)
		{
			case IDCMP_CLOSEWINDOW:
				ri.Cmd_ExecuteText(EXEC_NOW, "menu_quit\n");
		}

		ReplyMsg((struct Message *)imsg);
	}

	/* Grab and ungrab the mouse if the
	 * console or the menu is opened */
	if (vid_fullscreen->value)
	{
		if (!mouse_grabbed)
		{
			IN_GrabInput(true);
			mouse_grabbed = true;		
		}
	}
	
	if (in_grab->value == 0)
	{
		if (mouse_grabbed)
		{
			IN_GrabInput(false);
			mouse_grabbed = false;		
		}
	}
	else if (in_grab->value == 1)
	{
		if (!mouse_grabbed)
		{
			IN_GrabInput(true);
			mouse_grabbed = true;
		}
	}
	else
	{
		if (windowed_mouse->value)
		{
			if (!mouse_grabbed)
			{
				IN_GrabInput(true);
				mouse_grabbed = true;
			}
		}
		else
		{
			if (mouse_grabbed)
			{
				IN_GrabInput(false);
				mouse_grabbed = false;
			}
		}
	}

	/* Process the key events */
	while (keyq_head != keyq_tail)
	{
		in_state->Key_Event_fp(keyq[keyq_tail].key, keyq[keyq_tail].down);
		keyq_tail = (keyq_tail + 1) & 127;
	}

	IN_Update_Flag = 0;
}

/*
 * Closes all inputs and clears
 * the input queue.
 */
void
IN_Close(void)
{
	keyq_head = 0;
	keyq_tail = 0;

	memset(keyq, 0, sizeof(keyq));
}

/*
 * Gets the mouse state
 */
void
IN_GetMouseState(int *x, int *y/*, int *state*/)
{
	*x = mx;
	*y = my;
//	*state = mouse_buttonstate;
}

/*
 * Cleares the mouse state
 */
void
IN_ClearMouseState()
{
	mx = my = 0;
}

/*
 * Centers the view
 */
static void
IN_ForceCenterView(void)
{
	in_state->viewangles[PITCH] = 0;
}

/*
 * Look up
 */
static void
IN_MLookDown(void)
{
	mlooking = true;
}

/*
 * Look down
 */
static void
IN_MLookUp(void)
{
	mlooking = false;
	in_state->IN_CenterView_fp();
}

/*
 * Keyboard initialisation. Called
 * by the client via function pointer
 */
void
IN_KeyboardInit(Key_Event_fp_t fp)
{
	Key_Event_fp = fp;
}

/*
 * Initializes the backend
 */
void
IN_BackendInit(in_state_t *in_state_p)
{
	//ri.Con_Printf(PRINT_ALL, "in_backendinit\n");
	in_state = in_state_p;
	m_filter = ri.Cvar_Get("m_filter", "0", CVAR_ARCHIVE);
	in_mouse = ri.Cvar_Get("in_mouse", "0", CVAR_ARCHIVE);

	freelook = ri.Cvar_Get("freelook", "1", 0);
	lookstrafe = ri.Cvar_Get("lookstrafe", "0", 0);
	sensitivity = ri.Cvar_Get("sensitivity", "3", 0);
	exponential_speedup = ri.Cvar_Get("exponential_speedup", "0", CVAR_ARCHIVE);

	m_pitch = ri.Cvar_Get("m_pitch", "0.022", 0);
	m_yaw = ri.Cvar_Get("m_yaw", "0.022", 0);
	m_forward = ri.Cvar_Get("m_forward", "1", 0);
	m_side = ri.Cvar_Get("m_side", "0.8", 0);

	ri.Cmd_AddCommand("+mlook", IN_MLookDown);
	ri.Cmd_AddCommand("-mlook", IN_MLookUp);
	ri.Cmd_AddCommand("force_centerview", IN_ForceCenterView);

	mouse_x = mouse_y = 0.0;
	mouse_grabbed = false;
	
	windowed_mouse = ri.Cvar_Get("windowed_mouse", "1",
			CVAR_USERINFO | CVAR_ARCHIVE);
	in_grab = ri.Cvar_Get("in_grab", "2", CVAR_ARCHIVE);

	vid_fullscreen = ri.Cvar_Get("vid_fullscreen", "0", CVAR_ARCHIVE);
	
	g_pCursor = AllocVec(24, MEMF_ANY | MEMF_CLEAR);	
	
	if ((inputport = CreateMsgPort()))
	{
		if ((inputreq = CreateIORequest(inputport, sizeof(*inputreq))))
		{
			if (!OpenDevice("input.device", 0, (struct IORequest *)inputreq, 0))
			{
				InputHandler.is_Node.ln_Type = NT_INTERRUPT;
				InputHandler.is_Node.ln_Pri = 100;
				InputHandler.is_Node.ln_Name = "Quake2 input handler";
				//id->InputHandler.is_Data = id;
				InputHandler.is_Code = (void (*)())&myinputhandler;
				inputreq->io_Data = (void *)&InputHandler;
				inputreq->io_Command = IND_ADDHANDLER;
				DoIO((struct IORequest *)inputreq);	

				ri.Con_Printf(PRINT_ALL, "Input initialized.\n");

				return;
			}
			DeleteIORequest(inputreq);
		}
		DeleteMsgPort(inputport);
	}
	
	ri.Sys_Error (ERR_FATAL, "Couldn't initialize the input handler");
}

/*
 * Shuts the backend down
 */
void
IN_BackendShutdown(void)
{
	ri.Cmd_RemoveCommand("+mlook");
	ri.Cmd_RemoveCommand("-mlook");
	ri.Cmd_RemoveCommand("force_centerview");

	if (inputreq)
	{
		inputreq->io_Data = (void *)&InputHandler;
		inputreq->io_Command = IND_REMHANDLER;
		DoIO((struct IORequest *)inputreq);

		CloseDevice((struct IORequest *)inputreq);
		DeleteIORequest(inputreq);
	}

	if (inputport)
	{
		DeleteMsgPort(inputport);
	}
	
	if (g_pCursor)
	{
		FreeVec(g_pCursor);
		g_pCursor = 0;
	}

	ri.Con_Printf(PRINT_ALL, "Input shut down.\n");
}

/*
 * Mouse button handling
 */
void
IN_BackendMouseButtons(void)
{
	// handled in the input handler
}

/*
 * Move handling
 */
void
IN_BackendMove(usercmd_t *cmd)
{
	IN_GetMouseState(&mouse_x, &mouse_y);
	
	if (m_filter->value)
	{
		if ((mouse_x > 1) || (mouse_x < -1))
		{
			mouse_x = (mouse_x + old_mouse_x) * 0.5;
		}

		if ((mouse_y > 1) || (mouse_y < -1))
		{
			mouse_y = (mouse_y + old_mouse_y) * 0.5;
		}
	}

	old_mouse_x = mouse_x;
	old_mouse_y = mouse_y;

	if (mouse_x || mouse_y)
	{
		if (!exponential_speedup->value)
		{
			mouse_x *= sensitivity->value;
			mouse_y *= sensitivity->value;
		}
		else
		{
			if ((mouse_x > MOUSE_MIN) || (mouse_y > MOUSE_MIN) ||
				(mouse_x < -MOUSE_MIN) || (mouse_y < -MOUSE_MIN))
			{
				mouse_x = (mouse_x * mouse_x * mouse_x) / 4;
				mouse_y = (mouse_y * mouse_y * mouse_y) / 4;

				if (mouse_x > MOUSE_MAX)
				{
					mouse_x = MOUSE_MAX;
				}
				else if (mouse_x < -MOUSE_MAX)
				{
					mouse_x = -MOUSE_MAX;
				}

				if (mouse_y > MOUSE_MAX)
				{
					mouse_y = MOUSE_MAX;
				}
				else if (mouse_y < -MOUSE_MAX)
				{
					mouse_y = -MOUSE_MAX;
				}
			}
		}

		/* add mouse X/Y movement to cmd */
		if ((*in_state->in_strafe_state & 1) ||
			(lookstrafe->value && mlooking))
		{
			cmd->sidemove += m_side->value * mouse_x;
		}
		else
		{
			in_state->viewangles[YAW] -= m_yaw->value * mouse_x;
		}

		if ((mlooking || freelook->value) &&
			!(*in_state->in_strafe_state & 1))
		{
			in_state->viewangles[PITCH] += m_pitch->value * mouse_y;
		}
		else
		{
			cmd->forwardmove -= m_forward->value * mouse_y;
		}

		IN_ClearMouseState();
	}
}

#ifndef AROS
static struct InputEvent *myinputhandler_real(void);

static struct EmulLibEntry myinputhandler =
{
	TRAP_LIB, 0, (void (*)(void))myinputhandler_real
};
#endif

#ifdef AROS
static AROS_UFH1(struct InputEvent *, myinputhandler, AROS_UFHA(struct InputEvent *, moo, A0))
{
	AROS_USERFUNC_INIT
#else
static struct InputEvent *myinputhandler_real()
{
	struct InputEvent *moo = (struct InputEvent *)REG_A0;
#endif

	struct InputEvent *coin;

	ULONG screeninfront;

	if (!(g_pWindow->Flags & WFLG_WINDOWACTIVE))
		return moo;

	coin = moo;

	if (g_pWindow->WScreen)
	{
#ifdef __MORPHOS__
		if (IntuitionBase->LibNode.lib_Version > 50 || (IntuitionBase->LibNode.lib_Version == 50 && IntuitionBase->LibNode.lib_Revision >= 56))
			GetAttr(SA_Displayed, g_pWindow->WScreen, &screeninfront);
		else
#endif
			screeninfront = (g_pWindow->WScreen == IntuitionBase->FirstScreen);
	}
	else
		screeninfront = 1;

	do
	{
		// mouse wheel and keyboard
		if (coin->ie_Class == IECLASS_RAWKEY)
		{
			switch (coin->ie_Code)
			{
				case RAWKEY_NM_WHEEL_UP:
					//ri.Con_Printf(PRINT_ALL,"wheel up RAW\n");
					keyq[keyq_head].key = K_MWHEELUP;
					keyq[keyq_head].down = true;
					keyq_head = (keyq_head + 1) & 127;
					keyq[keyq_head].key = K_MWHEELUP;
					keyq[keyq_head].down = false;
					keyq_head = (keyq_head + 1) & 127;
					break;
				case RAWKEY_NM_WHEEL_DOWN:
					//ri.Con_Printf(PRINT_ALL,"wheel down RAW\n");
					keyq[keyq_head].key = K_MWHEELDOWN;
					keyq[keyq_head].down = true;
					keyq_head = (keyq_head + 1) & 127;
					keyq[keyq_head].key = K_MWHEELDOWN;
					keyq[keyq_head].down = false;
					keyq_head = (keyq_head + 1) & 127;
					break;
				default:
					if (coin->ie_Code & IECODE_UP_PREFIX)
					{
						//ri.Con_Printf(PRINT_ALL, "Key up - ie_Code %d quake key %d\n", coin->ie_Code & ~IECODE_UP_PREFIX, scantokey[coin->ie_Code & ~IECODE_UP_PREFIX]);
						keyq[keyq_head].key = scantokey[coin->ie_Code & ~IECODE_UP_PREFIX];
						keyq[keyq_head].down = false;
						keyq_head = (keyq_head + 1) & 127;
					}
					else
					{
						//ri.Con_Printf(PRINT_ALL, "Key down - ie_Code %d quake key %d\n", coin->ie_Code, scantokey[coin->ie_Code]);
						keyq[keyq_head].key = scantokey[coin->ie_Code];
						keyq[keyq_head].down = true;
						keyq_head = (keyq_head + 1) & 127;
					}
			}
		}
#ifdef __MORPHOS__		
		if ((coin->ie_Class == IECLASS_NEWMOUSE) && mouse_grabbed && screeninfront)
		{
			switch (coin->ie_Code)
			{
				case NM_WHEEL_UP:
					//ri.Con_Printf(PRINT_ALL,"wheel up NM\n");
					keyq[keyq_head].key = K_MWHEELUP;
					keyq[keyq_head].down = true;
					keyq_head = (keyq_head + 1) & 127;
					keyq[keyq_head].key = K_MWHEELUP;
					keyq[keyq_head].down = false;
					keyq_head = (keyq_head + 1) & 127;
					break;
				case NM_WHEEL_DOWN:
					//ri.Con_Printf(PRINT_ALL,"wheel down NM\n");
					keyq[keyq_head].key = K_MWHEELDOWN;
					keyq[keyq_head].down = true;
					keyq_head = (keyq_head + 1) & 127;
					keyq[keyq_head].key = K_MWHEELDOWN;
					keyq[keyq_head].down = false;
					keyq_head = (keyq_head + 1) & 127;
					break;
				case NM_BUTTON_FOURTH:
					keyq[keyq_head].key = K_MOUSE4;
					keyq[keyq_head].down = true;
					keyq_head = (keyq_head + 1) & 127;
					break;
				case (NM_BUTTON_FOURTH | IECODE_UP_PREFIX):
					keyq[keyq_head].key = K_MOUSE4;
					keyq[keyq_head].down = false;
					keyq_head = (keyq_head + 1) & 127;
					break;
			}
			
			coin->ie_Code = IECODE_NOBUTTON;
		}		
#endif
		if ((coin->ie_Class == IECLASS_RAWMOUSE) && mouse_grabbed && screeninfront)
		{
			if (coin->ie_Code != IECODE_NOBUTTON)
			{
				//mouse buttons
				switch (coin->ie_Code)
				{
					case IECODE_LBUTTON:
						keyq[keyq_head].key = K_MOUSE1;
						keyq[keyq_head].down = true;
						keyq_head = (keyq_head + 1) & 127;
						break;
					case (IECODE_LBUTTON | IECODE_UP_PREFIX):
						keyq[keyq_head].key = K_MOUSE1;
						keyq[keyq_head].down = false;
						keyq_head = (keyq_head + 1) & 127;
						break;
					case IECODE_RBUTTON:
						keyq[keyq_head].key = K_MOUSE2;
						keyq[keyq_head].down = true;
						keyq_head = (keyq_head + 1) & 127;
						break;
					case (IECODE_RBUTTON | IECODE_UP_PREFIX):
						keyq[keyq_head].key = K_MOUSE2;
						keyq[keyq_head].down = false;
						keyq_head = (keyq_head + 1) & 127;
						break;
					case IECODE_MBUTTON:
						keyq[keyq_head].key = K_MOUSE3;
						keyq[keyq_head].down = true;
						keyq_head = (keyq_head + 1) & 127;
						break;
					case (IECODE_MBUTTON | IECODE_UP_PREFIX):
						keyq[keyq_head].key = K_MOUSE3;
						keyq[keyq_head].down = false;
						keyq_head = (keyq_head + 1) & 127;
						break;
				}

				coin->ie_Code = IECODE_NOBUTTON;
			}

			// mouse movement
			mx += coin->ie_position.ie_xy.ie_x;
			my += coin->ie_position.ie_xy.ie_y;

			coin->ie_position.ie_xy.ie_x = 0;
			coin->ie_position.ie_xy.ie_y = 0;
		}
		
		coin = coin->ie_NextEvent;
	} while (coin);

	return moo;
	
#ifdef AROS
	AROS_USERFUNC_EXIT
#endif
}

