
/*
 * FS1541 - packet processor
 *

 * If you are new to this subject and want to learn about it, you *must*
 * have the Amiga Guru Book next to you and compare the routines for each
 * packet with the specifications in the book -- then you will understand!
 *
 * Copyright (C) 1996 - 1999 Michael Krause
 * Copyright (C) 1998        John "Graham" Selck (portions)
 * Copyright (C) 2000 - 2008 Helmut Schoettner (many portions)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <ctype.h>
#include <string.h>

#include <exec/types.h>
#include <exec/execbase.h>
#include <exec/memory.h>
#include <dos/dosextens.h>
#include <dos/datetime.h>
#include <dos/filehandler.h>
#include <clib/alib_stdio_protos.h>

#include <proto/exec.h>
#include <proto/dos.h>
#include <proto/utility.h>

#include "main.h"
#include "packet.h"
#include "support.h"
#include "volume.h"
#include "disk.h"

static BOOL GetFileData (UBYTE entry, struct FHArg1 *fharg1);
static BPTR TryLockFile (LONG *err2, BSTR name, LONG access, struct FHArg1 *fharg1, BOOL allow_root);
LONG Flags2Attrib (BYTE type);
void Volume2DiskAndId (struct BAM *bam, char *buffer);
unsigned char DetermineFileType (char *buffer);
void TodayStamp (struct FileInfoBlock *fib);

static UBYTE fname1[256]; /* 20 */
int inhibited= FALSE;

static LONG TestLock (struct FileLock *fl);
__inline static LONG TestLockRef (struct FileLock *fl);

int trytrace= 0;

/* CAUTION !!! empty_bam is in CBM-ASCII here !!! */

static struct BAM empty_bam=
{
	0x12, 0x01, 0x41, 0x00, 

	{
		0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 
		0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 
		0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 
		0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 0x15FFFF1F, 
		0x15FFFF1F, 0x15FFFF1F, 0x11FCFF07, 0x13FFFF07, 
		0x13FFFF07, 0x13FFFF07, 0x13FFFF07, 0x13FFFF07, 
		0x13FFFF07, 0x12FFFF03, 0x12FFFF03, 0x12FFFF03, 
		0x12FFFF03, 0x12FFFF03, 0x12FFFF03, 0x11FFFF01, 
		0x11FFFF01, 0x11FFFF01, 0x11FFFF01, 0x11FFFF01
	}, 

	{
		0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58, 0x58
	}, 

	{
		0xA0, 0xA0
	}, 
/*
	{
		0x59, 0x59, 0xA0, 0x32, 0x41
	}, 

	replaced by following:
*/

	/* default id= '41'     */
	{               
	  0x34, 0x31
	}, 

	/* separator= shift space */
	{
	  0xA0
	}, 
	
	/* dostype always= '2a' */	
	{
	  0x32, 0x41
	}, 

/*
	{
		0xA0, 0xA0, 0xA0, 0xA0, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	}
	
	replaced by following: 
*/

		{
			0xA0, 0xA0, 0xA0, 0xA0
		}, 
		
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00
		}, 
		
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
		}
};

/*-------------------------------------------------------------------------*/

BOOL DoPackets(void)
{
	BOOL bUpdate;
	static BOOL bWriteD64= FALSE;
	struct DosPacket *packet;

	bUpdate= FALSE;

	while((packet= GetPacket(ourport)))
	{
		LONG error= DOSFALSE;
		LONG err2= 0;
		BOOL f= TRUE;

		SDpSender= packet->dp_Port;

		if (inhibited)
		{
			switch(packet->dp_Type)
			{
				case ACTION_DISK_INFO:
				case ACTION_INHIBIT:
				case ACTION_FORMAT:
					break;

				default:
					f= FALSE;
					err2= ERROR_NOT_A_DOS_DISK;
					break;
			}
		}

		if (f) switch(packet->dp_Type)
		{
			case ACTION_LOCATE_OBJECT:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);
				struct FHArg1 fharg1;

				if ((err2= TestLock(fl)))
				{
					break;
				}
				else if (!curvolumenode)
				{
					err2= disk_inserted ? ERROR_NOT_A_DOS_DISK : ERROR_NO_DISK;
					break;
				}

				/* This FileSystem is easy since we don't have to take
				   care of any FileLock path tracking and resolving */

				trytrace= 198;
				error= TryLockFile(&err2, packet->dp_Arg2, packet->dp_Arg3, &fharg1, TRUE);
				MotorOff();
				break;
			}

			case ACTION_FREE_LOCK:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);

				if (fl)
				{
					FreeLock(fl);
				}

				MotorOff();
				error= DOSTRUE;
				break;
			}

			case ACTION_COPY_DIR:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);

				if (fl)
				{
					if (fl->fl_Access == EXCLUSIVE_LOCK)
					{
						/* Don't duplicate exclusive locks */
						
						err2= ERROR_OBJECT_IN_USE;
					}
					else
					{
						if (fl->fl_Volume == MKBADDR(curdoslist))
						{
							/* Try to duplicate it! */
							
							err2= ERROR_NO_FREE_STORE;
							error= MakeLock(fl->fl_Key, SHARED_LOCK);
						}
						else
						{
							/* The reference disk is currently not inserted. */
							
							err2= ERROR_DEVICE_NOT_MOUNTED;
						}
					}
				}
				else
				{
					if (curvolumenode)
					{
						/* Return shared lock on root directory */
						
						err2= ERROR_NO_FREE_STORE;
						error= MakeLock(0, SHARED_LOCK);
					}
					else
					{
						err2= ERROR_NO_DISK;
					}
				}

				break;
			}

			case ACTION_PARENT:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);

				if (fl)
				{
					if (!(err2= TestLockRef(fl)))
					{
						if (fl->fl_Key != 0)
						{
							/* Return shared lock on root directory */
							
							err2= ERROR_NO_FREE_STORE;
							error= MakeLock(0, SHARED_LOCK);
						}
						else
						{
							err2= 0;
						}
					}
				}
				else
				{
					err2= 0;
				}

				break;
			}

			case ACTION_SAME_LOCK:
			{
				struct FileLock *fl1= BADDR(packet->dp_Arg1);
				struct FileLock *fl2= BADDR(packet->dp_Arg2);

				err2= 0;

				if (fl1 == fl2 || ((fl1->fl_Volume == fl2->fl_Volume) && (fl1->fl_Key == fl2->fl_Key)))
				{
					error= DOSTRUE;
				}

				break;
			}

			case ACTION_EXAMINE_OBJECT:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);
				struct FileInfoBlock *fib= BADDR(packet->dp_Arg2);

				if (fl && fl->fl_Key != 0)
				{
					/* Examine file */
					
					struct FHArg1 fharg1;
					int entry;
					int special;

					fib->fib_DiskKey= 0;
					fib->fib_DirEntryType= fib->fib_EntryType= ST_FILE;
					fib->fib_Protection= FIBF_READ | FIBF_WRITE | FIBF_EXECUTE | FIBF_DELETE;
					fib->fib_Comment[0]= '\0';
					TodayStamp(fib);

					if (fl->fl_Volume != MKBADDR(curdoslist))
					{
						/* The reference disk is currently not inserted. */
						
						err2= ERROR_DEVICE_NOT_MOUNTED;
						break;
					}

					if ((special= fl->fl_Key & FLKEY_SPECIALMASK))
					{
						/* Examine virtual files */
						
						static struct { STRPTR s; ULONG l; } tbl[3]=
						{
							{ VIRTUAL_FILE_D64, D64_SIZE }, 
							{ VIRTUAL_FILE_OPT, 0 }
						};
						
						int index= (special>> FLKEY_SPECIALSHIFT)- 2;

						switch(special)
						{
							case FLKEY_DOLLAR:
								strcpy(&fib->fib_FileName[1], VIRTUAL_FILE_DOLLAR);
								fib->fib_FileName[0]= strlen(VIRTUAL_FILE_DOLLAR);
								fib->fib_Size= curvolumenode->dollarlen;
								fib->fib_Protection &= ~FIBF_READ;
								error= DOSTRUE;
								break;

							case FLKEY_D64: 
								fib->fib_Protection &= ~(FIBF_READ | FIBF_WRITE | FIBF_DELETE);

							case FLKEY_OPT:
								strcpy(&fib->fib_FileName[1], tbl[index].s);
								fib->fib_FileName[0]= strlen(tbl[index].s);
								fib->fib_Size= tbl[index].l;
								fib->fib_Protection &= ~FIBF_READ;
								error= DOSTRUE;
								break;

							case FLKEY_DISKINFO:
								strcpy(&fib->fib_FileName[1], VIRTUAL_FILE_DISKINFO);
								fib->fib_FileName[0]= strlen(VIRTUAL_FILE_DISKINFO);
								fib->fib_Size= diskiconlength;
								fib->fib_Protection &= ~FIBF_READ;
								error= DOSTRUE;
								break;

							default:
								err2= ERROR_OBJECT_NOT_FOUND;
								break;
						}
					}
					else
					{
						for (entry= 0; entry< dirsize; entry++)
						{
							if (directory[entry].type) /*  & 0x07 */
							{
								/* anything but DEL-Files */
							
								if (fl->fl_Key == ((directory[entry].datat<< 8)|directory[entry].datas))
								{
									break;
								}
							}
						}

						if (entry< dirsize)
						{
							if (GetFileData(entry, &fharg1))
							{
								Cbm2Ascii(&fib->fib_FileName[1], directory[entry].name, 16, CHR_LEVEL_ALL);
								fib->fib_FileName[0]= strlen(&fib->fib_FileName[1]);
								fib->fib_Size= fharg1.len;
								fib->fib_Protection= Flags2Attrib(directory[entry].type);
								error= DOSTRUE;
							}
							else
							{
								err2= ERROR_OBJECT_NOT_FOUND;
							}
						}
						else
						{
							err2= ERROR_OBJECT_NOT_FOUND;
						}
					}
				}
				else
				{
					/* Prepare for EXAMINE_NEXT, return volume name in fib_FileName */
					
					fib->fib_DiskKey= 0;
					fib->fib_DirEntryType= fib->fib_EntryType= ST_ROOT;
					strcpy(&fib->fib_FileName[1], &curvolumenode->name[1]);
					fib->fib_FileName[0]= strlen(&fib->fib_FileName[1]);
					fib->fib_Protection= FIBF_READ | FIBF_WRITE | FIBF_EXECUTE | FIBF_DELETE;
					fib->fib_Comment[0]= '\0';
					TodayStamp(fib);
					error= DOSTRUE;
				}

				break;
			}

			case ACTION_EXAMINE_NEXT:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);
				struct FileInfoBlock *fib= BADDR(packet->dp_Arg2);

				if ((err2= TestLock(fl)))
				{
					break;
				}

				fib->fib_DirEntryType= fib->fib_EntryType= ST_FILE;

				err2= ERROR_NO_MORE_ENTRIES;

				while(fib->fib_DiskKey < dirsize)
				{
					int entry;
					struct FHArg1 fharg1;

					entry= fib->fib_DiskKey++;

					if (GetFileData(entry, &fharg1))
					{
						Cbm2Ascii(&fib->fib_FileName[1], directory[entry].name, 16, CHR_LEVEL_ALL);
						fib->fib_FileName[0]= strlen(&fib->fib_FileName[1]);
						fib->fib_Size= fharg1.len;
						fib->fib_NumBlocks= fharg1.numblocks;
						//fib->fib_Protection= FIBF_READ | FIBF_WRITE | FIBF_EXECUTE | FIBF_DELETE;
						fib->fib_Protection= Flags2Attrib(directory[entry].type);
						TodayStamp(fib);
						error= DOSTRUE;				

						break;
					}
				}

				if (error == DOSFALSE)
				{
					MotorOff();
				}

				break;
			}

			case ACTION_FINDINPUT:
			case ACTION_FINDOUTPUT:
			case ACTION_FINDUPDATE:
			{
				struct FileHandle *fh= BADDR(packet->dp_Arg1);
				struct FileLock *fl= BADDR(packet->dp_Arg2);
				struct FHArg1 *fharg1;
				LONG access= (packet->dp_Type==ACTION_FINDINPUT) ? SHARED_LOCK : EXCLUSIVE_LOCK;

				if ((err2= TestLock(fl)))
				{
					break;
				}

				if ((fharg1= AllocVec(sizeof(struct FHArg1), MEMF_ANY|MEMF_CLEAR)))
				{
					trytrace= 490;
					if (TryLockFile(&err2, packet->dp_Arg3, access, fharg1, FALSE))
					{
						if (fharg1->number!= NUM_VIRTUAL || access== SHARED_LOCK
							|| (((struct FileLock*)BADDR(fharg1->lock))->fl_Key & FLKEY_SPECIALMASK) == FLKEY_D64)
						{
							/* Allow virtual files only for reading, exception is $d64 */
							
							fh->fh_Arg1= (LONG)fharg1;
							fh->fh_Port= DOSFALSE;
							error= DOSTRUE;
						}
						else
						{
							/* But new files with virtual file names can be created, 
								which will be real then. */
								
							FreeLock((struct FileLock*)BADDR(fharg1->lock));
							err2= ERROR_OBJECT_NOT_FOUND;
						}
					}

					if ((error == DOSFALSE && access == EXCLUSIVE_LOCK && err2 != ERROR_INVALID_COMPONENT_NAME) || packet->dp_Type == ACTION_FINDOUTPUT)
					{
						if (!iWriteProtect)
						{
							/* We only write to disks with valid cbm-dos type */
							
							if (!(Flags2Attrib(directory[fharg1->number].type | 0x80) & FIBF_WRITE))
							{
								/* We only write to non protected files */
								
								if (error == DOSFALSE)
								{
									/* Create new file */
									
									UWORD dblk;
		
									if (dirsize < 144)
									{
										if ((dblk= AllocBlock(17, 0)))
										{
											UBYTE clrblk[256];
		
											memset(&clrblk, 0, sizeof(clrblk));
											clrblk[1]= 1;
		
											if (PutBlockTS(dblk>> 8, dblk& 0xff, &clrblk))
											{
												struct DirEntry *de= &directory[dirsize++];
		
												memset(de, 0, sizeof(struct DirEntry));
												/* de->type= 0x82; *//* PRG, closed */
												de->type= DetermineFileType(fname1);
												de->datat= dblk>> 8;
												de->datas= dblk& 0xff;
												de->lengthl= 1;
												Ascii2Cbm(de->name, fname1, 16);
												StartUDSTimer();
		
												trytrace= 543;
												if (TryLockFile(&err2, packet->dp_Arg3, access, fharg1, FALSE))
												{
													fh->fh_Arg1= (LONG)fharg1;
													fh->fh_Port= DOSFALSE;
													error= DOSTRUE;
												}
											}
											else
											{
												err2= ABORT_DISK_ERROR;
												FreeBlock(dblk>> 8, dblk& 0xff);
											}
										}
										else
										{
											err2= ERROR_DISK_FULL;
										}
									}
									else
									{
										err2= ERROR_DISK_FULL;
									}
								}
								else if (fharg1->number != NUM_VIRTUAL) /* trying to overwrite real files, but not $d64 */
								{
									/* Overwrite existing file */
									
									struct DataBlock *block;
									UBYTE t, s;
									int i;
		
									for (i= 0, t= fharg1->t, s= fharg1->s; t; t= block->t, s= block->s, i++)
									{
										block= GetBlockTS(t, s);
										if (i> 0) FreeBlock(t, s);
									}
									
									fharg1->len= 0;
									fharg1->numblocks= 1;
									block= GetBlockTS(fharg1->t, fharg1->s);
									block->t= 0;
									block->s= 1;
									PutBlockTS(fharg1->t, fharg1->s, block);			
								}
							}
							else
							{
								err2= ERROR_WRITE_PROTECTED;
							}
						}
						else
						{
							err2= ERROR_DISK_WRITE_PROTECTED;
						}
					}
					else
					{
						if (error== DOSTRUE)
						{
							/* Open existing file, already done. */
						}
						
						/* else err2 still set from first TryLockFile() call */
					}

					if (error == DOSFALSE)
					{
						FreeVec(fharg1);
						MotorOff();
					}
				}
				else
				{
					err2= ERROR_NO_FREE_STORE;
				}

				break;
			}

			case ACTION_READ:
			{
				struct FHArg1 *fharg1= (struct FHArg1 *)packet->dp_Arg1;
				/* APTR buffer= (APTR)packet->dp_Arg2; */
				char *buffer= (APTR)packet->dp_Arg2;
				ULONG togo= packet->dp_Arg3;
				struct DataBlock *block= NULL;

				if ((err2= TestLockRef((struct FileLock*)BADDR(fharg1->lock))))
				{
					error= -1;
					break;
				}

				if (fharg1->pos + togo >= fharg1->len)
				{
					togo= fharg1->len - fharg1->pos;
				}

				error= 0;

				if (fharg1->number == NUM_VIRTUAL)
				{
					/* Handle virtual files */
					int special= ((struct FileLock*)BADDR(fharg1->lock))->fl_Key & FLKEY_SPECIALMASK;

					switch(special)
					{
						case FLKEY_DOLLAR:
							CopyMem(&curvolumenode->dollarbuf[fharg1->pos], buffer, togo);
							fharg1->pos += togo;
							error += togo;
							break;

						case FLKEY_DISKINFO:
							CopyMem(&diskiconimg[fharg1->pos], buffer, togo);
							fharg1->pos += togo;
							error += togo;
							break;

						case FLKEY_D64:						
							while(togo)
							{
								ULONG size, offset;

								if (!(block= GetPutBlock((fharg1->pos)>> 8, FALSE)))
									break;

								offset= (fharg1->pos)& 255;

								size= 256- offset;
								if (togo < size)
									size= togo;

								CopyMem(&((UBYTE*)block)[offset], buffer, size);

								buffer += size;
								fharg1->pos += size;
								error += size;
								togo -= size;
							}
							break;

						default:
							break;
					}
				}
				else
				{
					/* Handle real files */
					
					while(togo)
					{
						ULONG size;
	
						if (fharg1->p == 254)
						{
							if (!block)
							{
								if (!(block= GetBlockTS(fharg1->t, fharg1->s)))
								{
									break;
								}
							}
									
							fharg1->p= 0;
							fharg1->t= block->t;
							fharg1->s= block->s;
						}
	
						size= 254- fharg1->p;
						if (togo < size)
						{
							size= togo;
						}
	
						if (!(block= GetBlockTS(fharg1->t, fharg1->s)))
						{
							break;
						}
	
						CopyMem(&block->data[fharg1->p], buffer, size);
						buffer += size;
						fharg1->p += size;
						fharg1->pos += size;
						error += size;
						togo -= size;
					}
				}

				break;
			}

			case ACTION_WRITE:
			{
				struct FHArg1 *fharg1= (struct FHArg1 *)packet->dp_Arg1;
				/* APTR buffer= (APTR)packet->dp_Arg2; */
				char *buffer= (APTR)packet->dp_Arg2;
				ULONG togo= packet->dp_Arg3;
				struct DataBlock *block= NULL;
				struct DataBlock newblk;

				error= -1;

				if ((err2= TestLockRef((struct FileLock*)BADDR(fharg1->lock))))
				{
					break;
				}

				error= 0;

				if ((((struct FileLock*)BADDR(fharg1->lock))->fl_Key & FLKEY_SPECIALMASK) == FLKEY_D64)
				{
					/* Handle writes to the $d64 file */
					
					if (fharg1->pos + togo > D64_SIZE)
					{
						togo= D64_SIZE - fharg1->pos;
					}

					while(togo)
					{
						ULONG size, offset;

						if (!(block= GetPutBlock((fharg1->pos)>> 8, FALSE)))
						{
							break;
						}

						offset= (fharg1->pos)& 255;

						size= 256- offset;
						if (togo < size)
						{
							size= togo;
						}

						CopyMem(buffer, &((UBYTE*)block)[offset], size);
						GetPutBlock((fharg1->pos)>> 8, TRUE);

						buffer += size;
						fharg1->pos += size;
						error += size;
						togo -= size;
					}
					
					/* show write access to $d64 */
								
					bWriteD64= TRUE;		
										
					break;
				}
				else if (iWriteProtect)
				{
					/* iWriteProtect is taken into consideration only for real files */
					
					err2= ERROR_DISK_WRITE_PROTECTED;
					break;
				}
				else if ((Flags2Attrib(directory[fharg1->number].type | 0x80) & FIBF_WRITE))
				{
					/* We only write to non protected files */
					
					err2= ERROR_WRITE_PROTECTED;
					break;
				}

				while(togo)
				{
					ULONG size;

					if (!block)
						block= GetBlockTS(fharg1->t, fharg1->s);

					if (fharg1->p == 254)
					{
						/* Switch to next block */
						
						if (!block->t)
						{
							/* Allocate new block */
							
							UWORD dblk;

							if (!(dblk= AllocBlock(fharg1->t, fharg1->s)))
							{
								err2= ERROR_DISK_FULL;
								break;
							}

							fharg1->p= 0;
							block->t= dblk>> 8;
							block->s= dblk& 0xff;
							PutBlockTS(fharg1->t, fharg1->s, block);	/* link to previous block */
							fharg1->t= block->t;
							fharg1->s= block->s;
							fharg1->numblocks++;

							block= &newblk;
							memset(&newblk, 0, sizeof(newblk));
							block->t= 0;
							block->s= 1;
						}
						else
						{
							/* Jump to existing next block */
							fharg1->p= 0;
							fharg1->t= block->t;
							fharg1->s= block->s;
							block= GetBlockTS(fharg1->t, fharg1->s);
						}
					}

					size= 254 - fharg1->p;
					if (togo < size)
					{
						size= togo;
					}

					/* Insert data */
					
					CopyMem(buffer, &block->data[fharg1->p], size);
					buffer += size;

					/* Advance block pointer and dump it */
					
					fharg1->p += size;
					if (!block->t)
					{
						block->s= fharg1->p+ 1;
					}
					
					PutBlockTS(fharg1->t, fharg1->s, block);

					/* Adjust length field if necessary */
					
					fharg1->pos += size;
					if (fharg1->pos > fharg1->len)
					{
						fharg1->len= fharg1->pos;
					}

					error += size;
					togo -= size;
				}

				break;
			}

			case ACTION_SET_FILE_SIZE:
			{
				error= -1;
				err2= ERROR_ACTION_NOT_KNOWN;
			
				break;
			}

			case ACTION_SEEK:
			{
				struct FHArg1 fharg1bak, *fharg1= (struct FHArg1 *)packet->dp_Arg1;
				LONG move= packet->dp_Arg2;

				if ((err2= TestLockRef((struct FileLock*)BADDR(fharg1->lock))))
				{
					break;
				}

				error= fharg1->pos;
				err2= ERROR_SEEK_ERROR;

				switch(packet->dp_Arg3)
				{
					case OFFSET_BEGINNING:
						move= move- error;
						break;

					case OFFSET_CURRENT:
						break;

					case OFFSET_END:
						move= move+ fharg1->len- error;
						break;

					default:
						error= -1;
						break;
				}

				if (error != -1)
				{
					CopyMem(fharg1, &fharg1bak, sizeof(struct FHArg1));

					if ((error+move)<0 || (error+move)>fharg1->len)
					{
						error= -1;
					}
					else
					{
						if (((struct FileLock*)BADDR(fharg1->lock))->fl_Key & FLKEY_SPECIALMASK)
						{
							/* Virtual files are easy */

							fharg1->pos += move;
						}
						else
						{
							if (move< 0)
							{
								/* When moving back, we have to rescan the whole file */

								move= move+ error;
								fharg1->p= 0;
								fharg1->pos= 0;
								fharg1->t= fharg1->t0;
								fharg1->s= fharg1->s0;
							}
	
							while(move)
							{
								ULONG size;
								struct DataBlock *block;
	
								if (fharg1->p == 254)
								{
									block= GetBlockTS(fharg1->t, fharg1->s);

									if (!block)
									{
										CopyMem(&fharg1bak, fharg1, sizeof(struct FHArg1));
										error= -1;
										err2= ABORT_DISK_ERROR;

										break;
									}

									fharg1->p= 0;
									fharg1->t= block->t;
									fharg1->s= block->s;
								}
	
								size= 254- fharg1->p;
			
								if (move < size)
								{
									size= move;
								}
			
								fharg1->p += size;
								fharg1->pos += size;
								move -= size;
							}
						}
					}
				}

				break;
			}

			case ACTION_END:
			{
				struct FHArg1 *fharg1= (struct FHArg1 *)packet->dp_Arg1;

				if ((((struct FileLock*)BADDR(fharg1->lock))->fl_Key & FLKEY_SPECIALMASK) == FLKEY_D64)
				{
					if (bWriteD64)
					{
						// write to $d64 is complete - force disk change
						
						bWriteD64= FALSE;
						bUpdate= TRUE;
					}
				}
				
				if ((err2= TestLockRef((struct FileLock*)BADDR(fharg1->lock))))
				{
					break;
				}

				if (fharg1->len != fharg1->len0)
				{
					/* File has been modified, update disk structure */

					int i;

					for (i= 0; i< dirsize; i++)
					{
						if ((directory[i].type & 0x0F) && (directory[i].datat == fharg1->t0) && (directory[i].datas == fharg1->s0))
						{
							UWORD blks= fharg1->numblocks;
							directory[i].lengthl= blks& 0xff;
							directory[i].lengthh= blks>> 8;
							
							/* mark file as closed for valid file types */
							
							if (directory[i].type & 0x0F)
							{
								directory[i].type |= 0x80;
							}

							break;
						}
					}

					StartUDSTimer();
				}

				FreeLock(BADDR(fharg1->lock));
				FreeVec(fharg1);

				error= DOSTRUE;
				break;
			}

			case ACTION_IS_FILESYSTEM:
			{
				/* Yes, we are. */

				error= DOSTRUE;
				break;
			}

			case ACTION_CURRENT_VOLUME:
			{
				struct FHArg1 *fharg1= (struct FHArg1 *)packet->dp_Arg1;

				/* The packet name is somewhat misleading... */

				if (fharg1)
				{
					error= ((struct FileLock*)BADDR(fharg1->lock))->fl_Volume;
				}
				else
				{
					if (curvolumenode)
					{
						error= (LONG)MKBADDR(curdoslist);
					}
					else
					{
						error= NULL;
					}
				}

				break;
			}

			case ACTION_INFO:
			case ACTION_DISK_INFO:
			{
				struct InfoData *id;

				if (packet->dp_Type == ACTION_INFO)
				{
					// workbench's info
					
					struct FileLock *fl= BADDR(packet->dp_Arg1);

					if (fl && fl->fl_Volume != MKBADDR(curdoslist))
					{
						/* The reference disk is currently not inserted. */

						err2= ERROR_DEVICE_NOT_MOUNTED;

						break;
					}

					id= BADDR(packet->dp_Arg2);
				}
				else
				{
					// dos info command: ACTION_DISK_INFO
					
					id= BADDR(packet->dp_Arg1);
				}

				id->id_NumSoftErrors= lNumSoftErrors;
				id->id_NumBlocks= 683;
				id->id_NumBlocksUsed= UsedBlocks();
				id->id_BytesPerBlock= 254;
				id->id_DiskState= iHardWriteProtect ? ID_WRITE_PROTECTED : (iWriteProtect ? ID_VALIDATING : ID_VALIDATED);
				id->id_UnitNumber= fssm->fssm_Unit;

				if (curvolumenode)
				{
					id->id_DiskType= ID_CBM_DISK;
					id->id_VolumeNode= MKBADDR(curdoslist);
					id->id_InUse= curvolumenode->locklist ? DOSTRUE : DOSFALSE;
				}
				else
				{
					#ifndef ID_BUSY /* My v40 header files don't */
					#define ID_BUSY 0x42555359
					#endif
					id->id_DiskType= inhibited ? ID_BUSY : (disk_inserted ? ID_UNREADABLE_DISK : ID_NO_DISK_PRESENT);
				}

				error= DOSTRUE;

				break;
			}

			case ACTION_INHIBIT:
			{
				/* We could have trouble here if the DosList is locked, 
				   but then again, one shouldn't lock it for INHIBIT. */

				if (packet->dp_Arg1 == DOSTRUE)
				{
					if (inhibited == FALSE)
					{
						inhibited= TRUE;

						StopUDSTimer();
						MotorOff();
						
						/* remove the disk icon */
						DoDiskRemove(TRUE);
						
						ResetDisk();
					}
				}
				else if (inhibited == TRUE)
				{
					DoDiskInsert(FALSE);

					inhibited= FALSE;
				}

				error= DOSTRUE;

				break;
			}

			case ACTION_CREATE_DIR:
			{
				err2= ERROR_INVALID_COMPONENT_NAME;
				break;
			}

			case ACTION_DELETE_OBJECT:
			{
				struct FileLock *fl= BADDR(packet->dp_Arg1);
				struct FHArg1 fharg1;

				if ((err2= TestLock(fl)))
				{
					break;
				}

				if (!iWriteProtect)
				{
					trytrace= 1181;
					if (TryLockFile(&err2, packet->dp_Arg2, EXCLUSIVE_LOCK, &fharg1, FALSE))
					{
						struct DataBlock *block;
						UBYTE t, s;
	
						FreeLock((struct FileLock*)BADDR(fharg1.lock));
	
						if (fharg1.number != NUM_VIRTUAL)
						{
							if (!(Flags2Attrib(directory[fharg1.number].type) & FIBF_DELETE))
							{
								/* We only delete non protected files */
								/* We do this only for non-virtual files. */
	
								for (t= fharg1.t, s= fharg1.s; t; t= block->t, s= block->s)
								{
									block= GetBlockTS(t, s);
									FreeBlock(t, s);
								}
			
								/* set file type to DEL and open to avoid file is shown with type DEL */
	
								directory[fharg1.number].type= 0x00;
	
								StartUDSTimer();
								error= DOSTRUE;
							}
							else
							{
								err2= ERROR_DELETE_PROTECTED;
							}
						}
						else
						{
							/* fake deletion of virtual $d64 */
							
							if ((((struct FileLock*)BADDR(fharg1.lock))->fl_Key & FLKEY_SPECIALMASK) == FLKEY_D64)
							{
								error= DOSTRUE;
							}
							else
							{
								err2= ERROR_DELETE_PROTECTED;
							}
						}
					}
				}
				else
				{
					err2= ERROR_DISK_WRITE_PROTECTED;
				}

				MotorOff();

				break;
			}

			case ACTION_FORMAT:
			{
				UBYTE buf[16+ 3];
				STRPTR a;
				struct DataBlock blk1;	

				if (!inhibited)
				{
					/* Amiga Guru Book p. 431, 17.1.50 Format():
					   "Before a medium can be high-level formatted, the
					   associated DOS device must have been inhibited..." */

					err2= ERROR_OBJECT_IN_USE;

					break;
				}

				//DoDiskRemove(TRUE);
				ResetDisk();

				if (!iHardWriteProtect)
				{
					int l;
					int k;

					if ((a= BADDR(packet->dp_Arg1)))
					{
						l= a[0]> (16+ 3) ? (16+ 3) : a[0];
						CopyMem(&a[1], buf, l);
						buf[l]= '\0';
					}
					else
					{
					   /* buf[0]= '\0'; */
					   strcpy(&buf[0], "commodore");
					}
					
					/* empty_bam is pre-initialized !! (already cbm-ascii !!)                  */
					
					Volume2DiskAndId(&empty_bam, buf);
					
					/* put the given id in every sector's label */			
					/* rewrite each sector because of the sectors label information */
					
					for (k= 0; k< 683; k++)
					{
						ubDiskIdAr[k][0]= empty_bam.ubDiskIdAr[0];
						ubDiskIdAr[k][1]= empty_bam.ubDiskIdAr[1];
					
						GetPutBlock((ULONG)k, TRUE);
					}
					
					/* all done - write the bam block */

				/*	empty_bam[0].name2=(rand()&0x3F)+32;
					empty_bam[1].name2=(rand()&0x3F)+32; */

					PutBlockTS(18, 0, &empty_bam);
					memset(&blk1, 0, sizeof(blk1));
					blk1.s= 0xff;
					PutBlockTS(18, 1, &blk1);

					error= DOSTRUE;
				}
				else
				{
					err2= ERROR_DISK_WRITE_PROTECTED;
				}

				/* we don't have to DoDiskInsert() here --
				   it's done in the final ACTION_INHIBIT. */

				break;
			}

			case ACTION_RENAME_DISK:
			{
				UBYTE buf[16+ 3];
				STRPTR a;

				if (curvolumenode)
				{
					int l;
					
					if (!iWriteProtect)
					{
						if ((a= BADDR(packet->dp_Arg1)))
						{
							l= a[0]> (16+ 3) ? (16+ 3) : a[0];
							CopyMem(&a[1], buf, l);
							buf[l]= '\0';
						}
						else
						{
							/* buf[0]= '\0'; */
							strcpy(&buf[0], "commodore");
						}

						Volume2DiskAndId(bam, buf);

						while(!AttemptLockDosList(LDF_VOLUMES|LDF_WRITE))
						{
							DoPackets();
						}

						Cbm2Ascii(&curvolumenode->name[1], bam->name, 16, CHR_LEVEL_ALL);
						curvolumenode->name[0]= strlen(&curvolumenode->name[1]);
						UnLockDosList(LDF_VOLUMES|LDF_WRITE);

						StartUDSTimer();

						error= DOSTRUE;
					}
					else
					{
						err2= ERROR_DISK_WRITE_PROTECTED;
					}
				}
				else
				{
					err2= disk_inserted ? ERROR_NOT_A_DOS_DISK : ERROR_NO_DISK;
				}

				break;
			}

			case ACTION_SET_PROTECT:
			{
				struct FileLock *fl1= BADDR(packet->dp_Arg2);
				LONG attr= (packet->dp_Arg4);
				struct FHArg1 fharg1;
				
				if (!((err2= TestLock(fl1))))
				{
					if (!iWriteProtect)
					{					
						if (TryLockFile(&err2, packet->dp_Arg3, SHARED_LOCK, &fharg1, FALSE))						
						{
							FreeLock((struct FileLock*)BADDR(fharg1.lock));
							
							if (fharg1.number != NUM_VIRTUAL)
							{
								/* by default no access permissions - inverted logic */
								
								if ((attr & FIBF_WRITE) || (attr & FIBF_DELETE))
								{
									directory[fharg1.number].type |= 0x40;
								}
								else
								{
									directory[fharg1.number].type &= ~0x40;									
								}
								
								StartUDSTimer();

								error= DOSTRUE;
							}
							else
							{
								err2= ERROR_OBJECT_NOT_FOUND;
							}														
						}
					}
				}
										
				break;
			}
			
			case ACTION_RENAME_OBJECT:
			{
				struct FileLock *fl1= BADDR(packet->dp_Arg1), 
				                *fl2= BADDR(packet->dp_Arg3);
				struct FHArg1 fharg1;

				if (!((err2= TestLock(fl1)) || (err2= TestLock(fl2))))
				{
					if (!iWriteProtect)
					{
						trytrace= 1372;
						if (TryLockFile(&err2, packet->dp_Arg2, SHARED_LOCK, &fharg1, FALSE))
						{
							FreeLock((struct FileLock*)BADDR(fharg1.lock));

							if (fharg1.number != NUM_VIRTUAL)
							{
								STRPTR a;
								UBYTE buf[256];

								if ((a= BADDR(packet->dp_Arg4)))
								{
									CopyMem(&a[1], buf, a[0]);
									buf[a[0]]= '\0';

									a= strchr(buf, ':');
									strncpy(fname1, a ? a+1 : buf, 16+ 1+ 2);
									fname1[16+ 2]= '\0';

									directory[fharg1.number].type= (DetermineFileType(fname1) | 0x80);
									Ascii2Cbm(directory[fharg1.number].name, fname1, 16);
									StartUDSTimer();
									error= DOSTRUE;
								}
								else
								{
									err2= ERROR_INVALID_COMPONENT_NAME;
								}
							}
							else
							{
								err2= ERROR_OBJECT_NOT_FOUND;
							}
						}
					}
					else
					{
						err2= ERROR_DISK_WRITE_PROTECTED;
					}
				}

				break;
			}

			default:
			{
				err2= ERROR_ACTION_NOT_KNOWN;
				break;
			}
		}
	
		ReturnPacket(packet, error, err2);
	}
	
	return (bUpdate);
}

static LONG TestLock(struct FileLock *fl)
{
	if (fl)
	{
		if (fl->fl_Volume != MKBADDR(curdoslist))
		{
			/* The reference disk is currently not inserted. */

			return(ERROR_DEVICE_NOT_MOUNTED);
		}

		if (fl->fl_Key != 0)
		{
			/* An invalid lock has been passed. */

			return(ERROR_OBJECT_WRONG_TYPE);
		}
	}

	return(0);
}

__inline static LONG TestLockRef(struct FileLock *fl)
{
	if (fl->fl_Volume != MKBADDR(curdoslist))
	{
		/* The reference disk is currently not inserted. */

		return(ERROR_DEVICE_NOT_MOUNTED);
	}

	return(0);
}

/*-------------------------------------------------------------------------*/

static BOOL GetFileData(UBYTE entry, struct FHArg1 *fharg1)
{
	if (directory[entry].name[0] != 0xa0)
	{
		/* cannot get DEL and REL type file */
		
		UWORD type= (directory[entry].type); /*  & 0x07; */

		if (type) /* != 0x00) */
		{
			/* anything but DEL-Files */

			struct DataBlock *block;
			int i, t, s, size;

			/* i<683; a primitive solution to overcome circular links */

			if (iDiskScanned)
			{
				for (i= 0, t= directory[entry].datat, s= directory[entry].datas, size= 0; i< 683; t= block->t, s= block->s, i++)
				{
					if (!(block= GetBlockTS(t, s)))
					{
						return (FALSE);
					}
	
					if (block->t)
					{
						size += 254;
					}
					else
					{
						size += block->s- 1;
	
						break;
					}
				}
				
				i++;
			}
			else
			{
				/* Quick size for files with valid track number only */
				
				if (directory[entry].datat)
				{
					i= ((int)(directory[entry].lengthh<< 8)| directory[entry].lengthl);
					size= i* 254;
				}
				else
				{
					return (FALSE);
				}
			}
			
			if (i< 683)
			{
				fharg1->len= fharg1->len0= size;
				fharg1->numblocks= i;
				fharg1->t= fharg1->t0= directory[entry].datat;
				fharg1->s= fharg1->s0= directory[entry].datas;
				fharg1->number= entry;

				return (TRUE);
			}
		}
	}

	return (FALSE);
}

/*-------------------------------------------------------------------------*/

/* May only be called if curvolumenode is valid! */
static BPTR TryLockFile(LONG *err2, BSTR name, LONG access, struct FHArg1 *fharg1, BOOL allow_root)
{
	UBYTE fname[256];
	UBYTE buf[256];
	STRPTR a;
	LONG error;

	error= DOSFALSE;

	/* Convert BSTR argument into C string */
	
	if ((a= BADDR(name)))
	{
		CopyMem(&a[1], buf, a[0]);
		buf[a[0]]= '\0';

		a= strchr(buf, ':');
		strncpy(fname1, a ? a+ 1 : buf, 16+ 1+ 2);
		fname1[16+ 2]= '\0';
	}
	else
	{
		fname1[0]= '\0';
	}

	/* Cut any given directory entry type locally */
	
	strcpy(fname, fname1);
	DetermineFileType(fname);

	if (fname[0] == '\0')
	{
		if (allow_root)
		{
			/* Return lock on root directory */

			if (access == EXCLUSIVE_LOCK)
			{
				*err2= ERROR_OBJECT_IN_USE;
			}
			else
			{
				*err2= ERROR_NO_FREE_STORE;
				error= MakeLock(0, access);
			}
		}
		else
		{
			*err2= ERROR_OBJECT_WRONG_TYPE;
		}
	}
	else
	{
		int entry;

		*err2= ERROR_OBJECT_NOT_FOUND;
		//*err2= trytrace;

		/* Look up file in directory and lock it */

		for (entry= 0; entry< dirsize; entry++)
		{
			if (directory[entry].type) /* & 0x07) */
			{
				/* anything but DEL-Files */
				
				Cbm2Ascii(buf, directory[entry].name, 16, CHR_LEVEL_ALL);
				if (!Stricmp(buf, fname))
				{
					break;
				}
			}
		}

		if (entry == dirsize)
		{
			//*err2= trytrace+ 10000;
			
			/* Perhaps we can provide a virtual file */

			fharg1->number= NUM_VIRTUAL;

			if (!Stricmp(VIRTUAL_FILE_DISKINFO, fname))
			{
				*err2= ERROR_NO_FREE_STORE;
				error= fharg1->lock= MakeLock(FLKEY_DISKINFO, access);
				fharg1->len= fharg1->len0= diskiconlength;
			}
			else if (fname[0] == '$')
			{
				if (!Stricmp(VIRTUAL_FILE_DOLLAR, fname))
				{
					*err2= ERROR_NO_FREE_STORE;
					error= fharg1->lock= MakeLock(FLKEY_DOLLAR, access);
					fharg1->len= fharg1->len0= curvolumenode->dollarlen;
				}
				else if (!Stricmp(VIRTUAL_FILE_D64, fname))
				{
					*err2= ERROR_NO_FREE_STORE;
					error= fharg1->lock= MakeLock(FLKEY_D64, access);
					fharg1->len= fharg1->len0= D64_SIZE;
				}
				else if (!Stricmp(VIRTUAL_FILE_OPT, fname))
				{
					if (!iWriteProtect)
					{
						*err2= ERROR_NO_FREE_STORE;
						error= fharg1->lock= MakeLock(FLKEY_OPT, access);
						fharg1->len= fharg1->len0= 0;
						OptimizeDirectory();
						StartUDSTimer();
					}
					else
					{
						*err2= ERROR_DISK_WRITE_PROTECTED;
					}
				}
			}
		}
		else
		{
			if (GetFileData(entry, fharg1))
			{
				if (Lockable(fharg1->t, fharg1->s, access))
				{
					*err2= ERROR_NO_FREE_STORE;
					error= fharg1->lock= MakeLock(((fharg1->t)<< 8)| fharg1->s, access);
				}
				else
				{
					*err2= ERROR_OBJECT_IN_USE;
				}
			}
		}
	}

	return(error);
}

LONG Flags2Attrib (BYTE type)
{
	LONG result;

	/* by default no access permissions - inverted logic */

	result= (FIBF_READ | FIBF_WRITE | FIBF_EXECUTE | FIBF_DELETE);

	if (type & 0x80)
	{
		/* file is closed */

		/* deletion and write permission will be added in next block */
		
		switch((type & 7))
		{
			case 0:  /* DEL */
				/* nothing= default */
				break;

			case 1:  /* SEQ */
				result &= ~(FIBF_READ);
				break;

			case 4:  /* REL */
				/* nothing= default */
				break;

			case 2:  /* PRG */
			case 3:  /* USR */
				result &= ~(FIBF_READ | FIBF_EXECUTE);
				break;

			default:
				/* nothing= default */
				break;
		}

		/* is file protected from deletion and write access */
		
		if (!(type & 0x40))
		{
			/* allow deletion and write access */
			
			result &= ~(FIBF_WRITE | FIBF_DELETE);
		}
	}
	else
	{
		/* left open file - no access (needs disk validation) */

		/* nothing= default */
	}

	return (result);
}

void Volume2DiskAndId (struct BAM *bam, char *buffer)
{
	int iIdPos;
	char chIdBuffer[5];
	char *pchPos;

	iIdPos= -1;
	
	pchPos= strrchr(buffer, ',');
	if (pchPos)
	{
		iIdPos= (pchPos- buffer)+ 1;
		buffer[iIdPos- 1]= 0x00;
	}

	Ascii2Cbm(bam->name, buffer, 16);

	if (pchPos)
	{
		strcpy(chIdBuffer, "  ");
		strncpy(chIdBuffer, &buffer[iIdPos], 2);

		Ascii2Cbm(bam->ubDiskIdAr, chIdBuffer, 2);
	}
}

unsigned char DetermineFileType (char *buffer)
{
	int iIdPos;
	char *pchPos;
	unsigned char uchFileType;
	
	uchFileType= 0x02;  /* PRG, open */
	
	pchPos= strrchr(buffer, ',');
	if (pchPos)
	{
		iIdPos= (pchPos- buffer)+ 1;
		buffer[iIdPos- 1]= 0x00;
		
		switch (toupper(buffer[iIdPos]))
		{
			case 'D':
				uchFileType= 0x80;	/* DEL, closed */
				break;
				
			case 'S':
				uchFileType= 0x01;	/* SEQ, open */
				break;
				
			case 'P':
				uchFileType= 0x02;	/* PRG, open */
				break;

			case 'U':
				uchFileType= 0x03;	/* USR, open */
				break;

			case 'R':
				uchFileType= 0x04;	/* REL, open */
				break;

			default:
				uchFileType= 0x00;	/* DEL, open */
				break;
		}
	}

	/* make sure name's length is 16 characters maximum */
		
	buffer[16]= 0x00;

	return (uchFileType);
}

void TodayStamp (struct FileInfoBlock *fib)
{
	struct DateTime sDateTime;
	char szDate[LEN_DATSTRING];
	char szTime[LEN_DATSTRING];
	
	sDateTime.dat_Format= FORMAT_DOS;
	sDateTime.dat_Flags= DTF_SUBST;
	sDateTime.dat_StrDay= NULL;
	sDateTime.dat_StrDate= szDate;
	sDateTime.dat_StrTime= szTime;

	/* use the date stamp of disk insertion to avoid the date 01.01.1978 for files */

	if (fib && curdoslist)
	{		
		fib->fib_Date.ds_Days	= curdoslist->dol_misc.dol_volume.dol_VolumeDate.ds_Days;
		fib->fib_Date.ds_Minute	= curdoslist->dol_misc.dol_volume.dol_VolumeDate.ds_Minute;
		fib->fib_Date.ds_Tick	= curdoslist->dol_misc.dol_volume.dol_VolumeDate.ds_Tick;
		
		DateStamp(&sDateTime.dat_Stamp);
		DateToStr(&sDateTime);
		
		/* it looks like the first character of fib_Comment is ignored */
		
		sprintf(fib->fib_Comment, " (%s %s)", szDate, szTime);
	}
}
