﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

namespace BufferedGraphicsContextExample
{
    public partial class FMMatrixControl : UserControl
    {
        private int m_SelectedOP;
        private bool[] m_OpActive = new bool[8] { false, true, false, true, false, true, false, false };
        private int[,] m_Matrix = new int[8, 10];

        public int SelectedOP
        {
            get { return m_SelectedOP; }
            set { m_SelectedOP = value; }
        }

        public bool[] OpActive
        {
            get { return m_OpActive; }
        }

        public int[,] Matrix
        {
            get { return m_Matrix; }
        }

        public EventHandler MatrixChanged;
        public EventHandler SelectionChanged;

        private Point m_MousePrevPos;
        private Point m_MouseDownPos;
        private Point m_MouseDownCell;
        private int   m_MouseDownValue;

        int Rows = 16 + 3;
        int Cols = 8;

        // Double buffering related
        private bool m_InitializationComplete;
        private bool m_isDisposing;
        private BufferedGraphicsContext m_backbufferContext;
        private BufferedGraphics m_backbufferGraphics;
        private Graphics m_drawingGraphics;

        public FMMatrixControl()
        {
            InitializeComponent();

            // Set the control style to double buffer.
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.SupportsTransparentBackColor, false);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);

            // Assign our buffer context.
            m_backbufferContext = BufferedGraphicsManager.Current;  
            m_InitializationComplete = true;

            RecreateBuffers();

            Redraw();
        }

        protected override void OnFontChanged(EventArgs e)
        {
            base.OnFontChanged(e);
            Redraw();
        }

        public void DrawCell(Graphics g, Point pos, Color clr, Font fnt, string text)
        {
            int PadX = 7;
            int PadY = 2;
            int OffsetY = 0;
            int ShadowOff = 4;
            int CellHeight = ClientSize.Height / Rows - PadY * 2;
            int CellWidth = ClientSize.Width / Cols - PadX * 2;

            var st = new StringFormat();
            st.Alignment = StringAlignment.Center;
            st.LineAlignment = StringAlignment.Center;

            g.FillRectangle(new SolidBrush(Color.FromArgb(50, Color.Black)), new Rectangle(pos.X + PadX + ShadowOff, pos.Y + PadY + ShadowOff + OffsetY, CellWidth, CellHeight));
            g.FillRectangle(new SolidBrush(clr),                             new Rectangle(pos.X + PadX,             pos.Y + PadY + OffsetY,             CellWidth, CellHeight));
            g.DrawRectangle(new Pen(Color.FromArgb(70, Color.Black)),        new Rectangle(pos.X + PadX,             pos.Y + PadY + OffsetY,             CellWidth-1, CellHeight-1));
            g.DrawString(text, fnt, Brushes.Black, new RectangleF(pos.X + PadX-2, pos.Y + PadY + OffsetY, CellWidth+4, CellHeight-1), st);
        }

        public void Redraw()
        {
            if (m_drawingGraphics == null)
                return;

            Color bg_back = Color.FromArgb(unchecked((int)0xFF303030));
            Color bg_cell = Color.FromArgb(unchecked((int)0xFF404040));
            Color bg_out  = Color.FromArgb(unchecked((int)0xFF606060));
            Color bg_pan  = Color.FromArgb(unchecked((int)0xFF505050));
            Color ActiveOp = Color.FromArgb(unchecked((int)0xFFC0C0C0));
            Color UnactiveOp = Color.FromArgb(unchecked((int)0xFF606060));
            
            Brush bg_cell_brush = new SolidBrush(bg_cell);
            Brush bg_out_brush = new SolidBrush(bg_out);
            Brush bg_pan_brush = new SolidBrush(bg_pan);

            m_drawingGraphics.Clear(bg_back);


            // Draw Background
            int CellHeight = ClientSize.Height / Rows;
            int CellWidth  = ClientSize.Width  / Cols;
            int HalfWidth  = CellWidth / 2;
            int HalfHeight = CellHeight / 2;
            for (int iRow = 1; iRow < 8*2; iRow += 2)
            {
                for (int iCol = 0; iCol < Cols; iCol++)
                {
                    int iActRow = (iRow / 2 == iCol) ? iRow - 1 : iRow;
                    m_drawingGraphics.FillRectangle(bg_cell_brush, new Rectangle(1 + iCol * CellWidth, 1 + iActRow * CellHeight, CellWidth - 1, CellHeight - 1));
                }
            }

            // Draw Out and Pan background
            for (int iCol = 0; iCol < Cols; iCol++)
            {
                m_drawingGraphics.FillRectangle(bg_out_brush, new Rectangle(1 + iCol * CellWidth, 1 + 17 * CellHeight, CellWidth - 1, CellHeight - 1));
                m_drawingGraphics.FillRectangle(bg_pan_brush, new Rectangle(1 + iCol * CellWidth, 1 + 18 * CellHeight, CellWidth - 1, CellHeight - 1));
            }

            // Draw Operator lines
            for (int iOp = 0; iOp < 8; iOp++)
            {
                
                // Output not used, if the lowest gain cell
                int BottomMost = iOp;
                for (int iOut = iOp; iOut < 9; iOut++)
                    if (m_Matrix[iOp, iOut] != 0)
                        BottomMost = iOut;

                // Find the RightMost input
                int TopMost = iOp;
                for (int iTop = iOp; iTop >= 0; iTop--)
                    if (m_Matrix[iOp, iTop] != 0)
                        TopMost = iTop;

                // Find the LeftMost input
                int LeftMost = iOp;
                for (int iLeft = iOp; iLeft >= 0; iLeft--)
                    if (m_Matrix[iLeft, iOp] != 0)
                        LeftMost = iLeft;

                // Find the RightMost input
                int RightMost = iOp;
                for (int iRight = iOp; iRight < 8; iRight++)
                    if (m_Matrix[iRight, iOp] != 0)
                        RightMost = iRight;

                if (BottomMost != iOp)
                    m_drawingGraphics.DrawLine(Pens.White, 
                                           iOp * CellWidth + HalfWidth, (1 + iOp * 2)        * CellHeight + HalfHeight, 
                                           iOp * CellWidth + HalfWidth, (1 + BottomMost * 2) * CellHeight + HalfHeight);
                if (TopMost != iOp)
                    m_drawingGraphics.DrawLine(Pens.White,
                                           iOp * CellWidth + HalfWidth, (1 + iOp * 2) * CellHeight + HalfHeight,
                                           iOp * CellWidth + HalfWidth, (1 + TopMost * 2) * CellHeight + HalfHeight);
                if (LeftMost != iOp)
                    m_drawingGraphics.DrawLine(Pens.White,
                                           iOp * CellWidth + HalfWidth, (1 + iOp * 2) * CellHeight + HalfHeight,
                                           LeftMost * CellWidth + HalfWidth, (1 + iOp * 2) * CellHeight + HalfHeight);

                if (RightMost != iOp)
                    m_drawingGraphics.DrawLine(Pens.White,
                                           iOp * CellWidth + HalfWidth, (1 + iOp * 2) * CellHeight + HalfHeight,
                                           RightMost * CellWidth + HalfWidth, (1 + iOp * 2) * CellHeight + HalfHeight);

                // Draw Self feedback
                if (m_Matrix[iOp, iOp] != 0)
                    m_drawingGraphics.DrawRectangle(Pens.White,
                                           iOp * CellWidth + HalfWidth, (iOp * 2) * CellHeight + HalfHeight,
                                           HalfWidth, CellHeight);
            }

            // Draw Operators
            var fnt = new Font(Font.FontFamily, Font.Size, FontStyle.Bold);
            string[] ops = { "A", "B", "C", "D", "E", "F", "X", "Z" };
            for (int i = 0; i < ops.Length; i++)
            {
                var color = m_OpActive[i] ? ActiveOp : UnactiveOp;
                if (m_SelectedOP == i)
                    color = m_OpActive[i] ? Color.Red : Color.DarkRed;
                DrawCell(m_drawingGraphics, new Point(CellWidth * i, CellHeight * (1 + 2 * i)), color, fnt, ops[i]);
            }


            // Draw Operators
            for (int iRow = 0; iRow < 9; iRow++)
            {
                for (int iCol = 0; iCol < 8; iCol++)
                {
                    if (m_Matrix[iCol, iRow] != 0)
                    {
                        int cellYPos = iCol != iRow ? CellHeight * (1 + 2 * iRow) : CellHeight * (2 * iRow);

                        // Out
                        if (iRow == 8) cellYPos = CellHeight * (17);
                        if (iRow == 9) cellYPos = CellHeight * (18);

                        var color = m_OpActive[iCol] ? Color.LightBlue : UnactiveOp;

                        DrawCell(m_drawingGraphics, new Point(CellWidth * iCol, cellYPos), color, Font, m_Matrix[iCol, iRow].ToString());
                    }
                }
            }

            // Draw Panning
            for (int iCol = 0; iCol < 8; iCol++)
            {
                if (m_Matrix[iCol, 8] != 0)
                {
                    int cellXPos = CellWidth * iCol + 1 - 7 + 13 * (m_Matrix[iCol, 9] + 99) / 200;
                    int cellYPos = CellHeight * 18;
                    var color = m_OpActive[iCol] ? Color.LightBlue : UnactiveOp;
                    DrawCell(m_drawingGraphics, new Point(cellXPos, cellYPos), color, Font, m_Matrix[iCol, 9].ToString());
                }
            }


            // Force the control to both invalidate and update. 
            this.Refresh();
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            // If we've initialized the backbuffer properly, render it on the control. 
            // Otherwise, do just the standard control paint.
            if (!m_isDisposing && m_backbufferGraphics != null)
            {
                m_backbufferGraphics.Render(e.Graphics);
            }
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            RecreateBuffers();
            Redraw();
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            m_MousePrevPos = new Point(e.X, e.Y);
            m_MouseDownPos = m_MousePrevPos;

            // Compute cell position
            int CellHeight = ClientSize.Height / Rows;
            int CellWidth  = ClientSize.Width  / Cols;
            int CellX      = (e.X + 1) / CellWidth;
            int CellY      = (e.Y + 1) / CellHeight;

            // Decide where in the grid are we (matrix or outputs)
            bool bCellSelected = false;
            bool bOPlSelected  = false;
            if (CellY < 8 * 2)
            {
                // Normal cell 
                bCellSelected |= (CellX * 2 == CellY) && (CellY % 2 == 0);
                bCellSelected |= (CellX * 2 + 1 != CellY) && (CellY % 2 != 0);
                bOPlSelected = (CellX * 2 + 1 == CellY) && (CellY % 2 != 0);

                m_MouseDownCell = new Point(-1, -1);
                if (bCellSelected || bOPlSelected)
                    m_MouseDownCell = new Point(CellX, CellY / 2);
            }
            else
            {
                // Out and Pan
                bCellSelected = true;
                m_MouseDownCell = new Point(CellX, CellY-16 + 7);
            }

            // Check ranges
            if ((m_MouseDownCell.X > 8) || (m_MouseDownCell.Y > 9))
                m_MouseDownCell = new Point(-1, -1);

            // Left Click
            if ((e.Button & MouseButtons.Left) != 0)
            {
                if (m_MouseDownCell.X != -1)
                {
                    // Value draging
                    if (bCellSelected)
                        m_MouseDownValue = m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y];

                    // Operator selection
                    if (bOPlSelected)
                    {                        
                        // Check if selection was changed
                        if (m_SelectedOP != m_MouseDownCell.X)
                        {
                            m_SelectedOP = m_MouseDownCell.X;
                            if (SelectionChanged != null)
                                SelectionChanged(this, new EventArgs());
                        }

                        m_MouseDownCell = new Point(-1, -1);
                    }
                }
            }

            // Right Click
            if ((e.Button & MouseButtons.Right) != 0)
            {
                if (m_MouseDownCell.X != -1)
                {
                    if (bOPlSelected)
                    {
                        // Operator activation
                        m_OpActive[m_MouseDownCell.X] = !m_OpActive[m_MouseDownCell.X];
                        if (MatrixChanged != null)
                            MatrixChanged(this, new EventArgs());
                    }
                    else
                    {
                        // Value Reset
                        m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = 0;
                        if (MatrixChanged != null)
                            MatrixChanged(this, new EventArgs());

                    }
                    m_MouseDownCell = new Point(-1, -1);
                }
            }

            Redraw();
            base.OnMouseDown(e);
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) != 0)
            {            
                if (m_MouseDownCell.X != -1)
                {
                    var PrevValue = m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y];
                    m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = m_MouseDownValue + m_MouseDownPos.Y - e.Y;
                    if (m_MouseDownCell.Y < 9)
                    {
                        m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = Math.Min(m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y], 99);
                        m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = Math.Max(m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y], 0);
                    }
                    else
                    {
                        m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = Math.Min(m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y], 99);
                        m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y] = Math.Max(m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y], -99);
                    }

                    Redraw();

                    // Check if need to report a matrix change
                    if (PrevValue != m_Matrix[m_MouseDownCell.X, m_MouseDownCell.Y])
                        if (MatrixChanged != null)
                            MatrixChanged(this, new EventArgs());

                }
                m_MousePrevPos = new Point(e.X, e.Y);
            }

            base.OnMouseMove(e);
        }

        protected override void Dispose(bool disposing)
        {
            m_isDisposing = true;
            if (disposing)
            {
                if(components != null)
                    components.Dispose();

                // We must dispose of backbufferGraphics before we dispose of backbufferContext or we will get an exception.
                if (m_backbufferGraphics != null)
                    m_backbufferGraphics.Dispose();
                if (m_backbufferContext != null)
                    m_backbufferContext.Dispose();
            }
            base.Dispose(disposing);
        }

        private void OnMouseWheel(object sender, MouseEventArgs e)
        {
            int delta = e.Delta / 120;
        }

        private void RecreateBuffers()
        {
            // Check initialization has completed so we know backbufferContext has been assigned.
            // Check that we aren't disposing or this could be invalid.
            if (!m_InitializationComplete || m_isDisposing)
                return;

            // We recreate the buffer with a width and height of the control. The "+ 1" 
            // guarantees we never have a buffer with a width or height of 0. 
            m_backbufferContext.MaximumBuffer = new Size(this.Width + 1, this.Height + 1);

            // Dispose of old backbufferGraphics (if one has been created already)
            if (m_backbufferGraphics != null)
                m_backbufferGraphics.Dispose();


            // Create new backbufferGrpahics that matches the current size of buffer.
            var rect = new Rectangle(0, 0, Math.Max(this.Width, 1), Math.Max(this.Height, 1));
            m_backbufferGraphics = m_backbufferContext.Allocate(this.CreateGraphics(), rect);

            // Assign the Graphics object on backbufferGraphics to "drawingGraphics" for easy reference elsewhere.
            m_drawingGraphics = m_backbufferGraphics.Graphics;
            m_drawingGraphics.FillRectangle(new SolidBrush(Color.FromArgb(0)), rect);

            // This is a good place to assign drawingGraphics.SmoothingMode if you want a better anti-aliasing technique.
           
            // Invalidate the control so a repaint gets called somewhere down the line.
            this.Invalidate();
        }
    }
}
