﻿using Jacobi.Vst.Core;
using Jacobi.Vst.Framework;
using Jacobi.Vst.Framework.Plugin;
using KSDManager;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;
using Jacobi.Vst.Core.Plugin;
using Jacobi.Vst.Interop.Host;
using System.Reflection;
using System;
using System.Runtime.InteropServices;

namespace TinyFM8
{
    /// <summary>
    /// The Plugin root class that derives from the framework provided base class that also include the interface manager.
    /// </summary>
    public class Plugin : VstPluginWithInterfaceManagerBase
    {
        public ChannelData       ChannelData = new ChannelData() { Gain = 1 };
        public AudioProcessor    AudioProcessor;
        public MidiProcessor     MidiProcessor;
        public MainUserInterface UserInterface;
        public PluginEditor      PluginEditor;
        public PluginPersistence persistence;
        public PluginPrograms    program;
        public PluginCommandStub pluginStub;

        private Timer tmrReload = new Timer();

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool SetDllDirectory(string lpPathName);

        public Plugin(PluginCommandStub stub)
            : base("TinyFM8",
                new VstProductInfo("TinyFM8", "VST Tools", 1000),
                VstPluginCategory.Synth, 
                VstPluginCapabilities.None, 
                0,
                0x3A3A3A3A)
        {
            // Register and assembly resolver
            AppDomain currentDomain = AppDomain.CurrentDomain;
            currentDomain.AssemblyResolve += new ResolveEventHandler(MyResolveEventHandler);
            SetDllDirectory(Path.GetDirectoryName((new Uri(Assembly.GetExecutingAssembly().CodeBase).LocalPath)));

            // Save stub
            pluginStub = stub;

            // Create reload timer
            tmrReload.Interval = 1;
            tmrReload.Tick += new System.EventHandler(tmrReload_Tick);
        }

        void tmrReload_Tick(object sender, System.EventArgs e)
        {
            tmrReload.Stop();
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }

        public void BuildParameterCollection(VstParameterCollection collection)
        {
            var Cat = new VstParameterCategory();
            Cat.Name = "General";

            var Info = new VstParameterInfo();
            Info.Name = "Param";
            Info.Category = Cat;
            Info.DefaultValue = 1;
            Info.MinInteger = 0;
            Info.MaxInteger = 100;
            Info.StepInteger = 1;
            
            // Create Collection
            collection.Add(new VstParameter(Info));
        }

        public void LoadPatch(string filename)
        {
            // Load Instrument
            var patch = InstrumentLibrary.ReadObject<PatchData>(filename);

            // Assign instrument
            LoadPatch(patch);
        }

        public void LoadPatch(PatchData patch)
        {
            // Assign instrument
            ChannelData.patch = patch;

            // Load instrument
            AudioProcessor.LoadChannelData(ChannelData);

            // Caption
            var ProgramName = patch.DisplayName;
            if (ProgramName.Length > 23)
                ProgramName = ProgramName.Substring(0, 22);

            // Update Plugin name
            pluginStub.SetProgramName(ProgramName);
            var hostIF = Host.GetInstance<IVstHostCommands20>();
            hostIF.UpdateDisplay();
        }

        protected override IVstPluginAudioProcessor CreateAudioProcessor(IVstPluginAudioProcessor instance)
        {
            if (instance == null) return (AudioProcessor = new AudioProcessor(this));

            return base.CreateAudioProcessor(instance);
        }

        protected override IVstMidiProcessor CreateMidiProcessor(IVstMidiProcessor instance)
        {
            if (instance == null) return (MidiProcessor = new MidiProcessor(this));

            return base.CreateMidiProcessor(instance);
        }

        protected override IVstPluginEditor CreateEditor(IVstPluginEditor instance)
        {
            if (instance == null) return (PluginEditor =  new PluginEditor(this));

            return base.CreateEditor(instance);
        }

        protected override IVstPluginPersistence CreatePersistence(IVstPluginPersistence instance)
        {
            if (instance == null) return (persistence = new PluginPersistence(this));

            return base.CreatePersistence(instance);
        }

        protected override IVstPluginPrograms CreatePrograms(IVstPluginPrograms instance)
        {
            if (instance == null) return (program = new PluginPrograms(this));

            return base.CreatePrograms(instance);
        }

        private static Assembly MyResolveEventHandler(object sender, ResolveEventArgs args)
        {
            //Thisw handler is called only when the common language runtime tries to bind to the assembly and fails.
            var SearchFor = args.Name.Substring(0, args.Name.IndexOf(","));
            if (SearchFor == "Jacobi.Vst.Interop")
                SearchFor = Path.GetFileName((new Uri(Assembly.GetExecutingAssembly().CodeBase)).LocalPath).Replace(".net.dll", "");

            var VSTPath = Path.GetDirectoryName((new Uri(Assembly.GetExecutingAssembly().CodeBase)).LocalPath);
            var LoadPath = Path.Combine(VSTPath, SearchFor + ".dll");

            //Load the assembly from the specified path. 					
            var MyAssembly = Assembly.LoadFrom(LoadPath);

            //Return the loaded assembly.
            return MyAssembly;
        }
    }
}
