﻿using Jacobi.Vst.Core;
using Jacobi.Vst.Framework;
using Jacobi.Vst.Framework.Plugin;
using KSDManager;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;
using Jacobi.Vst.Core.Plugin;
using Jacobi.Vst.Interop.Host;
using System;
using System.Runtime.InteropServices;
using System.Reflection;

namespace ReverbFX
{
    /// <summary>
    /// The Plugin root class that derives from the framework provided base class that also include the interface manager.
    /// </summary>
    public class Plugin : VstPluginWithInterfaceManagerBase
    {
        public KSDData           ksd;
        public AudioProcessor    ap;
        public VSTUserControl    control;
        public PluginPersistence persistence;
        public PluginCommandStub pluginStub;

        // Parameters
        public ReverbData             Param = new ReverbData();
        public PluginParameterFactory     ParameterFactory;


        private Timer tmrReload = new Timer();

        public IntPtr LeftHandle;
        public IntPtr RightHandle;

        [DllImport("kernel32.dll", SetLastError = true)]
        static extern bool SetDllDirectory(string lpPathName);

        /// <summary>
        /// Constructs a new instance.
        /// </summary>
        public Plugin(PluginCommandStub stub)
            : base("ReverbFX", 
                new VstProductInfo("ReverbFX", "VST Tools", 1000),
                VstPluginCategory.Effect, 
                VstPluginCapabilities.None, 
                0,
                0x3A3A3A3C)
        {
            // Register and assembly resolver
            AppDomain currentDomain = AppDomain.CurrentDomain;
            currentDomain.AssemblyResolve += new ResolveEventHandler(MyResolveEventHandler);
            SetDllDirectory(Path.GetDirectoryName((new Uri(Assembly.GetExecutingAssembly().CodeBase).LocalPath)));

            pluginStub = stub;

            ParameterFactory = new PluginParameterFactory(this);
            ParameterFactory.ParameterInfos.AddRange(Param.ParameterInfos);

            tmrReload.Interval = 1;
            tmrReload.Tick += new System.EventHandler(tmrReload_Tick);
        }

        public void param_PropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            // Exit if not ready
            foreach (var param in Param.ParameterInfos)
                if (param.ParameterManager.ActiveParameter == null)
                    return;

            // Create data
            var list = new List<ReverbData>();
            list.Add(Param);
            list.Add(Param);


            // Serialze data
            var ms = new MemoryStream();
            var bw = new BinaryWriter(ms);
            SoundRenderInterface.SerilaizeReverb(list, bw);

            // Create instance
            var InstArray = new IntPtr[2];
            var buffer = ms.GetBuffer();
            SoundRenderInterface.CreateReverbs(buffer, InstArray);
            LeftHandle  = InstArray[0];
            RightHandle = InstArray[1];
        }

        void tmrReload_Tick(object sender, System.EventArgs e)
        {
            tmrReload.Stop();
        }
        protected override void Dispose(bool disposing)
        {
            control.Dispose();
            control = null;
            base.Dispose(disposing);
        }

        public void UpdateDisplay()
        {
            pluginStub.SetProgramName("Test");
            var hostIF = Host.GetInstance<IVstHostCommands20>();
            hostIF.UpdateDisplay();
        }

        /// <summary>
        /// Creates a default instance and reuses that for all threads.
        /// </summary>
        /// <param name="instance">A reference to the default instance or null.</param>
        /// <returns>Returns the default instance.</returns>
        protected override IVstPluginAudioProcessor CreateAudioProcessor(IVstPluginAudioProcessor instance)
        {
            if (instance == null) return (ap = new AudioProcessor(this));

            return base.CreateAudioProcessor(instance);
        }

        /// <summary>
        /// Creates a default instance and reuses that for all threads.
        /// </summary>
        /// <param name="instance">A reference to the default instance or null.</param>
        /// <returns>Returns the default instance.</returns>
        protected override IVstPluginEditor CreateEditor(IVstPluginEditor instance)
        {
            if (instance == null) return new PluginEditor(this);

            return base.CreateEditor(instance);
        }

        /// <summary>
        /// Creates a default instance and reuses that for all threads.
        /// </summary>
        /// <param name="instance">A reference to the default instance or null.</param>
        /// <returns>Returns the default instance.</returns>
        protected override IVstPluginPrograms CreatePrograms(IVstPluginPrograms instance)
        {
            if (instance == null) return new PluginPrograms(this);

            return instance;    // reuse initial instance
        }

        /// <summary>
        /// Creates a default instance and reuses that for all threads.
        /// </summary>
        /// <param name="instance">A reference to the default instance or null.</param>
        /// <returns>Returns the default instance.</returns>
        protected override IVstPluginPersistence CreatePersistence(IVstPluginPersistence instance)
        {
            if (instance == null) return (persistence = new PluginPersistence(this));

            return base.CreatePersistence(instance);
        }

        private static Assembly MyResolveEventHandler(object sender, ResolveEventArgs args)
        {
            //Thisw handler is called only when the common language runtime tries to bind to the assembly and fails.
            var SearchFor = args.Name.Substring(0, args.Name.IndexOf(","));
            if (SearchFor == "Jacobi.Vst.Interop")
                SearchFor = Path.GetFileName((new Uri(Assembly.GetExecutingAssembly().CodeBase)).LocalPath).Replace(".net.dll", "");

            var VSTPath = Path.GetDirectoryName((new Uri(Assembly.GetExecutingAssembly().CodeBase)).LocalPath);
            var LoadPath = Path.Combine(VSTPath, SearchFor + ".dll");

            //Load the assembly from the specified path. 					
            var MyAssembly = Assembly.LoadFrom(LoadPath);

            //Return the loaded assembly.
            return MyAssembly;
        }
    }
}
