﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using NAudio.Midi;
using NRenoiseTools;
using Song = NRenoiseTools.RenoiseSong;
using Instrument = NRenoiseTools.RenoiseInstrument;
using System.Text;
using KSDManager;
using System.Windows.Forms;

namespace NRenoiseTools.Xrns2MidiApp
{
    /// <summary>
    /// Simple association of a midi channel and a patch.
    /// </summary>
    public class InternalInstrument
    {
        public string name;
        public int shiftBaseNote;
        public float Volume;
        public string VST_Identfier;
        public byte[] VST_chunkData;
        public List<int> UsedByTracks = new List<int>();
    }

    public class EffectInfo
    {
        public string EffectName;
        public Dictionary<string, float> Parameters = new Dictionary<string, float>();
        public IntPtr[] Handles;
    }

    public class NoteInfo
    {
        public int InstIndex;
        public int Note;
        public int Velocity;

        public NoteInfo()
        {
        }

        public NoteInfo(int InstIndex, int Note, int Velocity)
        {
            this.InstIndex = InstIndex;
            this.Note = Note;
            this.Velocity = Velocity;
        }
    }

    public class RenoiseMidiSong : SongIterator
    {
        // Index    Description
        // 0        TimeStamp (Line #)
        // 1        Instrument
        // 2        Note
        // 3        Velocity
        public List<int[]> SongEvents = new List<int[]>();

        public List<List<IntPtr /*VoiceHandle*/> /*Instrument*/> InstrumentsHandles = new List<List<IntPtr>>();

        public List<EffectInfo>/*Chain*/[/*Tracks*/] TracksEffectChain;

        public float MasterVolume;

        public TextWriter Log { get; set; }

        public String Code;

        private const int MidiDivision = 96;

        private List<int> SortedInstrumentUseage = new List<int>();
        
        private Dictionary<int, NoteInfo> m_LastPlayedNote = new Dictionary<int, NoteInfo>(); 

        private InternalInstrument[] m_Instruments;
        
        private Dictionary<int, int> m_InstrumentMap = new Dictionary<int, int>();

        public RenoiseMidiSong(Song song): base(song)
        {
            Log = Console.Out;
        }

        private void AddEvent(int TimeStamp, int Instrument, int Note, int Velocity)
        {
            var data = new int[] { TimeStamp, Instrument, Note, Velocity };
            SongEvents.Add(data);
        }

        private string ArrayToCPP(byte[] Data)
        {
            StringBuilder sb = new StringBuilder(100000);
            for (int i=0;i<Data.Length;i++)
            {
                if ((i % 16) == 0)
                {
                    sb.AppendLine();
                    sb.Append("  ");
                }

                sb.AppendFormat("0x{0:x2}, ", Data[i]);
            }
            return sb.ToString();
        }

        private string ArrayToCPP(UInt16[] Data)
        {
            StringBuilder sb = new StringBuilder(100000);
            for (int i = 0; i < Data.Length; i++)
            {
                if ((i % 8) == 0)
                {
                    sb.AppendLine();
                    sb.Append("  ");
                }

                sb.AppendFormat("0x{0:x4}, ", Data[i]);
            }
            return sb.ToString();
        }

        private void LoadInstruments()
        {
            Log.WriteLine("Assign Instruments to channels");

            // Iterate on all note in the song to retrieve the instrument used
            m_Instruments = new InternalInstrument[Song.Instruments.Instrument.Length];
            Iterate(new SongIteratorEvent()
            {
                OnNote = delegate
                {
                    // Check if need to ignore note
                    if (Song.Tracks.SequencerTrack[TrackIndex].State != SequencerTrackState.Active)
                        return;

                    if (Song.Tracks.SequencerTrack[TrackIndex].NoteColumnStates[0] != SequencerTrackNoteColumnState.Active)
                        return;

                    if (NoteColumn.Instrument == null)
                        return;

                    // check that instrument is used for the first time
                    int instrumentIndex = Convert.ToInt32(NoteColumn.Instrument, 16);
                    if (m_Instruments[instrumentIndex] == null)
                    {
                        // Get renoise instrument
                        Instrument renoiseInstrument = Song.Instruments.Instrument[instrumentIndex];

                        // Create new export instrument
                        InternalInstrument instrument = new InternalInstrument();
                        m_Instruments[instrumentIndex] = instrument;

                        if (renoiseInstrument.PluginProperties.PluginDevice.IsActive)
                        {
                            // Load instrument settings
                            instrument.name = renoiseInstrument.Name;
                            instrument.shiftBaseNote = renoiseInstrument.PluginProperties.BaseNote - NoteHelper.ConvertToNumber("C-4");
                            instrument.Volume        = renoiseInstrument.PluginProperties.Volume;
                            instrument.VST_Identfier = renoiseInstrument.PluginProperties.PluginDevice.PluginIdentifier;

                            // Fix chunk length
                            string Base64Block = renoiseInstrument.PluginProperties.PluginDevice.ParameterChunk;
                            int mod4 = Base64Block.Length % 4;
                            if (mod4 > 0) Base64Block += new string('=', 4 - mod4);
                            instrument.VST_chunkData = Convert.FromBase64String(Base64Block);
                        }
                    }

                    // Add to track index list
                    if (!m_Instruments[instrumentIndex].UsedByTracks.Contains(TrackIndex))
                        m_Instruments[instrumentIndex].UsedByTracks.Add(TrackIndex);                                     
                }
            });

            // Load instruments, build instrumentMap
            var voiceList = new List<KSDManager.KSDData>();
            for (int i = 0; i < m_Instruments.Length; i++)
            {
                if (m_Instruments[i] == null)
                    continue;

                // Check that the instrument is not used by more than one track
                if (m_Instruments[i].UsedByTracks.Count > 1)
                {

                    MessageBox.Show(null, string.Format("The instrument \"{0}\" is used by track \"{1}\" and by track \"{2}\".\nMake sure instruments are not shared in several tracks.", 
                        m_Instruments[i].name, 
                        Song.Tracks.SequencerTrack[m_Instruments[i].UsedByTracks[0]].Name,
                        Song.Tracks.SequencerTrack[m_Instruments[i].UsedByTracks[1]].Name), 
                        "Loading Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }

                // Load KSDData
                KSDData inst = null;
                switch (m_Instruments[i].VST_Identfier.ToUpper())
                {
                    case "FM8":
                    {
                        inst = KSDData.LoadFromStream(new MemoryStream(m_Instruments[i].VST_chunkData));
                        break;
                    }

                    case "TINYFM8":
                    {
                        BinaryReader br = new BinaryReader(new MemoryStream(m_Instruments[i].VST_chunkData));

                        // Read programs count
                        var typeName = br.ReadString();
                        var progCount = br.ReadInt32();
                        for (int j = 0; j < progCount; j++)
                        {
                            // read program name
                            var programName = br.ReadString();

                            // read parameters
                            var paramType = br.ReadString();
                            var paramCount = br.ReadInt32();
                            for (int k = 0; k < paramCount; k++)
                            {
                                string name = br.ReadString();
                                float value = br.ReadSingle();
                            }
                        }

                        // Internal FM8 saved data
                        byte[] FM8data = new byte[br.BaseStream.Length - br.BaseStream.Position];
                        br.Read(FM8data, 0, FM8data.Length);

                        // Load FM8 instrument
                        inst = KSDData.LoadFromStream(new MemoryStream(FM8data));

                        // Set Gain
                        inst.Gain *= SoundRenderInterface.dB2lin(m_Instruments[i].Volume); 

                        break;
                    }

                    default:
                    {
                        MessageBox.Show(null, string.Format(@"Unsupported instrument ""{0}"".", m_Instruments[i].VST_Identfier), "Loading Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        break;
                    }

                }

                // Add instrument to instrument map
                if (inst != null)
                {
                    // Add to instrument map
                    m_InstrumentMap.Add(i, voiceList.Count);
                    voiceList.Add(inst);                    

                    // Update Instrument track index
                    inst.TrackIndex = m_Instruments[i].UsedByTracks.Min();
                }
            }

            // Update PreAmp gains
            foreach (var inst in voiceList)
            {
                // Convert Master volume to gain
                //inst.MasterVolume = 80;
                inst.Gain *= 0.000156f * inst.MasterVolume * inst.MasterVolume;

                // Convert Preamp to gain
                var PreAmp = TracksEffectChain[inst.TrackIndex][0];
                inst.Gain *= PreAmp.Parameters["Volume"];
            }


            // Serilaize instrumente
            var bw = new BinaryWriter(new MemoryStream());
            SoundRenderInterface.SerilaizeInstruments(voiceList, bw);

            // read data from stream
            var data = new byte[bw.BaseStream.Length];
            bw.BaseStream.Position = 0;
            bw.BaseStream.Read(data, 0, data.Length);

            // Save stream as C++ code
            Code += "unsigned char InstData[] = {";
            Code += ArrayToCPP(data).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";

            // Load Instrument by sound renderer
            var InstArray = new IntPtr[256];
            int nInstCount = SoundRenderInterface.CreateInstruments(data, InstArray);
            
            // Create Voices            
            for (int i = 0; i<nInstCount;i++)
            {
                // Create voice list and add main voice
                var voices = new List<IntPtr>();
                InstrumentsHandles.Add(voices);
                voices.Add(InstArray[i]);

                // Add other voices
                for (int j = 1; j < 9; j++)
                    voices.Add(SoundRenderInterface.DuplicateInstrument(InstArray[i]));
            }
        }

        private void LoadCompressors(List<CompressorData> compressors)
        {
            // Serialze compressor parameters
            var ms = new MemoryStream();
            var bw = new BinaryWriter(ms);
            SoundRenderInterface.SerilaizeCompressor(compressors, bw);

            // read data from stream
            var buffer = new byte[bw.BaseStream.Length];
            bw.BaseStream.Position = 0;
            bw.BaseStream.Read(buffer, 0, buffer.Length);

            // Save stream as C++ code
            Code += "unsigned char CompData[] = {";
            Code += ArrayToCPP(buffer).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";

            // Create compressors
            var compHandles = new IntPtr[compressors.Count];
            SoundRenderInterface.CreateCompressors(ms.GetBuffer(), compHandles);

            // Assign comp handles
            var compHandleList = compHandles.ToList();
            foreach (var track in TracksEffectChain)
                foreach (var effect in track)
                    if (effect.EffectName == "CompressorFX")
                    {
                        effect.Handles = new[] { compHandleList[0] };
                        compHandleList.RemoveAt(0);
                    }
            
        }

        private void LoadReverbs(List<ReverbData> reverbs)
        {
            // Serialze compressor parameters
            var ms = new MemoryStream();
            var bw = new BinaryWriter(ms);
            SoundRenderInterface.SerilaizeReverb(reverbs, bw);

            // read data from stream
            var buffer = new byte[bw.BaseStream.Length];
            bw.BaseStream.Position = 0;
            bw.BaseStream.Read(buffer, 0, buffer.Length);

            // Save stream as C++ code
            Code += "unsigned char ReverbData[] = {";
            Code += ArrayToCPP(buffer).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";

            // Create compressors
            var reverbHandles1 = new IntPtr[reverbs.Count];
            var reverbHandles2 = new IntPtr[reverbs.Count];
            SoundRenderInterface.CreateReverbs(ms.GetBuffer(), reverbHandles1);
            SoundRenderInterface.CreateReverbs(ms.GetBuffer(), reverbHandles2);

            // Assign comp handles
            var reverbHandleList1 = reverbHandles1.ToList();
            var reverbHandleList2 = reverbHandles2.ToList();
            foreach (var track in TracksEffectChain)
                foreach (var effect in track)
                    if (effect.EffectName == "ReverbFX")
                    {
                        effect.Handles = new[] {reverbHandleList1[0], reverbHandleList2[0]};
                        reverbHandleList1.RemoveAt(0);
                        reverbHandleList2.RemoveAt(0);
                    }

        }

        private void LoadTrackInfo_ProcessFX(object item, List<EffectInfo> effectsList)
        {
            if (item is SequencerTrackDevice)
            {
                var cast = item as SequencerTrackDevice;
                if (!cast.IsActive)
                    return;

                // Create effect
                var eff = new EffectInfo();
                effectsList.Add(eff);
                eff.EffectName = "Channel";
                eff.Parameters.Add("Volume", cast.Volume.Value);
                eff.Parameters.Add("Panning", cast.Panning.Value);
                eff.Parameters.Add("PostVolume", cast.PostVolume.Value);
                eff.Parameters.Add("PostPanning", cast.PostPanning.Value);
            }

            if (item is AudioPluginDevice)
            {
                var cast = item as AudioPluginDevice;
                if (!cast.IsActive)
                    return;

                // Create effect
                var eff = new EffectInfo();
                effectsList.Add(eff);
                eff.EffectName = cast.PluginIdentifier;

                // Create Chunk stream
                string Base64Block = cast.ParameterChunk;
                int mod4 = Base64Block.Length % 4;
                if (mod4 > 0) Base64Block += new string('=', 4 - mod4);
                var bytes = Convert.FromBase64String(Base64Block);
                var br = new BinaryReader(new MemoryStream(bytes));

                // Read programs count
                var typeName = br.ReadString();
                var progCount = br.ReadInt32();

                // read program name
                var programName = br.ReadString();

                // read parameters
                var paramType = br.ReadString();
                var paramCount = br.ReadInt32();
                for (int k = 0; k < paramCount; k++)
                {
                    string name = br.ReadString();
                    float value = br.ReadSingle();
                    eff.Parameters.Add(name, value);
                }
            }
        }

        private void LoadTrackInfo()
        {
            // Create tracks list
            TracksEffectChain = new List<EffectInfo>[Song.Tracks.SequencerTrack.Length + 1];
            for (int i = 0; i < TracksEffectChain.Length; i++)
                TracksEffectChain[i] = new List<EffectInfo>();

            // Load Master effects
            foreach (var item in Song.Tracks.SequencerMasterTrack[0].FilterDevices.Devices.Items)
                LoadTrackInfo_ProcessFX(item, TracksEffectChain[TracksEffectChain.Length-1]);

            // Load Tracsks effects
            for (int i = 0; i < Song.Tracks.SequencerTrack.Length; i++)
            {
                // Check if track is active
                var track = Song.Tracks.SequencerTrack[i];
                if (track.State != SequencerTrackState.Active)
                    continue;

                var chain = track.FilterDevices.Devices.Items;
                foreach (var item in chain)
                    LoadTrackInfo_ProcessFX(item, TracksEffectChain[i]);
            }

            // Create compressor list
            var compressors = new List<CompressorData>();
            var reverbs = new List<ReverbData>();
            foreach (var track in TracksEffectChain)
                foreach (var effect in track)
                {
                    if (effect.EffectName == "CompressorFX")
                    {
                        // Create data
                        var data = new CompressorData();
                        data.prmAttackTime.NormalizedValue = effect.Parameters["Attack"];
                        data.prmReleaseTime.NormalizedValue = effect.Parameters["Release"];
                        data.prmThreshold.NormalizedValue = effect.Parameters["Thshld"];
                        data.prmMakeupGain.NormalizedValue = effect.Parameters["Gain"];
                        data.prmRatio.NormalizedValue = effect.Parameters["Ratio"];
                        compressors.Add(data);
                    }
                    else if (effect.EffectName == "ReverbFX")
                    {
                        // Create data
                        var data = new ReverbData();
                        data.prmInputDiffuse1.NormalizedValue = effect.Parameters["Difuse1"];
                        data.prmInputDiffuse2.NormalizedValue = effect.Parameters["Difuse2"];
                        data.prmBandLimiter.NormalizedValue = effect.Parameters["BandLmt"];
                        data.prmTime.NormalizedValue = effect.Parameters["Time"];
                        data.prmDamping.NormalizedValue = effect.Parameters["Damping"];
                        data.prmOutGain.NormalizedValue = effect.Parameters["OutGain"];
                        reverbs.Add(data);
                    }
                }

            LoadCompressors(compressors);
            LoadReverbs(reverbs);
        }

        public void ConvertToInternal()
        {
            Code = "";

            // Load Master volume
            MasterVolume = ((NRenoiseTools.SequencerMasterTrackDevice)(Song.Tracks.SequencerMasterTrack[0].FilterDevices.Devices.Items[0])).PostVolume.Value;

            // Load Tracks info (effects)
            LoadTrackInfo();

            // Load instruments
            LoadInstruments();

            Log.WriteLine("Convert patterns");
            Iterate(new SongIteratorEvent
            {
                OnEndSong = delegate
                {
                    Log.WriteLine("\t-- EndSong --");
                },
                OnBeginPattern = delegate
                {
                    Log.WriteLine("\tConvert Pattern {0}", PatternIndex);
                },
                OnNote = delegate
                {
                    // Check if need to ignore note
                    if (Song.Tracks.SequencerTrack[TrackIndex].NoteColumnStates[0] != SequencerTrackNoteColumnState.Active)
                        return;

                    // Check if need to ignore note
                    if (Song.Tracks.SequencerTrack[TrackIndex].State != SequencerTrackState.Active)
                        return;

                    // Get the last played note
                    NoteInfo LastPlayed = null;
                    int UniqeIndex = TrackIndex * 100 + ColumnIndex;
                    if (m_LastPlayedNote.ContainsKey(UniqeIndex))
                        LastPlayed = m_LastPlayedNote[UniqeIndex];

                    // Get the instrument index
                    int event_instrument = -1;
                    if (LastPlayed != null)            event_instrument = LastPlayed.InstIndex;
                    if (NoteColumn.Instrument != null) event_instrument = Convert.ToInt32(NoteColumn.Instrument, 16);
                    if (event_instrument == -1)
                        return;

                    // compute velocity
                    int event_velocity = 0x80;  
                    if (NoteColumn.Volume != null) event_velocity = Convert.ToInt32(NoteColumn.Volume, 16);

                    // compute Note number
                    int event_noteNumber = -1;
                    if (NoteColumn.Note != null)
                       event_noteNumber = NoteHelper.ConvertToNumber(NoteColumn.Note);

                    // Check if need to terminate last note
                    if ((LastPlayed != null) && ((NoteColumn.Note == "OFF")  || (event_noteNumber >= 0)))
                    {
                        // Add Stop Event
                        AddEvent(LineIndexInSong, m_InstrumentMap[LastPlayed.InstIndex], LastPlayed.Note, 0);

                        // Remove the last pressed note from the channel
                        m_LastPlayedNote.Remove(UniqeIndex);
                        LastPlayed = null;
                    }
                    
                    // Check if need to play a new note
                    if (event_noteNumber >= 0)
                    {
                        // shift note by instrument
                        event_noteNumber -= m_Instruments[event_instrument].shiftBaseNote;

                        // Create last played info (if does not already exists)
                        LastPlayed = new NoteInfo(event_instrument, event_noteNumber, event_velocity);
                        m_LastPlayedNote.Add(UniqeIndex, LastPlayed);

                        // Add Note Event
                        AddEvent(LineIndexInSong, m_InstrumentMap[event_instrument], event_noteNumber, event_velocity);
                    }
                }
            });

            SongEvents = SongEvents.OrderBy(t => t[0]).ToList();

            // Splite into arrays
            int nIndex = 0;
            var me_LineTS        = new UInt16[SongEvents.Count];
            var me_InstIndex     = new byte[SongEvents.Count];
            var me_NoteIndex     = new byte[SongEvents.Count];
            var me_VelocityIndex = new byte[SongEvents.Count];
            foreach (var midiEvent in SongEvents)
            {
                me_LineTS[nIndex]        = (UInt16)midiEvent[0];
                me_InstIndex[nIndex]     = (byte)midiEvent[1];
                me_NoteIndex[nIndex]     = (byte)midiEvent[2];
                me_VelocityIndex[nIndex] = (byte)midiEvent[3];
                nIndex++;
            }

            // Compute samples per line
            int LinesPerMin = Tempo.Lpb * (int)Tempo.Bpm;
            int SamplesPerLine = 60 * 44100 / LinesPerMin;

            // Save stream as C++ code
            Code += "#define SAMPLES_PER_LINE (" + SamplesPerLine + ")\n\n";
            Code += "#define SONG_EVENTS      (" + me_LineTS.Length + ")\n\n";
            Code += "unsigned short songEvent_Time[] = {" + ArrayToCPP(me_LineTS).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";
            Code += "unsigned char songEvent_Inst[] = {" + ArrayToCPP(me_InstIndex).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";
            Code += "unsigned char songEvent_Note[] = {" + ArrayToCPP(me_NoteIndex).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";
            Code += "unsigned char songEvent_Velocity[] = {" + ArrayToCPP(me_VelocityIndex).TrimEnd(new char[] { ',', ' ' }) + "}; \r\n\r\n";

            // Export Track effect code
            int ReverbCounter = 0;
            int CompressorCounter = 0;
            int MasterIndex = TracksEffectChain.Length - 1;
            Code += "#define TRACKS_FX_PROCESS".PadRight(100) + "\\\r\n";
            Code += "    switch (iTrack)".PadRight(100)       + "\\\r\n";
            Code += "    {".PadRight(100)                     + "\\\r\n";
            for (int i = 0; i < MasterIndex; i++)
            {
                // Check if there are no effects in track
                if (TracksEffectChain[i].Where(t=>t.EffectName != "Channel").Count() == 0)
                    continue;

                Code += String.Format("        case {0}:", i).PadRight(100) + "\\\r\n";

                foreach (var effect in TracksEffectChain[i])
                {
                    switch (effect.EffectName)
                    {
                        case "CompressorFX":
                            Code += String.Format("            m_Compressors[{0}]->Render(LineWorkBuf, SAMPLES_PER_LINE);", CompressorCounter++).PadRight(100) + "\\\r\n";
                            break;

                        case "ReverbFX":
                            Code += String.Format("            FM8_ReverbRender(m_LReverbs[{0}], m_RReverbs[{0}], LineWorkBuf, SAMPLES_PER_LINE);", ReverbCounter++).PadRight(100) + "\\\r\n";
                            break;
                    }
                }

                Code += "            break;".PadRight(100) + "\\\r\n";
            }
            Code += "    }\r\n\r\n";
        
            // Export master channel effect
            int EffectCount = 0;
            int MasterEffects = TracksEffectChain[MasterIndex].Where(t => t.EffectName != "Channel").Count();
            Code += (MasterEffects == 0) ? "#define MASTER_FX_PROCESS" : "#define MASTER_FX_PROCESS".PadRight(100) + "\\\r\n";
            foreach (var effect in TracksEffectChain[MasterIndex])
            {
                string Line = "";
                switch (effect.EffectName)
                {
                    case "CompressorFX":
                        Line = String.Format("    m_Compressors[{0}]->Render(LineSumBuf, SAMPLES_PER_LINE);", CompressorCounter++);
                        break;

                    case "ReverbFX":
                        Line = String.Format("    FM8_ReverbRender(m_LReverbs[{0}], m_RReverbs[{0}], LineSumBuf, SAMPLES_PER_LINE);", ReverbCounter++);
                        break;

                    default:
                        continue;
                }

                // Add line
                if (Line != "")
                    Code += (++EffectCount == MasterEffects) ? Line : Line.PadRight(100) + "\\\r\n";
            }

            Code += "\r\n\r\n";
        }
    }
}
