﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using Microsoft.DirectX.DirectSound;
using NRenoiseTools.Xrns2MidiApp;

public delegate void NoteCommandSent(int Index);

public class SoundRenderer
{
    // Instruments and voices list
    public List</*Instruments*/List</*Voices*/IntPtr>> m_Instruments = new List<List<IntPtr>>();

    // Track Effect list
    public List<EffectInfo>/*Chain*/[/*Tracks*/] m_TrackEffects;

    // Song Data
    // Index    Description
    // 0        TimeStamp (Line #)
    // 1        Instrument
    // 2        Note
    // 3        Velocity    
    public List<int[]> m_songEvents = new List<int[]>();

    // Audio Setup
    public const int SAMPLE_RATE = 44100;
    public const int AUDIO_CHANNELS = 2;
    public const int BYTES_PER_SAMPLE = 2;

    // CHANNEL_BUFFER_DEPTH is the number of buffering samples for each channel
    public const int CHANNEL_BUFFER_DEPTH = SAMPLE_RATE * 20 /*seconds*/;

    // SamplesPerLine represent the number of samples required to play a single line at the required BPM/Tempo.
    // The value is given in "SampleRate" domain, and should NOT be double for stereo channeling.
    public int m_SamplesPerLine;

    // Output related
    private Thread          m_RenderThread;
    private Device          m_DirectSoundDevice;
    private SecondaryBuffer m_secondary;
    private int             m_secondary_prevPos;
    public  float           m_BufferingState;
    public  BinaryWriter    m_Dump;

    // Rendering counters
    public int m_QueuePos_Line;
    public int m_QueuePos_Samples;
    public int m_QueuePos_NextEvent;
    public int m_PlayPos_Line = -1;
    public int m_PlayPos_Samples;
    public int m_PlayPos_NextEvent;

    public NoteCommandSent OnNoteCommandSent;

    public WaveWriter m_waveWriter;

    public void Setup(Control Handle)
    {
        // Create output file
        //m_Dump = new BinaryWriter(new FileStream("c:\\raw2.out", FileMode.Create));

        // Create device
        m_DirectSoundDevice = new Device();
        m_DirectSoundDevice.SetCooperativeLevel(Handle, CooperativeLevel.Normal);

        // Create stream format
        var format = new WaveFormat();
        format.BitsPerSample = 16;
        format.Channels = 2;
        format.BlockAlign = 4;
        format.FormatTag = WaveFormatTag.Pcm;
        format.SamplesPerSecond = SAMPLE_RATE;
        format.AverageBytesPerSecond = 4 * SAMPLE_RATE;

        // Create secondary buffer and start playing
        var desc = new BufferDescription(format);
        desc.DeferLocation = true;
        desc.GlobalFocus = true;
        desc.BufferBytes = CHANNEL_BUFFER_DEPTH * AUDIO_CHANNELS * BYTES_PER_SAMPLE;
        m_secondary = new SecondaryBuffer(desc, m_DirectSoundDevice);

        // Set initial position (only needed for seeking)
        m_secondary_prevPos = m_PlayPos_Samples % CHANNEL_BUFFER_DEPTH;
        m_secondary.SetCurrentPosition(m_secondary_prevPos * AUDIO_CHANNELS * BYTES_PER_SAMPLE);

        // Start audio render thread
        m_RenderThread = new Thread(AudioRenderThread);
        m_RenderThread.IsBackground = true;
        m_RenderThread.Start();
    }

    public void Play()
    {
        m_secondary.SetCurrentPosition(m_secondary_prevPos * AUDIO_CHANNELS * BYTES_PER_SAMPLE);
        m_secondary.Play(0, BufferPlayFlags.Looping);
    }

    public void Pause()
    {
        m_secondary.Stop();
    }

    public void Close()
    {
        m_secondary.Stop();
        m_RenderThread.Abort();
        m_RenderThread.Join();
        m_secondary.Dispose();
        m_DirectSoundDevice.Dispose();

        if (m_waveWriter != null)
            m_waveWriter.Dispose();

    }

    private void UpdatePlayPosition()
    {
        // Get current positions (mind the "PlayPos / (AudioChannels * BytesPerSample)")
        int PlayPosition = m_secondary.PlayPosition / (AUDIO_CHANNELS * BYTES_PER_SAMPLE);
        if (!m_secondary.Status.Playing) PlayPosition = m_secondary_prevPos;
        int SamplesPlayed = PlayPosition - m_secondary_prevPos;
        m_secondary_prevPos = PlayPosition;

        // Handle overlapping
        SamplesPlayed = (SamplesPlayed >= 0) ? SamplesPlayed : SamplesPlayed + CHANNEL_BUFFER_DEPTH;


        // Update total played 
        m_PlayPos_Samples += SamplesPlayed;
        m_PlayPos_Line     = m_PlayPos_Samples / m_SamplesPerLine;

        // Update buffering state
        m_BufferingState = Math.Max(0, (m_QueuePos_Samples - m_PlayPos_Samples) * 100f / CHANNEL_BUFFER_DEPTH);

        // Track on the currently processed event
        while ((m_PlayPos_NextEvent < m_songEvents.Count) && (m_songEvents[m_PlayPos_NextEvent][0/*Time*/] <= m_PlayPos_Line))
        {
            // Notify on new note sent
            if (OnNoteCommandSent != null)
                OnNoteCommandSent(m_PlayPos_NextEvent);

            // Next event
            m_PlayPos_NextEvent++;
        }
    }

    private void ProcessChannelEffects(List<EffectInfo> effectChain, float[] AudioBuffer)
    {
        for (int index = 0; index < effectChain.Count; index++)
        {
            if (effectChain[index].EffectName == "CompressorFX")
            {
                SoundRenderInterface.CompressorRender(effectChain[index].Handles[0], AudioBuffer, m_SamplesPerLine);
            }
            else if (effectChain[index].EffectName == "ReverbFX")
            {
                SoundRenderInterface.ReverbRender(effectChain[index].Handles[0], effectChain[index].Handles[1], AudioBuffer, m_SamplesPerLine);
            }
        }
    }

    private void AudioRenderThread()
    {
        var LineWorkBuf    = new float[m_SamplesPerLine * AUDIO_CHANNELS]; // Used for each instrument as a temporary target buffer
        var LineSumBuf     = new float[m_SamplesPerLine * AUDIO_CHANNELS]; // Used to sum all channels
        var LineOutputBuf  = new Int16[m_SamplesPerLine * AUDIO_CHANNELS]; // The result of converting LineSumBuf into 16bit

        // Find m_QueuePos_NextEvent according to m_QueuePos_Line
        m_QueuePos_NextEvent = m_songEvents.FindIndex(t => t[0/*Time*/] >= m_QueuePos_Line);
        m_PlayPos_NextEvent  = m_QueuePos_NextEvent;

        // Turn off everything
        foreach (var instruments in m_Instruments)
            foreach (var voice in instruments)
                SoundRenderInterface.OffNote(voice);
        
        while (m_RenderThread.IsAlive)
        {
            // Update Play position before it is used 
            UpdatePlayPosition();
            
            // Check if we need to render a fragment (if there is a single row of free space in queue)
            while (CHANNEL_BUFFER_DEPTH + m_PlayPos_Samples - m_QueuePos_Samples > m_SamplesPerLine)
            {
                // Execute Song events
                while ((m_QueuePos_NextEvent < m_songEvents.Count) && (m_songEvents[m_QueuePos_NextEvent][0/*Time*/] <= m_QueuePos_Line))
                {
                    int nInstIndex = m_songEvents[m_QueuePos_NextEvent][1/*Instrument*/];

                    // Check if it's a release command
                    if (m_songEvents[m_QueuePos_NextEvent][3/*Velocity*/] == 0)
                    {
                        // Release all voices in instrument
                        foreach (var voice in m_Instruments[nInstIndex])
                            if (SoundRenderInterface.GetNote(voice) == m_songEvents[m_QueuePos_NextEvent][2/*Note*/])
                                SoundRenderInterface.OffNote(voice);
                    }
                    else
                    {
                        int nVoices = SoundRenderInterface.GetVoices(m_Instruments[nInstIndex][0]);
                        for (int nVoice = 0; nVoice < nVoices; nVoice++)
                        {
                            // move head->tail
                            var prevInst = m_Instruments[nInstIndex][0];
                            m_Instruments[nInstIndex].RemoveAt(0);
                            m_Instruments[nInstIndex].Insert(m_Instruments[nInstIndex].Count, prevInst);

                            // init head instrument
                            var inst = m_Instruments[nInstIndex][0];
                            SoundRenderInterface.PlayNote(inst, m_songEvents[m_QueuePos_NextEvent][2], nVoice);
                            SoundRenderInterface.SetVelocity(inst, m_songEvents[m_QueuePos_NextEvent][3] / 127f);
                        }
                    }

                    // Next event
                    m_QueuePos_NextEvent++;
                }

                // Render audio of all voices
                Array.Clear(LineSumBuf, 0, LineSumBuf.Length);
                for (int iTrack = 0; iTrack < m_TrackEffects.Length - 1; iTrack++)
                {
                    Array.Clear(LineWorkBuf, 0, LineWorkBuf.Length);
                    foreach (var instrument in m_Instruments)
                    {
                        // Skip no relevent track
                        if (SoundRenderInterface.GetTrackIndex(instrument[0]) != iTrack)
                            continue;

                        // Render all voices to track
                        foreach (var voice in instrument)
                            SoundRenderInterface.SynthRender(voice, LineWorkBuf, m_SamplesPerLine);
                    }

                    // Apply effect chain
                    ProcessChannelEffects(m_TrackEffects[iTrack], LineWorkBuf);

                    // add samples to main buffer
                    for (int i = 0; i < LineWorkBuf.Length; i++)
                        LineSumBuf[i] += LineWorkBuf[i];
                }

                // Applt master channel effects
                ProcessChannelEffects(m_TrackEffects[m_TrackEffects.Length - 1], LineSumBuf);

                // Convert to 16 bit
                for (int i = 0; i < LineOutputBuf.Length; i++)
                {
                    float sample = LineSumBuf[i] * 0.5f;
                    if (sample > 1) sample = 1;
                    if (sample < -1) sample = -1;

                    //m_Dump.Write(sample);
                    LineOutputBuf[i] = (Int16)(sample * Int16.MaxValue);
                }

                // Write data to secondary buffer
                int WriteOffset = (m_QueuePos_Samples % CHANNEL_BUFFER_DEPTH) * AUDIO_CHANNELS * BYTES_PER_SAMPLE;
                m_secondary.Write(WriteOffset, LineOutputBuf.ToArray(), LockFlag.None);

                // Update wave writer (is enabled)
                if (m_waveWriter != null)
                    m_waveWriter.WriteData(LineOutputBuf);

                // Increment queue counters
                m_QueuePos_Samples += m_SamplesPerLine;
                m_QueuePos_Line++;

                // Update Buffering state
                UpdatePlayPosition();
            }
        }
    }
}

