﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using KSDManager;
using System.IO;
using System.Runtime.InteropServices;
using TinyFM8;

class SoundRenderInterface
{
    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void BuildWaveforms();

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int CreateInstruments(byte[] pBlock, IntPtr[] pInstruments);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int CreateCompressors(byte[] pBlock, IntPtr[] pCompressors);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int CreateReverbs(byte[] pBlock, IntPtr[] pReverb);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern IntPtr DuplicateInstrument(IntPtr Handle);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void SynthRender(IntPtr Handle, float[] Buffer, int nBufferSize);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void CompressorRender(IntPtr Handle, float[] Buffer, int nBufferSize);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void ReverbRender(IntPtr LeftHandle, IntPtr RightHandle, float[] Buffer, int nBufferSize);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void PlayNote(IntPtr Handle, int NoteIndex, int VoiceIndex);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void SetVelocity(IntPtr Handle, float Velocity);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void OffNote(IntPtr Handle);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int GetVoices(IntPtr Handle);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int GetNote(IntPtr Handle);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern int GetTrackIndex(IntPtr Handle);
    
    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern  void  GetWaveform(int nWaveformIndex, float[] pBuffer);

    [DllImport("SoundRender.dll", CallingConvention = CallingConvention.StdCall)]
    public static extern void RenderEnvelope(IntPtr Handle, int OpIndex, float FromT, float ToT, int Samples, float[] pBuffer);

    public static void SerilaizeInstruments(List<ChannelData> Instruments, BinaryWriter sw)
    {
        List<BinaryWriter> instWriters = new List<BinaryWriter>();

        sw.Write((byte)Instruments.Count);
        if (Instruments.Count == 0)
            return;

        int nInst = 0;
        foreach (var inst in Instruments)
        {
            var memstream = new MemoryStream();
            var writer = new BinaryWriter(memstream);
            instWriters.Add(writer);

            // Clean disabled oscillators
            for (int i = 0; i < 6; i++)
            {
                if (inst.patch.OP_Active[i] == 0)
                {
                    inst.patch.OP_Waveform[i] = 0;
                    inst.patch.OP_Velocity[i] = 0;
                    inst.patch.OP_KeySync[i] = 0;
                    inst.patch.OP_OutVol[i] = 0;
                    inst.patch.OP_OutPan[i] = 0;
                    inst.patch.OP_KeyScale[i] = 0;
                    inst.patch.OP_KeySc_Level[i] = new float[0];
                    inst.patch.OP_Env_Level[i] = new float[0];
                    inst.patch.OP_Env_SustainLoopStart[i] = 0;
                    inst.patch.OP_Env_SustainLoopEnd[i] = 0;
                    inst.patch.OP_Ratio[i] = 0;
                    inst.patch.OP_Offset[i] = 0;
                }
            }

            // Clean X_OP
            if (inst.patch.OP_Active[6] == 0)
            {
                inst.patch.OP_Velocity[6] = 0;
                inst.patch.OP_OutVol[6] = 0;
                inst.patch.OP_OutPan[6] = 0;
                inst.patch.OP_KeyScale[6] = 0;
                inst.patch.OP_KeySc_Level[6] = new float[0];
                inst.patch.OP_Env_Level[6] = new float[0];
                inst.patch.OP_Env_SustainLoopStart[6] = 0;
                inst.patch.OP_Env_SustainLoopEnd[6] = 0;
            }

            // Clean Z_OP
            if (inst.patch.OP_Active[7] == 0)
            {
                inst.patch.OP_Velocity[7] = 0;
                inst.patch.OP_OutVol[7] = 0;
                inst.patch.OP_OutPan[7] = 0;
                inst.patch.OP_KeyScale[7] = 0;
                inst.patch.OP_KeySc_Level[7] = new float[0];
                inst.patch.OP_Env_Level[7] = new float[0];
                inst.patch.OP_Env_SustainLoopStart[7] = 0;
                inst.patch.OP_Env_SustainLoopEnd[7] = 0;
            }

            writer.Write((float)inst.Gain);
            writer.Write((sbyte)inst.TrackIndex);
            writer.Write((sbyte)inst.patch.Master_Polyphony_Voices);
            writer.Write((sbyte)inst.patch.Master_Unison_Voices);
            writer.Write((sbyte)inst.patch.Master_Unison_Detune);
            writer.Write((sbyte)inst.patch.Master_Unison_Pan);
            writer.Write((sbyte)inst.patch.OPX_NoiseCutOff);
            writer.Write((sbyte)inst.patch.OPX_NoiseReso);
            writer.Write((sbyte)inst.patch.OPX_NoiseAmp);
            writer.Write((sbyte)inst.patch.OPX_SatLimit);
            writer.Write((sbyte)inst.patch.OPX_SatGain);
            writer.Write((sbyte)inst.patch.OPZ_CutOff);
            writer.Write((sbyte)inst.patch.OPZ_Spread);
            writer.Write((sbyte)inst.patch.OPZ_Reso1);
            writer.Write((sbyte)inst.patch.OPZ_Reso2);
            writer.Write((sbyte)inst.patch.OPZ_Mode1);
            writer.Write((sbyte)inst.patch.OPZ_Mode2);
            writer.Write((sbyte)inst.patch.OPZ_Routing);
            writer.Write((sbyte)inst.patch.OPZ_Mix);
            writer.Write((sbyte)inst.patch.Pitch_Envelope);
            writer.Write((sbyte)inst.patch.Pitch_Transp);

            for (int nRow = 0; nRow < 8; nRow++)
                for (int nCol = 0; nCol < 8; nCol++)
                    writer.Write((sbyte)inst.patch.FM_Matrix[nRow][nCol]);

            var empty = new byte[] { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

            for (int i = 0; i < 6; i++) writer.Write((sbyte)inst.patch.OP_Waveform[i]); writer.Write(empty, 0, 3);
            for (int i = 0; i < 8; i++) writer.Write((sbyte)inst.patch.OP_Velocity[i]); writer.Write(empty, 0, 1);
            for (int i = 0; i < 6; i++) writer.Write((sbyte)inst.patch.OP_KeySync[i]); writer.Write(empty, 0, 3);
            for (int i = 0; i < 8; i++) writer.Write((sbyte)inst.patch.OP_OutVol[i]); writer.Write(empty, 0, 1);
            for (int i = 0; i < 8; i++) writer.Write((sbyte)inst.patch.OP_OutPan[i]); writer.Write(empty, 0, 1);
            for (int i = 0; i < 9; i++) writer.Write((sbyte)inst.patch.OP_KeyScale[i]);
            for (int i = 0; i < 8; i++) writer.Write((sbyte)inst.patch.OP_KeySc_Level[i].Length); writer.Write(empty, 0, 1);
            for (int i = 0; i < 9; i++) writer.Write((sbyte)inst.patch.OP_Env_Level[i].Length);
            for (int i = 0; i < 9; i++) writer.Write((sbyte)inst.patch.OP_Env_SustainLoopStart[i]);
            for (int i = 0; i < 9; i++) writer.Write((sbyte)inst.patch.OP_Env_SustainLoopEnd[i]);
            for (int i = 0; i < 6; i++) writer.Write((float)inst.patch.OP_Ratio[i]); writer.Write(empty, 0, 3 * 4);
            for (int i = 0; i < 6; i++) writer.Write((float)inst.patch.OP_Offset[i]); writer.Write(empty, 0, 3 * 4);

            writer.Flush();
        }

        // Reset to Zero position
        foreach (var stream in instWriters)
            stream.Seek(0, SeekOrigin.Begin);

        // Blend all streams into main stream
        for (int i = 0; i < instWriters[0].BaseStream.Length; i++)
            foreach (var inst in instWriters)
                sw.Write((byte)inst.BaseStream.ReadByte());

        foreach (var inst in Instruments)
        {
            if (inst.patch.OP_Env_RelTime[8][1] == 0)
            {
                inst.patch.OP_Env_RelTime[8][0] = 0.1f;
                inst.patch.OP_Env_RelTime[8][1] = 1.0f;
                inst.patch.OP_Env_RelTime[8][2] = 0.1f;
                inst.patch.OP_Env_Slope[8][0] = 0.5f;
                inst.patch.OP_Env_Slope[8][1] = 0.5f;
                inst.patch.OP_Env_Slope[8][2] = 0.5f;
                inst.patch.OP_Env_Level[8][0] = 0.5f;
                inst.patch.OP_Env_Level[8][1] = 0.5f;
                inst.patch.OP_Env_Level[8][2] = 0.5f;
            }      
        }

        // Check that structure is ok
        if (sw.BaseStream.Position != 1 + 250 * Instruments.Count)
            throw new Exception("Structure changed, please update reader");

        foreach (var inst in Instruments)
            for (int i = 0; i < 8; i++)
                for (int j = 0; j < inst.patch.OP_KeySc_Level[i].Length; j++) sw.Write((sbyte)inst.patch.OP_KeySc_Note[i][j]);

        foreach (var inst in Instruments)
            for (int i = 0; i < 8; i++)
                for (int j = 0; j < inst.patch.OP_KeySc_Level[i].Length; j++) sw.Write((sbyte)inst.patch.OP_KeySc_Level[i][j]);

        foreach (var inst in Instruments)
            for (int i = 0; i < 8; i++)
                for (int j = 0; j < inst.patch.OP_KeySc_Level[i].Length; j++) sw.Write(inst.patch.OP_KeySc_Slope[i][j]);

        foreach (var inst in Instruments)
            for (int i = 0; i < 9; i++)
                for (int j = 0; j < inst.patch.OP_Env_Level[i].Length; j++) sw.Write(inst.patch.OP_Env_Level[i][j]);

        foreach (var inst in Instruments)
            for (int i = 0; i < 9; i++)
                for (int j = 0; j < inst.patch.OP_Env_Level[i].Length; j++) sw.Write(inst.patch.OP_Env_Slope[i][j]);

        foreach (var inst in Instruments)
            for (int i = 0; i < 9; i++)
                for (int j = 0; j < inst.patch.OP_Env_Level[i].Length; j++) sw.Write(inst.patch.OP_Env_RelTime[i][j]);

        sw.Flush();
    }

    public static void SerilaizeReverb(List<ReverbData> Reverbs, BinaryWriter sw)
    {
        List<BinaryWriter> compWriters = new List<BinaryWriter>();

        sw.Write((byte)Reverbs.Count);
        if (Reverbs.Count == 0)
            return;

        foreach (var reverb in Reverbs)
        {
            var writer = new BinaryWriter(new MemoryStream());
            writer.Write((sbyte)(reverb.prmInputDiffuse1).TrackValue);
            writer.Write((sbyte)(reverb.prmInputDiffuse2).TrackValue);
            writer.Write((sbyte)(reverb.prmBandLimiter).TrackValue);
            writer.Write((sbyte)(reverb.prmTime).TrackValue);
            writer.Write((sbyte)(reverb.prmDamping).TrackValue);
            writer.Write((sbyte)(reverb.prmOutGain).TrackValue);
            compWriters.Add(writer);
        }

        // Reset to Zero position
        foreach (var stream in compWriters)
            stream.Seek(0, SeekOrigin.Begin);
        
        // Blend all streams into main stream
        for (int i = 0; i < compWriters[0].BaseStream.Length; i++)
            foreach (var comp in compWriters)
                sw.Write((byte)comp.BaseStream.ReadByte());

        sw.Flush();
    }

    public static void SerilaizeCompressor(List<CompressorData> Compressors, BinaryWriter sw)
    {
        List<BinaryWriter> compWriters = new List<BinaryWriter>();

        sw.Write((byte)Compressors.Count);
        if (Compressors.Count == 0)
            return;

        int nComp = 0;
        foreach (var comp in Compressors)
        {
            var writer = new BinaryWriter(new MemoryStream());
            writer.Write((sbyte)(comp.prmAttackTime).TrackValue);
            writer.Write((sbyte)(comp.prmReleaseTime).TrackValue);
            writer.Write((sbyte)(comp.prmThreshold).TrackValue);
            writer.Write((sbyte)(comp.prmMakeupGain).TrackValue);
            writer.Write((sbyte)(comp.prmRatio).TrackValue);
            compWriters.Add(writer);
        }

        // Reset to Zero position
        foreach (var stream in compWriters)
            stream.Seek(0, SeekOrigin.Begin);

        // Blend all streams into main stream
        for (int i = 0; i < compWriters[0].BaseStream.Length; i++)
            foreach (var comp in compWriters)
                sw.Write((byte)comp.BaseStream.ReadByte());

        sw.Flush();
    }

    // linear -> dB conversion
    public static float lin2dB(float lin) 
    {
	    const float LOG_2_DB = 6.02059991327962f;	// 20 / log( 10, 2)
        return (float)Math.Log(lin) * LOG_2_DB;
    }

    // dB -> linear conversion
    public static float dB2lin( float dB ) 
    {
	    const float DB_2_LOG = 0.11512925464970228420089957273422f;	// ln( 10 ) / 20
	    return (float)Math.Exp( dB * DB_2_LOG );
    }
}

