﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using ComponentAce.Compression.Libs.zlib;
using System.Reflection;
using LZF.NET;
using TinyFM8;
using System.Xml;

namespace KSDManager
{

    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    public struct FM8InternalStruct
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 12)]
        public byte[] Unknown_BaseHeader;

        public float Master_Polyphony_Voices;
        public float Master_Unison_Voices;
        public float Master_Unison_Detune;
        public float Master_Pitch_Tune;
        public float Master_Polyphony_Mono;

        public float Pitch_PBMode;
        public float Pitch_Transp;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] Unknown1C;

        public float Pitch_Time;
        public float Pitch_ToggleAuto;
        public float Pitch_ToggleOn;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] Unknown1D;

        public float Pitch_Envelope;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public byte[] Unknown1E;

        public float LFO1_ToggleOn;
        public int   LFO1_Waveform;
        public float LFO1_Rate;
        public float LFO1_Delay;
        public float LFO1_ToggleSync;
        public float LFO1_ToggleKeySync;
        public float LFO1_KeyScale;
        public float LFO1_VelScale;

        public float LFO2_ToggleOn;
        public int   LFO2_Waveform;
        public float LFO2_Rate;
        public float LFO2_Delay;
        public float LFO2_ToggleSync;
        public float LFO2_ToggleKeySync;
        public float LFO2_KeyScale;
        public float LFO2_VelScale;

        public float Eff_Overdrive_ToggleOn;
        public float Eff_Overdrive_Drive;
        public float Eff_Overdrive_Tone;
        public float Eff_Overdrive_Bass;
        public float Eff_Overdrive_Volume;

        public float Eff_TubeAmp_ToggleOn;
        public float Eff_TubeAmp_Drive;
        public float Eff_TubeAmp_Volume;

        public float Eff_Cabinet_ToggleOn;
        public float Eff_Cabinet_Type;
        public float Eff_Cabinet_Size;
        public float Eff_Cabinet_Air;
        public float Eff_Cabinet_Bass;
        public float Eff_Cabinet_Treble;

        public float Eff_ShelvingEQ_ToggleOn;
        public float Eff_ShelvingEQ_EQ1;
        public float Eff_ShelvingEQ_EQ2;
        public float Eff_ShelvingEQ_EQ3;
        public float Eff_ShelvingEQ_EQ4;
        public float Eff_ShelvingEQ_Volume;

        public float Eff_PeakEQ_ToggleOn;
        public float Eff_PeakEQ_EQ1;
        public float Eff_PeakEQ_EQ2;
        public float Eff_PeakEQ_Q1;
        public float Eff_PeakEQ_EQ3;
        public float Eff_PeakEQ_EQ4;
        public float Eff_PeakEQ_Q2;
        public float Eff_PeakEQ_Volume;

        public float Eff_TalkWah_ToggleOn;
        public float Eff_TalkWah_Mouth;
        public float Eff_TalkWah_ModWheel;
        public float Eff_TalkWah_Size;
        public float Eff_TalkWah_Bright;

        public float Eff_Phaser_ToggleOn;
        public float Eff_Phaser_Rate;
        public float Eff_Phaser_Color;
        public float Eff_Phaser_Rotate;
        public float Eff_Phaser_SweepMin;
        public float Eff_Phaser_SweepMax;
        public float Eff_Phaser_Sync;
        public float Eff_Phaser_DryWet;
        public float Eff_Phaser_Inv;
        public float Eff_Phaser_Notches;

        public float Eff_Flanger_ToggleOn;
        public float Eff_Flanger_Rate;
        public float Eff_Flanger_Static;
        public float Eff_Flanger_Depth;
        public float Eff_Flanger_Color;
        public float Eff_Flanger_Rotate;
        public float Eff_Flanger_Sync;
        public float Eff_Flanger_Inv;
        public float Eff_Flanger_DryWet;

        public float Eff_Tremolo_ToggleOn;
        public float Eff_Tremolo_Rate;
        public float Eff_Tremolo_Intensity;
        public float Eff_Tremolo_Sync;
        public float Eff_Tremolo_Stereo;
        public float Eff_Tremolo_Width;
        public float Eff_Tremolo_Attach;
        public float Eff_Tremolo_Decay;

        public float Eff_Reverb_ToggleOn;
        public float Eff_Reverb_DryWet;
        public float Eff_Reverb_Bright;
        public float Eff_Reverb_Time;        
        public float Eff_Reverb_Treble;

        public float Eff_PsycheDelay_ToggleOn;
        public float Eff_PsycheDelay_DryWet;
        public float Eff_PsycheDelay_Time;
        public float Eff_PsycheDelay_Reverse;
        public float Eff_PsycheDelay_Detune;
        public float Eff_PsycheDelay_Feedback;
        public float Eff_PsycheDelay_Tap;
        public float Eff_PsycheDelay_Pitch;
        public float Eff_PsycheDelay_Sync;
        public float Eff_PsycheDelay_Stereo;

        public float Eff_Chorus_Time;
        public float Eff_Chorus_Diffuse;
        public float Eff_Chorus_ModDepth;
        public float Eff_Chorus_Feedback;
        public float Eff_Chorus_HiCut;
        public float Eff_Chorus_LoCut;
        public float Eff_Chorus_Invert;
        public float Eff_Chorus_ToggleOn;
        public float Eff_Chorus_DryWet;
        public float Eff_Chorus_ModRate;
        public float Eff_Chorus_Sync;
        public float Eff_Chorus_SyncDelay;


        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 20)]
        public byte[] Unknown_EffectsData;

        public float LFO1_ToggleInvert;
        public float LFO2_ToggleInvert;

        public float Master_Unison_Pan;
        public float Master_Unison_Dynamic;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 28)]
        public byte[] Unknown3;

        public float OPZ_Routing;
        public float OPZ_Mix;
        public float OPZ_Mode1;
        public float OPZ_Mode2;
        public float OPZ_CutOff;
        public float OPZ_Spread;

        public float OPZ_EnvAmt;
        public float OPZ_Reso1;
        public float OPZ_Reso2;

        public float OPX_NoiseCutOff;
        public float OPX_NoiseReso;
        public float OPX_NoiseAmp;
        public float OPX_SatGain;
        public float OPX_SatLimit;
        public float OPX_SatAsym;

        public float Master_Quality_Analog;
        public float Master_Quality_Digital;

        public float Effects_Amount;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 44)]
        public byte[] Unknown4B;        

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = (184))]
        public byte[] Unknown5;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_TempoSync;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Env_SustainLoopEnd;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Env_SustainLoopStart;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] Unknown6;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_KeySync;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public sbyte[] OP_Velocity;

        public byte Pitch_Velocity;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_KeyScaling;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_VelScaling;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_PB_Up;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_PB_Dn;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_Modulation;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_AfterTouch;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_Breath;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_Ctrl1;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_Ctrl2;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_In_Envelope;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1_Modulation;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1_AfterTouch;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1_Breath;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1_Ctrl1;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO1_Ctrl2;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2_Modulation;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2_AfterTouch;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2_Breath;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2_Ctrl1;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Mod_LFO2_Ctrl2;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public sbyte[] Unknown_Mod_Row;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Invert;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_PitchEnv;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Sustain;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Release;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Active;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public byte[] OP_Waveform;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public float[] OP_Ratio;
        
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 9)]
        public float[] OP_Offset;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 270)]
        public byte[] Unknown9;

    }

    public class UnknownBlock
    {
        public UnknownBlock()
        {
        }

        public UnknownBlock(string Description, int Address, byte[] Block)
        {
            this.Block = Block;
            this.Description = Description;
            this.Address = Address;
        }

        public byte[] Block;
        public string Description;
        public int    Address;
    }

    public class FM8Instrument
    {
        #region KSD Data

        public Dictionary<string, Stream> Chunks;

        public FM8InternalStruct Data;

        public string Name;
        public string Authour;
        public string Comment;

        public float Gain = 1f;
        public int MasterVolume = 80;
        public int TrackIndex = 0;

        public int[,] FM_Matrix = new int[8, 9];
        public int[] OP_OutVol = new int[9];
        public int[] OP_OutPan = new int[9];
        public int[] OP_IN = new int[9];

        public float[][] OP_Env_RelTime = new float[9][];
        public float[][] OP_Env_AbsTime = new float[9][];
        public float[][] OP_Env_Level = new float[9][];
        public float[][] OP_Env_Slope = new float[9][];

        public float[][] OP_KeySc_Note = new float[8][];
        public float[][] OP_KeySc_Level = new float[8][];
        public float[][] OP_KeySc_Slope = new float[8][];

        public List<UnknownBlock> UnknownParts = new List<UnknownBlock>();

        #endregion

        #region Utility static functions

        static private void CopyStream(System.IO.Stream input, System.IO.Stream output)
        {
            byte[] buffer = new byte[2000];
            int len;
            while ((len = input.Read(buffer, 0, 2000)) > 0)
            {
                output.Write(buffer, 0, len);
            }
            output.Flush();
        }

        static private string FlipString(string ch)
        {
            string s = "";
            for (int i = 0; i < ch.Length; i++)
                s = ch[i] + s;
            return s;
        }

        static private string BytesToString(char[] ch)
        {
            string s = "";
            for (int i = 0; i < ch.Length; i++)
                s = s + ch[i];
            return s;
        }

        static private T ReadStruct<T>(Stream fs)
        {
            byte[] buffer = new byte[Marshal.SizeOf(typeof(T))];
            fs.Read(buffer, 0,
                Marshal.SizeOf(typeof(T)));
            GCHandle handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);
            T temp = (T)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(T));
            handle.Free();
            return temp;
        }

        static private float[] ReadFloatArray(BinaryReader br)
        {
            int nCount = br.ReadByte() + 1;
            var Ret = new float[nCount];

            for (int i = 0; i < nCount; i++)
                Ret[i] = br.ReadSingle();

            return Ret;
        }

        static private int IndexOf(byte[] full, byte[] part, int startIndex)
        {
            for (int i = startIndex; i <= full.Length - part.Length; i++)
            {
                for (int j = 0; j < part.Length; j++)
                {
                    if (full[i + j] != part[j])
                        break;

                    if (j == part.Length - 1)
                        return i;
                }
            }
            return -1;
        }

        #endregion

        #region Static FM8 Loader

        static private void KSD_DecodeInfo(FM8Instrument data, MemoryStream stream)
        {
            // Find the start of the document
            var index = IndexOf(stream.GetBuffer(), Encoding.ASCII.GetBytes(@"<?xml version=""1.0"""), 0);
            if (index < 0)
                return;

            // Build XML document
            XmlDocument dom = new XmlDocument();
            stream.Position = index;
            dom.Load(stream);

            // Find node
            XmlNodeList nodes = dom.SelectNodes("//info//commonAttr");
            if (nodes.Count == 0)
                return;

            // load name
            var name = nodes[0].SelectSingleNode("DisplayName").InnerText;
            if (!string.IsNullOrEmpty(name))
                data.Name = name;

            // load authour
            var authour = nodes[0].SelectSingleNode("Author").InnerText;
            if (!string.IsNullOrEmpty(authour))
                data.Authour = authour;

            // Load comment
            var comment = nodes[0].SelectSingleNode("Comment").InnerText;
            if (!string.IsNullOrEmpty(comment))
                data.Comment = comment;
        }

        static private Dictionary<string, Stream> KSD_DecodeChunks(Stream inStream)
		{
            var Chunks = new Dictionary<string, Stream>();
            var binaryReader = new BinaryReader(inStream);

            int ChunkID = 0;
            inStream.Position = 0x3C;
            while (inStream.Position != inStream.Length)
            {
                // Read "atad" or "data"
                var DataTag = FlipString(BytesToString(binaryReader.ReadChars(4)));
                if (DataTag != "data")
                    throw new Exception("Incorrect file format");

                // Read size usally 2 or 4
                uint nSize = binaryReader.ReadUInt32();

                // Read tag type
                var ChunkType = FlipString(BytesToString(binaryReader.ReadChars(4)));

                // Read tag type
                var ZLibTag = FlipString(BytesToString(binaryReader.ReadChars(4)));

                // Read packed size 
                uint nPackedSize = binaryReader.ReadUInt32();

                // Read Unpacked size 
                uint nUnpackedSize = binaryReader.ReadUInt32();

                // Extract packed data
                var ChunkData = binaryReader.ReadBytes((int)nPackedSize);
                var chunkStream = new MemoryStream(ChunkData);

                // Create output filef
                var outStream = new MemoryStream();
                if (!Chunks.ContainsKey(ChunkType))
                    Chunks.Add(ChunkType, outStream);
                ZOutputStream outZStream = new ZOutputStream(outStream);

                // Perform inflate
                try
                {
                    CopyStream(chunkStream, outZStream);
                }
                finally
                {
                    outZStream.Flush();
                }
            }

            return Chunks;
		}
        
        static public FM8Instrument Load(string KSDFile)
        {
            var file = new FileStream(KSDFile, FileMode.Open, FileAccess.Read);
            var data = LoadFM8(file);
            file.Close();
            return data;
        }

        static public FM8Instrument LoadFM8(Stream stream)
        {
            // Create binary reader
            stream.Position = 0;
            var br = new BinaryReader(stream);

            // Find what stream type is it (NFM8 or KSD)
            if (Encoding.ASCII.GetString(br.ReadBytes(4)) == "-in-")
            {
                // Decode the KSD stream
                var Chunks = KSD_DecodeChunks(stream);
                if (!Chunks.ContainsKey("prst"))
                    return null;

                // Decode main instrument data
                var Data = LoadFM8InstrumentBlock(Chunks["prst"]); 
                Data.Chunks = Chunks;

                // Decode info chunk
                if (Chunks.ContainsKey("info"))
                {
                    KSD_DecodeInfo(Data, Chunks["info"] as MemoryStream);
                }

                // Read IntC section (if exists)
                if (Chunks.ContainsKey("intc"))
                {
                    var intc_chunk = Chunks["intc"];
                    var intc_reader = new BinaryReader(intc_chunk);
                    intc_chunk.Position = 0x11;
                    Data.MasterVolume = (int)intc_reader.ReadSingle();

                    // Convert MasterVolume to gain
                    Data.Gain = 0.000156f * Data.MasterVolume * Data.MasterVolume;
                }

                // Open file
                return Data;
            }
            else
            {
                // NFM8 stream
                stream.Position = 0;
                var Chunks = NFM8_Load(br);
                
                var Data = LoadFM8InstrumentBlock(Chunks["prst"]);
                
                Data.Chunks = Chunks;
                return Data;
            }
        }

        static private string NFM8_ReadString(BinaryReader br)
        {
            int len = br.ReadUInt16();
            int tmp = br.ReadUInt16();
            return Encoding.Unicode.GetString(br.ReadBytes(len * 2));
        }

        static private void NFM8_ReadUntilSection(BinaryReader br, List<string> items, string section)
        {
            int MaxSubSections = 15;

            // Find tag
            while (true)
            {
                // Read 4 letter tag 
                var HeaderTag = BitConverter.ToString(br.ReadBytes(0x05)).ToUpper();

                // check if tag reached
                if (HeaderTag == section)
                    break;

                if (MaxSubSections-- == 0)
                    throw new Exception("Could not find section");

                // if not skip data
                items.Add("SkipData=" + BitConverter.ToString(br.ReadBytes(15)));
            }
        }

        static private byte[] NFM8_ReadSection(BinaryReader br, List<string> items)
        {
            // Find hsin tag
            NFM8_ReadUntilSection(br, items, "68-73-69-6E-01");

            // Dump HSIN data
            items.Add("HSINData1=" + BitConverter.ToString(br.ReadBytes(23)));

            // Read Size
            var Size = br.ReadUInt32();

            // Dump HSIN data
            items.Add("HSINData2=" + BitConverter.ToString(br.ReadBytes(0x4)));

            return br.ReadBytes((int)Size + 4);
        }

        static private Dictionary<string, Stream> NFM8_Load(BinaryReader br)
        {
            // Create outputs
            var retVal = new Dictionary<string, Stream>();
            var items = new List<string>();

            // Read total size
            uint TotalSize = br.ReadUInt32();

            // Read Extra header bytes
            items.Add("Header=" + BitConverter.ToString(br.ReadBytes(0x0c - 4)));
            
            // Read sections
            var Sections = new List<byte[]>();
            while (br.BaseStream.Position < TotalSize)
                Sections.Add(NFM8_ReadSection(br, items));

            //for (int i = 0; i < Sections.Count; i++)
            //    using (var fs = new FileStream("c:\\out2_." + i.ToString(), FileMode.Create))
            //        fs.Write(Sections[i], 0, Sections[i].Length);

            foreach (var section in Sections)
            {
                switch (section[4])
                {
                    case 0x6C:
                    {
                        var sbr = new BinaryReader(new MemoryStream(section, 52, section.Length - 52));
                        items.Add("InstrumentName=" + NFM8_ReadString(sbr));
                        items.Add("Author=" + NFM8_ReadString(sbr));
                        items.Add("Company=" + NFM8_ReadString(sbr));
                        items.Add("Comments=" + NFM8_ReadString(sbr));
                        items.Add("Unknown-1=" + BitConverter.ToString(br.ReadBytes(0x28))); // Probably color, rating, etc...
                        items.Add("FM8-Tag-1=" + NFM8_ReadString(sbr));
                        //items.Add("InstrumentType=" + LoadNFM8_ReadString(sbr));

                        /*uint tagsCount = sbr.ReadUInt32();
                        for (int i = 0; i < tagsCount; i++)
                            items.Add("Attribute_" + (i + 1).ToString() + "=" + LoadNFM8_ReadString(sbr));*/
                        break;
                    }

                    case 0x79:
                    {
                        var sbr = new BinaryReader(new MemoryStream(section, 56, section.Length - 56));
                        var Tag = NFM8_ReadString(sbr);
                        if (Tag == "")
                            break;
                        items.Add("FM8-Tag-2=" + Tag);

                        for (int j = 0; j < 16; j++)
                        {
                            for (int i = 0; i < 8; i++)
                            {
                                uint SomeInt = sbr.ReadUInt32();
                                string VariableName = NFM8_ReadString(sbr);
                                uint CheckIfNotZero = sbr.ReadUInt32();
                                float Value = sbr.ReadSingle();
                                uint Index1 = sbr.ReadUInt32();
                                uint Index2 = sbr.ReadUInt32();
                                items.Add(j.ToString() + "_" + i.ToString() + string.Format("={0}, {1}, {2}, {3}, {4}, {5}", SomeInt, VariableName, CheckIfNotZero, Value, Index1, Index2));
                            }

                            uint Equal1 = sbr.ReadUInt32();
                            uint Equal8 = sbr.ReadUInt32();
                            items.Add(j.ToString() + "_EOL=" + string.Format("{0}, {1}", Equal1, Equal8));
                            if (Equal8 != 8)
                                break;
                            string FM8_Tag_3 = NFM8_ReadString(sbr);
                        }

                        break;
                    }

                    case 0x74:
                    {
                        // Load LZF block
                        var sbr = new BinaryReader(new MemoryStream(section));
                        NFM8_ReadUntilSection(sbr, items, "44-53-49-4E-01");
                        items.Add("prst_header=" + BitConverter.ToString(sbr.ReadBytes(20)));

                        var SizeOfData = sbr.ReadUInt32();
                        var PackedData = sbr.ReadBytes((int)SizeOfData);
                        var OutputData = new byte[1024 * 128];
                        var lzf = new CLZF();
                        var outLen = lzf.lzf_decompress(PackedData, PackedData.Length, OutputData, OutputData.Length);

                        var sbr2 = new BinaryReader(new MemoryStream(OutputData, 0, outLen));
                        //using (var file = new FileStream("c:\\out.out1", FileMode.Create))
                        //    KSDData.CopyStream(sbr2.BaseStream, file);
                        //sbr2.BaseStream.Position = 0;

                        // Read preheader
                        items.Add("prst_header=" + BitConverter.ToString(sbr2.ReadBytes(12)));
                        
                        // Load section
                        var data1 = NFM8_ReadSection(sbr2, items);
                        var data2 = NFM8_ReadSection(sbr2, items);

                        items.Add("prst_heade_1r=" + BitConverter.ToString(data1));

                        // Find E8MF
                        var E8MFIndex = IndexOf(data2, new byte[] { 0x45, 0x38, 0x4D, 0x46 }, 0);

                        var sbr3 = new BinaryReader(new MemoryStream(data2));
                        //using (var file = new FileStream("c:\\out.out2", FileMode.Create))
                        //    KSDData.CopyStream(sbr3.BaseStream, file);
                        //sbr3.BaseStream.Position = 0;

                        // Read data upto E8MF
                        items.Add("prst_heade_1r=" + BitConverter.ToString(sbr3.ReadBytes(E8MFIndex)));

                        // Read data
                        var data3 = sbr3.ReadBytes((int)sbr3.BaseStream.Length - (int)sbr3.BaseStream.Position);

                        // Internal instrument block
                        retVal.Add("prst", new MemoryStream(data3));

                        break;
                    }
                }
            }

            // Export Loaded values
            var sw = new StreamWriter(new MemoryStream());
            foreach (var item in items)
                sw.WriteLine(item);

            retVal.Add("info", sw.BaseStream);
            
            return retVal;
        }

        static private FM8Instrument LoadFM8InstrumentBlock(Stream stream)
        {
            var Data = new FM8Instrument();
            //stream.Position = 0;
            //using (var ds = new FileStream("c:\\out.out", FileMode.Create))
            //    KSDData.CopyStream(stream, ds);

            var br = new BinaryReader(stream);
            stream.Position = 0;

            // Read Magic header
            br.ReadBytes(12);

            // Read Instrument name (4 times)
            for (int i = 0; i < 4; i++)
            {
                // Read name
                int nSize = br.ReadInt32();
                var bytes = br.ReadBytes(nSize);

                // Store in data
                Data.Name = Encoding.ASCII.GetString(bytes);
                Data.UnknownParts.Add(new UnknownBlock("Name", 0, bytes));
            }
            
            // Read Block1
            Data.Data = ReadStruct<FM8InternalStruct>(stream);

            // Add all Unknown fileds to list
            var Fields = Data.Data.GetType().GetFields();
            foreach (var field in Fields)
            {
                if (field.Name.StartsWith("Unknown", StringComparison.InvariantCultureIgnoreCase))
                    Data.UnknownParts.Add(new UnknownBlock(field.Name, 0, (byte[])field.GetValue(Data.Data)));
            }

            // Read EnvelopeAmp
            for (int i = 0; i < 9; i++) Data.OP_Env_RelTime[i] = ReadFloatArray(br);
            for (int i = 0; i < 9; i++) Data.OP_Env_Level[i] = ReadFloatArray(br);
            for (int i = 0; i < 9; i++) Data.OP_Env_Slope[i] = ReadFloatArray(br);

            // Post load processing
            for (int i = 0; i < 9; i++)
            {
                // Convert MS->Sec
                for (int j = 0; j < Data.OP_Env_RelTime[i].Length; j++)
                    Data.OP_Env_RelTime[i][j] /= 1000f;

                // Build Abs Table
                Data.OP_Env_AbsTime[i] = new float[Data.OP_Env_RelTime[i].Length + 1];
                for (int j = 0; j < Data.OP_Env_RelTime[i].Length; j++)
                    Data.OP_Env_AbsTime[i][j + 1] = Data.OP_Env_AbsTime[i][j] + Data.OP_Env_RelTime[i][j];
            }

            // Read KeySc
            for (int i = 0; i < 8; i++) Data.OP_KeySc_Note[i] = ReadFloatArray(br);
            for (int i = 0; i < 8; i++) Data.OP_KeySc_Level[i] = ReadFloatArray(br);
            for (int i = 0; i < 8; i++) Data.OP_KeySc_Slope[i] = ReadFloatArray(br);

            // Read FM Matrix
            for (int nOp = 0; nOp < 9; nOp++)
            {
                // Read Matrix values
                for (int i = 0; i < 8; i++)
                {
                    Data.FM_Matrix[i, nOp] = br.ReadByte();
                    br.ReadBytes(3);
                    //Data.UnknownParts.Add(new UnknownBlock("FM_Matrix_Gap1", (int)fs.Position, br.ReadBytes(3)));
                }

                // Read OutVol, OutPan and IN volume
                Data.OP_OutVol[nOp] = br.ReadByte();
                Data.UnknownParts.Add(new UnknownBlock("FM_Matrix_Gap2", (int)stream.Position, br.ReadBytes(3)));
                Data.OP_OutPan[nOp] = br.ReadInt32();
                Data.OP_IN[nOp] = br.ReadInt32(); 
            }

            // Read unread data
            Data.UnknownParts.Add(new UnknownBlock("Last_Block", (int)stream.Position, br.ReadBytes((int)(stream.Length - stream.Position))));

            return Data;
        }

        #endregion

        #region Public methods

        public PatchData ToPatchData()
        {
            var pd = new PatchData();
            pd.DisplayName = Name;
            pd.AuthorName  = Authour;
            pd.Comment     = Comment; 

            pd.Master_Polyphony_Voices = (byte)Data.Master_Polyphony_Voices;
            pd.Master_Unison_Voices    = (byte)Data.Master_Unison_Voices;
            pd.Master_Unison_Detune    = (byte)Data.Master_Unison_Detune;
            pd.Master_Unison_Pan       = (byte)Data.Master_Unison_Pan;
            
            pd.OPX_NoiseCutOff         = (byte)Data.OPX_NoiseCutOff;
            pd.OPX_NoiseReso           = (byte)Data.OPX_NoiseReso;
            pd.OPX_NoiseAmp            = (byte)Data.OPX_NoiseAmp;
            pd.OPX_SatGain             = (byte)Data.OPX_SatGain;
            pd.OPX_SatLimit            = (byte)Data.OPX_SatLimit;
            
            pd.OPZ_CutOff              = (byte)Data.OPZ_CutOff;
            pd.OPZ_Spread              = (byte)Data.OPZ_Spread;
            pd.OPZ_Reso1               = (byte)Data.OPZ_Reso1;
            pd.OPZ_Reso2               = (byte)Data.OPZ_Reso2;
            pd.OPZ_Mode1               = (byte)Data.OPZ_Mode1;
            pd.OPZ_Mode2               = (byte)Data.OPZ_Mode2;
            pd.OPZ_Routing             = (byte)Data.OPZ_Routing;
            pd.OPZ_Mix                 = (byte)Data.OPZ_Mix;
            
            pd.Pitch_Envelope          = (byte)Data.Pitch_Envelope;
            pd.Pitch_Transp            = (sbyte)Data.Pitch_Transp;

            pd.FM_Matrix = new byte[8][];
            for (int nRow = 0; nRow < 8; nRow++)
            {
                pd.FM_Matrix[nRow] = new byte[8];
                for (int nCol = 0; nCol < 8; nCol++)
                    pd.FM_Matrix[nRow][nCol] = (byte)FM_Matrix[nRow, nCol];
            }

            // Waveform converstion
            // [ 0] Sin            => [0] Sin
            // [ 1] Parabol        => [1] Parabol
            // [ 2] Triangle       => [2] Triangle
            // [ 3] Square         => [3] Square
            // [ 4] Sawtooth       => [4] Sawtooth 
            // [ 5] Soft Square    => [5] Soft Square
            // [ 6] Soft Tristate  => [6] Soft Tristate
            // [ 7] Short Tristate => [7] Short Tristate
            // [ 8] PWM Ramp Mod   => [4] Sawtooth 
            // [ 9] 1+3+5 Square   => [5] Soft Square
            // [10] 1+2+3 Saw      => [4] Sawtooth
            // [11] 1+2            => [0] Sin
            // [12] 1+3            => [0] Sin
            // [13] 1+4            => [0] Sin
            // [14] 1+5            => [0] Sin
            // [15] 1+6            => [0] Sin
            // [16] 1+7            => [0] Sin
            // [17] 1+8            => [0] Sin
            // [18] 2nd Formant    => [0] Sawtooth 
            // [19] 3rd Formant    => [4] Sawtooth 
            // [20] 4th Formant    => [4] Sawtooth 
            // [21] 5th Formant    => [4] Sawtooth 
            // [22] 6th Formant    => [4] Sawtooth 
            // [23] 8th Formant    => [4] Sawtooth 
            // [24] 10th Formant   => [4] Sawtooth 
            // [24] TX Wave 2      => [4] Sawtooth 
            // [25] TX Wave 3      => [4] Sawtooth 
            // [26] TX Wave 4      => [4] Sawtooth 
            // [27] TX Wave 5      => [4] Sawtooth 
            // [28] TX Wave 6      => [4] Sawtooth 
            // [29] TX Wave 7      => [4] Sawtooth 
            // [30] TX Wave 8      => [4] Sawtooth 
            byte[] WaveformConvert = { 0, 1, 2, 3, 4, 5, 6, 7, 4, 5, 4, 0, 0, 0, 0, 0, 0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4 };

            for (int i = 0; i < 8; i++) pd.OP_Active[i]               = (byte)Data.OP_Active[i];
            for (int i = 0; i < 6; i++) pd.OP_Waveform[i]             = (byte)WaveformConvert[Data.OP_Waveform[i]];
            for (int i = 0; i < 8; i++) pd.OP_Velocity[i]             = (sbyte)Data.OP_Velocity[i];
            for (int i = 0; i < 6; i++) pd.OP_KeySync[i]              = (byte)Data.OP_KeySync[i];
            for (int i = 0; i < 8; i++) pd.OP_OutVol[i]               = (byte)OP_OutVol[i];
            for (int i = 0; i < 8; i++) pd.OP_OutPan[i]               = (sbyte)OP_OutPan[i];
            for (int i = 0; i < 9; i++) pd.OP_KeyScale[i]             = (byte)Data.OP_KeyScaling[i];
            for (int i = 0; i < 9; i++) pd.OP_Env_SustainLoopStart[i] = (sbyte)Data.OP_Env_SustainLoopStart[i];
            for (int i = 0; i < 9; i++) pd.OP_Env_SustainLoopEnd[i]   = (sbyte)Data.OP_Env_SustainLoopEnd[i];
            for (int i = 0; i < 6; i++) pd.OP_Ratio[i]                = Data.OP_Ratio[i];
            for (int i = 0; i < 6; i++) pd.OP_Offset[i]               = Data.OP_Offset[i];

            for (int iEnv = 0; iEnv < 9; iEnv++)
            {
                pd.OP_Env_RelTime[iEnv] = OP_Env_RelTime[iEnv];
                pd.OP_Env_Level[iEnv]   = OP_Env_Level[iEnv];
                pd.OP_Env_Slope[iEnv]   = OP_Env_Slope[iEnv];
            }

            for (int iKeySc = 0; iKeySc < 8; iKeySc++)
            {
                pd.OP_KeySc_Note[iKeySc] = OP_KeySc_Note[iKeySc];
                pd.OP_KeySc_Level[iKeySc] = OP_KeySc_Level[iKeySc];
                pd.OP_KeySc_Slope[iKeySc] = OP_KeySc_Slope[iKeySc];
            }

            return pd;
        }

        #endregion
    }
}
