(** Generic utilities. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(** Executes the first function and guarantees the second function is run (even on exceptions). *)
let finally fn finalizer =
	let ret =
		try fn ()
		with x -> finalizer (); raise x
	in

	finalizer();
	ret

(** Converts a char into a string. *)
let string_of_char c =
	String.make 1 c

(** Converts a char list into a string. *)
let string_of_chars lst =
	let str_arr = Array.of_list lst in
	let str = String.create (Array.length str_arr) in

	Array.iteri (fun idx v -> str.[idx] <- v) str_arr;
	str

(** Returns the any non-unique object in the list or None. *)
let get_nonunique_item lst =
	let rec test left =
		match left with
		| [] -> None
		| x :: y :: _ when x = y -> Some x
		| _ :: tl -> test tl
	in

	test (Sort.list (<) lst)

(** Replaces the association list item with the given item. Preserves order. Throws Not_found if the value is not found. *)
let assoc_replace (k,v) lst =
	let rec do_it lst acc found =
		match lst with
		| [] -> if found then acc else raise Not_found
		| (lk,_) :: t when k = lk -> do_it t ((k, v) :: acc) true
		| x :: t -> do_it t (x :: acc) found
	in

	List.rev (do_it lst [] false)

(** Replaces or inserts the association list item with the given item.
    New items are inserted to the start, otherwise order is presrved. *)
let assoc_replace_or_insert keyval lst =
	try assoc_replace keyval lst
	with Not_found -> keyval :: lst

(** Converts a string into a char list. *)
let string_to_chars str =
	let lst = ref [] in
	String.iter (fun c -> lst := c :: !lst) str;
	List.rev !lst

(** List iterator that gives the item index while iterating. *)
let list_iteri f l =
	let rec iter idx lst =
		match lst with
		| [] -> ()
		| x :: tl -> f idx x; iter (idx + 1) tl
	in
	iter 0 l

(** List mapping with an index. *)
let list_mapi f lst =
	let rec map lst idx acc =
		match lst with
		| [] -> acc
		| hd :: tl -> map tl (idx + 1) ((f idx hd) :: acc)
	in
	List.rev (map lst 0 [])

(** Returns the index of the item in the list. Throws Not_found if item not in list. *)
let list_idx lst item =
	let rec find idx lst =
		match lst with
		| [] -> raise Not_found
		| x :: tl when x = item -> idx
		| _ :: tl -> find (idx + 1) tl
	in
	find 0 lst

(** Returns the index of the first item in the list that the predicate is true. Throws Not_found if all are false. *)
let list_find_idx pred lst =
	let rec find idx lst =
		match lst with
		| [] -> raise Not_found
		| x :: tl when pred(x) -> idx
		| _ :: tl -> find (idx + 1) tl
	in
	find 0 lst

(** Writes a text file *)
let write_file fn txt =
	let chn = open_out fn in
	let do_it () =
		output_string chn txt
	in

	finally do_it (fun () -> close_out chn)

(** Writes a file with the line strings being in a list. *)
let write_file_lines fn lines =
	write_file fn (String.concat "\n" lines)
