(** Debug writer for immediate bytecode. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

include Imbc

(** Returns a descriptive (very verbose) string from the bytecode. *)
let string_of_imbc imbc =
	(* Generate a string writer. *)
	let final_str = ref "" in
	let cur_indent = ref "" in

	let write_no_indent fmt = Printf.ksprintf (fun str -> final_str := !final_str ^ str ^ "\n") fmt in
	let write fmt = Printf.ksprintf (fun str -> final_str := !final_str ^ !cur_indent ^ str ^ "\n") fmt in
	let write_no_lf fmt = Printf.ksprintf (fun str -> final_str := !final_str ^ !cur_indent ^ str) fmt in

	let indent () = cur_indent := "    " ^ !cur_indent in
	let undent () = cur_indent := (String.sub !cur_indent 0 ((String.length !cur_indent)-4)) in

	(* Writes an opcode. *)
	let write_opcode op write_params =
		match op with
		| Label s -> if write_params then write_no_indent "Label %s" s else write_no_indent "Label"
		| Jump s -> if write_params then write "Jump %s" s else write "Jump"
		| JumpIfNot s -> if write_params then write "JumpIfNot %s" s else write "JumpIfNot"
		| PushLocal v -> if write_params then write "PushLocal %d" v else write "PushLocal"
		| PushPresetGlobal v -> if write_params then write "PushPresetGlobal #%d" v else write "PushPresetGlobal"
		| PushUninitGlobal v -> if write_params then write "PushUninitGlobal #%d" v else write "PushUninitGlobal"
		| PushConstant v -> if write_params then write "PushConstant #%d" v else write "PushConstant"
		| PushConstantByte v -> if write_params then write "PushConstantByte %d" v else write "PushConstantByte"
		| Pop -> write "Pop"
		| StoreLocal v -> if write_params then write "StoreLocal %d" v else write "StoreLocal"
		| StorePresetGlobal v -> if write_params then write "StorePresetGlobal #%d" v else write "StorePresetGlobal"
		| StoreUninitGlobal v -> if write_params then write "StoreUninitGlobal #%d" v else write "StoreUninitGlobal"
		| Return -> write "Return"
		| Call s -> if write_params then write "Call %s" s else write "Call"
		| Call_Cdecl v -> if write_params then write "Call_Cdecl %d" v else write "Call_Cdecl"
		| Call_CdeclFP v -> if write_params then write "Call_CdeclFP %d" v else write "Call_CdeclFP"
		| Call_Stdcall -> write "Call_Stdcall"
		| Call_StdcallFP -> write "Call_StdcallFP"
		| AsmFun (name,params,code,attrs) ->
			if write_params then
				write "AsmFun %s" name
			else
				begin
				let attr_str =
					String.concat ", " (List.map Program.string_of_fun_attr attrs)
				in

				write "AsmFun %s (params: %d, attrs: %s)" name params attr_str;
				indent ();
				List.iter (fun l -> write "%s" l) code;
				undent ()
				end

		| FixedPushLocal _
		| FixedPushPresetGlobal _
		| FixedPushUninitGlobal _
		| FixedPushConstant _
		| FixedPushConstantByte _
		| FixedStoreLocal _
		| FixedStorePresetGlobal _
		| FixedStoreUninitGlobal _
		| FixedCall _ -> write "%s" (Imbc.string_of_opcode op)
	in

	(* Writes a value (ivalue). *)
	let write_value v =
		match v with
		| Val_Int v -> write "Int: %ld (0x%lx)" v v
		| Val_Float v -> write "Float: %f" v
		| Val_String_Const v -> write "Const_String (idx: %d)" v
		| Val_External v -> write "External: '%s'" v
	in

	(* Write the settings. *)
	write "Heap size: %d" imbc.imbc_heapsize;
	write "";
	write "";

	(* Write the opcodes. *)
	write "Opcodes (count: %d)" (List.length imbc.imbc_opcodes);
	indent ();
	Util.list_iteri (fun idx o -> write_no_lf "%d. " idx; write_opcode o false) imbc.imbc_opcodes;
	undent ();
	write "";
	write "";

	(* Write the constants. *)
	write "Constants (count: %d)" (List.length imbc.imbc_constants);
	indent ();
	Util.list_iteri (fun idx c -> write_no_lf "%d. " idx; write_value c) imbc.imbc_constants;
	undent ();
	write "";
	write "";

	(* Write the strings. *)
	write "String constants (count: %d)" (List.length imbc.imbc_strings);
	indent ();
	Util.list_iteri (fun idx s -> write "%d. \"%s\"" idx (String.escaped s)) imbc.imbc_strings;
	undent ();
	write "";
	write "";

	(* Write the strings. *)
	write "External references (count: %d)" (List.length imbc.imbc_externals);
	indent ();
	Util.list_iteri (fun idx s -> write "%d \"%s\"" idx (String.escaped s)) imbc.imbc_externals;
	undent ();
	write "";
	write "";

	(* Write the preset globals. *)
	write "Preinitialized globals (count: %d)" (List.length imbc.imbc_preset_globals);
	indent ();
	let write_preset_global idx (name,v) =
		write "%d. Name: '%s'" idx name;
		indent (); write_value v; undent ()
	in
	Util.list_iteri write_preset_global imbc.imbc_preset_globals;
	undent ();
	write "";
	write "";

	(* Write the uninitialized globals. *)
	write "Uninitialized globals (count: %d)" (List.length imbc.imbc_uninit_globals);
	indent ();
	Util.list_iteri (fun idx s -> write "%d. Name: '%s'" idx (String.escaped s)) imbc.imbc_uninit_globals;
	undent ();
	write "";
	write "";

	(* Write the functions. *)
	let write_fun idx (name,f) =
		write "%d. Function '%s' (code opcodes: %d, parameters: %d, stack size: %d entries)" (idx + 1) name (List.length f.ifun_code) f.ifun_params f.ifun_stack_entries;
		indent ();
		List.iter (fun o -> write_opcode o true) f.ifun_code;
		undent ();
		write "";
	in

	write "Functions (count: %d)" (List.length imbc.imbc_funs);
	Util.list_iteri write_fun imbc.imbc_funs;
	write "";

	(* Done! *)
	!final_str
