(** Intermediate bytecode. *)

(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type opcode =
	| Label of string						(* Marks a label. *)
	| Jump of string						(* Unconditional jump. *)
	| JumpIfNot of string						(* Pops topmost and if it's zero, jumps. *)
	| PushLocal of int						(* Pushes a paramater (+5 onwards) or a local (-1 backwards) to the stack. *)
	| PushPresetGlobal of int					(* Pushes a preset global (index given) to the stack. *)
	| PushUninitGlobal of int					(* Pushes an uninitialized global (index given) to the stack. *)
	| PushConstant of int						(* Pushes a constant (index to the constant table). *)
	| PushConstantByte of int					(* Pushes a byte constant (raw value). *)
	| Pop								(* Pops the topmost item in stack. *)
	| StoreLocal of int						(* Stores the topmost to a paramater (+2 onwards) or to a local (-1 backwards). *)
	| StorePresetGlobal of int					(* Stores the topmost to a preset global (index given). *)
	| StoreUninitGlobal of int					(* Stores the topmost to an uninitialized global (index given). *)
	| Return							(* Leaves the function and returns the topmost item in stack. *)
	| Call of string						(* Calls a function. Pops the arguments and pushes the return value to the stack. Arguments are valuated from right to left. *)
	| Call_Cdecl of int						(* Calls a C function (cdecl calling convention). Argument count is given, and so much is popped (first is the function to call). The return value is pushed back. Arguments are valuated from right to left. *)
	| Call_CdeclFP of int						(* Calls a C function (cdecl calling convention). Argument count is given, and so much is popped (first is the function to call). The floating point return value is pushed back. Arguments are valuated from right to left. *)
	| Call_Stdcall							(* Calls a C function (stdcall calling convention). First argument is function to call, the rest are popped after call. The return value is pushed back. Arguments are valuated from right to left. *)
	| Call_StdcallFP						(* Calls a C function (stdcall calling convention). First argument is function to call, the rest are popped after call. The floating point return value is pushed back. Arguments are valuated from right to left. *)
	| AsmFun of string * int * (string list) * (Program.fun_attrs)	(* An opcode that is pure assembly. Parameters: descriptive name, parameter count and the assembly code itself. Arguments are valuated from left to right. *)
(* Fixed parameter opcodes (same as above, but the value "read" is fixed and not read from the byte stream. *)
	| FixedPushLocal of int
	| FixedPushPresetGlobal of int
	| FixedPushUninitGlobal of int
	| FixedPushConstant of int
	| FixedPushConstantByte of int 
	| FixedStoreLocal of int
	| FixedStorePresetGlobal of int
	| FixedStoreUninitGlobal of int
	| FixedCall of string

type ivalue =
	| Val_Int of int32
	| Val_Float of float
	| Val_String_Const of int
	| Val_External of string

type ifun =
{
	ifun_params : int;				(* Number of parameters. *)
	ifun_stack_entries : int;			(* How many 32-bit entries to allocate for local variables. *)
	ifun_code : opcode list;			(* The function code. *)
}

type imbc =
{
	imbc_heapsize : int;				(* The heap size to allocate. *)
	imbc_opcodes : opcode list;			(* List of all used opcodes (some may contain unused params). *)
	imbc_constants : ivalue list;			(* List of all used constants. *)
	imbc_strings : string list;			(* List of string constants. *)
	imbc_externals : string list;			(* List of referenced externals. *)
	imbc_preset_globals : (string * ivalue) list;	(* List of pre-initialized globals (name, value -list). *)
	imbc_uninit_globals : string list;		(* List of uninitialized globals (name list). *)

	imbc_funs : (string * ifun) list;		(* List of functions. *)
}

(** Converts the opcode to a string. *)
val string_of_opcode : opcode -> string

(** Tests if the two opcodes are semantically identical (some parameters may differ). *)
val opcode_eq : opcode -> opcode -> bool
