(*
    il4c  --  Compiler for the IL4 Lisp-ahtava langauge
    Copyright (C) 2007 Jere Sanisalo

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type options =
{
	opt_source_files : string list;
	opt_outname : string;
	opt_optimize_common_opcodes : bool;
	opt_optimize_sort_constants : bool;
	opt_optimize_sort_opcodes : bool;
	opt_save_debug_files : bool;
	opt_verbose : bool;
}

let default_options =
	{
		opt_source_files = [];
		opt_outname = "out";
		opt_optimize_common_opcodes = true;
		opt_optimize_sort_constants = true;
		opt_optimize_sort_opcodes = true;
		opt_save_debug_files = false;
		opt_verbose = false;
	}

let compile opts =
	(* Verbose printer. *)
	let debug_msg fmt = Printf.ksprintf (fun str -> if opts.opt_verbose then begin print_string str; print_newline () end) fmt in

	(* Returns a filename, given the extension. *)
	let fn ext = Printf.sprintf "%s.%s" opts.opt_outname ext in

	(* Read the source files and to the il4 parsing. *)
	debug_msg "Parsing source files..";
	let parse_tree = List.flatten (List.map (fun fn -> debug_msg "Parsing file '%s'.." fn; Il4_reader.parse fn) opts.opt_source_files) in

	(* Parse and optimize the program as a tree. *)
	debug_msg "Generating a syntax tree..";
	let prg = Program.generate parse_tree in
	if opts.opt_save_debug_files then
		begin
		debug_msg "Saving intermediate file '%s'.." (fn "ptree");
		Util.write_file (fn "ptree") (Program_writer.string_of_prg prg);
		end;

	debug_msg "Purging the program from unused functions/constants..";
	let prg = Program_optimizer.remove_unneeded prg in
	if opts.opt_save_debug_files then
		begin
		debug_msg "Saving intermediate file '%s'.." (fn "ptree2");
		Util.write_file (fn "ptree2") (Program_writer.string_of_prg prg);
		end;

	(* Convert to bytecode and optimize. *)
	debug_msg "Generating intermediate bytecode..";
	let imbc_prg = Program_to_imbc.convert prg in
	if opts.opt_save_debug_files then
		begin
		debug_msg "Saving intermediate file '%s'.." (fn "imbc");
		Util.write_file (fn "imbc") (Imbc_writer.string_of_imbc imbc_prg);
		end;

	let imbc_prg =
		if opts.opt_optimize_common_opcodes then
			begin
			debug_msg "Optimizing common opcode parameters to new opcodes..";
			let imbc_prg = Imbc_optimizer.optimize_common_opcode_params imbc_prg in
			if opts.opt_save_debug_files then
				begin
				debug_msg "Saving intermediate file '%s'.." (fn "imbc2");
				Util.write_file (fn "imbc2") (Imbc_writer.string_of_imbc imbc_prg);
				end;
			imbc_prg
			end
		else
			imbc_prg
	in

	let imbc_prg =
		if opts.opt_optimize_sort_constants then
			begin
			debug_msg "Sorting constants to rising order (by value)..";
			let imbc_prg = Imbc_optimizer.sort_constants imbc_prg in
			if opts.opt_save_debug_files then
				begin
				debug_msg "Saving intermediate file '%s'.." (fn "imbc3");
				Util.write_file (fn "imbc3") (Imbc_writer.string_of_imbc imbc_prg);
				end;
			imbc_prg
			end
		else
			imbc_prg
	in

	let imbc_prg =
		if opts.opt_optimize_sort_opcodes then
			begin
			debug_msg "Sorting opcodes to descending order (by use)..";
			let imbc_prg = Imbc_optimizer.sort_opcodes imbc_prg in
			if opts.opt_save_debug_files then
				begin
				debug_msg "Saving intermediate file '%s'.." (fn "imbc4");
				Util.write_file (fn "imbc4") (Imbc_writer.string_of_imbc imbc_prg);
				end;
			imbc_prg
			end
		else
			imbc_prg
	in

	(* Write the final assembly output. *)
	debug_msg "Saving final assembly file '%s'.." (fn "asm");
	Asm_writer.write_asm imbc_prg (fn "asm");

	(* Done *)
	debug_msg "Completed successfully!"

(*** MAIN ***)
let _ =
	(* Parse the command line*)
	let opts = ref default_options in
	let new_source_file fn =
		opts := { !opts with opt_source_files = fn :: !opts.opt_source_files }
	in
	let new_output_name n =
		opts := { !opts with opt_outname = n }
	in
	let disable_common_opcode_opt () =
		opts := { !opts with opt_optimize_common_opcodes = false }
	in
	let disable_sort_constants_opt () =
		opts := { !opts with opt_optimize_sort_constants = false }
	in
	let disable_sort_opcodes_opt () =
		opts := { !opts with opt_optimize_sort_opcodes = false }
	in
	let save_debug_files () =
		opts := { !opts with opt_save_debug_files = true }
	in
	let make_verbose () =
		opts := { !opts with opt_verbose = true }
	in

	let arg_spec = [
		("-o", Arg.String new_output_name, "<name>\t\t\tSpecifies the output file without the extension (default: 'out')");
		("-disable-common-opcode-opt", Arg.Unit disable_common_opcode_opt, "\tDisables the common opcode parameter optimization.");
		("-disable-constant-sort", Arg.Unit disable_sort_constants_opt, "\tDisables sorting the constants to a rising order");
		("-disable-opcode-sort", Arg.Unit disable_sort_opcodes_opt, "\tDisables sorting the opcodes to a descending order (by use)");
		("-save-debug", Arg.Unit save_debug_files, "\t\t\tSaves the intermediate debug files.");
		("-verbose", Arg.Unit make_verbose, "\t\t\tPrints more debug information on the console.");
		]
	in

	Arg.parse arg_spec new_source_file "il4c.exe [options] <source file(s)>";

	(* Check the arguments *)
	if (List.length !opts.opt_source_files) = 0 then failwith "No source files given!";
	opts := { !opts with opt_source_files = List.rev !opts.opt_source_files };

	(* Compile *)
	compile !opts
