//---------------------------------------------------------------------------
// NEOLoad by Jason "Joefish" Railton
//
// Version:     1.3
// Dated:       30 March 2016
//
//
// Purpose.
// --------
//
// A utility to load Atari-ST Neochrome (.NEO) format
// low-resolution 16-colour images, display and save as
// Windows Bitmap image files (.BMP).
//
// Also to load Windows Bitmap (.BMP) image files
// and save as Neochrome (.NEO) format files. The image
// should have been prepared as a 320x200 pixel image
// with approximately 16 colours. This software does
// not apply any scaling, re-colouring, shading or
// dithering to the image during conversion.
//
//
// License.
// --------
//
// Software is free to use for any purpose.
// No use is warranted or guaranteed.
//
// Permission given for free use of code and algorithms.
// The author would prefer if credit is displayed in any
// derivative works of this software.
//
//

//---------------------------------------------------------------------------
#include <vcl.h>
#include <math.h>
#include <stdio.h>
#pragma hdrstop

#include "NEOLoad1.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TFormNeoLoad *FormNeoLoad;
//---------------------------------------------------------------------------
__fastcall TFormNeoLoad::TFormNeoLoad(TComponent* Owner)
        : TForm(Owner)
{
    int     iN;

    // Get the colour panel control objects into an array:
    m_pnlColour[ 0] =  pnlColour0;
    m_pnlColour[ 1] =  pnlColour1;
    m_pnlColour[ 2] =  pnlColour2;
    m_pnlColour[ 3] =  pnlColour3;
    m_pnlColour[ 4] =  pnlColour4;
    m_pnlColour[ 5] =  pnlColour5;
    m_pnlColour[ 6] =  pnlColour6;
    m_pnlColour[ 7] =  pnlColour7;
    m_pnlColour[ 8] =  pnlColour8;
    m_pnlColour[ 9] =  pnlColour9;
    m_pnlColour[10] = pnlColour10;
    m_pnlColour[11] = pnlColour11;
    m_pnlColour[12] = pnlColour12;
    m_pnlColour[13] = pnlColour13;
    m_pnlColour[14] = pnlColour14;
    m_pnlColour[15] = pnlColour15;

    // No colour currently selected:
    m_iColSel = -1;

    // Initialise the BMP->NEO palette as white:
    for (iN=0; iN<16; ++iN)
    {
        m_colBmpPalette[iN] = clWhite;
        m_iRGB[iN][0] = 7;
        m_iRGB[iN][1] = 7;
        m_iRGB[iN][2] = 7;
        m_pnlColour[iN]->Color = clWhite;
        m_pnlColour[iN]->Caption = "";
    }


}
//---------------------------------------------------------------------------
void __fastcall TFormNeoLoad::FormCreate(TObject *Sender)
{
    TImage * imgPic1 = FormNeoLoad->ImgNeo;
    TImage * imgPal1 = FormNeoLoad->ImgNeoPalette;

    TImage * imgPic2 = FormNeoLoad->ImgBmp;
    TImage * imgPal2 = FormNeoLoad->ImgBmpPalette;


    // Set both pictures and palette sidebars to white:
    imgPic1->Canvas->Brush->Color = clWhite;
    imgPic1->Canvas->FillRect(Rect(0, 0, imgPic1->Width, imgPic1->Height));

    imgPal1->Canvas->Brush->Color = clWhite;
    imgPal1->Canvas->FillRect(Rect(0, 0, imgPal1->Width, imgPal1->Height));

    imgPic2->Canvas->Brush->Color = clWhite;
    imgPic2->Canvas->FillRect(Rect(0, 0, imgPic2->Width, imgPic2->Height));

    imgPal2->Canvas->Brush->Color = clWhite;
    imgPal2->Canvas->FillRect(Rect(0, 0, imgPal2->Width, imgPal2->Height));

    // Narrow the form to hide the palette swaps:
    FormNeoLoad->ClientWidth = 377;
}
//---------------------------------------------------------------------------


void __fastcall TFormNeoLoad::BtnExitClick(TObject *Sender)
{
    FormNeoLoad->Close();
}
//---------------------------------------------------------------------------

// Load an Atari-ST format Neochrome file as binary data,
// then convert and paint it into the upper image panel:
void __fastcall TFormNeoLoad::BtnLoadNeoClick(TObject *Sender)
{
    int             iFileHandle;
    bool            bLoadOk;
    int             iLoadSize;
    unsigned char   bNeoPic[32768];
    TColor          colNeoPalette[16];
    int             iN;
    int             iPtr, iInk;
    int             iR, iG, iB;
    int             iX, iY, iZ;
    unsigned int    uW0, uW1, uW2, uW3;
    unsigned int    uBit;
    TImage * imgPic = FormNeoLoad->ImgNeo;
    TImage * imgPal = FormNeoLoad->ImgNeoPalette;


    if (FormNeoLoad->dlgLoadNeo->Execute())
    {
        bLoadOk = false;
        iFileHandle = FileOpen(FormNeoLoad->dlgLoadNeo->FileName, fmOpenRead);
        if (iFileHandle > 0)
        {
            // Load 32128 bytes:
            iLoadSize = FileRead(iFileHandle, bNeoPic, 32128);
            if (iLoadSize == 32128)
            {
                bLoadOk = true;

                // Skip two words (Flag byte=0, assume resolution=0)
                // Extract the palette from the next 16 words, and
                // convert into a temporary array of TColor:
                for (iN=0; iN<16; ++iN)
                {
                    iR = floor((bNeoPic[iN*2 + 4] & 0x07) * 255.0 / 7.0);
                    iG = floor((bNeoPic[iN*2 + 5] & 0x70) * 255.0 / 7.0 / 16.0);
                    iB = floor((bNeoPic[iN*2 + 5] & 0x07) * 255.0 / 7.0);

                    colNeoPalette[iN] = TColor(RGB(iR, iG, iB));

                    // Paint each palette entry in the sidebar:
                    imgPal->Canvas->Brush->Color = colNeoPalette[iN];
                    imgPal->Canvas->FillRect(Rect(0, iN*8, 16, iN*8 + 8));
                }
                imgPal->Update();

                // Address pointer, past 128b header:
                iPtr = 128;

                // 200 rows of image:
                for (iY=0; iY<200; ++iY)
                {
                    // 20 column blocks:
                    for (iX=0; iX<20; ++iX)
                    {
                        // Fetch the 4 words that make up the
                        // next 16 pixels across 4 bitplanes:
                        uW0 = bNeoPic[iPtr+0] * 256 + bNeoPic[iPtr+1];
                        uW1 = bNeoPic[iPtr+2] * 256 + bNeoPic[iPtr+3];
                        uW2 = bNeoPic[iPtr+4] * 256 + bNeoPic[iPtr+5];
                        uW3 = bNeoPic[iPtr+6] * 256 + bNeoPic[iPtr+7];

                        // The first pixel is found in the highest bit:
                        uBit = 0x8000;

                        // 16 pixels to process:
                        for (iZ=0; iZ<16; ++iZ)
                        {
                            // Work out the colour index:
                            iInk = 0;
                            if (uW0 & uBit) iInk += 1;
                            if (uW1 & uBit) iInk += 2;
                            if (uW2 & uBit) iInk += 4;
                            if (uW3 & uBit) iInk += 8;

                            // Plot a pixel of that Tcolor:
                            imgPic->Canvas->Pixels[iX*16 + iZ][iY] = colNeoPalette[iInk];

                            uBit >>= 1;
                        }
                        iPtr += 8;
                    }

                    // Update picture after each row:
                    imgPic->Update();
                }
            }

            FileClose(iFileHandle);
        }

        if (!bLoadOk)
        {
              Application->MessageBox("Loading Error", NULL, MB_OK);
        }
    }
}
//---------------------------------------------------------------------------

// Simple to save a bitmap using the native format of the image object:
void __fastcall TFormNeoLoad::BtnSaveBmpClick(TObject *Sender)
{
     if (FormNeoLoad->dlgSaveBmp->Execute())
     {
        FormNeoLoad->ImgNeo->Picture->Bitmap->SaveToFile(FormNeoLoad->dlgSaveBmp->FileName);
     }
}
//---------------------------------------------------------------------------

// Load a bitmap into the lower image panel, then
// scan the pixels to get a palette and store it:
void __fastcall TFormNeoLoad::BtnLoadBmpClick(TObject *Sender)
{
    TColor  colPixel;
    int     iInk, iInkCount;
    int     iX, iY;
    int     iR, iG, iB;
    int     iN;
    TImage * imgPic = FormNeoLoad->ImgBmp;
    TImage * imgPal = FormNeoLoad->ImgBmpPalette;
    char    sRGB[64];


    if (FormNeoLoad->dlgLoadBmp->Execute())
    {
        // Load the image using the native formats of the image object:
        imgPic->Picture->Bitmap->LoadFromFile(FormNeoLoad->dlgLoadBmp->FileName);

        // Set the palette to magenta:
        for (iN=0; iN<16; ++iN)
        {
            m_colBmpPalette[iN] = TColor(0x00FF00FF);
            m_iRGB[iN][0] = 7;
            m_iRGB[iN][1] = 0;
            m_iRGB[iN][2] = 7;

            // Including the palette swap panels:
            m_pnlColour[iN]->Color = TColor(0x00FF00FF);
            m_pnlColour[iN]->Caption = "";
        }
        // Paint the palette sidebar magenta:
        imgPal->Canvas->Brush->Color = TColor(0x00FF00FF);
        imgPal->Canvas->FillRect(Rect(0, 0, imgPal->Width, imgPal->Height));

        // Start to discover colours:
        iInkCount = 0;

        // Scan the 320x200 grid of pixels:
        for (iY=0; iY<200; ++iY)
        {
            for (iX=0; iX<320; ++iX)
            {
                // Get a pixel colour:
                colPixel = imgPic->Canvas->Pixels[iX][iY];

                // Reduce it to 9 bits:
                colPixel = TColor((unsigned int)colPixel & 0x00E0E0E0);

                // See if we already have a match. Note that we don't
                // need an exact match for the 24-bit colour. We're
                // only matching on the 9-bit value, so several colours
                // may come together during the conversion:
                for (iInk=0; iInk<iInkCount; iInk++)
                {
                    if (iInk<16 && colPixel==TColor((unsigned int)m_colBmpPalette[iInk] & 0x00E0E0E0))
                    {
                        break;
                    }
                }

                // No match means a new colour:
                if (iInk==iInkCount)
                {
                    // Is there still space in the palette?
                    iInkCount++;
                    if (iInk<16)
                    {
                        // If so, work out the colour's 3+3+3 bit RGB value:
                        m_iRGB[iInk][0] = (colPixel & 0x000000E0) >> 5;
                        m_iRGB[iInk][1] = (colPixel & 0x0000E000) >> 13;
                        m_iRGB[iInk][2] = (colPixel & 0x00E00000) >> 21;

                        // Convert that back into a 24-bit colour for display:
                        iR = floor(m_iRGB[iInk][0] * 255.0 / 7.0);
                        iG = floor(m_iRGB[iInk][1] * 255.0 / 7.0);
                        iB = floor(m_iRGB[iInk][2] * 255.0 / 7.0);
                        m_colBmpPalette[iInk] = TColor(RGB(iR, iG, iB));

                        // Show the colour in the palette sidebar:
                        imgPal->Canvas->Brush->Color = m_colBmpPalette[iInk];
                        imgPal->Canvas->FillRect(Rect(0, iInk*8, 16, iInk*8 + 8));

                        // Display the colour in one of the palette swap panels:
                        sprintf(sRGB, "$%d%d%d", m_iRGB[iInk][0], m_iRGB[iInk][1], m_iRGB[iInk][2]);
                        m_pnlColour[iInk]->Color = m_colBmpPalette[iInk];
                        m_pnlColour[iInk]->Caption = sRGB;

                        // This just sets the text of the palette swap panel to black
                        // or white in contrast to its displayed colour:
                        if (m_iRGB[iInk][0]*2 + m_iRGB[iInk][1]*4 + m_iRGB[iInk][2] > 21)
                        {
                            m_pnlColour[iInk]->Font->Color = clBlack;
                        }
                        else
                        {
                            m_pnlColour[iInk]->Font->Color = clWhite;
                        }

                    }
                }
            }
        }

        // Warn if more than 16 colours were found:
        if (iInkCount>16)
        {
            Application->MessageBox("More than 16 colours identified.", NULL, MB_OK);
        }
    }
}
//---------------------------------------------------------------------------

// Toggle visibility of the palette swap controls:
void __fastcall TFormNeoLoad::spbPaletteClick(TObject *Sender)
{
    // If the button was just pressed and stays down:
    if (FormNeoLoad->spbPalette->Down)
    {
        // Widen the form to show the palette swap panels:
        FormNeoLoad->ClientWidth = 482;
    }

    // But if the button was just popped up:
    else
    {
        // Undo any half-done swap:
        if (m_iColSel>=0)
        {
            m_pnlColour[m_iColSel]->BevelOuter = bvRaised;
        }
        m_iColSel = -1;

        // Narrow the form to hide the palette swap panels:
        FormNeoLoad->ClientWidth = 377;
    }
}
//---------------------------------------------------------------------------

// A click on a palette swap panel. All the panels
// are in an array, and we get the index number
// from the panel's 'Tag' property:
void __fastcall TFormNeoLoad::pnlColourClick(TObject *Sender)
{
    // The index of the panel just clicked is:
    int         iNew = ((TPanel*)Sender)->Tag;
    int         iR, iG, iB;
    TColor      colInk;
    TColor      colFont;
    AnsiString  sCaption;
    TImage * imgPal = FormNeoLoad->ImgBmpPalette;


    if (m_iColSel < 0)
    {
        // If nothing already selected, select this panel
        // and highlight it by insetting its border:
        m_iColSel = iNew;
        m_pnlColour[m_iColSel]->BevelOuter = bvLowered;
    }
    else
    {
        // If this is a second click, swap details with the
        // previous selection.

        // First, get a temporary cache of the previously clicked
        // palette entry, its RGB value, the panel font colour,
        // and the RGB readout it displayed:
        colInk = m_colBmpPalette[m_iColSel];
        iR = m_iRGB[m_iColSel][0];
        iG = m_iRGB[m_iColSel][1];
        iB = m_iRGB[m_iColSel][2];
        sCaption = m_pnlColour[m_iColSel]->Caption;
        colFont = m_pnlColour[m_iColSel]->Font->Color;

        // Set the previously clicked panel with the values
        // of the just-clicked panel:
        m_colBmpPalette[m_iColSel] = m_colBmpPalette[iNew];
        m_iRGB[m_iColSel][0] = m_iRGB[iNew][0];
        m_iRGB[m_iColSel][1] = m_iRGB[iNew][1];
        m_iRGB[m_iColSel][2] = m_iRGB[iNew][2];
        m_pnlColour[m_iColSel]->Color = m_colBmpPalette[iNew];
        m_pnlColour[m_iColSel]->Caption = m_pnlColour[iNew]->Caption;
        m_pnlColour[m_iColSel]->Font->Color = m_pnlColour[iNew]->Font->Color;

        // Now set the just-clicked panel to the cached values:
        m_colBmpPalette[iNew] = colInk;
        m_iRGB[iNew][0] = iR;
        m_iRGB[iNew][1] = iG;
        m_iRGB[iNew][2] = iB;
        m_pnlColour[iNew]->Color = colInk;
        m_pnlColour[iNew]->Caption = sCaption;
        m_pnlColour[iNew]->Font->Color = colFont;

        // Re-paint the two palette entries in the palette sidebar:
        imgPal->Canvas->Brush->Color = m_colBmpPalette[m_iColSel];
        imgPal->Canvas->FillRect(Rect(0, m_iColSel*8, 16, m_iColSel*8 + 8));

        imgPal->Canvas->Brush->Color = m_colBmpPalette[iNew];
        imgPal->Canvas->FillRect(Rect(0, iNew*8, 16, iNew*8 + 8));

        // Clear the previous seletion and remove the highlight:
        m_pnlColour[m_iColSel]->BevelOuter = bvRaised;
        m_iColSel = -1;
    }

}
//---------------------------------------------------------------------------

// Save a Neochrome format file from the bottom image:
void __fastcall TFormNeoLoad::BtnSaveNeoClick(TObject *Sender)
{
    int             iFileHandle;
    int             iN;
    int             iX, iY, iZ;
    TColor          colPixel;
    unsigned int    uW0, uW1, uW2, uW3;
    unsigned int    uBitVal;
    unsigned int    uInk;
    unsigned char   uBytes[256];
    int             iWritten;
    bool            bWriteOk = false;
    TImage * imgPic = FormNeoLoad->ImgBmp;

    if (FormNeoLoad->dlgSaveNeo->Execute())
    {
        iFileHandle = FileCreate(FormNeoLoad->dlgSaveNeo->FileName);
        if (iFileHandle > 0)
        {
            bWriteOk = true;

            // Neochrome file header is 128 bytes:
            // Fill header with 0:
            for (iN=0; iN<128; ++iN)
            {
                uBytes[iN] = 0;
            }

            // Header is 1W Flag = 0, 1W Res=0,
            // 16W Palette:
            for (iN=0; iN<16; ++iN)
            {
                uBytes[iN*2+4] = m_iRGB[iN][0];
                uBytes[iN*2+5] = (m_iRGB[iN][1] << 4) | m_iRGB[iN][2];
            }

            // 12b empty filename:
            sprintf(&uBytes[36], "        .   ");

            // 3W colour animation definition,
            // 1W x-offset=0, 1W y-offset=0,
            // 1W width=320:
            uBytes[58] = 1;
            uBytes[59] = 64;

            // 1W height = 200:
            uBytes[61] = 200;

            // 33W unused.
            // Write 128-byte header to file:
            iWritten = FileWrite(iFileHandle, uBytes, 128);
            if (iWritten != 128) bWriteOk = false;


            // Neochrome file image data block is 32000 bytes,
            // made from 4 bitplanes, word-interleaved.
            // So, work with 8 bytes (4 words) at a time.

            // 200 screen rows:
            for (iY=0; iY<200; ++iY)
            {
                // 20 screen column blocks:
                for (iX=0; iX<20; ++iX)
                {

                    // 16 pixels per block row. These 16 pixels
                    // form 4 words of interleaved bitplanes:
                    uW0 = 0;
                    uW1 = 0;
                    uW2 = 0;
                    uW3 = 0;

                    // The first pixel is worth the high bit:
                    uBitVal = 0x8000;

                    // Scan 16 pixels:
                    for (iZ=0; iZ<16; iZ++)
                    {
                        // Get the pixel colour:
                        colPixel = imgPic->Canvas->Pixels[iX*16+iZ][iY];

                        // Trim it from 24 bits to 9 bits:
                        colPixel = TColor((unsigned int)colPixel & 0x00E0E0E0);

                        // Now look it up in the previously stored palette. IF
                        // it's not there, go with 0:
                        uInk = 0;
                        for (iN=0; iN<16; iN++)
                        {
                            if (colPixel==TColor((unsigned int)m_colBmpPalette[iN] & 0x00E0E0E0))
                            {
                                uInk = iN;
                                break;
                            }
                        }

                        // Stuff the colour index into the four
                        // interleaved words at the right pixel
                        // position:
                        if (uInk & 1) uW0 = uW0 | uBitVal;
                        if (uInk & 2) uW1 = uW1 | uBitVal;
                        if (uInk & 4) uW2 = uW2 | uBitVal;
                        if (uInk & 8) uW3 = uW3 | uBitVal;

                        // Next pixel is worth one bit lower:
                        uBitVal >>= 1;
                    }

                    // Store the 4 words in big-endian order in 8 bytes:
                    uBytes[0] = uW0 >> 8;
                    uBytes[1] = uW0 & 255;
                    uBytes[2] = uW1 >> 8;
                    uBytes[3] = uW1 & 255;
                    uBytes[4] = uW2 >> 8;
                    uBytes[5] = uW2 & 255;
                    uBytes[6] = uW3 >> 8;
                    uBytes[7] = uW3 & 255;

                    // Save 8 more bytes:
                    if (bWriteOk)
                    {
                        iWritten = FileWrite(iFileHandle, uBytes, 8);
                        if (iWritten != 8) bWriteOk = false;
                    }
                }
            }

            FileClose(iFileHandle);
        }

    }

    if (!bWriteOk)
    {
        Application->MessageBox("File write error.", NULL, MB_OK);
    }

}
//---------------------------------------------------------------------------

// Paint a preview of the lower image in the upper display,
// as it would appear if saved as a .NEO file and re-loaded:
void __fastcall TFormNeoLoad::btnPreviewClick(TObject *Sender)
{
    int             iN;
    int             iX, iY;
    TColor          colPixel;
    int             iInk;
    TImage * imgPic = FormNeoLoad->ImgBmp;
    TImage * imgTgt = FormNeoLoad->ImgNeo;
    TImage * imgPal = FormNeoLoad->ImgNeoPalette;

    // Repaint the upper palette:
    for (iN=0; iN<16; ++iN)
    {
        imgPal->Canvas->Brush->Color = m_colBmpPalette[iN];
        imgPal->Canvas->FillRect(Rect(0, iN*8, 16, iN*8 + 8));
    }
    imgPal->Update();

    // Scan the 320x200 image:
    for (iY=0; iY<200; ++iY)
    {
        for (iX=0; iX<320; ++iX)
        {
            // Get the pixel colour:
            colPixel = imgPic->Canvas->Pixels[iX][iY];

            // Trim it from 24 bits to 9 bits:
            colPixel = TColor((unsigned int)colPixel & 0x00E0E0E0);

            // Now look it up in the previously stored palette. IF
            // it's not there, go with 0:
            iInk = 0;
            for (iN=0; iN<16; iN++)
            {
                if (colPixel==TColor((unsigned int)m_colBmpPalette[iN] & 0x00E0E0E0))
                {
                    iInk = iN;
                    break;
                }
            }

            // Plot a pixel of that colour in the upper image:
            imgTgt->Canvas->Pixels[iX][iY] = m_colBmpPalette[iInk];
        }
        
        // Update picture after each row:
        imgTgt->Update();
    }

}
//---------------------------------------------------------------------------

