/*

	Geometry library
	
	M3D_geom.c

*/

#include "M3D_geom.h"

M3Dvertex	*vertex_list;
M3Dtriangle	*poly_list;
M3Dtexpoint	*tmap_list;
M3Dsort		*sort_list;
M3Dobject	*object_list;

int		num_vertices = 0;
int		num_polys = 0;
int		num_objects = 0;
int		visible_polys = 0;

float	M3Ddistance;


int		init_geom (void)
{

	if ((vertex_list = (M3Dvertex *)malloc(VERTEX_BUF * sizeof(M3Dvertex))) == NULL) {
		printf("Out of Memory! (vertices)\n");
		return (0);
	}
	if ((poly_list = (M3Dtriangle *)malloc(POLY_BUF * sizeof(M3Dtriangle))) == NULL) {
		printf("Out of Memory! (polys)\n");
		return (0);
	}
	if ((tmap_list = (M3Dtexpoint *)malloc(POLY_BUF * sizeof(M3Dtexpoint))) == NULL) {
		printf("Out of Memory! (texturepointlist)\n");
		return (0);
	}
	if ((sort_list = (M3Dsort *)malloc(POLY_BUF * sizeof(M3Dsort))) == NULL) {
		printf("Out of Memory!\n");
		return (0);
	}
	if ((object_list = (M3Dobject *)malloc(OBJECT_BUF * sizeof(M3Dobject))) == NULL) {
		printf("Out of Memory!\n");
		return (0);
	}

	return(1);
}


void	close_geom (void)
{
	free (vertex_list);
	free (poly_list);
	free (tmap_list);
	free (sort_list);
	free (object_list);
}


void	add_vertex (float x, float y, float z)
{

	vertex_list[num_vertices].local.x = x;
	vertex_list[num_vertices].local.y = y;
	vertex_list[num_vertices].local.z = z;
	vertex_list[num_vertices].local.w = 1;
	
	vertex_list[num_vertices].lnormal.x = 0;
	vertex_list[num_vertices].lnormal.y = 0;
	vertex_list[num_vertices].lnormal.z = 0;
	vertex_list[num_vertices].lnormal.w = 1;

	vertex_list[num_vertices].ncount = 0;
	
	num_vertices ++;

}


void	add_polygon (int a, int b, int c)
{
	M3Dvector	v1, v2, normal;

	poly_list[num_polys].v1 = a;
	poly_list[num_polys].v2 = b;
	poly_list[num_polys].v3 = c;
	poly_list[num_polys].material = 0;

	vector_sub(&vertex_list[a].local, &vertex_list[b].local,&v1);
	vector_sub(&vertex_list[b].local, &vertex_list[c].local,&v2);
		
	vector_cross(&v1, &v2, &normal);
	vector_normalize(&normal);

	vertex_list[a].lnormal.x += normal.x;
	vertex_list[a].lnormal.y += normal.y;
	vertex_list[a].lnormal.z += normal.z;
	vertex_list[a].ncount ++;

	vertex_list[b].lnormal.x += normal.x;
	vertex_list[b].lnormal.y += normal.y;
	vertex_list[b].lnormal.z += normal.z;
	vertex_list[b].ncount ++;

	vertex_list[c].lnormal.x += normal.x;
	vertex_list[c].lnormal.y += normal.y;
	vertex_list[c].lnormal.z += normal.z;
	vertex_list[c].ncount ++;

	vector_add(&normal, &vertex_list[a].local, &poly_list[num_polys].lnormal);

	num_polys ++;
}


void	add_object (int v_base, int num_v, int p_base, int num_p)
{

	object_list[num_objects].pivot.x = 0;
	object_list[num_objects].pivot.y = 0;
	object_list[num_objects].pivot.z = 0;
	object_list[num_objects].pivot.w = 1;
	
	object_list[num_objects].rotate_x = 0;
	object_list[num_objects].rotate_y = 0;
	object_list[num_objects].rotate_z = 0;

	object_list[num_objects].vertex_base = v_base;
	object_list[num_objects].num_vertices = num_v;

	object_list[num_objects].poly_base = p_base;
	object_list[num_objects].num_polys = num_p;

	object_list[num_objects].visible = TRUE;

	object_list[num_objects].num_child = 0;
	
	num_objects ++;

}


void	calc_vertex_normals (M3Dobject *obj)
{
	
	int		i;
	int		start, end;
	float	ratio;	

	/* Make averaged vertex normals */
	
	start = obj->vertex_base;
	end = obj->vertex_base + obj->num_vertices;

	for (i = start; i < end; i ++) {
		if(vertex_list[i].ncount > 0) {
			ratio = 1.0 / (float)(vertex_list[i].ncount);
			vertex_list[i].lnormal.x *= ratio;
			vertex_list[i].lnormal.y *= ratio;
			vertex_list[i].lnormal.z *= ratio;
			vector_normalize(&vertex_list[i].lnormal);
		}

		vertex_list[i].lnormal.x += vertex_list[i].local.x;
		vertex_list[i].lnormal.y += vertex_list[i].local.y;
		vertex_list[i].lnormal.z += vertex_list[i].local.z;
	}

}


void	pivot_rotate_object (M3Dobject *obj, int x, int y, int z, float pivot_x, float pivot_y, float pivot_z)
{
/*	int			i;*/
	M3Dmatrix	mat;
	M3Dvector	vec;

	obj->rotate_x += x;
	obj->rotate_y += y;
	obj->rotate_z += z;


	/* rotate pivot */

	init_matrix(mat);
	translate(mat, -pivot_x, -pivot_y, -pivot_z);
	rotate_XYZ(mat, x, y, z);
	translate(mat, pivot_x, pivot_y, pivot_z);

	vector_copy(&obj->pivot, &vec);
	mult_vector_matrix(&vec, mat, &obj->pivot);

	/* rotate children. */
/*	for (i = 0; i < obj->num_child; i ++) {
		pivot_rotate_object (&object_list[obj->child_list[i]], x, y, z,
							  obj->pivot.x + pivot_x, obj->pivot.y + pivot_y, obj->pivot.z + pivot_z);
	}*/
}


void	rotate_object (M3Dobject *obj, int x, int y, int z)
{
/*	int		i;*/

	obj->rotate_x += x;
	obj->rotate_y += y;
	obj->rotate_z += z;

	/* rotate children. */
/*	for (i = 0; i < obj->num_child; i ++) {
		pivot_rotate_object (&object_list[obj->child_list[i]], x, y, z, obj->pivot.x, obj->pivot.y, obj->pivot.z);
	}*/
}


void	translate_object (M3Dobject *obj, float x, float y, float z)
{
	obj->pivot.x += x;
	obj->pivot.y += y;
	obj->pivot.z += z;
}


void	set_pivot (M3Dobject *obj, float x, float y, float z)
{
	obj->pivot.x = x;
	obj->pivot.y = y;
	obj->pivot.z = z;
}


void	set_rotate (M3Dobject *obj, float x, float y, float z)
{
	obj->rotate_x = x;
	obj->rotate_y = y;
	obj->rotate_z = z;
}


void	transform_world (M3Dmatrix mat)
{

	int			i, j, start, end;
	M3Dvertex	*vptr;
	M3Dtriangle	*pptr;
	M3Dobject	*obj;
	M3Dlight	*light;
	M3Dmatrix	mat1, mat2;


	for (j = 0; j < num_objects; j ++) {

		obj = &object_list[j];
		
		if (obj->visible == TRUE) {
	
			init_matrix(mat1);
			init_matrix(mat2);
			
			rotate_XYZ(mat1, obj->rotate_x, obj->rotate_y, obj->rotate_z);
			translate(mat1, obj->pivot.x, obj->pivot.y, obj->pivot.z);

			mult_matrix_matrix(mat1, mat, mat2);
	
			/* transform vertices */
	
			start = obj->vertex_base;
			end = obj->vertex_base + obj->num_vertices;
		
			for (i = start; i < end; i ++) {
				vptr = &vertex_list[i];
				mult_vector_matrix(&vptr->local, mat2, &vptr->world);
		
				/* rotate normals */
				mult_vector_matrix(&vptr->lnormal, mat2, &vptr->wnormal);

				vptr->wnormal.x -= vptr->world.x;
				vptr->wnormal.y -= vptr->world.y;
				vptr->wnormal.z -= vptr->world.z;
			}
		
			start = obj->poly_base;
			end = obj->poly_base + obj->num_polys;

			/* rotate polys normals. */		
			for (i = start; i < end; i ++) {
				pptr = &poly_list[i];		
				mult_vector_matrix(&pptr->lnormal, mat2, &pptr->wnormal);
				
				pptr->wnormal.x -= vertex_list[pptr->v1].world.x;
				pptr->wnormal.y -= vertex_list[pptr->v1].world.y;
				pptr->wnormal.z -= vertex_list[pptr->v1].world.z;
			}
		}
	}

	/* move lights to camera space. */

	for (j = 0; j < num_lights; j ++) {

		light = &light_list[j];

		mult_vector_matrix(&light->position, mat, &light->cam_position);
		mult_vector_matrix(&light->target, mat, &light->cam_target);
	}
	
}


void	check_mesh_visibility (void)
{
	char			inside;
	int				i, j, start, end;
	float			dot;
	M3Dtriangle		*ptr;
	M3Dobject		*obj;

	
	visible_polys = 0;

	/* Process objects. trivial reject all possible polygons. */
	for (j = 0; j < num_objects; j ++) {

		obj = &object_list[j];
		/* process only if this object is marked as visible. */		
		if (obj->visible == TRUE) {

			start = obj->poly_base;
			end = obj->poly_base + obj->num_polys;
			/* process all polygons. */
			for (i = start; i < end; i ++) {
			
				ptr = &poly_list[i];
				/* back face culling. */
				dot = vector_dot(&vertex_list[ptr->v1].world, &ptr->wnormal);

				if (dot <= 0.0) {
					inside = 0;
					if (vertex_list[ptr->v1].world.z > Z_MIN) inside ++;
					if (vertex_list[ptr->v2].world.z > Z_MIN) inside ++;
					if (vertex_list[ptr->v3].world.z > Z_MIN) inside ++;

					/* No Z clipping needed */
					if (inside == 3) {
						
						sort_list[visible_polys].depth = (vertex_list[ptr->v1].world.z +
														  vertex_list[ptr->v2].world.z +
														  vertex_list[ptr->v3].world.z) / 3;
						sort_list[visible_polys].p_num = i;
			
						vertex_list[ptr->v1].status |= PROJECT + LIGHT;		/* mark these vertices for lightning calculation */
						vertex_list[ptr->v2].status |= PROJECT + LIGHT;		/* and perspective projection. */
						vertex_list[ptr->v3].status |= PROJECT + LIGHT;

						ptr->status = NONE;
			
						visible_polys ++;
						
					} 
					/* Z Clipping needed */
					else if (inside > 0) {

						sort_list[visible_polys].depth = (vertex_list[ptr->v1].world.z +
														  vertex_list[ptr->v2].world.z +
														  vertex_list[ptr->v3].world.z) / 3;
						sort_list[visible_polys].p_num = i;
			
						vertex_list[ptr->v1].status |= LIGHT;		/* these are only for intensity calculation. */
						vertex_list[ptr->v2].status |= LIGHT;		/* perspective coordinates are calculated later. */
						vertex_list[ptr->v3].status |= LIGHT;
						
						/* This indicates later that this poly should be clipped. */
						ptr->status = Z_CLIP;
			
						visible_polys ++;
						
					}
				}
			}
		}
	}
	
	for (i = 0; i < num_lights; i ++) {
		sort_list[visible_polys].depth = light_list[i].cam_position.z;
		sort_list[visible_polys].p_num = POLY_BUF + i;
		visible_polys ++;
	}

}


void	project (float distance)
{
	int				i;
	M3Dtriangle		*ptr;

	printf("visible_polys %06d \r", visible_polys); fflush(stdout);

	M3Ddistance = distance;

	/* project vertices. */
	for (i = 0; i < visible_polys; i ++) {
		if (sort_list[i].p_num < POLY_BUF) {
			ptr = &poly_list[sort_list[i].p_num];

			/* project vertice1 only if it is in the list and it haven't bee projected earlier. */
			if (vertex_list[ptr->v1].status & PROJECT == PROJECT) {
				project_vertex (&vertex_list[ptr->v1]);
			}
			
			/* project vertice2 only if it is in the list and it haven't bee projected earlier. */
			if (vertex_list[ptr->v2].status & PROJECT == PROJECT) {
				project_vertex (&vertex_list[ptr->v2]);
			}
			
			/* project vertice3 only if it is in the list and it haven't bee projected earlier. */
			if (vertex_list[ptr->v3].status & PROJECT == PROJECT) {
				project_vertex (&vertex_list[ptr->v3]);
			}
		}
	}
}	


int		z_clip (M3Dclip_point *source, M3Dclip_point *clipped, char type)
{
	int				i, num_points;
	float			ratio;
	M3Dclip_point	*start, *end;
	M3Dclip_point	I;
	
	/*	Clips 3 point polygon defined in source against plane z = Z_MIN. 3 or 4 point poly result is given in clipped. */
	num_points = 0;

	start = &source[2];
		
	for (i = 0; i < 3; i ++) {
		end = &source[i];
		
		if (end->z >= Z_MIN) {
			if (start->z >= Z_MIN) {										/* case 1:	polygon edge is completely inside. */
				memcpy(&clipped[num_points ++], end, sizeof(M3Dclip_point));
			}
			else {															/* case 4:	end point is inside, */
																			/*			startpoint is outside. */
				ratio = (Z_MIN - start->z) / (end->z - start->z);
				I.x = start->x + ((end->x - start->x) * ratio);
				I.y = start->y + ((end->y - start->y) * ratio);
				I.z = Z_MIN;
				I.color.R = start->color.R + (char)((float)(end->color.R - start->color.R) * ratio);
				I.color.G = start->color.G + (char)((float)(end->color.G - start->color.G) * ratio);
				I.color.B = start->color.B + (char)((float)(end->color.B - start->color.B) * ratio);
				
				memcpy(&clipped[num_points ++], &I, sizeof(M3Dclip_point));
				memcpy(&clipped[num_points ++], end, sizeof(M3Dclip_point));
			}
		}
		else {																/* case 2:	start point id inside, */
			if (start->z >= Z_MIN) {										/*			end point is outside. */

				ratio = (Z_MIN - end->z) / (start->z - end->z);
				I.x = end->x + ((start->x - end->x) * ratio);
				I.y = end->y + ((start->y - end->y) * ratio);
				I.z = Z_MIN;
				I.color.R = end->color.R + (char)((float)(start->color.R - end->color.R) * ratio);
				I.color.G = end->color.G + (char)((float)(start->color.G - end->color.G) * ratio);
				I.color.B = end->color.B + (char)((float)(start->color.B - end->color.B) * ratio);

				memcpy(&clipped[num_points ++], &I, sizeof(M3Dclip_point));
			}																/* case 3:  polygon edge is completely outside */
		}
		start = end;
	}

	return num_points;
}


void	draw_polys (void)
{
	int				i, j, num_points;
	float			x, y, z, ratio;
	M3Dtriangle		*ptr;
	M3Dclip_point	source[3];
	M3Dclip_point	clipped[4];
	M3Dpoly_point	points[4];

	/* draw polys. */

	for (i = 0; i < visible_polys; i ++) {

		if (sort_list[i].p_num < POLY_BUF) {
			ptr = &poly_list[sort_list[i].p_num];

			/* this polygon has "i-need-clipping" flag. */
			if (ptr->status == Z_CLIP) {
	
				/* copy vertice informations to clipping array. */
				source[0].x = vertex_list[ptr->v1].world.x;
				source[0].y = vertex_list[ptr->v1].world.y;
				source[0].z = vertex_list[ptr->v1].world.z;
				
				source[1].x = vertex_list[ptr->v2].world.x;
				source[1].y = vertex_list[ptr->v2].world.y;
				source[1].z = vertex_list[ptr->v2].world.z;
				
				source[2].x = vertex_list[ptr->v3].world.x;
				source[2].y = vertex_list[ptr->v3].world.y;
				source[2].z = vertex_list[ptr->v3].world.z;
		
				source[0].color.R = vertex_list[ptr->v1].color.R;
				source[0].color.G = vertex_list[ptr->v1].color.G;
				source[0].color.B = vertex_list[ptr->v1].color.B;
	
				source[1].color.R = vertex_list[ptr->v2].color.R;
				source[1].color.G = vertex_list[ptr->v2].color.G;
				source[1].color.B = vertex_list[ptr->v2].color.B;
	
				source[2].color.R = vertex_list[ptr->v3].color.R;
				source[2].color.G = vertex_list[ptr->v3].color.G;
				source[2].color.B = vertex_list[ptr->v3].color.B;
	
				num_points = z_clip(&source, &clipped, ptr->type);
	
				/* clear Z_CLIP status. */			
				ptr->status = NONE;
	
				for (j = 0; j < num_points; j ++) {
		
					/* project clipped points. */
					x = clipped[j].x;
					y = clipped[j].y;
					z = clipped[j].z;
					
					ratio = (float)(1.0) / z;
					points[j].x = (int)(M3Dview_center_x + x * M3Ddistance * ratio);
					points[j].y = (int)(M3Dview_center_y - y * M3Ddistance * ratio);
					
					points[j].color.R = clipped[j].color.R;
					points[j].color.G = clipped[j].color.G;
					points[j].color.B = clipped[j].color.B;
				}
							
				clip_and_draw_poly(points, ptr->type, num_points);
			}
			else {		
				points[0].x = vertex_list[ptr->v1].screen_x;
				points[0].y = vertex_list[ptr->v1].screen_y;
				
				points[1].x = vertex_list[ptr->v2].screen_x;
				points[1].y = vertex_list[ptr->v2].screen_y;
				
				points[2].x = vertex_list[ptr->v3].screen_x;
				points[2].y = vertex_list[ptr->v3].screen_y;
		
				points[0].color.R = vertex_list[ptr->v1].color.R;
				points[0].color.G = vertex_list[ptr->v1].color.G;
				points[0].color.B = vertex_list[ptr->v1].color.B;
	
				points[1].color.R = vertex_list[ptr->v2].color.R;
				points[1].color.G = vertex_list[ptr->v2].color.G;
				points[1].color.B = vertex_list[ptr->v2].color.B;
	
				points[2].color.R = vertex_list[ptr->v3].color.R;
				points[2].color.G = vertex_list[ptr->v3].color.G;
				points[2].color.B = vertex_list[ptr->v3].color.B;
	
				clip_and_draw_poly(points, ptr->type, 3);
				ptr->status = NONE;
			}
		}
		else {
			draw_light (sort_list[i].p_num - POLY_BUF);
		}
	}
}


void	clip_and_draw_poly (M3Dpoly_point *points, char type, int start_points)
{

	char				inside[4], all;
	int					i, out_points, num_points;
	long				ratio;
	M3Dpoly_point		*start, *end, *I;
	M3Dpoly_point		clipped1[8];
	M3Dpoly_point		clipped2[8];
	

	
	num_points = start_points;
	if (num_points == 0) return;
	
	all = 0;
	for (i = 0; i < num_points; i ++) {
		inside[i] = 0;
/*		if (points[i].y > M3Dclip_top) inside[i] |= 1;
		if (points[i].y < M3Dclip_bot) inside[i] |= 2;
		if (points[i].x > M3Dclip_left) inside[i] |= 4;
		if (points[i].x < M3Dclip_right) inside[i] |= 8;
		if (inside[i] == 15) all ++;*/

		if ((points[i].y > M3Dclip_top) && (points[i].y < M3Dclip_bot) &&
			(points[i].x > M3Dclip_left) && (points[i].x < M3Dclip_right))
				all ++;
	}
	

	if (all == num_points) {
		if (num_points == 3) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	points[0].x, points[0].y, points[1].x, points[1].y, points[2].x, points[2].y,
								points[0].color.R, points[0].color.G, points[0].color.B,
								points[1].color.R, points[1].color.G, points[1].color.B,
								points[2].color.R, points[2].color.G, points[2].color.B);
		}
	
		else if (num_points	== 4) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	points[0].x, points[0].y, points[1].x, points[1].y, points[2].x, points[2].y,
								points[0].color.R, points[0].color.G, points[0].color.B,
								points[1].color.R, points[1].color.G, points[1].color.B,
								points[2].color.R, points[2].color.G, points[2].color.B);
	
			drawrgbgouraudpoly(	points[0].x, points[0].y, points[3].x, points[3].y, points[2].x, points[2].y,
								points[0].color.R, points[0].color.G, points[0].color.B,
								points[3].color.R, points[3].color.G, points[3].color.B,
								points[2].color.R, points[2].color.G, points[2].color.B);
		}
	}
	else {

		/* clip top */
		out_points = 0;
		start = &points[num_points - 1];
		
		for (i = 0; i < num_points; i ++) {
			end = &points[i];
			
			if (end->y >= M3Dclip_top) {
				if (start->y >= M3Dclip_top) {									/* case 1:	polygon edge is completely inside. */
					memcpy(&clipped1[out_points ++], end, sizeof(M3Dpoly_point));
				}
				else {															/* case 4:	end point is inside, */
																				/*			startpoint is outside. */
					I = &clipped1[out_points ++];
		
					ratio = ((M3Dclip_top - start->y) << 16) / (end->y - start->y);
					I->x = start->x + (((end->x - start->x) * ratio) >> 16);
					I->y = M3Dclip_top;
					I->color.R = start->color.R + (((end->color.R - start->color.R) * ratio) >> 16);
					I->color.G = start->color.G + (((end->color.G - start->color.G) * ratio) >> 16);
					I->color.B = start->color.B + (((end->color.B - start->color.B) * ratio) >> 16);
	
					memcpy(&clipped1[out_points ++], end, sizeof(M3Dpoly_point));
				}
			}
			else {																/* case 2:	start point is inside, */
				if (start->y >= M3Dclip_top) {									/*			end point is outside. */
	
					I = &clipped1[out_points ++];
	
					ratio = ((M3Dclip_top - end->y) << 16) / (start->y - end->y);
					I->x = end->x + (((start->x - end->x) * ratio) >> 16);
					I->y = M3Dclip_top;
					I->color.R = end->color.R + (((start->color.R - end->color.R) * ratio) >> 16);
					I->color.G = end->color.G + (((start->color.G - end->color.G) * ratio) >> 16);
					I->color.B = end->color.B + (((start->color.B - end->color.B) * ratio) >> 16);
				}																/* case 3:  polygon edge is completely outside */
			}
			start = end;
		}
	
		/* clip bottom */
	
		num_points = out_points;
		if (num_points == 0) return;
	
		out_points = 0;
		start = &clipped1[num_points - 1];
			
		for (i = 0; i < num_points; i ++) {
			end = &clipped1[i];
			
			if (end->y <= M3Dclip_bot) {
				if (start->y <= M3Dclip_bot) {									/* case 1:	polygon edge is completely inside. */
					memcpy(&clipped2[out_points ++], end, sizeof(M3Dpoly_point));
				}
				else {															/* case 4:	end point is inside, */
																				/*			startpoint is outside. */
					I = &clipped2[out_points ++];
	
					ratio = ((M3Dclip_bot - start->y) << 16) / (end->y - start->y);
					I->x = start->x + (((end->x - start->x) * ratio) >> 16);
					I->y = M3Dclip_bot;
					I->color.R = start->color.R + (((end->color.R - start->color.R) * ratio) >> 16);
					I->color.G = start->color.G + (((end->color.G - start->color.G) * ratio) >> 16);
					I->color.B = start->color.B + (((end->color.B - start->color.B) * ratio) >> 16);
	
					memcpy(&clipped2[out_points ++], end, sizeof(M3Dpoly_point));
				}
			}
			else {																/* case 2:	start point is inside, */
				if (start->y <= M3Dclip_bot) {									/*			end point is outside. */
	
					I = &clipped2[out_points ++];
	
					ratio = ((M3Dclip_bot - end->y) << 16) / (start->y - end->y);
					I->x = end->x + (((start->x - end->x) * ratio) >> 16);
					I->y = M3Dclip_bot;
					I->color.R = end->color.R + (((start->color.R - end->color.R) * ratio) >> 16);
					I->color.G = end->color.G + (((start->color.G - end->color.G) * ratio) >> 16);
					I->color.B = end->color.B + (((start->color.B - end->color.B) * ratio) >> 16);
				}																/* case 3:  polygon edge is completely outside */
			}
			start = end;
		}
	
		/* clip left */
	
		num_points = out_points;
		if (num_points == 0) return;
	
		out_points = 0;
		start = &clipped2[num_points - 1];
			
		for (i = 0; i < num_points; i ++) {
			end = &clipped2[i];
			
			if (end->x >= M3Dclip_left) {
				if (start->x >= M3Dclip_left) {									/* case 1:	polygon edge is completely inside. */
					memcpy(&clipped1[out_points ++], end, sizeof(M3Dpoly_point));
				}
				else {															/* case 4:	end point is inside, */
																				/*			startpoint is outside. */
					I = &clipped1[out_points ++];
	
					ratio = ((M3Dclip_left - start->x) << 16) / (end->x - start->x);
					I->x = M3Dclip_left;
					I->y = start->y + (((end->y - start->y) * ratio) >> 16);
					I->color.R = start->color.R + (((end->color.R - start->color.R) * ratio) >> 16);
					I->color.G = start->color.G + (((end->color.G - start->color.G) * ratio) >> 16);
					I->color.B = start->color.B + (((end->color.B - start->color.B) * ratio) >> 16);
	
					memcpy(&clipped1[out_points ++], end, sizeof(M3Dpoly_point));
				}
			}
			else {																/* case 2:	start point is inside, */
				if (start->x >= M3Dclip_left) {									/*			end point is outside. */
	
					I = &clipped1[out_points ++];
	
					ratio = ((M3Dclip_left - end->x) << 16) / (start->x - end->x);
					I->x = M3Dclip_left;
					I->y = end->y + (((start->y - end->y) * ratio) >> 16);
					I->color.R = end->color.R + (((start->color.R - end->color.R) * ratio) >> 16);
					I->color.G = end->color.G + (((start->color.G - end->color.G) * ratio) >> 16);
					I->color.B = end->color.B + (((start->color.B - end->color.B) * ratio) >> 16);
				}																/* case 3:  polygon edge is completely outside */
			}
			start = end;
		}
	
		/* clip right */
	
		num_points = out_points;
		if (num_points == 0) return;
	
		out_points = 0;
		start = &clipped1[num_points - 1];
			
		for (i = 0; i < num_points; i ++) {
			end = &clipped1[i];
			
			if (end->x <= M3Dclip_right) {
				if (start->x <= M3Dclip_right) {								/* case 1:	polygon edge is completely inside. */
					memcpy(&clipped2[out_points ++], end, sizeof(M3Dpoly_point));
				}
				else {															/* case 4:	end point is inside, */
																				/*			startpoint is outside. */
					I = &clipped2[out_points ++];
	
					ratio = ((M3Dclip_right - start->x) << 16) / (end->x - start->x);
					I->x = M3Dclip_right;
					I->y = start->y + (((end->y - start->y) * ratio) >> 16);
					I->color.R = start->color.R + (((end->color.R - start->color.R) * ratio) >> 16);
					I->color.G = start->color.G + (((end->color.G - start->color.G) * ratio) >> 16);
					I->color.B = start->color.B + (((end->color.B - start->color.B) * ratio) >> 16);
	
					memcpy(&clipped2[out_points ++], end, sizeof(M3Dpoly_point));
				}
			}
			else {																/* case 2:	start point is inside, */
				if (start->x <= M3Dclip_right) {								/*			end point is outside. */
	
					I = &clipped2[out_points ++];
	
					ratio = ((M3Dclip_right - end->x) << 16) / (start->x - end->x);
					I->x = M3Dclip_right;
					I->y = end->y + (((start->y - end->y) * ratio) >> 16);
					I->color.R = end->color.R + (((start->color.R - end->color.R) * ratio) >> 16);
					I->color.G = end->color.G + (((start->color.G - end->color.G) * ratio) >> 16);
					I->color.B = end->color.B + (((start->color.B - end->color.B) * ratio) >> 16);
				}																/* case 3:  polygon edge is completely outside */
			}
			start = end;
		}

		/* clipped poly is in clipped2[] and number of points in this poly in out_points. */
	
		if (out_points == 3) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[1].x, clipped2[1].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[1].color.R, clipped2[1].color.G, clipped2[1].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
		}
	
		else if (out_points	== 4) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[1].x, clipped2[1].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[1].color.R, clipped2[1].color.G, clipped2[1].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-3-4 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[3].x, clipped2[3].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[3].color.R, clipped2[3].color.G, clipped2[3].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
		}
	
		else if (out_points	== 5) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[1].x, clipped2[1].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[1].color.R, clipped2[1].color.G, clipped2[1].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-3-4 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[3].x, clipped2[3].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[3].color.R, clipped2[3].color.G, clipped2[3].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-4-5 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[3].x, clipped2[3].y, clipped2[4].x, clipped2[4].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[3].color.R, clipped2[3].color.G, clipped2[3].color.B,
								clipped2[4].color.R, clipped2[4].color.G, clipped2[4].color.B);
		}
	
		else if (out_points	== 6) {
			/* draw 1-2-3 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[1].x, clipped2[1].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[1].color.R, clipped2[1].color.G, clipped2[1].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-3-4 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[3].x, clipped2[3].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[3].color.R, clipped2[3].color.G, clipped2[3].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-4-5 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[3].x, clipped2[3].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[3].color.R, clipped2[3].color.G, clipped2[3].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
			/* draw 1-5-6 */
			drawrgbgouraudpoly(	clipped2[0].x, clipped2[0].y, clipped2[5].x, clipped2[5].y, clipped2[2].x, clipped2[2].y,
								clipped2[0].color.R, clipped2[0].color.G, clipped2[0].color.B,
								clipped2[5].color.R, clipped2[5].color.G, clipped2[5].color.B,
								clipped2[2].color.R, clipped2[2].color.G, clipped2[2].color.B);
		}
	}
}


int		poly_compare (M3Dsort *a, M3Dsort *b)
{
	float	comp;
	
	comp = a->depth - b->depth;

	if (comp < 0)
		return 1;
	else
		return -1;

}

#include <stddef.h>

void	mqsort (void  *base, size_t nel, size_t width)
{
	size_t	wnel, gap, wgap, i, j, k;
	char	*a, *b, tmp;
	M3Dsort	*ta, *tb;

	wnel = width * nel;

	for (gap = 0; ++gap < nel;)
		gap *= 3;

	while ( gap /= 3 ) {
		wgap = width * gap;
		for (i = wgap; i < wnel; i += width) {
			for (j = i - wgap; ;j -= wgap) {
				a = j + (char *)base;
				b = a + wgap;
				
				ta = (M3Dsort *)a;
				tb = (M3Dsort *)b;

				if ( ta->depth >= tb->depth )
					break;

				k = width;
				do {
					tmp = *a;
					*a++ = *b;
					*b++ = tmp;
				} while ( --k );
				if (j < wgap)
					break;
			}
		}
	}
}


void	sort_polys (void)
{
	mqsort(sort_list, visible_polys, sizeof(M3Dsort));
}
