////////////////////////////////////////////////////////////////////////////
//
//        CIENKA    -    100 Kb intro by FAC/DLB featuring music by Uctumi
//
//        File:     bitmaps.c
//
//        Description:   This file implements the next two classes:
//                       class TBitmap which handles 32 bpp bitmaps and
//                       class TBitmap8bpp which handles 256 color bitmaps
//
////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include "globals.h"
#include "bitmaps.h"

// TBitmap class implementation

// constructor

TBitmap::TBitmap(int w, int h) {
     // get memory for image data
     datos = (unsigned *)malloc(w * h * 4);

     // let's make sure we got a valid pointer
     if (datos) {
        ancho = w;  // set width
        alto = h;   // set height

        // get memory for the Y-offset table and calculate it
        tablay = (int *)malloc(h * sizeof(int));
        for (int y = 0; y < h; y++) tablay[y] = ancho * y;
     }
     else {
          ancho = 0;
          alto = 0;
          tablay = NULL;
     }
}


// destructor

TBitmap::~TBitmap() {
     if (datos) free(datos);
     if (tablay) free(tablay);
}

// Clear bitmap with some color

void TBitmap::Borra(unsigned color) {
     unsigned *offs = datos;
	unsigned size = ancho * alto;
	_asm {
		mov eax, [color]
		mov ecx, [size]
		mov edi, [offs]
		rep stosd
	}
}

// Blur bitmap (this is a 32 bpp blur but the rgb components are blurred
// separately, so it's done like 8 bpp blur. We only have to skip the
// first byte in each dword and blur the next three bytes).

void TBitmap::Blur() {
     unsigned bpsl = ancho * 4;              // bytes per scanline
     unsigned size = ancho * (alto - 2);     // bitmap size (2 scanlines less)
     unsigned *offs = datos;                 // pointer to bitmap data

	_asm {
		mov edi, [offs]
          mov edx, [bpsl]     // edi is used to add the offset of 1 scanline
		mov ecx, [size]
          add edi, edx        // start from Y = 1
		xor eax, eax
          mov esi, edx
		xor ebx, ebx
          neg esi        // esi is used to substract the offset of 1 scanline

                    // blur the first byte of a dword
          blah:     mov al, [edi + esi]
				mov bl, [edi + edx]
				add eax, ebx
				mov bl, [edi - 4]
				add eax, ebx
				mov bl, [edi + 4]
				add eax, ebx
				shr eax, 2
                    stosb     // for some reason, this was faster
                              // than mov [edi], al; inc edi;

                    // blur the second byte
				mov al, [edi + esi]
				mov bl, [edi + edx]
				add eax, ebx
				mov bl, [edi - 4]
				add eax, ebx
				mov bl, [edi + 4]
				add eax, ebx
				shr eax, 2
				stosb

                    // blur the third byte
				mov al, [edi + esi]
				mov bl, [edi + edx]
				add eax, ebx
				mov bl, [edi - 4]
				add eax, ebx
				mov bl, [edi + 4]
				add eax, ebx
				shr eax, 2
				stosb

                    // continue with next pixel
				inc edi
				dec ecx				
				jnz blah
	}
}


// TBitmap8bpp implementation

// constructors (pretty similar to TBitmap::TBitmap())

TBitmap8bpp::TBitmap8bpp(int w, int h) {
     datos = (byte *)malloc(w * h);
     if (datos) {
        ancho = w;
        alto = h;
        tablay = (int *)malloc(h * sizeof(int));
        for (int y = 0; y < h; y++) tablay[y] = ancho * y;
        _memset(paleta, 0, 256 * sizeof(unsigned));
     }
     else {
          ancho = 0;
          alto = 0;
          tablay = NULL;
     }
}


// This constructors loads a bitmap from data already in memory
// (that is, the bitmaps stored in the executable)

TBitmap8bpp::TBitmap8bpp(int w, int h, unsigned char *data, unsigned *palette) {
     ancho = w;
     alto = h;
     datos = (byte *)data;
     _memcpy(paleta, palette, 256 * sizeof(unsigned));
     tablay = (int *)malloc(h * sizeof(int));
     for (int y = 0; y < h; y++) tablay[y] = ancho * y;
}

// Crappy destructor: doesn't free the image data memory
// (that has to be done manually with free(mybitmap.data) )

TBitmap8bpp::~TBitmap8bpp() {
     if (tablay) free(tablay);
}


// Clear the bitmap (only called at initialization time so no asm needed)

void TBitmap8bpp::Borra(byte color) {
        if (datos) _memset(datos, color, ancho * alto);
}


// This function copies an 8 bpp bitmap to a 32 bpp bitmap at (x0, y0)
// Clipping is also performed.

// I didn't bother to optimize this. Watcom does it pretty well.

void TBitmap8bpp::BlitTo(TBitmap *dest, int x0, int y0) {
	int xa = 0, ya = 0, width = ancho, height = alto, i;

     // (xa, ya) = starting coordinates in source (8 bpp) bitmap
     // (x0, y0) = starting coordinates in destination (32 bpp) bitmap
     // (width, height) = size of the clipped region

     // check if the source is blitted completely outside the destination
	if ((x0 >= dest->ancho) || (y0 >= dest->ancho)) return;

     // perform some clipping
	if ((x0 + width) > dest->ancho) width = dest->ancho - x0;
	if ((y0 + height) > dest->alto) height = dest->alto - y0;
	if (x0 < 0) { xa = -x0; width += x0; x0 = 0; }
	if (y0 < 0) { ya = -y0; height += y0; y0 = 0; }

     // and do the blit!
	byte *src;
	unsigned *dst;
	while ((height--) > 0) {
		src = datos + tablay[ya++] + xa;
		dst = dest->datos + dest->tablay[y0++] + x0;
		for (i = 0; i < width; i++) *dst++ = paleta[*src++];
	}
}


// This function copies an 8 bpp bitmap to a 32 bpp bitmap at (x0, y0)
// using tcolor as a transparent color. Works just like the function above.

void TBitmap8bpp::TransBlitTo(TBitmap *dest, int x0, int y0, byte tcolor) {
	int xa = 0, ya = 0, width = ancho, height = alto, i;

     // perform clipping and blah blah blah...
	if ((x0 >= dest->ancho) || (y0 >= dest->ancho)) return;
	if ((x0 + width) > dest->ancho) width = dest->ancho - x0;
	if ((y0 + height) > dest->alto) height = dest->alto - y0;
	if (x0 < 0) { xa = -x0; width += x0; x0 = 0; }
	if (y0 < 0) { ya = -y0; height += y0; y0 = 0; }

     // do the blit!
	byte *src;
	unsigned *dst;
	while ((height--) > 0) {
		src = datos + tablay[ya++] + xa;
		dst = dest->datos + dest->tablay[y0++] + x0;
                for (i = 0; i < width; i++) {
                        if (*src != tcolor) *dst = paleta[*src];
                        dst++; src++;
                }
	}
}


// This function blits an 8 bpp bitmap to another 8 bpp bitmap
// It works like the other blitting functions.
void TBitmap8bpp::BlitTo8bpp(TBitmap8bpp *dest, int x0, int y0) {
	int xa = 0, ya = 0, width = ancho, height = alto, i;
	if ((x0 >= dest->ancho) || (y0 >= dest->ancho)) return;
	if ((x0 + width) > dest->ancho) width = dest->ancho - x0;
	if ((y0 + height) > dest->alto) height = dest->alto - y0;
	if (x0 < 0) { xa = -x0; width += x0; x0 = 0; }
	if (y0 < 0) { ya = -y0; height += y0; y0 = 0; }
	byte *src;
	byte *dst;
	while ((height--) > 0) {
		src = datos + tablay[ya++] + xa;
		dst = dest->datos + dest->tablay[y0++] + x0;
		for (i = 0; i < width; i++) *dst++ = paleta[*src++];
	}
}


// Blur an 8 bpp bitmap (by just taking the mean of the adjacent pixels)

void TBitmap8bpp::Blur() {
	unsigned off = ancho + (unsigned)datos;
	unsigned size = ancho * (alto - 2);
	unsigned bpsl = ancho;
	_asm {
		mov edx, [bpsl]
		mov edi, [off]
		mov esi, [bpsl]
		mov ecx, [size]
		neg edx
		xor eax, eax
		xor ebx, ebx
          bla:      mov al, [edi + edx]
				mov bl, [edi + esi]
				add eax, ebx
				mov bl, [edi - 1]
				add eax, ebx
				mov bl, [edi + 1]
				add eax, ebx
				shr eax, 2
				mov [edi], al
				inc edi
				dec ecx
				jnz bla
	}
}

