#include <sifcmd.h>
#include <kernel.h>
#include <sifrpc.h>
#include <compat.h>

#include "spu.h"


// defines
#define SPU_RPC 0x80000601
#define STACK_SIZE	4096

// iop_heap global rpc variables
static struct t_rpc_client_data spu_sif __attribute__((aligned(128)));
static u8 send_buffer[512] __attribute__((aligned(128)));
static u8 recieve_buffer[512] __attribute__((aligned(128)));
static u8 stack[STACK_SIZE] __attribute__((aligned(128)));

// globals
static int spu_initialised = 0;
static void (*spu_callback_ptr)();
static void (*IRQ_callback_ptr)();
static void (*transfer_callback_ptr)();
static void (*auto_dma_callback_ptr)();

// imports
extern void * _gp;


// helpers

int spu_init()
{
  if(spu_initialised)
    return 0;

  spu_sif.server = NULL;
  
  do 
    {
      if (sif_bind_rpc(&spu_sif, SPU_RPC, 0) < 0) {
	return -1;
      }
      nopdelay();
    }
  while(!spu_sif.server);
  
  k_FlushCache(0);  
  spu_initialised = 1;

  //init spu
  spu_remote(1,spuInit,0,0,0,0,0,0);

  return 1;
}

void spu_callback(void *ptr)
{
  spu_callback_ptr = ptr;
}

int spu_remote(u32 mode, u32 function, u32 param1, u32 param2, u32 param3, u32 param4, u32 param5, u32 param6)
{
  int i=0,len=0,ret=0,m=0;
  u8  *pkt;
 
  if(!spu_initialised)
    return 0;

  m = !mode?1:0;


  /* Special case for setting common_attr or voice_attr */
  if(function == spuSetCommonAttr || function == spuSetVoiceAttr)
    {
      if ((ret = sif_call_rpc( &spu_sif, function, m, (void*)param1, param2, recieve_buffer, 0x10, 0, 0)) < 0)
	return -1;
      
      POPDATA( u32, UNCACHED_SEG(recieve_buffer), ret, i);  
    }
  
  /* Special case for getting common_attr or voice_attr */
  else if(function == spuGetCommonAttr || function == spuGetVoiceAttr)
    {
      if ((ret = sif_call_rpc( &spu_sif, function, m, 0, 0, (void*)param1, param2, 0, 0)) < 0)
	return -1;
      
      ret = 0;
    }
  /* General case for calling spu functions */	
  else
    {
      /* build packet */
      pkt = send_buffer; 
      PUSHDATA( u8*, pkt, send_buffer, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param1, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param2, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param3, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param4, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param5, i);
      pkt += i; len += i;
      PUSHDATA( u32, pkt, param6, i);
      pkt += i; len += i;
    
      if ((ret = sif_call_rpc( &spu_sif, function, m, send_buffer, len, recieve_buffer, 0x10, 0, 0)) < 0)
	return -1;
      
      POPDATA( u32, UNCACHED_SEG(recieve_buffer), ret, i);  
    }

  /* save callback handlers globally */
  if(function == spuSetIRQCallback)
    IRQ_callback_ptr = (void*)param1;
  else if(function == spuSetTransferCallback)
    transfer_callback_ptr = (void*)param1;
  else if(function == spuAutoDMASetCallback)
    auto_dma_callback_ptr = (void*)param1;
  
  return ret;    
}

void *spu_remote_callback_handler(u32 command, void *data, int size)
{
  //printf("Callback #%x called\n",*((u32*)(data)));

  switch(*((u32*)(data)))
    {
    case 	0:
      {
	if(spu_callback_ptr)
	  spu_callback_ptr();
	break;
      }
    case 	5:
      {
	if(transfer_callback_ptr)
	  transfer_callback_ptr();
	break;
      }
    case 	6:
      {
	if(IRQ_callback_ptr)
        IRQ_callback_ptr();
	break;
      }
    default:
      break;
    }		
}


void spu_remote_callback_thread()
{
  struct t_rpc_data_queue cb_queue __attribute__((aligned(128)));
  struct t_rpc_server_data cb_srv __attribute__((aligned(128)));
  u8 cb_rpc_buffer[512] __attribute__((aligned(128)));
  int thread_id;

  sif_rpc_init(0);
  thread_id = k_GetThreadID();
  sif_set_rpc_queue(&cb_queue,thread_id);
  sif_register_rpc(&cb_srv,0x80000603,	spu_remote_callback_handler, cb_rpc_buffer, 0 , 0, &cb_queue);
  sif_rpc_loop(&cb_queue);
}


int spu_remote_callback_init(int priority)
{
  struct thread_attr ta;
  int thread_id;
  int thread_priority;

  if(!spu_initialised)
    return -1;

  spu_remote(1,_spu_start_thread,0,0,0,0,0,0);

  ta.func = spu_remote_callback_thread;
  ta.stack = &stack;
  ta.stack_size = sizeof(stack);
  ta.gp_reg = &_gp;
  ta.option = priority;

  thread_id = k_CreateThread(&ta);

  if(k_StartThread(thread_id,0) < 0)
    return -1;
  
  return thread_id;
}

