#include "raceworld.hpp"
#include <math.h>
#include <SDL2/SDL.h>

using namespace cg_engine;

Title::Title(Config *c) {
	start = 0.0;
	end = 6.0;

	simple = c->getShader("title");
	screenquad = new Sprite();
	title1 = c->getTexture("title1");
	title2 = c->getTexture("title2");
}

void Title::show(double time) {

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	simple->use();
	if(time < 3.0){
		simple->addTexture("image", title1);
	} else {
		simple->addTexture("image", title2);
	}
	simple->setParam("vertical", 1.0f);
	screenquad->draw();
}

Intro::Intro(int w, int h, Config *c) {
	start = 6.0;
	end =   12.0;
	terrain = c->getModel("terrain");
	charger = c->getModel("charger");
	toycar = c->getModel("toycar");
	wheel = c->getModel("wheel");
	skybox = c->getModel("skybox");
	stars = c->getTexture("stars");
	wireframe = c->getShader("wireframe");
	simple = c->getShader("simple");
	pmatrix.perspective(35, (double)w / (double)h, 0.1, 1500);
	screenquad = new Sprite();
}

void Intro::geometry(Matrix camera){
	Matrix m, tr, rot;
	int i;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	wireframe->use();
	m = camera * pmatrix;
	wireframe->setParam("WIRE_COL", 0.01, 0.97, 0.98);
	wireframe->setParam("FILL_COL", 12.0 / 255.0, 37.0 / 255.0, 59.0 / 255.0);
	wireframe->setParam("WIN_SCALE", 12.0, 12.0);

	for(i = -5; i < 5; i++){
		tr.translate(0.0, 0.0, 132.0 * i);
		m = tr * camera * pmatrix;
		wireframe->setParam("pmatrix", m);
		terrain->draw();
	}

	wireframe->setParam("FILL_COL", 0.0, 1.0, 0.0);
	wireframe->setParam("WIRE_COL", 1.0, 0.0, 0.0);

	tr.translate(-10.0, 0.0, 44.0);
	m = tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	toycar->draw();

	//toycar wheels
	tr.translate(-3.2, 0.6, 43.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-3.2, 0.6, 45.40);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 43.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 45.40);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(3.5, 0.0, 44.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	charger->draw();

	// charger wheels
	tr.translate(4.9, 0.6, 40.82);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(4.9, 0.6, 46.6);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 40.82);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 46.6);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	simple->use();
	simple->addTexture("image", stars);
	m = camera * pmatrix;
	simple->setParam("pmatrix", m);
	skybox->draw();
}

void Intro::show(double time) {
	double pos[3] = {0.0,4.0, 24.0};
	double target[3] = {0.0, 3.0, 44.0};
	double up[3] = {0.0, 1.0, 0.0};
	Matrix camera;

	pos[0] = sin((time + 15.0) / 8.0) * 20.0;
	pos[2] = 44.0 + cos((time + 15.0) / 8.0) * 20.0;
	camera.lookAt(pos, target, up);
	geometry(camera);
}

TerrainAndCars::TerrainAndCars(int w, int h, Config *c) {
	start = 12.0;
	end =   47.553;
	terrain = c->getModel("terrain");
	charger = c->getModel("charger");
	toycar = c->getModel("toycar");
	wheel = c->getModel("wheel");
	skybox = c->getModel("skybox");
	stars = c->getTexture("stars");
	wireframe = c->getShader("wireframe");
	simple = c->getShader("simple");
	tinyplanet = c->getShader("tinyplanet");
	pmatrix.perspective(90, 1.0, 0.1, 1500); // We render to cubemap, which viewport is a square
	cubemap = new CubeMap(2048, w, h);
	screenquad = new Sprite();
}

void TerrainAndCars::geometry(double time, Matrix camera){
	Matrix m, tr, rot, rot2;
	int i;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	wireframe->use();
	m = camera * pmatrix;
	wireframe->setParam("WIRE_COL", 0.01, 0.97, 0.98);
	wireframe->setParam("FILL_COL", 12.0 / 255.0, 37.0 / 255.0, 59.0 / 255.0);
	wireframe->setParam("WIN_SCALE", 12.0, 12.0);

	for(i = -5; i < 5; i++){
		tr.translate(0.0, 0.0, 132.0 * i + (time * 20.0));
		m = tr * camera * pmatrix;
		wireframe->setParam("pmatrix", m);
		terrain->draw();
	}

	wireframe->setParam("FILL_COL", 0.0, 1.0, 0.0);
	wireframe->setParam("WIRE_COL", 1.0, 0.0, 0.0);

	tr.translate(-10.0, 0.0, 44.0);
	rot.rotate(0.0, 0.0, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	toycar->draw();

	//toycar wheels
	tr.translate(-3.2, 0.6, 43.0);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-3.2, 0.6, 45.40);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 43.0);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 45.40);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(3.5, 0.0, 44.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	charger->draw();

	// charger wheels
	tr.translate(4.9, 0.6, 40.82);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(4.9, 0.6, 46.6);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 40.82);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 46.6);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	simple->use();
	simple->addTexture("image", stars);
	m = camera * pmatrix;
	simple->setParam("pmatrix", m);
	skybox->draw();
}

void TerrainAndCars::show(double time) {
	double pos[3] = {0.11, 4.0, 24.0};
	double sides[] = {-10.0, 0.0, 0.0, 10.0, 0.0, 0.0, 0.0, -27.0, 0.0, 0.0, 27.0, 0.0, 0.0, 0.0, 40.0, 0.0, 0.0, -10.0};
	double target[18];
	double up[] = {0.0, 1.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, -1.0, 0.0, 1.0, 0.0, 0.0, 1.0, 0.0};
	Matrix camera;
	float param1 = 1.0f;
	float param2 = 0.25f;
	float param3 = 0.5f;
	float param4 = 1.49f;
	int i;

	// change parameters over time
	if(time < 18.0 ){
		pos[0] = pos[0] + sin(time) * 4.0;
		pos[2] = pos[2] + cos(time) * 4.0;
	} else if(time > 18.0 && time < 24.0){
		param2 = 0.25f + sin(time - 18.0) + 1.0;
	} else if(time > 24.0 && time < 29){
		param4 = 1.49 + (time - 24.0) / 10.0f;
		pos[2] = (time - 24.0) * 10.0;
	} else if(time > 29.0 && time < 38.0){
		pos[2] = 48.0;
		param4 = 1.49f - (time - 29.0f) / 10.0f;
		param2 = 0.25f - (time - 29.0f) / 50.0f;
	} else if(time > 36.0 && time < 41.0){
		param3 = 0.5f + (time - 36.0) / 10.0f;
	} else{
		param3 = 0.5f + 5.0f / 10.0f;
		param1 = 1.0f - (time - 41.0) / 10.0f;
	}

	for(i = 0; i < 18; i++){
		target[i] = pos[i % 3] + sides[i];
	}

	// render to cubemap
	cubemap->setAsRendertarget();
	for(i = 0; i < 6; i++){
		cubemap->setSide(i);
		camera.lookAt(pos, &target[i*3], &up[i*3]);
		geometry(time - start, camera);
	}
	cubemap->unset();

	// render to screen
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	tinyplanet->use();
	tinyplanet->addTexture("image", GL_TEXTURE_CUBE_MAP, cubemap->getID());
	tinyplanet->setParam("param1", param1);
	tinyplanet->setParam("param2", param2);
	tinyplanet->setParam("param3", param3);
	tinyplanet->setParam("param4", param4);
	screenquad->draw();
}

CarDrop::CarDrop(int w, int h, Config *c) {
	start = 47.553;
	end = 54.195;
	terrain = c->getModel("terrain");
	charger = c->getModel("charger");
	toycar = c->getModel("toycar");
	wheel  = c->getModel("wheel");
	skybox = c->getModel("skybox");
	stars = c->getTexture("stars");
	wireframe = c->getShader("wireframe");
	simple = c->getShader("simple");
	pmatrix.perspective(35, (double)w / (double)h, 0.1, 1500);
}

void CarDrop::show(double time) {
	double pos[3] = {0.0,4.0,0.0};
	double target[3] = {0.0,3.0,-44.0};
	double up[3] = {0.0,1.0,0.0};
	Matrix camera, m, tr, rot, tr1, tr2, rot2;
	double Y;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	wireframe->use();
	camera.lookAt(pos, target, up);
	m = camera * pmatrix;
	wireframe->setParam("WIRE_COL", 0.01, 0.97, 0.98);
	wireframe->setParam("FILL_COL", 12.0 / 255.0, 37.0 / 255.0, 59.0 / 255.0);
	wireframe->setParam("WIN_SCALE", 12.0, 12.0);

	if(time - start < 2.0){
		tr.translate(0.0, 0.0, (time - start) * 20.0);
	} else {
		tr.translate(0.0, 0.0, 40.0);
	}

	m = tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	terrain->draw();

	if(time - start < 2.621){
		Y = 0.0;
	} else {
		Y = (time - start - 2.621) * -30.0;
	}

	tr1.translate(3.5, Y, -44.0);
	tr2.translate(-10.0, Y, -44.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr1 * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wireframe->setParam("FILL_COL", 0.0, 1.0, 0.0);
	wireframe->setParam("WIRE_COL", 1.0, 0.0, 0.0);
	charger->draw();

	rot.rotate(0.0, 0.0, 0.0);
	m = rot * tr2 * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	toycar->draw();

	tr.translate(-3.2, 0.6 + Y, -43.0);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-3.2, 0.6 + Y, -45.40);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6 + Y, -43.0);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6 + Y, -45.40);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	// charger wheels
	tr.translate(4.9, 0.6 + Y, -40.82);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(4.9, 0.6 + Y, -46.6);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6 + Y, -40.82);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6 + Y, -46.6);
	rot.rotate(0.0, 0.0, time * 20.0);
	rot2.rotate(0.0, -1.6, 0.0);
	m = rot * rot2 * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	simple->use();
	simple->addTexture("image", stars);
	m = camera * pmatrix;
	simple->setParam("pmatrix", m);
	skybox->draw();

}

WholeWorld::WholeWorld(int w, int h, Config *c){
	start = 54.195;
	end =   70.536;
	terrain = c->getModel("terrain");
	turtle = c->getModel("turtle");
	charger = c->getModel("charger");
	toycar = c->getModel("toycar");
	skybox = c->getModel("skybox");
	wireframe = c->getShader("wireframe");
	simple = c->getShader("simple");
	title = c->getShader("title");
	poet = c->getTexture("poet");
	stars = c->getTexture("stars");
	pmatrix.perspective(35, (double)w / (double)h, 0.1, 1500);
	screenquad = new Sprite();
}

void WholeWorld::show(double time){
	double pos[3] = {0.0,30,80.0};
	double target[3] = {0.0,0.0,0.0};
	double up[3] = {0.0,1.0,0.0};
	Matrix camera, m, rot, sc, tr;
	double rtime;

	rtime = time - start;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	wireframe->use();
	pos[0] = sin((rtime + 30.0) / 10.0) * 60.0;
	pos[2] = cos((rtime + 30.0) / 10.0) * 60.0;
	camera.lookAt(pos, target, up);
	rot.rotate(-1.54,0,0);
	m = rot * camera * pmatrix;
	wireframe->setParam("WIRE_COL", 0.01, 0.97, 0.98);
	wireframe->setParam("FILL_COL", 12.0 / 255.0, 37.0 / 255.0, 59.0 / 255.0);
	wireframe->setParam("WIN_SCALE", 12.0, 12.0);
	wireframe->setParam("pmatrix", m);

	turtle->draw();

	sc.scale(0.15,0.15,0.15);
	tr.translate(0.0, 9.6, 0.0);
	m = sc * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	terrain->draw();

	wireframe->setParam("FILL_COL", 0.0, 1.0, 0.0);
	wireframe->setParam("WIRE_COL", 1.0, 0.0, 0.0);
	wireframe->setParam("WIN_SCALE", 40, 40);

	sc.scale(0.10, 0.10, 0.10);
	rot.rotate(0.0, time, time / 2.0);
	tr.translate(0.0, 9.0 - rtime, -12);
	m = sc * rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	charger->draw();

	sc.scale(0.10, 0.10, 0.10);
	tr.translate(-2.0, 9.0 - rtime, -12);
	m = sc * rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	toycar->draw();

	simple->use();
	simple->addTexture("image", stars);
	m = camera * pmatrix;
	simple->setParam("pmatrix", m);
	skybox->draw();

	title->use();
	title->addTexture("image", poet);
	title->setParam("vertical", (float)(time - start) / 10.0f);
	screenquad->draw();
}

ToInfinity::ToInfinity(int w, int h, Config *c) {
	start = 70.536;
	end = 100.0;

	wheel = c->getModel("wheel");
	charger = c->getModel("charger");
	toycar = c->getModel("toycar");
	wireframe = c->getShader("wireframe");
	starshader = c->getShader("star");
	pmatrix.perspective(35.0, (float)w / (float)h, 0.1, 1500);
	stars = new CircParticle(600, 0.0, 1.0, 10.0, 10.0, start);
	for(double i = 0.0; i < 100.0; i = i + 1.0){
		stars->update(start + i);
	}
}

void ToInfinity::show(double time) {
	Matrix camera, m, tr, rot;
	double pos[3] = {0.0,1.0, 24.0};
	double target[3] = {0.0, 3.0, 44.0};
	double up[3] = {1.0, 0.0, 0.0};

	pos[0] = sin( (time - start) / 10.0) * 20.0;
	pos[1] = pos[1];
	pos[2] = 44.0 + cos( (time - start) / 10.0) * 20.0;

	camera.lookAt(pos, target, up);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	wireframe->use();

	wireframe->setParam("FILL_COL", 0.0, 1.0, 0.0);
	wireframe->setParam("WIRE_COL", 1.0, 0.0, 0.0);
	wireframe->setParam("WIN_SCALE", 12.0, 12.0);

	tr.translate(-10.0, 0.0, 44.0);
	m = tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	toycar->draw();

	//toycar wheels
	tr.translate(-3.2, 0.6, 43.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-3.2, 0.6, 45.40);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 43.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(-4.8, 0.6, 45.40);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(3.5, 0.0, 44.0);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	charger->draw();

	// charger wheels
	tr.translate(4.9, 0.6, 40.82);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(4.9, 0.6, 46.6);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 40.82);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	tr.translate(2.0, 0.6, 46.6);
	rot.rotate(0.0, -1.6, 0.0);
	m = rot * tr * camera * pmatrix;
	wireframe->setParam("pmatrix", m);
	wheel->draw();

	starshader->use();
	m = camera * pmatrix;
	starshader->setParam("pmatrix", m);
	stars->update(time + 100.0);
	stars->draw();
}

RaceWorld::RaceWorld(int w, int h): Demo(w, h){
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_PROGRAM_POINT_SIZE);
	glEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
}

void RaceWorld::init(){
	Config *cfg = new Config("config.txt");

	timeframes.push_back(new Title(cfg));
	timeframes.push_back(new Intro(screenwidth, screenheight, cfg));
	timeframes.push_back(new TerrainAndCars(screenwidth, screenheight, cfg));
	timeframes.push_back(new CarDrop(screenwidth, screenheight, cfg));
	timeframes.push_back(new WholeWorld(screenwidth, screenheight, cfg));
	timeframes.push_back(new ToInfinity(screenwidth, screenheight, cfg));
}

