#include "Mesh3D.hpp"
#include <fstream>
#include <iostream>
#include <vector>
#include <cstdlib>
#include <cstring>
#include "MeshIndex.hpp"

using namespace std;
using namespace cg_engine;

Mesh3D::Mesh3D(const char *filename)
{
	has_normals = false;
	has_texture = false;

	loadOBJ(filename);
	buildMesh();
	createOpenGLBinding();
}


Mesh3D::~Mesh3D(void)
{
	glDeleteBuffers(BUFFERNUM, buffers);
	glDeleteVertexArrays(1, &vertexarray);
}

/**
  If all the information available, load it to
  the GPU
*/
void Mesh3D::createOpenGLBinding(){
	glGenVertexArrays(1, &vertexarray);
	glBindVertexArray(vertexarray);

	glGenBuffers(BUFFERNUM, buffers);

	glBindBuffer(GL_ARRAY_BUFFER, buffers[VERTEX]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * i_count * 3, vertices, GL_DYNAMIC_DRAW);
	glVertexAttribPointer(VERTEX, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glEnableVertexAttribArray(VERTEX);

	glBindBuffer(GL_ARRAY_BUFFER, buffers[NORMAL]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * i_count * 3, normals, GL_DYNAMIC_DRAW);
	glVertexAttribPointer(NORMAL, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glEnableVertexAttribArray(NORMAL);

	glBindBuffer(GL_ARRAY_BUFFER, buffers[TEXTURE]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * i_count * 2, textcoord, GL_DYNAMIC_DRAW);
	glVertexAttribPointer(TEXTURE, 2, GL_FLOAT, GL_FALSE, 0, 0);
	glEnableVertexAttribArray(TEXTURE);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, buffers[INDEX]);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLuint) * i_count, indices, GL_DYNAMIC_DRAW);

	glBindVertexArray(0);
}

/**
  Draw the model using vertex array
*/
void Mesh3D::draw(){
	glBindVertexArray(vertexarray);
	glDrawElements(GL_TRIANGLES, i_count, GL_UNSIGNED_INT, 0);
}

/**
  Parse OBJ file
*/
void Mesh3D::loadOBJ(const char *filename){
	ifstream obj(filename, ios::in);
	char line[200];
	char *d;

	while(obj.good()){
		obj.getline(line, 200);

		if(line[0] == 'v' && line[1] == ' '){
			MYSTRTOK(line, " \n", &d);
			vbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
			vbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
			vbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
		}
		if(line[0] == 'v' && line[1] == 't'){
			has_texture = true;
			MYSTRTOK(line, " \n", &d);
			tbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
			tbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
		}
		if(line[0] == 'v' && line[1] == 'n'){
			has_normals = true;
			MYSTRTOK(line, " \n", &d);
			nbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
			nbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
			nbuffer.push_back(atof(MYSTRTOK(NULL, " \n", &d)));
		}
		if(line[0] == 'f' && line[1] == ' '){
			ibuffer.push_back(new MeshIndex(line));
		}
	}

	obj.close();
}

/**
  Build mesh from the information available
  in OBJ file.

  For example in OBJ file the number of normals may differ
  from the number of vertices. OpenGL wants the same
  size of every buffer.
*/
void Mesh3D::buildMesh(){
	i_count   = ibuffer.size() * 3;

	vertices  = new GLfloat[i_count * 3];
	normals   = new GLfloat[i_count * 3];
	textcoord = new GLfloat[i_count * 2];
	indices   = new GLuint[i_count]; //FIXME some program use quads instead of triangels

	for(unsigned int i = 0; i < ibuffer.size(); i++){
		for(int j = 0; j < 3; j++){
			MeshIndex *mi = ibuffer.at(i);
			int vi = mi->getVertex(j);
			int ni = mi->getNormal(j);
			int ti = mi->getText(j);

			indices[i * 3 + j] = i * 3 + j;

			vertices[(i * 3 + j) * 3 + 0] = vbuffer.at(vi * 3 + 0);
			vertices[(i * 3 + j) * 3 + 1] = vbuffer.at(vi * 3 + 1);
			vertices[(i * 3 + j) * 3 + 2] = vbuffer.at(vi * 3 + 2);
			
			if(has_normals){
				normals[(i * 3 + j) * 3 + 0] = nbuffer.at(ni * 3 + 0);
				normals[(i * 3 + j) * 3 + 1] = nbuffer.at(ni * 3 + 1);
				normals[(i * 3 + j) * 3 + 2] = nbuffer.at(ni * 3 + 2);
			}
			
			if(has_texture){
				textcoord[(i * 3 + j) * 2  + 0] = tbuffer.at(ti * 2 + 0);
				textcoord[(i * 3 + j) * 2  + 1] = tbuffer.at(ti * 2 + 1);
			}
		}
	}
}
