use super::{fps::FpsValue, resource};
use crate::{
    RESOURCE_DIR,
    config::{self, Resolution},
};
use anyhow::{Context, Result};
use crevice::std140::AsStd140;
use glam::{Vec2, Vec4};
use image::{GenericImageView, ImageReader};
use sdl3::gpu::{
    Buffer, BufferBinding, BufferRegion, BufferUsageFlags, ColorTargetBlendState,
    ColorTargetDescription, CommandBuffer, CompareOp, CullMode, DepthStencilState, Device,
    FillMode, FrontFace, GraphicsPipeline, GraphicsPipelineTargetInfo, PrimitiveType,
    RasterizerState, RenderPass, ShaderFormat, Texture, TextureCreateInfo, TextureFormat,
    TextureRegion, TextureTransferInfo, TextureType, TextureUsage, TransferBuffer,
    TransferBufferLocation, TransferBufferUsage, VertexAttribute, VertexBufferDescription,
    VertexElementFormat, VertexInputRate, VertexInputState,
};
use std::{
    fmt::Write,
    io::Read,
    mem::size_of,
    ops::Range,
    path::Path,
    process,
    sync::mpsc::{self, Receiver, Sender},
    thread,
    time::Duration,
};

#[derive(AsStd140)]
struct Layer {
    offset: Vec2,
    color0: Vec4,
    color1: Vec4,
    color2: Vec4,
    color3: Vec4,
}

impl From<&config::TextLayer> for Layer {
    fn from(value: &config::TextLayer) -> Self {
        let (color0, color1, color2, color3) = match &value.color {
            config::TextColor::Gradient(c0, c1, c2, c3) => (
                Vec4::from(c0),
                Vec4::from(c1),
                Vec4::from(c2),
                Vec4::from(c3),
            ),
            config::TextColor::Flat(color) => {
                let color = Vec4::from(color);
                (color, color, color, color)
            }
        };

        Self {
            offset: value.offset.into(),
            color0,
            color1,
            color2,
            color3,
        }
    }
}
#[repr(transparent)]
#[derive(Copy, Clone)]
struct Instance {
    pos_12x_12y_text_8c: u32,
}

#[derive(AsStd140)]
struct Uniforms {
    resolution: Vec2,
}

type Update = (Option<Buffer>, usize);

pub struct Text {
    pipeline: GraphicsPipeline,
    font_texture: Texture<'static>,
    layers: Vec<Layer>,
    buffer: Buffer,
    instances: usize,
    rx: Receiver<Update>,
    timing: Option<Range<f32>>,
}

impl Text {
    pub fn new(
        gpu: &Device,
        target_format: TextureFormat,
        config: &config::Text,
        fps: &FpsValue,
    ) -> Result<Self> {
        let (code, stage) = resource::load_shader("text.vert")?;
        let vertex_shader = gpu
            .create_shader()
            .with_code(ShaderFormat::SpirV, &code, stage)
            .with_entrypoint(c"main")
            .with_uniform_buffers(2)
            .build()
            .context("Can't create vertex shader")?;

        let (code, stage) = resource::load_shader("text.frag")?;
        let fragment_shader = gpu
            .create_shader()
            .with_code(ShaderFormat::SpirV, &code, stage)
            .with_entrypoint(c"main")
            .with_storage_textures(1)
            .build()
            .context("Can't create fragment shader")?;

        let pipeline = gpu
            .create_graphics_pipeline()
            .with_vertex_shader(&vertex_shader)
            .with_fragment_shader(&fragment_shader)
            .with_vertex_input_state(
                VertexInputState::new()
                    .with_vertex_buffer_descriptions(&[VertexBufferDescription::new()
                        .with_slot(0)
                        .with_pitch(size_of::<Instance>() as u32)
                        .with_input_rate(VertexInputRate::Instance)
                        .with_instance_step_rate(0)])
                    .with_vertex_attributes(&[VertexAttribute::new()
                        .with_format(VertexElementFormat::Uint)
                        .with_location(0)
                        .with_buffer_slot(0)
                        .with_offset(0)]),
            )
            .with_primitive_type(PrimitiveType::TriangleStrip)
            .with_rasterizer_state(
                RasterizerState::new()
                    .with_fill_mode(FillMode::Fill)
                    .with_cull_mode(CullMode::None)
                    .with_front_face(FrontFace::Clockwise),
            )
            .with_depth_stencil_state(
                DepthStencilState::new()
                    .with_compare_op(CompareOp::Greater)
                    .with_enable_depth_test(false)
                    .with_enable_stencil_test(false),
            )
            .with_target_info(
                GraphicsPipelineTargetInfo::new()
                    .with_color_target_descriptions(&[ColorTargetDescription::new()
                        .with_format(target_format)
                        .with_blend_state(ColorTargetBlendState::default())])
                    .with_has_depth_stencil_target(false),
            )
            .build()
            .context("Can't create graphics pipeline")?;
        drop(fragment_shader);
        drop(vertex_shader);

        // Load font
        let font_texture = load_font(gpu)?;

        let layers = config.layers.iter().map(Into::into).collect();

        // Start text updater thread
        let (mut loader, rx) = Loader::new(gpu, fps.clone(), &config.sources)?;
        let refresh = config.refresh;
        thread::spawn(move || {
            loader.update().unwrap();
            let Some(refresh) = refresh else {
                return;
            };

            loop {
                thread::sleep(Duration::from_secs_f32(refresh));
                loader.update().unwrap();
            }
        });

        let Ok((Some(buffer), instances)) = rx.recv() else {
            panic!("Text update thread initialization failed");
        };

        Ok(Self {
            pipeline,
            font_texture,
            layers,
            buffer,
            instances,
            rx,
            timing: config.timing.clone(),
        })
    }

    pub fn draw(
        &mut self,
        command_buffer: &CommandBuffer,
        pass: &RenderPass,
        resolution: Resolution,
        time: f32,
    ) {
        if let Some(timing) = &self.timing {
            if time < timing.start || time >= timing.end {
                return;
            }
        }

        // Poll for changed buffer
        while let Ok((new_buffer, new_instances)) = self.rx.try_recv() {
            if let Some(new_buffer) = new_buffer {
                self.buffer = new_buffer;
            }
            self.instances = new_instances;
        }

        pass.bind_graphics_pipeline(&self.pipeline);
        pass.bind_vertex_buffers(
            0,
            &[BufferBinding::new()
                .with_buffer(&self.buffer)
                .with_offset(0)],
        );
        pass.bind_fragment_storage_textures(0, &[self.font_texture.clone()]);
        let param = Uniforms {
            resolution: Vec2::new(resolution.width as f32, resolution.height as f32),
        }
        .as_std140();
        command_buffer.push_vertex_uniform_data(0, &param);

        for layer in &self.layers {
            command_buffer.push_vertex_uniform_data(1, &layer.as_std140());
            pass.draw_primitives(
                4,
                self.instances.saturating_sub((time / 8.) as usize % 2), // TODO: Bad
                0,
                0,
            );
        }
    }
}

fn load_font(gpu: &Device) -> Result<Texture<'static>> {
    let image = ImageReader::open(Path::new(RESOURCE_DIR).join("font.png"))
        .context("Can't open font.png")?
        .decode()
        .context("Can't decode font.png")?;
    let (width, height) = image.dimensions();
    assert_eq!(width % 8, 0);
    let width = width / 8;

    // Start texture copy pass
    let command_buffer = gpu.acquire_command_buffer()?;
    let copy_pass = gpu.begin_copy_pass(&command_buffer)?;

    let texture = gpu.create_texture(
        TextureCreateInfo::new()
            .with_format(TextureFormat::R8Uint)
            .with_type(TextureType::_2D)
            .with_width(width)
            .with_height(height)
            .with_layer_count_or_depth(1)
            .with_num_levels(1)
            .with_usage(TextureUsage::GraphicsStorageRead),
    )?;

    let transfer_buffer = gpu
        .create_transfer_buffer()
        .with_size(width * height * u32::try_from(size_of::<u8>()).unwrap())
        .with_usage(TransferBufferUsage::Upload)
        .build()?;

    let mut buffer_mem = transfer_buffer.map::<u8>(gpu, false);
    let buf = buffer_mem.mem_mut();
    // Pack image to texture
    for i in 0..(width * height) {
        let x = i % width;
        let y = i / width;
        let mut texel = 0u8;
        for j in 0..8 {
            let pixel = image.get_pixel(x * 8 + j, y);
            let image::Rgba([red, ..]) = pixel;
            texel |= u8::from(red != 0) << j;
        }
        buf[usize::try_from(i).unwrap()] = texel;
    }
    buffer_mem.unmap();

    copy_pass.upload_to_gpu_texture(
        TextureTransferInfo::new()
            .with_transfer_buffer(&transfer_buffer)
            .with_offset(0),
        TextureRegion::new()
            .with_texture(&texture)
            .with_layer(0)
            .with_width(width)
            .with_height(height)
            .with_depth(1),
        false,
    );

    gpu.end_copy_pass(copy_pass);
    command_buffer.submit()?;

    Ok(texture)
}

struct Loader {
    gpu: Device,
    fps: FpsValue,
    sources: Vec<config::TextSource>,
    current_size: usize,
    transfer_buffer: TransferBuffer,
    target_buffer: Buffer,
    tx: Sender<Update>,
}

unsafe impl Send for Loader {}

impl Loader {
    fn new(
        gpu: &Device,
        fps: FpsValue,
        sources: &[config::TextSource],
    ) -> Result<(Self, Receiver<Update>)> {
        let initial_text = Self::compile_sources(sources, &fps)?;
        let (size, _) = Self::buffer_size(&initial_text);

        let (transfer_buffer, target_buffer) = Self::create_buffers(gpu, size)?;

        let (tx, rx) = mpsc::channel();
        tx.send((Some(target_buffer.clone()), 0)).unwrap();

        Ok((
            Self {
                gpu: gpu.clone(),
                fps,
                sources: Vec::from(sources),
                current_size: size,
                transfer_buffer,
                target_buffer,
                tx,
            },
            rx,
        ))
    }

    fn update(&mut self) -> Result<()> {
        let text = Self::compile_sources(&self.sources, &self.fps)?;
        let (size, chars) = Self::buffer_size(&text);

        // If buffers are too small, reallocate
        let reallocated = size > self.current_size;
        if reallocated {
            let (transfer_buffer, target_buffer) = Self::create_buffers(&self.gpu, size)?;
            self.transfer_buffer = transfer_buffer;
            self.target_buffer = target_buffer;
            self.current_size = size;
        }

        // Start copy pass
        let command_buffer = self.gpu.acquire_command_buffer()?;
        let copy_pass = self.gpu.begin_copy_pass(&command_buffer)?;

        let mut buffer_mem = self.transfer_buffer.map::<Instance>(&self.gpu, true);
        let mem = buffer_mem.mem_mut();
        let mut i = 0;
        let mut x: u32 = 0;
        let mut y: u32 = 0;
        for char in text.chars() {
            if char == '\n' {
                x = 0;
                y += 1;
                continue;
            }
            if !char.is_ascii() || char.is_ascii_whitespace() {
                x += 1;
                continue;
            }
            let mut ascii = [0u8];
            char.encode_utf8(&mut ascii);

            assert!(x < 2_u32.pow(12));
            assert!(y < 2_u32.pow(12));
            mem[i] = Instance {
                pos_12x_12y_text_8c: x << 20 | y << 8 | u32::from(ascii[0]),
            };

            x += 1;
            i += 1;
        }
        mem[i] = Instance {
            pos_12x_12y_text_8c: x << 20 | y << 8 | 127,
        };
        buffer_mem.unmap();

        copy_pass.upload_to_gpu_buffer(
            TransferBufferLocation::new()
                .with_transfer_buffer(&self.transfer_buffer)
                .with_offset(0),
            BufferRegion::new()
                .with_buffer(&self.target_buffer)
                .with_offset(0)
                .with_size(self.current_size as u32),
            true,
        );

        self.gpu.end_copy_pass(copy_pass);
        command_buffer.submit()?;

        // Send update to rendering thread
        self.tx
            .send((reallocated.then(|| self.target_buffer.clone()), chars))
            .unwrap();

        Ok(())
    }

    fn create_buffers(gpu: &Device, size: usize) -> Result<(TransferBuffer, Buffer)> {
        let transfer_buffer = gpu
            .create_transfer_buffer()
            .with_size(size as u32)
            .with_usage(TransferBufferUsage::Upload)
            .build()?;

        let buffer = gpu
            .create_buffer()
            .with_size(size as u32)
            .with_usage(BufferUsageFlags::Vertex)
            .build()?;

        Ok((transfer_buffer, buffer))
    }

    fn compile_sources(sources: &[config::TextSource], fps: &FpsValue) -> Result<String> {
        let mut buffer = String::new();
        for source in sources {
            match source {
                config::TextSource::File { name } => {
                    buffer.push_str(
                        &std::fs::read_to_string(Path::new(crate::RESOURCE_DIR).join(name))
                            .with_context(|| format!("Can't read {}", name.display()))?,
                    );
                }
                config::TextSource::Inline { text } => buffer.push_str(text),
                config::TextSource::Stdin => {
                    std::io::stdin().read_to_string(&mut buffer)?;
                }
                config::TextSource::Version => buffer.push_str(env!("CARGO_PKG_VERSION")),
                config::TextSource::Fps => write!(&mut buffer, "{}", fps.get())?,
                config::TextSource::Command { cmd, args } => {
                    let stdout = process::Command::new(cmd)
                        .args(args)
                        .output()
                        .context("Command execution failed")?
                        .stdout;
                    buffer.push_str(&String::from_utf8_lossy(&stdout));
                }
            }
        }
        Ok(buffer)
    }

    fn buffer_size(text: &str) -> (usize, usize) {
        let mut chars = 1; // One for the cursor
        for char in text.chars() {
            if !char.is_ascii() || char.is_ascii_whitespace() {
                continue;
            }
            chars += 1;
        }
        ((chars * size_of::<Instance>()).next_power_of_two(), chars)
    }
}
