use anyhow::{Result, bail};
use sdl3engine::{RESOURCE_DIR, SHADER_DIR, render::resource};
use std::{fs, path::Path};

pub fn compile_shader(path: impl AsRef<Path>) -> Result<Vec<u8>> {
    let path = path.as_ref();
    let stage = resource::infer_shader_stage(path)?;

    let (spv, warn) = resource::compile_shader(path, stage)?;

    if let Some(warn) = warn {
        eprintln!("In file {}: {}", path.display(), warn);
    }

    Ok(spv)
}

fn main() -> Result<()> {
    let Ok(shaders) = fs::read_dir(SHADER_DIR) else {
        bail!("{} unavailable", SHADER_DIR);
    };

    for entry in shaders.flatten() {
        let path = entry.path();
        match entry.file_type() {
            Ok(ft) if ft.is_file() => {
                let spv = compile_shader(&path)?;
                let out_dir = Path::new(RESOURCE_DIR).join(SHADER_DIR);
                fs::create_dir_all(&out_dir).ok();
                let mut out_file = out_dir.join(path.file_name().unwrap());
                let mut extension = path.extension().unwrap().to_os_string();
                extension.push(".spv");
                assert!(out_file.set_extension(extension));
                fs::write(out_file, spv).unwrap();
            }
            _ => println!("Skipping {}, not a regular file", path.display()),
        }
    }

    Ok(())
}
