#![cfg(feature = "audio")]

mod opus;

use anyhow::{Context, Result, bail};
use opus::OpusStream;
use sdl3::audio::{AudioCallback, AudioSpec, AudioStreamWithCallback};
use std::{
    alloc::{self, Layout},
    io::{Read, Seek},
    mem,
    time::Duration,
};

const NANOS_PER_SEC: u64 = 1_000_000_000;
const MAX_CHANNELS: usize = 2;
const MAX_OPUS_PACKET_MS: usize = 120;
const SAMPLE_RATE: i32 = 48000;
const BUFFER_LEN: usize = MAX_CHANNELS * MAX_OPUS_PACKET_MS * SAMPLE_RATE as usize;
type Buffer = [f32; BUFFER_LEN];

struct CallbackContext<T: Read + Seek> {
    opus: OpusStream<T>,
    buf: Box<Buffer>,
    skip: usize, // Samples
}

impl<T: Read + Seek> CallbackContext<T> {
    pub fn new(opus: OpusStream<T>) -> Self {
        let skip = opus.pre_skip() * usize::from(opus.channel_count());
        Self {
            opus,
            // Unsafe allocate buffer to ensure fixed size and no stack overflow
            buf: unsafe {
                Box::from_raw(
                    #[allow(clippy::cast_ptr_alignment, reason = "Alignment ensured in alloc")]
                    alloc::alloc(
                        Layout::from_size_align(
                            BUFFER_LEN * mem::size_of::<f32>(),
                            mem::align_of::<Buffer>(),
                        )
                        .unwrap(),
                    )
                    .cast::<Buffer>(),
                )
            },
            skip,
        }
    }
}

impl<T: Read + Seek + 'static> AudioCallback<f32> for CallbackContext<T> {
    fn callback(&mut self, stream: &mut sdl3::audio::AudioStream, requested: i32) {
        let mut put = 0;
        while put < requested as usize {
            let samples = self.opus.decode_packet(&mut self.buf);
            if samples == 0 {
                return;
            }

            let not_skipped = samples.saturating_sub(self.skip);
            let skipped = samples - not_skipped;
            self.skip = self.skip.saturating_sub(skipped);

            stream
                .put_data_f32(&self.buf[skipped..][..not_skipped])
                .unwrap();
            put += not_skipped;
        }
    }
}

pub struct Player<T: Read + Seek + 'static> {
    length: Duration,
    stream: AudioStreamWithCallback<CallbackContext<T>>,
}

impl<T: Read + Seek> Player<T> {
    pub fn new(rdr: T, audio: &sdl3::AudioSubsystem) -> Result<Self> {
        let opus = OpusStream::new(rdr)?;
        let channels: i32 = opus.channel_count().into();

        let nanos = opus.length() * NANOS_PER_SEC / SAMPLE_RATE as u64;
        let length = Duration::from_nanos(nanos);

        let callback = CallbackContext::new(opus);

        let spec = AudioSpec {
            freq: Some(SAMPLE_RATE),
            channels: Some(channels),
            format: Some(sdl3::audio::AudioFormat::F32LE),
        };

        let stream = audio
            .open_playback_stream(&spec, callback)
            .context("Can't open audio stream")?;
        Ok(Self { length, stream })
    }

    #[must_use]
    pub fn length(&self) -> &Duration {
        &self.length
    }

    pub fn pause(&mut self, state: bool) {
        if state {
            self.stream.pause().expect("Can't pause audio stream");
        } else {
            self.stream.resume().expect("Can't resume audio stream");
        }
    }

    pub fn seek(&mut self, to: Duration) -> Result<()> {
        let to_nanos = to.as_nanos() as u64;
        let pos_goal = (to_nanos * SAMPLE_RATE as u64) / NANOS_PER_SEC;

        {
            let Some(mut stream) = self.stream.lock() else {
                bail!("Can't lock audio stream");
            };
            if let Ok(skip) = stream.opus.seek(pos_goal) {
                stream.skip = skip * usize::from(stream.opus.channel_count());
            }
        }
        self.stream.clear().context("Can't clear audio stream")?;

        Ok(())
    }
}
