#include "sftg.hpp"
#include <math.h>
#include <fstream>
#include <cstring>
#include <SDL2/SDL.h>

using namespace cg_engine;

GroupTitle::GroupTitle(int w, int h, Config *c){
	start = 0;
	end = 4.817;

	drawshader = c->getShader("gloom");
	postprocessing = c->getShader("postgloom");

	background = new Mesh3D("data/demogroup.obj");
	creators   = c->getModel("creators");
	bgtex      = c->getTexture("background");

	pmatrix.perspective(35.0, (double)w / (double)h, 0.1, 1500);
	rb = new RenderBuffer(w, h, 0, 2, true);
	rb->addShader(postprocessing);
}

void GroupTitle::show(double time){
	rb->firstPass();
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	drawshader->use();
	Matrix r;
	r.rotate(1.51, 0.0, 0.0);
	Matrix t;
	t.translate(0.0, 0.0, -13.33);
	Matrix m;
	m = r * t * pmatrix;
	drawshader->setParam("pmatrix", m);
	drawshader->addTexture("img", bgtex);

	background->draw();
	// periods: 1.223->-16.0  2.823->-30.0 4.423-45.5>
	double newtime = time * 4.0;
	t.translate( (sin(newtime) + newtime) * -5.1, 0.0, -13.33);
	m = r * t * pmatrix;
	drawshader->setParam("pmatrix", m);
	creators->draw();

	rb->secondPass();
	rb->draw();
	glEnable(GL_DEPTH_TEST);
}

MarchingCubes::MarchingCubes(int w, int h, Config *c) {
	start = 4.817;
	end = 21.067;

	drawshader = c->getShader("gloom");
	postprocessing = c->getShader("postgloom");
	background = new Mesh3D("data/demogroup.obj");
	cube = new Mesh3D("data/cube.obj");

	bgtex = c->getTexture("background");
	cubetex = c->getTexture("cube");

	pm.perspective(35.0, (double)w / (double)h, 0.1, 150.0);
	for(int i = 0; i < 28; i++){
		x[i] = 10.0 + drand48() * 10.0;
		z[i] = -13.0 - drand48() * 3.0;
		t[i] = drand48();
	}

	rb = new RenderBuffer(w, h, 0, 2, true);
	rb->addShader(postprocessing);
}

void MarchingCubes::show(double time) {
	Matrix m;
	Matrix m2;

	rb->firstPass();
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	drawshader->use();

	m.rotate(1.51, 0.0, 0.0);
	m2.translate(0.0, 1.0, -13.33);
	m = m * m2 * pm;

	drawshader->addTexture("img", bgtex);
	drawshader->setParam("pmatrix", m);
	background->draw();

	drawshader->addTexture("img", cubetex);
	for(int i = 0; i < 28; i++){
		float r = 0.4341; // because our cube has 0.614 length
		float a = fmod((time + t[i]) - 2.0 * r, 2.0 * r) - r;
		float y = sqrt( (r * r) - (a * a) );
		m.translate(x[i] - (time + t[i]), -2.0 + y, z[i]);
		m2.rotate(0.0, 0.0, (time + t[i]) * 2.0);
		m = m2 * m * pm;

		drawshader->setParam("pmatrix", m);
		cube->draw();
	}

	rb->secondPass();
	rb->draw();
	glEnable(GL_DEPTH_TEST);
}

BouncingCubes::BouncingCubes(int w, int h, Config *c) {
	start = 21.067;
	end = 33.9;
	pm.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);

	cube = new Mesh3D("data/cube.obj");
	tunnel = new Mesh3D("data/tunnel.obj");

	tunneltex = c->getTexture("circuit");
	cubetex = c->getTexture("cube");

	drawshader = c->getShader("default");
	postprocessing = c->getShader("dof");
	
	for(int i = 0; i < 120; i++){
		double s = -1.0;
		if(drand48() - 0.5 > 0.0) s = 1.0;
		x[i] = (0.25 + drand48() * 2.2) * s;
		s = -1.0;
		if(drand48() - 0.5 > 0.0) s = 1.0;
		y[i] = (0.25 + drand48() * 2.2) * s;
		z[i] = drand48() * 34.0;

		ro[i] = drand48();
	}

	rb = new RenderBuffer(w, h, 0, 1, true);
	rb->addShader(postprocessing);
}

void BouncingCubes::show(double time) {
	Matrix r;
	Matrix m;

	rb->firstPass();

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	r.rotate(4.711, 0.0, time / 15.0);

	drawshader->use();
	drawshader->addTexture("img", tunneltex);
	for(int i = 0; i < 5; i++) {
		m.translate(0.0, 0.0, -28.0 * (double)i + (time - start));
		m = r * m * pm;
		drawshader->setParam("pmatrix", m);
		tunnel->draw();
	}

	drawshader->addTexture("img", cubetex);
	for(int i = 0; i < 50; i++){
		m.translate(x[i], y[i], z[i] - (time - start) * 3.0);
		r.rotate(ro[i] - (time - start), 0.0, 0.0);
		m = r * m * pm;
		drawshader->setParam("pmatrix", m);
		cube->draw();
	}

	rb->secondPass();
	rb->draw();
	glEnable(GL_DEPTH_TEST);
}

BouncingCubes2::BouncingCubes2(int w, int h, Config *c){
	start = 33.9;
	end = 42.067;
	pm.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);

	cube = new Mesh3D("data/cube.obj");
	tunnel = new Mesh3D("data/tunnel.obj");

	tunneltex = c->getTexture("circuit");
	cubetex = c->getTexture("cube");

	drawshader = c->getShader("default");

	for(int i = 0; i < 50; i++){
		z[i]  = -2.0 - drand48() * 4.0;
		x[i]  = drand48() * -25.0;
		y[i]  = drand48() * 6.28;
		ro[i] = drand48();
	}
}

void BouncingCubes2::show(double time){
	Matrix r;
	Matrix m;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	r.rotate(4.711, 4.711, 0.0);
	m.translate(-3.51 - (time - start), 0.0, -3.0);

	drawshader->use();
	drawshader->addTexture("img", tunneltex);
	r = r * m * pm;
	drawshader->setParam("pmatrix", r);
	tunnel->draw();

	drawshader->addTexture("img", cubetex);
	for(int i = 0; i < 50; i++){
		float ya = sin(time - start + y[i]) / 2.0;
		m.translate(x[i] + (time - start), ya, z[i]);
		r.rotate(ro[i] + time - start, 0.3, -time - start);
		r = r * m * pm;
		drawshader->setParam("pmatrix", r);
		cube->draw();
	}
}

Sinkhole::Sinkhole(int w, int h, Config *c){
	start      = 42.067;
	end        = 61.692;

	sinkhole   = new Mesh3D("data/sinkhole.obj");
	cube       = new Mesh3D("data/cube.obj");
	sinktex    = c->getTexture("sinkhole");
	cubetex    = c->getTexture("cube");
	drawshader = c->getShader("default");
	postprocessing = c->getShader("postprocessing");
	pm.perspective(90.0, (double)w / (double)h, 0.1, 1500.0);
	camera.lookAt(pos, target, up);

	for(int i = 0; i < 30; i++){
		double rnd = floor(drand48() * 10.0) / 10.0;
		delay[i] = start - 14.0 + rnd * 10.0;
	}

	path[0] = new Path();
	path[1] = new Path();
	path[2] = new Path();
	path[3] = new Path();

	path[0]->addPoint(Point3D(-0.10901901125907898, 0.3795681893825531, 1.48349928855896));
	path[0]->addPoint(Point3D(0.7171065807342529, -1.0550377368927002, 6.251735210418701));
	path[0]->addPoint(Point3D(-0.13532212376594543, 17.861162185668945, 5.222771644592285));
	path[0]->addPoint(Point3D(-0.11561470478773117, 19.08551597595215, 5.252367973327637));

	path[1]->addPoint(Point3D(0.3075043559074402, 0.32253849506378174, 1.550645351409912));
	path[1]->addPoint(Point3D(-1.117918848991394, -0.23861470818519592, 6.360003471374512));
	path[1]->addPoint(Point3D(17.801517486572266, -0.5849689245223999, 5.1178436279296875));
	path[1]->addPoint(Point3D(19.022666931152344, -0.6785464882850647, 5.136366367340088));

	path[2]->addPoint(Point3D(-0.10903352499008179, -0.29807135462760925, 1.9149399995803833));
	path[2]->addPoint(Point3D(-0.07851671427488327, 1.0912922620773315, 6.767298698425293));
	path[2]->addPoint(Point3D(0.13440543413162231, -17.82384490966797, 5.4330339431762695));
	path[2]->addPoint(Point3D(0.09075963497161865, -19.047754287719727, 5.4542036056518555));

	path[3]->addPoint(Point3D(-0.3007117509841919, 0.1580325961112976, 1.4463558197021484));
	path[3]->addPoint(Point3D(1.0878019332885742, 0.10064911842346191, 6.298714637756348));
	path[3]->addPoint(Point3D(-17.82791519165039, 0.25361379981040955, 4.964449882507324));
	path[3]->addPoint(Point3D(-19.050750732421875, 0.32092368602752686, 4.985620021820068));

	rb = new RenderBuffer(w, h, 0, 1, true);
	rb->addShader(postprocessing);
}

void Sinkhole::show(double time){
	rb->firstPass();

	glClearColor(0.0, 0.0, 0.0, 1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	drawshader->use();
	drawshader->addTexture("img", sinktex);
	Matrix m;

	if(time > 56.694){
		pos[1] = 8.0 - (time - 56.694);
	}

	camera.lookAt(pos, target, up);
	m = camera * pm;
	drawshader->setParam("pmatrix", m);
	sinkhole->draw();

	drawshader->addTexture("img", cubetex);
	for(int i = 0; i < 30; i++){
		if(time > delay[i]){
			double t = (time - delay[i]) / 24.0;
			if(t > 1.0) t = 1.0;
			for(int j = 0; j < 4; j++){
				Point3D p = path[j]->bezier(1.0 - t);
				if(t == 1.0) p.z = p.z - time - delay[i] - 24.0;
				m.translate(p.x, p.y, p.z);
				Matrix r;
				r.rotate(-1.57, -1.55, 0.0);
				Matrix r2;
				r2.rotate(delay[i] + time, delay[i] + time / 2.0, delay[i]);
				m = r2 * m * r * camera * pm;
				drawshader->setParam("pmatrix", m);
				cube->draw();
			}
		}
	}

	rb->secondPass();
	if(time > 56.694){
		postprocessing->setParam("fade", (float)( (time - 56.694) / (end - 56.694)));
	} else {
		postprocessing->setParam("fade", 0.0f);
	}
	rb->draw();
	glEnable(GL_DEPTH_TEST); // the second pass switch off depth test
}

Greetings::Greetings(int w, int h, Config *c) {
	start = 61.692;
	end   = 91.017;

	cube = c->getModel("cube");
	cubetex = c->getTexture("cube");
	groups = c->getModel("groups");
	groupstex = c->getTexture("groups");
	glasstube = c->getModel("glasstube");
	glasstex = c->getTexture("glasstube");
	drawshader = c->getShader("default");
	glassshader = c->getShader("glass");
	pmatrix.perspective(55.0, (double)w / (double)h, 0.1, 1500.0);

	for(int i = 0; i < 12; i++){
		cpos[i * 3 + 0] = (1.0 + drand48()) * sin(drand48() * 3.14);
		cpos[i * 3 + 1] = drand48() * 12.0;
		cpos[i * 3 + 2] = (1.0 + drand48()) * sin(drand48() * 3.14);
	}
}

void Greetings::show(double time){
	double          pos[3] = {0.0, 0.0, 0.0};
        double          target[3] = {-22.0, -540, 0.0};
        double          up[3] = {0.0, -1.0, 0.0};
	glClearColor(0.6, 0.6, 0.6, 1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	Matrix camera;
	pos[1] = 200.0 - (time - start);
	pos[0] = (drand48() - 0.5) / 100.0;
	pos[2] = (drand48() - 0.5) / 100.0;
	target[0] = -22.0 + sin(time) * 13.0;
	target[2] = cos(time) * 13.0;
	camera.lookAt(pos, target, up);
	camera = camera * pmatrix;

	drawshader->use();
	drawshader->addTexture("img", groupstex);
	drawshader->setParam("pmatrix", camera);
	groups->draw();

	drawshader->addTexture("img", cubetex);
	for(int i = 0; i < 12; i ++){
		Matrix posm;
		posm.translate(cpos[i * 3], 200.0 + cpos[i * 3 + 1] - (time - start) * 1.5, cpos[i * 3 + 2]);
		posm = posm * camera;
		drawshader->setParam("pmatrix", posm);
		cube->draw();
	}

	glassshader->use();
	glassshader->setParam("pmatrix", camera);
	glassshader->setParam("time", (float)(time - start));
	glassshader->addTexture("glasstex", glasstex);
	glasstube->draw();
}

RobotCave::RobotCave(int w, int h, Config *c) {
	start = 91.017;
	end   = 98.0;

	floor = c->getModel("robotfloor");
	floortex = c->getTexture("robotfloor");
	cave = c->getModel("robottunel");
	cavetex = c->getTexture("robottunel");
	cube = c->getModel("cube");
	cubetex = c->getTexture("cube");

	robot = parsehierfile(c->getFile("robothier"), cube);
	Matrix t;
	t.translate(0.0,1.1,-20.0);
	Matrix r;
	r.rotate(0.0, 4.68, 4.68);
	hiertoindividual(robot->getNode("center"), r * t);

	drawshader = c->getShader("hemisphere");

	pmatrix.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);
	for(int i =0; i < ROBOTCUBENUM; i++){
		starttime[i] = 2.0 + start + drand48() * 4.0;
	}
}

void RobotCave::show(double time) {
	Matrix move;
	Matrix rot;
	Matrix m;
	Matrix n;
	Matrix t;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glClearColor(0.0, 0.0, 0.0, 1.0);

	t.translate(57.0, -3.0, 0.0);
	m.rotate(0.0, 1.553846, 0.0);
	n = t * m;
	m = t * m * pmatrix;

	drawshader->use();
	drawshader->addTexture("img", floortex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	floor->draw();

	drawshader->addTexture("img", cavetex);
	drawshader->setParam("pmatrix", m);
	cave->draw();

	for(int i = 1; i < 4; i++){
		t.translate(100.0 * i + 51.0, -3.0, 0.0);
		m.rotate(0.0, 1.553846, 0.0);
		n = t * m;
		m = t * m * pmatrix;
		drawshader->addTexture("img", floortex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		floor->draw();
		drawshader->addTexture("img", cavetex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		cave->draw();
	}

	unsigned int counter = 0;
	for(auto &i: robot->all_nodes){
		t = i.second->getMatrix();
		if(time < starttime[counter]){
			double px = 0.0 + (t.matrix()[12] - 0.0) * ((time - start) / (starttime[counter] - start));
			double py = 10.0 + (t.matrix()[13] - 10.0) * ((time - start) / (starttime[counter] - start));
			double pz = 0.0 + (t.matrix()[14] - 0.0) * ((time - start) / (starttime[counter] - start));
			t.translate(px, py, pz);
		} else {
			t.translate(t.matrix()[12], t.matrix()[13], t.matrix()[14]);
		}
		m = t * pmatrix;
		n = t;
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		drawshader->addTexture("img", cubetex);
		cube->draw();

		counter++;
	}
}

RobotIntro::RobotIntro(int w, int h, Config *c) {
	start = 98.0;
	end   = 102.476;

	floor = c->getModel("robotfloor");
	floortex = c->getTexture("robotfloor");
	cave = c->getModel("robottunel");
	cavetex = c->getTexture("robottunel");
	cube = c->getModel("cubel");
	cubetex = c->getTexture("cube");
	robot = parsehierfile(c->getFile("robothier"), cube);
	drawshader = c->getShader("hemisphere");
	pmatrix.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);
}

/* Drawing hierarchy in a recursive way */
void hierdraw(Node *root, Matrix m, ShaderProgram *s, Matrix pmatrix) {
	Matrix rm = root->getMatrix();
	Matrix parentm = rm * m;
	Matrix t = parentm * pmatrix;
	s->setParam("pmatrix", t);
	s->setParam("nmatrix", parentm);
	root->draw();
	for(unsigned int i = 0; i < root->childs.size(); i++){
		hierdraw(root->childs.at(i), parentm, s, pmatrix);
	}
}

/* Update child matrices with parents information. No need recursion if you call this function */
void hiertoindividual(Node *root, Matrix m) {
	Matrix parent = root->getMatrix() * m;
	root->updateStateMatrix(parent);
	for(unsigned int i = 0; i < root->childs.size(); i++){
		hiertoindividual(root->childs.at(i), parent);
	}
}

void RobotIntro::show(double time) {
	double pos[] =    {0.0, 0.0, 0.0};
	double target[] = {0.423, 2.2, -22.0};
	double up[] =     {0.0, 1.0, 0.0};

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glClearColor(0.0, 0.0, 0.0, 1.0);


	Matrix camera;
	if(time < 99.5){
		pos[0] = -3.0;
		pos[1] = 2.2;
		pos[2] = -16.2;
	} else if(time < 100.983){
		pos[0] = 3.96;
		pos[1] = -2.93;
		pos[2] = -16.2;
	} else if(time < 102.476){
		pos[0] = 0.41;
		pos[1] = -1.27;
		pos[2] = -3.7;
	}
	camera.lookAt(pos, target, up);
	Matrix m;
	Matrix t;
	Matrix n;
	t.translate(57.0, -3.0, 0.0);
	m.rotate(0.0, 1.553846, 0.0);
	n = t * m * camera;
	m = t * m * camera * pmatrix;

	drawshader->use();
	drawshader->addTexture("img", floortex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	floor->draw();

	drawshader->addTexture("img", cavetex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	cave->draw();

	for(int i = 1; i < 4; i++){
		t.translate(100.0 * i + 51.0, -3.0, 0.0);
		m.rotate(0.0, 1.553846, 0.0);
		n = t * m * camera;
		m = t * m * camera * pmatrix;
		drawshader->addTexture("img", floortex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		floor->draw();
		drawshader->addTexture("img", cavetex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		cave->draw();
	}


	Matrix move;
	move.translate(0.7, 1.01, -22.0);
	Matrix rot;
	rot.rotate(0.0, 4.68, 4.67);
	drawshader->addTexture("img", cubetex);
	hierdraw(robot->getNode("center"), rot * move * camera, drawshader, pmatrix);
}

RobotRunning::RobotRunning(int w, int h, Config *c) {
	start = 102.476;
	end   = 121.639;

	floor = c->getModel("robotfloor");
	floortex = c->getTexture("robotfloor");
	cave = c->getModel("robottunel");
	cavetex = c->getTexture("robottunel");

	cube = c->getModel("cubel");
	robot = parsehierfile(c->getFile("robothier"), cube);
	drawshader = c->getShader("hemisphere");
	cubetex = c->getTexture("cube");
	pmatrix.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);
}

int indentcount(const char *txt) {
	unsigned int i;
	for(i=0; i < strlen(txt); i++){
		if(txt[i] != ' '){
			return i;
		}
	}
	return 0;
}

Node *createhier(char *text, char **id) {
	Node *entry;
	char *val;
	float x;

	*id = strtok(text, " ");
	Matrix m;
	entry = new Node();
	for(int i = 0; i < 16; i++){
		val = strtok(NULL, " ");
		x = atof(val);
		m.modify(i, x);
	}
	m.transpose();
	entry->updateStateMatrix(m);
	return entry;
}

Hierarchy3D* parsehierfile(string filename, Mesh3D *mesh) {
	char line[200];
	char *id;
	Node *child;
	vector<Node*> parents;
	Hierarchy3D *hier;

	hier = new Hierarchy3D();
	ifstream hierfile(filename, ios::in);
	while(hierfile.good()){
		hierfile.getline(line, 200);
		if(strlen(line) > 0) {
			unsigned int spacecount = indentcount(line);
			child = createhier(&line[spacecount], &id);
			hier->addNode(id, child);
			child->setMesh(mesh);
			if(spacecount == parents.size()){
				parents.push_back(child);
			}
			parents.at(spacecount) = child;
			if(spacecount > 0){
				parents.at(spacecount-1)->addChild(child);
			}
		}
	}
	hierfile.close();

	return hier;
}

void RobotRunning::show(double time) {
	Matrix m;
	Matrix t;
	Matrix n;
	Matrix camera;
	double pos[] =    {0.0, 0.0, 0.0};
	double target[] = {0.423, 2.2, -22.0};
	double up[] =     {0.0, 1.0, 0.0};
	double runpos;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	if(time < 109.237){
		runpos = (time - start) * 7.0;
	} else if(time < 112.47){
		runpos = (time - 112.47) * 7.0;
		pos[1] = 4.08;
		pos[2] = -5.4;
		target[0] = 0.423077;
		target[1] = 1.2 + fabs(sin(time * 3.0)/2.0);
		target[2] = -72.0 + runpos;
	} else {
		runpos = (time - 115.5) * 7.0;
		target[1] = 1.2;
		target[2] = -72.0 + runpos;
		pos[0] = 2.69;
		pos[1] = 4.19;
		pos[2] = runpos - 60.0;
	}

	camera.lookAt(pos, target, up);

	t.translate(57.0, -3.0, 0.0);
	m.rotate(0.0, 1.553846, 0.0);
	n = t * m * camera;
	m = t * m * camera * pmatrix;

	drawshader->use();
	drawshader->addTexture("img", floortex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	floor->draw();

	drawshader->addTexture("img", cavetex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	cave->draw();
	for(int i = 1; i < 4; i++){
		t.translate(100.0 * i + 51.0, -3.0, 0.0);
		m.rotate(0.0, 1.553846, 0.0);
		n = t * m * camera;
		m = t * m * camera * pmatrix;
		drawshader->addTexture("img", floortex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		floor->draw();
		drawshader->addTexture("img", cavetex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		cave->draw();
	}


	Matrix j;
	j.rotate(0.0, sin(time * 3.0), 0.0);
	Matrix b;
	b.rotate(0.0, -sin(time * 3.0), 0.0);
	Matrix elbowrot;
	elbowrot.rotate(0.0, -1.47, 0.0);
	Matrix lkneerot;
	Matrix rkneerot;
	double s = sin(time * 3.0) * 2.0 + 2.0;
	if(s > 1.0) s = 1.0;
	double s2 = -sin(time * 3.0) * 2.0 + 2.0;
	if(s2 > 1.0) s2 = 1.0;
	lkneerot.rotate(0.0, s, 0.0);
	rkneerot.rotate(0.0, s2, 0.0);
	robot->getNode("leftleg")->updateMatrix(j, false);
	robot->getNode("leftknee")->updateMatrix(lkneerot, false);
	robot->getNode("rightshoulder")->updateMatrix(j, false);
	robot->getNode("rightleg")->updateMatrix(b, false);
	robot->getNode("rightknee")->updateMatrix(rkneerot, false);
	robot->getNode("leftshoulder")->updateMatrix(b, false);
	robot->getNode("leftelbow")->updateMatrix(elbowrot, false);
	robot->getNode("rightelbow")->updateMatrix(elbowrot, false);

	m.translate(0.423077, 1.2 + fabs(sin(time * 3.0)/2.0), -72.0 + runpos);
	Matrix rot;
	rot.rotate(0.0, 4.68, 4.67);
	t = rot * m * camera;
	drawshader->addTexture("img", cubetex);
	hierdraw(robot->getNode("center"), t, drawshader, pmatrix);
}

FallingApart::FallingApart(int w, int h, Config *c) {
	start = 121.639;
	end = 128.0;

	floor = c->getModel("robotfloor");
	floortex = c->getTexture("robotfloor");
	cave = c->getModel("robottunel");
	cavetex = c->getTexture("robottunel");

	cube = c->getModel("cubel");
	robot = parsehierfile(c->getFile("robothier"), cube);
	bounchofcubes = parsehierfile(c->getFile("robothier"), cube);
	drawshader = c->getShader("hemisphere");
	layershader = c->getShader("layer");
	cubetex = c->getTexture("cube");

	pmatrix.perspective(35.0, (double)w / (double)h, 0.1, 1500.0);
	Matrix t;
	Matrix r;
	t.translate(-0.166923, 1.2, -24.043);
	r.rotate(-0.097242, 4.68, 4.67);
	hiertoindividual(bounchofcubes->getNode("center"), r * t);

	for(int i = 0; i < ROBOTCUBENUM; i++){
		direction[i*2] = drand48() - 0.5;
		direction[i*2+1] = drand48() - 0.5;
	}

	layer = new Sprite();
	titletex = c->getTexture("title");
}

void FallingApart::show(double time){
	Matrix m;
	Matrix n;
	Matrix t;
	Matrix camera;
	double pos[] =    {0.0, 0.0, 0.0};
	double target[] = {0.423, 2.2, -22.0};
	double up[] =     {0.0, 1.0, 0.0};

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	pos[0] = 0.0 + sin(0.56) * 12.0;
	pos[1] = 0.94;
	pos[2] = -22.0 + cos(0.56) * 12.0;
	target[1] = 0.5;
	camera.lookAt(pos, target, up);

	t.translate(57.0, -3.0, 0.0);
	m.rotate(0.0, 1.553846, 0.0);
	n = t * m * camera;
	m = t * m * camera * pmatrix;

	drawshader->use();
	drawshader->addTexture("img", floortex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	floor->draw();

	drawshader->addTexture("img", cavetex);
	drawshader->setParam("pmatrix", m);
	drawshader->setParam("nmatrix", n);
	cave->draw();
	for(int i = 1; i < 4; i++){
		t.translate(100.0 * i + 51.0, -3.0, 0.0);
		m.rotate(0.0, 1.553846, 0.0);
		n = t * m * camera;
		m = t * m * camera * pmatrix;
		drawshader->addTexture("img", floortex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		floor->draw();
		drawshader->addTexture("img", cavetex);
		drawshader->setParam("pmatrix", m);
		drawshader->setParam("nmatrix", n);
		cave->draw();
	}


	drawshader->addTexture("img", cubetex);

	if(time < 125.0){
		//Limping
		Matrix ll;
		Matrix rl;
		ll.rotate(0.0, abs(sin(time * 2.0) / 2.0), 0.0);
		robot->getNode("leftleg")->updateMatrix(ll, false);
		rl.rotate(0.0, -abs(sin(time * 2.0) / 3.0), 0.0);
		robot->getNode("rightleg")->updateMatrix(rl, false);
		robot->getNode("rightknee")->updateMatrix(ll, false);
		robot->getNode("leftshoulder")->updateMatrix(rl, false);
		robot->getNode("rightshoulder")->updateMatrix(ll, false);

		// whole robot movement
		t.translate(-0.166923, 1.2, -27.4 + (time - start));
		m.rotate(0.0 - abs(sin(time * 2.0)) / 10.0, 4.68, 4.67);
		m = m * t * camera;
		hierdraw(robot->getNode("center"), m, drawshader, pmatrix);
	} else {
		// falling apart
		int count = 0;
		for(auto &i: bounchofcubes->all_nodes){
			t = i.second->getMatrix();
			double rtime = time - 125.0;
			rtime = rtime * rtime;
			t.matrix()[12] = t.matrix()[12] + (time - 125.0) * direction[count*2];
			t.matrix()[14] = t.matrix()[14] + (time - 125.0) * direction[count*2+1];
			if(t.matrix()[13] - (4.9 * rtime) > -2.57){
				t.matrix()[13] = t.matrix()[13] - (4.9 * rtime);
			} else t.matrix()[13] = -2.57;
			m = t * camera * pmatrix;
			n = t * camera;
			drawshader->setParam("pmatrix", m);
			drawshader->setParam("nmatrix", n);
			drawshader->addTexture("img", cubetex);
			cube->draw();
			count++;
		}
		if(time > 126.573){
			layershader->use();
			layershader->addTexture("img", titletex);
			layer->draw();
		}
	}
}

MyDemo::MyDemo(int w, int h): Demo(w, h){
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_PROGRAM_POINT_SIZE);
	glEnable(GL_BLEND);
	glEnable(GL_ALPHA_TEST);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void MyDemo::init(){
	Config *conf = new Config("data/config.txt");

	scenes.push_back(new GroupTitle(screenwidth, screenheight, conf));
	scenes.push_back(new MarchingCubes(screenwidth, screenheight, conf));       // 4.817 - 21.067 Marching cubes
	scenes.push_back(new BouncingCubes(screenwidth, screenheight, conf));       // 21.067 - 33.9 Bouncing cubes
	scenes.push_back(new BouncingCubes2(screenwidth, screenheight, conf));      // 33.9 - 42.067 Different bouncing cubes
	scenes.push_back(new Sinkhole(screenwidth, screenheight, conf));            // 42.067 - 61.692 Sulyeszto. Big dig
	scenes.push_back(new Greetings(screenwidth, screenheight, conf));           // 61.692 - 71.017 Greetings, everything is falling
	scenes.push_back(new RobotCave(screenwidth, screenheight, conf));           // 71.017 - 98 Transcendental state, the falling is ended. We see the big robot in blur
	scenes.push_back(new RobotIntro(screenwidth, screenheight, conf));          // 98 - 102.476 We see the robot in different angles, but only small parts
	scenes.push_back(new RobotRunning(screenwidth, screenheight, conf));        // 102.476 - 121.639 Running robot
	scenes.push_back(new FallingApart(screenwidth, screenheight, conf));        // 121.639 - 128.0 The robot is falling apart
}

