#include <math.h>
#include <stdio.h>
#include <string.h>
#include "Matrix.hpp"

using namespace cg_engine;

Matrix Matrix::operator * (Matrix &right) {
   int i,x,y;
   Matrix c;

   for(i = 0; i < 16; i++){
      x = i & 12;
      y = i % 4;
      c.m[i] = this->m[x + 0] * right.m[y + 0] +
               this->m[x + 1] * right.m[y + 4] +
               this->m[x + 2] * right.m[y + 8] +
               this->m[x + 3] * right.m[y + 12];
   }
   return c;
}

Matrix::Matrix() {
	for(int i = 0; i < 16; i++) {
		m[i] = 0.0;
	}
}

Matrix::~Matrix() {
}

void Matrix::perspective(double fov, double aspect, double near, double far){
  double range;

  range = tan(fov * M_PI/360.0) * near;
  m[0] = (2.0 * near) / ((range * aspect) - (-range * aspect));
  m[5] = (2.0 * near) / (2.0 * range);
  m[10] = -(far + near) / (far - near);
  m[11] = -1.0;
  m[14] = -(2.0 * far * near) / (far - near);
}

void Matrix::translate(double x, double y, double z){
   int i;

   /* Identity matrix */
   for(i = 0; i < 16; i++){
      if( (i % 5) == 0){
         m[i] = 1.0;
      }
      else m[i] = 0.0;
   }

   m[12] = x;
   m[13] = y;
   m[14] = z;
}

void Matrix::scale(double sx, double sy, double sz){
   int i;

   for(i = 0; i < 16; i++){
      if( (i%5) == 0) m[i] = 1.0; else m[i] = 0.0;
   }

   m[0] = sx;
   m[5] = sy;
   m[10] = sz;
   m[15] = 1.0;
}

void Matrix::rotate(double rx, double ry, double rz){
   int i;
   /* Identity matrix */
   for(i = 0; i < 16; i++){
      if( (i % 5) == 0){
         m[i] = 1.0;
      }
      else m[i] = 0.0;
   }

   /* General rotation */
   m[0]  =  cosf(ry) * cosf(rz);
   m[4]  = -cosf(rx) * sinf(rz) + sinf(rx) * sinf(ry) * cosf(rz);
   m[8]  =  sinf(rx) * sinf(rz) + cosf(rx) * sinf(ry) * cosf(rz);
   m[12] =  0.0;
   m[1]  =  cosf(ry) * sinf(rz);
   m[5]  =  cosf(rx) * cosf(rz) + sinf(rx) * sinf(ry) * sinf(rz);
   m[9]  = -sinf(rx) * cosf(rz) + cosf(rx) * sinf(ry) * sinf(rz);
   m[13] =  0.0;
   m[2]  = -sinf(ry);
   m[6]  =  sinf(rx) * cosf(ry);
   m[10] =  cosf(rx) * cosf(ry);
   m[14] =  0.0;
   m[3]  =  0.0;
   m[7]  =  0.0;
   m[11] =  0.0;
   m[15] =  1.0;

}

void normalize(double *vector, int size){
   float d = 0.0;
   int i;

   for(i = 0; i < size; i++) d += vector[i] * vector[i];
   d = sqrtf(d);
   for(i = 0; i < size; i++) vector[i] = vector[i] / d;
}

void crossproduct3v(double *a, double *b, double *result){
   result[0] = a[1] * b[2] - a[2] * b[1];
   result[1] = a[2] * b[0] - a[0] * b[2];
   result[2] = a[0] * b[1] - a[1] * b[0];
}

float dotproduct(double *a, double *b, int size){
   int i;
   float result = 0.0;

   for(i = 0; i < size; i++) result += a[i] * b[i];
   return(result);
}

void Matrix::lookAt(double *eye, double *target, double *up){
   int i;
   double n[3];
   double u[3];
   double v[3];
   double d[3];

   for(i = 0; i < 3; i++) n[i] = eye[i] - target[i];
   crossproduct3v(up, n, u);
   crossproduct3v(n, u, v);
   normalize(u, 3);
   d[0] = dotproduct(eye, u, 3) * -1.0;
   normalize(v, 3);
   d[1] = dotproduct(eye, v, 3) * -1.0;
   normalize(n, 3);
   d[2] = dotproduct(eye, n, 3) * -1.0;

   m[0]  = u[0];
   m[4]  = u[1];
   m[8]  = u[2];
   m[12] = d[0];
   m[1]  = v[0];
   m[5]  = v[1];
   m[9]  = v[2];
   m[13] = d[1];
   m[2]  = n[0];
   m[6]  = n[1];
   m[10] = n[2];
   m[14] = d[2];
   m[3]  = 0.0;
   m[7]  = 0.0;
   m[11] = 0.0;
   m[15] = 1.0;
}

GLfloat *Matrix::matrix() {
	return(m);
}

void Matrix::modify(int i, GLfloat value) {
	if(i >= 0 && i < 16){
		m[i] = value;
	}
}

/**
 * Print out the content of the matrix
 * */
void Matrix::dump() {
	int i;
	for(i = 0; i < 16; i++){
		printf("%f, ", m[i]);
	}
	printf("\n");
}

/**
 * Simple function to convert Blender coordinate system to OpenGL coordinate system
 * */
void Matrix::blender2opengl() {
	float tmp;
	tmp = m[1];
	m[1] = m[2];
	m[2] = tmp * -1.0;
}

void Matrix::transpose() {
	float tmp;

	tmp = m[1];
	m[1] = m[4];
	m[4] = tmp;

	tmp = m[2];
	m[2] = m[8];
	m[8] = tmp;

	tmp = m[3];
	m[3] = m[12];
	m[12] = tmp;

	tmp = m[6];
	m[6] = m[9];
	m[9] = tmp;

	tmp = m[7];
	m[7] = m[13];
	m[13] = tmp;

	tmp = m[11];
	m[11] = m[14];
	m[14] = tmp;
}

/*
void Matrix::blender2openglmatrix(){
   Matrix *rot;
   rot->rotate(0.0, -90.0, 0.0);
   this = this * rot;
}
*/
