#include "Texture.hpp"
#include <stdio.h>
#include <stdlib.h>
#include <GL/glew.h>
#include <turbojpeg.h>
#include <png.h>

using namespace cg_engine;

Texture::Texture(char *filename, ImgType imgtype)
{
	glGenTextures(1, &texid);

	switch(imgtype){
		case JPEGType:
			loadJPEG(filename, texid);
			break;
		case PNGType:
			loadPNG(filename, texid);
			break;
		case TGAType:
			loadTGA(filename, texid);
			break;
	};
}


Texture::~Texture()
{
}

GLuint Texture::getTexID()
{
	return texid;
}

void Texture::loadTGA(char *filename, GLuint texid)
{
	char id;
	char colormaptype;
	char imgtype;
	char colormap[5];
	short int w;
	short int h;
	char bpp;
	char *image;
	GLenum type;

	FILE *f = fopen(filename, "rb");
	fread(&id, 1, 1, f);
	fread(&colormaptype, 1, 1, f);
	fread(&imgtype, 1, 1, f);
	fread(&colormap, 5, 1, f);
	fread(&colormap, 4, 1, f);
	fread(&w, 2, 1, f);
	fread(&h, 2, 1, f);
	width = w;
	height = h;
	fread(&bpp, 1, 1, f);
	bpp = bpp / 8;

	if (bpp == 1) {
		type = GL_RED;
	}
	else if(bpp == 3){
		type = GL_BGR;
	}
	else {
		type = GL_BGRA;
	}

	fread(&id, 1, 1, f);
	image = (char*)malloc(w * h * bpp);
	fread(image, sizeof(char), w * h * bpp, f);
	fclose(f);

	glBindTexture(GL_TEXTURE_2D, texid);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w,
		h, 0, type, GL_UNSIGNED_BYTE,
		image);

	free(image);
}

void Texture::loadJPEG(char *filename, GLuint texid)
{
	tjhandle jpegh;
	FILE *jpgfile;
	unsigned long filesize;
	unsigned char *rawjpg;
	int subsample;
	unsigned char *image;

	jpgfile = fopen(filename, "rb");
	fseek(jpgfile, 0L, SEEK_END);
	filesize = ftell(jpgfile);
	fseek(jpgfile, 0L, SEEK_SET);
	rawjpg = (unsigned char*)malloc(filesize);
	fread(rawjpg, filesize, 1, jpgfile);
	fclose(jpgfile);

	jpegh = tjInitDecompress();
	tjDecompressHeader2(jpegh, rawjpg, filesize, &width, &height, &subsample);
	image = (unsigned char*)malloc(width * height * 3); // 3 bytes per pixel
	tjDecompress2(jpegh, rawjpg, filesize, image, width, 0, height, TJPF_RGB, TJFLAG_BOTTOMUP);
	tjDestroy(jpegh);
	free(rawjpg);

	glBindTexture(GL_TEXTURE_2D, texid);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, 0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGB, GL_UNSIGNED_BYTE, image);

	free(image);
}

void Texture::loadPNG(char *filename, GLuint texid)
{
	FILE        *texturefile;
	png_structp  texture_ptr;
	png_infop    texture_info;
	png_infop    png_end;
	png_bytep   *row;
	GLubyte     *image;
	GLenum       png_format;
	int          channels;
	int i;

	/* initialization steps */
	texturefile  = fopen(filename, "rb");
	if(texturefile == NULL){
		return;
	}
	texture_ptr  = png_create_read_struct(PNG_LIBPNG_VER_STRING,
                                       NULL,
                                       NULL,
                                       NULL);
	if(texture_ptr == NULL){
		return;
	}
	texture_info = png_create_info_struct(texture_ptr);
	png_end      = png_create_info_struct(texture_ptr);
	png_init_io(texture_ptr, texturefile);
	png_read_info(texture_ptr, texture_info);
	width        = png_get_image_width(texture_ptr, texture_info);
	height       = png_get_image_height(texture_ptr, texture_info);
	channels     = png_get_channels(texture_ptr, texture_info);
	row          = (png_bytep*)malloc(sizeof(png_bytep) * height);
	image        = (GLubyte*)malloc(width * height * channels);

	switch(channels){
		case 4:  png_format = GL_RGBA;            break;
		case 3:  png_format = GL_RGB;             break;
		case 2:  png_format = GL_LUMINANCE_ALPHA; break;
		default: png_format = GL_R;
	};


	/* We set the png row pointer to the stored texture data */
	for(i = 0; i < height; i++){
		row[height - 1 - i] = image + i * (width * channels);
	}

	/* Read the file */
	png_read_image(texture_ptr, row);  

	/* OpenGL related stuff. The png data go to the OpenGL state machine */
	glBindTexture(GL_TEXTURE_2D, texid);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width,
		height, 0, png_format, GL_UNSIGNED_BYTE,
		image);

	/* The data is in the state machine we do not need this datas any more */
	png_destroy_read_struct(&texture_ptr, &texture_info, &png_end);
	free(image);
	free(row);
	fclose(texturefile);
}

int Texture::getWidth()
{
	return width;
}

int Texture::getHeight()
{
	return height;
}
