"use strict";

class TextManager {
    _scene;
    _lines = [];

    constructor(scene) {
        this._scene = scene;
    }

    showLine(x, y, text) {
        var textMesh = this.createLabel(text, 1, "ShadowsIntoLight", 64);
        textMesh.position.set(x, y, 0.001);

        var line = new TextLine();
        line.mesh = textMesh;
        line.age = 0;
        
        this._lines.push(line);
        this._scene.add(textMesh);
    }

    render() {
        this._lines.forEach((textLine, index) => {
            textLine.age++;
            if (textLine.age < 60)
                return;
            var mesh = textLine.mesh;
            mesh.position.add(new THREE.Vector3(0, 0.006, 0));
            mesh.material.opacity -= 0.01;
            if (mesh.position.y < -2) {
                this._scene.remove(mesh);
                this._lines.splice(index, 1);
            }
        });
    }

    createLabel(text, height, fontName, fontSize, color) {
        color = color || "white";
        var canvas = document.createElement("canvas");
        var context = canvas.getContext("2d");

        context.font = fontSize + "pt " + fontName;

        var metrics = this.measureText(text, fontName, fontSize);
        canvas.style.width = metrics.width + "px";
        canvas.style.height = metrics.height + "px";

        var dpi = window.devicePixelRatio;
        //create a style object that returns width and height
        let style = {
            height() {
                return +getComputedStyle(canvas).getPropertyValue('height').slice(0, -2);
            },
            width() {
                return +getComputedStyle(canvas).getPropertyValue('width').slice(0, -2);
            }
        }
        canvas.width = metrics.width * dpi;
        canvas.height = metrics.height * dpi;

        context.clearRect(0, 0, canvas.width, canvas.height);

        context.font = fontSize + "pt " + fontName;
        context.textAlign = "left";
        context.textBaseline = "top";
        context.fillStyle = color;
        context.fillText(text, 0, 0);

        var texture = new THREE.Texture(canvas);
        texture.needsUpdate = true;

        var material = new THREE.MeshBasicMaterial({
            map: texture,
            transparent: true
        });

        var width = canvas.width / canvas.height * height;

        var mesh = new THREE.Mesh(new THREE.PlaneGeometry(width, height, 1, 1), material);
        mesh.overdraw = true;

        return mesh;
    }

    measureText(text, font, size) {
        var div = document.createElement("div");
        div.innerHTML = text;
        div.style.position = 'absolute';
        div.style.top = '-9999px';
        div.style.left = '-9999px';
        div.style.fontFamily = font;
        div.style.fontWeight = "normal";
        div.style.fontSize = size + 'pt'; // or 'px'
        document.body.appendChild(div);
        var size = { width: div.offsetWidth, height: div.offsetHeight };
        document.body.removeChild(div);
        return size;
    }
}

class TextLine {
    mesh;
    age;
}