#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <curl/curl.h>
#include <unistd.h>
#include <ftdi.h>
#include <time.h>

const int NOF_ARTICLES = 24;
const int NOF_LINES = 24;
const int CHARS_PER_LINE = 64;
const int TYPE_HEADLINES = 6;
const int TYPE_TEXT = 1;
const int TYPE_KOALA = 2;
const int TYPE_HIRES = 3;
const int TYPE_HIRES_SELECT_PAPER = 4;
const int TYPE_INTERLACE_IMAGE = 5;

int c64_state = 0;
const int C64_SELECT_PAPER = 0;
const int C64_INTROBILD = 1;
const int C64_HEADLINES = 2;
const int C64_ARTICLE = 3;
const int C64_CREDITS = 4;
int which_paper = -1;

// All data to C64 is sent as
// size_lsb
// size_msb
// TYPE_xxx
// data...

// size is transfer size including the TYPE_xxx byte


// we're assuming it's unix/linux/macos if _WIN32 _WIN64 or MSC_VER (mingw)
// have not been defined
#if defined (__WIN32__)
  #include <conio.h>
#else
  #include <termios.h>
  #include <sys/ioctl.h>
//  #include <stdargs.h>
#endif

int product_id = 0x8738;
int deviceNumber = 0;
int mode128 = 0;

int init(struct ftdi_context *p_ftdic) {
  if (ftdi_init(p_ftdic)) {
    fprintf(stderr, "can't initialize ftdi_context");
    return 1;
  }
  return 0;
}

int fail(struct ftdi_context *p_ftdic) {
  int rc = 0;
  char *err;
  err = ftdi_get_error_string(p_ftdic);
  if (strncmp(err, "all fine", 8)) {
    fprintf(stderr, "libftdi: %s\n", err);
    rc = 1;
  }
  ftdi_deinit(p_ftdic);
  return rc;
}

int listDevices(struct ftdi_context *p_ftdic, struct ftdi_device_list *p_deviceList, int product_id) {
  struct ftdi_device_list *p_currentDevice;
  char p_manufacturer[128], p_description[128], p_serial[128];
  int i = 0;
  if (ftdi_usb_find_all(p_ftdic, &p_deviceList, 0x0403, product_id) < 0) {
    fprintf(stderr, "There was a problem listing the devices.\n");
    fail(p_ftdic);
    return 1;
  }
  if (p_deviceList == NULL) {
    printf("No devices found.\n Check connection and access permissions!\n");
    return 1;
  }
  p_currentDevice = p_deviceList;
  if (p_currentDevice != NULL) {
    if (ftdi_usb_get_strings(p_ftdic, p_currentDevice->dev,
      p_manufacturer, 128, p_description, 128,
    p_serial, 128) < 0) {
      ftdi_list_free(&p_deviceList);
      ftdi_deinit(p_ftdic);
      return 1;
    }
    printf("Device: %d, Manufacturer: %s, Description: %s, Serial: %s\n",
      i, p_manufacturer, p_description, p_serial);
    p_currentDevice = p_currentDevice->next;
    ++i;
  }
  ftdi_list_free(&p_deviceList);
  return 0;
}

int usednum;
unsigned char usedtr[1024];
unsigned char usedse[1024];
unsigned char tempb[65535];
struct ftdi_context ftdic;
struct ftdi_device_list *p_deviceList = NULL;

static void s_strncpy(char *to, const char *from, int l) {
  memccpy(to, from, '\0', l);
  to[l-1] = '\0';
}

static char * eat(char *s, int (*p)(int), int r) {
  while(s != '\0' && p(*s) == r)
    s++;
  return s;
}

static char* skip(char *s, char c) {
  while(*s != c && *s != '\0')
    s++;
  if(*s != '\0')
    *s++ = '\0';
  return s;
}

unsigned char petToAscTable[256] = {
0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x14,0x09,0x0d,0x11,0x93,0x0a,0x0e,0x0f,
0x10,0x0b,0x12,0x13,0x08,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
0x40,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x5b,0x5c,0x5d,0x5e,0x5f,
0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf,
0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdd,0xde,0xdf,
0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
0x90,0x91,0x92,0x0c,0x94,0x95,0x96,0x97,0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,
0x60,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x7b,0x7c,0x7d,0x7e,0x7f,
0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf
};

unsigned char ascToPetTable[256] = {
0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x14,0x20,0x0d,0x11,0x93,0x0a,0x0e,0x0f,
0x10,0x0b,0x12,0x13,0x08,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
0x40,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf,
0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0x5b,0x5c,0x5d,0x5e,0x5f,
0xc0,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0xdb,0xdc,0xdd,0xde,0xdf,
0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
0x90,0x91,0x92,0x0c,0x94,0x95,0x96,0x97,0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,
0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,
0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,
0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
};

unsigned char ascToScrTable[256] = {
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
0x00,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x1b,0x1c,0x1d,0x1e,0x64,
0x27,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0xdd,0x1d,0x77,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20
};

int ftdiwrite(unsigned char * buf, int len) {
  printf("USB transmit to C64:0x%02x 0x%02x 0x%02x...\n", buf[0], buf[1], buf[2]);
  int i;
  int written;
  for (i = 0; i < len; i++) {
    written = 0;
    while (written != 1) {
      written = ftdi_write_data(&ftdic, buf + i, 1);
    }
  }
  return len;
}

int ftdiread(unsigned char * buf, int len) {
  int i;
  int written;
  for (i = 0; i < len; i++) {
    written = 0;
    while (written != 1) {
      written = ftdi_read_data(&ftdic, buf + i, 1);
    }
  }
  return len;
}

unsigned char * bufread;
unsigned char * bufcmd;
int written;
int nof_read;

void print_net_status(int code, char *message) {
}

int debug=0;
unsigned char * bufstart;
FILE * fp;

void startprg() {
  // SEND PRG TO EF3 MENU
  size_t size;
  uint8_t buffer[65536];

  int waiting;
  bufstart = (unsigned char *) malloc(30);
  bufread = (unsigned char *) malloc(30);
  printf("\nStarting nyheter.prg on c64\n");

//  do
//  {
//    waiting = 0;
//    printf("Handshake: EFSTART:PRG\n");
//    sprintf((char*)bufstart,"EFSTART:PRG%c",0);
//    written = ftdiwrite(bufstart, 12);
//    printf("1\n");
//    nof_read = ftdiread(bufread, 5);
//    printf("2\n");
//
//    if (nof_read != 5)
//    {
//      printf("Error 3 on the C64 side ... exiting...\n");
//      goto exitpoint;
//    }
//    printf("Response: [%c%c%c%c]\n",bufread[0],bufread[1],bufread[2],bufread[3]);
//    printf("Response: [%d]\n",bufread[4]);
//
//    if (bufread[0] == 'W') {
//      waiting = 1;
//      printf("Waiting...\n");
//    } else if (bufread[0] != 'L') {
//      printf("Error on the C64 side... exiting...\n");
//      goto exitpoint;
//    }
//    printf("3\n");
//  }
//  while (waiting == 1);

  printf("\n - Sending PRG file. Bytes sent: ");
  fp = fopen("nyheter.prg", "rb");
//  ftdiread(bufread, 2);   // get how many bytes to send
//
//  written = bufread[0] + bufread[1]*256;
  written = 65535;
  do {
    size = fread(buffer, 1, written, fp);
    bufstart[0] = (unsigned char) (size & 0xff);
    bufstart[1] = (unsigned char) (size >> 8);
    if (ftdiwrite(bufstart, 2) != 2)
    {
      printf("\nERROR 1!\n");
      break;
    }        
    if (ftdi_write_data(&ftdic, buffer, size) != size)
    {
      printf("\nERROR 2!\n");
      break;
    }        
    printf("%6d%c%c%c%c%c%c",(int)size,8,8,8,8,8,8);
  }
  while (size);

  printf("\n\nDONE!\n\n");
  exitpoint:
    fclose(fp);
}

char get_command_from_c64() {
  unsigned char * bufread;
  bufread = (unsigned char *) malloc(30);
  printf("\nWaiting for command from c64\n");
  nof_read = ftdiread(bufread, 1);
  printf("Got a '%c' = 0x%02x\n", bufread[0], bufread[0]);
  return bufread[0];
}

#if !defined(__WIN32__)
  void enable_raw_mode() {
    struct termios term;
    tcgetattr(0, &term);
    term.c_lflag &= ~(ICANON | ECHO); // Disable echo as well
    tcsetattr(0, TCSANOW, &term);
  }

  void disable_raw_mode() {
    struct termios term;
    tcgetattr(0, &term);
    term.c_lflag |= ICANON | ECHO;
    tcsetattr(0, TCSANOW, &term);
  }

  int kbhit() {
    int byteswaiting;
    ioctl(0, FIONREAD, &byteswaiting);
    return byteswaiting > 0;
  }
#endif

int current_page_no = 0;
int nof_pages = 1;
char* current_headline_url = NULL;
char* current_article_url = NULL;

struct MemoryStruct {
  char *memory;
  size_t size;
};

struct MemoryStruct headlines;
struct MemoryStruct article;
struct MemoryStruct koala;

static size_t WriteMemoryCallback(void *contents, size_t size, size_t nmemb, void *userp) {
  size_t realsize = size * nmemb;
  struct MemoryStruct *mem = (struct MemoryStruct *)userp;
  char *ptr = realloc(mem->memory, mem->size + realsize + 1);
  if(ptr == NULL) {
    /* out of memory! */
    printf("not enough memory (realloc returned NULL)\n");
    return 0;
  }
  mem->memory = ptr;
  memcpy(&(mem->memory[mem->size]), contents, realsize);
  mem->size += realsize;
  mem->memory[mem->size] = 0;
  return realsize;
}


void send_to_usb(unsigned char * buf, int size) {
  buf[0] = (unsigned char) ((size-2) & 0xff);
  buf[1] = (unsigned char) ((size-2) >> 8);
  printf("USB transmit to C64:0x%02x 0x%02x 0x%02x...\n", buf[0], buf[1], buf[2]);
  int i = 0;
  int written = 0;
  int bytes_left_to_send = size;
  while (bytes_left_to_send > 0) {
    written = ftdi_write_data(&ftdic, buf + i, bytes_left_to_send);
    if (written < 0) {
      printf("\nERROR %d!\n", written);
    }
    bytes_left_to_send -= written;
    i += written;
  }
}

void send_to_usb_with_separate_header(unsigned char * buf, int size, unsigned char data_type) {
  unsigned char header[10];
  printf ("USB transmit to C64, separate header, size=%d\n", size);
  header[0] = (unsigned char) ((size+3) & 0xff);
  header[1] = (unsigned char) ((size+3) >> 8);
  header[2] = data_type;
  if (ftdiwrite(header, 3) != 3)
  {
    printf("\nERROR sep!\n");
  }
  int i = 0;
  int written = 0;
  int bytes_left_to_send = size;
  while (bytes_left_to_send > 0) {
    written = ftdi_write_data(&ftdic, buf + i, bytes_left_to_send);
    if (written < 0) {
      printf("\nERROR %d!\n", written);
    }
    bytes_left_to_send -= written;
    i += written;
  }
  printf ("...done sending!\n");
}


void send_intro_koala(char * filename) {
  const int MAXBUFLEN = 65536;
  size_t nof_read_bytes = 0;
  unsigned char read_data[MAXBUFLEN + 1];
  FILE *fp = fopen(filename, "rb");
  if (fp != NULL) {
    nof_read_bytes = fread(read_data, sizeof(char), MAXBUFLEN, fp);
    if ( ferror( fp ) != 0 ) {
      fputs("Error reading file", stderr);
    }
    fclose(fp);
  }
  int size = nof_read_bytes;
  printf ("sending koala image, size=%d\n", size);
  // Skip sending the first 2 bytes, they are 0x6000 "loading address"
  send_to_usb_with_separate_header(&read_data[2], size-2+1, TYPE_KOALA);
}


void send_koala() {
  int size = koala.size - 2 + 1;
  printf ("sending koala image, size=%d\n", size);
  // Skip sending the first 2 bytes, they are 0x6000 "loading address"
  send_to_usb_with_separate_header((unsigned char*)&koala.memory[2], size, TYPE_KOALA);
}


void grab_koala(char* url) {
  printf ("grabbing image %s\n", url);
  CURL* curl_handle;
  CURLcode res;

  koala.memory = malloc(1);  /* will be grown as needed by the realloc above */
  koala.size = 0;    /* no data at this point */

  /* init the curl session */
  curl_handle = curl_easy_init();
  if(curl_handle) {
    /* specify URL to get */
    curl_easy_setopt(curl_handle, CURLOPT_URL, url);
    /* send all data to this function  */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    /* we pass our 'chunk' struct to the callback function */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)&koala);
    /* some servers don't like requests that are made without a user-agent
       field, so we provide one */
    curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "c64_newsreader/1.0");
    /* get it! */
    res = curl_easy_perform(curl_handle);
    /* check for errors */
    if(res != CURLE_OK) {
      fprintf(stderr, "curl_easy_perform() failed: %s\n",
              curl_easy_strerror(res));
    } else {
      // Now, our koala.memory points to a memory block that is koala.size
      // bytes big and contains the remote file.
      printf("%lu bytes retrieved\n", (unsigned long)headlines.size);
    }
    /* cleanup curl stuff */
    curl_easy_cleanup(curl_handle);
  }
  /* we're done with libcurl, so clean it up */
  curl_global_cleanup();
}


void send_headlines() {
  uint8_t buffer[65536];
  size_t size = 0;

  //printf("GOT:%s\n", headlines.memory);
//1906122336
//3JVy59;1906122252;Hj�rnsl�pp av stj�rnan - r�ddades av VAR-straff
//jdw4Kz;1906122251;Blixten har slagit ner i flerfamiljshus i Limhamn
  int offset = 0;
  int output_offset = 2;

  // Tell C64 what type of data it is getting:
  buffer[output_offset++] = TYPE_HEADLINES;

  while ((offset < headlines.size) && (headlines.memory[offset] != 0x0a)) {
    offset++;
  }
  offset++;
  //printf("offset=%d\n", offset);
  // We're now at the second row of the headlines, there should be exactly 24 lines of headlines here.
  int article_no = 0;
  while ((offset < headlines.size) && (article_no < NOF_ARTICLES)) {
    int char_no = 0;
    while ((offset < headlines.size) && (headlines.memory[offset] != ';')) {
      offset++;
    }
    offset++;
    // We're now at the date.
    while ((offset < headlines.size) && (headlines.memory[offset] != ';')) {
      offset++;
    }
    offset++;
    // We're now at the headline text.
    while ((offset < headlines.size) && (headlines.memory[offset] != 0x0a) && (char_no < CHARS_PER_LINE)) {
      unsigned char this_c = headlines.memory[offset];
      //buffer[output_offset++] = ascToScrTable[headlines.memory[offset]];
      if (this_c == 0xe5) this_c = 0x00; // å
      if (this_c == 0xe4) this_c = 0x01; // ä
      if (this_c == 0xf6) this_c = 0x02; // ö
      if (this_c == 0xc5) this_c = 0x03; // Å
      if (this_c == 0xc4) this_c = 0x04; // Ä
      if (this_c == 0xd6) this_c = 0x05; // Ö

      if (this_c == 0xe6) this_c = 0x06; // æ
      if (this_c == 0xc6) this_c = 0x07; // Æ
      if (this_c == 0xf8) this_c = 0x08; // ø
      if (this_c == 0xd8) this_c = 0x09; // Ø
      if (this_c == 0xe9) this_c = 0x0a; // é

      //printf("%c=%02x,", this_c, this_c);
      buffer[output_offset++] = this_c;
      offset++;
      char_no++;
    }
    // Skip any "too many" chars on this row:
    while ((offset < headlines.size) && (headlines.memory[offset] != 0x0a)) {
      offset++;
    }
    offset++;
    // Fill with spaces until 64 chars:
    while (char_no < CHARS_PER_LINE) {
      //printf("_");
      buffer[output_offset++] = ascToScrTable[' '];
      char_no++;
    }
    //printf("\n");
    article_no++;
  }
  buffer[output_offset++] = 0xff;

  // Send it all over USB:
  size = output_offset;
  send_to_usb(buffer, size);
  c64_state = C64_HEADLINES;
  current_page_no = 0;
}

void send_page() {
  printf ("sending page no #%d\n", current_page_no);
  //printf("GOT:%s\n", article.memory);
  // Skip until the current page
  // output 24 rows of text
  // or, if row starts with IMG; - download the .kla file, and send that one.

  uint8_t buffer[65536];
  size_t size = 0;

  //printf("GOT:%s\n", article.memory);
  int offset = 0;

  // Find the offset of this page:
  int pages_left_to_skip = current_page_no;
  while (pages_left_to_skip > 0) {
    printf("Line start:%c%c%c", article.memory[offset], article.memory[offset+1], article.memory[offset+2]);
    if ((article.memory[offset] == 'I') &&
        (article.memory[offset+1] == 'M') &&
        (article.memory[offset+2] == 'G') &&
        (article.memory[offset+3] == ';')) {
      // This is an image. It fills the whole screen.
      while ((offset < article.size) && (article.memory[offset] != 0x0a)) {
        offset++;
      }
      offset++;
    } else {
      int lines_to_skip = NOF_LINES;
      while ((offset < article.size) && (lines_to_skip > 0)) {
        int char_no = 0;
        // We're now at the text for this line.
        while ((offset < article.size) && (article.memory[offset] != 0x0a)) {
          offset++;
        }
        offset++;
        if ((article.memory[offset] == 'I') &&
            (article.memory[offset+1] == 'M') &&
            (article.memory[offset+2] == 'G') &&
            (article.memory[offset+3] == ';')) {
          //There is an image on this line. Let's end the page here.
          lines_to_skip = 0;
        }
        lines_to_skip--;
      }
    }
    pages_left_to_skip--;
  }

  if (offset >= article.size) {
    printf ("last page already sent, sending headlines instead...\n");
    send_headlines();
    c64_state = C64_HEADLINES;
    return;
  }


  if ((article.memory[offset] == 'I') &&
      (article.memory[offset+1] == 'M') &&
      (article.memory[offset+2] == 'G') &&
      (article.memory[offset+3] == ';')) {
    // This is an image. It fills the whole screen.
    // Let's send it.
    char koala_url[1024];
    int i = 0;
    int url_offset = offset + 4;
    while (((article.memory[url_offset] == 0x0a) || ((unsigned char)article.memory[url_offset] == (unsigned char)0xff)) == 0) {
      koala_url[i++] = article.memory[url_offset];
      url_offset++;
    }
    koala_url[i++] = (char)0x00;
    grab_koala(koala_url);
    send_koala();
    return;
  }


  c64_state = C64_ARTICLE;
  int output_offset = 2;

  // Tell C64 what type of data it is getting:
  buffer[output_offset++] = TYPE_TEXT;

  //printf("offset=%d\n", offset);
  // We're now at the first row of the article page, we should send exactly 24 lines of text here.
  int line_no = 0;
  int end_page = 0;
  while ((offset < article.size) && (line_no < NOF_LINES) && (end_page == 0)) {
    int char_no = 0;

    if ((article.memory[offset] == 'I') &&
        (article.memory[offset+1] == 'M') &&
        (article.memory[offset+2] == 'G') &&
        (article.memory[offset+3] == ';')) {
      // This is an image. It will be seen on the next page. End the page here.
      end_page = 1;
    } else {
      // We're now at the text for this line.
      while ((offset < article.size) && (((article.memory[offset] == 0x0a) || ((unsigned char)article.memory[offset] == 0xff)) == 0) && (char_no < CHARS_PER_LINE)) {
        unsigned char this_c = article.memory[offset];
        if (this_c == 0xe5) this_c = 0x00; // å
        if (this_c == 0xe4) this_c = 0x01; // ä
        if (this_c == 0xf6) this_c = 0x02; // ö
        if (this_c == 0xc5) this_c = 0x03; // Å
        if (this_c == 0xc4) this_c = 0x04; // Ä
        if (this_c == 0xd6) this_c = 0x05; // Ö
        if (this_c == 0xe6) this_c = 0x06; // æ
        if (this_c == 0xc6) this_c = 0x07; // Æ
        if (this_c == 0xf8) this_c = 0x08; // ø
        if (this_c == 0xd8) this_c = 0x09; // Ø
        if (this_c == 0xe9) this_c = 0x0a; // é
        //printf("%c=%02x,", this_c, this_c);
        buffer[output_offset++] = this_c;
        offset++;
        char_no++;
      }
      // Skip any "too many" chars on this row:
      while ((offset < article.size) && (article.memory[offset] != 0x0a)) {
        offset++;
      }
      offset++;
      // Fill with spaces until 64 chars:
      while (char_no < CHARS_PER_LINE) {
        //printf("_");
        buffer[output_offset++] = ascToScrTable[' '];
        char_no++;
      }
      //printf("\n");
      line_no++;
    }
  }

  // Make sure that 24 lines are sent. Always.
  while (line_no < NOF_LINES) {
    int char_no = 0;
    while (char_no < CHARS_PER_LINE) {
      buffer[output_offset++] = ascToScrTable[' '];
      char_no++;
    }
    line_no++;
  }
  buffer[output_offset++] = 0xff;

  // Send it all over USB:
  size = output_offset;
  send_to_usb(buffer, size);
  c64_state = C64_ARTICLE;
}


void grab_headlines(char* url) {
  current_headline_url = url;
  CURL* curl_handle;
  CURLcode res;

  headlines.memory = malloc(1);  /* will be grown as needed by the realloc above */
  headlines.size = 0;    /* no data at this point */

  /* init the curl session */
  curl_handle = curl_easy_init();
  if(curl_handle) {
    /* specify URL to get */
    curl_easy_setopt(curl_handle, CURLOPT_URL, url);
    /* send all data to this function  */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    /* we pass our 'chunk' struct to the callback function */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)&headlines);
    /* some servers don't like requests that are made without a user-agent
       field, so we provide one */
    curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "c64_newsreader/1.0");
    /* get it! */
    res = curl_easy_perform(curl_handle);
    /* check for errors */
    if(res != CURLE_OK) {
      fprintf(stderr, "curl_easy_perform() failed: %s\n",
              curl_easy_strerror(res));
    } else {
      // Now, our headlines.memory points to a memory block that is headlines.size
      // bytes big and contains the remote file.
      printf("%lu bytes retrieved\n", (unsigned long)headlines.size);
    }
    /* cleanup curl stuff */
    curl_easy_cleanup(curl_handle);
  }
  /* we're done with libcurl, so clean it up */
  curl_global_cleanup();
}

void grab_article(char articles_to_skip) {
  printf ("grabbing article #%d\n", articles_to_skip);
  char url[256];

  article.memory = malloc(1);  /* will be grown as needed by the realloc above */
  article.size = 0;    /* no data at this point */

  // Need to find the article_id:
  int offset = 0;
  while ((offset < headlines.size) && (headlines.memory[offset] != 0x0a)) {
    offset++;
  }
  offset++;
  //printf("offset=%d\n", offset);
  // We're now at the second row of the headlines, there should be approximately 24 lines of headlines here.
  int article_no = 0;
  while ((offset < headlines.size) && (article_no < articles_to_skip)) {
    while ((offset < headlines.size) && (headlines.memory[offset] != 0x0a)) {
      offset++;
    }
    offset++;
    article_no++;
  }

  sprintf(url, "%s&a=%c%c%c%c%c%c", current_headline_url, headlines.memory[offset], headlines.memory[offset+1], headlines.memory[offset+2], headlines.memory[offset+3], headlines.memory[offset+4], headlines.memory[offset+5]);
  printf ("url=%s\n", url);

  CURL* curl_handle;
  CURLcode res;
  /* init the curl session */
  curl_handle = curl_easy_init();
  if(curl_handle) {
    /* specify URL to get */
    curl_easy_setopt(curl_handle, CURLOPT_URL, url);
    /* send all data to this function  */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    /* we pass our 'chunk' struct to the callback function */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)&article);
    /* some servers don't like requests that are made without a user-agent
       field, so we provide one */
    curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "c64_newsreader/1.0");
    /* get it! */
    res = curl_easy_perform(curl_handle);
    /* check for errors */
    if(res != CURLE_OK) {
      fprintf(stderr, "curl_easy_perform() failed: %s\n",
              curl_easy_strerror(res));
    } else {
      // Now, our headlines.memory points to a memory block that is article.size
      // bytes big and contains the remote file.
      printf("%lu bytes retrieved\n", (unsigned long)article.size);
      current_page_no = 0;
    }
    /* cleanup curl stuff */
    curl_easy_cleanup(curl_handle);
  }
  /* we're done with libcurl, so clean it up */
  curl_global_cleanup();
}


void send_next_page() {
  current_page_no++;
  send_page();
}

void send_papers() {
  // ToDo: send hires image with paper selector image
  const int MAXBUFLEN = 65536;
  size_t nof_read_bytes = 0;
  unsigned char read_data[MAXBUFLEN + 1];
  FILE *fp = fopen("nyheter_start_v03_bw.map", "rb");
  if (fp != NULL) {
    nof_read_bytes = fread(read_data, sizeof(char), MAXBUFLEN, fp);
    if ( ferror( fp ) != 0 ) {
      fputs("Error reading file", stderr);
    }
    fclose(fp);
  }
  int size = nof_read_bytes;
  printf ("sending hires image, size=%d\n", size);
  send_to_usb_with_separate_header(read_data, size, TYPE_HIRES_SELECT_PAPER);
}

void send_credits() {
  // ToDo: send hires image with credits
  const int MAXBUFLEN = 65536;
  size_t nof_read_bytes = 0;
  unsigned char read_data[MAXBUFLEN + 1];
  FILE *fp = fopen("nyheter_credits_v02_bw.map", "rb");
  if (fp != NULL) {
    nof_read_bytes = fread(read_data, sizeof(char), MAXBUFLEN, fp);
    if ( ferror( fp ) != 0 ) {
      fputs("Error reading file", stderr);
    }
    fclose(fp);
  }
  int size = nof_read_bytes;
  printf ("sending hires image, size=%d\n", size);
  // Skip sending the first 2 bytes, they are 0x6000 "loading address"
  send_to_usb_with_separate_header(read_data, size, TYPE_HIRES);
}

void send_introbild(char * filename) {
  // ToDo: send interlaced image with introbild
  const int MAXBUFLEN = 65536;
  size_t nof_read_bytes = 0;
  unsigned char read_data[MAXBUFLEN + 1];
  FILE *fp = fopen(filename, "rb");
  if (fp != NULL) {
    nof_read_bytes = fread(read_data, sizeof(char), MAXBUFLEN, fp);
    if ( ferror( fp ) != 0 ) {
      fputs("Error reading file", stderr);
    }
    fclose(fp);
  }
  int size = nof_read_bytes;
  printf ("sending introbild image, size=%d\n", size);
  // Skip sending the first 2 bytes, they are 0x6000 "loading address"
  send_to_usb_with_separate_header(read_data, size, TYPE_INTERLACE_IMAGE);
  c64_state = C64_INTROBILD;
}

int main(void) {
  printf ("\nVälkommen till Nyheter, för Commodore 64: Aftonbladet, SvD, VG och Aftenposten. [ctrl + d] för att avsluta.\n");
  if (init(&ftdic)) return 1;
  if (listDevices(&ftdic, p_deviceList, product_id)) return 1;

  if (ftdi_usb_open_desc_index(&ftdic, 0x0403, product_id, NULL, NULL, deviceNumber))
    return 1;
  while (ftdi_usb_reset(&ftdic)) {
    fprintf(stderr, "%s", ftdi_get_error_string(&ftdic));
    usleep(1000000);
  }
  printf ("Init...\n");
  startprg();
  printf ("Allt är klart!\n");

  curl_global_init(CURL_GLOBAL_DEFAULT);

  headlines.memory = malloc(1);  /* will be grown as needed by the realloc above */
  headlines.size = 0;    /* no data at this point */

  char *line = NULL;  /* forces getline to allocate with malloc */
  size_t len = 0;     /* ignored when line = NULL */
  ssize_t read;

//  while ((read = getline(&line, &len, stdin)) != -1) {
//    if (read > 0) {
//      printf ("read %zd chars from stdin, allocated %zd bytes for line : %s\n", read, len, line);
//
//      if (line[0] == 'A') {
//        grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=ab");
//      } else if (line[0] == 'B') {
//        grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=svd");
//      } else if (line[0] == 'C') {
//        grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=vg");
//      } else if (line[0] == 'D') {
//        grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=ap");
//      } else if ((line[0] >= 'a') && (line[0] <= 'x')) {
//        grab_article(line[0] - 'a');
//      } else if (line[0] == 'E') {
//        prev_page();
//      } else if (line[0] == 'F') {
//        next_page();
//      } else if (line[0] == 'G') {
//        grab_koala("https://gubbdata.scenesat.com/img/ab/70xdKB230a85f8eb12eae5416fd3fb0b879aa4.kla");
//      } else if (line[0] == 'H') {
//        grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6wba492e4302f27ef9efb6950622d01361.kla");
//      } else if (line[0] == 'I') {
//        grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6we21e18fcc30b6e867f9861f1eb14943b.kla");
//      } else if (line[0] == 'J') {
//        grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6w268914cf6df800dd547aad62a4193c1f.kla");
//      }
//    }
//    if (line != NULL) {
//      free (line);  /* free memory allocated by getline */
//      line = NULL;
//    }
//    free(headlines.memory);
//  }

// There are two zero bytes from C64 after bootloader:
  char cmd = get_command_from_c64();
  cmd = get_command_from_c64();

  // Show all intro images:
  cmd = get_command_from_c64();
  send_intro_koala("intro_01_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_02_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_03_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_04_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_05_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_06_v02.kla");
  cmd = get_command_from_c64();
  send_intro_koala("intro_07_v02.kla");
  cmd = get_command_from_c64();
  send_papers();


  while (1==1) {
    char cmd = get_command_from_c64();
//      grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=ab");
////      grab_koala("https://gubbdata.scenesat.com/img/ab/70xdKB230a85f8eb12eae5416fd3fb0b879aa4.kla");
//    cmd = get_command_from_c64();
//      grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6wba492e4302f27ef9efb6950622d01361.kla");
////      grab_article(1);
//    cmd = get_command_from_c64();
////      grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6we21e18fcc30b6e867f9861f1eb14943b.kla");
//      grab_article(1);
//    cmd = get_command_from_c64();
////      grab_koala("https://gubbdata.scenesat.com/img/ab/70xd6we21e18fcc30b6e867f9861f1eb14943b.kla");
//      grab_article(2);

    if (cmd == '0') {
      // Run/Stop pressed on C64
      send_papers();
      c64_state = C64_SELECT_PAPER;
    } else if (cmd == 'A') {
      which_paper = 0;
      send_introbild("introbild_ab.bin");
      grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=ab");
    } else if (cmd == 'B') {
      which_paper = 1;
      send_introbild("introbild_svd.bin");
      grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=svd");
    } else if (cmd == 'C') {
      which_paper = 2;
      send_introbild("introbild_vg.bin");
      grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=vg");
    } else if (cmd == 'D') {
      which_paper = 3;
      send_introbild("introbild_af.bin");
      grab_headlines("https://gubbdata.scenesat.com/?c64version=1&nr=ap");
    } else {
      if (c64_state == C64_INTROBILD) {
        send_headlines();
      } else if (c64_state == C64_HEADLINES) {
        if ((cmd >= 'a') && (cmd <= 'x')) {
          // Grab article
          grab_article(cmd - 'a');
          c64_state = C64_ARTICLE;
          send_page();
        } else if (cmd == (char)0x80) {
          // Clicked at the top of the page (uppermost 6 pixels)
          send_papers();
          c64_state = C64_SELECT_PAPER;
        } else {
          // Clicked below articles
          send_credits();
          c64_state = C64_CREDITS;
        }
      } else if (c64_state == C64_ARTICLE) {
        if ((cmd >= 'a') && (cmd <= 'x')) {
          // goto next page
          send_next_page();
        } else if (cmd == (char)0x80) {
          // Clicked at the top of the page (uppermost 6 pixels)
          send_headlines();
        } else {
          // Clicked below article
          send_next_page();
        }
      }
    }

    printf("c64_state=");
    if (c64_state == C64_SELECT_PAPER) { printf("SELECT_PAPER"); }
    if (c64_state == C64_INTROBILD) { printf("INTROBILD"); }
    if (c64_state == C64_HEADLINES) { printf("HEADLINES"); }
    if (c64_state == C64_ARTICLE) { printf("ARTICLE"); }
    if (c64_state == C64_CREDITS) { printf("CREDITS"); }
    printf("\n");

  }


  return 0;
}
