//--------------------------------------------------------------------------------------
// Constant Buffer Variables
//--------------------------------------------------------------------------------------
Texture2D tex0 : register(t0);
SamplerState sampLinearClamp : register(s0);
SamplerState sampNearestClamp : register(s1);
SamplerState sampLinearWrap : register(s2);
SamplerState sampNearestWrap : register(s3);

cbuffer GlobalRenderData : register(b3)
{
    matrix g_matWorld;
    matrix g_matView;
    matrix g_matProjection;
};

cbuffer GlobalRenderData : register(b4)
{
    matrix g_matModel;
    float g_fTime;
    float g_fFloat1;    // Outer circle width
    float g_fFloat2;    // Outer circle stroke width
    float g_fFloat3;    // Outer circle form angle
    float g_fFloat4;    // Central visibility
    float g_fFloat5;    // Inner coloured circle size
    float g_fFloat6;    // Inner opacity
    float g_fSeed;
};

#define PI 3.14159265359

//--------------------------------------------------------------------------------------
struct VS_INPUT
{
    float3 vecPosition : POSITION;
    float2 vecTexCoord0 : TEXCOORD0;
};

struct PS_INPUT
{
    float4 vecPosition : SV_POSITION;
    float2 vecTexCoord0 : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Support functions
//--------------------------------------------------------------------------------------
float HueToRGB(float f1, float f2, float hue)
{
    if (hue < 0.0)
        hue += 1.0;
    else if (hue > 1.0)
        hue -= 1.0;
    float res;
    if ((6.0 * hue) < 1.0)
        res = f1 + (f2 - f1) * 6.0 * hue;
    else if ((2.0 * hue) < 1.0)
        res = f2;
    else if ((3.0 * hue) < 2.0)
        res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
    else
        res = f1;
    return res;
}

float3 HSLToRGB(float3 hsl)
{
    float3 rgb;

    if (hsl.y == 0.0)
        rgb = float3(hsl.z, hsl.z, hsl.z); // Luminance
    else
    {
        float f2;

        if (hsl.z < 0.5)
            f2 = hsl.z * (1.0 + hsl.y);
        else
            f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);

        float f1 = 2.0 * hsl.z - f2;

        rgb.r = HueToRGB(f1, f2, hsl.x + (1.0 / 3.0));
        rgb.g = HueToRGB(f1, f2, hsl.x);
        rgb.b = HueToRGB(f1, f2, hsl.x - (1.0 / 3.0));
    }

    return rgb;
}

float ShapeCheck(float2 vecOffset)
{
    float fCircleWidth = 0.06;
    float fCircleStroke = 0.0015;

    float fDistance = length(vecOffset) / g_fFloat5;
    float fEdgeDistance = abs(fDistance - fCircleWidth);

    float fStrength = 0;
    if (abs(fEdgeDistance) < fCircleStroke)
    {
        fStrength = cos(fEdgeDistance / fCircleStroke * PI / 2) * 0.5;
    }
    if (fDistance < fCircleWidth)
    {
        fStrength = max(fStrength, 0.07);
    }
    return fStrength;
}


//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
PS_INPUT VS( VS_INPUT input )
{
    PS_INPUT output;

    matrix matWorld = mul(g_matWorld, g_matModel);
    matrix matWorldView = mul(matWorld, g_matView);
    matrix matWorldViewProjection = mul(matWorldView, g_matProjection);

    float2 vecTexCoord0 = input.vecTexCoord0.xy * float2(2, -2) + float2(-1, 1);

    output.vecPosition = mul(float4(input.vecPosition, 1), matWorldViewProjection);
//    output.vecPosition = float4(vecTexCoord0 * float2(1, 1280.0f / 720.0f), 0, 1);
    output.vecTexCoord0 = input.vecTexCoord0.xy - 0.5;

//     vecTex = float2(cos(g_fRotation) * vecTex.x - sin(g_fRotation) * vecTex.y, 
//                     sin(g_fRotation) * vecTex.x + cos(g_fRotation) * vecTex.y);
//     vecTex -= float2(g_fPosX, g_fPosY);
//     output.vecTex = (vecTex * g_fScale) / float2(2, -2) + 0.5;

    return output;
}


//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS(PS_INPUT input) : SV_Target
{
    // Outer circle
    float fDistance = length(input.vecTexCoord0);
    if (fDistance > g_fFloat1)
    {
        // Outer half of outer circle
        float fDistanceFromCircle = fDistance - g_fFloat1;
        if (fDistanceFromCircle < g_fFloat2 / 2)
        {
            float fAngle = atan2(input.vecTexCoord0.x, -input.vecTexCoord0.y);
            if (abs(fAngle) < g_fFloat3)
            {
                return cos(fDistanceFromCircle / g_fFloat2 * PI);
            }
        }
    }
    else
    {
        // Inner half of outer circle needs to be added in
        float fDistanceFromCircle = g_fFloat1 - fDistance;
        float fEdgeColour = 0;
        if (fDistanceFromCircle < g_fFloat2 / 2)
        {
            float fAngle = atan2(input.vecTexCoord0.x, -input.vecTexCoord0.y);
            if (abs(fAngle) < g_fFloat3)
            {
                fEdgeColour = cos(fDistanceFromCircle / g_fFloat2 * PI);
            }
        }

        // Central shapes
        float fStrength = 0;

        float fNumOuterCircles = 6;// floor((g_fSeed * 123.45) % 20) + 3;
        float fOuterCircleStep = PI * 2 / fNumOuterCircles;
        for (float i = 0; i < fNumOuterCircles; i++)
        {
            float2 vecCircleOffset = float2(sin(i * fOuterCircleStep), cos(i * fOuterCircleStep));
            fStrength += ShapeCheck(input.vecTexCoord0 + vecCircleOffset * 0.125);
        }

        float fNumInnerCircles = 10;// floor((g_fSeed * 345.67) % 20) + 3;
        float fInnerCircleStep = PI * 2 / fNumInnerCircles;
        for (float i = 0; i < fNumInnerCircles; i++)
        {
            float2 vecCircleOffset = float2(sin(i * fInnerCircleStep), cos(i * fInnerCircleStep));
            fStrength += ShapeCheck(input.vecTexCoord0 + vecCircleOffset * 0.075);
        }

        float fAngle = atan2(input.vecTexCoord0.x, -input.vecTexCoord0.y);
        float3 vecRGB = HSLToRGB(float3(fAngle / PI / 2, 0.6, fStrength));
        return max(fEdgeColour, float4(vecRGB.rgb, g_fFloat4)) + g_fFloat6;
    }

    return 0;
}
