//--------------------------------------------------------------------------------------
// Constant Buffer Variables
//--------------------------------------------------------------------------------------
Texture2D tex0 : register(t0);
SamplerState sampLinearClamp : register(s0);
SamplerState sampNearestClamp : register(s1);
SamplerState sampLinearWrap : register(s2);
SamplerState sampNearestWrap : register(s3);


cbuffer GlobalRenderData: register(b0)
{
    float g_fHue;
    float g_fSaturation;
    float g_fLuminance;
    float gbrpad1;
};



//--------------------------------------------------------------------------------------
struct VS_INPUT
{
    float3 vecPosition : POSITION;
    float2 vecTexCoord0 : TEXCOORD0;
};

struct PS_INPUT
{
    float4 vecPosition : SV_POSITION;
    float2 vecTexCoord0 : TEXCOORD0;
};


//--------------------------------------------------------------------------------------
// Support functions
//--------------------------------------------------------------------------------------
float3 RGBToHSL(float3 color)
{
    float3 hsl; // init to 0 to avoid warnings ? (and reverse if + remove first part)

    float fmin = min(min(color.r, color.g), color.b);    //Min. value of RGB
    float fmax = max(max(color.r, color.g), color.b);    //Max. value of RGB
    float delta = fmax - fmin;             //Delta RGB value

    hsl.z = (fmax + fmin) / 2.0; // Luminance

    if (delta == 0.0)		//This is a gray, no chroma...
    {
        hsl.x = 0.0;	// Hue
        hsl.y = 0.0;	// Saturation
    }
    else                                    //Chromatic data...
    {
        if (hsl.z < 0.5)
            hsl.y = delta / (fmax + fmin); // Saturation
        else
            hsl.y = delta / (2.0 - fmax - fmin); // Saturation

        float deltaR = (((fmax - color.r) / 6.0) + (delta / 2.0)) / delta;
        float deltaG = (((fmax - color.g) / 6.0) + (delta / 2.0)) / delta;
        float deltaB = (((fmax - color.b) / 6.0) + (delta / 2.0)) / delta;

        if (color.r == fmax)
            hsl.x = deltaB - deltaG; // Hue
        else if (color.g == fmax)
            hsl.x = (1.0 / 3.0) + deltaR - deltaB; // Hue
        else if (color.b == fmax)
            hsl.x = (2.0 / 3.0) + deltaG - deltaR; // Hue

        if (hsl.x < 0.0)
            hsl.x += 1.0; // Hue
        else if (hsl.x > 1.0)
            hsl.x -= 1.0; // Hue
    }

    return hsl;
}

float HueToRGB(float f1, float f2, float hue)
{
    if (hue < 0.0)
        hue += 1.0;
    else if (hue > 1.0)
        hue -= 1.0;
    float res;
    if ((6.0 * hue) < 1.0)
        res = f1 + (f2 - f1) * 6.0 * hue;
    else if ((2.0 * hue) < 1.0)
        res = f2;
    else if ((3.0 * hue) < 2.0)
        res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
    else
        res = f1;
    return res;
}

float3 HSLToRGB(float3 hsl)
{
    float3 rgb;

    if (hsl.y == 0.0)
        rgb = float3(hsl.z, hsl.z, hsl.z); // Luminance
    else
    {
        float f2;

        if (hsl.z < 0.5)
            f2 = hsl.z * (1.0 + hsl.y);
        else
            f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);

        float f1 = 2.0 * hsl.z - f2;

        rgb.r = HueToRGB(f1, f2, hsl.x + (1.0 / 3.0));
        rgb.g = HueToRGB(f1, f2, hsl.x);
        rgb.b = HueToRGB(f1, f2, hsl.x - (1.0 / 3.0));
    }

    return rgb;
}


//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
PS_INPUT VS( VS_INPUT input )
{
    PS_INPUT output;

    float2 vecTexCoord0 = input.vecTexCoord0.xy * float2(2, -2) + float2(-1, 1);
    output.vecPosition = float4(vecTexCoord0, 0, 1);
    output.vecTexCoord0 = input.vecTexCoord0;

    return output;
}


//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS(PS_INPUT input) : SV_Target
{
    float4 vecSourceColour = tex0.Sample(sampLinearClamp, input.vecTexCoord0);

    float3 vecHSL = RGBToHSL(vecSourceColour.rgb);
    vecHSL.x = ((vecHSL.x + 2.0 + g_fHue)) - 2.0;
    vecHSL.y *= g_fSaturation;
    vecHSL.z *= g_fLuminance;

    float4 vecOutput;
    vecOutput.rgb = HSLToRGB(vecHSL);
    vecOutput.a = vecSourceColour.a;
    return vecOutput;
}
