#version 330

uniform mat4 uViewMatrix;
uniform sampler2D uDiffuseMap;
uniform sampler2D uMetallicMap;

uniform float ucCameraFar;
uniform float ucSkyColor;

in vec3 fPosition;
in vec3 fNormal;
in vec2 fTexCoord;

out vec4 fragColor; // first out variable is automatically written to the screen

/** Lights
 * 
 * 
 * 
 * 
 * 
 */




 
 
#define MAX_SPECULAR_POW 64
#define LIGHT_AMOUNT 1

struct Light
{
	vec3 color;
	float linearAttenuation;
	vec3 positionDirection;
	float quadraticAttenuation;
};

struct Material
{
	float diffuseStrength;
	float specularPower;
	float specularStrength;
};


uniform Light uLights[LIGHT_AMOUNT];
 
 Material getMaterial(float specularValue)
{
	Material mat;

	float diffuseS = 1.1f - specularValue;
	mat.diffuseStrength = diffuseS;
	float specularS = specularValue - 0.1f;
	mat.specularStrength = specularS;
	mat.specularPower = 2 + specularValue * MAX_SPECULAR_POW;
	
	return mat;
}
 
 float getAttenuation(float distanceToLight, float linear, float quadratic)
{
	return 1.0f / (1.0f + linear * distanceToLight + quadratic * (distanceToLight * distanceToLight));
}
 
vec4 getDirectionalLight(
	  vec3 viewPos
	, vec3 viewNormal
	, Light l
	, Material m)
{
	// On direction vector the w component must be 0.0f
	vec3 viewLightDir = vec3(uViewMatrix * vec4(l.positionDirection, 0.0f));
	
	float ambientStrength = l.linearAttenuation;
	float attenuation = l.quadraticAttenuation; // Intensity
	vec3 toFragment = viewLightDir;
	vec3 toLight = toFragment * -1.0f;
	// Ambient color
	vec4 lightColor4 = vec4(l.color, 1.0f);
	vec4 ambientResult = lightColor4 * ambientStrength;
	
	// Diffuse color
	vec3 norm = normalize(viewNormal);
	float diffuse = max(dot(norm, toLight), 0.0);
	vec4 diffuseResult = m.diffuseStrength * diffuse * lightColor4 * attenuation;
	
	// Specular color
	// In view space the view position is (0,0,0)
	vec3 viewDir = normalize(-viewPos);
	vec3 reflectDir = reflect(toFragment, norm);
	float specular = pow(max(dot(viewDir, reflectDir), 0.0), m.specularPower);
	vec4 specularResult = m.specularStrength * specular * lightColor4 * attenuation;
	
	return (ambientResult + diffuseResult + specularResult);
}

vec4 calculateLights(
	  vec3 viewPos
	, vec3 viewNormal
	, Material mat)
{
	vec4 lightColor = vec4(0,0,0,1);
		// Light 0 is directional
	lightColor += getDirectionalLight(viewPos, viewNormal, uLights[0], mat);
	
	return lightColor;
}

float near = 0.1; 
float far  = ucCameraFar;
  
float LinearizeDepth(float depth) 
{
    float z = depth * 2.0 - 1.0; // back to NDC 
    float distance = (2.0 * near * far) / (far + near - z * (far - near));	
    return distance / far;
}

void main()
{
	vec4 texColor = texture(uDiffuseMap, fTexCoord);
	vec4 specularColor = texture(uMetallicMap, fTexCoord);
	
	Material mat = getMaterial(specularColor.x);
	
	vec4 color = texColor * calculateLights(fPosition, fNormal, mat);
	vec4 skyColorDay = vec4(76.0/255.0, 153.0/255.0, 235.0/255.0, 1.0);
	vec4 skyColorEvening = vec4(220.0/255.0, 72.0/255.0, 14.0/255.0, 1.0);
	float depth = LinearizeDepth(gl_FragCoord.z);
	depth = depth * depth;
	float colorPower = 1.0 - depth; 
	vec4 skyColorActive = ucSkyColor * skyColorDay + (1.0f - ucSkyColor) * skyColorEvening;
	if (ucSkyColor < 0.0f)
	{
		float black = ucSkyColor * -1.0f;
		skyColorActive = skyColorDay * black + (1.0f - black) * skyColorEvening;
	}
	skyColorActive = skyColorActive * uLights[0].linearAttenuation;
	vec4 finalColor = color * colorPower + skyColorActive * depth;
	finalColor.w = texColor.w; // Keep texture alpha
	fragColor = finalColor;
}
