// Encoding: UTF-8
function backingScale(context) {
  if ('devicePixelRatio' in window) {
    if (window.devicePixelRatio > 1) {
      return window.devicePixelRatio;
    }
  }
  return 1;
}

App = function app(){

  //Shorthand for App object itself:
  var a = this;

  //Handles window resize
  a.resize = function(){
    var width = window.innerWidth;
    var height = window.innerHeight;

    a.camera.aspect = width / height;
    a.camera.updateProjectionMatrix();
    a.renderer.setSize(width, height);

    a.canvas.width = width * a.canvasScaleFactor;
    a.canvas.height = height * a.canvasScaleFactor;
    a.canvas.style.width = width;
    a.canvas.style.height = height;
    a.context = a.canvas.getContext("2d");
    a.context.scale(a.canvasScaleFactor, a.canvasScaleFactor);

    var outerRadius = width * 0.5;
    var innerRadius = width * 0.3;
    a.vignetteGradient = a.context.createRadialGradient(width / 2, height / 2, innerRadius, width / 2, height / 2, outerRadius);

  };

  //Initialises object
  a.init = function(){

    var audio = new Audio('assets/thykka_hhsf_invite.ogg');

    //Get canvas context
    a.canvas = document.getElementById('fraktopy');
    a.context = a.canvas.getContext('2d');
    a.canvasScaleFactor = backingScale(a.context);

    //Viewport dimensions
    var width = window.innerWidth;
    var height = window.innerHeight;

    //Setup canvas
    a.canvas.width = width;
    a.canvas.height = height;

    a.canvas.width = width * a.canvasScaleFactor;
    a.canvas.height = height * a.canvasScaleFactor;
    a.canvas.style.width = width;
    a.canvas.style.height = height;
    a.context = a.canvas.getContext("2d");
    a.context.scale(a.canvasScaleFactor, a.canvasScaleFactor);

    //WebGL renderer
    a.renderer = new THREE.WebGLRenderer({antialias: true});
    a.renderer.setSize(document.body.clientWidth, document.body.clientHeight);
    document.body.appendChild(a.renderer.domElement);

    //Bind to window resize event
    window.addEventListener('resize', a.resize, false);

    //Click to go fullscreen
    a.renderer.domElement.addEventListener('click', a.fullscreen, false);

    //Create a scene
    a.scene = new THREE.Scene();

    //Add fog
    a.scene.fog = new THREE.Fog( 0x000000, 1, 1000 );

    //Camera
    a.fov = 60;
    a.aspect = width / height;
    a.camera = new THREE.PerspectiveCamera( a.fov, a.aspect );

    a.camera.position.z = 550;
    a.camera.position.y = 190;
    a.camera.lookAt(new THREE.Vector3(0,200,0));

    //Lights
    a.lights = [];
    a.lights['ambientLight'] = new THREE.AmbientLight( 0xffffff );
    a.scene.add(a.lights['ambientLight']);

    //Shaders
    a.mountain_heightmap = new THREE.ImageUtils.loadTexture("assets/heightmap.png");
    a.mountain_texture = new THREE.ImageUtils.loadTexture("assets/color2.png");
    a.ground_texture = new THREE.ImageUtils.loadTexture("assets/color.png");

    a.shader = THREE.ShaderLib["normalmap"];
    a.uniforms = THREE.UniformsUtils.clone(a.shader.uniforms);

    a.uniforms["enableDisplacement"].value = true;
    a.uniforms["enableDiffuse"].value = true;
    a.uniforms["tDisplacement"].value = a.mountain_heightmap;
    a.uniforms["tDiffuse"].value = a.mountain_texture;
    a.uniforms["uDisplacementScale"].value = 60;

    //Create materials
    a.mountain_material = new THREE.ShaderMaterial({
      fragmentShader: a.shader.fragmentShader,
      vertexShader: a.shader.vertexShader,
      uniforms: a.uniforms,
      lights: true,
      wireframe: true
    });

    a.ground_material = new THREE.MeshPhongMaterial({
      color: '#ffffff',
      map: a.ground_texture,
      wireframe: true,
      blending: THREE.AdditiveBlending
    });

    //Create mountain geometry
    a.mountains_geometry = new THREE.PlaneGeometry(800, 600, 200, 150);
    a.mountains_geometry.applyMatrix( new THREE.Matrix4().makeTranslation( 80, 20, -38 ) );
    a.mountains_geometry.computeTangents();

    a.mountains = new THREE.Mesh(a.mountains_geometry, a.mountain_material);
    a.mountains.rotation.x = -Math.PI / 3;
    a.mountains.position.x = -200;
    a.mountains.position.y = -20;
    a.mountains.position.z = -20;

    a.scene.add(a.mountains);

    //Create ground geometry
    a.ground_geometry = new THREE.PlaneGeometry(800, 600, 200, 150);
    a.ground_geometry.applyMatrix( new THREE.Matrix4().makeTranslation( 80, 20, -38 ) );

    a.ground = new THREE.Mesh(a.ground_geometry, a.ground_material);
    a.ground.rotation.x = -Math.PI / 3;
    a.ground.position.x = -200;
    a.ground.position.y = -35;
    a.ground.position.z = -20;

    a.scene.add(a.ground);

    //Create gradient for canvas vignette effect
    var outerRadius = width * 0.5;
    var innerRadius = width * 0.3;
    a.vignetteGradient = a.context.createRadialGradient(width / 2, height / 2, innerRadius, width / 2, height / 2, outerRadius);
    a.vignetteGradient.addColorStop(0, 'rgba(0, 0, 0, 0)');
    a.vignetteGradient.addColorStop(1, 'rgba(0, 0 ,0, 0.3)');

    a.triangleScale = 0;

    //Reset time
    a.lastUpdate = 0;
    a.firstUpdate = Date.now();
    a.lastBeat = false;
    a.currentBeat = 0;
    a.offset = 0;

    a.text = '';

    a.greetz = [
      'hey you!', ' ', 'bionik', 'invites you', 'to hhsf2018', 'at someplace', 'in jyväskylä', '1.-4.3.', '2018', 'ebin party', 'hosted by', 'elite dekkerz', 'the ultimate', 'hacker collective', 'prepare', 'for satisfactory', 'beverage', 'consumption', 'greetings to:', 'netl', 'anacron', 'ermuggo', 'zidit', 'vurpo', 'faplab', 'hacklab.fi', ' ', 'ಠ_ಠ', ' ', 'music:', 'thykka' ,' ', 'never', 'stop', 'hacking', ' '
    ];
    a.current_greet = 0;

    a.beatratio = 6857;

    audio.addEventListener('ended', function() {
      audio.currentTime = 0;
      audio.play();
    }, false);

    audio.addEventListener('canplaythrough', function(){
      setTimeout(function(){ audio.play(); }, 20);
      //Start loop
      a.update();
    }, false);

  };

  //Sets fullscreen mode
  a.fullscreen = function(){
    if (document.body.requestFullscreen) {
      document.body.requestFullscreen();
    } else if (document.body.msRequestFullscreen) {
      document.body.msRequestFullscreen();
    } else if (document.body.mozRequestFullScreen) {
      document.body.mozRequestFullScreen();
    } else if (document.body.webkitRequestFullscreen) {
      document.body.webkitRequestFullscreen();
    }
  };

  //Update state and draw
  a.update = function() {
    requestAnimationFrame(a.update);

    var time = Date.now() - a.firstUpdate;
    var dt = time - a.lastUpdate;
    a.lastUpdate = time;

    a.animate(time, dt);
    a.draw();
  };

  a.doBeat = function(time){
    if(a.currentBeat == 0){
      a.offset = Math.sin(time*8008135)*2000+10000;
    }
  };

  //Animate objects based on time
  a.animate = function(time, dt){

    //Make the beat do something... calculate beat on bpm
    var beatposition = ((time + 40) % a.beatratio)/a.beatratio;

    var beatposition_r = ((time + 30) % a.beatratio)/a.beatratio;
    //var beatposition_g = ((time + 20) % a.beatratio)/a.beatratio;
    var beatposition_b = ((time + 10) % a.beatratio)/a.beatratio;

    a.triangleScale = Math.sin( 2*Math.PI * 2 * beatposition ) * 100;

    a.triangleScale_r = Math.sin( 2*Math.PI * 2 * beatposition_r ) * 100;
    //a.triangleScale_g = Math.sin( 2*Math.PI * 2 * beatposition_g ) * 100;
    a.triangleScale_b = Math.sin( 2*Math.PI * 2 * beatposition_b ) * 98;

    var currentBeat = Math.floor(4 * beatposition);
    if(a.lastBeat !== currentBeat){

      if(currentBeat == 0) a.doBeat(time);

      a.current_greet++;
      if(a.current_greet >= a.greetz.length){
        a.current_greet = 0;
      }
      a.text = a.greetz[a.current_greet];

      a.lastBeat = currentBeat;
    }

    //Move stuff
    a.mountains.rotation.z = Math.cos(time*0.0002+a.offset)*5;
    a.mountains.rotation.y = Math.cos(time*0.0002+a.offset)*0.2;
    a.mountains.position.x = Math.cos(time*0.00044+a.offset)*6 + 30;
    a.mountains.position.z = Math.cos(time*0.00003+a.offset)*2 + 30;

    a.ground.rotation.z = Math.cos(time*0.0002+a.offset)*5;
    a.ground.rotation.y = Math.cos(time*0.0002+a.offset)*0.2;
    a.ground.position.x = Math.cos(time*0.00044+a.offset)*6 + 30;
    a.ground.position.z = Math.cos(time*0.00003+a.offset)*2 + 30;

    //Camera
    a.camera.fov = Math.sin(time*0.00008+a.offset)*10 + 66;
    a.camera.updateProjectionMatrix();

    a.camera.position.y = Math.sin(time*0.0006+a.offset)*30 + -80;
    a.camera.position.z = Math.sin(time*0.0002+a.offset)*5 + 220;
    a.camera.position.x = Math.sin(time*0.00013+a.offset)*2;

  };

  //Draws the three.js scene and canvas overlay
  a.draw = function() {
    a.renderer.setClearColor(0x000000, 1);
    a.context.clearRect(0, 0, a.canvas.width, a.canvas.height);
    a.renderer.render(a.scene, a.camera);

    var width = window.innerWidth;
    var height = window.innerHeight;

    //Draw vignette
    a.context.rect(0, 0, width, height);
    a.context.fillStyle = a.vignetteGradient;
    a.context.fill();

    //Draw fraktopy bg
    a.context.fillStyle = '#ffffff';
    a.context.fillRect(100, 0, 260, 123);

    //Draw fraktopy text
    a.context.font = '50px Oswald';
    a.context.fillStyle = '#000000';
    a.context.textBaseline = 'top';
    a.context.textAlign = 'left';
    a.context.fillText('hhsf2018', 135, 42);

    var screenfactor = height/1080;
    var scaleoffset = 2.1 * screenfactor;
    var positionoffset;

    /*a.context.save();
    positionoffset = 0.3;
    a.context.transform(1,0,0,1,width/2,height/2+a.triangleScale*positionoffset);
    a.context.beginPath();
    a.context.strokeStyle = "#95ff0090";
    a.context.lineWidth = 32;
    a.context.moveTo(-2*a.triangleScale_g*scaleoffset,1*a.triangleScale_g*scaleoffset);
    a.context.lineTo(2*a.triangleScale_g*scaleoffset,1*a.triangleScale_g*scaleoffset);
    a.context.lineTo(0,-2*a.triangleScale_g*scaleoffset);
    a.context.lineTo(-2*a.triangleScale_g*scaleoffset,1*a.triangleScale_g*scaleoffset);
    a.context.closePath();
    a.context.stroke();
    a.context.restore();*/

    a.context.save();
    positionoffset = 0.1;
    a.context.transform(1,0,0,1,width/2+1,height/2+a.triangleScale*positionoffset);
    a.context.beginPath();
    a.context.strokeStyle = "#0076ff90";
    a.context.lineWidth = screenfactor *  30;
    a.context.moveTo(-2*a.triangleScale_b*scaleoffset,1*a.triangleScale_b*scaleoffset);
    a.context.lineTo(2*a.triangleScale_b*scaleoffset,1*a.triangleScale_b*scaleoffset);
    a.context.lineTo(0,-2*a.triangleScale_b*scaleoffset);
    a.context.lineTo(-2*a.triangleScale_b*scaleoffset,1*a.triangleScale_b*scaleoffset);
    a.context.closePath();
    a.context.stroke();
    a.context.restore();

    a.context.save();
    positionoffset = -0.05;
    a.context.transform(1,0,0,1,width/2-1,height/2+a.triangleScale*positionoffset);
    a.context.beginPath();
    a.context.strokeStyle = "#ff006d90";
    a.context.lineWidth = screenfactor * 32;
    a.context.moveTo(-2*a.triangleScale_r*scaleoffset,1*a.triangleScale_r*scaleoffset);
    a.context.lineTo(2*a.triangleScale_r*scaleoffset,1*a.triangleScale_r*scaleoffset);
    a.context.lineTo(0,-2*a.triangleScale_r*scaleoffset);
    a.context.lineTo(-2*a.triangleScale_r*scaleoffset,1*a.triangleScale_r*scaleoffset);
    a.context.closePath();
    a.context.stroke();
    a.context.restore();

    scaleoffset = 2.0 * screenfactor;
    a.context.save();
    a.context.transform(1,0,0,1,width/2,height/2+a.triangleScale*positionoffset);
    a.context.beginPath();
    a.context.strokeStyle = "#ffffffff";
    a.context.lineWidth = screenfactor * 28;
    a.context.moveTo(-2*a.triangleScale*scaleoffset,1*a.triangleScale*scaleoffset);
    a.context.lineTo(2*a.triangleScale*scaleoffset,1*a.triangleScale*scaleoffset);
    a.context.lineTo(0,-2*a.triangleScale*scaleoffset);
    a.context.lineTo(-2*a.triangleScale*scaleoffset,1*a.triangleScale*scaleoffset);
    a.context.closePath();
    a.context.stroke();
    a.context.restore();

    a.context.save();
    a.context.font = '50px Oswald';
    if(a.text.length >= 16) a.context.font = '40px Oswald';
    if(a.text.length <= 8) a.context.font = '65px Oswald';
    a.context.fillStyle = '#ffffff';
    a.context.textBaseline = 'middle';
    a.context.textAlign = 'center';
    a.context.translate(width/2, height/2);
    var factor = screenfactor * Math.abs(a.triangleScale/100)*1.3;
    a.context.scale(0.3 + factor, 0.3 + factor);
    a.context.fillText(a.text, 0, 0);
    a.context.restore();


  };

  a.init();

};