package ai;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import bot.Bot;
import bot.Bot.Direction;
import game.Ghost;
import game.Map;
import game.Map.ObstacleType;
import util.Position;

public class MiniMax {

	public enum NodeType {
		MAX, MIN
	}
	
	private Map map;
	
	private int depth;
	
	private Node root;
	
	private int nodes = 0;
	
	public MiniMax(Map map, int depth) {
		this.map = map;
		this.depth = depth;
	}
	
	public void calculateTree(Bot bot) {
		long time = System.currentTimeMillis();
		nodes = 0;
		
		Node root = new Node();
		root.position = bot.ghost.position;
		root.children = generateChildren(map, root, bot.ghost.position);
		
		for (Node child : root.children) {
			child.score = minimax(map, child, depth, NodeType.MIN);
		}
		this.root = root;
		
		System.out.println("Created " + nodes + " nodes in " + (System.currentTimeMillis() - time) + " ms.");
	}
	
	private float minimax(Map map, Node node, int depth, NodeType type) {
		if (depth == 0) {
			return node.score;
		}
		
		float v = Integer.MIN_VALUE;
		List<Node> children = generateChildren(map, node, node.position);
		for (Node child : children) {
			v = Math.max(v, minimax(map, child, depth - 1, NodeType.MIN));
		}
		return v;
	}
	
	private List<Node> generateChildren(Map map, Node node, Position<Integer, Integer> position) {
		List<Node> children = new ArrayList<Node>();
		
		List<Direction> directions = map.getAvailableMoves(position);
		for (Direction direction : directions) {
			Position<Integer, Integer> p = Map.getPosition(position, direction);
			
			Node child = new Node();
			child.direction = direction;
			child.position = p;
			child.score = map.calculateScore(position) + node.score;
	
			children.add(child);
		}
		
		return children;
	}

	public Direction getBestDirection() {
		if (root == null || root.children == null) {
			return null;
		}
		float bestScore = Integer.MIN_VALUE;
		List<Node> potential = new ArrayList<Node>();
		for (Node node : root.children) {
			if (node.score > bestScore && node.score > 0) {
				potential.clear();
				potential.add(node);
				bestScore = node.score;
			} else if (node.score == bestScore) {
				potential.add(node);
			}
		}
		if (potential.isEmpty()) {
			return findNearest(root);
		}
		Node best = potential.get(new Random().nextInt(potential.size()));
		
		return best.direction;
	}
	
	private Direction findNearest(Node root) {
		Position<Integer, Integer> p = root.position;
		
		ObstacleType[][] obstacles = map.obstacles;
		
		List<Node> visited = new ArrayList<Node>();
		Node[][] nodes = new Node[obstacles.length][obstacles[0].length];
		for (int x = 0; x < nodes.length; x++) {
			for (int y = 0; y < nodes[0].length; y++) {
				nodes[x][y] = new Node();
				nodes[x][y].position = new Position<Integer, Integer>(x, y);
			}
		}
		
		LinkedList<Node> queue = new LinkedList<Node>();
		queue.add(nodes[p.x][p.y]);
		while (!queue.isEmpty()) {
			Node n = queue.pop();
			
			if (obstacles[n.position.x][n.position.y] == ObstacleType.WALL) {
				continue;
			}
			
			if (obstacles[n.position.x][n.position.y] == ObstacleType.PELLET ||
					obstacles[n.position.x][n.position.y] == ObstacleType.SUPERPELLET) {
				Direction direction = null;
				while (n.parent != null) {
					n = n.parent;
					if (n.direction != null) {
						direction = n.direction;
					}
				}
				return direction;
			}
			
			try {
				Node up = nodes[n.position.x][n.position.y - 1];
				if (!visited.contains(up)) {
					up.parent = n;
					up.direction = Direction.UP;
					queue.add(up);
				}
			} catch (Exception e) {}
			
			try {
				Node down = nodes[n.position.x][n.position.y + 1];
				if (!visited.contains(down)) {
					down.parent = n;
					down.direction = Direction.DOWN;
					queue.add(down);
				}
			} catch (Exception e) {}
			
			try {
				Node left = nodes[n.position.x - 1][n.position.y];
				if (!visited.contains(left)) {
					left.parent = n;
					left.direction = Direction.LEFT;
					queue.add(left);
				}
			} catch (Exception e) {}
			
			try {
				Node right = nodes[n.position.x + 1][n.position.y];
				if (!visited.contains(right)) {
					right.parent = n;
					right.direction = Direction.RIGHT;
					queue.add(right);
				}
			} catch (Exception e) {}
			
			visited.add(n);
		}
		return null;
	}
	
	/*private Direction getDirectionToClosest(Node root) {
		Position<Integer, Integer> p = root.position;
		
		ObstacleType[][] obstacles = map.obstacles;
		
		float distance = Integer.MIN_VALUE;
		//Direction direction = null;
		
		List<Direction> available = map.getAvailableMoves(root.position);
		List<Direction> potential = new ArrayList<Direction>();
		
		for (int x = 0; x < obstacles.length; x++) {
			for (int y = 0; y < obstacles[x].length; y++) {
				if (obstacles[x][y] == ObstacleType.PELLET || obstacles[x][y] == ObstacleType.SUPERPELLET) {
					float d0 = (float) Math.sqrt(Math.pow(x - (p.x - 1), 2) + Math.pow(y - p.y, 2));
					float d1 = (float) Math.sqrt(Math.pow(x - (p.x + 1), 2) + Math.pow(y - p.y, 2));
					float d2 = (float) Math.sqrt(Math.pow(x - p.x, 2) + Math.pow(y - (p.y - 1), 2));
					float d3 = (float) Math.sqrt(Math.pow(x - p.x, 2) + Math.pow(y - (p.y + 1), 2));
					
					if (d0 > distance && available.contains(Direction.RIGHT)) {
						distance = d0;
						potential.clear();
						potential.add(Direction.RIGHT);
					} else if (d0 == distance && available.contains(Direction.RIGHT)) {
						potential.add(Direction.RIGHT);
					}
					if (d1 > distance && available.contains(Direction.LEFT)) {
						distance = d1;
						potential.add(Direction.LEFT);
					} else if (d1 == distance && available.contains(Direction.LEFT)) {
						potential.add(Direction.LEFT);
					}
					if (d2 > distance && available.contains(Direction.DOWN)) {
						distance = d2;
						potential.add(Direction.DOWN);
					} else if (d2 == distance && available.contains(Direction.DOWN)) {
						potential.add(Direction.DOWN);
					}
					if (d3 > distance && available.contains(Direction.UP)) {
						distance = d3;
						potential.add(Direction.UP);
					} else if (d3 == distance && available.contains(Direction.UP)) {
						potential.add(Direction.UP);
					}
				}
			}
		}
		
		return potential.get(new Random().nextInt(potential.size()));
	}*/
	
	private class Node {

		public Node parent;
		
		public List<Node> children;
		
		public Direction direction;
		
		public Position<Integer, Integer> position;
		public float score;
		
		public Node() {
			nodes++;
		}
		
	}
}
