// KugelBlitz Lite
// Mike Erwin aka DangerCobraM
// SynchroNY 2017 /// NYC --> Montreal

// Developed on & for a Raspberry Pi 2 + PiTFT 320x240 display
// "physics" here are super fake, optimized for fun at the expense of accuracy

#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <unistd.h>
#include <assert.h>
#include <algorithm>
#include <array>
#include <vector>

using byte = uint8_t;
using u16 = uint16_t;
using s16 = int16_t;
using u32 = uint32_t;


int f; // file descriptor for /dev/fb1

u32 rnd(u32 x)
	{
	return random() % x;
	}

float rndf()
	{
	return 0.001f * rnd(1000);
	}

struct Pixel
	{
	u16 opaque : 1;
	u16 b : 5;
	u16 g : 5;
	u16 r : 5;
	};

struct Vec2
	{
	float x, y;

	float mag2() const
		{
		return x * x + y * y;
		}

	float mag() const
		{
		return sqrtf(mag2());
		}

	Vec2& operator+=(const Vec2& rhs)
		{
		x += rhs.x;
		y += rhs.y;
		return *this;
		}
	};

Vec2 operator+(const Vec2& a, const Vec2& b)
	{
	return { a.x + b.x, a.y + b.y };
	}

Vec2 operator-(const Vec2& a, const Vec2& b)
	{
	return { a.x - b.x, a.y - b.y };
	}

Vec2 operator*(float s, const Vec2& v)
	{
	return { s * v.x, s * v.y };
	}

Vec2 operator-(const Vec2& v)
	{
	return { -v.x, -v.y };
	}

Vec2 normalize(const Vec2& v)
	{
	return (1.0f / v.mag()) * v;
	}

Vec2 rnd_unit()
	{
	Vec2 v;
	float m2;
	do {
		u32 bits = random();
		v.x = (bits & 0xffff) * (2.0f / 65535.0f) - 1.0f;
		v.y = ((bits >> 15) & 0xffff) * (2.0f / 65535.0f) - 1.0f;
		m2 = v.mag2();
		} while (m2 > 1.0f);
	float s = 1.0f / sqrtf(m2);
	return s * v;
	}

struct Pos
	{
	u16 x, y;

	Pos(int X=0, int Y=0)
		: x { (u16)X }
		, y { (u16)Y }
		{ }

	Pos(const Vec2 v)
		: x { (u16)v.x } // should round?
		, y { (u16)v.y }
		{ }
	};

struct Offset
	{
	s16 x, y;
	};

struct Image
	{
	const u16 width;
	const u16 height; 
	Pixel* pixels {nullptr};

	Image(u16 w, u16 h)
		: width { w }
		, height { h }
		, pixels { new Pixel[w * h] }
		{ }

	const Pixel& pixel(Pos p) const
		{
		if (p.x < width && p.y < height)
			return pixels[p.y * width + p.x];
		else
			return pixels[0];
		}

	void clear()
		{
		memset(pixels, 0, width * height * sizeof(Pixel));
		}

	void pset(Pos p, Pixel color)
		{
		if (p.x < width && p.y < height)
			pixels[p.y * width + p.x] = color;
		}

	void line(Pos a, Pos b, Pixel color)
		{
#if 0
		// "approximation" v1
		pset(a, color);
		pset(b, color);
#endif

		// v2
		u16 first, last;
		if (a.x == b.x) // vertical
			{
			if (a.y <= b.y)
				{
				first = a.y;
				last = b.y;
				}
			else
				{
				first = b.y;
				last = a.y;
				}
			for (u16 y = first; y <= last; ++y)
				pset({a.x, y}, color);
			}
		else if (a.y == b.y) // horizontal
			{
			if (a.x <= b.x)
				{
				first = a.x;
				last = b.x;
				}
			else
				{
				first = b.x;
				last = a.x;
				}
			for (u16 x = first; x <= last; ++x)
				pset({x, a.y}, color);
			}
		else
			{
			int dx = b.x - a.x;
			int dy = b.y - a.y;

			if (abs(dx) >= abs(dy)) // more horizontal
				{
				if (b.x < a.x)
					{
					std::swap(a, b);
					dx = -dx;
					dy = -dy;
					}
				float perpx = (float)dy / (float)dx;
				float yf = a.y;
				for (u16 x = a.x; x <= b.x; ++x)
					{
					pset({x, (u16)yf}, color);
					yf += perpx;
					}
				}
			else // more vertical
				{
				if (b.y < a.y)
					{
					std::swap(a, b);
					dx = -dx;
					dy = -dy;
					}
				float perpx = (float)dx / (float)dy;
				float xf = a.x;
				for (u16 y = a.y; y <= b.y; ++y)
					{
					pset({(u16)xf, y}, color);
					xf += perpx;
					}
				}
			}
		}

	// blit another image/sprite onto this one
	void blitImage(const Image& img, Offset off)
		{
		u16 x_ct = img.width;
		u16 y_ct = img.height;
		Pos src_base, dst_base;
		// TODO: clip to dest image (this) bounding rect
		for (u16 y = 0; y < y_ct; ++y)
			for (u16 x = 0; x < x_ct; ++x)
			{
				const Pixel& px = img.pixel({src_base.x + x, src_base.y + y});
				if (px.opaque)
					pset({dst_base.x + x, dst_base.y + y}, px);
			}
		}

	void blitToScreen() const
		{
		assert(width == 320);
		pwrite(f, pixels, width * height * sizeof(Pixel), 0);
		}
	};

Pixel convert(byte r, byte g, byte b, byte a)
	{
	return {
		.r = (u16)(r >> 3),
		.g = (u16)(g >> 3),
		.b = (u16)(b >> 3),
		.opaque = (a != 0)
		};
	}

Pixel convert(const byte rgba[4])
	{
	return {
		.r = (u16)(rgba[0] >> 3),
		.g = (u16)(rgba[1] >> 3),
		.b = (u16)(rgba[2] >> 3),
		.opaque = (rgba[3] != 0)
		};
	}

#if 0 // if only time permits...
Image loadPNG(const char* filename)
	{
	// open file with libpng

	u16 width = 0; // get these from libpng
	u16 height = 0;

	Image img(width, height);

	// decode PNG pixels into PiTFT pixels
 
 	Pos p;
	for (p.y = 0; p.y < height; ++p.y)
		{
		// get scanline
		for (p.x = 0; p.x < width; ++p.x)
			{
			img.pset(p, convert(pixel));
			}
		}

	return img;
	}
#endif

Image fb { 320, 240 }; // framebuffer image

void init()
	{
	srandom(time(0));
	f = open("/dev/fb1", O_WRONLY);	
	}

struct Spark
	{
	Vec2 pos;
	Vec2 delta;
	float age;
	u16 col_id;

	Pixel color() const
		{
		// based on age (and velocity?)
		constexpr u16 color_ct = 7;
		static const std::array<Pixel,color_ct> colors = {
			convert(255, 195, 0, 255),
			convert(255, 162, 0, 255),
			convert(255, 130, 0, 255),
			convert(255, 97, 0, 255),
			convert(192, 74, 0, 255),
			convert(128, 49, 0, 255),
			convert(50, 25, 0, 255),
			};
		return colors[col_id];
		}
	};

#define SPARK_CT 500

Spark sparks[SPARK_CT];

void init_sparks()
	{
	for (auto& spark : sparks)
		{
		spark.pos.x = rnd(fb.width);
		spark.pos.y = rnd(fb.height);
		spark.delta = 5.0f * rnd_unit();
		spark.delta.y += 2.0f; // tend upwards
		spark.age = 0.0f;
		spark.col_id = rnd(7);
		}
	}

// const Pixel spark_col = { .r = 31, .g = 10, .b = 0 };

// void wiggle_sparks()
// 	{
// 	for (int i = 0; i < SPARK_CT; ++i)
// 		{
// 		sparks[i].pos += sparks[i].delta;
// 		sparks[i].delta += 0.1f * rnd_unit();
// 		fb.pset(sparks[i].pos, spark_col);
// 		}
// 	}

void move_sparks()
	{
	constexpr float spark_grav = 0.05f;
	for (auto& spark : sparks)
		{
		spark.delta.y += spark_grav;

		if (spark.delta.mag() > 5)
			spark.delta = 0.95f * spark.delta; // decay

		if (spark.pos.y + spark.delta.y >= fb.height)
			{
			spark.delta.y = -spark.delta.y;
			// spark.delta += 0.1f * rnd_unit();
			Vec2 boost = rnd_unit();
			boost.y = fabsf(boost.y);
			spark.delta += boost;
			}

		spark.pos += spark.delta;

		fb.pset(spark.pos, spark.color());
		}
	}


struct Bolt
	{
	static const u16 ct { 10 };
	static const u16 old { 200 };

	std::array<Pos,9> pts;
	u16 age { 0 };

	void fracture(u16 a, u16 b)
		{
		Vec2 pa { (float)pts[a].x, (float)pts[a].y };
		Vec2 pb { (float)pts[b].x, (float)pts[b].y };
		Vec2 mid = 0.5f * (pa + pb);
		// Vec2 tweak = pa - pb;
		// float len = tweak.mag();
		// std::swap(tweak.x, tweak.y);
		// float s = (rndf() - 0.5f) * len;
		Vec2 tweak = rnd_unit();
		float s = 0.15f * (pa - pb).mag();
		Vec2 p = mid + s * tweak;
		u16 ab = (a + b) / 2;
		pts[ab] = p;
		if (b - a > 2)
			{
			fracture(a, ab);
			fracture(ab, b);
			}
		}

	Bolt(Pos a, Pos b)
		{
		// simple line for now
		// fracture later!
		pts[0] = a;
		pts.back() = b;
		fracture(0, pts.size() - 1);
		}

	Pixel color() const
		{
#if 0
		// based on
		constexpr u16 color_ct = 7;
		static const std::array<Pixel,color_ct> colors = {
			convert(255, 195, 0, 255),
			convert(255, 162, 0, 255),
			convert(255, 130, 0, 255),
			convert(255, 97, 0, 255),
			convert(192, 74, 0, 255),
			convert(128, 49, 0, 255),
			convert(50, 25, 0, 255),
			};

		if (age > old)
			return {0,0,0,0};
		else
			return colors[age / old];
#else
		if (age >= old)
			return {0,0,0,0};
		else
			return convert(0, 0, 255 - age, 255);
#endif
		}

	void draw() const
		{
		if (age >= old) return;
		assert(pts.size() >= 2);
		const Pixel col = color();
		for (int i = 1; i < pts.size(); ++i)
			{
			fb.line(pts[i - 1], pts[i], col);
			}
		}
	};

struct Ball
	{
	const float radius { 13.0f };
	Vec2 pos;
	Vec2 delta;
	};

Ball balls[2]; // always 2

void init_balls()
	{
	for (auto& ball : balls)
		{
		ball.pos.x = rnd(fb.width);
		ball.pos.y = rnd(fb.height);
		ball.delta = rnd_unit();
		}
	}

const Pixel ball_col = convert(200, 200, 200, 255);



void move_balls()
	{
	constexpr float ball_grav = 0.035f;
	// TODO: repel during lightning

	// bounce off each other
	const float touch = balls[0].radius + balls[1].radius;
	const float touch2 = touch * touch;
	const Vec2 diff = balls[0].pos - balls[1].pos;
	if (diff.mag2() < touch2)
		{
		// change direction, keep magnitude
		balls[0].delta = balls[0].delta.mag() * normalize(diff);
		balls[1].delta = balls[1].delta.mag() * normalize(-diff);
		}

	for (auto& ball : balls)
		{
		ball.delta.y += ball_grav;

		ball.pos += ball.delta;

		// bounce off walls
		if (ball.delta.x < 0 && ball.pos.x < ball.radius)
			{
			ball.delta.x = -ball.delta.x;
			}
		else if (ball.delta.x > 0 && ball.pos.x >= fb.width - ball.radius)
			{
			ball.delta.x = -ball.delta.x;
			}

		if (ball.delta.y < 0 && ball.pos.y < ball.radius)
			{
			ball.delta.y = -ball.delta.y;
			}
		else if (ball.delta.y > 0 && ball.pos.y >= fb.height - ball.radius)
			{
			ball.delta.y = -ball.delta.y;

			Vec2 boost = rnd_unit();
			boost.y = 3.5f * fabsf(boost.y);
			boost.x *= 0.1f; // boost is mostly upward
			ball.delta += boost;

			ball.pos.y = fb.height - ball.radius;
			}

		// draw ball (the slowest way possible)
		const float rad2 = ball.radius * ball.radius;
		for (float y = ball.pos.y - ball.radius; y < ball.pos.y + ball.radius; ++y)
			for (float x = ball.pos.x - ball.radius; x < ball.pos.x + ball.radius; ++x)
				{
				Vec2 raster_pos { x, y };
				if ((ball.pos - raster_pos).mag2() < rad2)
					fb.pset(raster_pos, ball_col);
				}
		}
	}

bool zap_mode = false;
std::vector<Bolt> bolts;

void strike()
{
	bolts.clear();
	zap_mode = true;
}

void step()
	{
	fb.clear();
	move_sparks();
	if (zap_mode)
		{
		for (auto& bolt : bolts)
			if (bolt.age < Bolt::old) bolt.age++;
		if (bolts.size() < Bolt::ct)
			if (bolts.empty() || bolts.back().age > 10)
				bolts.emplace_back(balls[0].pos, balls[1].pos);
		for (auto& bolt : bolts)
			bolt.draw();

		if (bolts.back().age >= Bolt::old)
			zap_mode = false;
		}
	move_balls();
	fb.blitToScreen();
	}

int main()
	{
	init();

	for (int i = 0; i < 30; ++i) {
		for (int p = rnd(40 - 14); p; --p)
			putchar(' ');
		puts("SynchroNY 2017");
		usleep(100000);
	}

	printf("\n\nREADY GO!!!\n\n");
	sleep(4);

	init_balls();

	while (true)
		{
		init_sparks();
		for (int i = 0; i < 1000; ++i)
			{
			step();
			usleep(5000);
			}
		strike();
		}

	close(f);
	}
