#include "Missile.h"

Missile::Missile()
	: AbstractGameObject()
	, m_ownerId(0)
	, m_type(Type::Invalid)
{
}

// IMPORTANT, id of -1 tells the bot that its a simulated fire
Missile::Missile(int ownerId, Missile::Type type, const Angle &rotation, const Position &position)
	: AbstractGameObject(-1, rotation, position)
	, m_ownerId(ownerId)
	, m_type(type)
{
}

Missile::Missile(const QJsonObject &jsonObject)
	: AbstractGameObject(jsonObject)
{
	m_ownerId = jsonObject.find("owner").value().toInt();
	const auto& typeString = jsonObject.find("type").value().toString();
	if (typeString == "NORMAL")
	{
		m_type = Type::Normal;
	}
	else if (typeString == "MINE")
	{
		m_type = Type::Mine;
	}
	else if (typeString == "SEEKING")
	{
		m_type = Type::Seeking;
	}
	else
	{
		qCritical() << "Unknown missile type:" << typeString;
	}
}

void Missile::step()
{
	const auto& distanceToSun = position().distanceTo(Position::sun());
	if (distanceToSun < 0.1)
	{
		setAlive(false);
		return;
	}

	const auto& velocityMagnitude = velocity().magnitude();
	if (velocityMagnitude > C::Missile::MAX_SPEED)
	{
		const auto& velocityAngle = velocity().angle();
		velocity()
			.setX(qCos(velocityAngle.asRadians()) * C::Missile::MAX_SPEED)
			.setY(qSin(velocityAngle.asRadians()) * C::Missile::MAX_SPEED);
	}

	const auto& angleToSun = position().angleTo(Position::sun());
	const auto& gravityPull = distanceToSun / 1000.;
	velocity() += Velocity(
		qCos(angleToSun.asRadians()) * gravityPull,
		qSin(angleToSun.asRadians()) * gravityPull
	);

	position() += velocity();

	if (type() == Type::Normal)
	{
		rotation() = velocity().angle();
	}

	if (energy().value() < 10)
	{
		velocity() /= Velocity(1.01, 1.01);
		return;
	}

	if (type() == Type::Mine)
	{
		velocity() += Velocity(
			qCos(rotation().asRadians()) * 0.0005,
			qSin(rotation().asRadians()) * 0.0005
		);
		energy().drain(1);
		return;
	}

	energy().drain(50);

	if (type() == Type::Normal)
	{
		velocity() += Velocity(
			qCos(rotation().asRadians()) * (energy().value() / 1000000.0),
			qSin(rotation().asRadians()) * (energy().value() / 1000000.0)
		);
	}
	else if (type() == Type::Seeking)
	{
		velocity() += Velocity(
			qCos(rotation().asRadians()) * (energy().value() / 100000.0),
			qSin(rotation().asRadians()) * (energy().value() / 100000.0)
		);
	}
}
