#pragma once

#include <cmath>

#include <QtCore>

class Angle
{
	public:
		Angle() = default;

		constexpr static Angle fromDegrees(double degrees)
		{
			return Angle(degrees, Type::Degrees);
		}

		constexpr static Angle fromRadians(double radians)
		{
			return Angle(radians, Type::Radians);
		}

		constexpr int asDegrees() const
		{
			return (m_type == Type::Degrees) ? m_value : qRadiansToDegrees(m_value);
		}

		constexpr double asRadians() const
		{
			return (m_type == Type::Radians) ? m_value : qDegreesToRadians(m_value);
		}

		constexpr bool operator==(const Angle& other) const
		{
			if (m_type == other.m_type)
			{
				return qFuzzyCompare(m_value, other.m_value);
			}
			else
			{
				return asDegrees() == other.asDegrees();
			}
		}

		constexpr bool operator!=(const Angle& other) const
		{
			return !operator==(other);
		}

		constexpr void operator+=(const Angle& other)
		{
			if (m_type == other.m_type)
			{
				m_value += other.m_value;
			}
			else if (m_type == Type::Degrees)
			{
				m_value += other.asDegrees();
			}
			else if (m_type == Type::Radians)
			{
				m_value += other.asRadians();
			}
		}

		constexpr void operator-=(const Angle& other)
		{
			if (m_type == other.m_type)
			{
				m_value -= other.m_value;
			}
			else if (m_type == Type::Degrees)
			{
				m_value -= other.asDegrees();
			}
			else if (m_type == Type::Radians)
			{
				m_value -= other.asRadians();
			}
		}

	private:
		enum class Type
		{
			Degrees,
			Radians
		};

		constexpr Angle(double value, Type type)
			: m_value(value)
			, m_type(type)
		{}

		double m_value;
		Type m_type;
};
