// Create our Mixins namespace
Game.EntityMixins = {};

// ****** Combat ******

Game.EntityMixins.Attacker = {
    name: 'Attacker',
    groupName: 'Attacker',
    init: function(template) {
        this._attackValue = template['attackValue'] || 1;
    },
	// Get composite attack value -- base + level bonus + weapon bonus
    getAttackValue: function() {
        var weaponBonus = 0, levelBonus = 0;
        if (this.hasMixin(Game.EntityMixins.Equipper)) {
            if (this.getWeapon())
                weaponBonus += this.getWeapon().getAttackValue();
            if (this.getArmor())
                weaponBonus += this.getArmor().getAttackValue();
        }
		if (this.hasMixin(Game.EntityMixins.ExperienceGainer))
			levelBonus = this.getLevel()-1;		
		
        return this._attackValue + levelBonus + weaponBonus;
    },
    increaseAttackValue: function(value) {
        value = value || 1;
        this._attackValue += value;
        Game.sendMessage(this, "You look stronger!");
    },
    attack: function(target) {
        if (!target.hasMixin('Destructible')) 
			return;
		
		var glancingChance = .05;						// Percentage chance of glancing blow; decreased by accuracy stat/perk
		var critChance = .05;							// Percentage chance of crit; increased by "big crits" stat/perk
		var stopHitChance = .02;						// Percentage change of stophit; increased by dexterity stat/perk
		var att = this.getAttackValue();
		var def = target.getDefenseValue();
		var hitcheck = att/(att+def);					// To hit, die roll must be LESS than this threshold
		glancingChance *= hitcheck;						// Scale these so the percentages are accurate assuming a hit!
		critChance *= hitcheck;
		stopHitChance *= hitcheck;
		
		// Roll to hit
		var dieroll = Math.random();
		//console.log("Hitcheck threshold is "+hitcheck+". Below "+critChance+": critical. Below "+(hitcheck-glancingChance)+": normal. Below "+hitcheck+": glancing. Below "+(1 - stopHitChance)+": miss. Above that, stop hit!");
		//console.log("Rolled: "+dieroll);
		var attMessage = 'You '+Game.randomMessage(['strike','thrust at','attack','swing at','assault','assail'])+' '+target.describeThe(false);
		var defMessage = this.describeThe(true)+' '+Game.randomMessage(['strikes','thrusts','attacks','swings','assaults','assails']);
		var damageMin,damageMax,reverseDamage=false;
		if (dieroll<critChance) { 						// critical
			damageMin = 1.5*att;
			damageMax = 3*att;
			attMessage += Game.randomMessage(['; a HUGE hit'+Game.randomMessage(['!','.'])+' ',' - CRITICAL HIT'+Game.randomMessage(['!','.'])+' ']);
			defMessage += Game.randomMessage(['; a HUGE hit'+Game.randomMessage(['!','.'])+' ',' - CRITICAL HIT'+Game.randomMessage(['!','.'])+' ']);
		} else if (dieroll < (hitcheck-glancingChance)) {	// normal
			damageMin = .5*att;
			damageMax = 1.5*att;
			attMessage += Game.randomMessage([' well; ', ' for ', ' well, for ', ' and hit: ',', landing your blow for ',' and hit for ',' and cause ',', inflicting ',' and inflict',', causing ']);
			defMessage += Game.randomMessage([' well; ', ' for ', ' well, for ', ' and hits: ',', landing a blow for ',' and hits for ',' and causes ',', inflicting ',' and inflicts',', causing ']);				
		} else if (dieroll < hitcheck) {				// glancing blow
			damageMin = 0;
			damageMax = .5*att;
			attMessage += Game.randomMessage(['; a POOR hit'+Game.randomMessage(['!','.'])+' ',' but the blow glances; ','; glancing blow - ',', landing a glancing blow for ']);
			defMessage += Game.randomMessage(['; a POOR hit'+Game.randomMessage(['!','.'])+' ',' but the blow glances; ','; glancing blow - ',', landing a glancing blow for ']);			
		} else if (dieroll < 1 - stopHitChance) {		// miss
			damageMin = 0;
			damageMax = 0;
			attMessage += Game.randomMessage([' but are parried'+Game.randomMessage(['!','.'])+' ',' but your foe blocks'+Game.randomMessage(['!','.'])+' ',' but miss'+Game.randomMessage(['!','.'])+' ', ' but the enemy dodges'+Game.randomMessage(['!','.'])+' ']);
			defMessage += Game.randomMessage([' but you parry'+Game.randomMessage(['!','.'])+' ',' but you block'+Game.randomMessage(['!','.'])+' ',' but misses'+Game.randomMessage(['!','.'])+' ', ' but you dodge'+Game.randomMessage(['!','.'])+' ']);
		} else {										// stop hit
			reverseDamage = true;
			damageMin = def * .5;
			damageMax = def;
			attMessage += Game.randomMessage([' but your foe performs a time hit causing you ', ' but your opponent times you, causing ', ' but the enemy manages a stop-hit for ']);
			defMessage += Game.randomMessage([' but you perform a time hit and cause the foe ', ' but you time the opponent, causing ', ' but you manage a stop-hit on the enemy for ']);				
		}
		
		// Roll for damage
		var damage = Math.floor(2.*Game.getRandomFloat(damageMin,damageMax));
		//console.log("Damage: "+damage);
		
		// Send the messages
		var dmgMessage="No";
		if (damage > 0)
			dmgMessage = damage + " HP";	
		attMessage+=' '+dmgMessage+' damage'+Game.randomMessage(['!','.']);
		defMessage+=' '+dmgMessage+' damage'+Game.randomMessage(['!','.']);
		Game.sendMessage(this, attMessage);
		Game.sendMessage(target, defMessage);
		
		// Hit them with the damage!
		if (damage > 0) {
			if (reverseDamage)
				this.takeDamage(target, damage);
			else
				target.takeDamage(this, damage);
		}
    },
    listeners: {
        details: function() {
            return [{key: 'attack', value: this.getAttackValue()}];
        }
    }
};

// This mixin signifies an entity can take damage and be destroyed
Game.EntityMixins.Destructible = {
    name: 'Destructible',
    init: function(template) {
        this._maxHp = template['maxHp'] || 10;
        this._hp = template['hp'] || this._maxHp;			// Allow health passed from template in case we want an entity to start with diff hp than usual max
        this._defenseValue = template['defenseValue'] || 0;
    },
    getDefenseValue: function() {
        var armourBonus = 0, levelBonus = 0;
        if (this.hasMixin(Game.EntityMixins.Equipper)) {
            if (this.getWeapon())
                armourBonus += this.getWeapon().getDefenseValue();
            if (this.getArmor())
                armourBonus += this.getArmor().getDefenseValue();
        }
		if (this.hasMixin(Game.EntityMixins.ExperienceGainer))
			levelBonus = this.getLevel()-1;
		return this._defenseValue + armourBonus + levelBonus;
    },
    getHp: function() {
        return this._hp;
    },
    getMaxHp: function() {
        return this._maxHp;
    },
    setHp: function(hp) {
        this._hp = hp;
    },
    increaseDefenseValue: function(value) {
        value = value || 1;
        this._defenseValue += value;
        Game.sendMessage(this, "You look tougher!");
    },
    increaseMaxHp: function(value) {
        value = value || 10;
        this._maxHp += value;
        this._hp += value;
        Game.sendMessage(this, "You look healthier!");
    },
    takeDamage: function(attacker, damage) {
        this._hp -= damage;
        if (this._hp <= 0) {
			this._hp = 0;
            Game.sendMessage(attacker, Game.randomMessage(['Your foe is vanquished','You killed the %s','You vanquished the %s','The %s is defeated','You destroyed the %s','Your adversary is finished','You dispatched the %s','The %s is dead','The %s is finished'])+Game.randomMessage(['!','.']), [this.getName()]);
            
            this.raiseEvent('onDeath', attacker);
            attacker.raiseEvent('onKill', this);
            this.kill();
        }
    },
    listeners: {
        onGainLevel: function() {
            // Heal the entity.
            this.setHp(this.getMaxHp());
        },
        details: function() {
            return [
                {key: 'defense', value: this.getDefenseValue()},
                {key: 'hp', value: this.getHp()}
            ];
        }
    }
};

// Main player's actor mixin
Game.EntityMixins.PlayerActor = {
    name: 'PlayerActor',
    groupName: 'Actor',
    act: function() {	// act() is called by the ROT engine/scheduler and is responsible for processing the entity's turn
        if (this._acting)
            return;
        this._acting = true;
		this.raiseEvent('onTurn');				// Handle mixins that want on-turn processing here!
        Game.refresh();							// Re-render the screen
		this.getMap().getEngine().lock();		// The engine was still running when it called act(), so we need to stop it! It gets restarted again by PlayState's handleInput function.
		if(this._autoPilotRunning){
			var e=this;
			setTimeout(e.getMap().getEngine().unlock(),200);
		}
        this._acting = false;
    },
    listeners: {
        onDeath: function(attacker) {
			Game.State.LoseState.setDeathMessage("Killed by "+attacker.describeA()+" at a depth of "+Game.levelToFeet(this.getZ())+" feet.");
            ThreeEngine.startDeathSpiral();
			Game.State.PlayState.setGameEnded(true);
			$('#threeDisplay').fadeOut(6000, function() {
				Game.switchState(Game.State.LoseState);
			});
			$("#playerHP").fadeOut(500,function(){
				$("#minimap").fadeOut(500,function(){
					$("#legend").fadeOut(500,function(){
						$("#playerStats").fadeOut(500,function(){
							$("#playerHunger").fadeOut(500,function(){
								$("#locationDescription").fadeOut(500,function(){
									$("#messagesContainer").fadeOut(500);
								});
							});
						});
					});
				});
			});
        }
    }	
};

Game.EntityMixins.FungusActor = {
    name: 'FungusActor',
    groupName: 'Actor',
    init: function() {
        this._growthsRemaining = 10;
    },
    act: function() { 
        // Check if we are going to try growing this turn
        if (this._growthsRemaining > 0) {
            if (Math.random() <= 0.05) {
                var xOffset = Math.floor(Math.random() * 3) - 1;
                var yOffset = Math.floor(Math.random() * 3) - 1;
                if (xOffset != 0 || yOffset != 0) {
                    if (this.getMap().isEmptyFloor(this.getX() + xOffset, this.getY() + yOffset, this.getZ())) {
                        var entity = Game.EntityRepository.create('slime mold');
                        entity.setPosition(this.getX() + xOffset, this.getY() + yOffset, this.getZ());
                        this.getMap().addEntity(entity);
                        this._growthsRemaining--;
                        // Send a message nearby!
                        Game.sendMessageNearby(this.getMap(), entity.getX(), entity.getY(), entity.getZ(), 'The smile mold is spreading!');
                    }
                }
            }
        }
    }
};

Game.EntityMixins.TaskActor = {
    name: 'TaskActor',
    groupName: 'Actor',
    init: function(template) {
        this._tasks = template['tasks'] || ['wander']; 
    },
    act: function() {							// act() is called by the ROT engine/scheduler and is responsible for processing the entity's turn
		this.raiseEvent('onTurn');				// Handle mixins that want on-turn processing here!
        for (var i = 0; i < this._tasks.length; i++) {
            if (this.canDoTask(this._tasks[i])) {	// If we can do it... do it!
                this[this._tasks[i]]();
                return;
            }
        }
    },
    canDoTask: function(task) {
        if (task === 'hunt') {
            return this.hasMixin('Sight') && this.canSee(this.getMap().getPlayer());
        } else if (task === 'wander') {
            return true;
        } else {
            throw new Error('Tried to perform undefined task ' + task);
        }
    },
    hunt: function() {
        var player = this.getMap().getPlayer();

        // If we are adjacent to the player, then attack instead of hunting.
        var xoff = Math.abs(player.getX() - this.getX()), yoff = Math.abs(player.getY() - this.getY());
		if ((xoff == 1 && yoff == 0) || (xoff == 0 && yoff == 1) || (xoff == 1 && yoff == 1))
            if (this.hasMixin('Attacker')) {
                this.attack(player);
                return;
            }

        // Generate the path and move to the first tile.
        var source = this;
        var z = source.getZ();
        var path = new ROT.Path.AStar(player.getX(), player.getY(), function(x, y) {
            // If an entity is present at the tile, can't move there.
            var entity = source.getMap().getEntityAt(x, y, z);
            if (entity && entity !== player && entity !== source) {
                return false;
            }
            return source.getMap().getTile(x, y, z).isWalkable();
        }, {topology: 8});
        // Once we've gotten the path, we want to move to the second cell that is
        // passed in the callback (the first is the entity's strting point)
        var count = 0;
        path.compute(source.getX(), source.getY(), function(x, y) {
            if (count == 1) {
                source.tryMove(x, y, z);
            }
            count++;
        });
    },
    wander: function() {
        // Flip coin to determine if moving by 1 in the positive or negative direction
        var moveOffset = (Math.round(Math.random()) === 1) ? 1 : -1;
        // Flip coin to determine if moving in x direction or y direction
        if (Math.round(Math.random()) === 1) {
            this.tryMove(this.getX() + moveOffset, this.getY(), this.getZ());
        } else {
            this.tryMove(this.getX(), this.getY() + moveOffset, this.getZ());
        }
    }
};

Game.EntityMixins.GiantZombieActor = Game.extend(Game.EntityMixins.TaskActor, {
    init: function(template) {
        // Call the task actor init with the right tasks.
        Game.EntityMixins.TaskActor.init.call(this, Game.extend(template, {
            'tasks' : ['growArm', 'spawnSlime', 'hunt', 'wander']
        }));
        // We only want to grow the arm once.
        this._hasGrownArm = false;
    },
    canDoTask: function(task) {
        // If we haven't already grown arm and HP <= 20, then we can grow.
        if (task === 'growArm') {
            return this.getHp() <= 20 && !this._hasGrownArm;
        // Spawn a slime only a 10% of turns.
        } else if (task === 'spawnSlime') {
            return Math.round(Math.random() * 100) <= 10;
        // Call parent canDoTask
        } else {
            return Game.EntityMixins.TaskActor.canDoTask.call(this, task);
        }
    },
    growArm: function() {
        this._hasGrownArm = true;
        this.increaseAttackValue(5);
        // Send a message saying the zombie grew an arm.
        Game.sendMessageNearby(this.getMap(),
            this.getX(), this.getY(), this.getZ(),
            'An extra arm appears on the giant zombie!');
    },
    spawnSlime: function() {
        // Generate a random position nearby.
        var xOffset = Math.floor(Math.random() * 3) - 1;
        var yOffset = Math.floor(Math.random() * 3) - 1;

        // Check if we can spawn an entity at that position.
        if (!this.getMap().isEmptyFloor(this.getX() + xOffset, this.getY() + yOffset,
            this.getZ())) {
            // If we cant, do nothing
            return;
        }
        // Create the entity
        var slime = Game.EntityRepository.create('Fungleater');
        slime.setX(this.getX() + xOffset);
        slime.setY(this.getY() + yOffset)
        slime.setZ(this.getZ());
        this.getMap().addEntity(slime);
    },
    listeners: {
        onDeath: function(attacker) {
            // Switch to win state when killed!
            Game.switchState(Game.State.WinState);
        }
    }
});

Game.EntityMixins.MessageRecipient = {
    name: 'MessageRecipient',
    init: function(template) {
        this._messages = [];
    },
    receiveMessage: function(message) {
        this._messages.push(message);
    },
    getMessages: function() {
        return this._messages;
    },
    clearMessages: function() {
        this._messages = [];
    }
};

// This signifies our entity posseses a field of vision of a given radius.
Game.EntityMixins.Sight = {
    name: 'Sight',
    groupName: 'Sight',
    init: function(template) {
        this._sightRadius = template['sightRadius'] || 5;
    },
    getSightRadius: function() {
        return this._sightRadius;
    },
    increaseSightRadius: function(value) {
        // If no value was passed, default to 1.
        value = value || 1;
        // Add to sight radius.
        this._sightRadius += 1;
        Game.sendMessage(this, "You are more aware of your surroundings!");
    },
    canSee: function(entity) {
        // If not on the same map or on different floors, then exit early
        if (!entity || this._map !== entity.getMap() || this._z !== entity.getZ()) {
            return false;
        }

        var otherX = entity.getX();
        var otherY = entity.getY();

        // If we're not in a square field of view, then we won't be in a real
        // field of view either.
        if ((otherX - this._x) * (otherX - this._x) +
            (otherY - this._y) * (otherY - this._y) >
            this._sightRadius * this._sightRadius) {
            return false;
        }

        // Compute the FOV and check if the coordinates are in there.
        var found = false;
        this.getMap().getFov(this.getZ()).compute(
            this.getX(), this.getY(), 
            this.getSightRadius(), 
            function(x, y, radius, visibility) {
                if (x === otherX && y === otherY) {
                    found = true;
                }
            });
        return found;
    }
};

// Message sending functions
Game.sendMessage = function(recipient, message, args) {
    // Make sure the recipient can receive the message 
    // before doing any work.
    if (recipient.hasMixin(Game.EntityMixins.MessageRecipient)) {
        // If args were passed, then we format the message, else
        // no formatting is necessary
        if (args) {
            message = vsprintf(message, args);
        }
        recipient.receiveMessage(message);
    }
};
Game.sendMessageNearby = function(map, centerX, centerY, centerZ, message, args) {
    // If args were passed, then we format the message, else
    // no formatting is necessary
    if (args) {
        message = vsprintf(message, args);
    }
    // Get the nearby entities
    entities = map.getEntitiesWithinRadius(centerX, centerY, centerZ, 5);
    // Iterate through nearby entities, sending the message if
    // they can receive it.
    for (var i = 0; i < entities.length; i++) {
        if (entities[i].hasMixin(Game.EntityMixins.MessageRecipient)) {
            entities[i].receiveMessage(message);
        }
    }
};

Game.EntityMixins.InventoryHolder = {
    name: 'InventoryHolder',
    init: function(template) {
        // Default to 10 inventory slots.
        var inventorySlots = template['inventorySlots'] || 10;
        // Set up an empty inventory.
        this._items = new Array(inventorySlots);
    },
    getItems: function() {
        return this._items;
    },
    getItem: function(i) {
        return this._items[i];
    },
    addItem: function(item) {
        // Try to find a slot, returning true only if we could add the item.
        for (var i = 0; i < this._items.length; i++) {
            if (!this._items[i]) {
                this._items[i] = item;
                return true;
            }
        }
        return false;
    },
    removeItem: function(i) {
        // If we can equip items, then make sure we unequip the item we are removing.
        if (this._items[i] && this.hasMixin(Game.EntityMixins.Equipper)) {
            this.unequip(this._items[i]);
        }
        // Simply clear the inventory slot.
        this._items[i] = null;
    },
    canAddItem: function() {
        // Check if we have an empty slot.
        for (var i = 0; i < this._items.length; i++) {
            if (!this._items[i]) {
                return true;
            }
        }
        return false;
    },
    pickupItems: function(indices) {
        // Allows the user to pick up items from the map, where indices is
        // the indices for the array returned by map.getItemsAt
        var mapItems = this._map.getItemsAt(this.getX(), this.getY(), this.getZ());
        var added = 0;
        // Iterate through all indices.
        for (var i = 0; i < indices.length; i++) {
            // Try to add the item. If our inventory is not full, then splice the 
            // item out of the list of items. In order to fetch the right item, we
            // have to offset the number of items already added.
            if (this.addItem(mapItems[indices[i]  - added])) {
                mapItems.splice(indices[i] - added, 1);
                added++;
            } else {
                // Inventory is full
                break;
            }
        }
        // Update the map items
        this._map.setItemsAt(this.getX(), this.getY(), this.getZ(), mapItems);
        // Return true only if we added all items
        return added === indices.length;
    },
    dropItem: function(i) {
        // Drops an item to the current map tile
        if (this._items[i]) {
            if (this._map) {
                this._map.addItem(this.getX(), this.getY(), this.getZ(), this._items[i]);
            }
            this.removeItem(i);      
        }
    }
};

Game.EntityMixins.FoodConsumer = {
    name: 'FoodConsumer',
    init: function(template) {
        this._maxFullness = template['maxFullness'] || 1000;
        this._fullness = template['fullness'] || (this._maxFullness / 2);
        this._fullnessDepletionRate = template['fullnessDepletionRate'] || 1;
    },
    modifyFullnessBy: function(points) {
        this._fullness = this._fullness + points;
        if (this._fullness <= 0)
            this.kill("You have died of starvation!");
        else if (this._fullness > this._maxFullness)
            this.kill("You choke and die!");
    },
	getHungerPercent: function() {
		return this._fullness / this._maxFullness * 100;
	},
    getHungerState: function() {
        var perPercent = this._maxFullness / 100;
		
        if (this._fullness <= perPercent * 5)
            return '<span class="stathighlight">Starving</span>';
        else if (this._fullness <= perPercent * 25)
            return '<span class="stathighlight">Hungry</span>';
        else if (this._fullness <= perPercent * 50)
            return 'Somewhat peckish';
        else if (this._fullness <= perPercent * 60)
            return 'Could nibble';
        else if (this._fullness <= perPercent * 75)
			return 'Not hungry';
		else if (this._fullness <= perPercent * 95)
            return 'Full';
        else
            return '<span class="stathighlight">Oversatiated</span>';
    },
    listeners: {
        onTurn: function() {
            this.modifyFullnessBy(-this._fullnessDepletionRate);
        }
    }
};

Game.EntityMixins.Recoverer = {
    name: 'Recoverer',
    init: function(template) {
        this._healthRecoveryRate = template['healthRecoveryRate'] || .1;
    },
    listeners: {
        onTurn: function() {
            this.setHp(Math.min(this.getMaxHp(),this.getHp()+this._healthRecoveryRate));
        }
    }	
};

Game.EntityMixins.CorpseDropper = {
    name: 'CorpseDropper',
    init: function(template) {
        this._corpseDropRate = template['corpseDropRate'] || 75;	// Percent chance of dropping
    },
    listeners: {
        onDeath: function(attacker) {
            if (Math.round(Math.random() * 100) <= this._corpseDropRate) {
                this._map.addItem(this.getX(), this.getY(), this.getZ(),
                    Game.ItemRepository.create('corpse', {
                        name: this._name + ' corpse',
                        foreground: this._foreground
                    }));
            }    
        }
    }
};

Game.EntityMixins.Equipper = {
    name: 'Equipper',
    init: function(template) {
        this._weapon = null;
        this._armor = null;
    },
    wield: function(item) {
        this._weapon = item;
    },
    unwield: function() {
        this._weapon = null;
    },
    wear: function(item) {
        this._armor = item;
    },
    takeOff: function() {
        this._armor = null;
    },
    getWeapon: function() {
        return this._weapon;
    },
    getArmor: function() {
        return this._armor;
    },
    unequip: function(item) {
        if (this._weapon === item)
            this.unwield();
			
        if (this._armor === item)
            this.takeOff();
    }
};

Game.EntityMixins.ExperienceGainer = {
    name: 'ExperienceGainer',
    init: function(template) {
        this._level = template['level'] || 1;
        this._experience = template['experience'] || 0;
        this._statPointsPerLevel = template['statPointsPerLevel'] || 1;
        this._statPoints = 0;
        // Determine what stats can be levelled up.
        this._statOptions = [];
        if (this.hasMixin('Attacker')) {
            this._statOptions.push(['Increase attack value', this.increaseAttackValue]);
        }
        if (this.hasMixin('Destructible')) {
            this._statOptions.push(['Increase defense value', this.increaseDefenseValue]);   
            this._statOptions.push(['Increase max health', this.increaseMaxHp]);
        }
        if (this.hasMixin('Sight')) {
            this._statOptions.push(['Increase sight range', this.increaseSightRadius]);
        }
    },
    getLevel: function() {
        return this._level;
    },
    getExperience: function() {
        return this._experience;
    },
    getNextLevelExperience: function() {
        return (this._level * this._level) * 10;
    },
    getStatPoints: function() {
        return this._statPoints;
    },
    setStatPoints: function(statPoints) {
        this._statPoints = statPoints;
    },
    getStatOptions: function() {
        return this._statOptions;
    },
    giveExperience: function(points) {
        var statPointsGained = 0;
        var levelsGained = 0;
        // Loop until we've allocated all points.
        while (points > 0) {
            // Check if adding in the points will surpass the level threshold.
            if (this._experience + points >= this.getNextLevelExperience()) {
                // Fill our experience till the next threshold.
                var usedPoints = this.getNextLevelExperience() - this._experience;
                points -= usedPoints;
                this._experience += usedPoints;
                // Level up our entity!
                this._level++;
                levelsGained++;
                this._statPoints += this._statPointsPerLevel;
                statPointsGained += this._statPointsPerLevel;
            } else {
                // Simple case - just give the experience.
                this._experience += points;
                points = 0;
            }
        }
        // Check if we gained at least one level.
        if (levelsGained > 0) {
            Game.sendMessage(this, "You have reached the rank of %s.", [Game.RankNames[this._level-1]]);
            this.raiseEvent('onGainLevel');
			Game.sendMessageNearby(this.getMap(), this.getX(), this.getY(), this.getZ(), this.describeThe(true)+' leveled up'+Game.randomMessage(['!','.']));			
        }
    },
    listeners: {
        onKill: function(victim) {
            var exp = victim.getMaxHp() + victim.getDefenseValue();
            if (victim.hasMixin('Attacker'))
                exp += victim.getAttackValue();
            
            if (victim.hasMixin('ExperienceGainer'))					// Account for level differences
                exp -= (this.getLevel() - victim.getLevel()) * 3;
				
            if (exp > 0)
                this.giveExperience(exp);
        },
        details: function() {
            return [{key: 'level', value: this.getLevel()}];
        }
    }
};

Game.EntityMixins.RandomStatGainer = {
    name: 'RandomStatGainer',
    groupName: 'StatGainer',
    listeners: {
        onGainLevel: function() {
            var statOptions = this.getStatOptions();
            while (this.getStatPoints() > 0) {
                statOptions.random()[1].call(this);
                this.setStatPoints(this.getStatPoints() - 1);
            }
        }
    }
};

Game.EntityMixins.PlayerStatGainer = {
    name: 'PlayerStatGainer',
    groupName: 'StatGainer',
    listeners: {
        onGainLevel: function() {
			Game.refresh();
            Game.State.GainStatState.setup(this);
			Game.State.PlayState.setSubState(Game.State.GainStatState);
        }
    }
};

Game.EntityMixins.Autopilotable = {
    name: 'Autopilotable',
    init: function(template) {
        this._targetX = null;
		this._targetY = null;
		this._path = null;
        this._autoPilotRunning = false;
    },
	setAutopilot: function(x,y) {
		this._targetX=x;
		this._targetY=y;
		
		var ent=this,z=this.getZ();
		this._path = new ROT.Path.AStar(this._targetX, this._targetY, function(x, y) {	// If an entity is present at the tile, can't move there.
            var entity = ent.getMap().getEntityAt(x, y, z);
            if (entity && entity !== ent)
                return false;
            return ent.getMap().getTile(x, y, z).isWalkable();
        }, {topology: 8});

		this._autoPilotRunning = true;
	},
    listeners: {
        onTurn: function() {
			if (!this._autoPilotRunning)
				return;
			
			var i=0, ent=this;
			this._path.compute(this.getX(), this.getY(), function(x, y) {
				if (i==1) {
					ent.tryMove(x, y, ent.getZ());
					if(x==ent._targetX && y==ent._targetY)
						ent._autoPilotRunning = false;
				}
				i++;
			});
        }
    }
};