var ThreeEngine = {
	_scene: null,
	_camera: null,
	_container: null,
	_composer: null,
	_renderer: null,
	_controls: null,
	_stats: null,
	_fovTex: null,
	_tex: null,
	_clock: null,
	_deathSpiral: null,
	_effectRGBShift: null,
	_effectSepia: null,
	_effectDamageFlash: null,
	_damageFlashStart: null,
	_torchNoise: null,
	_mat: null,
	_floor: null,
	_planeWidth: null,
	_planeHeight: null,
	
	init: function() {
		_planeWidth=720;
		_planeHeight=375;
		//_planeWidth=1100;
		//_planeHeight=2040;
		
		_scene = new THREE.Scene();
		
		_deathSpiral = false;
		_damageFlashStart = 0;
		_torchNoise = new ROT.Noise.Simplex();
		
		var SCREEN_WIDTH = window.innerWidth - 10, SCREEN_HEIGHT = window.innerHeight - 10;
		var VIEW_ANGLE = 45, ASPECT = SCREEN_WIDTH / SCREEN_HEIGHT, NEAR = 0.1, FAR = 20000;
		
		_camera = new THREE.PerspectiveCamera(VIEW_ANGLE, ASPECT, NEAR, FAR);
		_scene.add(_camera);
		
		if (Detector.webgl)
			_renderer = new THREE.WebGLRenderer({antialias:true});
		else
			_renderer = new THREE.CanvasRenderer(); 
		_renderer.setClearColor(0x000000,0);
		_renderer.setSize(SCREEN_WIDTH, SCREEN_HEIGHT);
		
		_container = document.getElementById('threeDisplay');
		_container.appendChild(_renderer.domElement);
		
		_controls = new THREE.DungeonControls(_camera, _renderer.domElement);
		_controls.maxPolarAngle = Math.PI/2 - 0.7;
		
		_clock = new THREE.Clock();
		
		_stats = new Stats();
		_stats.domElement.style.position = 'absolute';
		_stats.domElement.style.top = '0px';
		_stats.domElement.style.right = '0px';
		_stats.domElement.style.zIndex = 100;
		_stats.domElement.style.display = 'none';
		_container.appendChild(_stats.domElement);
		
		// Materials and geometry
		_fovTex = new THREE.Texture(RotEngine.getDisplayFOV().getContainer());
		_fovTex.needsUpdate = true;
		_fovTex.minFilter = THREE.LinearFilter;
		
		_tex = new THREE.Texture(RotEngine.getDisplay().getContainer());
		_tex.needsUpdate = true;
		_tex.minFilter = THREE.LinearFilter;
		
		_mat = new THREE.ShaderMaterial({
			uniforms: {
				diffuse: {type: "t", value: _tex},
				multimap: {type: "t", value: _fovTex},
				flicker: {type: "f", value: 1},
			},
			vertexShader: THREE.DungeonShader.vertexShader,
			fragmentShader: THREE.DungeonShader.fragmentShader,
		});
		
		var geo = new THREE.PlaneBufferGeometry(_planeWidth, _planeHeight, 1, 1);

		var layerStep = 10., currY = 0;
		for (var i=0; i<7; i++){
			var mesh = new THREE.Mesh(geo, _mat);
			if(i==0)
				_floor = mesh;
			mesh.position.y = currY;
			mesh.rotation.x = -Math.PI / 2;
			_scene.add(mesh);
			currY += layerStep;
		}
		
		// Postprocessing
		_renderer.autoClear = false;
				
		_composer = new THREE.EffectComposer(_renderer);
		_composer.addPass(new THREE.RenderPass(_scene, _camera));
		
		//var effectFilm = new THREE.FilmPass(.42, 0, 0, false);
		//_composer.addPass(effectFilm);
		
		var effectVignette = new THREE.ShaderPass(THREE.VignetteShader);		
		effectVignette.uniforms["offset"].value = .95;
		effectVignette.uniforms["darkness"].value = 2.;
		_composer.addPass(effectVignette);
		
		var effectBloom = new THREE.BloomPass(.6);
		_composer.addPass(effectBloom);
				
		_effectRGBShift = new THREE.ShaderPass(THREE.RGBShiftShader);
		_effectRGBShift.uniforms['amount'].value = 0;
		_composer.addPass(_effectRGBShift);
		
		_effectSepia = new THREE.ShaderPass(THREE.SepiaShader);
		_effectSepia.uniforms['amount'].value = 0;
		_composer.addPass(_effectSepia);
		
		_effectDamageFlash = new THREE.ShaderPass(THREE.DamageFlashShader);
		_composer.addPass(_effectDamageFlash);
		
		var effectCopy = new THREE.ShaderPass(THREE.CopyShader);
		effectCopy.renderToScreen = true;
		_composer.addPass(effectCopy);
		
		window.addEventListener('resize', ThreeEngine.onWindowResize, false);

		this.resetCamera();
		this.tick();
	},
	
	toggleStatsDisplay: function() {
		if(_stats.domElement.style.display=="none")
			_stats.domElement.style.display="block";
		else
			_stats.domElement.style.display="none";
	},
	
	onWindowResize: function () {
		_camera.aspect = (window.innerWidth - 10) / (window.innerHeight - 10);
		_camera.updateProjectionMatrix();

		_renderer.setSize(window.innerWidth - 10, window.innerHeight - 10);
		_composer.reset();
	},
	
	startDeathSpiral: function() {
		_deathSpiral = true;
		_controls.controlsEnabled = false;
	},
	
	startDamageFlash: function() {
		_damageFlashStart = _clock.getElapsedTime();
	},

	setEffects: function(nutrition, health) {	
		var n=Math.max(0,(25-nutrition)*0.00025);
		var h=Math.min(1,Math.max(0,(25-health)*0.04));
		
		_effectRGBShift.enabled = (n<25)?true:false;
		_effectRGBShift.uniforms['amount'].value = n;
		
		_effectSepia.enabled = (h<25)?true:false;
		_effectSepia.uniforms['amount'].value = h;
	},
	
	resetCamera: function(distance) {
		_deathSpiral = false;
		_controls.controlsEnabled = true;	
		if (distance===undefined)
			distance=500;
		_camera.position.set(0,distance,0);
		_camera.rotation.x=Math.PI/2;
		_camera.rotation.y=Math.PI;
		_camera.rotation.z=0;
	},
	
	doPicking: function(mouseX,mouseY){
        var mouse3D = new THREE.Vector3((mouseX/window.innerWidth)*2 - 1, -(mouseY/window.innerHeight)*2 + 1, .5);
		var raycaster = new THREE.Raycaster();
		raycaster.setFromCamera(mouse3D.clone(),_camera);
		var intersections = raycaster.intersectObject(_floor);
        if (intersections.length > 0) {
			var x=intersections[0].point.x, y=intersections[0].point.z;
			
			x += _planeWidth/2;				// make positive
			x /= _planeWidth;				// scale to [0,1]
			x *= RotEngine.getConsoleWidth();		// scale to eg 80x25 to get cell
			x = Math.floor(x);
			
			y += _planeHeight/2;
			y /= _planeHeight;
			y *= RotEngine.getConsoleHeight();
			y = Math.floor(y);
			
			var offsets = Game.State.PlayState.getScreenOffsets();
			x += offsets.x, y += offsets.y+1;	// Apply offset to find world cell coords of that screen cell coord

			var player=Game.State.PlayState.getPlayer();
			player.setAutopilot(x,y);		// Set target...
			player.getMap().getEngine().unlock(); // ... aaaand GO!!!
		}
	},
	
	tick: function () {
		if (Game.getCurrentState()!=Game.State.PlayState)
			_effectSepia.enabled = _effectRGBShift.enabled = _deathSpiral = false;
			
		requestAnimationFrame(ThreeEngine.tick);
		
		_fovTex.needsUpdate = true;
		_tex.needsUpdate = true;
		
		var delta=_clock.getDelta();
		if(_deathSpiral){
			_controls.zoomIn(1.005);
			_controls.rotateLeft(delta*.4);
			_controls.rotateDown(delta*.05);
		}
		
		_mat.uniforms.flicker.value = (_torchNoise.get(_clock.getElapsedTime()*5.,0) + 1)/2;
		
		var t=_clock.getElapsedTime(),length=.05,damageFlashEnd=_damageFlashStart+length;
		_effectDamageFlash.enabled = (t<damageFlashEnd)?true:false;
		_effectDamageFlash.uniforms['amount'].value = Math.sin((t-_damageFlashStart)/length * Math.PI/2);
		
		_composer.render(delta);
		_controls.update();
		_stats.update();
	}
};