function animate(keyframes, time)
{
  var last = keyframes.length - 1;
  if (time <= keyframes[0][0]) return keyframes[0][1];
  if (time >= keyframes[last][0]) return keyframes[last][1];
  
  // we must have at least 2 keyframes, or it will crash
  var prev = [], next = [];
  for (var i = 0; i < keyframes[1][1].length; i++) {
    prev.push(2 * keyframes[0][1][i] - keyframes[1][1][i]);
    next.push(2 * keyframes[last][1][i] - keyframes[last-1][1][i]);
  }

  keyframes.push([2 * keyframes[last][0] - keyframes[last-1][0], next]);
  keyframes.unshift([keyframes[0][0] - keyframes[1][0], prev]);
  
  var i = 1;
  while ((i <= last) && (keyframes[i][0] < time)) i++;
  
  var k0 = keyframes[i - 2];
  var k1 = keyframes[i - 1];
  var k2 = keyframes[i];
  var k3 = keyframes[i + 1];
  
  var t = (time - k1[0]) / (k2[0] - k1[0]);
  
  var h1 = 2 * t * t * t - 3 * t * t + 1;          // calculate basis function 1
  var h2 = -2 * t * t * t + 3 * t * t;              // calculate basis function 2
  var h3 = t * t * t - 2 * t * t + t;         // calculate basis function 3
  var h4 = t * t * t - t * t;
  
  var out = []
  for (var i = 0; i < k1[1].length; i++) {
    var t1 = (k2[1][i] - k0[1][i]) / 4;
    var t2 = (k3[1][i] - k1[1][i]) / 4;
    out.push(h1 * k1[1][i] + h2 * k2[1][i] + h3 * t1 + h4 * t2);
  }
  return out;
}
var scenes = [];
var start_time = 0;
var snd;
var M = Math;


function minify_context(ctx)
{
  var names = []
  for (var name in ctx) names.push(name);
  names.sort();
  
  for (var i in names)
  {
    var name = names[i]
    
    
    var m, newName = "";
    var re = /([A-Z0-9])[A-Z]*_?/g;
    if (name.match(/[a-z]/))
      re = /(^[a-z]|[A-Z0-9])[a-z]*/g;
    while (m = re.exec(name)) newName += m[1];
    
    // add an underscore to shader variables, to avoid conflict with glsl-unit minification
    
    if (newName in ctx)
    {
      var index = 2;
      while ((newName + index) in ctx) index++;
      newName = newName + index;
    }
    
    ctx[newName] = ctx[name];
    
  }
}

// export for minifcation tools

function engine_render(current_time)
{
  var start_time = 0;
  for (var i = 0; i < scenes.length; i++) {
    var scene = scenes[i]
    var scene_time = current_time - start_time;
    if ((scene_time >= 0) && (scene_time < scene.duration)) {
      render_scene(scene, current_time, scene_time);
      break;
    }

    start_time += scene.duration;
  }
}

function main_loop() {
  var current_time = snd.t();
  engine_render(current_time);
  requestAnimationFrame(main_loop);
}

function main() {
  var body = document.body
  body.innerHTML = "";
  canvas = document.createElement("canvas");
  body.appendChild(canvas);
  body.style.margin = 0;

  canvas.width = innerWidth;
  canvas.height = innerHeight;

  gl_init();
  text_init();
  demo_init();

  load_shaders();
  load_geometries();
  load_textures();

  gfx_init();

  
  render_scene(scenes[0], 0, 0);

  //snd = new SND(SONG);
  snd = new SND_Ogg("tdf15_master3")
  // If you want to shut the music up comment this out and also comment
  // out the equivalent line in engine-driver.js:~100
  snd.p();

  main_loop();
}

function editor_main() {
  canvas = document.getElementById("engine-view")
  gl_init();
  text_init();
}

// general naming rule: things that have offset in the name are offsets in
// an array, while things with index in the name are indices that should be
// multiplied by a stride to obtain the offset.

// ring: [[x,y,z]]
// geom: {vbo, ibo, v_stride, v_cursor, i_cursor}
// v_cursor is an index (in vertex, not an offset in the array).
// Use v_cursor * v_stride for an offset in the array.

var SEED = 1;
function seedable_random() {
    return (SEED = (69069 * SEED + 1) & 0x7FFFFFFF) / 0x80000000;
}

function mid_point(a, b) {
    return [
        (a[0]+b[0])/2,
        (a[1]+b[1])/2,
        (a[2]+b[2])/2
    ];
}

function get_vec3(buffer, offset) {
    return [
        buffer[offset],
        buffer[offset+1],
        buffer[offset+2]
    ];
}

//      c
//     / \
//    /   \
//  ac --- bc
//  / \   / \
// /   \ /   \
//a-----ab----b

function subdivide(prev_buffer) {
    var output = [];
    for (var i=0; i<prev_buffer.length; i+=9) {
        var a = get_vec3(prev_buffer, i);
        var b = get_vec3(prev_buffer, i+3);
        var c = get_vec3(prev_buffer, i+6);
        var ab = mid_point(a, b);
        var bc = mid_point(b, c);
        var ac = mid_point(a, c);
        pack_vertices(output,[
            a,  ab, ac,
            ac, ab, bc,
            bc, ab, b,
            ac, bc, c
        ]);
    }
    return output;
}

//  a          b          c           d
// (1, 1, 1), (1,-1,-1), (-1, 1,-1), (-1,-1, 1)
function make_tetrahedron() {
    return [
         1, 1, 1,   1,-1,-1,  -1, 1,-1,  // abc
        -1,-1, 1,   1,-1,-1,   1, 1, 1,  // dba
        -1,-1, 1,  -1, 1,-1,   1,-1,-1,  // dcb
        -1,-1, 1,   1, 1, 1,  -1, 1,-1   // dac
    ];
}

function make_sphere(radius, num_subdivs) {
    var buffer = make_tetrahedron();
    while (num_subdivs-- > 0) {
        buffer = subdivide(buffer);
    }
    for (var i = 0; i < buffer.length; i+=3) {
        var len = vec3.length([buffer[i], buffer[i+1], buffer[i+2]]);
        buffer[i] *= radius/len;
        buffer[i+1] *= radius/len;
        buffer[i+2] *= radius/len;
    }
    return buffer;
}

// TODO: it's sorta convenient to have this for prototyping but I assume we'll
// have to not use this in the shipping demos and always generate from unpacked
// geometry rather than packing and unpacking to re-pack afterwards like
// map_triangles does.
// turns [a, b, c, d, e, f, g, h, i] into [[a, b, c], [d, e, f], [g, h, i]]
function unpack_vertices(vertices, offset, num_vertices) {
    var output = [];
    for(var i = offset; i < offset+num_vertices*3; i+=3) {
        output.push([vertices[i], vertices[i+1], vertices[i+2]]);
    }
    return output;
}

function map_triangles(positions, fn) {
    var output = [];
    for (var i = 0; i < positions.length; i+=9) {
        pack_vertices(output, fn(unpack_vertices(positions, i, 3), i));
    }
    return output;
}

// triangle: unpacked vertices [[x, y, z], [x, y, z], [x, y, z]]
function flat_normals(triangle) {
    var a = triangle[0];
    var b = triangle[1];
    var c = triangle[2];
    var ab = vec3.create();
    var ac = vec3.create();
    var normal = vec3.create();
    vec3.sub(ab, b, a);
    vec3.sub(ac, c, a);
    vec3.cross(normal, ab, ac);
    vec3.normalize(normal, normal);
    return [normal, normal, normal];
}

function triangle_index(triangle, i) {
    return [[i],[i],[i]];
}

function translate(dx, dy, dz) {
    var identity = mat4.create();
    return mat4.translate(identity, identity, [dx, dy, dz]);
}
function rotate_x(angle) {
    var identity = mat4.create();
    return mat4.rotate(identity, identity, angle, [1, 0, 0]);
}
function rotate_y(angle) {
    var identity = mat4.create();
    return mat4.rotate(identity, identity, angle, [0, 1, 0]);
}
function rotate_z(angle) {
    var identity = mat4.create();
    return mat4.rotate(identity, identity, angle, [0, 0, 1]);
}
function scale(sx, sy, sz) {
    var identity = mat4.create();
    return mat4.scale(identity, identity, [sx, sy, sz]);
}

function matrix_str(mat) {
    return "[ " + mat[0] + " "
                + mat[1] + " "
                + mat[2] + " "
                + mat[3] + " | "
                + mat[4] + " "
                + mat[5] + " "
                + mat[6] + " "
                + mat[7] + " | "
                + mat[8] + " "
                + mat[9] + " "
                + mat[10] + " "
                + mat[11] + " | "
                + mat[12] + " "
                + mat[13] + " "
                + mat[14] + " "
                + mat[15] + "]";
}

function vector_str(vec) {
    var vec_3 = vec[3]||"";
    return "[ " + vec[0] + " "
                + vec[1] + " "
                + vec[2] + " "
                + vec_3 + " ]";
}

function extrude_geom(geom, cmd_list) {
    var base_paths;
    var transform = mat4.create();
    var previous_paths;
    for (var i = 0; i < cmd_list.length; ++i) {
        var item = cmd_list[i];
        if (item.transform) {
            mat4.multiply(transform, transform, item.transform);
        }
        if (item.apply) {
            var transformed_paths = transform_paths(base_paths, transform);
            if (previous_paths) {
                item.apply(geom, previous_paths, transformed_paths);
            }
            previous_paths = transformed_paths;
        }
        if (item.set_path) {
            base_paths = item.set_path(base_paths);
        }
        if (item.jump) {
            i = item.jump(i);
        }
    }
}

function create_geom_from_cmd_list(commands) {
    var geom = {}

    if (asset.positions) { geom.positions = []; }
    if (asset.normals) { geom.normals = []; }
    if (asset.uvs) { geom.uvs = []; }

    extrude_geom(geom, commands);

    var buffers = [];
    if (asset.positions) { buffers.push(make_vbo(POS, geom.positions)); }
    if (asset.normals) { buffers.push(make_vbo(NORMAL, geom.normals)); }
    if (asset.uvs) { buffers.push(make_vbo(UV, geom.uvs)); }

    geometries[name] = {
      buffers: buffers,
      mode: gl.TRIANGLES,
      vertex_count: geom.positions.length / 3
    };
}

// XXX - apply_extrusion
function apply_fn(geom, previous_rings, new_rings) {
  previous_rings.forEach(
      function(prev_item, i) {
          console.log(new_rings);
          join_rings(
            geom,
            prev_item,
            new_rings[i],
            function() { return uv_buffer(0, 0, 1, 1) }
          );
      }
  );
}

function apply_fill(geom, ring) {
  var normal = [0, 1, 0];
  for (var i = 1; i < ring.length - 1; i++) {
      pack_vertices(geom.positions, [ring[0], ring[i], ring[i + 1]]);
      //pack_vertices(geom.normals, [normal, normal, normal]);
  }
}


function jump_if(pc, cond) {
    return function(i) { if (cond(i)) { return pc; } };
}

function transform_paths(path_array, transform) {
    var out_array = [];
    for (var i = 0; i < path_array.length; ++i) {
        var path = path_array[i];
        var new_path = [];
        for (var v = 0; v < path.length; ++v) {
            var vertex = vec3.fromValues(
                path[v][0],
                path[v][1],
                path[v][2]
            );
            vec3.transformMat4(vertex, vertex, transform);
            new_path.push(vertex);
        }
        out_array.push(new_path);
    }
    return out_array;
}

function uv_buffer(u1, v1, u2, v2) {
  return [[
    u1, v1,
    u2, v1,
    u2, v2,
    u2, v2,
    u1, v2,
    u1, v1
  ]];
}

// For a continuous ring of 4 points the indices are:
//    0    1
//  7 A----B 2
//    |    |
//    |    |
//  6 D----C 3
//    5    4
//
// The slice of the vbo for this ring looks like:
// [A, B, B, C, C, D, D, A]
//
// Continuous rings are what the city generator outputs, but join_rings
// takes discontinuous rings as inputs:
//
// For a discontinuous ring of 4 points the indices are:
//    0    1
//    A----B
//
//
//    C----D
//    3    2
//
// The slice of the vbo for this ring looks like:
// [A, B, C, D]

function is_path_convex(path) {
    var path_length = path.length;
    var c = vec3.create();
    var v1 = vec2.create();
    var v2 = vec2.create();
    for (var i = 0; i < path_length; ++i) {
        vec2.subtract(v1, path[(i+1)%path_length], path[i]);
        vec2.subtract(v2, path[(i+2)%path_length], path[(i+1)%path_length]);
        vec2.cross(c, v1, v2);
        if (c[2] > 0) {
            return false;
        }
    }
    return true;
}

function make_ring(path, y) {
  return path.map(function(point)
  {
    return [point[0], y, -point[1]]
  })
}

function pack_vertices(to, v) {
    for (var i = 0; i<v.length; ++i) {
        for (var j = 0; j<v[i].length; ++j) {
            to.push(v[i][j]);
        }
    }
}

function join_rings(geom, r1, r2, uv_fn) {

    var e1 = vec3.create()
    var e2 = vec3.create()
    var normal = [0,0,0]
    for (var i = 0; i < r1.length; i++)
    {
      var next = (i + 1) % r1.length;
      pack_vertices(geom.positions, [r1[i], r1[next], r2[next], r2[next], r2[i], r1[i]]);

      vec3.sub(e1, r2[next], r1[i]);
      vec3.sub(e2, r1[next], r1[i]);
      vec3.cross(normal, e1, e2);
      vec3.normalize(normal, normal);
      pack_vertices(geom.normals, [normal, normal, normal, normal, normal, normal]);
      var head_or_tail = rand_int(2) == 1 ? 0.3 : 0.5;
      pack_vertices(geom.uvs, uv_fn(vec3.length(e2), head_or_tail));
    }
}

function rand_int(max) {
    return M.floor(seedable_random() * max);
}

function mod(a, m) {
  return (a%m+m)%m;
}

// Yeah. I know.
function deep_clone(obj) {
    return JSON.parse(JSON.stringify(obj));
}

function _vector_2d(a,b) { return vec2.subtract([], b, a) }
function _vec2_scale(v, f) { return [v[0]*f, v[1]*f] }

function tangent(v) {
    var l = vec2.length(v);
    return [-v[1]/l, v[0]/l]
}

function lines_intersection_2d(a1, a2, b1, b2) {
    var det = (a1[0]-a2[0])*(b1[1]-b2[1]) - (a1[1]-a2[1])*(b1[0]-b2[0]);
    if (det*det < 0.0001) { return null }
    var a = (a1[0]*a2[1]- a1[1]*a2[0]);
    var b = (b1[0]*b2[1]- b1[1]*b2[0]);
    return [
        (a * (b1[0] - b2[0]) - b * (a1[0] - a2[0])) / det,
        (a * (b1[1] - b2[1]) - b * (a1[1] - a2[1])) / det
    ];
}

// returns a transformed ring. 
//displacements : vec3, translations along : 
//                     - normale to the ring
//                     - center to 1st vertice of the ring direction
//                     - normale to both of the above
//rotations : vec3, in radiatns, of the rotation along those same axis
//homotecies, scalar
function transform_ring ( ring_old, displacements, rotations, homotecies ){
  var i = 0;
  var center = vec3.create();
  
  var e1;
  var e2;
  var normal;
  var a1;
  
  var ring;
  
  //first, make a copy of it
  var ring = deep_clone(ring_old);
  
  //find out the normal
  vec3.sub(e1, ring[0], ring[1]);
  vec3.sub(e2, ring[0], ring[2]);
  vec3.cross(normal, e1, e2);
  vec3.normalize(normal, normal);
  
  //at first, we will have to make sure the ring is coplanar.
  
  //compute the center of the ring.
  for(i = 0; i < ring.length; i++){
    vec3.add(center, ring[i]);
  }
  vec3.divide(center,center, i);
  
  //find out the two other axis
  vec3.sub(a1, ring[0], center);
  vec3.normalize(a1, a1);
  
  vec3.cross(a2, normal, a1);
  
  
  //translate !
  var temp;
  for(i = 0; ring.length; i++){
    temp = vec3.scale(temp, normal, displacement[0]);
    vec3.add(ring[i], temp);
    temp = vec3.scale(temp, a1, displacement[1]);
    vec3.add(ring[i], temp);
    temp = vec3.scale(temp, a2, displacement[2]);
    vec3.add(ring[i], temp);
  }
  
  //rotate !
  console.log("transform_ring : rotation not yet implemented !");
  
  //homotecies 
  console.log("transform_ring : homotecies not yet implemented !");
  
  
  return ring;
  

}

function shrink_path(path, amount, z, use_subdiv, disp) {
    var new_path = [];
    var path_length = path.length;
    var pna = vec2.create();
    var pnxa = vec2.create();
    var pnb = vec2.create();
    var pnxb = vec2.create();
    for (var i = 0; i < path_length; ++i) {
        var pa = path[mod(i-1, path_length)];
        var px = path[mod(i,   path_length)];
        var pb = path[mod(i+1, path_length)];
        use_subdiv = use_subdiv || 0;
        var displacement;
        //if(disp)
        //  console.log("on a disp=" + disp);
        displacement = disp || [0,0];
        // avoid shrinking too much
        if (vec2.distance(pa, pb) < amount*(1+pa.subdiv*use_subdiv*2)) {
            return deep_clone(path);
        }
        var pa_sub = pa.subdiv || 0;
        var px_sub = px.subdiv || 0;
        var na = _vec2_scale(tangent(_vector_2d(pa, px)), amount * (1+pa_sub*use_subdiv));
        var nb = _vec2_scale(tangent(_vector_2d(px, pb)), amount * (1+px_sub*use_subdiv));

        vec2.add(pna, pa, na);
        vec2.add(pnb, pb, nb);
        vec2.add(pnxa, px, na);
        vec2.add(pnxb, px, nb);

        var inter = lines_intersection_2d(pna, pnxa, pnxb, pnb );

        // If inter is null (pa, px and pb are aligned)
        inter = inter || [pnxa[0], pnxa[1]];
        inter = vec2.add(inter, inter, displacement);
        inter.subdiv = path[i].subdiv;
        new_path.push(inter);
    }

    var old_segment = vec2.create();
    var new_segment = vec2.create();
    for (var i = 0; i < path_length; ++i) {
        vec2.subtract(old_segment, path[(i+1)%path_length], path[i]);
        vec2.subtract(new_segment, new_path[(i+1)%path_length], new_path[i]);

        if (vec2.dot(old_segment, new_segment) < 0) {
            return null;
        }
    }
    return new_path;
}

function fill_convex_ring(geom, ring, uv) {
  var normal = [0, 1, 0];
  // roof top or grass
  uv = uv || [0.5, 0.95];
  for (var i = 1; i < ring.length - 1; i++) {
      pack_vertices(geom.positions, [ring[0], ring[i], ring[i + 1]]);
      pack_vertices(geom.normals, [normal, normal, normal]);
      pack_vertices(geom.uvs, [uv, uv, uv]);
  }
}


// TODO make this show in the editor: it defines how the min size of city blocks
var MIN_PERIMETER = 260;

function city_subdivision(path, sub_id) {
    var path_length = path.length;

    // a1 is the index of the point starting the first edge we'll cut.
    // b1 is the index of the point starting the second edge we'll cut.
    var a1;
    var maxd = 0;
    var perimeter = 0;
    var i; // loop index, taken out to win a few bytes
    // pick the longest segment
    for (i = 0; i < path_length; ++i) {
        var d = vec2.distance(path[i], path[(i+1)%path_length]);
        if (d > maxd) {
            maxd = d;
            a1 = i;
        }
        perimeter += d;
    }

    if (perimeter < MIN_PERIMETER) { return null; }

    var a2 = (a1+1) % path_length;
    var b1, b2;

    do {
        b1 = rand_int(path_length);
        if (a1 == b1 || a1 == b1 + 1) { continue; }

        b2 = (b1+1) % path_length;

        var f1 = 0.5 + (0.5 - M.abs(seedable_random() - 0.5)) * 0.2;
        var f2 = 0.5 + (0.5 - M.abs(seedable_random() - 0.5)) * 0.2;

        var p_a3_1 = { '0': path[a1][0]*f1 + path[a2][0]*(1.0-f1), '1': path[a1][1]*f1 + path[a2][1]*(1-f1), subdiv: sub_id};
        var p_a3_2 = { '0': path[a1][0]*f1 + path[a2][0]*(1.0-f1), '1': path[a1][1]*f1 + path[a2][1]*(1-f1), subdiv: path[a1].subdiv};
        var p_b3_1 = { '0': path[b1][0]*f2 + path[b2][0]*(1.0-f2), '1': path[b1][1]*f2 + path[b2][1]*(1-f2), subdiv: sub_id};
        var p_b3_2 = { '0': path[b1][0]*f2 + path[b2][0]*(1.0-f2), '1': path[b1][1]*f2 + path[b2][1]*(1-f2), subdiv: path[b1].subdiv};

        break;
    } while (1);

    var path1 = [p_a3_1, p_b3_2]
    for (i = b2; i != a2; i = mod((i+1), path_length)) {
        path1.push(path[i]);
    }

    var path2 = [p_b3_1, p_a3_2]
    for (i = a2; i != b2; i = mod((i+1), path_length)) {
        path2.push(path[i]);
    }

    return [path1, path2];
}

function circle_path(center, radius, n_points) {
    var path = []
    for (i = 0; i < n_points; ++i) {
        path.push([
            center[0] + -M.cos(i/n_points * 2 * M.PI) * radius,
            center[1] + M.sin(i/n_points * 2 * M.PI) * radius
        ]);
    }
    return path;
}

function circle_path_vec3(center, radius, n_points) {
    var path = []
    for (i = 0; i < n_points; ++i) {
        path.push([
            center[0] + -M.cos(i/n_points * 2 * M.PI) * radius,
            0,
            center[1] + M.sin(i/n_points * 2 * M.PI) * radius
        ]);
    }
    return path;
}





// Testing...
// if this code below ends up in the minified export, something's wrong.

function debug_draw_path(path, color, offset_x, offset_y) {
/*    map_ctx.strokeStyle = color;
    for (var i in path) {
        map_ctx.beginPath();
        map_ctx.moveTo(
            (path[i][0] + offset_x + 300) / 3,
            (path[i][1] + offset_y) / 3
        );
        map_ctx.lineTo(
            (path[mod(i-1, path.length)][0] + offset_x + 300) / 3,
            (path[mod(i-1, path.length)][1] + offset_y) / 3
        );
        map_ctx.stroke();
        map_ctx.closePath();
    }*/
}
var gl
var canvas
var textures = {}
var uniforms = {}
var geometries = {}
var programs = {}
var fragment_shaders = {}
var vertex_shaders = {}
var ctx_2d

var use_texture_float = true;
var gl_ext_half_float;

function gl_init() {
  gl = canvas.getContext("webgl", {alpha: false});
  minify_context(gl);

  gl.getExtension("WEBGL_depth_texture");

  if (use_texture_float) {
    gl_ext_half_float = gl.getExtension("OES_texture_half_float");
    gl.getExtension("OES_texture_half_float_linear");
    gl.getExtension("EXT_color_buffer_half_float");
    minify_context(gl_ext_half_float);
  }

  gl.viewport(0, 0, canvas.width, canvas.height);

  // get readable strings for error enum values

  canvas_2d = document.createElement("canvas");
  canvas_2d.width = canvas_2d.height = 2048;
  ctx_2d = canvas_2d.getContext("2d");
  minify_context(ctx_2d);
}


var _locations = [
  "a_position",
  "a_uv",
  "a_normal",
  "a_color",
  "a_triangle_id"
];

var POS = 0;
var UV = 1;
var NORMAL = 2;
var COLOR = 3;
var TRIANGLE_ID = 4;

function gfx_init() {
  // replace the render passes' texture arrays by actual frame buffer objects
  // this is far from optimal...
  for (var s=0; s<scenes.length; ++s) {
    var scene = scenes[s];
    for (var p=0; p<scene.passes.length; ++p) {
      var pass = scene.passes[p];
      if (pass.render_to) {
        pass.fbo = frame_buffer(pass.render_to);
      }
    }
  }
  
  uniforms["cam_pos"] = [0, 1, 0]
  uniforms["cam_target"] = [0, 0, 0]
  uniforms["cam_fov"] = 75
  uniforms["cam_tilt"] = 0
  
  // hack to make the export toolchain minify attribute and uniform names

  // edition placeholders
}

function make_vbo(location, buffer) {
  var vbo = gl.createBuffer();
  gl.bindBuffer(gl.ARRAY_BUFFER, vbo);
  gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(buffer), gl.STATIC_DRAW);
  return {location: location, vbo: vbo, length: buffer.length};
}

// editor only

// actually renders
function draw_geom(data) {
  for (var i in data.buffers) {
    var buffer = data.buffers[i];
    gl.bindBuffer(gl.ARRAY_BUFFER, buffer.vbo);
    gl.enableVertexAttribArray(buffer.location);
    gl.vertexAttribPointer(buffer.location, buffer.length / data.vertex_count, gl.FLOAT, false, 0, 0);
  }
  gl.drawArrays(data.mode, 0, data.vertex_count);
}

// type: gl.VERTEX_SHADER or gl.FRAGMENT_SHADER
function compile_shader(txt_src, type) {
  var shader = gl.createShader(type);
  gl.shaderSource(shader, txt_src);
  gl.compileShader(shader);
  return shader;
}

function load_shader_program(vs_entry_point, fs_entry_point) {
  var vs = vs_shader_source.replace(vs_entry_point + "()", "main()");
  var fs = fs_shader_source.replace(fs_entry_point + "()", "main()");
  var program = gl.createProgram();
  gl.attachShader(program, compile_shader(vs, gl.VERTEX_SHADER));
  gl.attachShader(program, compile_shader(fs, gl.FRAGMENT_SHADER));

  for (var i in _locations) {
    gl.bindAttribLocation(program, i, _locations[i]);
  }

  gl.linkProgram(program);
  return program;
}


function set_texture_flags(texture, allow_repeat, linear_filtering, mipmaps) {
  // XXX - Getting the following error associated to the bind texture call:
  // WebGL: A texture is going to be rendered as if it were black, as per the
  // OpenGL ES 2.0.24 spec section 3.8.2, because it is a 2D texture, with a
  // minification filter requiring a mipmap, and is not mipmap complete (as
  // defined in section 3.7.10).
  gl.bindTexture(gl.TEXTURE_2D, texture);

  var wrap = allow_repeat ? gl.REPEAT : gl.CLAMP_TO_EDGE;
  var min_filtering = linear_filtering
                    ? mipmaps ? gl.LINEAR_MIPMAP_LINEAR : gl.LINEAR
                    : gl.NEAREST;
  var mag_filtering = linear_filtering ? gl.LINEAR : gl.NEAREST;

  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, wrap);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, wrap);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, min_filtering);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, mag_filtering);
  if (mipmaps) {
    gl.generateMipmap(gl.TEXTURE_2D);
  }
}

function create_texture(width, height, format, data, allow_repeat, linear_filtering, mipmaps, float_tex) {
  format = format || gl.RGBA;
  width = width || canvas.width;
  height = height || canvas.height;

  var texture = gl.createTexture();

  gl.bindTexture(gl.TEXTURE_2D, texture);
  gl.texImage2D(gl.TEXTURE_2D, 0, format, width, height, 0,
                format,
                (use_texture_float && float_tex) ? gl_ext_half_float.HALF_FLOAT_OES
                          : (format == gl.DEPTH_COMPONENT) ? gl.UNSIGNED_SHORT
                                                           : gl.UNSIGNED_BYTE,
                data ? new Uint8Array(data, 0, 0) : null);

  (format == gl.DEPTH_COMPONENT) || set_texture_flags(texture, allow_repeat, linear_filtering, mipmaps);
  //debug{{
  if (float_tex && data) {
    // wouldn't be hard to add, but we haven't needed it yet.
    console.log("!!! We don't support uploading data to float textures, something may be busted.");
  }
  //debug}}

  return {
    tex: texture,
    width: width,
    height: height
  };
}

function destroy_texture(texture) {
  gl.deleteTexture(texture.tex);
}

function texture_unit(i) { return gl.TEXTURE0+i; }

function frame_buffer(target) {
  var fbo = gl.createFramebuffer();
  gl.bindFramebuffer(gl.FRAMEBUFFER, fbo);

  if (target.color && textures[target.color]) gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, textures[target.color].tex, 0);
  if (target.depth && textures[target.depth]) gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT, gl.TEXTURE_2D, textures[target.depth].tex, 0);


  return fbo;
}

function set_uniforms(program, ratio) {
  var viewMatrix = mat4.create()
  var projectionMatrix = mat4.create()
  var viewProjectionMatrix = mat4.create()
  var viewProjectionMatrixInv = mat4.create()
  
  // derive camera matrices from simpler parameters
  //mat4.lookAt(viewMatrix, uniforms["cam_pos"], uniforms["cam_target"], [0.0, 1.0, 0.0]);
  mat4.lookAtTilt(viewMatrix, uniforms["cam_pos"], uniforms["cam_target"], uniforms["cam_tilt"]);
  mat4.perspective(projectionMatrix, uniforms["cam_fov"] * M.PI / 180.0, ratio, 2.0, 10000.0)
  mat4.multiply(viewProjectionMatrix, projectionMatrix, viewMatrix);
  mat4.invert(viewProjectionMatrixInv, viewProjectionMatrix);
  uniforms["view_proj_mat"] = viewProjectionMatrix;
  uniforms["view_proj_mat_inv"] = viewProjectionMatrixInv;
  
  for (var uniformName in uniforms) {
    var val = uniforms[uniformName];

    var location = gl.getUniformLocation(program, uniformName);
    if (!location)
      continue;

    // if val is a bare number, make a one-element array
    if (typeof val == "number")
      val = [val];

    switch (val.length) {
      case 1: gl.uniform1fv(location, val); break;
      case 2: gl.uniform2fv(location, val); break;
      case 3: gl.uniform3fv(location, val); break;
      case 4: gl.uniform4fv(location, val); break;
      case 9: gl.uniformMatrix3fv(location, 0, val); break;
      case 16: gl.uniformMatrix4fv(location, 0, val); break;
    }
  }
}

function render_scene(scene, demo_time, scene_time) {
  var clip_time_norm = scene_time/scene.duration;
  uniforms["clip_time"] = scene_time;
  var t = {
    scene_norm: clip_time_norm,
    demo: demo_time,
    scene: scene_time
  };
  if (scene.update) {
    scene.update(t);
  }
  gl.disable(gl.BLEND);
  for (var p in scene.passes) {
    var pass = scene.passes[p];

    var texture_inputs = [];
    if (pass.texture_inputs) {
      for (var i=0; i<pass.texture_inputs.length; ++i) {
        texture_inputs.push(textures[pass.texture_inputs[i]]);
      }
    }

    if (pass.update) {
      pass.update(t, texture_inputs);
    }

    var program = programs[pass.program]
    
    var shader_program = program;
    gl.useProgram(shader_program);
    var rx = canvas.width;
    var ry = canvas.height;
    if (pass.render_to) {
      rx = textures[pass.render_to.color].width;
      ry = textures[pass.render_to.color].height;
    }
    uniforms["u_resolution"] = [rx,ry];
    set_uniforms(shader_program, rx / ry);
    gl.viewport(0, 0, rx, ry);

    gl.bindFramebuffer(gl.FRAMEBUFFER, pass.fbo);
    
    for (var i=0; i<texture_inputs.length; ++i) {
      var tex = texture_inputs[i].tex;
      gl.activeTexture(texture_unit(i));
      gl.bindTexture(gl.TEXTURE_2D, tex);
      gl.uniform1i(gl.getUniformLocation(shader_program,"texture_"+i), i);
    }

    if (pass.blend) {
      gl.enable(gl.BLEND);
      gl.blendFunc.apply(gl, pass.blend);
    }
    
    if (pass.depth_test) {
      gl.enable(gl.DEPTH_TEST);
    }
    else {
      gl.disable(gl.DEPTH_TEST);
    }
    
    if (pass.clear) {
      gl.clearColor(pass.clear[0], pass.clear[1], pass.clear[2], pass.clear[3]);
      gl.clearDepth(1.0);
      gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
    }
    
    if (pass.geometry) {
      var geometry = geometries[pass.geometry]

      draw_geom(geometry)
    }
  }
}


/**
 * @fileoverview gl-matrix - High performance matrix and vector operations
 * @author Brandon Jones
 * @author Colin MacKenzie IV
 * @version 2.2.1
 */

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

var GLMAT_EPSILON = 0.000001;
var GLMAT_ARRAY_TYPE = Float32Array;
var GLMAT_RANDOM = M.random;

/**
 * @class Common utilities
 * @name glMatrix
 */
var glMatrix = {};

/**
 * Sets the type of array used when creating new vectors and matricies
 *
 * @param {Type} type Array type, such as Float32Array or Array
 */
glMatrix.setMatrixArrayType = function(type) {
    GLMAT_ARRAY_TYPE = type;
}

var degree = M.PI / 180;

/**
* Convert Degree To Radian
*
* @param {Number} Angle in Degrees
*/
glMatrix.toRadian = function(a){
     return a * degree;
}
;
/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 2 Dimensional Vector
 * @name vec2
 */

var vec2 = {};

/**
 * Creates a new, empty vec2
 *
 * @returns {vec2} a new 2D vector
 */
vec2.create = function() {
    var out = new GLMAT_ARRAY_TYPE(2);
    out[0] = 0;
    out[1] = 0;
    return out;
};

/**
 * Creates a new vec2 initialized with values from an existing vector
 *
 * @param {vec2} a vector to clone
 * @returns {vec2} a new 2D vector
 */
vec2.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(2);
    out[0] = a[0];
    out[1] = a[1];
    return out;
};

/**
 * Creates a new vec2 initialized with the given values
 *
 * @param {Number} x X component
 * @param {Number} y Y component
 * @returns {vec2} a new 2D vector
 */
vec2.fromValues = function(x, y) {
    var out = new GLMAT_ARRAY_TYPE(2);
    out[0] = x;
    out[1] = y;
    return out;
};

/**
 * Copy the values from one vec2 to another
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the source vector
 * @returns {vec2} out
 */
vec2.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    return out;
};

/**
 * Set the components of a vec2 to the given values
 *
 * @param {vec2} out the receiving vector
 * @param {Number} x X component
 * @param {Number} y Y component
 * @returns {vec2} out
 */
vec2.set = function(out, x, y) {
    out[0] = x;
    out[1] = y;
    return out;
};

/**
 * Adds two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.add = function(out, a, b) {
    out[0] = a[0] + b[0];
    out[1] = a[1] + b[1];
    return out;
};

/**
 * Subtracts vector b from vector a
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.subtract = function(out, a, b) {
    out[0] = a[0] - b[0];
    out[1] = a[1] - b[1];
    return out;
};

/**
 * Alias for {@link vec2.subtract}
 * @function
 */
vec2.sub = vec2.subtract;

/**
 * Multiplies two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.multiply = function(out, a, b) {
    out[0] = a[0] * b[0];
    out[1] = a[1] * b[1];
    return out;
};

/**
 * Alias for {@link vec2.multiply}
 * @function
 */
vec2.mul = vec2.multiply;

/**
 * Divides two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.divide = function(out, a, b) {
    out[0] = a[0] / b[0];
    out[1] = a[1] / b[1];
    return out;
};

/**
 * Alias for {@link vec2.divide}
 * @function
 */
vec2.div = vec2.divide;

/**
 * Returns the minimum of two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.min = function(out, a, b) {
    out[0] = M.min(a[0], b[0]);
    out[1] = M.min(a[1], b[1]);
    return out;
};

/**
 * Returns the maximum of two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec2} out
 */
vec2.max = function(out, a, b) {
    out[0] = M.max(a[0], b[0]);
    out[1] = M.max(a[1], b[1]);
    return out;
};

/**
 * Scales a vec2 by a scalar number
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the vector to scale
 * @param {Number} b amount to scale the vector by
 * @returns {vec2} out
 */
vec2.scale = function(out, a, b) {
    out[0] = a[0] * b;
    out[1] = a[1] * b;
    return out;
};

/**
 * Adds two vec2's after scaling the second operand by a scalar value
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @param {Number} scale the amount to scale b by before adding
 * @returns {vec2} out
 */
vec2.scaleAndAdd = function(out, a, b, scale) {
    out[0] = a[0] + (b[0] * scale);
    out[1] = a[1] + (b[1] * scale);
    return out;
};

/**
 * Calculates the euclidian distance between two vec2's
 *
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {Number} distance between a and b
 */
vec2.distance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1];
    return M.sqrt(x*x + y*y);
};

/**
 * Alias for {@link vec2.distance}
 * @function
 */
vec2.dist = vec2.distance;

/**
 * Calculates the squared euclidian distance between two vec2's
 *
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {Number} squared distance between a and b
 */
vec2.squaredDistance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1];
    return x*x + y*y;
};

/**
 * Alias for {@link vec2.squaredDistance}
 * @function
 */
vec2.sqrDist = vec2.squaredDistance;

/**
 * Calculates the length of a vec2
 *
 * @param {vec2} a vector to calculate length of
 * @returns {Number} length of a
 */
vec2.length = function (a) {
    var x = a[0],
        y = a[1];
    return M.sqrt(x*x + y*y);
};

/**
 * Alias for {@link vec2.length}
 * @function
 */
vec2.len = vec2.length;

/**
 * Calculates the squared length of a vec2
 *
 * @param {vec2} a vector to calculate squared length of
 * @returns {Number} squared length of a
 */
vec2.squaredLength = function (a) {
    var x = a[0],
        y = a[1];
    return x*x + y*y;
};

/**
 * Alias for {@link vec2.squaredLength}
 * @function
 */
vec2.sqrLen = vec2.squaredLength;

/**
 * Negates the components of a vec2
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a vector to negate
 * @returns {vec2} out
 */
vec2.negate = function(out, a) {
    out[0] = -a[0];
    out[1] = -a[1];
    return out;
};

/**
 * Normalize a vec2
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a vector to normalize
 * @returns {vec2} out
 */
vec2.normalize = function(out, a) {
    var x = a[0],
        y = a[1];
    var len = x*x + y*y;
    if (len > 0) {
        //TODO: evaluate use of glm_invsqrt here?
        len = 1 / M.sqrt(len);
        out[0] = a[0] * len;
        out[1] = a[1] * len;
    }
    return out;
};

/**
 * Calculates the dot product of two vec2's
 *
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {Number} dot product of a and b
 */
vec2.dot = function (a, b) {
    return a[0] * b[0] + a[1] * b[1];
};

/**
 * Computes the cross product of two vec2's
 * Note that the cross product must by definition produce a 3D vector
 *
 * @param {vec3} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @returns {vec3} out
 */
vec2.cross = function(out, a, b) {
    var z = a[0] * b[1] - a[1] * b[0];
    out[0] = out[1] = 0;
    out[2] = z;
    return out;
};

/**
 * Performs a linear interpolation between two vec2's
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the first operand
 * @param {vec2} b the second operand
 * @param {Number} t interpolation amount between the two inputs
 * @returns {vec2} out
 */
vec2.lerp = function (out, a, b, t) {
    var ax = a[0],
        ay = a[1];
    out[0] = ax + t * (b[0] - ax);
    out[1] = ay + t * (b[1] - ay);
    return out;
};

/**
 * Generates a random vector with the given scale
 *
 * @param {vec2} out the receiving vector
 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
 * @returns {vec2} out
 */
vec2.random = function (out, scale) {
    scale = scale || 1.0;
    var r = GLMAT_RANDOM() * 2.0 * M.PI;
    out[0] = M.cos(r) * scale;
    out[1] = M.sin(r) * scale;
    return out;
};

/**
 * Transforms the vec2 with a mat2
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the vector to transform
 * @param {mat2} m matrix to transform with
 * @returns {vec2} out
 */
vec2.transformMat2 = function(out, a, m) {
    var x = a[0],
        y = a[1];
    out[0] = m[0] * x + m[2] * y;
    out[1] = m[1] * x + m[3] * y;
    return out;
};

/**
 * Transforms the vec2 with a mat2d
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the vector to transform
 * @param {mat2d} m matrix to transform with
 * @returns {vec2} out
 */
vec2.transformMat2d = function(out, a, m) {
    var x = a[0],
        y = a[1];
    out[0] = m[0] * x + m[2] * y + m[4];
    out[1] = m[1] * x + m[3] * y + m[5];
    return out;
};

/**
 * Transforms the vec2 with a mat3
 * 3rd vector component is implicitly '1'
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the vector to transform
 * @param {mat3} m matrix to transform with
 * @returns {vec2} out
 */
vec2.transformMat3 = function(out, a, m) {
    var x = a[0],
        y = a[1];
    out[0] = m[0] * x + m[3] * y + m[6];
    out[1] = m[1] * x + m[4] * y + m[7];
    return out;
};

/**
 * Transforms the vec2 with a mat4
 * 3rd vector component is implicitly '0'
 * 4th vector component is implicitly '1'
 *
 * @param {vec2} out the receiving vector
 * @param {vec2} a the vector to transform
 * @param {mat4} m matrix to transform with
 * @returns {vec2} out
 */
vec2.transformMat4 = function(out, a, m) {
    var x = a[0], 
        y = a[1];
    out[0] = m[0] * x + m[4] * y + m[12];
    out[1] = m[1] * x + m[5] * y + m[13];
    return out;
};

/**
 * Perform some operation over an array of vec2s.
 *
 * @param {Array} a the array of vectors to iterate over
 * @param {Number} stride Number of elements between the start of each vec2. If 0 assumes tightly packed
 * @param {Number} offset Number of elements to skip at the beginning of the array
 * @param {Number} count Number of vec2s to iterate over. If 0 iterates over entire array
 * @param {Function} fn Function to call for each vector in the array
 * @param {Object} [arg] additional argument to pass to fn
 * @returns {Array} a
 * @function
 */
vec2.forEach = function(a, stride, offset, count, fn, arg) {
  var vec = vec2.create();
  var i, l;
  if(!stride) {
      stride = 2;
  }

  if(!offset) {
      offset = 0;
  }
  
  if(count) {
      l = M.min((count * stride) + offset, a.length);
  } else {
      l = a.length;
  }

  for(i = offset; i < l; i += stride) {
      vec[0] = a[i]; vec[1] = a[i+1];
      fn(vec, vec, arg);
      a[i] = vec[0]; a[i+1] = vec[1];
  }
  
  return a;
};

/**
 * Returns a string representation of a vector
 *
 * @param {vec2} vec vector to represent as a string
 * @returns {String} string representation of the vector
 */
vec2.str = function (a) {
    return 'vec2(' + a[0] + ', ' + a[1] + ')';
};

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 3 Dimensional Vector
 * @name vec3
 */

var vec3 = {};

/**
 * Creates a new, empty vec3
 *
 * @returns {vec3} a new 3D vector
 */
vec3.create = function() {
    var out = new GLMAT_ARRAY_TYPE(3);
    out[0] = 0;
    out[1] = 0;
    out[2] = 0;
    return out;
};

/**
 * Creates a new vec3 initialized with values from an existing vector
 *
 * @param {vec3} a vector to clone
 * @returns {vec3} a new 3D vector
 */
vec3.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(3);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    return out;
};

/**
 * Creates a new vec3 initialized with the given values
 *
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @returns {vec3} a new 3D vector
 */
vec3.fromValues = function(x, y, z) {
    var out = new GLMAT_ARRAY_TYPE(3);
    out[0] = x;
    out[1] = y;
    out[2] = z;
    return out;
};

/**
 * Copy the values from one vec3 to another
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the source vector
 * @returns {vec3} out
 */
vec3.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    return out;
};

/**
 * Set the components of a vec3 to the given values
 *
 * @param {vec3} out the receiving vector
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @returns {vec3} out
 */
vec3.set = function(out, x, y, z) {
    out[0] = x;
    out[1] = y;
    out[2] = z;
    return out;
};

/**
 * Adds two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.add = function(out, a, b) {
    out[0] = a[0] + b[0];
    out[1] = a[1] + b[1];
    out[2] = a[2] + b[2];
    return out;
};

/**
 * Subtracts vector b from vector a
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.subtract = function(out, a, b) {
    out[0] = a[0] - b[0];
    out[1] = a[1] - b[1];
    out[2] = a[2] - b[2];
    return out;
};

/**
 * Alias for {@link vec3.subtract}
 * @function
 */
vec3.sub = vec3.subtract;

/**
 * Multiplies two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.multiply = function(out, a, b) {
    out[0] = a[0] * b[0];
    out[1] = a[1] * b[1];
    out[2] = a[2] * b[2];
    return out;
};

/**
 * Alias for {@link vec3.multiply}
 * @function
 */
vec3.mul = vec3.multiply;

/**
 * Divides two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.divide = function(out, a, b) {
    out[0] = a[0] / b[0];
    out[1] = a[1] / b[1];
    out[2] = a[2] / b[2];
    return out;
};

/**
 * Alias for {@link vec3.divide}
 * @function
 */
vec3.div = vec3.divide;

/**
 * Returns the minimum of two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.min = function(out, a, b) {
    out[0] = M.min(a[0], b[0]);
    out[1] = M.min(a[1], b[1]);
    out[2] = M.min(a[2], b[2]);
    return out;
};

/**
 * Returns the maximum of two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.max = function(out, a, b) {
    out[0] = M.max(a[0], b[0]);
    out[1] = M.max(a[1], b[1]);
    out[2] = M.max(a[2], b[2]);
    return out;
};

/**
 * Scales a vec3 by a scalar number
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the vector to scale
 * @param {Number} b amount to scale the vector by
 * @returns {vec3} out
 */
vec3.scale = function(out, a, b) {
    out[0] = a[0] * b;
    out[1] = a[1] * b;
    out[2] = a[2] * b;
    return out;
};

/**
 * Adds two vec3's after scaling the second operand by a scalar value
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @param {Number} scale the amount to scale b by before adding
 * @returns {vec3} out
 */
vec3.scaleAndAdd = function(out, a, b, scale) {
    out[0] = a[0] + (b[0] * scale);
    out[1] = a[1] + (b[1] * scale);
    out[2] = a[2] + (b[2] * scale);
    return out;
};

/**
 * Calculates the euclidian distance between two vec3's
 *
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {Number} distance between a and b
 */
vec3.distance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1],
        z = b[2] - a[2];
    return M.sqrt(x*x + y*y + z*z);
};

/**
 * Alias for {@link vec3.distance}
 * @function
 */
vec3.dist = vec3.distance;

/**
 * Calculates the squared euclidian distance between two vec3's
 *
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {Number} squared distance between a and b
 */
vec3.squaredDistance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1],
        z = b[2] - a[2];
    return x*x + y*y + z*z;
};

/**
 * Alias for {@link vec3.squaredDistance}
 * @function
 */
vec3.sqrDist = vec3.squaredDistance;

/**
 * Calculates the length of a vec3
 *
 * @param {vec3} a vector to calculate length of
 * @returns {Number} length of a
 */
vec3.length = function (a) {
    var x = a[0],
        y = a[1],
        z = a[2];
    return M.sqrt(x*x + y*y + z*z);
};

/**
 * Alias for {@link vec3.length}
 * @function
 */
vec3.len = vec3.length;

/**
 * Calculates the squared length of a vec3
 *
 * @param {vec3} a vector to calculate squared length of
 * @returns {Number} squared length of a
 */
vec3.squaredLength = function (a) {
    var x = a[0],
        y = a[1],
        z = a[2];
    return x*x + y*y + z*z;
};

/**
 * Alias for {@link vec3.squaredLength}
 * @function
 */
vec3.sqrLen = vec3.squaredLength;

/**
 * Negates the components of a vec3
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a vector to negate
 * @returns {vec3} out
 */
vec3.negate = function(out, a) {
    out[0] = -a[0];
    out[1] = -a[1];
    out[2] = -a[2];
    return out;
};

/**
 * Normalize a vec3
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a vector to normalize
 * @returns {vec3} out
 */
vec3.normalize = function(out, a) {
    var x = a[0],
        y = a[1],
        z = a[2];
    var len = x*x + y*y + z*z;
    if (len > 0) {
        //TODO: evaluate use of glm_invsqrt here?
        len = 1 / M.sqrt(len);
        out[0] = a[0] * len;
        out[1] = a[1] * len;
        out[2] = a[2] * len;
    }
    return out;
};

/**
 * Calculates the dot product of two vec3's
 *
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {Number} dot product of a and b
 */
vec3.dot = function (a, b) {
    return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
};

/**
 * Computes the cross product of two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @returns {vec3} out
 */
vec3.cross = function(out, a, b) {
    var ax = a[0], ay = a[1], az = a[2],
        bx = b[0], by = b[1], bz = b[2];

    out[0] = ay * bz - az * by;
    out[1] = az * bx - ax * bz;
    out[2] = ax * by - ay * bx;
    return out;
};

/**
 * Performs a linear interpolation between two vec3's
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the first operand
 * @param {vec3} b the second operand
 * @param {Number} t interpolation amount between the two inputs
 * @returns {vec3} out
 */
vec3.lerp = function (out, a, b, t) {
    var ax = a[0],
        ay = a[1],
        az = a[2];
    out[0] = ax + t * (b[0] - ax);
    out[1] = ay + t * (b[1] - ay);
    out[2] = az + t * (b[2] - az);
    return out;
};

/**
 * Generates a random vector with the given scale
 *
 * @param {vec3} out the receiving vector
 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
 * @returns {vec3} out
 */
vec3.random = function (out, scale) {
    scale = scale || 1.0;

    var r = GLMAT_RANDOM() * 2.0 * M.PI;
    var z = (GLMAT_RANDOM() * 2.0) - 1.0;
    var zScale = M.sqrt(1.0-z*z) * scale;

    out[0] = M.cos(r) * zScale;
    out[1] = M.sin(r) * zScale;
    out[2] = z * scale;
    return out;
};

/**
 * Transforms the vec3 with a mat4.
 * 4th vector component is implicitly '1'
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the vector to transform
 * @param {mat4} m matrix to transform with
 * @returns {vec3} out
 */
vec3.transformMat4 = function(out, a, m) {
    var x = a[0], y = a[1], z = a[2];
    out[0] = m[0] * x + m[4] * y + m[8] * z + m[12];
    out[1] = m[1] * x + m[5] * y + m[9] * z + m[13];
    out[2] = m[2] * x + m[6] * y + m[10] * z + m[14];
    return out;
};

/**
 * Transforms the vec3 with a mat3.
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the vector to transform
 * @param {mat4} m the 3x3 matrix to transform with
 * @returns {vec3} out
 */
vec3.transformMat3 = function(out, a, m) {
    var x = a[0], y = a[1], z = a[2];
    out[0] = x * m[0] + y * m[3] + z * m[6];
    out[1] = x * m[1] + y * m[4] + z * m[7];
    out[2] = x * m[2] + y * m[5] + z * m[8];
    return out;
};

/**
 * Transforms the vec3 with a quat
 *
 * @param {vec3} out the receiving vector
 * @param {vec3} a the vector to transform
 * @param {quat} q quaternion to transform with
 * @returns {vec3} out
 */
vec3.transformQuat = function(out, a, q) {
    // benchmarks: http://jsperf.com/quaternion-transform-vec3-implementations

    var x = a[0], y = a[1], z = a[2],
        qx = q[0], qy = q[1], qz = q[2], qw = q[3],

        // calculate quat * vec
        ix = qw * x + qy * z - qz * y,
        iy = qw * y + qz * x - qx * z,
        iz = qw * z + qx * y - qy * x,
        iw = -qx * x - qy * y - qz * z;

    // calculate result * inverse quat
    out[0] = ix * qw + iw * -qx + iy * -qz - iz * -qy;
    out[1] = iy * qw + iw * -qy + iz * -qx - ix * -qz;
    out[2] = iz * qw + iw * -qz + ix * -qy - iy * -qx;
    return out;
};

/**
* Rotate a 3D vector around the x-axis
* @param {vec3} out The receiving vec3
* @param {vec3} a The vec3 point to rotate
* @param {vec3} b The origin of the rotation
* @param {Number} c The angle of rotation
* @returns {vec3} out
*/
vec3.rotateX = function(out, a, b, c){
   var p = [], r=[];
    //Translate point to the origin
    p[0] = a[0] - b[0];
    p[1] = a[1] - b[1];
    p[2] = a[2] - b[2];

    //perform rotation
    r[0] = p[0];
    r[1] = p[1]*M.cos(c) - p[2]*M.sin(c);
    r[2] = p[1]*M.sin(c) + p[2]*M.cos(c);

    //translate to correct position
    out[0] = r[0] + b[0];
    out[1] = r[1] + b[1];
    out[2] = r[2] + b[2];

    return out;
};

/**
* Rotate a 3D vector around the y-axis
* @param {vec3} out The receiving vec3
* @param {vec3} a The vec3 point to rotate
* @param {vec3} b The origin of the rotation
* @param {Number} c The angle of rotation
* @returns {vec3} out
*/
vec3.rotateY = function(out, a, b, c){
    var p = [], r=[];
    //Translate point to the origin
    p[0] = a[0] - b[0];
    p[1] = a[1] - b[1];
    p[2] = a[2] - b[2];
  
    //perform rotation
    r[0] = p[2]*M.sin(c) + p[0]*M.cos(c);
    r[1] = p[1];
    r[2] = p[2]*M.cos(c) - p[0]*M.sin(c);
  
    //translate to correct position
    out[0] = r[0] + b[0];
    out[1] = r[1] + b[1];
    out[2] = r[2] + b[2];
  
    return out;
};

/**
* Rotate a 3D vector around the z-axis
* @param {vec3} out The receiving vec3
* @param {vec3} a The vec3 point to rotate
* @param {vec3} b The origin of the rotation
* @param {Number} c The angle of rotation
* @returns {vec3} out
*/
vec3.rotateZ = function(out, a, b, c){
    var p = [], r=[];
    //Translate point to the origin
    p[0] = a[0] - b[0];
    p[1] = a[1] - b[1];
    p[2] = a[2] - b[2];
  
    //perform rotation
    r[0] = p[0]*M.cos(c) - p[1]*M.sin(c);
    r[1] = p[0]*M.sin(c) + p[1]*M.cos(c);
    r[2] = p[2];
  
    //translate to correct position
    out[0] = r[0] + b[0];
    out[1] = r[1] + b[1];
    out[2] = r[2] + b[2];
  
    return out;
};

/**
 * Perform some operation over an array of vec3s.
 *
 * @param {Array} a the array of vectors to iterate over
 * @param {Number} stride Number of elements between the start of each vec3. If 0 assumes tightly packed
 * @param {Number} offset Number of elements to skip at the beginning of the array
 * @param {Number} count Number of vec3s to iterate over. If 0 iterates over entire array
 * @param {Function} fn Function to call for each vector in the array
 * @param {Object} [arg] additional argument to pass to fn
 * @returns {Array} a
 * @function
 */
vec3.forEach = function(a, stride, offset, count, fn, arg) {
  var vec = vec3.create();
  var i, l;
  if(!stride) {
      stride = 3;
  }

  if(!offset) {
      offset = 0;
  }
  
  if(count) {
      l = M.min((count * stride) + offset, a.length);
  } else {
      l = a.length;
  }

  for(i = offset; i < l; i += stride) {
      vec[0] = a[i]; vec[1] = a[i+1]; vec[2] = a[i+2];
      fn(vec, vec, arg);
      a[i] = vec[0]; a[i+1] = vec[1]; a[i+2] = vec[2];
  }
  
  return a;
};

/**
 * Returns a string representation of a vector
 *
 * @param {vec3} vec vector to represent as a string
 * @returns {String} string representation of the vector
 */
vec3.str = function (a) {
    return 'vec3(' + a[0] + ', ' + a[1] + ', ' + a[2] + ')';
};

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 4 Dimensional Vector
 * @name vec4
 */

var vec4 = {};

/**
 * Creates a new, empty vec4
 *
 * @returns {vec4} a new 4D vector
 */
vec4.create = function() {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = 0;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    return out;
};

/**
 * Creates a new vec4 initialized with values from an existing vector
 *
 * @param {vec4} a vector to clone
 * @returns {vec4} a new 4D vector
 */
vec4.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    return out;
};

/**
 * Creates a new vec4 initialized with the given values
 *
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @param {Number} w W component
 * @returns {vec4} a new 4D vector
 */
vec4.fromValues = function(x, y, z, w) {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = x;
    out[1] = y;
    out[2] = z;
    out[3] = w;
    return out;
};

/**
 * Copy the values from one vec4 to another
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the source vector
 * @returns {vec4} out
 */
vec4.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    return out;
};

/**
 * Set the components of a vec4 to the given values
 *
 * @param {vec4} out the receiving vector
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @param {Number} w W component
 * @returns {vec4} out
 */
vec4.set = function(out, x, y, z, w) {
    out[0] = x;
    out[1] = y;
    out[2] = z;
    out[3] = w;
    return out;
};

/**
 * Adds two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.add = function(out, a, b) {
    out[0] = a[0] + b[0];
    out[1] = a[1] + b[1];
    out[2] = a[2] + b[2];
    out[3] = a[3] + b[3];
    return out;
};

/**
 * Subtracts vector b from vector a
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.subtract = function(out, a, b) {
    out[0] = a[0] - b[0];
    out[1] = a[1] - b[1];
    out[2] = a[2] - b[2];
    out[3] = a[3] - b[3];
    return out;
};

/**
 * Alias for {@link vec4.subtract}
 * @function
 */
vec4.sub = vec4.subtract;

/**
 * Multiplies two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.multiply = function(out, a, b) {
    out[0] = a[0] * b[0];
    out[1] = a[1] * b[1];
    out[2] = a[2] * b[2];
    out[3] = a[3] * b[3];
    return out;
};

/**
 * Alias for {@link vec4.multiply}
 * @function
 */
vec4.mul = vec4.multiply;

/**
 * Divides two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.divide = function(out, a, b) {
    out[0] = a[0] / b[0];
    out[1] = a[1] / b[1];
    out[2] = a[2] / b[2];
    out[3] = a[3] / b[3];
    return out;
};

/**
 * Alias for {@link vec4.divide}
 * @function
 */
vec4.div = vec4.divide;

/**
 * Returns the minimum of two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.min = function(out, a, b) {
    out[0] = M.min(a[0], b[0]);
    out[1] = M.min(a[1], b[1]);
    out[2] = M.min(a[2], b[2]);
    out[3] = M.min(a[3], b[3]);
    return out;
};

/**
 * Returns the maximum of two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {vec4} out
 */
vec4.max = function(out, a, b) {
    out[0] = M.max(a[0], b[0]);
    out[1] = M.max(a[1], b[1]);
    out[2] = M.max(a[2], b[2]);
    out[3] = M.max(a[3], b[3]);
    return out;
};

/**
 * Scales a vec4 by a scalar number
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the vector to scale
 * @param {Number} b amount to scale the vector by
 * @returns {vec4} out
 */
vec4.scale = function(out, a, b) {
    out[0] = a[0] * b;
    out[1] = a[1] * b;
    out[2] = a[2] * b;
    out[3] = a[3] * b;
    return out;
};

/**
 * Adds two vec4's after scaling the second operand by a scalar value
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @param {Number} scale the amount to scale b by before adding
 * @returns {vec4} out
 */
vec4.scaleAndAdd = function(out, a, b, scale) {
    out[0] = a[0] + (b[0] * scale);
    out[1] = a[1] + (b[1] * scale);
    out[2] = a[2] + (b[2] * scale);
    out[3] = a[3] + (b[3] * scale);
    return out;
};

/**
 * Calculates the euclidian distance between two vec4's
 *
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {Number} distance between a and b
 */
vec4.distance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1],
        z = b[2] - a[2],
        w = b[3] - a[3];
    return M.sqrt(x*x + y*y + z*z + w*w);
};

/**
 * Alias for {@link vec4.distance}
 * @function
 */
vec4.dist = vec4.distance;

/**
 * Calculates the squared euclidian distance between two vec4's
 *
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {Number} squared distance between a and b
 */
vec4.squaredDistance = function(a, b) {
    var x = b[0] - a[0],
        y = b[1] - a[1],
        z = b[2] - a[2],
        w = b[3] - a[3];
    return x*x + y*y + z*z + w*w;
};

/**
 * Alias for {@link vec4.squaredDistance}
 * @function
 */
vec4.sqrDist = vec4.squaredDistance;

/**
 * Calculates the length of a vec4
 *
 * @param {vec4} a vector to calculate length of
 * @returns {Number} length of a
 */
vec4.length = function (a) {
    var x = a[0],
        y = a[1],
        z = a[2],
        w = a[3];
    return M.sqrt(x*x + y*y + z*z + w*w);
};

/**
 * Alias for {@link vec4.length}
 * @function
 */
vec4.len = vec4.length;

/**
 * Calculates the squared length of a vec4
 *
 * @param {vec4} a vector to calculate squared length of
 * @returns {Number} squared length of a
 */
vec4.squaredLength = function (a) {
    var x = a[0],
        y = a[1],
        z = a[2],
        w = a[3];
    return x*x + y*y + z*z + w*w;
};

/**
 * Alias for {@link vec4.squaredLength}
 * @function
 */
vec4.sqrLen = vec4.squaredLength;

/**
 * Negates the components of a vec4
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a vector to negate
 * @returns {vec4} out
 */
vec4.negate = function(out, a) {
    out[0] = -a[0];
    out[1] = -a[1];
    out[2] = -a[2];
    out[3] = -a[3];
    return out;
};

/**
 * Normalize a vec4
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a vector to normalize
 * @returns {vec4} out
 */
vec4.normalize = function(out, a) {
    var x = a[0],
        y = a[1],
        z = a[2],
        w = a[3];
    var len = x*x + y*y + z*z + w*w;
    if (len > 0) {
        len = 1 / M.sqrt(len);
        out[0] = a[0] * len;
        out[1] = a[1] * len;
        out[2] = a[2] * len;
        out[3] = a[3] * len;
    }
    return out;
};

/**
 * Calculates the dot product of two vec4's
 *
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @returns {Number} dot product of a and b
 */
vec4.dot = function (a, b) {
    return a[0] * b[0] + a[1] * b[1] + a[2] * b[2] + a[3] * b[3];
};

/**
 * Performs a linear interpolation between two vec4's
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the first operand
 * @param {vec4} b the second operand
 * @param {Number} t interpolation amount between the two inputs
 * @returns {vec4} out
 */
vec4.lerp = function (out, a, b, t) {
    var ax = a[0],
        ay = a[1],
        az = a[2],
        aw = a[3];
    out[0] = ax + t * (b[0] - ax);
    out[1] = ay + t * (b[1] - ay);
    out[2] = az + t * (b[2] - az);
    out[3] = aw + t * (b[3] - aw);
    return out;
};

/**
 * Generates a random vector with the given scale
 *
 * @param {vec4} out the receiving vector
 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
 * @returns {vec4} out
 */
vec4.random = function (out, scale) {
    scale = scale || 1.0;

    //TODO: This is a pretty awful way of doing this. Find something better.
    out[0] = GLMAT_RANDOM();
    out[1] = GLMAT_RANDOM();
    out[2] = GLMAT_RANDOM();
    out[3] = GLMAT_RANDOM();
    vec4.normalize(out, out);
    vec4.scale(out, out, scale);
    return out;
};

/**
 * Transforms the vec4 with a mat4.
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the vector to transform
 * @param {mat4} m matrix to transform with
 * @returns {vec4} out
 */
vec4.transformMat4 = function(out, a, m) {
    var x = a[0], y = a[1], z = a[2], w = a[3];
    out[0] = m[0] * x + m[4] * y + m[8] * z + m[12] * w;
    out[1] = m[1] * x + m[5] * y + m[9] * z + m[13] * w;
    out[2] = m[2] * x + m[6] * y + m[10] * z + m[14] * w;
    out[3] = m[3] * x + m[7] * y + m[11] * z + m[15] * w;
    return out;
};

/**
 * Transforms the vec4 with a quat
 *
 * @param {vec4} out the receiving vector
 * @param {vec4} a the vector to transform
 * @param {quat} q quaternion to transform with
 * @returns {vec4} out
 */
vec4.transformQuat = function(out, a, q) {
    var x = a[0], y = a[1], z = a[2],
        qx = q[0], qy = q[1], qz = q[2], qw = q[3],

        // calculate quat * vec
        ix = qw * x + qy * z - qz * y,
        iy = qw * y + qz * x - qx * z,
        iz = qw * z + qx * y - qy * x,
        iw = -qx * x - qy * y - qz * z;

    // calculate result * inverse quat
    out[0] = ix * qw + iw * -qx + iy * -qz - iz * -qy;
    out[1] = iy * qw + iw * -qy + iz * -qx - ix * -qz;
    out[2] = iz * qw + iw * -qz + ix * -qy - iy * -qx;
    return out;
};

/**
 * Perform some operation over an array of vec4s.
 *
 * @param {Array} a the array of vectors to iterate over
 * @param {Number} stride Number of elements between the start of each vec4. If 0 assumes tightly packed
 * @param {Number} offset Number of elements to skip at the beginning of the array
 * @param {Number} count Number of vec2s to iterate over. If 0 iterates over entire array
 * @param {Function} fn Function to call for each vector in the array
 * @param {Object} [arg] additional argument to pass to fn
 * @returns {Array} a
 * @function
 */
vec4.forEach = function(a, stride, offset, count, fn, arg) {
  var vec = vec4.create();
  var i, l;
  if(!stride) {
      stride = 4;
  }

  if(!offset) {
      offset = 0;
  }
  
  if(count) {
      l = M.min((count * stride) + offset, a.length);
  } else {
      l = a.length;
  }

  for(i = offset; i < l; i += stride) {
      vec[0] = a[i]; vec[1] = a[i+1]; vec[2] = a[i+2]; vec[3] = a[i+3];
      fn(vec, vec, arg);
      a[i] = vec[0]; a[i+1] = vec[1]; a[i+2] = vec[2]; a[i+3] = vec[3];
  }
  
  return a;
};

/**
 * Returns a string representation of a vector
 *
 * @param {vec4} vec vector to represent as a string
 * @returns {String} string representation of the vector
 */
vec4.str = function (a) {
    return 'vec4(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
};

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 2x2 Matrix
 * @name mat2
 */

var mat2 = {};

/**
 * Creates a new identity mat2
 *
 * @returns {mat2} a new 2x2 matrix
 */
mat2.create = function() {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    return out;
};

/**
 * Creates a new mat2 initialized with values from an existing matrix
 *
 * @param {mat2} a matrix to clone
 * @returns {mat2} a new 2x2 matrix
 */
mat2.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    return out;
};

/**
 * Copy the values from one mat2 to another
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the source matrix
 * @returns {mat2} out
 */
mat2.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    return out;
};

/**
 * Set a mat2 to the identity matrix
 *
 * @param {mat2} out the receiving matrix
 * @returns {mat2} out
 */
mat2.identity = function(out) {
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    return out;
};

/**
 * Transpose the values of a mat2
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the source matrix
 * @returns {mat2} out
 */
mat2.transpose = function(out, a) {
    // If we are transposing ourselves we can skip a few steps but have to cache some values
    if (out === a) {
        var a1 = a[1];
        out[1] = a[2];
        out[2] = a1;
    } else {
        out[0] = a[0];
        out[1] = a[2];
        out[2] = a[1];
        out[3] = a[3];
    }
    
    return out;
};

/**
 * Inverts a mat2
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the source matrix
 * @returns {mat2} out
 */
mat2.invert = function(out, a) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3],

        // Calculate the determinant
        det = a0 * a3 - a2 * a1;

    if (!det) {
        return null;
    }
    det = 1.0 / det;
    
    out[0] =  a3 * det;
    out[1] = -a1 * det;
    out[2] = -a2 * det;
    out[3] =  a0 * det;

    return out;
};

/**
 * Calculates the adjugate of a mat2
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the source matrix
 * @returns {mat2} out
 */
mat2.adjoint = function(out, a) {
    // Caching this value is nessecary if out == a
    var a0 = a[0];
    out[0] =  a[3];
    out[1] = -a[1];
    out[2] = -a[2];
    out[3] =  a0;

    return out;
};

/**
 * Calculates the determinant of a mat2
 *
 * @param {mat2} a the source matrix
 * @returns {Number} determinant of a
 */
mat2.determinant = function (a) {
    return a[0] * a[3] - a[2] * a[1];
};

/**
 * Multiplies two mat2's
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the first operand
 * @param {mat2} b the second operand
 * @returns {mat2} out
 */
mat2.multiply = function (out, a, b) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
    out[0] = a0 * b0 + a2 * b1;
    out[1] = a1 * b0 + a3 * b1;
    out[2] = a0 * b2 + a2 * b3;
    out[3] = a1 * b2 + a3 * b3;
    return out;
};

/**
 * Alias for {@link mat2.multiply}
 * @function
 */
mat2.mul = mat2.multiply;

/**
 * Rotates a mat2 by the given angle
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat2} out
 */
mat2.rotate = function (out, a, rad) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3],
        s = M.sin(rad),
        c = M.cos(rad);
    out[0] = a0 *  c + a2 * s;
    out[1] = a1 *  c + a3 * s;
    out[2] = a0 * -s + a2 * c;
    out[3] = a1 * -s + a3 * c;
    return out;
};

/**
 * Scales the mat2 by the dimensions in the given vec2
 *
 * @param {mat2} out the receiving matrix
 * @param {mat2} a the matrix to rotate
 * @param {vec2} v the vec2 to scale the matrix by
 * @returns {mat2} out
 **/
mat2.scale = function(out, a, v) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3],
        v0 = v[0], v1 = v[1];
    out[0] = a0 * v0;
    out[1] = a1 * v0;
    out[2] = a2 * v1;
    out[3] = a3 * v1;
    return out;
};

/**
 * Returns a string representation of a mat2
 *
 * @param {mat2} mat matrix to represent as a string
 * @returns {String} string representation of the matrix
 */
mat2.str = function (a) {
    return 'mat2(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
};

/**
 * Returns Frobenius norm of a mat2
 *
 * @param {mat2} a the matrix to calculate Frobenius norm of
 * @returns {Number} Frobenius norm
 */
mat2.frob = function (a) {
    return(M.sqrt(M.pow(a[0], 2) + M.pow(a[1], 2) + M.pow(a[2], 2) + M.pow(a[3], 2)))
};

/**
 * Returns L, D and U matrices (Lower triangular, Diagonal and Upper triangular) by factorizing the input matrix
 * @param {mat2} L the lower triangular matrix 
 * @param {mat2} D the diagonal matrix 
 * @param {mat2} U the upper triangular matrix 
 * @param {mat2} a the input matrix to factorize
 */

mat2.LDU = function (L, D, U, a) { 
    L[2] = a[2]/a[0]; 
    U[0] = a[0]; 
    U[1] = a[1]; 
    U[3] = a[3] - L[2] * U[1]; 
    return [L, D, U];       
}; 

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 2x3 Matrix
 * @name mat2d
 * 
 * @description 
 * A mat2d contains six elements defined as:
 * <pre>
 * [a, c, tx,
 *  b, d, ty]
 * </pre>
 * This is a short form for the 3x3 matrix:
 * <pre>
 * [a, c, tx,
 *  b, d, ty,
 *  0, 0, 1]
 * </pre>
 * The last row is ignored so the array is shorter and operations are faster.
 */

var mat2d = {};

/**
 * Creates a new identity mat2d
 *
 * @returns {mat2d} a new 2x3 matrix
 */
mat2d.create = function() {
    var out = new GLMAT_ARRAY_TYPE(6);
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    out[4] = 0;
    out[5] = 0;
    return out;
};

/**
 * Creates a new mat2d initialized with values from an existing matrix
 *
 * @param {mat2d} a matrix to clone
 * @returns {mat2d} a new 2x3 matrix
 */
mat2d.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(6);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    return out;
};

/**
 * Copy the values from one mat2d to another
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the source matrix
 * @returns {mat2d} out
 */
mat2d.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    return out;
};

/**
 * Set a mat2d to the identity matrix
 *
 * @param {mat2d} out the receiving matrix
 * @returns {mat2d} out
 */
mat2d.identity = function(out) {
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    out[4] = 0;
    out[5] = 0;
    return out;
};

/**
 * Inverts a mat2d
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the source matrix
 * @returns {mat2d} out
 */
mat2d.invert = function(out, a) {
    var aa = a[0], ab = a[1], ac = a[2], ad = a[3],
        atx = a[4], aty = a[5];

    var det = aa * ad - ab * ac;
    if(!det){
        return null;
    }
    det = 1.0 / det;

    out[0] = ad * det;
    out[1] = -ab * det;
    out[2] = -ac * det;
    out[3] = aa * det;
    out[4] = (ac * aty - ad * atx) * det;
    out[5] = (ab * atx - aa * aty) * det;
    return out;
};

/**
 * Calculates the determinant of a mat2d
 *
 * @param {mat2d} a the source matrix
 * @returns {Number} determinant of a
 */
mat2d.determinant = function (a) {
    return a[0] * a[3] - a[1] * a[2];
};

/**
 * Multiplies two mat2d's
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the first operand
 * @param {mat2d} b the second operand
 * @returns {mat2d} out
 */
mat2d.multiply = function (out, a, b) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5],
        b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3], b4 = b[4], b5 = b[5];
    out[0] = a0 * b0 + a2 * b1;
    out[1] = a1 * b0 + a3 * b1;
    out[2] = a0 * b2 + a2 * b3;
    out[3] = a1 * b2 + a3 * b3;
    out[4] = a0 * b4 + a2 * b5 + a4;
    out[5] = a1 * b4 + a3 * b5 + a5;
    return out;
};

/**
 * Alias for {@link mat2d.multiply}
 * @function
 */
mat2d.mul = mat2d.multiply;


/**
 * Rotates a mat2d by the given angle
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat2d} out
 */
mat2d.rotate = function (out, a, rad) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5],
        s = M.sin(rad),
        c = M.cos(rad);
    out[0] = a0 *  c + a2 * s;
    out[1] = a1 *  c + a3 * s;
    out[2] = a0 * -s + a2 * c;
    out[3] = a1 * -s + a3 * c;
    out[4] = a4;
    out[5] = a5;
    return out;
};

/**
 * Scales the mat2d by the dimensions in the given vec2
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the matrix to translate
 * @param {vec2} v the vec2 to scale the matrix by
 * @returns {mat2d} out
 **/
mat2d.scale = function(out, a, v) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5],
        v0 = v[0], v1 = v[1];
    out[0] = a0 * v0;
    out[1] = a1 * v0;
    out[2] = a2 * v1;
    out[3] = a3 * v1;
    out[4] = a4;
    out[5] = a5;
    return out;
};

/**
 * Translates the mat2d by the dimensions in the given vec2
 *
 * @param {mat2d} out the receiving matrix
 * @param {mat2d} a the matrix to translate
 * @param {vec2} v the vec2 to translate the matrix by
 * @returns {mat2d} out
 **/
mat2d.translate = function(out, a, v) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5],
        v0 = v[0], v1 = v[1];
    out[0] = a0;
    out[1] = a1;
    out[2] = a2;
    out[3] = a3;
    out[4] = a0 * v0 + a2 * v1 + a4;
    out[5] = a1 * v0 + a3 * v1 + a5;
    return out;
};

/**
 * Returns a string representation of a mat2d
 *
 * @param {mat2d} a matrix to represent as a string
 * @returns {String} string representation of the matrix
 */
mat2d.str = function (a) {
    return 'mat2d(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + 
                    a[3] + ', ' + a[4] + ', ' + a[5] + ')';
};

/**
 * Returns Frobenius norm of a mat2d
 *
 * @param {mat2d} a the matrix to calculate Frobenius norm of
 * @returns {Number} Frobenius norm
 */
mat2d.frob = function (a) { 
    return(M.sqrt(M.pow(a[0], 2) + M.pow(a[1], 2) + M.pow(a[2], 2) + M.pow(a[3], 2) + M.pow(a[4], 2) + M.pow(a[5], 2) + 1))
}; 

/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 3x3 Matrix
 * @name mat3
 */

var mat3 = {};

/**
 * Creates a new identity mat3
 *
 * @returns {mat3} a new 3x3 matrix
 */
mat3.create = function() {
    var out = new GLMAT_ARRAY_TYPE(9);
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 1;
    out[5] = 0;
    out[6] = 0;
    out[7] = 0;
    out[8] = 1;
    return out;
};

/**
 * Copies the upper-left 3x3 values into the given mat3.
 *
 * @param {mat3} out the receiving 3x3 matrix
 * @param {mat4} a   the source 4x4 matrix
 * @returns {mat3} out
 */
mat3.fromMat4 = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[4];
    out[4] = a[5];
    out[5] = a[6];
    out[6] = a[8];
    out[7] = a[9];
    out[8] = a[10];
    return out;
};

/**
 * Creates a new mat3 initialized with values from an existing matrix
 *
 * @param {mat3} a matrix to clone
 * @returns {mat3} a new 3x3 matrix
 */
mat3.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(9);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    out[6] = a[6];
    out[7] = a[7];
    out[8] = a[8];
    return out;
};

/**
 * Copy the values from one mat3 to another
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the source matrix
 * @returns {mat3} out
 */
mat3.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    out[6] = a[6];
    out[7] = a[7];
    out[8] = a[8];
    return out;
};

/**
 * Set a mat3 to the identity matrix
 *
 * @param {mat3} out the receiving matrix
 * @returns {mat3} out
 */
mat3.identity = function(out) {
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 1;
    out[5] = 0;
    out[6] = 0;
    out[7] = 0;
    out[8] = 1;
    return out;
};

/**
 * Transpose the values of a mat3
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the source matrix
 * @returns {mat3} out
 */
mat3.transpose = function(out, a) {
    // If we are transposing ourselves we can skip a few steps but have to cache some values
    if (out === a) {
        var a01 = a[1], a02 = a[2], a12 = a[5];
        out[1] = a[3];
        out[2] = a[6];
        out[3] = a01;
        out[5] = a[7];
        out[6] = a02;
        out[7] = a12;
    } else {
        out[0] = a[0];
        out[1] = a[3];
        out[2] = a[6];
        out[3] = a[1];
        out[4] = a[4];
        out[5] = a[7];
        out[6] = a[2];
        out[7] = a[5];
        out[8] = a[8];
    }
    
    return out;
};

/**
 * Inverts a mat3
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the source matrix
 * @returns {mat3} out
 */
mat3.invert = function(out, a) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8],

        b01 = a22 * a11 - a12 * a21,
        b11 = -a22 * a10 + a12 * a20,
        b21 = a21 * a10 - a11 * a20,

        // Calculate the determinant
        det = a00 * b01 + a01 * b11 + a02 * b21;

    if (!det) { 
        return null; 
    }
    det = 1.0 / det;

    out[0] = b01 * det;
    out[1] = (-a22 * a01 + a02 * a21) * det;
    out[2] = (a12 * a01 - a02 * a11) * det;
    out[3] = b11 * det;
    out[4] = (a22 * a00 - a02 * a20) * det;
    out[5] = (-a12 * a00 + a02 * a10) * det;
    out[6] = b21 * det;
    out[7] = (-a21 * a00 + a01 * a20) * det;
    out[8] = (a11 * a00 - a01 * a10) * det;
    return out;
};

/**
 * Calculates the adjugate of a mat3
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the source matrix
 * @returns {mat3} out
 */
mat3.adjoint = function(out, a) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8];

    out[0] = (a11 * a22 - a12 * a21);
    out[1] = (a02 * a21 - a01 * a22);
    out[2] = (a01 * a12 - a02 * a11);
    out[3] = (a12 * a20 - a10 * a22);
    out[4] = (a00 * a22 - a02 * a20);
    out[5] = (a02 * a10 - a00 * a12);
    out[6] = (a10 * a21 - a11 * a20);
    out[7] = (a01 * a20 - a00 * a21);
    out[8] = (a00 * a11 - a01 * a10);
    return out;
};

/**
 * Calculates the determinant of a mat3
 *
 * @param {mat3} a the source matrix
 * @returns {Number} determinant of a
 */
mat3.determinant = function (a) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8];

    return a00 * (a22 * a11 - a12 * a21) + a01 * (-a22 * a10 + a12 * a20) + a02 * (a21 * a10 - a11 * a20);
};

/**
 * Multiplies two mat3's
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the first operand
 * @param {mat3} b the second operand
 * @returns {mat3} out
 */
mat3.multiply = function (out, a, b) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8],

        b00 = b[0], b01 = b[1], b02 = b[2],
        b10 = b[3], b11 = b[4], b12 = b[5],
        b20 = b[6], b21 = b[7], b22 = b[8];

    out[0] = b00 * a00 + b01 * a10 + b02 * a20;
    out[1] = b00 * a01 + b01 * a11 + b02 * a21;
    out[2] = b00 * a02 + b01 * a12 + b02 * a22;

    out[3] = b10 * a00 + b11 * a10 + b12 * a20;
    out[4] = b10 * a01 + b11 * a11 + b12 * a21;
    out[5] = b10 * a02 + b11 * a12 + b12 * a22;

    out[6] = b20 * a00 + b21 * a10 + b22 * a20;
    out[7] = b20 * a01 + b21 * a11 + b22 * a21;
    out[8] = b20 * a02 + b21 * a12 + b22 * a22;
    return out;
};

/**
 * Alias for {@link mat3.multiply}
 * @function
 */
mat3.mul = mat3.multiply;

/**
 * Translate a mat3 by the given vector
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the matrix to translate
 * @param {vec2} v vector to translate by
 * @returns {mat3} out
 */
mat3.translate = function(out, a, v) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8],
        x = v[0], y = v[1];

    out[0] = a00;
    out[1] = a01;
    out[2] = a02;

    out[3] = a10;
    out[4] = a11;
    out[5] = a12;

    out[6] = x * a00 + y * a10 + a20;
    out[7] = x * a01 + y * a11 + a21;
    out[8] = x * a02 + y * a12 + a22;
    return out;
};

/**
 * Rotates a mat3 by the given angle
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat3} out
 */
mat3.rotate = function (out, a, rad) {
    var a00 = a[0], a01 = a[1], a02 = a[2],
        a10 = a[3], a11 = a[4], a12 = a[5],
        a20 = a[6], a21 = a[7], a22 = a[8],

        s = M.sin(rad),
        c = M.cos(rad);

    out[0] = c * a00 + s * a10;
    out[1] = c * a01 + s * a11;
    out[2] = c * a02 + s * a12;

    out[3] = c * a10 - s * a00;
    out[4] = c * a11 - s * a01;
    out[5] = c * a12 - s * a02;

    out[6] = a20;
    out[7] = a21;
    out[8] = a22;
    return out;
};

/**
 * Scales the mat3 by the dimensions in the given vec2
 *
 * @param {mat3} out the receiving matrix
 * @param {mat3} a the matrix to rotate
 * @param {vec2} v the vec2 to scale the matrix by
 * @returns {mat3} out
 **/
mat3.scale = function(out, a, v) {
    var x = v[0], y = v[1];

    out[0] = x * a[0];
    out[1] = x * a[1];
    out[2] = x * a[2];

    out[3] = y * a[3];
    out[4] = y * a[4];
    out[5] = y * a[5];

    out[6] = a[6];
    out[7] = a[7];
    out[8] = a[8];
    return out;
};

/**
 * Copies the values from a mat2d into a mat3
 *
 * @param {mat3} out the receiving matrix
 * @param {mat2d} a the matrix to copy
 * @returns {mat3} out
 **/
mat3.fromMat2d = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = 0;

    out[3] = a[2];
    out[4] = a[3];
    out[5] = 0;

    out[6] = a[4];
    out[7] = a[5];
    out[8] = 1;
    return out;
};

/**
* Calculates a 3x3 matrix from the given quaternion
*
* @param {mat3} out mat3 receiving operation result
* @param {quat} q Quaternion to create matrix from
*
* @returns {mat3} out
*/
mat3.fromQuat = function (out, q) {
    var x = q[0], y = q[1], z = q[2], w = q[3],
        x2 = x + x,
        y2 = y + y,
        z2 = z + z,

        xx = x * x2,
        yx = y * x2,
        yy = y * y2,
        zx = z * x2,
        zy = z * y2,
        zz = z * z2,
        wx = w * x2,
        wy = w * y2,
        wz = w * z2;

    out[0] = 1 - yy - zz;
    out[3] = yx - wz;
    out[6] = zx + wy;

    out[1] = yx + wz;
    out[4] = 1 - xx - zz;
    out[7] = zy - wx;

    out[2] = zx - wy;
    out[5] = zy + wx;
    out[8] = 1 - xx - yy;

    return out;
};

/**
* Calculates a 3x3 normal matrix (transpose inverse) from the 4x4 matrix
*
* @param {mat3} out mat3 receiving operation result
* @param {mat4} a Mat4 to derive the normal matrix from
*
* @returns {mat3} out
*/
mat3.normalFromMat4 = function (out, a) {
    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3],
        a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7],
        a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11],
        a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15],

        b00 = a00 * a11 - a01 * a10,
        b01 = a00 * a12 - a02 * a10,
        b02 = a00 * a13 - a03 * a10,
        b03 = a01 * a12 - a02 * a11,
        b04 = a01 * a13 - a03 * a11,
        b05 = a02 * a13 - a03 * a12,
        b06 = a20 * a31 - a21 * a30,
        b07 = a20 * a32 - a22 * a30,
        b08 = a20 * a33 - a23 * a30,
        b09 = a21 * a32 - a22 * a31,
        b10 = a21 * a33 - a23 * a31,
        b11 = a22 * a33 - a23 * a32,

        // Calculate the determinant
        det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;

    if (!det) { 
        return null; 
    }
    det = 1.0 / det;

    out[0] = (a11 * b11 - a12 * b10 + a13 * b09) * det;
    out[1] = (a12 * b08 - a10 * b11 - a13 * b07) * det;
    out[2] = (a10 * b10 - a11 * b08 + a13 * b06) * det;

    out[3] = (a02 * b10 - a01 * b11 - a03 * b09) * det;
    out[4] = (a00 * b11 - a02 * b08 + a03 * b07) * det;
    out[5] = (a01 * b08 - a00 * b10 - a03 * b06) * det;

    out[6] = (a31 * b05 - a32 * b04 + a33 * b03) * det;
    out[7] = (a32 * b02 - a30 * b05 - a33 * b01) * det;
    out[8] = (a30 * b04 - a31 * b02 + a33 * b00) * det;

    return out;
};

/**
 * Returns a string representation of a mat3
 *
 * @param {mat3} mat matrix to represent as a string
 * @returns {String} string representation of the matrix
 */
mat3.str = function (a) {
    return 'mat3(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + 
                    a[3] + ', ' + a[4] + ', ' + a[5] + ', ' + 
                    a[6] + ', ' + a[7] + ', ' + a[8] + ')';
};

/**
 * Returns Frobenius norm of a mat3
 *
 * @param {mat3} a the matrix to calculate Frobenius norm of
 * @returns {Number} Frobenius norm
 */
mat3.frob = function (a) {
    return(M.sqrt(M.pow(a[0], 2) + M.pow(a[1], 2) + M.pow(a[2], 2) + M.pow(a[3], 2) + M.pow(a[4], 2) + M.pow(a[5], 2) + M.pow(a[6], 2) + M.pow(a[7], 2) + M.pow(a[8], 2)))
};


/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class 4x4 Matrix
 * @name mat4
 */

var mat4 = {};

/**
 * Creates a new identity mat4
 *
 * @returns {mat4} a new 4x4 matrix
 */
mat4.create = function() {
    var out = new GLMAT_ARRAY_TYPE(16);
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 0;
    out[5] = 1;
    out[6] = 0;
    out[7] = 0;
    out[8] = 0;
    out[9] = 0;
    out[10] = 1;
    out[11] = 0;
    out[12] = 0;
    out[13] = 0;
    out[14] = 0;
    out[15] = 1;
    return out;
};

/**
 * Creates a new mat4 initialized with values from an existing matrix
 *
 * @param {mat4} a matrix to clone
 * @returns {mat4} a new 4x4 matrix
 */
mat4.clone = function(a) {
    var out = new GLMAT_ARRAY_TYPE(16);
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    out[6] = a[6];
    out[7] = a[7];
    out[8] = a[8];
    out[9] = a[9];
    out[10] = a[10];
    out[11] = a[11];
    out[12] = a[12];
    out[13] = a[13];
    out[14] = a[14];
    out[15] = a[15];
    return out;
};

/**
 * Copy the values from one mat4 to another
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the source matrix
 * @returns {mat4} out
 */
mat4.copy = function(out, a) {
    out[0] = a[0];
    out[1] = a[1];
    out[2] = a[2];
    out[3] = a[3];
    out[4] = a[4];
    out[5] = a[5];
    out[6] = a[6];
    out[7] = a[7];
    out[8] = a[8];
    out[9] = a[9];
    out[10] = a[10];
    out[11] = a[11];
    out[12] = a[12];
    out[13] = a[13];
    out[14] = a[14];
    out[15] = a[15];
    return out;
};

/**
 * Set a mat4 to the identity matrix
 *
 * @param {mat4} out the receiving matrix
 * @returns {mat4} out
 */
mat4.identity = function(out) {
    out[0] = 1;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 0;
    out[5] = 1;
    out[6] = 0;
    out[7] = 0;
    out[8] = 0;
    out[9] = 0;
    out[10] = 1;
    out[11] = 0;
    out[12] = 0;
    out[13] = 0;
    out[14] = 0;
    out[15] = 1;
    return out;
};

/**
 * Transpose the values of a mat4
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the source matrix
 * @returns {mat4} out
 */
mat4.transpose = function(out, a) {
    // If we are transposing ourselves we can skip a few steps but have to cache some values
    if (out === a) {
        var a01 = a[1], a02 = a[2], a03 = a[3],
            a12 = a[6], a13 = a[7],
            a23 = a[11];

        out[1] = a[4];
        out[2] = a[8];
        out[3] = a[12];
        out[4] = a01;
        out[6] = a[9];
        out[7] = a[13];
        out[8] = a02;
        out[9] = a12;
        out[11] = a[14];
        out[12] = a03;
        out[13] = a13;
        out[14] = a23;
    } else {
        out[0] = a[0];
        out[1] = a[4];
        out[2] = a[8];
        out[3] = a[12];
        out[4] = a[1];
        out[5] = a[5];
        out[6] = a[9];
        out[7] = a[13];
        out[8] = a[2];
        out[9] = a[6];
        out[10] = a[10];
        out[11] = a[14];
        out[12] = a[3];
        out[13] = a[7];
        out[14] = a[11];
        out[15] = a[15];
    }
    
    return out;
};

/**
 * Inverts a mat4
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the source matrix
 * @returns {mat4} out
 */
mat4.invert = function(out, a) {
    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3],
        a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7],
        a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11],
        a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15],

        b00 = a00 * a11 - a01 * a10,
        b01 = a00 * a12 - a02 * a10,
        b02 = a00 * a13 - a03 * a10,
        b03 = a01 * a12 - a02 * a11,
        b04 = a01 * a13 - a03 * a11,
        b05 = a02 * a13 - a03 * a12,
        b06 = a20 * a31 - a21 * a30,
        b07 = a20 * a32 - a22 * a30,
        b08 = a20 * a33 - a23 * a30,
        b09 = a21 * a32 - a22 * a31,
        b10 = a21 * a33 - a23 * a31,
        b11 = a22 * a33 - a23 * a32,

        // Calculate the determinant
        det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;

    if (!det) { 
        return null; 
    }
    det = 1.0 / det;

    out[0] = (a11 * b11 - a12 * b10 + a13 * b09) * det;
    out[1] = (a02 * b10 - a01 * b11 - a03 * b09) * det;
    out[2] = (a31 * b05 - a32 * b04 + a33 * b03) * det;
    out[3] = (a22 * b04 - a21 * b05 - a23 * b03) * det;
    out[4] = (a12 * b08 - a10 * b11 - a13 * b07) * det;
    out[5] = (a00 * b11 - a02 * b08 + a03 * b07) * det;
    out[6] = (a32 * b02 - a30 * b05 - a33 * b01) * det;
    out[7] = (a20 * b05 - a22 * b02 + a23 * b01) * det;
    out[8] = (a10 * b10 - a11 * b08 + a13 * b06) * det;
    out[9] = (a01 * b08 - a00 * b10 - a03 * b06) * det;
    out[10] = (a30 * b04 - a31 * b02 + a33 * b00) * det;
    out[11] = (a21 * b02 - a20 * b04 - a23 * b00) * det;
    out[12] = (a11 * b07 - a10 * b09 - a12 * b06) * det;
    out[13] = (a00 * b09 - a01 * b07 + a02 * b06) * det;
    out[14] = (a31 * b01 - a30 * b03 - a32 * b00) * det;
    out[15] = (a20 * b03 - a21 * b01 + a22 * b00) * det;

    return out;
};

/**
 * Calculates the adjugate of a mat4
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the source matrix
 * @returns {mat4} out
 */
mat4.adjoint = function(out, a) {
    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3],
        a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7],
        a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11],
        a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];

    out[0]  =  (a11 * (a22 * a33 - a23 * a32) - a21 * (a12 * a33 - a13 * a32) + a31 * (a12 * a23 - a13 * a22));
    out[1]  = -(a01 * (a22 * a33 - a23 * a32) - a21 * (a02 * a33 - a03 * a32) + a31 * (a02 * a23 - a03 * a22));
    out[2]  =  (a01 * (a12 * a33 - a13 * a32) - a11 * (a02 * a33 - a03 * a32) + a31 * (a02 * a13 - a03 * a12));
    out[3]  = -(a01 * (a12 * a23 - a13 * a22) - a11 * (a02 * a23 - a03 * a22) + a21 * (a02 * a13 - a03 * a12));
    out[4]  = -(a10 * (a22 * a33 - a23 * a32) - a20 * (a12 * a33 - a13 * a32) + a30 * (a12 * a23 - a13 * a22));
    out[5]  =  (a00 * (a22 * a33 - a23 * a32) - a20 * (a02 * a33 - a03 * a32) + a30 * (a02 * a23 - a03 * a22));
    out[6]  = -(a00 * (a12 * a33 - a13 * a32) - a10 * (a02 * a33 - a03 * a32) + a30 * (a02 * a13 - a03 * a12));
    out[7]  =  (a00 * (a12 * a23 - a13 * a22) - a10 * (a02 * a23 - a03 * a22) + a20 * (a02 * a13 - a03 * a12));
    out[8]  =  (a10 * (a21 * a33 - a23 * a31) - a20 * (a11 * a33 - a13 * a31) + a30 * (a11 * a23 - a13 * a21));
    out[9]  = -(a00 * (a21 * a33 - a23 * a31) - a20 * (a01 * a33 - a03 * a31) + a30 * (a01 * a23 - a03 * a21));
    out[10] =  (a00 * (a11 * a33 - a13 * a31) - a10 * (a01 * a33 - a03 * a31) + a30 * (a01 * a13 - a03 * a11));
    out[11] = -(a00 * (a11 * a23 - a13 * a21) - a10 * (a01 * a23 - a03 * a21) + a20 * (a01 * a13 - a03 * a11));
    out[12] = -(a10 * (a21 * a32 - a22 * a31) - a20 * (a11 * a32 - a12 * a31) + a30 * (a11 * a22 - a12 * a21));
    out[13] =  (a00 * (a21 * a32 - a22 * a31) - a20 * (a01 * a32 - a02 * a31) + a30 * (a01 * a22 - a02 * a21));
    out[14] = -(a00 * (a11 * a32 - a12 * a31) - a10 * (a01 * a32 - a02 * a31) + a30 * (a01 * a12 - a02 * a11));
    out[15] =  (a00 * (a11 * a22 - a12 * a21) - a10 * (a01 * a22 - a02 * a21) + a20 * (a01 * a12 - a02 * a11));
    return out;
};

/**
 * Calculates the determinant of a mat4
 *
 * @param {mat4} a the source matrix
 * @returns {Number} determinant of a
 */
mat4.determinant = function (a) {
    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3],
        a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7],
        a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11],
        a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15],

        b00 = a00 * a11 - a01 * a10,
        b01 = a00 * a12 - a02 * a10,
        b02 = a00 * a13 - a03 * a10,
        b03 = a01 * a12 - a02 * a11,
        b04 = a01 * a13 - a03 * a11,
        b05 = a02 * a13 - a03 * a12,
        b06 = a20 * a31 - a21 * a30,
        b07 = a20 * a32 - a22 * a30,
        b08 = a20 * a33 - a23 * a30,
        b09 = a21 * a32 - a22 * a31,
        b10 = a21 * a33 - a23 * a31,
        b11 = a22 * a33 - a23 * a32;

    // Calculate the determinant
    return b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
};

/**
 * Multiplies two mat4's
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the first operand
 * @param {mat4} b the second operand
 * @returns {mat4} out
 */
mat4.multiply = function (out, a, b) {
    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3],
        a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7],
        a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11],
        a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];

    // Cache only the current line of the second matrix
    var b0  = b[0], b1 = b[1], b2 = b[2], b3 = b[3];  
    out[0] = b0*a00 + b1*a10 + b2*a20 + b3*a30;
    out[1] = b0*a01 + b1*a11 + b2*a21 + b3*a31;
    out[2] = b0*a02 + b1*a12 + b2*a22 + b3*a32;
    out[3] = b0*a03 + b1*a13 + b2*a23 + b3*a33;

    b0 = b[4]; b1 = b[5]; b2 = b[6]; b3 = b[7];
    out[4] = b0*a00 + b1*a10 + b2*a20 + b3*a30;
    out[5] = b0*a01 + b1*a11 + b2*a21 + b3*a31;
    out[6] = b0*a02 + b1*a12 + b2*a22 + b3*a32;
    out[7] = b0*a03 + b1*a13 + b2*a23 + b3*a33;

    b0 = b[8]; b1 = b[9]; b2 = b[10]; b3 = b[11];
    out[8] = b0*a00 + b1*a10 + b2*a20 + b3*a30;
    out[9] = b0*a01 + b1*a11 + b2*a21 + b3*a31;
    out[10] = b0*a02 + b1*a12 + b2*a22 + b3*a32;
    out[11] = b0*a03 + b1*a13 + b2*a23 + b3*a33;

    b0 = b[12]; b1 = b[13]; b2 = b[14]; b3 = b[15];
    out[12] = b0*a00 + b1*a10 + b2*a20 + b3*a30;
    out[13] = b0*a01 + b1*a11 + b2*a21 + b3*a31;
    out[14] = b0*a02 + b1*a12 + b2*a22 + b3*a32;
    out[15] = b0*a03 + b1*a13 + b2*a23 + b3*a33;
    return out;
};

/**
 * Alias for {@link mat4.multiply}
 * @function
 */
mat4.mul = mat4.multiply;

/**
 * Translate a mat4 by the given vector
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to translate
 * @param {vec3} v vector to translate by
 * @returns {mat4} out
 */
mat4.translate = function (out, a, v) {
    var x = v[0], y = v[1], z = v[2],
        a00, a01, a02, a03,
        a10, a11, a12, a13,
        a20, a21, a22, a23;

    if (a === out) {
        out[12] = a[0] * x + a[4] * y + a[8] * z + a[12];
        out[13] = a[1] * x + a[5] * y + a[9] * z + a[13];
        out[14] = a[2] * x + a[6] * y + a[10] * z + a[14];
        out[15] = a[3] * x + a[7] * y + a[11] * z + a[15];
    } else {
        a00 = a[0]; a01 = a[1]; a02 = a[2]; a03 = a[3];
        a10 = a[4]; a11 = a[5]; a12 = a[6]; a13 = a[7];
        a20 = a[8]; a21 = a[9]; a22 = a[10]; a23 = a[11];

        out[0] = a00; out[1] = a01; out[2] = a02; out[3] = a03;
        out[4] = a10; out[5] = a11; out[6] = a12; out[7] = a13;
        out[8] = a20; out[9] = a21; out[10] = a22; out[11] = a23;

        out[12] = a00 * x + a10 * y + a20 * z + a[12];
        out[13] = a01 * x + a11 * y + a21 * z + a[13];
        out[14] = a02 * x + a12 * y + a22 * z + a[14];
        out[15] = a03 * x + a13 * y + a23 * z + a[15];
    }

    return out;
};

/**
 * Scales the mat4 by the dimensions in the given vec3
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to scale
 * @param {vec3} v the vec3 to scale the matrix by
 * @returns {mat4} out
 **/
mat4.scale = function(out, a, v) {
    var x = v[0], y = v[1], z = v[2];

    out[0] = a[0] * x;
    out[1] = a[1] * x;
    out[2] = a[2] * x;
    out[3] = a[3] * x;
    out[4] = a[4] * y;
    out[5] = a[5] * y;
    out[6] = a[6] * y;
    out[7] = a[7] * y;
    out[8] = a[8] * z;
    out[9] = a[9] * z;
    out[10] = a[10] * z;
    out[11] = a[11] * z;
    out[12] = a[12];
    out[13] = a[13];
    out[14] = a[14];
    out[15] = a[15];
    return out;
};

/**
 * Rotates a mat4 by the given angle
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @param {vec3} axis the axis to rotate around
 * @returns {mat4} out
 */
mat4.rotate = function (out, a, rad, axis) {
    var x = axis[0], y = axis[1], z = axis[2],
        len = M.sqrt(x * x + y * y + z * z),
        s, c, t,
        a00, a01, a02, a03,
        a10, a11, a12, a13,
        a20, a21, a22, a23,
        b00, b01, b02,
        b10, b11, b12,
        b20, b21, b22;

    if (M.abs(len) < GLMAT_EPSILON) { return null; }
    
    len = 1 / len;
    x *= len;
    y *= len;
    z *= len;

    s = M.sin(rad);
    c = M.cos(rad);
    t = 1 - c;

    a00 = a[0]; a01 = a[1]; a02 = a[2]; a03 = a[3];
    a10 = a[4]; a11 = a[5]; a12 = a[6]; a13 = a[7];
    a20 = a[8]; a21 = a[9]; a22 = a[10]; a23 = a[11];

    // Construct the elements of the rotation matrix
    b00 = x * x * t + c; b01 = y * x * t + z * s; b02 = z * x * t - y * s;
    b10 = x * y * t - z * s; b11 = y * y * t + c; b12 = z * y * t + x * s;
    b20 = x * z * t + y * s; b21 = y * z * t - x * s; b22 = z * z * t + c;

    // Perform rotation-specific matrix multiplication
    out[0] = a00 * b00 + a10 * b01 + a20 * b02;
    out[1] = a01 * b00 + a11 * b01 + a21 * b02;
    out[2] = a02 * b00 + a12 * b01 + a22 * b02;
    out[3] = a03 * b00 + a13 * b01 + a23 * b02;
    out[4] = a00 * b10 + a10 * b11 + a20 * b12;
    out[5] = a01 * b10 + a11 * b11 + a21 * b12;
    out[6] = a02 * b10 + a12 * b11 + a22 * b12;
    out[7] = a03 * b10 + a13 * b11 + a23 * b12;
    out[8] = a00 * b20 + a10 * b21 + a20 * b22;
    out[9] = a01 * b20 + a11 * b21 + a21 * b22;
    out[10] = a02 * b20 + a12 * b21 + a22 * b22;
    out[11] = a03 * b20 + a13 * b21 + a23 * b22;

    if (a !== out) { // If the source and destination differ, copy the unchanged last row
        out[12] = a[12];
        out[13] = a[13];
        out[14] = a[14];
        out[15] = a[15];
    }
    return out;
};

/**
 * Rotates a matrix by the given angle around the X axis
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat4} out
 */
mat4.rotateX = function (out, a, rad) {
    var s = M.sin(rad),
        c = M.cos(rad),
        a10 = a[4],
        a11 = a[5],
        a12 = a[6],
        a13 = a[7],
        a20 = a[8],
        a21 = a[9],
        a22 = a[10],
        a23 = a[11];

    if (a !== out) { // If the source and destination differ, copy the unchanged rows
        out[0]  = a[0];
        out[1]  = a[1];
        out[2]  = a[2];
        out[3]  = a[3];
        out[12] = a[12];
        out[13] = a[13];
        out[14] = a[14];
        out[15] = a[15];
    }

    // Perform axis-specific matrix multiplication
    out[4] = a10 * c + a20 * s;
    out[5] = a11 * c + a21 * s;
    out[6] = a12 * c + a22 * s;
    out[7] = a13 * c + a23 * s;
    out[8] = a20 * c - a10 * s;
    out[9] = a21 * c - a11 * s;
    out[10] = a22 * c - a12 * s;
    out[11] = a23 * c - a13 * s;
    return out;
};

/**
 * Rotates a matrix by the given angle around the Y axis
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat4} out
 */
mat4.rotateY = function (out, a, rad) {
    var s = M.sin(rad),
        c = M.cos(rad),
        a00 = a[0],
        a01 = a[1],
        a02 = a[2],
        a03 = a[3],
        a20 = a[8],
        a21 = a[9],
        a22 = a[10],
        a23 = a[11];

    if (a !== out) { // If the source and destination differ, copy the unchanged rows
        out[4]  = a[4];
        out[5]  = a[5];
        out[6]  = a[6];
        out[7]  = a[7];
        out[12] = a[12];
        out[13] = a[13];
        out[14] = a[14];
        out[15] = a[15];
    }

    // Perform axis-specific matrix multiplication
    out[0] = a00 * c - a20 * s;
    out[1] = a01 * c - a21 * s;
    out[2] = a02 * c - a22 * s;
    out[3] = a03 * c - a23 * s;
    out[8] = a00 * s + a20 * c;
    out[9] = a01 * s + a21 * c;
    out[10] = a02 * s + a22 * c;
    out[11] = a03 * s + a23 * c;
    return out;
};

/**
 * Rotates a matrix by the given angle around the Z axis
 *
 * @param {mat4} out the receiving matrix
 * @param {mat4} a the matrix to rotate
 * @param {Number} rad the angle to rotate the matrix by
 * @returns {mat4} out
 */
mat4.rotateZ = function (out, a, rad) {
    var s = M.sin(rad),
        c = M.cos(rad),
        a00 = a[0],
        a01 = a[1],
        a02 = a[2],
        a03 = a[3],
        a10 = a[4],
        a11 = a[5],
        a12 = a[6],
        a13 = a[7];

    if (a !== out) { // If the source and destination differ, copy the unchanged last row
        out[8]  = a[8];
        out[9]  = a[9];
        out[10] = a[10];
        out[11] = a[11];
        out[12] = a[12];
        out[13] = a[13];
        out[14] = a[14];
        out[15] = a[15];
    }

    // Perform axis-specific matrix multiplication
    out[0] = a00 * c + a10 * s;
    out[1] = a01 * c + a11 * s;
    out[2] = a02 * c + a12 * s;
    out[3] = a03 * c + a13 * s;
    out[4] = a10 * c - a00 * s;
    out[5] = a11 * c - a01 * s;
    out[6] = a12 * c - a02 * s;
    out[7] = a13 * c - a03 * s;
    return out;
};

/**
 * Creates a matrix from a quaternion rotation and vector translation
 * This is equivalent to (but much faster than):
 *
 *     mat4.identity(dest);
 *     mat4.translate(dest, vec);
 *     var quatMat = mat4.create();
 *     quat4.toMat4(quat, quatMat);
 *     mat4.multiply(dest, quatMat);
 *
 * @param {mat4} out mat4 receiving operation result
 * @param {quat4} q Rotation quaternion
 * @param {vec3} v Translation vector
 * @returns {mat4} out
 */
mat4.fromRotationTranslation = function (out, q, v) {
    // Quaternion math
    var x = q[0], y = q[1], z = q[2], w = q[3],
        x2 = x + x,
        y2 = y + y,
        z2 = z + z,

        xx = x * x2,
        xy = x * y2,
        xz = x * z2,
        yy = y * y2,
        yz = y * z2,
        zz = z * z2,
        wx = w * x2,
        wy = w * y2,
        wz = w * z2;

    out[0] = 1 - (yy + zz);
    out[1] = xy + wz;
    out[2] = xz - wy;
    out[3] = 0;
    out[4] = xy - wz;
    out[5] = 1 - (xx + zz);
    out[6] = yz + wx;
    out[7] = 0;
    out[8] = xz + wy;
    out[9] = yz - wx;
    out[10] = 1 - (xx + yy);
    out[11] = 0;
    out[12] = v[0];
    out[13] = v[1];
    out[14] = v[2];
    out[15] = 1;
    
    return out;
};

mat4.fromQuat = function (out, q) {
    var x = q[0], y = q[1], z = q[2], w = q[3],
        x2 = x + x,
        y2 = y + y,
        z2 = z + z,

        xx = x * x2,
        yx = y * x2,
        yy = y * y2,
        zx = z * x2,
        zy = z * y2,
        zz = z * z2,
        wx = w * x2,
        wy = w * y2,
        wz = w * z2;

    out[0] = 1 - yy - zz;
    out[1] = yx + wz;
    out[2] = zx - wy;
    out[3] = 0;

    out[4] = yx - wz;
    out[5] = 1 - xx - zz;
    out[6] = zy + wx;
    out[7] = 0;

    out[8] = zx + wy;
    out[9] = zy - wx;
    out[10] = 1 - xx - yy;
    out[11] = 0;

    out[12] = 0;
    out[13] = 0;
    out[14] = 0;
    out[15] = 1;

    return out;
};

/**
 * Generates a frustum matrix with the given bounds
 *
 * @param {mat4} out mat4 frustum matrix will be written into
 * @param {Number} left Left bound of the frustum
 * @param {Number} right Right bound of the frustum
 * @param {Number} bottom Bottom bound of the frustum
 * @param {Number} top Top bound of the frustum
 * @param {Number} near Near bound of the frustum
 * @param {Number} far Far bound of the frustum
 * @returns {mat4} out
 */
mat4.frustum = function (out, left, right, bottom, top, near, far) {
    var rl = 1 / (right - left),
        tb = 1 / (top - bottom),
        nf = 1 / (near - far);
    out[0] = (near * 2) * rl;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 0;
    out[5] = (near * 2) * tb;
    out[6] = 0;
    out[7] = 0;
    out[8] = (right + left) * rl;
    out[9] = (top + bottom) * tb;
    out[10] = (far + near) * nf;
    out[11] = -1;
    out[12] = 0;
    out[13] = 0;
    out[14] = (far * near * 2) * nf;
    out[15] = 0;
    return out;
};

/**
 * Generates a perspective projection matrix with the given bounds
 *
 * @param {mat4} out mat4 frustum matrix will be written into
 * @param {number} fovy Vertical field of view in radians
 * @param {number} aspect Aspect ratio. typically viewport width/height
 * @param {number} near Near bound of the frustum
 * @param {number} far Far bound of the frustum
 * @returns {mat4} out
 */
mat4.perspective = function (out, fovy, aspect, near, far) {
    var f = 1.0 / M.tan(fovy / 2),
        nf = 1 / (near - far);
    out[0] = f / aspect;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 0;
    out[5] = f;
    out[6] = 0;
    out[7] = 0;
    out[8] = 0;
    out[9] = 0;
    out[10] = (far + near) * nf;
    out[11] = -1;
    out[12] = 0;
    out[13] = 0;
    out[14] = (2 * far * near) * nf;
    out[15] = 0;
    return out;
};

/**
 * Generates a orthogonal projection matrix with the given bounds
 *
 * @param {mat4} out mat4 frustum matrix will be written into
 * @param {number} left Left bound of the frustum
 * @param {number} right Right bound of the frustum
 * @param {number} bottom Bottom bound of the frustum
 * @param {number} top Top bound of the frustum
 * @param {number} near Near bound of the frustum
 * @param {number} far Far bound of the frustum
 * @returns {mat4} out
 */
mat4.ortho = function (out, left, right, bottom, top, near, far) {
    var lr = 1 / (left - right),
        bt = 1 / (bottom - top),
        nf = 1 / (near - far);
    out[0] = -2 * lr;
    out[1] = 0;
    out[2] = 0;
    out[3] = 0;
    out[4] = 0;
    out[5] = -2 * bt;
    out[6] = 0;
    out[7] = 0;
    out[8] = 0;
    out[9] = 0;
    out[10] = 2 * nf;
    out[11] = 0;
    out[12] = (left + right) * lr;
    out[13] = (top + bottom) * bt;
    out[14] = (far + near) * nf;
    out[15] = 1;
    return out;
};

/**
 * Generates a look-at matrix with the given eye position, focal point, and up axis
 *
 * @param {mat4} out mat4 frustum matrix will be written into
 * @param {vec3} eye Position of the viewer
 * @param {vec3} center Point the viewer is looking at
 * @param {vec3} up vec3 pointing up
 * @returns {mat4} out
 */
mat4.lookAt = function (out, eye, center, up) {
    var x = [0,0,0];
    var y = [0,0,0];
    var z = [0,0,0];
    
    vec3.subtract(z, eye, center);
    vec3.normalize(z, z);
    vec3.cross(x, up, z);
    vec3.normalize(x, x);
    vec3.cross(y, z, x);

    out[0] = x[0];
    out[1] = y[0];
    out[2] = z[0];
    out[3] = 0;
    out[4] = x[1];
    out[5] = y[1];
    out[6] = z[1];
    out[7] = 0;
    out[8] = x[2];
    out[9] = y[2];
    out[10] = z[2];
    out[11] = 0;
    out[12] = -vec3.dot(x, eye);
    out[13] = -vec3.dot(y, eye);
    out[14] = -vec3.dot(z, eye);
    out[15] = 1;

    return out;
};

/**
 * Generates a look-at matrix with the given eye position, focal point, and tilt
 *
 * @param {mat4} out mat4 frustum matrix will be written into
 * @param {vec3} eye Position of the viewer
 * @param {vec3} center Point the viewer is looking at
 * @param {vec3} tilt Tilt angle (radians)
 * @returns {mat4} out
 */
mat4.lookAtTilt = function (out, eye, center, tilt) {
    var x = [0,0,0];
    var y = [0,0,0];
    var z = [0,0,0];
    
    vec3.subtract(z, eye, center);
    vec3.normalize(z, z);
    vec3.cross(x, [0,1,0], z);
    vec3.normalize(x, x);
    vec3.cross(y, z, x);
    vec3.scale(y, y, M.cos(tilt))
    vec3.scaleAndAdd(y, y, x, M.sin(tilt));
    vec3.cross(x, y, z);

    out[0] = x[0];
    out[1] = y[0];
    out[2] = z[0];
    out[3] = 0;
    out[4] = x[1];
    out[5] = y[1];
    out[6] = z[1];
    out[7] = 0;
    out[8] = x[2];
    out[9] = y[2];
    out[10] = z[2];
    out[11] = 0;
    out[12] = -vec3.dot(x, eye);
    out[13] = -vec3.dot(y, eye);
    out[14] = -vec3.dot(z, eye);
    out[15] = 1;

    return out;
};

/**
 * Returns a string representation of a mat4
 *
 * @param {mat4} mat matrix to represent as a string
 * @returns {String} string representation of the matrix
 */
mat4.str = function (a) {
    return 'mat4(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ', ' +
                    a[4] + ', ' + a[5] + ', ' + a[6] + ', ' + a[7] + ', ' +
                    a[8] + ', ' + a[9] + ', ' + a[10] + ', ' + a[11] + ', ' + 
                    a[12] + ', ' + a[13] + ', ' + a[14] + ', ' + a[15] + ')';
};

/**
 * Returns Frobenius norm of a mat4
 *
 * @param {mat4} a the matrix to calculate Frobenius norm of
 * @returns {Number} Frobenius norm
 */
mat4.frob = function (a) {
    return(M.sqrt(M.pow(a[0], 2) + M.pow(a[1], 2) + M.pow(a[2], 2) + M.pow(a[3], 2) + M.pow(a[4], 2) + M.pow(a[5], 2) + M.pow(a[6], 2) + M.pow(a[6], 2) + M.pow(a[7], 2) + M.pow(a[8], 2) + M.pow(a[9], 2) + M.pow(a[10], 2) + M.pow(a[11], 2) + M.pow(a[12], 2) + M.pow(a[13], 2) + M.pow(a[14], 2) + M.pow(a[15], 2) ))
};


/* Copyright (c) 2013, Brandon Jones, Colin MacKenzie IV. All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/**
 * @class Quaternion
 * @name quat
 */

var quat = {};

/**
 * Creates a new identity quat
 *
 * @returns {quat} a new quaternion
 */
quat.create = function() {
    var out = new GLMAT_ARRAY_TYPE(4);
    out[0] = 0;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    return out;
};

/**
 * Sets a quaternion to represent the shortest rotation from one
 * vector to another.
 *
 * Both vectors are assumed to be unit length.
 *
 * @param {quat} out the receiving quaternion.
 * @param {vec3} a the initial vector
 * @param {vec3} b the destination vector
 * @returns {quat} out
 */
quat.rotationTo = function(out, a, b) {
  var tmpvec3 = vec3.create();
  var dot = vec3.dot(a, b);
  if (dot < -0.999999) {
      vec3.cross(tmpvec3, [1,0,0], a);
      if (vec3.length(tmpvec3) < 0.000001)
          vec3.cross(tmpvec3, [0,1,0], a);
      vec3.normalize(tmpvec3, tmpvec3);
      quat.setAxisAngle(out, tmpvec3, M.PI);
      return out;
  } else if (dot > 0.999999) {
      out[0] = 0;
      out[1] = 0;
      out[2] = 0;
      out[3] = 1;
      return out;
  } else {
      vec3.cross(tmpvec3, a, b);
      out[0] = tmpvec3[0];
      out[1] = tmpvec3[1];
      out[2] = tmpvec3[2];
      out[3] = 1 + dot;
      return quat.normalize(out, out);
  }
}

/**
 * Sets the specified quaternion with values corresponding to the given
 * axes. Each axis is a vec3 and is expected to be unit length and
 * perpendicular to all other specified axes.
 *
 * @param {vec3} view  the vector representing the viewing direction
 * @param {vec3} right the vector representing the local "right" direction
 * @param {vec3} up    the vector representing the local "up" direction
 * @returns {quat} out
 */
quat.setAxes = function(out, view, right, up) {
    var matr = mat3.create();
    matr[0] = right[0];
    matr[3] = right[1];
    matr[6] = right[2];

    matr[1] = up[0];
    matr[4] = up[1];
    matr[7] = up[2];

    matr[2] = -view[0];
    matr[5] = -view[1];
    matr[8] = -view[2];

    return quat.normalize(out, quat.fromMat3(out, matr));
};

/**
 * Creates a new quat initialized with values from an existing quaternion
 *
 * @param {quat} a quaternion to clone
 * @returns {quat} a new quaternion
 * @function
 */
quat.clone = vec4.clone;

/**
 * Creates a new quat initialized with the given values
 *
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @param {Number} w W component
 * @returns {quat} a new quaternion
 * @function
 */
quat.fromValues = vec4.fromValues;

/**
 * Copy the values from one quat to another
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a the source quaternion
 * @returns {quat} out
 * @function
 */
quat.copy = vec4.copy;

/**
 * Set the components of a quat to the given values
 *
 * @param {quat} out the receiving quaternion
 * @param {Number} x X component
 * @param {Number} y Y component
 * @param {Number} z Z component
 * @param {Number} w W component
 * @returns {quat} out
 * @function
 */
quat.set = vec4.set;

/**
 * Set a quat to the identity quaternion
 *
 * @param {quat} out the receiving quaternion
 * @returns {quat} out
 */
quat.identity = function(out) {
    out[0] = 0;
    out[1] = 0;
    out[2] = 0;
    out[3] = 1;
    return out;
};

/**
 * Sets a quat from the given angle and rotation axis,
 * then returns it.
 *
 * @param {quat} out the receiving quaternion
 * @param {vec3} axis the axis around which to rotate
 * @param {Number} rad the angle in radians
 * @returns {quat} out
 **/
quat.setAxisAngle = function(out, axis, rad) {
    rad = rad * 0.5;
    var s = M.sin(rad);
    out[0] = s * axis[0];
    out[1] = s * axis[1];
    out[2] = s * axis[2];
    out[3] = M.cos(rad);
    return out;
};

/**
 * Adds two quat's
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a the first operand
 * @param {quat} b the second operand
 * @returns {quat} out
 * @function
 */
quat.add = vec4.add;

/**
 * Multiplies two quat's
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a the first operand
 * @param {quat} b the second operand
 * @returns {quat} out
 */
quat.multiply = function(out, a, b) {
    var ax = a[0], ay = a[1], az = a[2], aw = a[3],
        bx = b[0], by = b[1], bz = b[2], bw = b[3];

    out[0] = ax * bw + aw * bx + ay * bz - az * by;
    out[1] = ay * bw + aw * by + az * bx - ax * bz;
    out[2] = az * bw + aw * bz + ax * by - ay * bx;
    out[3] = aw * bw - ax * bx - ay * by - az * bz;
    return out;
};

/**
 * Alias for {@link quat.multiply}
 * @function
 */
quat.mul = quat.multiply;

/**
 * Scales a quat by a scalar number
 *
 * @param {quat} out the receiving vector
 * @param {quat} a the vector to scale
 * @param {Number} b amount to scale the vector by
 * @returns {quat} out
 * @function
 */
quat.scale = vec4.scale;

/**
 * Rotates a quaternion by the given angle about the X axis
 *
 * @param {quat} out quat receiving operation result
 * @param {quat} a quat to rotate
 * @param {number} rad angle (in radians) to rotate
 * @returns {quat} out
 */
quat.rotateX = function (out, a, rad) {
    rad *= 0.5; 

    var ax = a[0], ay = a[1], az = a[2], aw = a[3],
        bx = M.sin(rad), bw = M.cos(rad);

    out[0] = ax * bw + aw * bx;
    out[1] = ay * bw + az * bx;
    out[2] = az * bw - ay * bx;
    out[3] = aw * bw - ax * bx;
    return out;
};

/**
 * Rotates a quaternion by the given angle about the Y axis
 *
 * @param {quat} out quat receiving operation result
 * @param {quat} a quat to rotate
 * @param {number} rad angle (in radians) to rotate
 * @returns {quat} out
 */
quat.rotateY = function (out, a, rad) {
    rad *= 0.5; 

    var ax = a[0], ay = a[1], az = a[2], aw = a[3],
        by = M.sin(rad), bw = M.cos(rad);

    out[0] = ax * bw - az * by;
    out[1] = ay * bw + aw * by;
    out[2] = az * bw + ax * by;
    out[3] = aw * bw - ay * by;
    return out;
};

/**
 * Rotates a quaternion by the given angle about the Z axis
 *
 * @param {quat} out quat receiving operation result
 * @param {quat} a quat to rotate
 * @param {number} rad angle (in radians) to rotate
 * @returns {quat} out
 */
quat.rotateZ = function (out, a, rad) {
    rad *= 0.5; 

    var ax = a[0], ay = a[1], az = a[2], aw = a[3],
        bz = M.sin(rad), bw = M.cos(rad);

    out[0] = ax * bw + ay * bz;
    out[1] = ay * bw - ax * bz;
    out[2] = az * bw + aw * bz;
    out[3] = aw * bw - az * bz;
    return out;
};

/**
 * Calculates the W component of a quat from the X, Y, and Z components.
 * Assumes that quaternion is 1 unit in length.
 * Any existing W component will be ignored.
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a quat to calculate W component of
 * @returns {quat} out
 */
quat.calculateW = function (out, a) {
    var x = a[0], y = a[1], z = a[2];

    out[0] = x;
    out[1] = y;
    out[2] = z;
    out[3] = -M.sqrt(M.abs(1.0 - x * x - y * y - z * z));
    return out;
};

/**
 * Calculates the dot product of two quat's
 *
 * @param {quat} a the first operand
 * @param {quat} b the second operand
 * @returns {Number} dot product of a and b
 * @function
 */
quat.dot = vec4.dot;

/**
 * Performs a linear interpolation between two quat's
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a the first operand
 * @param {quat} b the second operand
 * @param {Number} t interpolation amount between the two inputs
 * @returns {quat} out
 * @function
 */
quat.lerp = vec4.lerp;

/**
 * Performs a spherical linear interpolation between two quat
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a the first operand
 * @param {quat} b the second operand
 * @param {Number} t interpolation amount between the two inputs
 * @returns {quat} out
 */
quat.slerp = function (out, a, b, t) {
    // benchmarks:
    //    http://jsperf.com/quaternion-slerp-implementations

    var ax = a[0], ay = a[1], az = a[2], aw = a[3],
        bx = b[0], by = b[1], bz = b[2], bw = b[3];

    var        omega, cosom, sinom, scale0, scale1;

    // calc cosine
    cosom = ax * bx + ay * by + az * bz + aw * bw;
    // adjust signs (if necessary)
    if ( cosom < 0.0 ) {
        cosom = -cosom;
        bx = - bx;
        by = - by;
        bz = - bz;
        bw = - bw;
    }
    // calculate coefficients
    if ( (1.0 - cosom) > 0.000001 ) {
        // standard case (slerp)
        omega  = M.acos(cosom);
        sinom  = M.sin(omega);
        scale0 = M.sin((1.0 - t) * omega) / sinom;
        scale1 = M.sin(t * omega) / sinom;
    } else {        
        // "from" and "to" quaternions are very close 
        //  ... so we can do a linear interpolation
        scale0 = 1.0 - t;
        scale1 = t;
    }
    // calculate final values
    out[0] = scale0 * ax + scale1 * bx;
    out[1] = scale0 * ay + scale1 * by;
    out[2] = scale0 * az + scale1 * bz;
    out[3] = scale0 * aw + scale1 * bw;
    
    return out;
};

/**
 * Calculates the inverse of a quat
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a quat to calculate inverse of
 * @returns {quat} out
 */
quat.invert = function(out, a) {
    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3],
        dot = a0*a0 + a1*a1 + a2*a2 + a3*a3,
        invDot = dot ? 1.0/dot : 0;
    
    // TODO: Would be faster to return [0,0,0,0] immediately if dot == 0

    out[0] = -a0*invDot;
    out[1] = -a1*invDot;
    out[2] = -a2*invDot;
    out[3] = a3*invDot;
    return out;
};

/**
 * Calculates the conjugate of a quat
 * If the quaternion is normalized, this function is faster than quat.inverse and produces the same result.
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a quat to calculate conjugate of
 * @returns {quat} out
 */
quat.conjugate = function (out, a) {
    out[0] = -a[0];
    out[1] = -a[1];
    out[2] = -a[2];
    out[3] = a[3];
    return out;
};

/**
 * Calculates the length of a quat
 *
 * @param {quat} a vector to calculate length of
 * @returns {Number} length of a
 * @function
 */
quat.length = vec4.length;

/**
 * Alias for {@link quat.length}
 * @function
 */
quat.len = quat.length;

/**
 * Calculates the squared length of a quat
 *
 * @param {quat} a vector to calculate squared length of
 * @returns {Number} squared length of a
 * @function
 */
quat.squaredLength = vec4.squaredLength;

/**
 * Alias for {@link quat.squaredLength}
 * @function
 */
quat.sqrLen = quat.squaredLength;

/**
 * Normalize a quat
 *
 * @param {quat} out the receiving quaternion
 * @param {quat} a quaternion to normalize
 * @returns {quat} out
 * @function
 */
quat.normalize = vec4.normalize;

/**
 * Creates a quaternion from the given 3x3 rotation matrix.
 *
 * NOTE: The resultant quaternion is not normalized, so you should be sure
 * to renormalize the quaternion yourself where necessary.
 *
 * @param {quat} out the receiving quaternion
 * @param {mat3} m rotation matrix
 * @returns {quat} out
 * @function
 */
quat.fromMat3 = function(out, m) {
    // Algorithm in Ken Shoemake's article in 1987 SIGGRAPH course notes
    // article "Quaternion Calculus and Fast Animation".
    var fTrace = m[0] + m[4] + m[8];
    var fRoot;

    if ( fTrace > 0.0 ) {
        // |w| > 1/2, may as well choose w > 1/2
        fRoot = M.sqrt(fTrace + 1.0);  // 2w
        out[3] = 0.5 * fRoot;
        fRoot = 0.5/fRoot;  // 1/(4w)
        out[0] = (m[7]-m[5])*fRoot;
        out[1] = (m[2]-m[6])*fRoot;
        out[2] = (m[3]-m[1])*fRoot;
    } else {
        // |w| <= 1/2
        var i = 0;
        if ( m[4] > m[0] )
          i = 1;
        if ( m[8] > m[i*3+i] )
          i = 2;
        var j = (i+1)%3;
        var k = (i+2)%3;
        
        fRoot = M.sqrt(m[i*3+i]-m[j*3+j]-m[k*3+k] + 1.0);
        out[i] = 0.5 * fRoot;
        fRoot = 0.5 / fRoot;
        out[3] = (m[k*3+j] - m[j*3+k]) * fRoot;
        out[j] = (m[j*3+i] + m[i*3+j]) * fRoot;
        out[k] = (m[k*3+i] + m[i*3+k]) * fRoot;
    }
    
    return out;
};

/**
 * Returns a string representation of a quatenion
 *
 * @param {quat} vec vector to represent as a string
 * @returns {String} string representation of the vector
 */
quat.str = function (a) {
    return 'quat(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
};


function paris_subdivision_rec(paths, num_subdivs, sub_id) {
    if (sub_id < 0) { sub_id = 0; }
    var sub_paths = [];
    for (var i in paths) {
        var sub = city_subdivision(paths[i], sub_id)
        if (!sub) {
            sub_paths.push(paths[i]);
        }
        else {
            sub_paths.push(sub[0], sub[1]);
        }
    }
    if (num_subdivs == 1) {
        return sub_paths;
    }
    return paris_subdivision_rec(sub_paths, num_subdivs - 1, sub_id - 1);
}


// TODO make this show in the editor: it defines how the min size of city blocks
var MIN_PERIMETER = 260;

function paris_subdivision(path, sub_id) {
    var path_length = path.length;

    // a1 is the index of the point starting the first edge we'll cut.
    // b1 is the index of the point starting the second edge we'll cut.
    var a1;
    var maxd = 0;
    var perimeter = 0;
    var i; // loop index, taken out to win a few bytes
    // pick the longest segment
    for (i = 0; i < path_length; ++i) {
        var d = vec2.distance(path[i], path[(i+1)%path_length]);
        if (d > maxd) {
            maxd = d;
            a1 = i;
        }
        perimeter += d;
    }

    if (perimeter < MIN_PERIMETER) { return null; }

    var a2 = (a1+1) % path_length;
    var b1, b2;

    do {
        b1 = rand_int(path_length);
        if (a1 == b1 || a1 == b1 + 1) { continue; }

        b2 = (b1+1) % path_length;

        var f1 = 0.5 + (0.5 - M.abs(seedable_random() - 0.5)) * 0.2;
        var f2 = 0.5 + (0.5 - M.abs(seedable_random() - 0.5)) * 0.2;

        var p_a3_1 = { '0': path[a1][0]*f1 + path[a2][0]*(1.0-f1), '1': path[a1][1]*f1 + path[a2][1]*(1-f1), subdiv: sub_id};
        var p_a3_2 = { '0': path[a1][0]*f1 + path[a2][0]*(1.0-f1), '1': path[a1][1]*f1 + path[a2][1]*(1-f1), subdiv: path[a1].subdiv};
        var p_b3_1 = { '0': path[b1][0]*f2 + path[b2][0]*(1.0-f2), '1': path[b1][1]*f2 + path[b2][1]*(1-f2), subdiv: sub_id};
        var p_b3_2 = { '0': path[b1][0]*f2 + path[b2][0]*(1.0-f2), '1': path[b1][1]*f2 + path[b2][1]*(1-f2), subdiv: path[b1].subdiv};

        break;
    } while (1);

    var path1 = [p_a3_1, p_b3_2]
    for (i = b2; i != a2; i = mod((i+1), path_length)) {
        path1.push(path[i]);
    }

    var path2 = [p_b3_1, p_a3_2]
    for (i = a2; i != b2; i = mod((i+1), path_length)) {
        path2.push(path[i]);
    }

    return [path1, path2];
}

function plazza(path, pos, rad) {
    for (p=0; p<path.length; ++p) {
      if (vec2.distance(path[p], pos) < rad) {
        return true;
      }
    }
    return false;
}



function text_init() {
  canvas_2d = document.createElement("canvas");
  canvas_2d.width = canvas_2d.height = 2048;
  ctx_2d = canvas_2d.getContext("2d");
  minify_context(ctx_2d);
  ctx_2d.textAlign = "center";
  ctx_2d.fillStyle = "#fff";
}

function clear_texture_canvas() {
  ctx_2d.clearRect(0, 0, 2048, 2048);
}

function texture_fill_rect(x, y, w, h, style) {
  var sz = 2048;
  ctx_2d.fillStyle = style;
  ctx_2d.fillRect(x*sz, y*sz, w*sz, h*sz);
}

function create_text_texture(fontSize, text) {
  clear_texture_canvas();
  
  fontSize *= 100;
  ctx_2d.font = fontSize + "px Calibri";

  var measure = ctx_2d.measureText(text);
  var width = 3 + measure.width|0,
    height = fontSize * 1.5;
	
  ctx_2d.fillText(text, width / 2, fontSize);
  
  return create_texture(
    width, height, gl.RGBA,
    ctx_2d.getImageData(0, 0, width, height).data,
    false, true
  );
}

function create_vertical_text_texture(fontSize, text) {
  clear_texture_canvas();
  
  fontSize *= 100;
  ctx_2d.font = fontSize + "px Calibri";

  var width = fontSize,
    height = fontSize;
	
  for (var i = 0; i < text.length; ++i) {
	ctx_2d.fillText(text[i], width / 2, height);
	height += fontSize * 0.7;
  }
  
  height += fontSize * 0.3;
  
  return create_texture(
    width, height, gl.RGBA,
    ctx_2d.getImageData(0, 0, width, height).data,
    false, true
  );
}  // change that to true to log
  function log() {
    // console.log.apply(console, arguments);
  }
  function editing() { return false; }
  function n2f(n) {
    return M.pow(2, (n - 69) / 12) * 440;
  }

  AudioNode.prototype.c = AudioNode.prototype.connect;

  ac = new AudioContext();
  minify_context(ac);

  /** @constructor */
  function SND() {
    log('SND.constr', this);
    this.playing = false;
  };
  
  SND.AD = function(p/*aram*/, l/*start*/, u/*end*/, t/*startTime*/, a/*attack*/, d/*decay*/) {
    p.setValueAtTime(l, t);
    p.linearRampToValueAtTime(u, t + a);
    // XXX change that to setTargetAtTime
    p.linearRampToValueAtTime(l, t + d);
  };
  SND.D = function(p, t, v, k) {
    p.value = v;
    p.setValueAtTime(v, t);
    p.setTargetAtTime(0, t, k);
  }
  SND.DCA = function(i, v, t, a, d) {
    var g = ac.createGain();
    i.c(g);
    SND.AD(g.gain, 0, v, t, a, d);
    return g;
  };
  function NoiseBuffer() {
    var i,l;
    if (!SND._noisebuffer) {
      SND._noisebuffer = ac.createBuffer(1, ac.sampleRate * 0.5, ac.sampleRate / 2);
      var cdata = SND._noisebuffer.getChannelData(0);
      for(i=0,l=cdata.length;i<l;i++) {
        cdata[i] = M.random() * 2.0 - 1.0;
      }
    }
    return SND._noisebuffer;
  }
  SND.ReverbBuffer = function(opts) {
    var i,l;
    var len = ac.sampleRate * opts.l
    var buffer = ac.createBuffer(2, len, ac.sampleRate)
    for(i=0,l=buffer.length;i<l;i++) {
      var s =  M.pow(1 - i / len, opts.d);
      buffer.getChannelData(0)[i] = (M.random() * 2 - 1)*2;
      buffer.getChannelData(1)[i] = (M.random() * 2 - 1)*2;
    }
    return buffer;
  }

  SND.DistCurve = function(k) {
    var c = new Float32Array(ac.sampleRate);
    var deg = M.PI / 180;
    for (var i = 0; i < c.length; i++) {
      var x = i * 2 / c.length - 1;
      c[i] = (3 + k) * x * 20 * deg / (M.PI + k * M.abs(x));
    }
    return c;
  }
  SND.setSends = function(sendGains, out) {
    sends.forEach(function(send, i) {
      var amp = ac.createGain();
      amp.gain.value = sendGains[i] || 0.0;
      out.c(amp);
      amp.c(send);
    });
  };

  // In fractional beat
  SND.prototype.t = function() {
    return (ac.currentTime - this.startTime) * (125/ 60);
  }

  SND.prototype.p = function() {
    if (this.playing == true) return;
    if (!this.startTime) this.startTime = ac.currentTime;
    var stepTime = 15 / 125,
        patternTime = stepTime * 64,
        currentTime = ac.currentTime;

    this.currentPos = 0;
    if (editing()) {
      // the patter to loop, or -1 to just play the track
      this.loop = this.loop != undefined ? this.loop : -1;
      // start at the loop if specified, beginning otherwise
      this.currentPos = this.loop != -1 ? this.loop : 0;
    }

    this.playing = true;

    var patternScheduler = (function() {
      if (this.playing == false) return;
      if (currentTime - ac.currentTime < (patternTime / 4)) {
        SND.st = [];
        for(i=0;i<64;i++) { SND.st[i] = currentTime + (stepTime * i); }
        if (SONG.playlist.length == this.currentPos) {
          return;
        }
        var cP = SONG.playlist[this.currentPos];
        log(cP);
        for (var instrId in cP) {
          if (cP.hasOwnProperty(instrId)) {
            log("scheduling", cP[instrId], "for", instrId)
            var data = SONG.patterns[cP[instrId]];
            SND.playPattern(instruments[instrId], SND.st, stepTime, data); 
          }
        }
        if (editing()) {
          if (this.loop == -1) {
            this.currentPos = (this.currentPos + 1) % SONG.playlist.length;
          } else {
            this.currentPos = this.loop;
          }
        } else{
          this.currentPos++;
        }
        currentTime += patternTime;
      }
      setTimeout(patternScheduler, 1000);
    }).bind(this);
    patternScheduler();
  };
  SND.prototype.s = function() {
    this.playing = false;
  }
  
  // SEND EFFECTS
  
  /** @constructor */
  SND.DEL = function() {
    var opts = {t: 0.36, fb: 0.4, m: 0.6, f: 800, q: 2};
    var delay = ac.createDelay();
    delay.delayTime.value = opts.t;
    var fb = ac.createGain();
    var flt = ac.createBiquadFilter();
    flt.type = 'highpass';
    flt.frequency.value = opts.f;
    flt.Q.value = opts.q;
    fb.gain.value = opts.fb;
    var mix = ac.createGain();
    mix.gain.value = opts.m;
    delay.c(mix);
    delay.c(flt);
    flt.c(fb);
    fb.c(delay);
    mix.c(ac.destination);
    return delay;
  }
  
  /** @constructor */
  SND.REV = function() {
    var opts = {d: 0.05, m: 1};
    var cnv = ac.createConvolver();
    var mix = ac.createGain();
    cnv.buffer = SND.ReverbBuffer({l: 2, d: opts.d});
    mix.gain.value = opts.m;
    cnv.c(mix);
    mix.c(ac.destination);
    return cnv;
  }

  /** @constructor */
  SND.DIST = function() {
    var ws = ac.createWaveShaper();
    mix = ac.createGain();
    ws.curve = SND.DistCurve(50);
    mix.gain.value = 0.5;
    ws.c(mix);
    mix.c(ac.destination);
    return ws;
  }
  
  // INSTRUMENTS
  
  SND.playPattern = function(instrument, times, stepTime, data) {
    times.forEach(function(t, i) {
      note = data[i];
      if (typeof(note) !== 'object') {
        note = [note, {}]
      }
      if (note[0] != 0) {
        instrument(t, stepTime, note);
      }
    });
  };
  
  var noise = NoiseBuffer();
  SND.Noise = function(t) {
    var smp = ac.createBufferSource();
    var flt = ac.createBiquadFilter();
    smp.c(flt);
    var amp = SND.DCA(flt, 0.1, t, 0.001, 0.06);
    flt.frequency.value = 8000;
    flt.type = "highpass";
    flt.Q.value = 8;
    smp.buffer = noise;
    smp.c(amp);
    SND.setSends([0.3], amp);
    amp.c(ac.destination);
    smp.start(t);smp.stop(t + 0.06);
  }
  
  SND.Drum = function(t) {
    var osc = ac.createOscillator();
    var click = ac.createOscillator();
    click.type = "square";
    click.frequency.value = 40;

    // SND.AD(osc.frequency, opts.en, opts.st, t, 0, opts.k * 8);
    osc.frequency.value = 90;
    osc.frequency.setValueAtTime(90, t);
    osc.frequency.setTargetAtTime(50, t+0.001, 0.03)

    function d(o, e){
      var amp = ac.createGain();
      o.c(amp);
      SND.D(amp.gain, t, 1.3, e);
      amp.c(ac.destination);
    }

    d(osc, 0.03)
    d(click, 0.005)

    osc.start(t);osc.stop(t + 0.2);
    click.start(t);click.stop(t + 0.009);
  }

  SND.Snare = function(t) {
    var f = [111 + 175, 111 + 224];
    var o = [];

    // filter for noise and osc
    var fl = ac.createBiquadFilter();
    // fl.type = "lowpass" // default
    fl.frequency.value = 3000;

    // amp for oscillator
    var amposc = ac.createGain();
    SND.D(amposc.gain, t, 0.4, 0.015);

    // two osc
    f.forEach(function(e, i) {
      o[i] = ac.createOscillator();
      o[i].type = "triangle";
      o[i].frequency.value = f[i];
      o[i].c(amposc);
      o[i].start(t); o[i].stop(t + 0.4);
    })

    // noise
    var smp = ac.createBufferSource();
    smp.buffer = noise;
    var ampnoise = ac.createGain();
    smp.c(ampnoise);
    SND.D(ampnoise.gain, t, 0.24, 0.045);
    smp.start(t);smp.stop(t + 0.1);

    ampnoise.c(fl);
    amposc.c(fl);

    SND.setSends([0.3, 0.2], fl);
    fl.c(ac.destination);
  }
  
  SND.Synth = function(t, stepTime, data) {
    var osc = ac.createOscillator();
    var flt = ac.createBiquadFilter();
    flt.Q.value = 2;
    osc.frequency.value = n2f(data[0]);
    osc.type = "square"
    len = stepTime * (data[1].l || 1);
    osc.c(flt);
    var amp = SND.DCA(flt, data[1].v || 0.1, t, 0.01, len);
    SND.setSends([0.5, 0.6], amp);
    amp.c(ac.destination);
    SND.AD(flt.frequency, 200, 2000, t, 0.01, len / 2);
    osc.start(t);osc.stop(t + len);
  }

  SND.Sub = function(t, stepTime, data) {
    var osc = ac.createOscillator();
    osc.frequency.value = n2f(data[0]);
    len = stepTime * data[1].l;
    // len = stepTime * (data[1].l || 1);
    var amp = SND.DCA(osc, 0.6, t, 0.05, len);
    amp.c(ac.destination);
    osc.start(t);osc.stop(t + len);
  }

  SND.Reese = function(t, stepTime, data) {
    var note = data[0];
    var len = stepTime * data[1].l;

    var flt = ac.createBiquadFilter();
    var o = ac.createOscillator();
    o.frequency.value = data[1].f * (125 / 120); // fetch tempo here.
    var s = ac.createGain();
    s.gain.value = 8000;
    o.c(s);
    s.c(flt.frequency);
    o.start(t); o.stop(t + 10); // long tail
    amp = SND.DCA(flt, data[1].v, t, 0, len);
    for (var i = 0; i < 2; i++) {
      o = ac.createOscillator();
      o.frequency.value = n2f(note);
      o.type = "square";
      o.detune.value = i * 50;
      o.c(flt);
      o.start(t);o.stop(t+len);
    }
    amp.c(ac.destination)
    SND.setSends([0,0.4,1], amp);
  }

  SND.Glitch = function(t, stepTime, data) {
    var len = (data[1].l || 1) * stepTime;
    var source = ac.createBufferSource();
    var end = t + len;
    var sources = [];
    var i = 0;
    var sink = ac.createGain();
    sink.gain.value = 0.05;
    while (t < end) {
      sources[i] = ac.createBufferSource();
      sources[i].buffer = noise;
      sources[i].loop = true;
      sources[i].loopStart = 0;
      sources[i].loopEnd = M.random() * 0.05;
      sources[i].start(t);
      t += M.random() * 0.5;
      t = M.min(t, end);
      sources[i].stop(t);
      sources[i].c(sink);
      i++;
    }
    sink.c(ac.destination);
    SND.setSends([0.3, 0.8], sink);
  }





////////////////// Audio tag implementation

function SND_Ogg(name, data)
{
  var src, audioElement;

  if (data)
    src = "/data/project/tdf15/asset/" + name + ".ogg"
  else
    src = name + ".ogg"

  var audioElement = new Audio(src)

  this.p = function()
  {
    audioElement.play();
  }

  this.s = function()
  {
    audioElement.pause();
  }

  this.t = function()
  {
    return audioElement.currentTime * (173.43 / 60);
  }

  this.seek = function(beat)
  {
    audioElement.currentTime = beat * 60 / 173.43
  }
}
var vs_shader_source='precision lowp float;uniform vec3 cam_pos,u_light;uniform mat4 view_proj_mat,view_proj_mat_inv,world_mat;uniform vec2 u_resolution,u_direction;uniform float focus,u_fade,u_threshold,u_param,clip_time,u_scale;uniform vec4 u_color1,u_color2;uniform sampler2D texture_0,texture_1,texture_2,texture_3,texture_4;varying vec3 a,b;varying vec2 c;varying vec4 d;varying float e;attribute vec3 a_position,a_normal,a_color;attribute vec2 a_uv;attribute float a_triangle_id;void j(){gl_Position=vec4(a_position,1);c=a_position.xy*.5+.5;}void k(){gl_Position=vec4(a_position,1);c=a_position.xy*.5+.5;}void l(){gl_Position=vec4(a_position,1);c=a_position.xy*.5+.5;a=vec3(u_direction/u_resolution,0);}float m(float g){return fract(sin(g)*43758.5453123);}vec3 n(vec3 g,float h,float i){return g*sin(i/10.)*50.*m(h);}void o(){gl_Position=view_proj_mat*vec4(a_position+n(a_normal,a_triangle_id,u_param),1);a=a_position;b=a_normal;d=mix(u_color1,u_color2,u_fade);c=a_uv;}void p(){j();}void q(){a=(world_mat*vec4(a_position,1)).xyz;gl_Position=view_proj_mat*vec4(a,1);b=mat3(world_mat)*a_normal;d.rgb=a_color;d.a=1.;}vec3 f=vec3(.9,.9,.8);void r(){vec3 g=a_position;g.y=1e2-mod(g.y+clip_time,1e2);float h=distance(cam_pos,g);e=abs(h-focus)/5.;b=a_normal;gl_Position=view_proj_mat*vec4(g,1);gl_PointSize=u_resolution.y*(1.+e)/gl_Position.w;}void s(){gl_Position=view_proj_mat*vec4(a_position*u_scale,1);a=a_position+a_triangle_id;d=mix(u_color1,u_color2,u_fade);}void t(){gl_Position=vec4(a_position,1);c=a_position.xy*.5+.5;}void u(){a=(world_mat*vec4(a_position,1)).xyz;gl_Position=view_proj_mat*vec4(a,1);b=mat3(world_mat)*a_normal;c=a_position.xy*vec2(.5,-.5)+.5;}'
var fs_shader_source='precision lowp float;uniform vec3 cam_pos,u_light;uniform mat4 view_proj_mat,view_proj_mat_inv,world_mat;uniform vec2 u_resolution;uniform float focus,u_fade,u_threshold,u_param,clip_time;uniform vec4 u_color1,u_color2,u_color;uniform sampler2D texture_0,texture_1,texture_2,texture_3,texture_4;varying vec3 a,b;varying vec2 c;varying vec4 d;varying float e;void o(){gl_FragColor=texture2D(texture_0,c)+texture2D(texture_1,c);}vec4 p(sampler2D g,vec2 h,vec2 i){vec4 j=vec4(0);float k[6];k[0]=.16;k[1]=.15;k[2]=.12;k[3]=.09;k[4]=.05;k[5]=.01;for(int l=-5;l<=5;++l)j+=texture2D(g,h+float(l)*i)*k[l>0?l:-l];return j;}void q(){gl_FragColor=p(texture_0,c,a.xy);}void r(){gl_FragColor=d;}void s(){vec4 g=texture2D(texture_1,c);if(g.r+g.g+g.b<u_threshold)g=vec4(0);gl_FragColor=g;}vec3 f=vec3(.9,.9,.8);float t(vec3 g){return dot(g,u_light)*.5+.5;}vec3 u(vec3 g){return mix(vec3(.75,.8,.9),vec3(1,.9,.7),t(g));}vec3 v(vec3 g,vec3 h){float i,j;i=distance(a,cam_pos);j=exp(-i*.001);return mix(u(g),h,j);}void w(){vec3 g,h,i,j,k,l,m;g=normalize(vec3(1));h=normalize(b);i=(dot(h,g)*.5+.5)*d.rgb;j=normalize(cam_pos-a);k=normalize(j+g);l=vec3(pow(dot(k,h),1e2));m=clamp(i+l,0.,1.);gl_FragColor=vec4(v(h,m),1);}void x(){vec3 g=b;float h,k,l;h=g.z*17.+clip_time*.2*g.x;mat2 i=mat2(sin(h),cos(h),-cos(h),sin(h));vec2 j=(gl_PointCoord-.5)*2.;j=i*j;k=pow(clamp(dot(g.yz-.5,vec2(j.y,-j.x)),0.,1.),1.5)*3.;j.x+=sin(j.y*g.x*6.)*.2;j*=vec2(1.+g.y*.1,1.+g.z*2.);l=1.-smoothstep(.9,1.,length(j));l/=1.+e;gl_FragColor=vec4(mix(vec3(1),vec3(1,.7,.7+g.z*.1),max(l-k,0.)),1);}void y(){gl_FragColor=d;}void z(){gl_FragColor=texture2D(texture_0,c);}void A(){vec4 g=texture2D(texture_0,c);vec3 h,i,j,k,l,m,n;h=normalize(vec3(1));i=normalize(b);j=(dot(i,h)*.5+.5)*g.rgb;k=normalize(cam_pos-a);l=normalize(k+h);m=vec3(pow(dot(l,i),1e2));n=clamp(j+m,0.,1.);gl_FragColor=vec4(v(i,n),g.a);}'
function load_shaders()
{
programs["add"] = load_shader_program("k", "o");
programs["directional_blur"] = load_shader_program("l", "q");
programs["explode"] = load_shader_program("o", "r");
programs["high_pass"] = load_shader_program("p", "s");
programs["model"] = load_shader_program("q", "w");
programs["particles"] = load_shader_program("r", "x");
programs["scale"] = load_shader_program("s", "y");
programs["show_texture"] = load_shader_program("t", "z");
programs["text"] = load_shader_program("u", "A");
}
function load_textures() {
textures["greets_ctrlalttest"] = create_vertical_text_texture(1, "CtrlAlt*");
textures["greets_cocoon"] = create_vertical_text_texture(1, "Cocoon");
textures["greets_vitalmotion"] = create_vertical_text_texture(1, "VitalMotion");
textures["greets_poobrain"] = create_vertical_text_texture(1, "PooBrain");
textures["greets_farbrausch"] = create_vertical_text_texture(1, "Farbrausch");
textures["greets_trsi"] = create_vertical_text_texture(1, "TRSi");
textures["greets_fairlight"] = create_vertical_text_texture(1, "Fairlight");
textures["greets_systemk"] = create_vertical_text_texture(1, "SystemK");
textures["text_travel_to"] = create_text_texture(1, "Travel to");
textures["text_japan"] = create_text_texture(1, "Japan");
textures["text_credits"] = create_text_texture(1, "tmp + X-Men | 2015");
textures["texture_1"] = create_texture( undefined ,  undefined ,  undefined ,  0,  undefined ,  1 ,  undefined ,  1 );
textures["texture_2"] = create_texture( 512 ,  512 ,  undefined ,  0,  undefined ,  1 ,  undefined ,  1 );
textures["texture_3"] = create_texture( 512 ,  256 ,  undefined ,  0,  undefined ,  1 ,  undefined ,  1 );
textures["texture_depth"] = create_texture( canvas.width ,  canvas.height ,  gl.DEPTH_COMPONENT ,  0,  undefined ,  undefined ,  undefined ,  undefined );
} // load_textures
function load_geometries() {
geometries["cube"] = {
  buffers: [
     make_vbo(POS, [1,-1,1,1,1,1,-1,1,1,-1,1,1,1,-1,1,-1,-1,1,1,-0.999999,-1,-1,-1,-1,-1,1,-1,-1,1,-1,1,-0.999999,-1,0.999999,1.000001,-1,1,-1,1,1,-0.999999,-1,0.999999,1.000001,-1,0.999999,1.000001,-1,1,-1,1,1,1,1,1,1,1,0.999999,1.000001,-1,-1,1,-1,-1,1,-1,1,1,1,-1,1,1,-1,1,1,-1,1,-1,-1,-1,-1,-1,-1,-1,-1,1,1,-1,-1,1,1,-0.999999,-1,1,-1,1,-1,-1,1,-1,-1,1,1,-0.999999,-1,-1,-1,-1] )
   , make_vbo(NORMAL, [0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1] )
   , make_vbo(COLOR, [0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621,0.085831,0.64,0.065621] )
  ],
  mode: gl.TRIANGLES ,
  vertex_count: 36
};
geometries["lantern1"] = {
  buffers: [
     make_vbo(POS, [-7.1987,30.8244,-15.2442,-3.9316,36.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,36.8294,-9.3897,-7.1987,30.8244,-15.2442,0.1425,30.8244,-16.9197,-13.0859,30.8244,-10.5493,-7.1987,36.8294,-5.8544,-3.9316,36.8294,-8.4599,-3.9316,36.8294,-8.4599,-13.0859,30.8244,-10.5493,-7.1987,30.8244,-15.2442,-16.353,30.8244,-3.765,-9.0118,36.8294,-2.0894,-7.1987,36.8294,-5.8544,-7.1987,36.8294,-5.8544,-16.353,30.8244,-3.765,-13.0859,30.8244,-10.5493,-16.353,30.8244,3.765,-9.0118,36.8294,2.0894,-9.0118,36.8294,-2.0894,-9.0118,36.8294,-2.0894,-16.353,30.8244,3.765,-16.353,30.8244,-3.765,-13.0859,30.8244,10.5493,-7.1987,36.8294,5.8544,-9.0118,36.8294,2.0894,-9.0118,36.8294,2.0894,-13.0859,30.8244,10.5493,-16.353,30.8244,3.765,-7.1987,30.8244,15.2442,-3.9316,36.8294,8.4599,-7.1987,36.8294,5.8544,-7.1987,36.8294,5.8544,-7.1987,30.8244,15.2442,-13.0859,30.8244,10.5493,0.1425,30.8244,16.9197,0.1425,36.8294,9.3897,-3.9316,36.8294,8.4599,-3.9316,36.8294,8.4599,0.1425,30.8244,16.9197,-7.1987,30.8244,15.2442,7.4837,30.8244,15.2442,4.2166,36.8294,8.4599,0.1425,36.8294,9.3897,0.1425,36.8294,9.3897,7.4837,30.8244,15.2442,0.1425,30.8244,16.9197,13.3709,30.8244,10.5493,7.4837,36.8294,5.8544,4.2166,36.8294,8.4599,4.2166,36.8294,8.4599,13.3709,30.8244,10.5493,7.4837,30.8244,15.2442,16.638,30.8244,3.765,9.2968,36.8294,2.0894,7.4837,36.8294,5.8544,7.4837,36.8294,5.8544,16.638,30.8244,3.765,13.3709,30.8244,10.5493,16.638,30.8244,-3.765,9.2968,36.8294,-2.0894,9.2968,36.8294,2.0894,9.2968,36.8294,2.0894,16.638,30.8244,-3.765,16.638,30.8244,3.765,13.3709,30.8244,-10.5493,7.4837,36.8294,-5.8544,9.2968,36.8294,-2.0894,9.2968,36.8294,-2.0894,13.3709,30.8244,-10.5493,16.638,30.8244,-3.765,7.4837,30.8244,-15.2442,4.2166,36.8294,-8.4599,7.4837,36.8294,-5.8544,7.4837,36.8294,-5.8544,7.4837,30.8244,-15.2442,13.3709,30.8244,-10.5493,0.1425,30.8244,-16.9197,0.1425,36.8294,-9.3897,4.2166,36.8294,-8.4599,4.2166,36.8294,-8.4599,0.1425,30.8244,-16.9197,7.4837,30.8244,-15.2442,-9.0118,22.147,-19.0091,-7.1987,30.8244,-15.2442,0.1425,30.8244,-16.9197,0.1425,30.8244,-16.9197,-9.0118,22.147,-19.0091,0.1425,22.147,-21.0986,-16.353,22.147,-13.1547,-13.0859,30.8244,-10.5493,-7.1987,30.8244,-15.2442,-7.1987,30.8244,-15.2442,-16.353,22.147,-13.1547,-9.0118,22.147,-19.0091,-20.4271,22.147,-4.6949,-16.353,30.8244,-3.765,-13.0859,30.8244,-10.5493,-13.0859,30.8244,-10.5493,-20.4271,22.147,-4.6949,-16.353,22.147,-13.1547,-20.4271,22.147,4.6949,-16.353,30.8244,3.765,-16.353,30.8244,-3.765,-16.353,30.8244,-3.765,-20.4271,22.147,4.6949,-20.4271,22.147,-4.6949,-16.353,22.147,13.1547,-13.0859,30.8244,10.5493,-16.353,30.8244,3.765,-16.353,30.8244,3.765,-16.353,22.147,13.1547,-20.4271,22.147,4.6949,-9.0118,22.147,19.0091,-7.1987,30.8244,15.2442,-13.0859,30.8244,10.5493,-13.0859,30.8244,10.5493,-9.0118,22.147,19.0091,-16.353,22.147,13.1547,0.1425,22.147,21.0986,0.1425,30.8244,16.9197,-7.1987,30.8244,15.2442,-7.1987,30.8244,15.2442,0.1425,22.147,21.0986,-9.0118,22.147,19.0091,9.2968,22.147,19.0091,7.4837,30.8244,15.2442,0.1425,30.8244,16.9197,0.1425,30.8244,16.9197,9.2968,22.147,19.0091,0.1425,22.147,21.0986,16.638,22.147,13.1547,13.3709,30.8244,10.5493,7.4837,30.8244,15.2442,7.4837,30.8244,15.2442,16.638,22.147,13.1547,9.2968,22.147,19.0091,20.7121,22.147,4.6949,16.638,30.8244,3.765,13.3709,30.8244,10.5493,13.3709,30.8244,10.5493,20.7121,22.147,4.6949,16.638,22.147,13.1547,20.7121,22.147,-4.6949,16.638,30.8244,-3.765,16.638,30.8244,3.765,16.638,30.8244,3.765,20.7121,22.147,-4.6949,20.7121,22.147,4.6949,16.638,22.147,-13.1548,13.3709,30.8244,-10.5493,16.638,30.8244,-3.765,16.638,30.8244,-3.765,16.638,22.147,-13.1548,20.7121,22.147,-4.6949,9.2968,22.147,-19.0092,7.4837,30.8244,-15.2442,13.3709,30.8244,-10.5493,13.3709,30.8244,-10.5493,9.2968,22.147,-19.0092,16.638,22.147,-13.1548,0.1425,22.147,-21.0986,0.1425,30.8244,-16.9197,7.4837,30.8244,-15.2442,7.4837,30.8244,-15.2442,0.1425,22.147,-21.0986,9.2968,22.147,-19.0092,-9.0118,-22.0147,-19.0091,-9.0118,22.147,-19.0091,0.1425,22.147,-21.0986,0.1425,22.147,-21.0986,-9.0118,-22.0147,-19.0091,0.1425,-22.0147,-21.0986,-16.353,-22.0147,-13.1547,-16.353,22.147,-13.1547,-9.0118,22.147,-19.0091,-9.0118,22.147,-19.0091,-16.353,-22.0147,-13.1547,-9.0118,-22.0147,-19.0091,-20.4271,-22.0147,-4.6949,-20.4271,22.147,-4.6949,-16.353,22.147,-13.1547,-16.353,22.147,-13.1547,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,-13.1547,-20.4271,-22.0147,4.6949,-20.4271,22.147,4.6949,-20.4271,22.147,-4.6949,-20.4271,22.147,-4.6949,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,13.1547,-16.353,22.147,13.1547,-20.4271,22.147,4.6949,-20.4271,22.147,4.6949,-16.353,-22.0147,13.1547,-20.4271,-22.0147,4.6949,-9.0118,-22.0147,19.0091,-9.0118,22.147,19.0091,-16.353,22.147,13.1547,-16.353,22.147,13.1547,-9.0118,-22.0147,19.0091,-16.353,-22.0147,13.1547,0.1425,-22.0147,21.0986,0.1425,22.147,21.0986,-9.0118,22.147,19.0091,-9.0118,22.147,19.0091,0.1425,-22.0147,21.0986,-9.0118,-22.0147,19.0091,9.2968,-22.0147,19.0091,9.2968,22.147,19.0091,0.1425,22.147,21.0986,0.1425,22.147,21.0986,9.2968,-22.0147,19.0091,0.1425,-22.0147,21.0986,16.638,-22.0147,13.1547,16.638,22.147,13.1547,9.2968,22.147,19.0091,9.2968,22.147,19.0091,16.638,-22.0147,13.1547,9.2968,-22.0147,19.0091,20.7121,-22.0147,4.6949,20.7121,22.147,4.6949,16.638,22.147,13.1547,16.638,22.147,13.1547,20.7121,-22.0147,4.6949,16.638,-22.0147,13.1547,20.7121,-22.0147,-4.6949,20.7121,22.147,-4.6949,20.7121,22.147,4.6949,20.7121,22.147,4.6949,20.7121,-22.0147,-4.6949,20.7121,-22.0147,4.6949,16.638,-22.0147,-13.1548,16.638,22.147,-13.1548,20.7121,22.147,-4.6949,20.7121,22.147,-4.6949,16.638,-22.0147,-13.1548,20.7121,-22.0147,-4.6949,9.2968,-22.0147,-19.0092,9.2968,22.147,-19.0092,16.638,22.147,-13.1548,16.638,22.147,-13.1548,9.2968,-22.0147,-19.0092,16.638,-22.0147,-13.1548,0.1425,-22.0147,-21.0986,0.1425,22.147,-21.0986,9.2968,22.147,-19.0092,9.2968,22.147,-19.0092,0.1425,-22.0147,-21.0986,9.2968,-22.0147,-19.0092,-7.1987,-30.6921,-15.2442,-9.0118,-22.0147,-19.0091,0.1425,-22.0147,-21.0986,0.1425,-22.0147,-21.0986,-7.1987,-30.6921,-15.2442,0.1425,-30.6921,-16.9197,-13.0859,-30.6921,-10.5493,-16.353,-22.0147,-13.1547,-9.0118,-22.0147,-19.0091,-9.0118,-22.0147,-19.0091,-13.0859,-30.6921,-10.5493,-7.1987,-30.6921,-15.2442,-16.353,-30.6921,-3.765,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,-13.1547,-16.353,-22.0147,-13.1547,-16.353,-30.6921,-3.765,-13.0859,-30.6921,-10.5493,-16.353,-30.6921,3.765,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,-4.6949,-20.4271,-22.0147,-4.6949,-16.353,-30.6921,3.765,-16.353,-30.6921,-3.765,-13.0859,-30.6921,10.5493,-16.353,-22.0147,13.1547,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,4.6949,-13.0859,-30.6921,10.5493,-16.353,-30.6921,3.765,-7.1987,-30.6921,15.2442,-9.0118,-22.0147,19.0091,-16.353,-22.0147,13.1547,-16.353,-22.0147,13.1547,-7.1987,-30.6921,15.2442,-13.0859,-30.6921,10.5493,0.1425,-30.6921,16.9197,0.1425,-22.0147,21.0986,-9.0118,-22.0147,19.0091,-9.0118,-22.0147,19.0091,0.1425,-30.6921,16.9197,-7.1987,-30.6921,15.2442,7.4837,-30.6921,15.2442,9.2968,-22.0147,19.0091,0.1425,-22.0147,21.0986,0.1425,-22.0147,21.0986,7.4837,-30.6921,15.2442,0.1425,-30.6921,16.9197,13.3709,-30.6921,10.5493,16.638,-22.0147,13.1547,9.2968,-22.0147,19.0091,9.2968,-22.0147,19.0091,13.3709,-30.6921,10.5493,7.4837,-30.6921,15.2442,16.638,-30.6921,3.765,20.7121,-22.0147,4.6949,16.638,-22.0147,13.1547,16.638,-22.0147,13.1547,16.638,-30.6921,3.765,13.3709,-30.6921,10.5493,16.638,-30.6921,-3.765,20.7121,-22.0147,-4.6949,20.7121,-22.0147,4.6949,20.7121,-22.0147,4.6949,16.638,-30.6921,-3.765,16.638,-30.6921,3.765,13.3709,-30.6921,-10.5493,16.638,-22.0147,-13.1548,20.7121,-22.0147,-4.6949,20.7121,-22.0147,-4.6949,13.3709,-30.6921,-10.5493,16.638,-30.6921,-3.765,7.4837,-30.6921,-15.2442,9.2968,-22.0147,-19.0092,16.638,-22.0147,-13.1548,16.638,-22.0147,-13.1548,7.4837,-30.6921,-15.2442,13.3709,-30.6921,-10.5493,0.1425,-30.6921,-16.9197,0.1425,-22.0147,-21.0986,9.2968,-22.0147,-19.0092,9.2968,-22.0147,-19.0092,0.1425,-30.6921,-16.9197,7.4837,-30.6921,-15.2442,-3.9316,-36.6971,-8.4599,-7.1987,-30.6921,-15.2442,0.1425,-30.6921,-16.9197,0.1425,-30.6921,-16.9197,-3.9316,-36.6971,-8.4599,0.1425,-36.6971,-9.3897,-7.1987,-36.6971,-5.8544,-13.0859,-30.6921,-10.5493,-7.1987,-30.6921,-15.2442,-7.1987,-30.6921,-15.2442,-7.1987,-36.6971,-5.8544,-3.9316,-36.6971,-8.4599,-9.0118,-36.6971,-2.0894,-16.353,-30.6921,-3.765,-13.0859,-30.6921,-10.5493,-13.0859,-30.6921,-10.5493,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,-5.8544,-9.0118,-36.6971,2.0894,-16.353,-30.6921,3.765,-16.353,-30.6921,-3.765,-16.353,-30.6921,-3.765,-9.0118,-36.6971,2.0894,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,5.8544,-13.0859,-30.6921,10.5493,-16.353,-30.6921,3.765,-16.353,-30.6921,3.765,-7.1987,-36.6971,5.8544,-9.0118,-36.6971,2.0894,-3.9316,-36.6971,8.4599,-7.1987,-30.6921,15.2442,-13.0859,-30.6921,10.5493,-13.0859,-30.6921,10.5493,-3.9316,-36.6971,8.4599,-7.1987,-36.6971,5.8544,0.1425,-36.6971,9.3897,0.1425,-30.6921,16.9197,-7.1987,-30.6921,15.2442,-7.1987,-30.6921,15.2442,0.1425,-36.6971,9.3897,-3.9316,-36.6971,8.4599,4.2166,-36.6971,8.4599,7.4837,-30.6921,15.2442,0.1425,-30.6921,16.9197,0.1425,-30.6921,16.9197,4.2166,-36.6971,8.4599,0.1425,-36.6971,9.3897,7.4837,-36.6971,5.8544,13.3709,-30.6921,10.5493,7.4837,-30.6921,15.2442,7.4837,-30.6921,15.2442,7.4837,-36.6971,5.8544,4.2166,-36.6971,8.4599,9.2968,-36.6971,2.0894,16.638,-30.6921,3.765,13.3709,-30.6921,10.5493,13.3709,-30.6921,10.5493,9.2968,-36.6971,2.0894,7.4837,-36.6971,5.8544,9.2968,-36.6971,-2.0894,16.638,-30.6921,-3.765,16.638,-30.6921,3.765,16.638,-30.6921,3.765,9.2968,-36.6971,-2.0894,9.2968,-36.6971,2.0894,7.4837,-36.6971,-5.8544,13.3709,-30.6921,-10.5493,16.638,-30.6921,-3.765,16.638,-30.6921,-3.765,7.4837,-36.6971,-5.8544,9.2968,-36.6971,-2.0894,4.2166,-36.6971,-8.4599,7.4837,-30.6921,-15.2442,13.3709,-30.6921,-10.5493,13.3709,-30.6921,-10.5493,4.2166,-36.6971,-8.4599,7.4837,-36.6971,-5.8544,0.1425,-36.6971,-9.3897,0.1425,-30.6921,-16.9197,7.4837,-30.6921,-15.2442,7.4837,-30.6921,-15.2442,0.1425,-36.6971,-9.3897,4.2166,-36.6971,-8.4599,0.1425,-44.7884,0,-3.9316,-44.7984,-8.4599,0.1425,-44.7984,-9.3897,0.1425,-44.7884,0,-7.1987,-44.7984,-5.8544,-3.9316,-44.7984,-8.4599,0.1425,-44.7884,0,-9.0118,-44.7984,-2.0894,-7.1987,-44.7984,-5.8544,0.1425,-44.7884,0,-9.0118,-44.7984,2.0894,-9.0118,-44.7984,-2.0894,0.1425,-44.7884,0,-7.1987,-44.7984,5.8544,-9.0118,-44.7984,2.0894,0.1425,-44.7884,0,-3.9316,-44.7984,8.4599,-7.1987,-44.7984,5.8544,0.1425,-44.7884,0,0.1425,-44.7984,9.3897,-3.9316,-44.7984,8.4599,0.1425,-44.7884,0,4.2166,-44.7984,8.4599,0.1425,-44.7984,9.3897,0.1425,-44.7884,0,7.4837,-44.7984,5.8544,4.2166,-44.7984,8.4599,0.1425,-44.7884,0,9.2968,-44.7984,2.0894,7.4837,-44.7984,5.8544,0.1425,-44.7884,0,9.2968,-44.7984,-2.0894,9.2968,-44.7984,2.0894,0.1425,-44.7884,0,7.4837,-44.7984,-5.8544,9.2968,-44.7984,-2.0894,0.1425,-44.7884,0,4.2166,-44.7984,-8.4599,7.4837,-44.7984,-5.8544,0.1425,-44.7884,0,0.1425,-44.7984,-9.3897,4.2166,-44.7984,-8.4599,0.1425,36.8294,-9.3897,-3.9316,36.8294,-8.4599,-3.9316,37.8294,-8.4599,-3.9316,37.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,37.8294,-9.3897,-3.9316,36.8294,-8.4599,-7.1987,36.8294,-5.8544,-7.1987,37.8294,-5.8544,-7.1987,37.8294,-5.8544,-3.9316,36.8294,-8.4599,-3.9316,37.8294,-8.4599,-7.1987,36.8294,-5.8544,-9.0118,36.8294,-2.0894,-9.0118,37.8294,-2.0894,-9.0118,37.8294,-2.0894,-7.1987,36.8294,-5.8544,-7.1987,37.8294,-5.8544,-9.0118,36.8294,-2.0894,-9.0118,36.8294,2.0894,-9.0118,37.8294,2.0894,-9.0118,37.8294,2.0894,-9.0118,36.8294,-2.0894,-9.0118,37.8294,-2.0894,-9.0118,36.8294,2.0894,-7.1987,36.8294,5.8544,-7.1987,37.8294,5.8544,-7.1987,37.8294,5.8544,-9.0118,36.8294,2.0894,-9.0118,37.8294,2.0894,-7.1987,36.8294,5.8544,-3.9316,36.8294,8.4599,-3.9316,37.8294,8.4599,-3.9316,37.8294,8.4599,-7.1987,36.8294,5.8544,-7.1987,37.8294,5.8544,-3.9316,36.8294,8.4599,0.1425,36.8294,9.3897,0.1425,37.8294,9.3897,0.1425,37.8294,9.3897,-3.9316,36.8294,8.4599,-3.9316,37.8294,8.4599,0.1425,36.8294,9.3897,4.2166,36.8294,8.4599,4.2166,37.8294,8.4599,4.2166,37.8294,8.4599,0.1425,36.8294,9.3897,0.1425,37.8294,9.3897,4.2166,36.8294,8.4599,7.4837,36.8294,5.8544,7.4837,37.8294,5.8544,7.4837,37.8294,5.8544,4.2166,36.8294,8.4599,4.2166,37.8294,8.4599,7.4837,36.8294,5.8544,9.2968,36.8294,2.0894,9.2968,37.8294,2.0894,9.2968,37.8294,2.0894,7.4837,36.8294,5.8544,7.4837,37.8294,5.8544,9.2968,36.8294,2.0894,9.2968,36.8294,-2.0894,9.2968,37.8294,-2.0894,9.2968,37.8294,-2.0894,9.2968,36.8294,2.0894,9.2968,37.8294,2.0894,9.2968,36.8294,-2.0894,7.4837,36.8294,-5.8544,7.4837,37.8294,-5.8544,7.4837,37.8294,-5.8544,9.2968,36.8294,-2.0894,9.2968,37.8294,-2.0894,7.4837,36.8294,-5.8544,4.2166,36.8294,-8.4599,4.2166,37.8294,-8.4599,4.2166,37.8294,-8.4599,7.4837,36.8294,-5.8544,7.4837,37.8294,-5.8544,4.2166,36.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,37.8294,-9.3897,0.1425,37.8294,-9.3897,4.2166,36.8294,-8.4599,4.2166,37.8294,-8.4599,-3.9316,-36.6971,-8.4599,0.1425,-36.6971,-9.3897,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-9.3897,-3.9316,-36.6971,-8.4599,-3.9316,-37.6971,-8.4599,0.1425,-36.6971,-9.3897,4.2166,-36.6971,-8.4599,4.2166,-37.6971,-8.4599,4.2166,-37.6971,-8.4599,0.1425,-36.6971,-9.3897,0.1425,-37.6971,-9.3897,4.2166,-36.6971,-8.4599,7.4837,-36.6971,-5.8544,7.4837,-37.6971,-5.8544,7.4837,-37.6971,-5.8544,4.2166,-36.6971,-8.4599,4.2166,-37.6971,-8.4599,7.4837,-36.6971,-5.8544,9.2968,-36.6971,-2.0894,9.2968,-37.6971,-2.0894,9.2968,-37.6971,-2.0894,7.4837,-36.6971,-5.8544,7.4837,-37.6971,-5.8544,9.2968,-36.6971,-2.0894,9.2968,-36.6971,2.0894,9.2968,-37.6971,2.0894,9.2968,-37.6971,2.0894,9.2968,-36.6971,-2.0894,9.2968,-37.6971,-2.0894,9.2968,-36.6971,2.0894,7.4837,-36.6971,5.8544,7.4837,-37.6971,5.8544,7.4837,-37.6971,5.8544,9.2968,-36.6971,2.0894,9.2968,-37.6971,2.0894,7.4837,-36.6971,5.8544,4.2166,-36.6971,8.4599,4.2166,-37.6971,8.4599,4.2166,-37.6971,8.4599,7.4837,-36.6971,5.8544,7.4837,-37.6971,5.8544,4.2166,-36.6971,8.4599,0.1425,-36.6971,9.3897,0.1425,-37.6971,9.3897,0.1425,-37.6971,9.3897,4.2166,-36.6971,8.4599,4.2166,-37.6971,8.4599,0.1425,-36.6971,9.3897,-3.9316,-36.6971,8.4599,-3.9316,-37.6971,8.4599,-3.9316,-37.6971,8.4599,0.1425,-36.6971,9.3897,0.1425,-37.6971,9.3897,-3.9316,-36.6971,8.4599,-7.1987,-36.6971,5.8544,-7.1987,-37.6971,5.8544,-7.1987,-37.6971,5.8544,-3.9316,-36.6971,8.4599,-3.9316,-37.6971,8.4599,-7.1987,-36.6971,5.8544,-9.0118,-36.6971,2.0894,-9.0118,-37.6971,2.0894,-9.0118,-37.6971,2.0894,-7.1987,-36.6971,5.8544,-7.1987,-37.6971,5.8544,-9.0118,-36.6971,2.0894,-9.0118,-36.6971,-2.0894,-9.0118,-37.6971,-2.0894,-9.0118,-37.6971,-2.0894,-9.0118,-36.6971,2.0894,-9.0118,-37.6971,2.0894,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,-5.8544,-7.1987,-37.6971,-5.8544,-7.1987,-37.6971,-5.8544,-9.0118,-36.6971,-2.0894,-9.0118,-37.6971,-2.0894,-7.1987,-36.6971,-5.8544,-3.9316,-36.6971,-8.4599,-3.9316,-37.6971,-8.4599,-3.9316,-37.6971,-8.4599,-7.1987,-36.6971,-5.8544,-7.1987,-37.6971,-5.8544,0.1424,37.8294,-13.3932,-5.6686,37.8294,-12.0669,-5.6686,44.9308,-12.0669,-5.6686,44.9308,-12.0669,0.1424,37.8294,-13.3932,0.1424,44.9308,-13.3932,-5.6686,37.8294,-12.0669,-10.3288,37.8294,-8.3506,-10.3288,44.9308,-8.3506,-10.3288,44.9308,-8.3506,-5.6686,37.8294,-12.0669,-5.6686,44.9308,-12.0669,-10.3288,37.8294,-8.3506,-12.9149,37.8294,-2.9803,-12.9149,44.9308,-2.9803,-12.9149,44.9308,-2.9803,-10.3288,37.8294,-8.3506,-10.3288,44.9308,-8.3506,-12.9149,37.8294,-2.9803,-12.9149,37.8294,2.9803,-12.9149,44.9308,2.9803,-12.9149,44.9308,2.9803,-12.9149,37.8294,-2.9803,-12.9149,44.9308,-2.9803,-12.9149,37.8294,2.9803,-10.3288,37.8294,8.3506,-10.3288,44.9308,8.3506,-10.3288,44.9308,8.3506,-12.9149,37.8294,2.9803,-12.9149,44.9308,2.9803,-10.3288,37.8294,8.3506,-5.6687,37.8294,12.0669,-5.6687,44.9308,12.0669,-5.6687,44.9308,12.0669,-10.3288,37.8294,8.3506,-10.3288,44.9308,8.3506,-5.6687,37.8294,12.0669,0.1425,37.8294,13.3932,0.1425,44.9308,13.3932,0.1425,44.9308,13.3932,-5.6687,37.8294,12.0669,-5.6687,44.9308,12.0669,0.1425,37.8294,13.3932,5.9537,37.8294,12.0669,5.9537,44.9308,12.0669,5.9537,44.9308,12.0669,0.1425,37.8294,13.3932,0.1425,44.9308,13.3932,5.9537,37.8294,12.0669,10.6138,37.8294,8.3506,10.6138,44.9308,8.3506,10.6138,44.9308,8.3506,5.9537,37.8294,12.0669,5.9537,44.9308,12.0669,10.6138,37.8294,8.3506,13.2,37.8294,2.9803,13.2,44.9308,2.9803,13.2,44.9308,2.9803,10.6138,37.8294,8.3506,10.6138,44.9308,8.3506,13.2,37.8294,2.9803,13.2,37.8294,-2.9802,13.2,44.9308,-2.9802,13.2,44.9308,-2.9802,13.2,37.8294,2.9803,13.2,44.9308,2.9803,13.2,37.8294,-2.9802,10.6137,37.8294,-8.3506,10.6137,44.9308,-8.3506,10.6137,44.9308,-8.3506,13.2,37.8294,-2.9802,13.2,44.9308,-2.9802,10.6137,37.8294,-8.3506,5.9537,37.8294,-12.0669,5.9537,44.9308,-12.0669,5.9537,44.9308,-12.0669,10.6137,37.8294,-8.3506,10.6137,44.9308,-8.3506,5.9537,37.8294,-12.0669,0.1424,37.8294,-13.3932,0.1424,44.9308,-13.3932,0.1424,44.9308,-13.3932,5.9537,37.8294,-12.0669,5.9537,44.9308,-12.0669,-5.6686,-37.6971,-12.0669,0.1425,-37.6971,-13.3932,0.1425,-44.7984,-13.3932,0.1425,-44.7984,-13.3932,-5.6686,-37.6971,-12.0669,-5.6686,-44.7984,-12.0669,0.1425,-37.6971,-13.3932,5.9536,-37.6971,-12.0669,5.9536,-44.7984,-12.0669,5.9536,-44.7984,-12.0669,0.1425,-37.6971,-13.3932,0.1425,-44.7984,-13.3932,5.9536,-37.6971,-12.0669,10.6138,-37.6971,-8.3506,10.6138,-44.7984,-8.3506,10.6138,-44.7984,-8.3506,5.9536,-37.6971,-12.0669,5.9536,-44.7984,-12.0669,10.6138,-37.6971,-8.3506,13.2,-37.6971,-2.9803,13.2,-44.7984,-2.9803,13.2,-44.7984,-2.9803,10.6138,-37.6971,-8.3506,10.6138,-44.7984,-8.3506,13.2,-37.6971,-2.9803,13.2,-37.6971,2.9802,13.2,-44.7984,2.9802,13.2,-44.7984,2.9802,13.2,-37.6971,-2.9803,13.2,-44.7984,-2.9803,13.2,-37.6971,2.9802,10.6138,-37.6971,8.3505,10.6138,-44.7984,8.3505,10.6138,-44.7984,8.3505,13.2,-37.6971,2.9802,13.2,-44.7984,2.9802,10.6138,-37.6971,8.3505,5.9535,-37.6971,12.0669,5.9535,-44.7984,12.0669,5.9535,-44.7984,12.0669,10.6138,-37.6971,8.3505,10.6138,-44.7984,8.3505,5.9535,-37.6971,12.0669,0.1425,-37.6971,13.3932,0.1425,-44.7984,13.3932,0.1425,-44.7984,13.3932,5.9535,-37.6971,12.0669,5.9535,-44.7984,12.0669,0.1425,-37.6971,13.3932,-5.6686,-37.6971,12.0669,-5.6686,-44.7984,12.0669,-5.6686,-44.7984,12.0669,0.1425,-37.6971,13.3932,0.1425,-44.7984,13.3932,-5.6686,-37.6971,12.0669,-10.3288,-37.6971,8.3506,-10.3288,-44.7984,8.3506,-10.3288,-44.7984,8.3506,-5.6686,-37.6971,12.0669,-5.6686,-44.7984,12.0669,-10.3288,-37.6971,8.3506,-12.9149,-37.6971,2.9803,-12.9149,-44.7984,2.9803,-12.9149,-44.7984,2.9803,-10.3288,-37.6971,8.3506,-10.3288,-44.7984,8.3506,-12.9149,-37.6971,2.9803,-12.9149,-37.6971,-2.9803,-12.9149,-44.7984,-2.9803,-12.9149,-44.7984,-2.9803,-12.9149,-37.6971,2.9803,-12.9149,-44.7984,2.9803,-12.9149,-37.6971,-2.9803,-10.3288,-37.6971,-8.3506,-10.3288,-44.7984,-8.3506,-10.3288,-44.7984,-8.3506,-12.9149,-37.6971,-2.9803,-12.9149,-44.7984,-2.9803,-10.3288,-37.6971,-8.3506,-5.6686,-37.6971,-12.0669,-5.6686,-44.7984,-12.0669,-5.6686,-44.7984,-12.0669,-10.3288,-37.6971,-8.3506,-10.3288,-44.7984,-8.3506,0.1425,37.8294,-9.3897,-3.9316,37.8294,-8.4599,-5.6686,37.8294,-12.0669,-5.6686,37.8294,-12.0669,0.1425,37.8294,-9.3897,0.1424,37.8294,-13.3932,-3.9316,37.8294,-8.4599,-7.1987,37.8294,-5.8544,-10.3288,37.8294,-8.3506,-10.3288,37.8294,-8.3506,-3.9316,37.8294,-8.4599,-5.6686,37.8294,-12.0669,-7.1987,37.8294,-5.8544,-9.0118,37.8294,-2.0894,-12.9149,37.8294,-2.9803,-12.9149,37.8294,-2.9803,-7.1987,37.8294,-5.8544,-10.3288,37.8294,-8.3506,-9.0118,37.8294,-2.0894,-9.0118,37.8294,2.0894,-12.9149,37.8294,2.9803,-12.9149,37.8294,2.9803,-9.0118,37.8294,-2.0894,-12.9149,37.8294,-2.9803,-9.0118,37.8294,2.0894,-7.1987,37.8294,5.8544,-10.3288,37.8294,8.3506,-10.3288,37.8294,8.3506,-9.0118,37.8294,2.0894,-12.9149,37.8294,2.9803,-7.1987,37.8294,5.8544,-3.9316,37.8294,8.4599,-5.6687,37.8294,12.0669,-5.6687,37.8294,12.0669,-7.1987,37.8294,5.8544,-10.3288,37.8294,8.3506,-3.9316,37.8294,8.4599,0.1425,37.8294,9.3897,0.1425,37.8294,13.3932,0.1425,37.8294,13.3932,-3.9316,37.8294,8.4599,-5.6687,37.8294,12.0669,0.1425,37.8294,9.3897,4.2166,37.8294,8.4599,5.9537,37.8294,12.0669,5.9537,37.8294,12.0669,0.1425,37.8294,9.3897,0.1425,37.8294,13.3932,4.2166,37.8294,8.4599,7.4837,37.8294,5.8544,10.6138,37.8294,8.3506,10.6138,37.8294,8.3506,4.2166,37.8294,8.4599,5.9537,37.8294,12.0669,7.4837,37.8294,5.8544,9.2968,37.8294,2.0894,13.2,37.8294,2.9803,13.2,37.8294,2.9803,7.4837,37.8294,5.8544,10.6138,37.8294,8.3506,9.2968,37.8294,2.0894,9.2968,37.8294,-2.0894,13.2,37.8294,-2.9802,13.2,37.8294,-2.9802,9.2968,37.8294,2.0894,13.2,37.8294,2.9803,9.2968,37.8294,-2.0894,7.4837,37.8294,-5.8544,10.6137,37.8294,-8.3506,10.6137,37.8294,-8.3506,9.2968,37.8294,-2.0894,13.2,37.8294,-2.9802,7.4837,37.8294,-5.8544,4.2166,37.8294,-8.4599,5.9537,37.8294,-12.0669,5.9537,37.8294,-12.0669,7.4837,37.8294,-5.8544,10.6137,37.8294,-8.3506,4.2166,37.8294,-8.4599,0.1425,37.8294,-9.3897,0.1424,37.8294,-13.3932,0.1424,37.8294,-13.3932,4.2166,37.8294,-8.4599,5.9537,37.8294,-12.0669,0.1424,44.9308,-13.3932,-5.6686,44.9308,-12.0669,-3.9316,44.9308,-8.4599,-3.9316,44.9308,-8.4599,0.1424,44.9308,-13.3932,0.1425,44.9308,-9.3897,5.9537,44.9308,-12.0669,0.1424,44.9308,-13.3932,0.1425,44.9308,-9.3897,0.1425,44.9308,-9.3897,5.9537,44.9308,-12.0669,4.2166,44.9308,-8.4599,10.6137,44.9308,-8.3506,5.9537,44.9308,-12.0669,4.2166,44.9308,-8.4599,4.2166,44.9308,-8.4599,10.6137,44.9308,-8.3506,7.4837,44.9308,-5.8544,13.2,44.9308,-2.9802,10.6137,44.9308,-8.3506,7.4837,44.9308,-5.8544,7.4837,44.9308,-5.8544,13.2,44.9308,-2.9802,9.2968,44.9308,-2.0894,13.2,44.9308,2.9803,13.2,44.9308,-2.9802,9.2968,44.9308,-2.0894,9.2968,44.9308,-2.0894,13.2,44.9308,2.9803,9.2968,44.9308,2.0894,10.6138,44.9308,8.3506,13.2,44.9308,2.9803,9.2968,44.9308,2.0894,9.2968,44.9308,2.0894,10.6138,44.9308,8.3506,7.4837,44.9308,5.8544,5.9537,44.9308,12.0669,10.6138,44.9308,8.3506,7.4837,44.9308,5.8544,7.4837,44.9308,5.8544,5.9537,44.9308,12.0669,4.2166,44.9308,8.4599,0.1425,44.9308,13.3932,5.9537,44.9308,12.0669,4.2166,44.9308,8.4599,4.2166,44.9308,8.4599,0.1425,44.9308,13.3932,0.1425,44.9308,9.3897,-5.6687,44.9308,12.0669,0.1425,44.9308,13.3932,0.1425,44.9308,9.3897,0.1425,44.9308,9.3897,-5.6687,44.9308,12.0669,-3.9316,44.9308,8.4599,-10.3288,44.9308,8.3506,-5.6687,44.9308,12.0669,-3.9316,44.9308,8.4599,-3.9316,44.9308,8.4599,-10.3288,44.9308,8.3506,-7.1987,44.9308,5.8544,-12.9149,44.9308,2.9803,-10.3288,44.9308,8.3506,-7.1987,44.9308,5.8544,-7.1987,44.9308,5.8544,-12.9149,44.9308,2.9803,-9.0118,44.9308,2.0894,-12.9149,44.9308,-2.9803,-12.9149,44.9308,2.9803,-9.0118,44.9308,2.0894,-9.0118,44.9308,2.0894,-12.9149,44.9308,-2.9803,-9.0118,44.9308,-2.0894,-10.3288,44.9308,-8.3506,-12.9149,44.9308,-2.9803,-9.0118,44.9308,-2.0894,-9.0118,44.9308,-2.0894,-10.3288,44.9308,-8.3506,-7.1987,44.9308,-5.8544,-5.6686,44.9308,-12.0669,-10.3288,44.9308,-8.3506,-7.1987,44.9308,-5.8544,-7.1987,44.9308,-5.8544,-5.6686,44.9308,-12.0669,-3.9316,44.9308,-8.4599,-3.9316,-37.6971,-8.4599,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-13.3932,0.1425,-37.6971,-13.3932,-3.9316,-37.6971,-8.4599,-5.6686,-37.6971,-12.0669,0.1425,-37.6971,-9.3897,4.2166,-37.6971,-8.4599,5.9536,-37.6971,-12.0669,5.9536,-37.6971,-12.0669,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-13.3932,4.2166,-37.6971,-8.4599,7.4837,-37.6971,-5.8544,10.6138,-37.6971,-8.3506,10.6138,-37.6971,-8.3506,4.2166,-37.6971,-8.4599,5.9536,-37.6971,-12.0669,7.4837,-37.6971,-5.8544,9.2968,-37.6971,-2.0894,13.2,-37.6971,-2.9803,13.2,-37.6971,-2.9803,7.4837,-37.6971,-5.8544,10.6138,-37.6971,-8.3506,9.2968,-37.6971,-2.0894,9.2968,-37.6971,2.0894,13.2,-37.6971,2.9802,13.2,-37.6971,2.9802,9.2968,-37.6971,-2.0894,13.2,-37.6971,-2.9803,9.2968,-37.6971,2.0894,7.4837,-37.6971,5.8544,10.6138,-37.6971,8.3505,10.6138,-37.6971,8.3505,9.2968,-37.6971,2.0894,13.2,-37.6971,2.9802,7.4837,-37.6971,5.8544,4.2166,-37.6971,8.4599,5.9535,-37.6971,12.0669,5.9535,-37.6971,12.0669,7.4837,-37.6971,5.8544,10.6138,-37.6971,8.3505,4.2166,-37.6971,8.4599,0.1425,-37.6971,9.3897,0.1425,-37.6971,13.3932,0.1425,-37.6971,13.3932,4.2166,-37.6971,8.4599,5.9535,-37.6971,12.0669,0.1425,-37.6971,9.3897,-3.9316,-37.6971,8.4599,-5.6686,-37.6971,12.0669,-5.6686,-37.6971,12.0669,0.1425,-37.6971,9.3897,0.1425,-37.6971,13.3932,-3.9316,-37.6971,8.4599,-7.1987,-37.6971,5.8544,-10.3288,-37.6971,8.3506,-10.3288,-37.6971,8.3506,-3.9316,-37.6971,8.4599,-5.6686,-37.6971,12.0669,-7.1987,-37.6971,5.8544,-9.0118,-37.6971,2.0894,-12.9149,-37.6971,2.9803,-12.9149,-37.6971,2.9803,-7.1987,-37.6971,5.8544,-10.3288,-37.6971,8.3506,-9.0118,-37.6971,2.0894,-9.0118,-37.6971,-2.0894,-12.9149,-37.6971,-2.9803,-12.9149,-37.6971,-2.9803,-9.0118,-37.6971,2.0894,-12.9149,-37.6971,2.9803,-9.0118,-37.6971,-2.0894,-7.1987,-37.6971,-5.8544,-10.3288,-37.6971,-8.3506,-10.3288,-37.6971,-8.3506,-9.0118,-37.6971,-2.0894,-12.9149,-37.6971,-2.9803,-7.1987,-37.6971,-5.8544,-3.9316,-37.6971,-8.4599,-5.6686,-37.6971,-12.0669,-5.6686,-37.6971,-12.0669,-7.1987,-37.6971,-5.8544,-10.3288,-37.6971,-8.3506,0.1425,-44.7984,-9.3897,-3.9316,-44.7984,-8.4599,-5.6686,-44.7984,-12.0669,-5.6686,-44.7984,-12.0669,0.1425,-44.7984,-9.3897,0.1425,-44.7984,-13.3932,-3.9316,-44.7984,-8.4599,-7.1987,-44.7984,-5.8544,-10.3288,-44.7984,-8.3506,-10.3288,-44.7984,-8.3506,-3.9316,-44.7984,-8.4599,-5.6686,-44.7984,-12.0669,-7.1987,-44.7984,-5.8544,-9.0118,-44.7984,-2.0894,-12.9149,-44.7984,-2.9803,-12.9149,-44.7984,-2.9803,-7.1987,-44.7984,-5.8544,-10.3288,-44.7984,-8.3506,-9.0118,-44.7984,-2.0894,-9.0118,-44.7984,2.0894,-12.9149,-44.7984,2.9803,-12.9149,-44.7984,2.9803,-9.0118,-44.7984,-2.0894,-12.9149,-44.7984,-2.9803,-9.0118,-44.7984,2.0894,-7.1987,-44.7984,5.8544,-10.3288,-44.7984,8.3506,-10.3288,-44.7984,8.3506,-9.0118,-44.7984,2.0894,-12.9149,-44.7984,2.9803,-7.1987,-44.7984,5.8544,-3.9316,-44.7984,8.4599,-5.6686,-44.7984,12.0669,-5.6686,-44.7984,12.0669,-7.1987,-44.7984,5.8544,-10.3288,-44.7984,8.3506,-3.9316,-44.7984,8.4599,0.1425,-44.7984,9.3897,0.1425,-44.7984,13.3932,0.1425,-44.7984,13.3932,-3.9316,-44.7984,8.4599,-5.6686,-44.7984,12.0669,0.1425,-44.7984,9.3897,4.2166,-44.7984,8.4599,5.9535,-44.7984,12.0669,5.9535,-44.7984,12.0669,0.1425,-44.7984,9.3897,0.1425,-44.7984,13.3932,4.2166,-44.7984,8.4599,7.4837,-44.7984,5.8544,10.6138,-44.7984,8.3505,10.6138,-44.7984,8.3505,4.2166,-44.7984,8.4599,5.9535,-44.7984,12.0669,7.4837,-44.7984,5.8544,9.2968,-44.7984,2.0894,13.2,-44.7984,2.9802,13.2,-44.7984,2.9802,7.4837,-44.7984,5.8544,10.6138,-44.7984,8.3505,9.2968,-44.7984,2.0894,9.2968,-44.7984,-2.0894,13.2,-44.7984,-2.9803,13.2,-44.7984,-2.9803,9.2968,-44.7984,2.0894,13.2,-44.7984,2.9802,9.2968,-44.7984,-2.0894,7.4837,-44.7984,-5.8544,10.6138,-44.7984,-8.3506,10.6138,-44.7984,-8.3506,9.2968,-44.7984,-2.0894,13.2,-44.7984,-2.9803,7.4837,-44.7984,-5.8544,4.2166,-44.7984,-8.4599,5.9536,-44.7984,-12.0669,5.9536,-44.7984,-12.0669,7.4837,-44.7984,-5.8544,10.6138,-44.7984,-8.3506,4.2166,-44.7984,-8.4599,0.1425,-44.7984,-9.3897,0.1425,-44.7984,-13.3932,0.1425,-44.7984,-13.3932,4.2166,-44.7984,-8.4599,5.9536,-44.7984,-12.0669,0.1425,40.3208,-9.3897,0.1425,44.9308,-9.3897,-3.9316,44.9308,-8.4599,-3.9316,44.9308,-8.4599,0.1425,40.3208,-9.3897,-3.9316,40.3208,-8.4599,4.2166,40.3208,-8.4599,4.2166,44.9308,-8.4599,0.1425,44.9308,-9.3897,0.1425,44.9308,-9.3897,4.2166,40.3208,-8.4599,0.1425,40.3208,-9.3897,7.4837,40.3208,-5.8544,7.4837,44.9308,-5.8544,4.2166,44.9308,-8.4599,4.2166,44.9308,-8.4599,7.4837,40.3208,-5.8544,4.2166,40.3208,-8.4599,9.2968,40.3208,-2.0894,9.2968,44.9308,-2.0894,7.4837,44.9308,-5.8544,7.4837,44.9308,-5.8544,9.2968,40.3208,-2.0894,7.4837,40.3208,-5.8544,9.2968,40.3208,2.0894,9.2968,44.9308,2.0894,9.2968,44.9308,-2.0894,9.2968,44.9308,-2.0894,9.2968,40.3208,2.0894,9.2968,40.3208,-2.0894,7.4837,40.3208,5.8544,7.4837,44.9308,5.8544,9.2968,44.9308,2.0894,9.2968,44.9308,2.0894,7.4837,40.3208,5.8544,9.2968,40.3208,2.0894,4.2166,40.3208,8.4599,4.2166,44.9308,8.4599,7.4837,44.9308,5.8544,7.4837,44.9308,5.8544,4.2166,40.3208,8.4599,7.4837,40.3208,5.8544,0.1425,40.3208,9.3897,0.1425,44.9308,9.3897,4.2166,44.9308,8.4599,4.2166,44.9308,8.4599,0.1425,40.3208,9.3897,4.2166,40.3208,8.4599,-3.9316,40.3208,8.4599,-3.9316,44.9308,8.4599,0.1425,44.9308,9.3897,0.1425,44.9308,9.3897,-3.9316,40.3208,8.4599,0.1425,40.3208,9.3897,-7.1987,40.3208,5.8544,-7.1987,44.9308,5.8544,-3.9316,44.9308,8.4599,-3.9316,44.9308,8.4599,-7.1987,40.3208,5.8544,-3.9316,40.3208,8.4599,-9.0118,40.3208,2.0894,-9.0118,44.9308,2.0894,-7.1987,44.9308,5.8544,-7.1987,44.9308,5.8544,-9.0118,40.3208,2.0894,-7.1987,40.3208,5.8544,-9.0118,40.3208,-2.0894,-9.0118,44.9308,-2.0894,-9.0118,44.9308,2.0894,-9.0118,44.9308,2.0894,-9.0118,40.3208,-2.0894,-9.0118,40.3208,2.0894,-7.1987,40.3208,-5.8544,-7.1987,44.9308,-5.8544,-9.0118,44.9308,-2.0894,-9.0118,44.9308,-2.0894,-7.1987,40.3208,-5.8544,-9.0118,40.3208,-2.0894,-3.9316,40.3208,-8.4599,-3.9316,44.9308,-8.4599,-7.1987,44.9308,-5.8544,-7.1987,44.9308,-5.8544,-3.9316,40.3208,-8.4599,-7.1987,40.3208,-5.8544,-7.1987,40.3208,-5.8544,-9.0118,40.3208,-2.0894,-9.0118,40.3208,2.0894,-9.0118,40.3208,2.0894,-7.1987,40.3208,-5.8544,0.1425,40.3208,-9.3897,-9.0118,40.3208,2.0894,-7.1987,40.3208,5.8544,-3.9316,40.3208,8.4599,-3.9316,40.3208,8.4599,-9.0118,40.3208,2.0894,4.2166,40.3208,8.4599,-3.9316,40.3208,8.4599,0.1425,40.3208,9.3897,4.2166,40.3208,8.4599,4.2166,40.3208,8.4599,7.4837,40.3208,5.8544,9.2968,40.3208,2.0894,9.2968,40.3208,2.0894,4.2166,40.3208,8.4599,7.4837,40.3208,-5.8544,9.2968,40.3208,2.0894,9.2968,40.3208,-2.0894,7.4837,40.3208,-5.8544,7.4837,40.3208,-5.8544,4.2166,40.3208,-8.4599,0.1425,40.3208,-9.3897,0.1425,40.3208,-9.3897,7.4837,40.3208,-5.8544,4.2166,40.3208,8.4599,-9.0118,40.3208,2.0894,4.2166,40.3208,8.4599,0.1425,40.3208,-9.3897,-3.9316,40.3208,-8.4599,-7.1987,40.3208,-5.8544,0.1425,40.3208,-9.3897,10.7801,46.4706,0.25,11.0675,43.9792,0.25,11.1468,43.9792,0,11.1468,43.9792,0,10.7801,46.4706,0.25,10.8572,46.4887,0,10.5692,46.4214,0.433,10.851,43.9792,0.433,11.0675,43.9792,0.25,11.0675,43.9792,0.25,10.5692,46.4214,0.433,10.7801,46.4706,0.25,10.2812,46.3542,0.5,10.5553,43.9792,0.5,10.851,43.9792,0.433,10.851,43.9792,0.433,10.2812,46.3542,0.5,10.5692,46.4214,0.433,9.9932,46.287,0.433,10.2595,43.9792,0.433,10.5553,43.9792,0.5,10.5553,43.9792,0.5,9.9932,46.287,0.433,10.2812,46.3542,0.5,9.7824,46.2378,0.25,10.043,43.9792,0.25,10.2595,43.9792,0.433,10.2595,43.9792,0.433,9.7824,46.2378,0.25,9.9932,46.287,0.433,9.7052,46.2198,0,9.9637,43.9792,0,10.043,43.9792,0.25,10.043,43.9792,0.25,9.7052,46.2198,0,9.7824,46.2378,0.25,9.7824,46.2378,-0.25,10.043,43.9792,-0.25,9.9637,43.9792,0,9.9637,43.9792,0,9.7824,46.2378,-0.25,9.7052,46.2198,0,9.9932,46.287,-0.433,10.2595,43.9792,-0.433,10.043,43.9792,-0.25,10.043,43.9792,-0.25,9.9932,46.287,-0.433,9.7824,46.2378,-0.25,10.2812,46.3542,-0.5,10.5553,43.9792,-0.5,10.2595,43.9792,-0.433,10.2595,43.9792,-0.433,10.2812,46.3542,-0.5,9.9932,46.287,-0.433,10.5692,46.4214,-0.433,10.851,43.9792,-0.433,10.5553,43.9792,-0.5,10.5553,43.9792,-0.5,10.5692,46.4214,-0.433,10.2812,46.3542,-0.5,10.7801,46.4706,-0.25,11.0675,43.9792,-0.25,10.851,43.9792,-0.433,10.851,43.9792,-0.433,10.7801,46.4706,-0.25,10.5692,46.4214,-0.433,10.8572,46.4887,0,11.1468,43.9792,0,11.0675,43.9792,-0.25,11.0675,43.9792,-0.25,10.8572,46.4887,0,10.7801,46.4706,-0.25,9.9608,48.7604,0.25,10.7801,46.4706,0.25,10.8572,46.4887,0,10.8572,46.4887,0,9.9608,48.7604,0.25,10.0319,48.7952,0,9.7664,48.6653,0.433,10.5692,46.4214,0.433,10.7801,46.4706,0.25,10.7801,46.4706,0.25,9.7664,48.6653,0.433,9.9608,48.7604,0.25,9.5008,48.5353,0.5,10.2812,46.3542,0.5,10.5692,46.4214,0.433,10.5692,46.4214,0.433,9.5008,48.5353,0.5,9.7664,48.6653,0.433,9.2353,48.4054,0.433,9.9932,46.287,0.433,10.2812,46.3542,0.5,10.2812,46.3542,0.5,9.2353,48.4054,0.433,9.5008,48.5353,0.5,9.0409,48.3102,0.25,9.7824,46.2378,0.25,9.9932,46.287,0.433,9.9932,46.287,0.433,9.0409,48.3102,0.25,9.2353,48.4054,0.433,8.9697,48.2754,0,9.7052,46.2198,0,9.7824,46.2378,0.25,9.7824,46.2378,0.25,8.9697,48.2754,0,9.0409,48.3102,0.25,9.0409,48.3102,-0.25,9.7824,46.2378,-0.25,9.7052,46.2198,0,9.7052,46.2198,0,9.0409,48.3102,-0.25,8.9697,48.2754,0,9.2353,48.4054,-0.433,9.9932,46.287,-0.433,9.7824,46.2378,-0.25,9.7824,46.2378,-0.25,9.2353,48.4054,-0.433,9.0409,48.3102,-0.25,9.5008,48.5353,-0.5,10.2812,46.3542,-0.5,9.9932,46.287,-0.433,9.9932,46.287,-0.433,9.5008,48.5353,-0.5,9.2353,48.4054,-0.433,9.7664,48.6653,-0.433,10.5692,46.4214,-0.433,10.2812,46.3542,-0.5,10.2812,46.3542,-0.5,9.7664,48.6653,-0.433,9.5008,48.5353,-0.5,9.9608,48.7604,-0.25,10.7801,46.4706,-0.25,10.5692,46.4214,-0.433,10.5692,46.4214,-0.433,9.9608,48.7604,-0.25,9.7664,48.6653,-0.433,10.0319,48.7952,0,10.8572,46.4887,0,10.7801,46.4706,-0.25,10.7801,46.4706,-0.25,10.0319,48.7952,0,9.9608,48.7604,-0.25,8.6759,50.7803,0.25,9.9608,48.7604,0.25,10.0319,48.7952,0,10.0319,48.7952,0,8.6759,50.7803,0.25,8.7377,50.8298,0,8.5071,50.6449,0.433,9.7664,48.6653,0.433,9.9608,48.7604,0.25,9.9608,48.7604,0.25,8.5071,50.6449,0.433,8.6759,50.7803,0.25,8.2765,50.46,0.5,9.5008,48.5353,0.5,9.7664,48.6653,0.433,9.7664,48.6653,0.433,8.2765,50.46,0.5,8.5071,50.6449,0.433,8.0459,50.275,0.433,9.2353,48.4054,0.433,9.5008,48.5353,0.5,9.5008,48.5353,0.5,8.0459,50.275,0.433,8.2765,50.46,0.5,7.8771,50.1396,0.25,9.0409,48.3102,0.25,9.2353,48.4054,0.433,9.2353,48.4054,0.433,7.8771,50.1396,0.25,8.0459,50.275,0.433,7.8154,50.0901,0,8.9697,48.2754,0,9.0409,48.3102,0.25,9.0409,48.3102,0.25,7.8154,50.0901,0,7.8771,50.1396,0.25,7.8771,50.1396,-0.25,9.0409,48.3102,-0.25,8.9697,48.2754,0,8.9697,48.2754,0,7.8771,50.1396,-0.25,7.8154,50.0901,0,8.0459,50.275,-0.433,9.2353,48.4054,-0.433,9.0409,48.3102,-0.25,9.0409,48.3102,-0.25,8.0459,50.275,-0.433,7.8771,50.1396,-0.25,8.2765,50.46,-0.5,9.5008,48.5353,-0.5,9.2353,48.4054,-0.433,9.2353,48.4054,-0.433,8.2765,50.46,-0.5,8.0459,50.275,-0.433,8.5071,50.6449,-0.433,9.7664,48.6653,-0.433,9.5008,48.5353,-0.5,9.5008,48.5353,-0.5,8.5071,50.6449,-0.433,8.2765,50.46,-0.5,8.6759,50.7803,-0.25,9.9608,48.7604,-0.25,9.7664,48.6653,-0.433,9.7664,48.6653,-0.433,8.6759,50.7803,-0.25,8.5071,50.6449,-0.433,8.7377,50.8298,0,10.0319,48.7952,0,9.9608,48.7604,-0.25,9.9608,48.7604,-0.25,8.7377,50.8298,0,8.6759,50.7803,-0.25,6.9911,52.465,0.25,8.6759,50.7803,0.25,8.7377,50.8298,0,8.7377,50.8298,0,6.9911,52.465,0.25,7.0407,52.5268,0,6.8557,52.2962,0.433,8.5071,50.6449,0.433,8.6759,50.7803,0.25,8.6759,50.7803,0.25,6.8557,52.2962,0.433,6.9911,52.465,0.25,6.6708,52.0657,0.5,8.2765,50.46,0.5,8.5071,50.6449,0.433,8.5071,50.6449,0.433,6.6708,52.0657,0.5,6.8557,52.2962,0.433,6.4859,51.8351,0.433,8.0459,50.275,0.433,8.2765,50.46,0.5,8.2765,50.46,0.5,6.4859,51.8351,0.433,6.6708,52.0657,0.5,6.3505,51.6663,0.25,7.8771,50.1396,0.25,8.0459,50.275,0.433,8.0459,50.275,0.433,6.3505,51.6663,0.25,6.4859,51.8351,0.433,6.3009,51.6045,0,7.8154,50.0901,0,7.8771,50.1396,0.25,7.8771,50.1396,0.25,6.3009,51.6045,0,6.3505,51.6663,0.25,6.3505,51.6663,-0.25,7.8771,50.1396,-0.25,7.8154,50.0901,0,7.8154,50.0901,0,6.3505,51.6663,-0.25,6.3009,51.6045,0,6.4859,51.8351,-0.433,8.0459,50.275,-0.433,7.8771,50.1396,-0.25,7.8771,50.1396,-0.25,6.4859,51.8351,-0.433,6.3505,51.6663,-0.25,6.6708,52.0657,-0.5,8.2765,50.46,-0.5,8.0459,50.275,-0.433,8.0459,50.275,-0.433,6.6708,52.0657,-0.5,6.4859,51.8351,-0.433,6.8557,52.2962,-0.433,8.5071,50.6449,-0.433,8.2765,50.46,-0.5,8.2765,50.46,-0.5,6.8557,52.2962,-0.433,6.6708,52.0657,-0.5,6.9911,52.465,-0.25,8.6759,50.7803,-0.25,8.5071,50.6449,-0.433,8.5071,50.6449,-0.433,6.9911,52.465,-0.25,6.8557,52.2962,-0.433,7.0407,52.5268,0,8.7377,50.8298,0,8.6759,50.7803,-0.25,8.6759,50.7803,-0.25,7.0407,52.5268,0,6.9911,52.465,-0.25,4.9712,53.7499,0.25,6.9911,52.465,0.25,7.0407,52.5268,0,7.0407,52.5268,0,4.9712,53.7499,0.25,5.006,53.8211,0,4.8761,53.5555,0.433,6.8557,52.2962,0.433,6.9911,52.465,0.25,6.9911,52.465,0.25,4.8761,53.5555,0.433,4.9712,53.7499,0.25,4.7462,53.29,0.5,6.6708,52.0657,0.5,6.8557,52.2962,0.433,6.8557,52.2962,0.433,4.7462,53.29,0.5,4.8761,53.5555,0.433,4.6162,53.0244,0.433,6.4859,51.8351,0.433,6.6708,52.0657,0.5,6.6708,52.0657,0.5,4.6162,53.0244,0.433,4.7462,53.29,0.5,4.5211,52.83,0.25,6.3505,51.6663,0.25,6.4859,51.8351,0.433,6.4859,51.8351,0.433,4.5211,52.83,0.25,4.6162,53.0244,0.433,4.4863,52.7589,0,6.3009,51.6045,0,6.3505,51.6663,0.25,6.3505,51.6663,0.25,4.4863,52.7589,0,4.5211,52.83,0.25,4.5211,52.83,-0.25,6.3505,51.6663,-0.25,6.3009,51.6045,0,6.3009,51.6045,0,4.5211,52.83,-0.25,4.4863,52.7589,0,4.6162,53.0244,-0.433,6.4859,51.8351,-0.433,6.3505,51.6663,-0.25,6.3505,51.6663,-0.25,4.6162,53.0244,-0.433,4.5211,52.83,-0.25,4.7462,53.29,-0.5,6.6708,52.0657,-0.5,6.4859,51.8351,-0.433,6.4859,51.8351,-0.433,4.7462,53.29,-0.5,4.6162,53.0244,-0.433,4.8761,53.5555,-0.433,6.8557,52.2962,-0.433,6.6708,52.0657,-0.5,6.6708,52.0657,-0.5,4.8761,53.5555,-0.433,4.7462,53.29,-0.5,4.9712,53.7499,-0.25,6.9911,52.465,-0.25,6.8557,52.2962,-0.433,6.8557,52.2962,-0.433,4.9712,53.7499,-0.25,4.8761,53.5555,-0.433,5.006,53.8211,0,7.0407,52.5268,0,6.9911,52.465,-0.25,6.9911,52.465,-0.25,5.006,53.8211,0,4.9712,53.7499,-0.25,2.6815,54.5692,0.25,4.9712,53.7499,0.25,5.006,53.8211,0,5.006,53.8211,0,2.6815,54.5692,0.25,2.6995,54.6464,0,2.6323,54.3584,0.433,4.8761,53.5555,0.433,4.9712,53.7499,0.25,4.9712,53.7499,0.25,2.6323,54.3584,0.433,2.6815,54.5692,0.25,2.5651,54.0704,0.5,4.7462,53.29,0.5,4.8761,53.5555,0.433,4.8761,53.5555,0.433,2.5651,54.0704,0.5,2.6323,54.3584,0.433,2.4979,53.7824,0.433,4.6162,53.0244,0.433,4.7462,53.29,0.5,4.7462,53.29,0.5,2.4979,53.7824,0.433,2.5651,54.0704,0.5,2.4487,53.5716,0.25,4.5211,52.83,0.25,4.6162,53.0244,0.433,4.6162,53.0244,0.433,2.4487,53.5716,0.25,2.4979,53.7824,0.433,2.4307,53.4944,0,4.4863,52.7589,0,4.5211,52.83,0.25,4.5211,52.83,0.25,2.4307,53.4944,0,2.4487,53.5716,0.25,2.4487,53.5716,-0.25,4.5211,52.83,-0.25,4.4863,52.7589,0,4.4863,52.7589,0,2.4487,53.5716,-0.25,2.4307,53.4944,0,2.4979,53.7824,-0.433,4.6162,53.0244,-0.433,4.5211,52.83,-0.25,4.5211,52.83,-0.25,2.4979,53.7824,-0.433,2.4487,53.5716,-0.25,2.5651,54.0704,-0.5,4.7462,53.29,-0.5,4.6162,53.0244,-0.433,4.6162,53.0244,-0.433,2.5651,54.0704,-0.5,2.4979,53.7824,-0.433,2.6323,54.3584,-0.433,4.8761,53.5555,-0.433,4.7462,53.29,-0.5,4.7462,53.29,-0.5,2.6323,54.3584,-0.433,2.5651,54.0704,-0.5,2.6815,54.5692,-0.25,4.9712,53.7499,-0.25,4.8761,53.5555,-0.433,4.8761,53.5555,-0.433,2.6815,54.5692,-0.25,2.6323,54.3584,-0.433,2.6995,54.6464,0,5.006,53.8211,0,4.9712,53.7499,-0.25,4.9712,53.7499,-0.25,2.6995,54.6464,0,2.6815,54.5692,-0.25,0.1901,54.8567,0.25,2.6815,54.5692,0.25,2.6995,54.6464,0,2.6995,54.6464,0,0.1901,54.8567,0.25,0.1901,54.9359,0,0.1901,54.6402,0.433,2.6323,54.3584,0.433,2.6815,54.5692,0.25,2.6815,54.5692,0.25,0.1901,54.6402,0.433,0.1901,54.8567,0.25,0.1901,54.3444,0.5,2.5651,54.0704,0.5,2.6323,54.3584,0.433,2.6323,54.3584,0.433,0.1901,54.3444,0.5,0.1901,54.6402,0.433,0.1901,54.0487,0.433,2.4979,53.7824,0.433,2.5651,54.0704,0.5,2.5651,54.0704,0.5,0.1901,54.0487,0.433,0.1901,54.3444,0.5,0.1901,53.8321,0.25,2.4487,53.5716,0.25,2.4979,53.7824,0.433,2.4979,53.7824,0.433,0.1901,53.8321,0.25,0.1901,54.0487,0.433,0.1901,53.7529,0,2.4307,53.4944,0,2.4487,53.5716,0.25,2.4487,53.5716,0.25,0.1901,53.7529,0,0.1901,53.8321,0.25,0.1901,53.8321,-0.25,2.4487,53.5716,-0.25,2.4307,53.4944,0,2.4307,53.4944,0,0.1901,53.8321,-0.25,0.1901,53.7529,0,0.1901,54.0487,-0.433,2.4979,53.7824,-0.433,2.4487,53.5716,-0.25,2.4487,53.5716,-0.25,0.1901,54.0487,-0.433,0.1901,53.8321,-0.25,0.1901,54.3444,-0.5,2.5651,54.0704,-0.5,2.4979,53.7824,-0.433,2.4979,53.7824,-0.433,0.1901,54.3444,-0.5,0.1901,54.0487,-0.433,0.1901,54.6402,-0.433,2.6323,54.3584,-0.433,2.5651,54.0704,-0.5,2.5651,54.0704,-0.5,0.1901,54.6402,-0.433,0.1901,54.3444,-0.5,0.1901,54.8567,-0.25,2.6815,54.5692,-0.25,2.6323,54.3584,-0.433,2.6323,54.3584,-0.433,0.1901,54.8567,-0.25,0.1901,54.6402,-0.433,0.1901,54.9359,0,2.6995,54.6464,0,2.6815,54.5692,-0.25,2.6815,54.5692,-0.25,0.1901,54.9359,0,0.1901,54.8567,-0.25,-2.3013,54.5692,0.25,0.1901,54.8567,0.25,0.1901,54.9359,0,0.1901,54.9359,0,-2.3013,54.5692,0.25,-2.3193,54.6464,0,-2.2521,54.3584,0.433,0.1901,54.6402,0.433,0.1901,54.8567,0.25,0.1901,54.8567,0.25,-2.2521,54.3584,0.433,-2.3013,54.5692,0.25,-2.1849,54.0704,0.5,0.1901,54.3444,0.5,0.1901,54.6402,0.433,0.1901,54.6402,0.433,-2.1849,54.0704,0.5,-2.2521,54.3584,0.433,-2.1177,53.7824,0.433,0.1901,54.0487,0.433,0.1901,54.3444,0.5,0.1901,54.3444,0.5,-2.1177,53.7824,0.433,-2.1849,54.0704,0.5,-2.0685,53.5716,0.25,0.1901,53.8321,0.25,0.1901,54.0487,0.433,0.1901,54.0487,0.433,-2.0685,53.5716,0.25,-2.1177,53.7824,0.433,-2.0505,53.4944,0,0.1901,53.7529,0,0.1901,53.8321,0.25,0.1901,53.8321,0.25,-2.0505,53.4944,0,-2.0685,53.5716,0.25,-2.0685,53.5716,-0.25,0.1901,53.8321,-0.25,0.1901,53.7529,0,0.1901,53.7529,0,-2.0685,53.5716,-0.25,-2.0505,53.4944,0,-2.1177,53.7824,-0.433,0.1901,54.0487,-0.433,0.1901,53.8321,-0.25,0.1901,53.8321,-0.25,-2.1177,53.7824,-0.433,-2.0685,53.5716,-0.25,-2.1849,54.0704,-0.5,0.1901,54.3444,-0.5,0.1901,54.0487,-0.433,0.1901,54.0487,-0.433,-2.1849,54.0704,-0.5,-2.1177,53.7824,-0.433,-2.2521,54.3584,-0.433,0.1901,54.6402,-0.433,0.1901,54.3444,-0.5,0.1901,54.3444,-0.5,-2.2521,54.3584,-0.433,-2.1849,54.0704,-0.5,-2.3013,54.5692,-0.25,0.1901,54.8567,-0.25,0.1901,54.6402,-0.433,0.1901,54.6402,-0.433,-2.3013,54.5692,-0.25,-2.2521,54.3584,-0.433,-2.3193,54.6464,0,0.1901,54.9359,0,0.1901,54.8567,-0.25,0.1901,54.8567,-0.25,-2.3193,54.6464,0,-2.3013,54.5692,-0.25,-4.5911,53.7499,0.25,-2.3013,54.5692,0.25,-2.3193,54.6464,0,-2.3193,54.6464,0,-4.5911,53.7499,0.25,-4.6259,53.8211,0,-4.496,53.5555,0.433,-2.2521,54.3584,0.433,-2.3013,54.5692,0.25,-2.3013,54.5692,0.25,-4.496,53.5555,0.433,-4.5911,53.7499,0.25,-4.366,53.29,0.5,-2.1849,54.0704,0.5,-2.2521,54.3584,0.433,-2.2521,54.3584,0.433,-4.366,53.29,0.5,-4.496,53.5555,0.433,-4.2361,53.0244,0.433,-2.1177,53.7824,0.433,-2.1849,54.0704,0.5,-2.1849,54.0704,0.5,-4.2361,53.0244,0.433,-4.366,53.29,0.5,-4.1409,52.83,0.25,-2.0685,53.5716,0.25,-2.1177,53.7824,0.433,-2.1177,53.7824,0.433,-4.1409,52.83,0.25,-4.2361,53.0244,0.433,-4.1061,52.7589,0,-2.0505,53.4944,0,-2.0685,53.5716,0.25,-2.0685,53.5716,0.25,-4.1061,52.7589,0,-4.1409,52.83,0.25,-4.1409,52.83,-0.25,-2.0685,53.5716,-0.25,-2.0505,53.4944,0,-2.0505,53.4944,0,-4.1409,52.83,-0.25,-4.1061,52.7589,0,-4.2361,53.0244,-0.433,-2.1177,53.7824,-0.433,-2.0685,53.5716,-0.25,-2.0685,53.5716,-0.25,-4.2361,53.0244,-0.433,-4.1409,52.83,-0.25,-4.366,53.29,-0.5,-2.1849,54.0704,-0.5,-2.1177,53.7824,-0.433,-2.1177,53.7824,-0.433,-4.366,53.29,-0.5,-4.2361,53.0244,-0.433,-4.496,53.5555,-0.433,-2.2521,54.3584,-0.433,-2.1849,54.0704,-0.5,-2.1849,54.0704,-0.5,-4.496,53.5555,-0.433,-4.366,53.29,-0.5,-4.5911,53.7499,-0.25,-2.3013,54.5692,-0.25,-2.2521,54.3584,-0.433,-2.2521,54.3584,-0.433,-4.5911,53.7499,-0.25,-4.496,53.5555,-0.433,-4.6259,53.8211,0,-2.3193,54.6464,0,-2.3013,54.5692,-0.25,-2.3013,54.5692,-0.25,-4.6259,53.8211,0,-4.5911,53.7499,-0.25,-6.611,52.465,0.25,-4.5911,53.7499,0.25,-4.6259,53.8211,0,-4.6259,53.8211,0,-6.611,52.465,0.25,-6.6605,52.5268,0,-6.4756,52.2962,0.433,-4.496,53.5555,0.433,-4.5911,53.7499,0.25,-4.5911,53.7499,0.25,-6.4756,52.2962,0.433,-6.611,52.465,0.25,-6.2906,52.0657,0.5,-4.366,53.29,0.5,-4.496,53.5555,0.433,-4.496,53.5555,0.433,-6.2906,52.0657,0.5,-6.4756,52.2962,0.433,-6.1057,51.8351,0.433,-4.2361,53.0244,0.433,-4.366,53.29,0.5,-4.366,53.29,0.5,-6.1057,51.8351,0.433,-6.2906,52.0657,0.5,-5.9703,51.6663,0.25,-4.1409,52.83,0.25,-4.2361,53.0244,0.433,-4.2361,53.0244,0.433,-5.9703,51.6663,0.25,-6.1057,51.8351,0.433,-5.9208,51.6045,0,-4.1061,52.7589,0,-4.1409,52.83,0.25,-4.1409,52.83,0.25,-5.9208,51.6045,0,-5.9703,51.6663,0.25,-5.9703,51.6663,-0.25,-4.1409,52.83,-0.25,-4.1061,52.7589,0,-4.1061,52.7589,0,-5.9703,51.6663,-0.25,-5.9208,51.6045,0,-6.1057,51.8351,-0.433,-4.2361,53.0244,-0.433,-4.1409,52.83,-0.25,-4.1409,52.83,-0.25,-6.1057,51.8351,-0.433,-5.9703,51.6663,-0.25,-6.2906,52.0657,-0.5,-4.366,53.29,-0.5,-4.2361,53.0244,-0.433,-4.2361,53.0244,-0.433,-6.2906,52.0657,-0.5,-6.1057,51.8351,-0.433,-6.4756,52.2962,-0.433,-4.496,53.5555,-0.433,-4.366,53.29,-0.5,-4.366,53.29,-0.5,-6.4756,52.2962,-0.433,-6.2906,52.0657,-0.5,-6.611,52.465,-0.25,-4.5911,53.7499,-0.25,-4.496,53.5555,-0.433,-4.496,53.5555,-0.433,-6.611,52.465,-0.25,-6.4756,52.2962,-0.433,-6.6605,52.5268,0,-4.6259,53.8211,0,-4.5911,53.7499,-0.25,-4.5911,53.7499,-0.25,-6.6605,52.5268,0,-6.611,52.465,-0.25,-8.2957,50.7803,0.25,-6.611,52.465,0.25,-6.6605,52.5268,0,-6.6605,52.5268,0,-8.2957,50.7803,0.25,-8.3575,50.8298,0,-8.1269,50.6449,0.433,-6.4756,52.2962,0.433,-6.611,52.465,0.25,-6.611,52.465,0.25,-8.1269,50.6449,0.433,-8.2957,50.7803,0.25,-7.8964,50.4599,0.5,-6.2906,52.0657,0.5,-6.4756,52.2962,0.433,-6.4756,52.2962,0.433,-7.8964,50.4599,0.5,-8.1269,50.6449,0.433,-7.6658,50.275,0.433,-6.1057,51.8351,0.433,-6.2906,52.0657,0.5,-6.2906,52.0657,0.5,-7.6658,50.275,0.433,-7.8964,50.4599,0.5,-7.497,50.1396,0.25,-5.9703,51.6663,0.25,-6.1057,51.8351,0.433,-6.1057,51.8351,0.433,-7.497,50.1396,0.25,-7.6658,50.275,0.433,-7.4352,50.0901,0,-5.9208,51.6045,0,-5.9703,51.6663,0.25,-5.9703,51.6663,0.25,-7.4352,50.0901,0,-7.497,50.1396,0.25,-7.497,50.1396,-0.25,-5.9703,51.6663,-0.25,-5.9208,51.6045,0,-5.9208,51.6045,0,-7.497,50.1396,-0.25,-7.4352,50.0901,0,-7.6658,50.275,-0.433,-6.1057,51.8351,-0.433,-5.9703,51.6663,-0.25,-5.9703,51.6663,-0.25,-7.6658,50.275,-0.433,-7.497,50.1396,-0.25,-7.8964,50.4599,-0.5,-6.2906,52.0657,-0.5,-6.1057,51.8351,-0.433,-6.1057,51.8351,-0.433,-7.8964,50.4599,-0.5,-7.6658,50.275,-0.433,-8.1269,50.6449,-0.433,-6.4756,52.2962,-0.433,-6.2906,52.0657,-0.5,-6.2906,52.0657,-0.5,-8.1269,50.6449,-0.433,-7.8964,50.4599,-0.5,-8.2957,50.7803,-0.25,-6.611,52.465,-0.25,-6.4756,52.2962,-0.433,-6.4756,52.2962,-0.433,-8.2957,50.7803,-0.25,-8.1269,50.6449,-0.433,-8.3575,50.8298,0,-6.6605,52.5268,0,-6.611,52.465,-0.25,-6.611,52.465,-0.25,-8.3575,50.8298,0,-8.2957,50.7803,-0.25,-9.5806,48.7604,0.25,-8.2957,50.7803,0.25,-8.3575,50.8298,0,-8.3575,50.8298,0,-9.5806,48.7604,0.25,-9.6518,48.7952,0,-9.3862,48.6653,0.433,-8.1269,50.6449,0.433,-8.2957,50.7803,0.25,-8.2957,50.7803,0.25,-9.3862,48.6653,0.433,-9.5806,48.7604,0.25,-9.1207,48.5353,0.5,-7.8964,50.4599,0.5,-8.1269,50.6449,0.433,-8.1269,50.6449,0.433,-9.1207,48.5353,0.5,-9.3862,48.6653,0.433,-8.8551,48.4054,0.433,-7.6658,50.275,0.433,-7.8964,50.4599,0.5,-7.8964,50.4599,0.5,-8.8551,48.4054,0.433,-9.1207,48.5353,0.5,-8.6607,48.3102,0.25,-7.497,50.1396,0.25,-7.6658,50.275,0.433,-7.6658,50.275,0.433,-8.6607,48.3102,0.25,-8.8551,48.4054,0.433,-8.5896,48.2754,0,-7.4352,50.0901,0,-7.497,50.1396,0.25,-7.497,50.1396,0.25,-8.5896,48.2754,0,-8.6607,48.3102,0.25,-8.6607,48.3102,-0.25,-7.497,50.1396,-0.25,-7.4352,50.0901,0,-7.4352,50.0901,0,-8.6607,48.3102,-0.25,-8.5896,48.2754,0,-8.8551,48.4054,-0.433,-7.6658,50.275,-0.433,-7.497,50.1396,-0.25,-7.497,50.1396,-0.25,-8.8551,48.4054,-0.433,-8.6607,48.3102,-0.25,-9.1207,48.5353,-0.5,-7.8964,50.4599,-0.5,-7.6658,50.275,-0.433,-7.6658,50.275,-0.433,-9.1207,48.5353,-0.5,-8.8551,48.4054,-0.433,-9.3862,48.6653,-0.433,-8.1269,50.6449,-0.433,-7.8964,50.4599,-0.5,-7.8964,50.4599,-0.5,-9.3862,48.6653,-0.433,-9.1207,48.5353,-0.5,-9.5806,48.7604,-0.25,-8.2957,50.7803,-0.25,-8.1269,50.6449,-0.433,-8.1269,50.6449,-0.433,-9.5806,48.7604,-0.25,-9.3862,48.6653,-0.433,-9.6518,48.7952,0,-8.3575,50.8298,0,-8.2957,50.7803,-0.25,-8.2957,50.7803,-0.25,-9.6518,48.7952,0,-9.5806,48.7604,-0.25,-10.3999,46.4706,0.25,-9.5806,48.7604,0.25,-9.6518,48.7952,0,-9.6518,48.7952,0,-10.3999,46.4706,0.25,-10.4771,46.4886,0,-10.1891,46.4214,0.433,-9.3862,48.6653,0.433,-9.5806,48.7604,0.25,-9.5806,48.7604,0.25,-10.1891,46.4214,0.433,-10.3999,46.4706,0.25,-9.9011,46.3542,0.5,-9.1207,48.5353,0.5,-9.3862,48.6653,0.433,-9.3862,48.6653,0.433,-9.9011,46.3542,0.5,-10.1891,46.4214,0.433,-9.6131,46.287,0.433,-8.8551,48.4054,0.433,-9.1207,48.5353,0.5,-9.1207,48.5353,0.5,-9.6131,46.287,0.433,-9.9011,46.3542,0.5,-9.4022,46.2378,0.25,-8.6607,48.3102,0.25,-8.8551,48.4054,0.433,-8.8551,48.4054,0.433,-9.4022,46.2378,0.25,-9.6131,46.287,0.433,-9.3251,46.2198,0,-8.5896,48.2754,0,-8.6607,48.3102,0.25,-8.6607,48.3102,0.25,-9.3251,46.2198,0,-9.4022,46.2378,0.25,-9.4022,46.2378,-0.25,-8.6607,48.3102,-0.25,-8.5896,48.2754,0,-8.5896,48.2754,0,-9.4022,46.2378,-0.25,-9.3251,46.2198,0,-9.6131,46.287,-0.433,-8.8551,48.4054,-0.433,-8.6607,48.3102,-0.25,-8.6607,48.3102,-0.25,-9.6131,46.287,-0.433,-9.4022,46.2378,-0.25,-9.9011,46.3542,-0.5,-9.1207,48.5353,-0.5,-8.8551,48.4054,-0.433,-8.8551,48.4054,-0.433,-9.9011,46.3542,-0.5,-9.6131,46.287,-0.433,-10.1891,46.4214,-0.433,-9.3862,48.6653,-0.433,-9.1207,48.5353,-0.5,-9.1207,48.5353,-0.5,-10.1891,46.4214,-0.433,-9.9011,46.3542,-0.5,-10.3999,46.4706,-0.25,-9.5806,48.7604,-0.25,-9.3862,48.6653,-0.433,-9.3862,48.6653,-0.433,-10.3999,46.4706,-0.25,-10.1891,46.4214,-0.433,-10.4771,46.4886,0,-9.6518,48.7952,0,-9.5806,48.7604,-0.25,-9.5806,48.7604,-0.25,-10.4771,46.4886,0,-10.3999,46.4706,-0.25,-10.6874,43.9792,0.25,-10.3999,46.4706,0.25,-10.4771,46.4886,0,-10.4771,46.4886,0,-10.6874,43.9792,0.25,-10.7666,43.9792,0,-10.4709,43.9792,0.433,-10.1891,46.4214,0.433,-10.3999,46.4706,0.25,-10.3999,46.4706,0.25,-10.4709,43.9792,0.433,-10.6874,43.9792,0.25,-10.1751,43.9792,0.5,-9.9011,46.3542,0.5,-10.1891,46.4214,0.433,-10.1891,46.4214,0.433,-10.1751,43.9792,0.5,-10.4709,43.9792,0.433,-9.8793,43.9792,0.433,-9.6131,46.287,0.433,-9.9011,46.3542,0.5,-9.9011,46.3542,0.5,-9.8793,43.9792,0.433,-10.1751,43.9792,0.5,-9.6628,43.9792,0.25,-9.4022,46.2378,0.25,-9.6131,46.287,0.433,-9.6131,46.287,0.433,-9.6628,43.9792,0.25,-9.8793,43.9792,0.433,-9.5836,43.9792,0,-9.3251,46.2198,0,-9.4022,46.2378,0.25,-9.4022,46.2378,0.25,-9.5836,43.9792,0,-9.6628,43.9792,0.25,-9.6628,43.9792,-0.25,-9.4022,46.2378,-0.25,-9.3251,46.2198,0,-9.3251,46.2198,0,-9.6628,43.9792,-0.25,-9.5836,43.9792,0,-9.8793,43.9792,-0.433,-9.6131,46.287,-0.433,-9.4022,46.2378,-0.25,-9.4022,46.2378,-0.25,-9.8793,43.9792,-0.433,-9.6628,43.9792,-0.25,-10.1751,43.9792,-0.5,-9.9011,46.3542,-0.5,-9.6131,46.287,-0.433,-9.6131,46.287,-0.433,-10.1751,43.9792,-0.5,-9.8793,43.9792,-0.433,-10.4709,43.9792,-0.433,-10.1891,46.4214,-0.433,-9.9011,46.3542,-0.5,-9.9011,46.3542,-0.5,-10.4709,43.9792,-0.433,-10.1751,43.9792,-0.5,-10.6874,43.9792,-0.25,-10.3999,46.4706,-0.25,-10.1891,46.4214,-0.433,-10.1891,46.4214,-0.433,-10.6874,43.9792,-0.25,-10.4709,43.9792,-0.433,-10.7666,43.9792,0,-10.4771,46.4886,0,-10.3999,46.4706,-0.25,-10.3999,46.4706,-0.25,-10.7666,43.9792,0,-10.6874,43.9792,-0.25] )
   , make_vbo(NORMAL, [-0.3443,0.6085,-0.715,-0.3853,0.4599,-0.8,0,0.4599,-0.888,0,0.4599,-0.888,-0.3443,0.6085,-0.715,0,0.6085,-0.7935,-0.6204,0.6085,-0.4948,-0.6942,0.4599,-0.5536,-0.3853,0.4599,-0.8,-0.3853,0.4599,-0.8,-0.6204,0.6085,-0.4948,-0.3443,0.6085,-0.715,-0.7737,0.6085,-0.1766,-0.8657,0.4599,-0.1976,-0.6942,0.4599,-0.5536,-0.6942,0.4599,-0.5536,-0.7737,0.6085,-0.1766,-0.6204,0.6085,-0.4948,-0.7737,0.6085,0.1766,-0.8657,0.4599,0.1976,-0.8657,0.4599,-0.1976,-0.8657,0.4599,-0.1976,-0.7737,0.6085,0.1766,-0.7737,0.6085,-0.1766,-0.6204,0.6085,0.4948,-0.6942,0.4599,0.5536,-0.8657,0.4599,0.1976,-0.8657,0.4599,0.1976,-0.6204,0.6085,0.4948,-0.7737,0.6085,0.1766,-0.3443,0.6085,0.715,-0.3853,0.4599,0.8,-0.6942,0.4599,0.5536,-0.6942,0.4599,0.5536,-0.3443,0.6085,0.715,-0.6204,0.6085,0.4948,0,0.6085,0.7935,0,0.4599,0.888,-0.3853,0.4599,0.8,-0.3853,0.4599,0.8,0,0.6085,0.7935,-0.3443,0.6085,0.715,0.3443,0.6085,0.715,0.3853,0.4599,0.8,0,0.4599,0.888,0,0.4599,0.888,0.3443,0.6085,0.715,0,0.6085,0.7935,0.6204,0.6085,0.4948,0.6942,0.4599,0.5536,0.3853,0.4599,0.8,0.3853,0.4599,0.8,0.6204,0.6085,0.4948,0.3443,0.6085,0.715,0.7737,0.6085,0.1766,0.8657,0.4599,0.1976,0.6942,0.4599,0.5536,0.6942,0.4599,0.5536,0.7737,0.6085,0.1766,0.6204,0.6085,0.4948,0.7737,0.6085,-0.1766,0.8657,0.4599,-0.1976,0.8657,0.4599,0.1976,0.8657,0.4599,0.1976,0.7737,0.6085,-0.1766,0.7737,0.6085,0.1766,0.6204,0.6085,-0.4948,0.6942,0.4599,-0.5536,0.8657,0.4599,-0.1976,0.8657,0.4599,-0.1976,0.6204,0.6085,-0.4948,0.7737,0.6085,-0.1766,0.3443,0.6085,-0.715,0.3853,0.4599,-0.8,0.6942,0.4599,-0.5536,0.6942,0.4599,-0.5536,0.3443,0.6085,-0.715,0.6204,0.6085,-0.4948,0,0.6085,-0.7935,0,0.4599,-0.888,0.3853,0.4599,-0.8,0.3853,0.4599,-0.8,0,0.6085,-0.7935,0.3443,0.6085,-0.715,-0.4236,0.216,-0.8797,-0.3443,0.6085,-0.715,0,0.6085,-0.7935,0,0.6085,-0.7935,-0.4236,0.216,-0.8797,0,0.216,-0.9764,-0.7634,0.216,-0.6088,-0.6204,0.6085,-0.4948,-0.3443,0.6085,-0.715,-0.3443,0.6085,-0.715,-0.7634,0.216,-0.6088,-0.4236,0.216,-0.8797,-0.9519,0.216,-0.2173,-0.7737,0.6085,-0.1766,-0.6204,0.6085,-0.4948,-0.6204,0.6085,-0.4948,-0.9519,0.216,-0.2173,-0.7634,0.216,-0.6088,-0.9519,0.216,0.2173,-0.7737,0.6085,0.1766,-0.7737,0.6085,-0.1766,-0.7737,0.6085,-0.1766,-0.9519,0.216,0.2173,-0.9519,0.216,-0.2173,-0.7634,0.216,0.6088,-0.6204,0.6085,0.4948,-0.7737,0.6085,0.1766,-0.7737,0.6085,0.1766,-0.7634,0.216,0.6088,-0.9519,0.216,0.2173,-0.4236,0.216,0.8797,-0.3443,0.6085,0.715,-0.6204,0.6085,0.4948,-0.6204,0.6085,0.4948,-0.4236,0.216,0.8797,-0.7634,0.216,0.6088,0,0.216,0.9764,0,0.6085,0.7935,-0.3443,0.6085,0.715,-0.3443,0.6085,0.715,0,0.216,0.9764,-0.4236,0.216,0.8797,0.4236,0.216,0.8797,0.3443,0.6085,0.715,0,0.6085,0.7935,0,0.6085,0.7935,0.4236,0.216,0.8797,0,0.216,0.9764,0.7634,0.216,0.6088,0.6204,0.6085,0.4948,0.3443,0.6085,0.715,0.3443,0.6085,0.715,0.7634,0.216,0.6088,0.4236,0.216,0.8797,0.9519,0.216,0.2173,0.7737,0.6085,0.1766,0.6204,0.6085,0.4948,0.6204,0.6085,0.4948,0.9519,0.216,0.2173,0.7634,0.216,0.6088,0.9519,0.216,-0.2173,0.7737,0.6085,-0.1766,0.7737,0.6085,0.1766,0.7737,0.6085,0.1766,0.9519,0.216,-0.2173,0.9519,0.216,0.2173,0.7634,0.216,-0.6088,0.6204,0.6085,-0.4948,0.7737,0.6085,-0.1766,0.7737,0.6085,-0.1766,0.7634,0.216,-0.6088,0.9519,0.216,-0.2173,0.4236,0.216,-0.8797,0.3443,0.6085,-0.715,0.6204,0.6085,-0.4948,0.6204,0.6085,-0.4948,0.4236,0.216,-0.8797,0.7634,0.216,-0.6088,0,0.216,-0.9764,0,0.6085,-0.7935,0.3443,0.6085,-0.715,0.3443,0.6085,-0.715,0,0.216,-0.9764,0.4236,0.216,-0.8797,-0.4236,-0.216,-0.8797,-0.4236,0.216,-0.8797,0,0.216,-0.9764,0,0.216,-0.9764,-0.4236,-0.216,-0.8797,0,-0.216,-0.9764,-0.7634,-0.216,-0.6088,-0.7634,0.216,-0.6088,-0.4236,0.216,-0.8797,-0.4236,0.216,-0.8797,-0.7634,-0.216,-0.6088,-0.4236,-0.216,-0.8797,-0.9519,-0.216,-0.2173,-0.9519,0.216,-0.2173,-0.7634,0.216,-0.6088,-0.7634,0.216,-0.6088,-0.9519,-0.216,-0.2173,-0.7634,-0.216,-0.6088,-0.9519,-0.216,0.2173,-0.9519,0.216,0.2173,-0.9519,0.216,-0.2173,-0.9519,0.216,-0.2173,-0.9519,-0.216,0.2173,-0.9519,-0.216,-0.2173,-0.7634,-0.216,0.6088,-0.7634,0.216,0.6088,-0.9519,0.216,0.2173,-0.9519,0.216,0.2173,-0.7634,-0.216,0.6088,-0.9519,-0.216,0.2173,-0.4236,-0.216,0.8797,-0.4236,0.216,0.8797,-0.7634,0.216,0.6088,-0.7634,0.216,0.6088,-0.4236,-0.216,0.8797,-0.7634,-0.216,0.6088,0,-0.216,0.9764,0,0.216,0.9764,-0.4236,0.216,0.8797,-0.4236,0.216,0.8797,0,-0.216,0.9764,-0.4236,-0.216,0.8797,0.4236,-0.216,0.8797,0.4236,0.216,0.8797,0,0.216,0.9764,0,0.216,0.9764,0.4236,-0.216,0.8797,0,-0.216,0.9764,0.7634,-0.216,0.6088,0.7634,0.216,0.6088,0.4236,0.216,0.8797,0.4236,0.216,0.8797,0.7634,-0.216,0.6088,0.4236,-0.216,0.8797,0.9519,-0.216,0.2173,0.9519,0.216,0.2173,0.7634,0.216,0.6088,0.7634,0.216,0.6088,0.9519,-0.216,0.2173,0.7634,-0.216,0.6088,0.9519,-0.216,-0.2173,0.9519,0.216,-0.2173,0.9519,0.216,0.2173,0.9519,0.216,0.2173,0.9519,-0.216,-0.2173,0.9519,-0.216,0.2173,0.7634,-0.216,-0.6088,0.7634,0.216,-0.6088,0.9519,0.216,-0.2173,0.9519,0.216,-0.2173,0.7634,-0.216,-0.6088,0.9519,-0.216,-0.2173,0.4236,-0.216,-0.8797,0.4236,0.216,-0.8797,0.7634,0.216,-0.6088,0.7634,0.216,-0.6088,0.4236,-0.216,-0.8797,0.7634,-0.216,-0.6088,0,-0.216,-0.9764,0,0.216,-0.9764,0.4236,0.216,-0.8797,0.4236,0.216,-0.8797,0,-0.216,-0.9764,0.4236,-0.216,-0.8797,-0.3443,-0.6085,-0.715,-0.4236,-0.216,-0.8797,0,-0.216,-0.9764,0,-0.216,-0.9764,-0.3443,-0.6085,-0.715,0,-0.6085,-0.7935,-0.6204,-0.6085,-0.4948,-0.7634,-0.216,-0.6088,-0.4236,-0.216,-0.8797,-0.4236,-0.216,-0.8797,-0.6204,-0.6085,-0.4948,-0.3443,-0.6085,-0.715,-0.7737,-0.6085,-0.1766,-0.9519,-0.216,-0.2173,-0.7634,-0.216,-0.6088,-0.7634,-0.216,-0.6088,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,-0.4948,-0.7737,-0.6085,0.1766,-0.9519,-0.216,0.2173,-0.9519,-0.216,-0.2173,-0.9519,-0.216,-0.2173,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,0.4948,-0.7634,-0.216,0.6088,-0.9519,-0.216,0.2173,-0.9519,-0.216,0.2173,-0.6204,-0.6085,0.4948,-0.7737,-0.6085,0.1766,-0.3443,-0.6085,0.715,-0.4236,-0.216,0.8797,-0.7634,-0.216,0.6088,-0.7634,-0.216,0.6088,-0.3443,-0.6085,0.715,-0.6204,-0.6085,0.4948,0,-0.6085,0.7935,0,-0.216,0.9764,-0.4236,-0.216,0.8797,-0.4236,-0.216,0.8797,0,-0.6085,0.7935,-0.3443,-0.6085,0.715,0.3443,-0.6085,0.715,0.4236,-0.216,0.8797,0,-0.216,0.9764,0,-0.216,0.9764,0.3443,-0.6085,0.715,0,-0.6085,0.7935,0.6204,-0.6085,0.4948,0.7634,-0.216,0.6088,0.4236,-0.216,0.8797,0.4236,-0.216,0.8797,0.6204,-0.6085,0.4948,0.3443,-0.6085,0.715,0.7737,-0.6085,0.1766,0.9519,-0.216,0.2173,0.7634,-0.216,0.6088,0.7634,-0.216,0.6088,0.7737,-0.6085,0.1766,0.6204,-0.6085,0.4948,0.7737,-0.6085,-0.1766,0.9519,-0.216,-0.2173,0.9519,-0.216,0.2173,0.9519,-0.216,0.2173,0.7737,-0.6085,-0.1766,0.7737,-0.6085,0.1766,0.6204,-0.6085,-0.4948,0.7634,-0.216,-0.6088,0.9519,-0.216,-0.2173,0.9519,-0.216,-0.2173,0.6204,-0.6085,-0.4948,0.7737,-0.6085,-0.1766,0.3443,-0.6085,-0.715,0.4236,-0.216,-0.8797,0.7634,-0.216,-0.6088,0.7634,-0.216,-0.6088,0.3443,-0.6085,-0.715,0.6204,-0.6085,-0.4948,0,-0.6085,-0.7935,0,-0.216,-0.9764,0.4236,-0.216,-0.8797,0.4236,-0.216,-0.8797,0,-0.6085,-0.7935,0.3443,-0.6085,-0.715,-0.3853,-0.4599,-0.8,-0.3443,-0.6085,-0.715,0,-0.6085,-0.7935,0,-0.6085,-0.7935,-0.3853,-0.4599,-0.8,0,-0.4599,-0.888,-0.6942,-0.4599,-0.5536,-0.6204,-0.6085,-0.4948,-0.3443,-0.6085,-0.715,-0.3443,-0.6085,-0.715,-0.6942,-0.4599,-0.5536,-0.3853,-0.4599,-0.8,-0.8657,-0.4599,-0.1976,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,-0.4948,-0.6204,-0.6085,-0.4948,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,-0.5536,-0.8657,-0.4599,0.1976,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,-0.1766,-0.7737,-0.6085,-0.1766,-0.8657,-0.4599,0.1976,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,0.5536,-0.6204,-0.6085,0.4948,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,0.1766,-0.6942,-0.4599,0.5536,-0.8657,-0.4599,0.1976,-0.3853,-0.4599,0.8,-0.3443,-0.6085,0.715,-0.6204,-0.6085,0.4948,-0.6204,-0.6085,0.4948,-0.3853,-0.4599,0.8,-0.6942,-0.4599,0.5536,0,-0.4599,0.888,0,-0.6085,0.7935,-0.3443,-0.6085,0.715,-0.3443,-0.6085,0.715,0,-0.4599,0.888,-0.3853,-0.4599,0.8,0.3853,-0.4599,0.8,0.3443,-0.6085,0.715,0,-0.6085,0.7935,0,-0.6085,0.7935,0.3853,-0.4599,0.8,0,-0.4599,0.888,0.6942,-0.4599,0.5536,0.6204,-0.6085,0.4948,0.3443,-0.6085,0.715,0.3443,-0.6085,0.715,0.6942,-0.4599,0.5536,0.3853,-0.4599,0.8,0.767,-0.5895,0.2533,0.7737,-0.6085,0.1766,0.6204,-0.6085,0.4948,0.6204,-0.6085,0.4948,0.767,-0.5895,0.2533,0.6942,-0.4599,0.5536,0.767,-0.5895,-0.2533,0.7737,-0.6085,-0.1766,0.7737,-0.6085,0.1766,0.7737,-0.6085,0.1766,0.767,-0.5895,-0.2533,0.767,-0.5895,0.2533,0.6942,-0.4599,-0.5536,0.6204,-0.6085,-0.4948,0.7737,-0.6085,-0.1766,0.7737,-0.6085,-0.1766,0.6942,-0.4599,-0.5536,0.767,-0.5895,-0.2533,0.3853,-0.4599,-0.8,0.3443,-0.6085,-0.715,0.6204,-0.6085,-0.4948,0.6204,-0.6085,-0.4948,0.3853,-0.4599,-0.8,0.6942,-0.4599,-0.5536,0,-0.4599,-0.888,0,-0.6085,-0.7935,0.3443,-0.6085,-0.715,0.3443,-0.6085,-0.715,0,-0.4599,-0.888,0.3853,-0.4599,-0.8,0,-1,0,0.0005,-1,0.001,0,-1,0.0011,0,-1,0,0.0008,-1,0.0007,0.0005,-1,0.001,0,-1,0,0.001,-1,0.0002,0.0008,-1,0.0007,0,-1,0,0.001,-1,-0.0002,0.001,-1,0.0002,0,-1,0,0.0008,-1,-0.0007,0.001,-1,-0.0002,0,-1,0,0.0005,-1,-0.001,0.0008,-1,-0.0007,0,-1,0,0,-1,-0.0011,0.0005,-1,-0.001,0,-1,0,-0.0005,-1,-0.001,0,-1,-0.0011,0,-1,0,-0.0008,-1,-0.0007,-0.0005,-1,-0.001,0,-1,0,-0.001,-1,-0.0002,-0.0008,-1,-0.0007,0,-1,0,-0.001,-1,0.0002,-0.001,-1,-0.0002,0,-1,0,-0.0008,-1,0.0007,-0.001,-1,0.0002,0,-1,0,-0.0005,-1,0.001,-0.0008,-1,0.0007,0,-1,0,0,-1,0.0011,-0.0005,-1,0.001,0,0.4599,-0.888,-0.3853,0.4599,-0.8,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0.4599,-0.888,0,0,-1,-0.3853,0.4599,-0.8,-0.6942,0.4599,-0.5536,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.3853,0.4599,-0.8,-0.4339,0,-0.901,-0.6942,0.4599,-0.5536,-0.8657,0.4599,-0.1976,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.6942,0.4599,-0.5536,-0.7818,0,-0.6235,-0.8657,0.4599,-0.1976,-0.8657,0.4599,0.1976,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.8657,0.4599,-0.1976,-0.9749,0,-0.2225,-0.8657,0.4599,0.1976,-0.6942,0.4599,0.5536,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.8657,0.4599,0.1976,-0.9749,0,0.2225,-0.6942,0.4599,0.5536,-0.3853,0.4599,0.8,-0.4339,0,0.901,-0.4339,0,0.901,-0.6942,0.4599,0.5536,-0.7818,0,0.6235,-0.3853,0.4599,0.8,0,0.4599,0.888,0,0,1,0,0,1,-0.3853,0.4599,0.8,-0.4339,0,0.901,0,0.4599,0.888,0.3853,0.4599,0.8,0.4339,0,0.901,0.4339,0,0.901,0,0.4599,0.888,0,0,1,0.3853,0.4599,0.8,0.6942,0.4599,0.5536,0.7818,0,0.6235,0.7818,0,0.6235,0.3853,0.4599,0.8,0.4339,0,0.901,0.6942,0.4599,0.5536,0.8657,0.4599,0.1976,0.9749,0,0.2225,0.9749,0,0.2225,0.6942,0.4599,0.5536,0.7818,0,0.6235,0.8657,0.4599,0.1976,0.8657,0.4599,-0.1976,0.9749,0,-0.2225,0.9749,0,-0.2225,0.8657,0.4599,0.1976,0.9749,0,0.2225,0.8657,0.4599,-0.1976,0.6942,0.4599,-0.5536,0.7818,0,-0.6235,0.7818,0,-0.6235,0.8657,0.4599,-0.1976,0.9749,0,-0.2225,0.6942,0.4599,-0.5536,0.3853,0.4599,-0.8,0.4339,0,-0.901,0.4339,0,-0.901,0.6942,0.4599,-0.5536,0.7818,0,-0.6235,0.3853,0.4599,-0.8,0,0.4599,-0.888,0,0,-1,0,0,-1,0.3853,0.4599,-0.8,0.4339,0,-0.901,-0.3853,-0.4599,-0.8,0,-0.4599,-0.888,0,0,-1,0,0,-1,-0.3853,-0.4599,-0.8,-0.4339,0,-0.901,0,-0.4599,-0.888,0.3853,-0.4599,-0.8,0.4339,0,-0.901,0.4339,0,-0.901,0,-0.4599,-0.888,0,0,-1,0.3853,-0.4599,-0.8,0.6942,-0.4599,-0.5536,0.7818,0,-0.6235,0.7818,0,-0.6235,0.3853,-0.4599,-0.8,0.4339,0,-0.901,0.6942,-0.4599,-0.5536,0.767,-0.5895,-0.2533,0.901,0,-0.4339,0.901,0,-0.4339,0.6942,-0.4599,-0.5536,0.7818,0,-0.6235,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0.767,-0.5895,0.2533,0.6942,-0.4599,0.5536,0.7818,0,0.6235,0.7818,0,0.6235,0.767,-0.5895,0.2533,0.901,0,0.4339,0.6942,-0.4599,0.5536,0.3853,-0.4599,0.8,0.4339,0,0.901,0.4339,0,0.901,0.6942,-0.4599,0.5536,0.7818,0,0.6235,0.3853,-0.4599,0.8,0,-0.4599,0.888,0,0,1,0,0,1,0.3853,-0.4599,0.8,0.4339,0,0.901,0,-0.4599,0.888,-0.3853,-0.4599,0.8,-0.4339,0,0.901,-0.4339,0,0.901,0,-0.4599,0.888,0,0,1,-0.3853,-0.4599,0.8,-0.6942,-0.4599,0.5536,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.3853,-0.4599,0.8,-0.4339,0,0.901,-0.6942,-0.4599,0.5536,-0.8657,-0.4599,0.1976,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.6942,-0.4599,0.5536,-0.7818,0,0.6235,-0.8657,-0.4599,0.1976,-0.8657,-0.4599,-0.1976,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.8657,-0.4599,0.1976,-0.9749,0,0.2225,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,-0.5536,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.8657,-0.4599,-0.1976,-0.9749,0,-0.2225,-0.6942,-0.4599,-0.5536,-0.3853,-0.4599,-0.8,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.6942,-0.4599,-0.5536,-0.7818,0,-0.6235,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,0,0,1,0,0,1,0,0,1,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0.4339,0,0.901,0,0,1,0,0,1,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0,0,-1,0,0,-1,0,0,-1,0.4339,0,-0.901,0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0,0,-1,0,0,-1,0.4339,0,-0.901,0.7818,0,-0.6235,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.4339,0,0.901,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0,0,1,0,0,1,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0,0,1,-0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0,0,1,-0.4339,0,0.901,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,0,1,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0,0,1,0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0,0,1,-0.4339,0,0.901,0,0,1,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.4339,0,0.901,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.7818,0,-0.6235,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,-0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0,0,-1,0,0,-1,0.4339,0,-0.901,0,0,-1,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.4339,0,-0.901,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.7818,0,-0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.9749,0,0.2225,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0.7818,0,0.6235,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0.8467,0.1976,0.4941,0.8651,0.0998,0.4916,0.9934,0.1146,0,0.9934,0.1146,0,0.8467,0.1976,0.4941,0.9738,0.2272,0,0.4983,0.1163,0.8591,0.5109,0.0589,0.8576,0.8651,0.0998,0.4916,0.8651,0.0998,0.4916,0.4983,0.1163,0.8591,0.8467,0.1976,0.4941,0.0182,0.0043,0.9998,0.0188,0.0022,0.9998,0.5109,0.0589,0.8576,0.5109,0.0589,0.8576,0.0182,0.0043,0.9998,0.4983,0.1163,0.8591,-0.4708,-0.1099,0.8754,-0.4827,-0.0557,0.874,0.0188,0.0022,0.9998,0.0188,0.0022,0.9998,-0.4708,-0.1099,0.8754,0.0182,0.0043,0.9998,-0.8374,-0.1954,0.5105,-0.8556,-0.0987,0.5081,-0.4827,-0.0557,0.874,-0.4827,-0.0557,0.874,-0.8374,-0.1954,0.5105,-0.4708,-0.1099,0.8754,-0.9738,-0.2272,0,-0.9934,-0.1146,0,-0.8556,-0.0987,0.5081,-0.8556,-0.0987,0.5081,-0.9738,-0.2272,0,-0.8374,-0.1954,0.5105,-0.8374,-0.1954,-0.5105,-0.8556,-0.0987,-0.5081,-0.9934,-0.1146,0,-0.9934,-0.1146,0,-0.8374,-0.1954,-0.5105,-0.9738,-0.2272,0,-0.4708,-0.1099,-0.8754,-0.4827,-0.0557,-0.874,-0.8556,-0.0987,-0.5081,-0.8556,-0.0987,-0.5081,-0.4708,-0.1099,-0.8754,-0.8374,-0.1954,-0.5105,0.0182,0.0043,-0.9998,0.0188,0.0022,-0.9998,-0.4827,-0.0557,-0.874,-0.4827,-0.0557,-0.874,0.0182,0.0043,-0.9998,-0.4708,-0.1099,-0.8754,0.4983,0.1163,-0.8591,0.5109,0.0589,-0.8576,0.0188,0.0022,-0.9998,0.0188,0.0022,-0.9998,0.4983,0.1163,-0.8591,0.0182,0.0043,-0.9998,0.8467,0.1976,-0.4941,0.8651,0.0998,-0.4916,0.5109,0.0589,-0.8576,0.5109,0.0589,-0.8576,0.8467,0.1976,-0.4941,0.4983,0.1163,-0.8591,0.9738,0.2272,0,0.9934,0.1146,0,0.8651,0.0998,-0.4916,0.8651,0.0998,-0.4916,0.9738,0.2272,0,0.8467,0.1976,-0.4941,0.7809,0.3821,0.4942,0.8467,0.1976,0.4941,0.9738,0.2272,0,0.9738,0.2272,0,0.7809,0.3821,0.4942,0.8982,0.4395,0,0.4594,0.2248,0.8593,0.4983,0.1163,0.8591,0.8467,0.1976,0.4941,0.8467,0.1976,0.4941,0.4594,0.2248,0.8593,0.7809,0.3821,0.4942,0.0164,0.008,0.9998,0.0182,0.0043,0.9998,0.4983,0.1163,0.8591,0.4983,0.1163,0.8591,0.0164,0.008,0.9998,0.4594,0.2248,0.8593,-0.4347,-0.2127,0.8751,-0.4708,-0.1099,0.8754,0.0182,0.0043,0.9998,0.0182,0.0043,0.9998,-0.4347,-0.2127,0.8751,0.0164,0.008,0.9998,-0.7725,-0.378,0.5102,-0.8374,-0.1954,0.5105,-0.4708,-0.1099,0.8754,-0.4708,-0.1099,0.8754,-0.7725,-0.378,0.5102,-0.4347,-0.2127,0.8751,-0.8982,-0.4395,0,-0.9738,-0.2272,0,-0.8374,-0.1954,0.5105,-0.8374,-0.1954,0.5105,-0.8982,-0.4395,0,-0.7725,-0.378,0.5102,-0.7725,-0.378,-0.5102,-0.8374,-0.1954,-0.5105,-0.9738,-0.2272,0,-0.9738,-0.2272,0,-0.7725,-0.378,-0.5102,-0.8982,-0.4395,0,-0.4347,-0.2127,-0.8751,-0.4708,-0.1099,-0.8754,-0.8374,-0.1954,-0.5105,-0.8374,-0.1954,-0.5105,-0.4347,-0.2127,-0.8751,-0.7725,-0.378,-0.5102,0.0164,0.008,-0.9998,0.0182,0.0043,-0.9998,-0.4708,-0.1099,-0.8754,-0.4708,-0.1099,-0.8754,0.0164,0.008,-0.9998,-0.4347,-0.2127,-0.8751,0.4594,0.2248,-0.8593,0.4983,0.1163,-0.8591,0.0182,0.0043,-0.9998,0.0182,0.0043,-0.9998,0.4594,0.2248,-0.8593,0.0164,0.008,-0.9998,0.7809,0.3821,-0.4942,0.8467,0.1976,-0.4941,0.4983,0.1163,-0.8591,0.4983,0.1163,-0.8591,0.7809,0.3821,-0.4942,0.4594,0.2248,-0.8593,0.8982,0.4395,0,0.9738,0.2272,0,0.8467,0.1976,-0.4941,0.8467,0.1976,-0.4941,0.8982,0.4395,0,0.7809,0.3821,-0.4942,0.6781,0.5439,0.4943,0.7809,0.3821,0.4942,0.8982,0.4395,0,0.8982,0.4395,0,0.6781,0.5439,0.4943,0.7801,0.6257,0,0.3989,0.3199,0.8594,0.4594,0.2248,0.8593,0.7809,0.3821,0.4942,0.7809,0.3821,0.4942,0.3989,0.3199,0.8594,0.6781,0.5439,0.4943,0.014,0.0112,0.9998,0.0164,0.008,0.9998,0.4594,0.2248,0.8593,0.4594,0.2248,0.8593,0.014,0.0112,0.9998,0.3989,0.3199,0.8594,-0.3778,-0.303,0.8749,-0.4347,-0.2127,0.8751,0.0164,0.008,0.9998,0.0164,0.008,0.9998,-0.3778,-0.303,0.8749,0.014,0.0112,0.9998,-0.671,-0.5382,0.5099,-0.7725,-0.378,0.5102,-0.4347,-0.2127,0.8751,-0.4347,-0.2127,0.8751,-0.671,-0.5382,0.5099,-0.3778,-0.303,0.8749,-0.7801,-0.6257,0,-0.8982,-0.4395,0,-0.7725,-0.378,0.5102,-0.7725,-0.378,0.5102,-0.7801,-0.6257,0,-0.671,-0.5382,0.5099,-0.671,-0.5382,-0.5099,-0.7725,-0.378,-0.5102,-0.8982,-0.4395,0,-0.8982,-0.4395,0,-0.671,-0.5382,-0.5099,-0.7801,-0.6257,0,-0.3778,-0.303,-0.8749,-0.4347,-0.2127,-0.8751,-0.7725,-0.378,-0.5102,-0.7725,-0.378,-0.5102,-0.3778,-0.303,-0.8749,-0.671,-0.5382,-0.5099,0.014,0.0112,-0.9998,0.0164,0.008,-0.9998,-0.4347,-0.2127,-0.8751,-0.4347,-0.2127,-0.8751,0.014,0.0112,-0.9998,-0.3778,-0.303,-0.8749,0.3989,0.3199,-0.8594,0.4594,0.2248,-0.8593,0.0164,0.008,-0.9998,0.0164,0.008,-0.9998,0.3989,0.3199,-0.8594,0.014,0.0112,-0.9998,0.6781,0.5439,-0.4943,0.7809,0.3821,-0.4942,0.4594,0.2248,-0.8593,0.4594,0.2248,-0.8593,0.6781,0.5439,-0.4943,0.3989,0.3199,-0.8594,0.7801,0.6257,0,0.8982,0.4395,0,0.7809,0.3821,-0.4942,0.7809,0.3821,-0.4942,0.7801,0.6257,0,0.6781,0.5439,-0.4943,0.5439,0.6781,0.4943,0.6781,0.5439,0.4943,0.7801,0.6257,0,0.7801,0.6257,0,0.5439,0.6781,0.4943,0.6257,0.7801,0,0.3199,0.3989,0.8594,0.3989,0.3199,0.8594,0.6781,0.5439,0.4943,0.6781,0.5439,0.4943,0.3199,0.3989,0.8594,0.5439,0.6781,0.4943,0.0112,0.014,0.9998,0.014,0.0112,0.9998,0.3989,0.3199,0.8594,0.3989,0.3199,0.8594,0.0112,0.014,0.9998,0.3199,0.3989,0.8594,-0.303,-0.3778,0.8749,-0.3778,-0.303,0.8749,0.014,0.0112,0.9998,0.014,0.0112,0.9998,-0.303,-0.3778,0.8749,0.0112,0.014,0.9998,-0.5382,-0.671,0.5099,-0.671,-0.5382,0.5099,-0.3778,-0.303,0.8749,-0.3778,-0.303,0.8749,-0.5382,-0.671,0.5099,-0.303,-0.3778,0.8749,-0.6257,-0.7801,0,-0.7801,-0.6257,0,-0.671,-0.5382,0.5099,-0.671,-0.5382,0.5099,-0.6257,-0.7801,0,-0.5382,-0.671,0.5099,-0.5382,-0.671,-0.5099,-0.671,-0.5382,-0.5099,-0.7801,-0.6257,0,-0.7801,-0.6257,0,-0.5382,-0.671,-0.5099,-0.6257,-0.7801,0,-0.303,-0.3778,-0.8749,-0.3778,-0.303,-0.8749,-0.671,-0.5382,-0.5099,-0.671,-0.5382,-0.5099,-0.303,-0.3778,-0.8749,-0.5382,-0.671,-0.5099,0.0112,0.014,-0.9998,0.014,0.0112,-0.9998,-0.3778,-0.303,-0.8749,-0.3778,-0.303,-0.8749,0.0112,0.014,-0.9998,-0.303,-0.3778,-0.8749,0.3199,0.3989,-0.8594,0.3989,0.3199,-0.8594,0.014,0.0112,-0.9998,0.014,0.0112,-0.9998,0.3199,0.3989,-0.8594,0.0112,0.014,-0.9998,0.5439,0.6781,-0.4943,0.6781,0.5439,-0.4943,0.3989,0.3199,-0.8594,0.3989,0.3199,-0.8594,0.5439,0.6781,-0.4943,0.3199,0.3989,-0.8594,0.6257,0.7801,0,0.7801,0.6257,0,0.6781,0.5439,-0.4943,0.6781,0.5439,-0.4943,0.6257,0.7801,0,0.5439,0.6781,-0.4943,0.3821,0.7809,0.4942,0.5439,0.6781,0.4943,0.6257,0.7801,0,0.6257,0.7801,0,0.3821,0.7809,0.4942,0.4395,0.8982,0,0.2248,0.4594,0.8593,0.3199,0.3989,0.8594,0.5439,0.6781,0.4943,0.5439,0.6781,0.4943,0.2248,0.4594,0.8593,0.3821,0.7809,0.4942,0.008,0.0164,0.9998,0.0112,0.014,0.9998,0.3199,0.3989,0.8594,0.3199,0.3989,0.8594,0.008,0.0164,0.9998,0.2248,0.4594,0.8593,-0.2127,-0.4347,0.8751,-0.303,-0.3778,0.8749,0.0112,0.014,0.9998,0.0112,0.014,0.9998,-0.2127,-0.4347,0.8751,0.008,0.0164,0.9998,-0.378,-0.7725,0.5102,-0.5382,-0.671,0.5099,-0.303,-0.3778,0.8749,-0.303,-0.3778,0.8749,-0.378,-0.7725,0.5102,-0.2127,-0.4347,0.8751,-0.4395,-0.8982,0,-0.6257,-0.7801,0,-0.5382,-0.671,0.5099,-0.5382,-0.671,0.5099,-0.4395,-0.8982,0,-0.378,-0.7725,0.5102,-0.378,-0.7725,-0.5102,-0.5382,-0.671,-0.5099,-0.6257,-0.7801,0,-0.6257,-0.7801,0,-0.378,-0.7725,-0.5102,-0.4395,-0.8982,0,-0.2127,-0.4347,-0.8751,-0.303,-0.3778,-0.8749,-0.5382,-0.671,-0.5099,-0.5382,-0.671,-0.5099,-0.2127,-0.4347,-0.8751,-0.378,-0.7725,-0.5102,0.008,0.0164,-0.9998,0.0112,0.014,-0.9998,-0.303,-0.3778,-0.8749,-0.303,-0.3778,-0.8749,0.008,0.0164,-0.9998,-0.2127,-0.4347,-0.8751,0.2248,0.4594,-0.8593,0.3199,0.3989,-0.8594,0.0112,0.014,-0.9998,0.0112,0.014,-0.9998,0.2248,0.4594,-0.8593,0.008,0.0164,-0.9998,0.3821,0.7809,-0.4942,0.5439,0.6781,-0.4943,0.3199,0.3989,-0.8594,0.3199,0.3989,-0.8594,0.3821,0.7809,-0.4942,0.2248,0.4594,-0.8593,0.4395,0.8982,0,0.6257,0.7801,0,0.5439,0.6781,-0.4943,0.5439,0.6781,-0.4943,0.4395,0.8982,0,0.3821,0.7809,-0.4942,0.1976,0.8467,0.4941,0.3821,0.7809,0.4942,0.4395,0.8982,0,0.4395,0.8982,0,0.1976,0.8467,0.4941,0.2272,0.9738,0,0.1163,0.4983,0.8591,0.2248,0.4594,0.8593,0.3821,0.7809,0.4942,0.3821,0.7809,0.4942,0.1163,0.4983,0.8591,0.1976,0.8467,0.4941,0.0043,0.0182,0.9998,0.008,0.0164,0.9998,0.2248,0.4594,0.8593,0.2248,0.4594,0.8593,0.0043,0.0182,0.9998,0.1163,0.4983,0.8591,-0.1099,-0.4708,0.8754,-0.2127,-0.4347,0.8751,0.008,0.0164,0.9998,0.008,0.0164,0.9998,-0.1099,-0.4708,0.8754,0.0043,0.0182,0.9998,-0.1954,-0.8374,0.5105,-0.378,-0.7725,0.5102,-0.2127,-0.4347,0.8751,-0.2127,-0.4347,0.8751,-0.1954,-0.8374,0.5105,-0.1099,-0.4708,0.8754,-0.2272,-0.9738,0,-0.4395,-0.8982,0,-0.378,-0.7725,0.5102,-0.378,-0.7725,0.5102,-0.2272,-0.9738,0,-0.1954,-0.8374,0.5105,-0.1954,-0.8374,-0.5105,-0.378,-0.7725,-0.5102,-0.4395,-0.8982,0,-0.4395,-0.8982,0,-0.1954,-0.8374,-0.5105,-0.2272,-0.9738,0,-0.1099,-0.4708,-0.8754,-0.2127,-0.4347,-0.8751,-0.378,-0.7725,-0.5102,-0.378,-0.7725,-0.5102,-0.1099,-0.4708,-0.8754,-0.1954,-0.8374,-0.5105,0.0043,0.0182,-0.9998,0.008,0.0164,-0.9998,-0.2127,-0.4347,-0.8751,-0.2127,-0.4347,-0.8751,0.0043,0.0182,-0.9998,-0.1099,-0.4708,-0.8754,0.1163,0.4983,-0.8591,0.2248,0.4594,-0.8593,0.008,0.0164,-0.9998,0.008,0.0164,-0.9998,0.1163,0.4983,-0.8591,0.0043,0.0182,-0.9998,0.1976,0.8467,-0.4941,0.3821,0.7809,-0.4942,0.2248,0.4594,-0.8593,0.2248,0.4594,-0.8593,0.1976,0.8467,-0.4941,0.1163,0.4983,-0.8591,0.2272,0.9738,0,0.4395,0.8982,0,0.3821,0.7809,-0.4942,0.3821,0.7809,-0.4942,0.2272,0.9738,0,0.1976,0.8467,-0.4941,0,0.8694,0.4941,0.1976,0.8467,0.4941,0.2272,0.9738,0,0.2272,0.9738,0,0,0.8694,0.4941,0,1,0,0,0.5118,0.8591,0.1163,0.4983,0.8591,0.1976,0.8467,0.4941,0.1976,0.8467,0.4941,0,0.5118,0.8591,0,0.8694,0.4941,0,0.0188,0.9998,0.0043,0.0182,0.9998,0.1163,0.4983,0.8591,0.1163,0.4983,0.8591,0,0.0188,0.9998,0,0.5118,0.8591,0,-0.4834,0.8754,-0.1099,-0.4708,0.8754,0.0043,0.0182,0.9998,0.0043,0.0182,0.9998,0,-0.4834,0.8754,0,0.0188,0.9998,0,-0.8598,0.5106,-0.1954,-0.8374,0.5105,-0.1099,-0.4708,0.8754,-0.1099,-0.4708,0.8754,0,-0.8598,0.5106,0,-0.4834,0.8754,0,-1,0,-0.2272,-0.9738,0,-0.1954,-0.8374,0.5105,-0.1954,-0.8374,0.5105,0,-1,0,0,-0.8598,0.5106,0,-0.8598,-0.5106,-0.1954,-0.8374,-0.5105,-0.2272,-0.9738,0,-0.2272,-0.9738,0,0,-0.8598,-0.5106,0,-1,0,0,-0.4834,-0.8754,-0.1099,-0.4708,-0.8754,-0.1954,-0.8374,-0.5105,-0.1954,-0.8374,-0.5105,0,-0.4834,-0.8754,0,-0.8598,-0.5106,0,0.0188,-0.9998,0.0043,0.0182,-0.9998,-0.1099,-0.4708,-0.8754,-0.1099,-0.4708,-0.8754,0,0.0188,-0.9998,0,-0.4834,-0.8754,0,0.5118,-0.8591,0.1163,0.4983,-0.8591,0.0043,0.0182,-0.9998,0.0043,0.0182,-0.9998,0,0.5118,-0.8591,0,0.0188,-0.9998,0,0.8694,-0.4941,0.1976,0.8467,-0.4941,0.1163,0.4983,-0.8591,0.1163,0.4983,-0.8591,0,0.8694,-0.4941,0,0.5118,-0.8591,0,1,0,0.2272,0.9738,0,0.1976,0.8467,-0.4941,0.1976,0.8467,-0.4941,0,1,0,0,0.8694,-0.4941,-0.1976,0.8467,0.4941,0,0.8694,0.4941,0,1,0,0,1,0,-0.1976,0.8467,0.4941,-0.2272,0.9738,0,-0.1163,0.4983,0.8591,0,0.5118,0.8591,0,0.8694,0.4941,0,0.8694,0.4941,-0.1163,0.4983,0.8591,-0.1976,0.8467,0.4941,-0.0043,0.0182,0.9998,0,0.0188,0.9998,0,0.5118,0.8591,0,0.5118,0.8591,-0.0043,0.0182,0.9998,-0.1163,0.4983,0.8591,0.1099,-0.4708,0.8754,0,-0.4834,0.8754,0,0.0188,0.9998,0,0.0188,0.9998,0.1099,-0.4708,0.8754,-0.0043,0.0182,0.9998,0.1954,-0.8374,0.5105,0,-0.8598,0.5106,0,-0.4834,0.8754,0,-0.4834,0.8754,0.1954,-0.8374,0.5105,0.1099,-0.4708,0.8754,0.2272,-0.9738,0,0,-1,0,0,-0.8598,0.5106,0,-0.8598,0.5106,0.2272,-0.9738,0,0.1954,-0.8374,0.5105,0.1954,-0.8374,-0.5105,0,-0.8598,-0.5106,0,-1,0,0,-1,0,0.1954,-0.8374,-0.5105,0.2272,-0.9738,0,0.1099,-0.4708,-0.8754,0,-0.4834,-0.8754,0,-0.8598,-0.5106,0,-0.8598,-0.5106,0.1099,-0.4708,-0.8754,0.1954,-0.8374,-0.5105,-0.0043,0.0182,-0.9998,0,0.0188,-0.9998,0,-0.4834,-0.8754,0,-0.4834,-0.8754,-0.0043,0.0182,-0.9998,0.1099,-0.4708,-0.8754,-0.1163,0.4983,-0.8591,0,0.5118,-0.8591,0,0.0188,-0.9998,0,0.0188,-0.9998,-0.1163,0.4983,-0.8591,-0.0043,0.0182,-0.9998,-0.1976,0.8467,-0.4941,0,0.8694,-0.4941,0,0.5118,-0.8591,0,0.5118,-0.8591,-0.1976,0.8467,-0.4941,-0.1163,0.4983,-0.8591,-0.2272,0.9738,0,0,1,0,0,0.8694,-0.4941,0,0.8694,-0.4941,-0.2272,0.9738,0,-0.1976,0.8467,-0.4941,-0.3821,0.7809,0.4942,-0.1976,0.8467,0.4941,-0.2272,0.9738,0,-0.2272,0.9738,0,-0.3821,0.7809,0.4942,-0.4395,0.8982,0,-0.2248,0.4594,0.8593,-0.1163,0.4983,0.8591,-0.1976,0.8467,0.4941,-0.1976,0.8467,0.4941,-0.2248,0.4594,0.8593,-0.3821,0.7809,0.4942,-0.008,0.0164,0.9998,-0.0043,0.0182,0.9998,-0.1163,0.4983,0.8591,-0.1163,0.4983,0.8591,-0.008,0.0164,0.9998,-0.2248,0.4594,0.8593,0.2127,-0.4347,0.8751,0.1099,-0.4708,0.8754,-0.0043,0.0182,0.9998,-0.0043,0.0182,0.9998,0.2127,-0.4347,0.8751,-0.008,0.0164,0.9998,0.378,-0.7725,0.5102,0.1954,-0.8374,0.5105,0.1099,-0.4708,0.8754,0.1099,-0.4708,0.8754,0.378,-0.7725,0.5102,0.2127,-0.4347,0.8751,0.4395,-0.8982,0,0.2272,-0.9738,0,0.1954,-0.8374,0.5105,0.1954,-0.8374,0.5105,0.4395,-0.8982,0,0.378,-0.7725,0.5102,0.378,-0.7725,-0.5102,0.1954,-0.8374,-0.5105,0.2272,-0.9738,0,0.2272,-0.9738,0,0.378,-0.7725,-0.5102,0.4395,-0.8982,0,0.2127,-0.4347,-0.8751,0.1099,-0.4708,-0.8754,0.1954,-0.8374,-0.5105,0.1954,-0.8374,-0.5105,0.2127,-0.4347,-0.8751,0.378,-0.7725,-0.5102,-0.008,0.0164,-0.9998,-0.0043,0.0182,-0.9998,0.1099,-0.4708,-0.8754,0.1099,-0.4708,-0.8754,-0.008,0.0164,-0.9998,0.2127,-0.4347,-0.8751,-0.2248,0.4594,-0.8593,-0.1163,0.4983,-0.8591,-0.0043,0.0182,-0.9998,-0.0043,0.0182,-0.9998,-0.2248,0.4594,-0.8593,-0.008,0.0164,-0.9998,-0.3821,0.7809,-0.4942,-0.1976,0.8467,-0.4941,-0.1163,0.4983,-0.8591,-0.1163,0.4983,-0.8591,-0.3821,0.7809,-0.4942,-0.2248,0.4594,-0.8593,-0.4395,0.8982,0,-0.2272,0.9738,0,-0.1976,0.8467,-0.4941,-0.1976,0.8467,-0.4941,-0.4395,0.8982,0,-0.3821,0.7809,-0.4942,-0.5439,0.6781,0.4943,-0.3821,0.7809,0.4942,-0.4395,0.8982,0,-0.4395,0.8982,0,-0.5439,0.6781,0.4943,-0.6257,0.7801,0,-0.3199,0.3989,0.8594,-0.2248,0.4594,0.8593,-0.3821,0.7809,0.4942,-0.3821,0.7809,0.4942,-0.3199,0.3989,0.8594,-0.5439,0.6781,0.4943,-0.0112,0.014,0.9998,-0.008,0.0164,0.9998,-0.2248,0.4594,0.8593,-0.2248,0.4594,0.8593,-0.0112,0.014,0.9998,-0.3199,0.3989,0.8594,0.303,-0.3778,0.8749,0.2127,-0.4347,0.8751,-0.008,0.0164,0.9998,-0.008,0.0164,0.9998,0.303,-0.3778,0.8749,-0.0112,0.014,0.9998,0.5382,-0.671,0.5099,0.378,-0.7725,0.5102,0.2127,-0.4347,0.8751,0.2127,-0.4347,0.8751,0.5382,-0.671,0.5099,0.303,-0.3778,0.8749,0.6257,-0.7801,0,0.4395,-0.8982,0,0.378,-0.7725,0.5102,0.378,-0.7725,0.5102,0.6257,-0.7801,0,0.5382,-0.671,0.5099,0.5382,-0.671,-0.5099,0.378,-0.7725,-0.5102,0.4395,-0.8982,0,0.4395,-0.8982,0,0.5382,-0.671,-0.5099,0.6257,-0.7801,0,0.303,-0.3778,-0.8749,0.2127,-0.4347,-0.8751,0.378,-0.7725,-0.5102,0.378,-0.7725,-0.5102,0.303,-0.3778,-0.8749,0.5382,-0.671,-0.5099,-0.0112,0.014,-0.9998,-0.008,0.0164,-0.9998,0.2127,-0.4347,-0.8751,0.2127,-0.4347,-0.8751,-0.0112,0.014,-0.9998,0.303,-0.3778,-0.8749,-0.3199,0.3989,-0.8594,-0.2248,0.4594,-0.8593,-0.008,0.0164,-0.9998,-0.008,0.0164,-0.9998,-0.3199,0.3989,-0.8594,-0.0112,0.014,-0.9998,-0.5439,0.6781,-0.4943,-0.3821,0.7809,-0.4942,-0.2248,0.4594,-0.8593,-0.2248,0.4594,-0.8593,-0.5439,0.6781,-0.4943,-0.3199,0.3989,-0.8594,-0.6257,0.7801,0,-0.4395,0.8982,0,-0.3821,0.7809,-0.4942,-0.3821,0.7809,-0.4942,-0.6257,0.7801,0,-0.5439,0.6781,-0.4943,-0.6781,0.5439,0.4943,-0.5439,0.6781,0.4943,-0.6257,0.7801,0,-0.6257,0.7801,0,-0.6781,0.5439,0.4943,-0.7801,0.6257,0,-0.3989,0.3199,0.8594,-0.3199,0.3989,0.8594,-0.5439,0.6781,0.4943,-0.5439,0.6781,0.4943,-0.3989,0.3199,0.8594,-0.6781,0.5439,0.4943,-0.014,0.0112,0.9998,-0.0112,0.014,0.9998,-0.3199,0.3989,0.8594,-0.3199,0.3989,0.8594,-0.014,0.0112,0.9998,-0.3989,0.3199,0.8594,0.3778,-0.303,0.8749,0.303,-0.3778,0.8749,-0.0112,0.014,0.9998,-0.0112,0.014,0.9998,0.3778,-0.303,0.8749,-0.014,0.0112,0.9998,0.671,-0.5382,0.5099,0.5382,-0.671,0.5099,0.303,-0.3778,0.8749,0.303,-0.3778,0.8749,0.671,-0.5382,0.5099,0.3778,-0.303,0.8749,0.7801,-0.6257,0,0.6257,-0.7801,0,0.5382,-0.671,0.5099,0.5382,-0.671,0.5099,0.7801,-0.6257,0,0.671,-0.5382,0.5099,0.671,-0.5382,-0.5099,0.5382,-0.671,-0.5099,0.6257,-0.7801,0,0.6257,-0.7801,0,0.671,-0.5382,-0.5099,0.7801,-0.6257,0,0.3778,-0.303,-0.8749,0.303,-0.3778,-0.8749,0.5382,-0.671,-0.5099,0.5382,-0.671,-0.5099,0.3778,-0.303,-0.8749,0.671,-0.5382,-0.5099,-0.014,0.0112,-0.9998,-0.0112,0.014,-0.9998,0.303,-0.3778,-0.8749,0.303,-0.3778,-0.8749,-0.014,0.0112,-0.9998,0.3778,-0.303,-0.8749,-0.3989,0.3199,-0.8594,-0.3199,0.3989,-0.8594,-0.0112,0.014,-0.9998,-0.0112,0.014,-0.9998,-0.3989,0.3199,-0.8594,-0.014,0.0112,-0.9998,-0.6781,0.5439,-0.4943,-0.5439,0.6781,-0.4943,-0.3199,0.3989,-0.8594,-0.3199,0.3989,-0.8594,-0.6781,0.5439,-0.4943,-0.3989,0.3199,-0.8594,-0.7801,0.6257,0,-0.6257,0.7801,0,-0.5439,0.6781,-0.4943,-0.5439,0.6781,-0.4943,-0.7801,0.6257,0,-0.6781,0.5439,-0.4943,-0.7809,0.3821,0.4942,-0.6781,0.5439,0.4943,-0.7801,0.6257,0,-0.7801,0.6257,0,-0.7809,0.3821,0.4942,-0.8982,0.4395,0,-0.4594,0.2248,0.8593,-0.3989,0.3199,0.8594,-0.6781,0.5439,0.4943,-0.6781,0.5439,0.4943,-0.4594,0.2248,0.8593,-0.7809,0.3821,0.4942,-0.0164,0.008,0.9998,-0.014,0.0112,0.9998,-0.3989,0.3199,0.8594,-0.3989,0.3199,0.8594,-0.0164,0.008,0.9998,-0.4594,0.2248,0.8593,0.4347,-0.2127,0.8751,0.3778,-0.303,0.8749,-0.014,0.0112,0.9998,-0.014,0.0112,0.9998,0.4347,-0.2127,0.8751,-0.0164,0.008,0.9998,0.7725,-0.378,0.5102,0.671,-0.5382,0.5099,0.3778,-0.303,0.8749,0.3778,-0.303,0.8749,0.7725,-0.378,0.5102,0.4347,-0.2127,0.8751,0.8982,-0.4395,0,0.7801,-0.6257,0,0.671,-0.5382,0.5099,0.671,-0.5382,0.5099,0.8982,-0.4395,0,0.7725,-0.378,0.5102,0.7725,-0.378,-0.5102,0.671,-0.5382,-0.5099,0.7801,-0.6257,0,0.7801,-0.6257,0,0.7725,-0.378,-0.5102,0.8982,-0.4395,0,0.4347,-0.2127,-0.8751,0.3778,-0.303,-0.8749,0.671,-0.5382,-0.5099,0.671,-0.5382,-0.5099,0.4347,-0.2127,-0.8751,0.7725,-0.378,-0.5102,-0.0164,0.008,-0.9998,-0.014,0.0112,-0.9998,0.3778,-0.303,-0.8749,0.3778,-0.303,-0.8749,-0.0164,0.008,-0.9998,0.4347,-0.2127,-0.8751,-0.4594,0.2248,-0.8593,-0.3989,0.3199,-0.8594,-0.014,0.0112,-0.9998,-0.014,0.0112,-0.9998,-0.4594,0.2248,-0.8593,-0.0164,0.008,-0.9998,-0.7809,0.3821,-0.4942,-0.6781,0.5439,-0.4943,-0.3989,0.3199,-0.8594,-0.3989,0.3199,-0.8594,-0.7809,0.3821,-0.4942,-0.4594,0.2248,-0.8593,-0.8982,0.4395,0,-0.7801,0.6257,0,-0.6781,0.5439,-0.4943,-0.6781,0.5439,-0.4943,-0.8982,0.4395,0,-0.7809,0.3821,-0.4942,-0.8467,0.1976,0.4941,-0.7809,0.3821,0.4942,-0.8982,0.4395,0,-0.8982,0.4395,0,-0.8467,0.1976,0.4941,-0.9738,0.2272,0,-0.4983,0.1163,0.8591,-0.4594,0.2248,0.8593,-0.7809,0.3821,0.4942,-0.7809,0.3821,0.4942,-0.4983,0.1163,0.8591,-0.8467,0.1976,0.4941,-0.0182,0.0043,0.9998,-0.0164,0.008,0.9998,-0.4594,0.2248,0.8593,-0.4594,0.2248,0.8593,-0.0182,0.0043,0.9998,-0.4983,0.1163,0.8591,0.4708,-0.1099,0.8754,0.4347,-0.2127,0.8751,-0.0164,0.008,0.9998,-0.0164,0.008,0.9998,0.4708,-0.1099,0.8754,-0.0182,0.0043,0.9998,0.8374,-0.1954,0.5105,0.7725,-0.378,0.5102,0.4347,-0.2127,0.8751,0.4347,-0.2127,0.8751,0.8374,-0.1954,0.5105,0.4708,-0.1099,0.8754,0.9738,-0.2272,0,0.8982,-0.4395,0,0.7725,-0.378,0.5102,0.7725,-0.378,0.5102,0.9738,-0.2272,0,0.8374,-0.1954,0.5105,0.8374,-0.1954,-0.5105,0.7725,-0.378,-0.5102,0.8982,-0.4395,0,0.8982,-0.4395,0,0.8374,-0.1954,-0.5105,0.9738,-0.2272,0,0.4708,-0.1099,-0.8754,0.4347,-0.2127,-0.8751,0.7725,-0.378,-0.5102,0.7725,-0.378,-0.5102,0.4708,-0.1099,-0.8754,0.8374,-0.1954,-0.5105,-0.0182,0.0043,-0.9998,-0.0164,0.008,-0.9998,0.4347,-0.2127,-0.8751,0.4347,-0.2127,-0.8751,-0.0182,0.0043,-0.9998,0.4708,-0.1099,-0.8754,-0.4983,0.1163,-0.8591,-0.4594,0.2248,-0.8593,-0.0164,0.008,-0.9998,-0.0164,0.008,-0.9998,-0.4983,0.1163,-0.8591,-0.0182,0.0043,-0.9998,-0.8467,0.1976,-0.4941,-0.7809,0.3821,-0.4942,-0.4594,0.2248,-0.8593,-0.4594,0.2248,-0.8593,-0.8467,0.1976,-0.4941,-0.4983,0.1163,-0.8591,-0.9738,0.2272,0,-0.8982,0.4395,0,-0.7809,0.3821,-0.4942,-0.7809,0.3821,-0.4942,-0.9738,0.2272,0,-0.8467,0.1976,-0.4941,-0.8651,0.0998,0.4916,-0.8467,0.1976,0.4941,-0.9738,0.2272,0,-0.9738,0.2272,0,-0.8651,0.0998,0.4916,-0.9934,0.1146,0,-0.5109,0.0589,0.8576,-0.4983,0.1163,0.8591,-0.8467,0.1976,0.4941,-0.8467,0.1976,0.4941,-0.5109,0.0589,0.8576,-0.8651,0.0998,0.4916,-0.0188,0.0022,0.9998,-0.0182,0.0043,0.9998,-0.4983,0.1163,0.8591,-0.4983,0.1163,0.8591,-0.0188,0.0022,0.9998,-0.5109,0.0589,0.8576,0.4827,-0.0557,0.874,0.4708,-0.1099,0.8754,-0.0182,0.0043,0.9998,-0.0182,0.0043,0.9998,0.4827,-0.0557,0.874,-0.0188,0.0022,0.9998,0.8556,-0.0987,0.5081,0.8374,-0.1954,0.5105,0.4708,-0.1099,0.8754,0.4708,-0.1099,0.8754,0.8556,-0.0987,0.5081,0.4827,-0.0557,0.874,0.9934,-0.1146,0,0.9738,-0.2272,0,0.8374,-0.1954,0.5105,0.8374,-0.1954,0.5105,0.9934,-0.1146,0,0.8556,-0.0987,0.5081,0.8556,-0.0987,-0.5081,0.8374,-0.1954,-0.5105,0.9738,-0.2272,0,0.9738,-0.2272,0,0.8556,-0.0987,-0.5081,0.9934,-0.1146,0,0.4827,-0.0557,-0.874,0.4708,-0.1099,-0.8754,0.8374,-0.1954,-0.5105,0.8374,-0.1954,-0.5105,0.4827,-0.0557,-0.874,0.8556,-0.0987,-0.5081,-0.0188,0.0022,-0.9998,-0.0182,0.0043,-0.9998,0.4708,-0.1099,-0.8754,0.4708,-0.1099,-0.8754,-0.0188,0.0022,-0.9998,0.4827,-0.0557,-0.874,-0.5109,0.0589,-0.8576,-0.4983,0.1163,-0.8591,-0.0182,0.0043,-0.9998,-0.0182,0.0043,-0.9998,-0.5109,0.0589,-0.8576,-0.0188,0.0022,-0.9998,-0.8651,0.0998,-0.4916,-0.8467,0.1976,-0.4941,-0.4983,0.1163,-0.8591,-0.4983,0.1163,-0.8591,-0.8651,0.0998,-0.4916,-0.5109,0.0589,-0.8576,-0.9934,0.1146,0,-0.9738,0.2272,0,-0.8467,0.1976,-0.4941,-0.8467,0.1976,-0.4941,-0.9934,0.1146,0,-0.8651,0.0998,-0.4916] )
   , make_vbo(COLOR, [0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0.8902,0.1137,0.1451,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882] )
  ],
  mode: gl.TRIANGLES ,
  vertex_count: 2262
};
geometries["lantern2"] = {
  buffers: [
     make_vbo(POS, [-7.1987,30.8244,-15.2442,-3.9316,36.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,36.8294,-9.3897,-7.1987,30.8244,-15.2442,0.1425,30.8244,-16.9197,-13.0859,30.8244,-10.5493,-7.1987,36.8294,-5.8544,-3.9316,36.8294,-8.4599,-3.9316,36.8294,-8.4599,-13.0859,30.8244,-10.5493,-7.1987,30.8244,-15.2442,-16.353,30.8244,-3.765,-9.0118,36.8294,-2.0894,-7.1987,36.8294,-5.8544,-7.1987,36.8294,-5.8544,-16.353,30.8244,-3.765,-13.0859,30.8244,-10.5493,-16.353,30.8244,3.765,-9.0118,36.8294,2.0894,-9.0118,36.8294,-2.0894,-9.0118,36.8294,-2.0894,-16.353,30.8244,3.765,-16.353,30.8244,-3.765,-13.0859,30.8244,10.5493,-7.1987,36.8294,5.8544,-9.0118,36.8294,2.0894,-9.0118,36.8294,2.0894,-13.0859,30.8244,10.5493,-16.353,30.8244,3.765,-7.1987,30.8244,15.2442,-3.9316,36.8294,8.4599,-7.1987,36.8294,5.8544,-7.1987,36.8294,5.8544,-7.1987,30.8244,15.2442,-13.0859,30.8244,10.5493,0.1425,30.8244,16.9197,0.1425,36.8294,9.3897,-3.9316,36.8294,8.4599,-3.9316,36.8294,8.4599,0.1425,30.8244,16.9197,-7.1987,30.8244,15.2442,7.4837,30.8244,15.2442,4.2166,36.8294,8.4599,0.1425,36.8294,9.3897,0.1425,36.8294,9.3897,7.4837,30.8244,15.2442,0.1425,30.8244,16.9197,13.3709,30.8244,10.5493,7.4837,36.8294,5.8544,4.2166,36.8294,8.4599,4.2166,36.8294,8.4599,13.3709,30.8244,10.5493,7.4837,30.8244,15.2442,16.638,30.8244,3.765,9.2968,36.8294,2.0894,7.4837,36.8294,5.8544,7.4837,36.8294,5.8544,16.638,30.8244,3.765,13.3709,30.8244,10.5493,16.638,30.8244,-3.765,9.2968,36.8294,-2.0894,9.2968,36.8294,2.0894,9.2968,36.8294,2.0894,16.638,30.8244,-3.765,16.638,30.8244,3.765,13.3709,30.8244,-10.5493,7.4837,36.8294,-5.8544,9.2968,36.8294,-2.0894,9.2968,36.8294,-2.0894,13.3709,30.8244,-10.5493,16.638,30.8244,-3.765,7.4837,30.8244,-15.2442,4.2166,36.8294,-8.4599,7.4837,36.8294,-5.8544,7.4837,36.8294,-5.8544,7.4837,30.8244,-15.2442,13.3709,30.8244,-10.5493,0.1425,30.8244,-16.9197,0.1425,36.8294,-9.3897,4.2166,36.8294,-8.4599,4.2166,36.8294,-8.4599,0.1425,30.8244,-16.9197,7.4837,30.8244,-15.2442,-9.0118,22.147,-19.0091,-7.1987,30.8244,-15.2442,0.1425,30.8244,-16.9197,0.1425,30.8244,-16.9197,-9.0118,22.147,-19.0091,0.1425,22.147,-21.0986,-16.353,22.147,-13.1547,-13.0859,30.8244,-10.5493,-7.1987,30.8244,-15.2442,-7.1987,30.8244,-15.2442,-16.353,22.147,-13.1547,-9.0118,22.147,-19.0091,-20.4271,22.147,-4.6949,-16.353,30.8244,-3.765,-13.0859,30.8244,-10.5493,-13.0859,30.8244,-10.5493,-20.4271,22.147,-4.6949,-16.353,22.147,-13.1547,-20.4271,22.147,4.6949,-16.353,30.8244,3.765,-16.353,30.8244,-3.765,-16.353,30.8244,-3.765,-20.4271,22.147,4.6949,-20.4271,22.147,-4.6949,-16.353,22.147,13.1547,-13.0859,30.8244,10.5493,-16.353,30.8244,3.765,-16.353,30.8244,3.765,-16.353,22.147,13.1547,-20.4271,22.147,4.6949,-9.0118,22.147,19.0091,-7.1987,30.8244,15.2442,-13.0859,30.8244,10.5493,-13.0859,30.8244,10.5493,-9.0118,22.147,19.0091,-16.353,22.147,13.1547,0.1425,22.147,21.0986,0.1425,30.8244,16.9197,-7.1987,30.8244,15.2442,-7.1987,30.8244,15.2442,0.1425,22.147,21.0986,-9.0118,22.147,19.0091,9.2968,22.147,19.0091,7.4837,30.8244,15.2442,0.1425,30.8244,16.9197,0.1425,30.8244,16.9197,9.2968,22.147,19.0091,0.1425,22.147,21.0986,16.638,22.147,13.1547,13.3709,30.8244,10.5493,7.4837,30.8244,15.2442,7.4837,30.8244,15.2442,16.638,22.147,13.1547,9.2968,22.147,19.0091,20.7121,22.147,4.6949,16.638,30.8244,3.765,13.3709,30.8244,10.5493,13.3709,30.8244,10.5493,20.7121,22.147,4.6949,16.638,22.147,13.1547,20.7121,22.147,-4.6949,16.638,30.8244,-3.765,16.638,30.8244,3.765,16.638,30.8244,3.765,20.7121,22.147,-4.6949,20.7121,22.147,4.6949,16.638,22.147,-13.1548,13.3709,30.8244,-10.5493,16.638,30.8244,-3.765,16.638,30.8244,-3.765,16.638,22.147,-13.1548,20.7121,22.147,-4.6949,9.2968,22.147,-19.0092,7.4837,30.8244,-15.2442,13.3709,30.8244,-10.5493,13.3709,30.8244,-10.5493,9.2968,22.147,-19.0092,16.638,22.147,-13.1548,0.1425,22.147,-21.0986,0.1425,30.8244,-16.9197,7.4837,30.8244,-15.2442,7.4837,30.8244,-15.2442,0.1425,22.147,-21.0986,9.2968,22.147,-19.0092,-9.0118,-22.0147,-19.0091,-9.0118,22.147,-19.0091,0.1425,22.147,-21.0986,0.1425,22.147,-21.0986,-9.0118,-22.0147,-19.0091,0.1425,-22.0147,-21.0986,-16.353,-22.0147,-13.1547,-16.353,22.147,-13.1547,-9.0118,22.147,-19.0091,-9.0118,22.147,-19.0091,-16.353,-22.0147,-13.1547,-9.0118,-22.0147,-19.0091,-20.4271,-22.0147,-4.6949,-20.4271,22.147,-4.6949,-16.353,22.147,-13.1547,-16.353,22.147,-13.1547,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,-13.1547,-20.4271,-22.0147,4.6949,-20.4271,22.147,4.6949,-20.4271,22.147,-4.6949,-20.4271,22.147,-4.6949,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,13.1547,-16.353,22.147,13.1547,-20.4271,22.147,4.6949,-20.4271,22.147,4.6949,-16.353,-22.0147,13.1547,-20.4271,-22.0147,4.6949,-9.0118,-22.0147,19.0091,-9.0118,22.147,19.0091,-16.353,22.147,13.1547,-16.353,22.147,13.1547,-9.0118,-22.0147,19.0091,-16.353,-22.0147,13.1547,0.1425,-22.0147,21.0986,0.1425,22.147,21.0986,-9.0118,22.147,19.0091,-9.0118,22.147,19.0091,0.1425,-22.0147,21.0986,-9.0118,-22.0147,19.0091,9.2968,-22.0147,19.0091,9.2968,22.147,19.0091,0.1425,22.147,21.0986,0.1425,22.147,21.0986,9.2968,-22.0147,19.0091,0.1425,-22.0147,21.0986,16.638,-22.0147,13.1547,16.638,22.147,13.1547,9.2968,22.147,19.0091,9.2968,22.147,19.0091,16.638,-22.0147,13.1547,9.2968,-22.0147,19.0091,20.7121,-22.0147,4.6949,20.7121,22.147,4.6949,16.638,22.147,13.1547,16.638,22.147,13.1547,20.7121,-22.0147,4.6949,16.638,-22.0147,13.1547,20.7121,-22.0147,-4.6949,20.7121,22.147,-4.6949,20.7121,22.147,4.6949,20.7121,22.147,4.6949,20.7121,-22.0147,-4.6949,20.7121,-22.0147,4.6949,16.638,-22.0147,-13.1548,16.638,22.147,-13.1548,20.7121,22.147,-4.6949,20.7121,22.147,-4.6949,16.638,-22.0147,-13.1548,20.7121,-22.0147,-4.6949,9.2968,-22.0147,-19.0092,9.2968,22.147,-19.0092,16.638,22.147,-13.1548,16.638,22.147,-13.1548,9.2968,-22.0147,-19.0092,16.638,-22.0147,-13.1548,0.1425,-22.0147,-21.0986,0.1425,22.147,-21.0986,9.2968,22.147,-19.0092,9.2968,22.147,-19.0092,0.1425,-22.0147,-21.0986,9.2968,-22.0147,-19.0092,-7.1987,-30.6921,-15.2442,-9.0118,-22.0147,-19.0091,0.1425,-22.0147,-21.0986,0.1425,-22.0147,-21.0986,-7.1987,-30.6921,-15.2442,0.1425,-30.6921,-16.9197,-13.0859,-30.6921,-10.5493,-16.353,-22.0147,-13.1547,-9.0118,-22.0147,-19.0091,-9.0118,-22.0147,-19.0091,-13.0859,-30.6921,-10.5493,-7.1987,-30.6921,-15.2442,-16.353,-30.6921,-3.765,-20.4271,-22.0147,-4.6949,-16.353,-22.0147,-13.1547,-16.353,-22.0147,-13.1547,-16.353,-30.6921,-3.765,-13.0859,-30.6921,-10.5493,-16.353,-30.6921,3.765,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,-4.6949,-20.4271,-22.0147,-4.6949,-16.353,-30.6921,3.765,-16.353,-30.6921,-3.765,-13.0859,-30.6921,10.5493,-16.353,-22.0147,13.1547,-20.4271,-22.0147,4.6949,-20.4271,-22.0147,4.6949,-13.0859,-30.6921,10.5493,-16.353,-30.6921,3.765,-7.1987,-30.6921,15.2442,-9.0118,-22.0147,19.0091,-16.353,-22.0147,13.1547,-16.353,-22.0147,13.1547,-7.1987,-30.6921,15.2442,-13.0859,-30.6921,10.5493,0.1425,-30.6921,16.9197,0.1425,-22.0147,21.0986,-9.0118,-22.0147,19.0091,-9.0118,-22.0147,19.0091,0.1425,-30.6921,16.9197,-7.1987,-30.6921,15.2442,7.4837,-30.6921,15.2442,9.2968,-22.0147,19.0091,0.1425,-22.0147,21.0986,0.1425,-22.0147,21.0986,7.4837,-30.6921,15.2442,0.1425,-30.6921,16.9197,13.3709,-30.6921,10.5493,16.638,-22.0147,13.1547,9.2968,-22.0147,19.0091,9.2968,-22.0147,19.0091,13.3709,-30.6921,10.5493,7.4837,-30.6921,15.2442,16.638,-30.6921,3.765,20.7121,-22.0147,4.6949,16.638,-22.0147,13.1547,16.638,-22.0147,13.1547,16.638,-30.6921,3.765,13.3709,-30.6921,10.5493,16.638,-30.6921,-3.765,20.7121,-22.0147,-4.6949,20.7121,-22.0147,4.6949,20.7121,-22.0147,4.6949,16.638,-30.6921,-3.765,16.638,-30.6921,3.765,13.3709,-30.6921,-10.5493,16.638,-22.0147,-13.1548,20.7121,-22.0147,-4.6949,20.7121,-22.0147,-4.6949,13.3709,-30.6921,-10.5493,16.638,-30.6921,-3.765,7.4837,-30.6921,-15.2442,9.2968,-22.0147,-19.0092,16.638,-22.0147,-13.1548,16.638,-22.0147,-13.1548,7.4837,-30.6921,-15.2442,13.3709,-30.6921,-10.5493,0.1425,-30.6921,-16.9197,0.1425,-22.0147,-21.0986,9.2968,-22.0147,-19.0092,9.2968,-22.0147,-19.0092,0.1425,-30.6921,-16.9197,7.4837,-30.6921,-15.2442,-3.9316,-36.6971,-8.4599,-7.1987,-30.6921,-15.2442,0.1425,-30.6921,-16.9197,0.1425,-30.6921,-16.9197,-3.9316,-36.6971,-8.4599,0.1425,-36.6971,-9.3897,-7.1987,-36.6971,-5.8544,-13.0859,-30.6921,-10.5493,-7.1987,-30.6921,-15.2442,-7.1987,-30.6921,-15.2442,-7.1987,-36.6971,-5.8544,-3.9316,-36.6971,-8.4599,-9.0118,-36.6971,-2.0894,-16.353,-30.6921,-3.765,-13.0859,-30.6921,-10.5493,-13.0859,-30.6921,-10.5493,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,-5.8544,-9.0118,-36.6971,2.0894,-16.353,-30.6921,3.765,-16.353,-30.6921,-3.765,-16.353,-30.6921,-3.765,-9.0118,-36.6971,2.0894,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,5.8544,-13.0859,-30.6921,10.5493,-16.353,-30.6921,3.765,-16.353,-30.6921,3.765,-7.1987,-36.6971,5.8544,-9.0118,-36.6971,2.0894,-3.9316,-36.6971,8.4599,-7.1987,-30.6921,15.2442,-13.0859,-30.6921,10.5493,-13.0859,-30.6921,10.5493,-3.9316,-36.6971,8.4599,-7.1987,-36.6971,5.8544,0.1425,-36.6971,9.3897,0.1425,-30.6921,16.9197,-7.1987,-30.6921,15.2442,-7.1987,-30.6921,15.2442,0.1425,-36.6971,9.3897,-3.9316,-36.6971,8.4599,4.2166,-36.6971,8.4599,7.4837,-30.6921,15.2442,0.1425,-30.6921,16.9197,0.1425,-30.6921,16.9197,4.2166,-36.6971,8.4599,0.1425,-36.6971,9.3897,7.4837,-36.6971,5.8544,13.3709,-30.6921,10.5493,7.4837,-30.6921,15.2442,7.4837,-30.6921,15.2442,7.4837,-36.6971,5.8544,4.2166,-36.6971,8.4599,9.2968,-36.6971,2.0894,16.638,-30.6921,3.765,13.3709,-30.6921,10.5493,13.3709,-30.6921,10.5493,9.2968,-36.6971,2.0894,7.4837,-36.6971,5.8544,9.2968,-36.6971,-2.0894,16.638,-30.6921,-3.765,16.638,-30.6921,3.765,16.638,-30.6921,3.765,9.2968,-36.6971,-2.0894,9.2968,-36.6971,2.0894,7.4837,-36.6971,-5.8544,13.3709,-30.6921,-10.5493,16.638,-30.6921,-3.765,16.638,-30.6921,-3.765,7.4837,-36.6971,-5.8544,9.2968,-36.6971,-2.0894,4.2166,-36.6971,-8.4599,7.4837,-30.6921,-15.2442,13.3709,-30.6921,-10.5493,13.3709,-30.6921,-10.5493,4.2166,-36.6971,-8.4599,7.4837,-36.6971,-5.8544,0.1425,-36.6971,-9.3897,0.1425,-30.6921,-16.9197,7.4837,-30.6921,-15.2442,7.4837,-30.6921,-15.2442,0.1425,-36.6971,-9.3897,4.2166,-36.6971,-8.4599,0.1425,-44.7884,0,-3.9316,-44.7984,-8.4599,0.1425,-44.7984,-9.3897,0.1425,-44.7884,0,-7.1987,-44.7984,-5.8544,-3.9316,-44.7984,-8.4599,0.1425,-44.7884,0,-9.0118,-44.7984,-2.0894,-7.1987,-44.7984,-5.8544,0.1425,-44.7884,0,-9.0118,-44.7984,2.0894,-9.0118,-44.7984,-2.0894,0.1425,-44.7884,0,-7.1987,-44.7984,5.8544,-9.0118,-44.7984,2.0894,0.1425,-44.7884,0,-3.9316,-44.7984,8.4599,-7.1987,-44.7984,5.8544,0.1425,-44.7884,0,0.1425,-44.7984,9.3897,-3.9316,-44.7984,8.4599,0.1425,-44.7884,0,4.2166,-44.7984,8.4599,0.1425,-44.7984,9.3897,0.1425,-44.7884,0,7.4837,-44.7984,5.8544,4.2166,-44.7984,8.4599,0.1425,-44.7884,0,9.2968,-44.7984,2.0894,7.4837,-44.7984,5.8544,0.1425,-44.7884,0,9.2968,-44.7984,-2.0894,9.2968,-44.7984,2.0894,0.1425,-44.7884,0,7.4837,-44.7984,-5.8544,9.2968,-44.7984,-2.0894,0.1425,-44.7884,0,4.2166,-44.7984,-8.4599,7.4837,-44.7984,-5.8544,0.1425,-44.7884,0,0.1425,-44.7984,-9.3897,4.2166,-44.7984,-8.4599,0.1425,36.8294,-9.3897,-3.9316,36.8294,-8.4599,-3.9316,37.8294,-8.4599,-3.9316,37.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,37.8294,-9.3897,-3.9316,36.8294,-8.4599,-7.1987,36.8294,-5.8544,-7.1987,37.8294,-5.8544,-7.1987,37.8294,-5.8544,-3.9316,36.8294,-8.4599,-3.9316,37.8294,-8.4599,-7.1987,36.8294,-5.8544,-9.0118,36.8294,-2.0894,-9.0118,37.8294,-2.0894,-9.0118,37.8294,-2.0894,-7.1987,36.8294,-5.8544,-7.1987,37.8294,-5.8544,-9.0118,36.8294,-2.0894,-9.0118,36.8294,2.0894,-9.0118,37.8294,2.0894,-9.0118,37.8294,2.0894,-9.0118,36.8294,-2.0894,-9.0118,37.8294,-2.0894,-9.0118,36.8294,2.0894,-7.1987,36.8294,5.8544,-7.1987,37.8294,5.8544,-7.1987,37.8294,5.8544,-9.0118,36.8294,2.0894,-9.0118,37.8294,2.0894,-7.1987,36.8294,5.8544,-3.9316,36.8294,8.4599,-3.9316,37.8294,8.4599,-3.9316,37.8294,8.4599,-7.1987,36.8294,5.8544,-7.1987,37.8294,5.8544,-3.9316,36.8294,8.4599,0.1425,36.8294,9.3897,0.1425,37.8294,9.3897,0.1425,37.8294,9.3897,-3.9316,36.8294,8.4599,-3.9316,37.8294,8.4599,0.1425,36.8294,9.3897,4.2166,36.8294,8.4599,4.2166,37.8294,8.4599,4.2166,37.8294,8.4599,0.1425,36.8294,9.3897,0.1425,37.8294,9.3897,4.2166,36.8294,8.4599,7.4837,36.8294,5.8544,7.4837,37.8294,5.8544,7.4837,37.8294,5.8544,4.2166,36.8294,8.4599,4.2166,37.8294,8.4599,7.4837,36.8294,5.8544,9.2968,36.8294,2.0894,9.2968,37.8294,2.0894,9.2968,37.8294,2.0894,7.4837,36.8294,5.8544,7.4837,37.8294,5.8544,9.2968,36.8294,2.0894,9.2968,36.8294,-2.0894,9.2968,37.8294,-2.0894,9.2968,37.8294,-2.0894,9.2968,36.8294,2.0894,9.2968,37.8294,2.0894,9.2968,36.8294,-2.0894,7.4837,36.8294,-5.8544,7.4837,37.8294,-5.8544,7.4837,37.8294,-5.8544,9.2968,36.8294,-2.0894,9.2968,37.8294,-2.0894,7.4837,36.8294,-5.8544,4.2166,36.8294,-8.4599,4.2166,37.8294,-8.4599,4.2166,37.8294,-8.4599,7.4837,36.8294,-5.8544,7.4837,37.8294,-5.8544,4.2166,36.8294,-8.4599,0.1425,36.8294,-9.3897,0.1425,37.8294,-9.3897,0.1425,37.8294,-9.3897,4.2166,36.8294,-8.4599,4.2166,37.8294,-8.4599,-3.9316,-36.6971,-8.4599,0.1425,-36.6971,-9.3897,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-9.3897,-3.9316,-36.6971,-8.4599,-3.9316,-37.6971,-8.4599,0.1425,-36.6971,-9.3897,4.2166,-36.6971,-8.4599,4.2166,-37.6971,-8.4599,4.2166,-37.6971,-8.4599,0.1425,-36.6971,-9.3897,0.1425,-37.6971,-9.3897,4.2166,-36.6971,-8.4599,7.4837,-36.6971,-5.8544,7.4837,-37.6971,-5.8544,7.4837,-37.6971,-5.8544,4.2166,-36.6971,-8.4599,4.2166,-37.6971,-8.4599,7.4837,-36.6971,-5.8544,9.2968,-36.6971,-2.0894,9.2968,-37.6971,-2.0894,9.2968,-37.6971,-2.0894,7.4837,-36.6971,-5.8544,7.4837,-37.6971,-5.8544,9.2968,-36.6971,-2.0894,9.2968,-36.6971,2.0894,9.2968,-37.6971,2.0894,9.2968,-37.6971,2.0894,9.2968,-36.6971,-2.0894,9.2968,-37.6971,-2.0894,9.2968,-36.6971,2.0894,7.4837,-36.6971,5.8544,7.4837,-37.6971,5.8544,7.4837,-37.6971,5.8544,9.2968,-36.6971,2.0894,9.2968,-37.6971,2.0894,7.4837,-36.6971,5.8544,4.2166,-36.6971,8.4599,4.2166,-37.6971,8.4599,4.2166,-37.6971,8.4599,7.4837,-36.6971,5.8544,7.4837,-37.6971,5.8544,4.2166,-36.6971,8.4599,0.1425,-36.6971,9.3897,0.1425,-37.6971,9.3897,0.1425,-37.6971,9.3897,4.2166,-36.6971,8.4599,4.2166,-37.6971,8.4599,0.1425,-36.6971,9.3897,-3.9316,-36.6971,8.4599,-3.9316,-37.6971,8.4599,-3.9316,-37.6971,8.4599,0.1425,-36.6971,9.3897,0.1425,-37.6971,9.3897,-3.9316,-36.6971,8.4599,-7.1987,-36.6971,5.8544,-7.1987,-37.6971,5.8544,-7.1987,-37.6971,5.8544,-3.9316,-36.6971,8.4599,-3.9316,-37.6971,8.4599,-7.1987,-36.6971,5.8544,-9.0118,-36.6971,2.0894,-9.0118,-37.6971,2.0894,-9.0118,-37.6971,2.0894,-7.1987,-36.6971,5.8544,-7.1987,-37.6971,5.8544,-9.0118,-36.6971,2.0894,-9.0118,-36.6971,-2.0894,-9.0118,-37.6971,-2.0894,-9.0118,-37.6971,-2.0894,-9.0118,-36.6971,2.0894,-9.0118,-37.6971,2.0894,-9.0118,-36.6971,-2.0894,-7.1987,-36.6971,-5.8544,-7.1987,-37.6971,-5.8544,-7.1987,-37.6971,-5.8544,-9.0118,-36.6971,-2.0894,-9.0118,-37.6971,-2.0894,-7.1987,-36.6971,-5.8544,-3.9316,-36.6971,-8.4599,-3.9316,-37.6971,-8.4599,-3.9316,-37.6971,-8.4599,-7.1987,-36.6971,-5.8544,-7.1987,-37.6971,-5.8544,0.1424,37.8294,-13.3932,-5.6686,37.8294,-12.0669,-5.6686,44.9308,-12.0669,-5.6686,44.9308,-12.0669,0.1424,37.8294,-13.3932,0.1424,44.9308,-13.3932,-5.6686,37.8294,-12.0669,-10.3288,37.8294,-8.3506,-10.3288,44.9308,-8.3506,-10.3288,44.9308,-8.3506,-5.6686,37.8294,-12.0669,-5.6686,44.9308,-12.0669,-10.3288,37.8294,-8.3506,-12.9149,37.8294,-2.9803,-12.9149,44.9308,-2.9803,-12.9149,44.9308,-2.9803,-10.3288,37.8294,-8.3506,-10.3288,44.9308,-8.3506,-12.9149,37.8294,-2.9803,-12.9149,37.8294,2.9803,-12.9149,44.9308,2.9803,-12.9149,44.9308,2.9803,-12.9149,37.8294,-2.9803,-12.9149,44.9308,-2.9803,-12.9149,37.8294,2.9803,-10.3288,37.8294,8.3506,-10.3288,44.9308,8.3506,-10.3288,44.9308,8.3506,-12.9149,37.8294,2.9803,-12.9149,44.9308,2.9803,-10.3288,37.8294,8.3506,-5.6687,37.8294,12.0669,-5.6687,44.9308,12.0669,-5.6687,44.9308,12.0669,-10.3288,37.8294,8.3506,-10.3288,44.9308,8.3506,-5.6687,37.8294,12.0669,0.1425,37.8294,13.3932,0.1425,44.9308,13.3932,0.1425,44.9308,13.3932,-5.6687,37.8294,12.0669,-5.6687,44.9308,12.0669,0.1425,37.8294,13.3932,5.9537,37.8294,12.0669,5.9537,44.9308,12.0669,5.9537,44.9308,12.0669,0.1425,37.8294,13.3932,0.1425,44.9308,13.3932,5.9537,37.8294,12.0669,10.6138,37.8294,8.3506,10.6138,44.9308,8.3506,10.6138,44.9308,8.3506,5.9537,37.8294,12.0669,5.9537,44.9308,12.0669,10.6138,37.8294,8.3506,13.2,37.8294,2.9803,13.2,44.9308,2.9803,13.2,44.9308,2.9803,10.6138,37.8294,8.3506,10.6138,44.9308,8.3506,13.2,37.8294,2.9803,13.2,37.8294,-2.9802,13.2,44.9308,-2.9802,13.2,44.9308,-2.9802,13.2,37.8294,2.9803,13.2,44.9308,2.9803,13.2,37.8294,-2.9802,10.6137,37.8294,-8.3506,10.6137,44.9308,-8.3506,10.6137,44.9308,-8.3506,13.2,37.8294,-2.9802,13.2,44.9308,-2.9802,10.6137,37.8294,-8.3506,5.9537,37.8294,-12.0669,5.9537,44.9308,-12.0669,5.9537,44.9308,-12.0669,10.6137,37.8294,-8.3506,10.6137,44.9308,-8.3506,5.9537,37.8294,-12.0669,0.1424,37.8294,-13.3932,0.1424,44.9308,-13.3932,0.1424,44.9308,-13.3932,5.9537,37.8294,-12.0669,5.9537,44.9308,-12.0669,-5.6686,-37.6971,-12.0669,0.1425,-37.6971,-13.3932,0.1425,-44.7984,-13.3932,0.1425,-44.7984,-13.3932,-5.6686,-37.6971,-12.0669,-5.6686,-44.7984,-12.0669,0.1425,-37.6971,-13.3932,5.9536,-37.6971,-12.0669,5.9536,-44.7984,-12.0669,5.9536,-44.7984,-12.0669,0.1425,-37.6971,-13.3932,0.1425,-44.7984,-13.3932,5.9536,-37.6971,-12.0669,10.6138,-37.6971,-8.3506,10.6138,-44.7984,-8.3506,10.6138,-44.7984,-8.3506,5.9536,-37.6971,-12.0669,5.9536,-44.7984,-12.0669,10.6138,-37.6971,-8.3506,13.2,-37.6971,-2.9803,13.2,-44.7984,-2.9803,13.2,-44.7984,-2.9803,10.6138,-37.6971,-8.3506,10.6138,-44.7984,-8.3506,13.2,-37.6971,-2.9803,13.2,-37.6971,2.9802,13.2,-44.7984,2.9802,13.2,-44.7984,2.9802,13.2,-37.6971,-2.9803,13.2,-44.7984,-2.9803,13.2,-37.6971,2.9802,10.6138,-37.6971,8.3505,10.6138,-44.7984,8.3505,10.6138,-44.7984,8.3505,13.2,-37.6971,2.9802,13.2,-44.7984,2.9802,10.6138,-37.6971,8.3505,5.9535,-37.6971,12.0669,5.9535,-44.7984,12.0669,5.9535,-44.7984,12.0669,10.6138,-37.6971,8.3505,10.6138,-44.7984,8.3505,5.9535,-37.6971,12.0669,0.1425,-37.6971,13.3932,0.1425,-44.7984,13.3932,0.1425,-44.7984,13.3932,5.9535,-37.6971,12.0669,5.9535,-44.7984,12.0669,0.1425,-37.6971,13.3932,-5.6686,-37.6971,12.0669,-5.6686,-44.7984,12.0669,-5.6686,-44.7984,12.0669,0.1425,-37.6971,13.3932,0.1425,-44.7984,13.3932,-5.6686,-37.6971,12.0669,-10.3288,-37.6971,8.3506,-10.3288,-44.7984,8.3506,-10.3288,-44.7984,8.3506,-5.6686,-37.6971,12.0669,-5.6686,-44.7984,12.0669,-10.3288,-37.6971,8.3506,-12.9149,-37.6971,2.9803,-12.9149,-44.7984,2.9803,-12.9149,-44.7984,2.9803,-10.3288,-37.6971,8.3506,-10.3288,-44.7984,8.3506,-12.9149,-37.6971,2.9803,-12.9149,-37.6971,-2.9803,-12.9149,-44.7984,-2.9803,-12.9149,-44.7984,-2.9803,-12.9149,-37.6971,2.9803,-12.9149,-44.7984,2.9803,-12.9149,-37.6971,-2.9803,-10.3288,-37.6971,-8.3506,-10.3288,-44.7984,-8.3506,-10.3288,-44.7984,-8.3506,-12.9149,-37.6971,-2.9803,-12.9149,-44.7984,-2.9803,-10.3288,-37.6971,-8.3506,-5.6686,-37.6971,-12.0669,-5.6686,-44.7984,-12.0669,-5.6686,-44.7984,-12.0669,-10.3288,-37.6971,-8.3506,-10.3288,-44.7984,-8.3506,0.1425,37.8294,-9.3897,-3.9316,37.8294,-8.4599,-5.6686,37.8294,-12.0669,-5.6686,37.8294,-12.0669,0.1425,37.8294,-9.3897,0.1424,37.8294,-13.3932,-3.9316,37.8294,-8.4599,-7.1987,37.8294,-5.8544,-10.3288,37.8294,-8.3506,-10.3288,37.8294,-8.3506,-3.9316,37.8294,-8.4599,-5.6686,37.8294,-12.0669,-7.1987,37.8294,-5.8544,-9.0118,37.8294,-2.0894,-12.9149,37.8294,-2.9803,-12.9149,37.8294,-2.9803,-7.1987,37.8294,-5.8544,-10.3288,37.8294,-8.3506,-9.0118,37.8294,-2.0894,-9.0118,37.8294,2.0894,-12.9149,37.8294,2.9803,-12.9149,37.8294,2.9803,-9.0118,37.8294,-2.0894,-12.9149,37.8294,-2.9803,-9.0118,37.8294,2.0894,-7.1987,37.8294,5.8544,-10.3288,37.8294,8.3506,-10.3288,37.8294,8.3506,-9.0118,37.8294,2.0894,-12.9149,37.8294,2.9803,-7.1987,37.8294,5.8544,-3.9316,37.8294,8.4599,-5.6687,37.8294,12.0669,-5.6687,37.8294,12.0669,-7.1987,37.8294,5.8544,-10.3288,37.8294,8.3506,-3.9316,37.8294,8.4599,0.1425,37.8294,9.3897,0.1425,37.8294,13.3932,0.1425,37.8294,13.3932,-3.9316,37.8294,8.4599,-5.6687,37.8294,12.0669,0.1425,37.8294,9.3897,4.2166,37.8294,8.4599,5.9537,37.8294,12.0669,5.9537,37.8294,12.0669,0.1425,37.8294,9.3897,0.1425,37.8294,13.3932,4.2166,37.8294,8.4599,7.4837,37.8294,5.8544,10.6138,37.8294,8.3506,10.6138,37.8294,8.3506,4.2166,37.8294,8.4599,5.9537,37.8294,12.0669,7.4837,37.8294,5.8544,9.2968,37.8294,2.0894,13.2,37.8294,2.9803,13.2,37.8294,2.9803,7.4837,37.8294,5.8544,10.6138,37.8294,8.3506,9.2968,37.8294,2.0894,9.2968,37.8294,-2.0894,13.2,37.8294,-2.9802,13.2,37.8294,-2.9802,9.2968,37.8294,2.0894,13.2,37.8294,2.9803,9.2968,37.8294,-2.0894,7.4837,37.8294,-5.8544,10.6137,37.8294,-8.3506,10.6137,37.8294,-8.3506,9.2968,37.8294,-2.0894,13.2,37.8294,-2.9802,7.4837,37.8294,-5.8544,4.2166,37.8294,-8.4599,5.9537,37.8294,-12.0669,5.9537,37.8294,-12.0669,7.4837,37.8294,-5.8544,10.6137,37.8294,-8.3506,4.2166,37.8294,-8.4599,0.1425,37.8294,-9.3897,0.1424,37.8294,-13.3932,0.1424,37.8294,-13.3932,4.2166,37.8294,-8.4599,5.9537,37.8294,-12.0669,0.1424,44.9308,-13.3932,-5.6686,44.9308,-12.0669,-3.9316,44.9308,-8.4599,-3.9316,44.9308,-8.4599,0.1424,44.9308,-13.3932,0.1425,44.9308,-9.3897,5.9537,44.9308,-12.0669,0.1424,44.9308,-13.3932,0.1425,44.9308,-9.3897,0.1425,44.9308,-9.3897,5.9537,44.9308,-12.0669,4.2166,44.9308,-8.4599,10.6137,44.9308,-8.3506,5.9537,44.9308,-12.0669,4.2166,44.9308,-8.4599,4.2166,44.9308,-8.4599,10.6137,44.9308,-8.3506,7.4837,44.9308,-5.8544,13.2,44.9308,-2.9802,10.6137,44.9308,-8.3506,7.4837,44.9308,-5.8544,7.4837,44.9308,-5.8544,13.2,44.9308,-2.9802,9.2968,44.9308,-2.0894,13.2,44.9308,2.9803,13.2,44.9308,-2.9802,9.2968,44.9308,-2.0894,9.2968,44.9308,-2.0894,13.2,44.9308,2.9803,9.2968,44.9308,2.0894,10.6138,44.9308,8.3506,13.2,44.9308,2.9803,9.2968,44.9308,2.0894,9.2968,44.9308,2.0894,10.6138,44.9308,8.3506,7.4837,44.9308,5.8544,5.9537,44.9308,12.0669,10.6138,44.9308,8.3506,7.4837,44.9308,5.8544,7.4837,44.9308,5.8544,5.9537,44.9308,12.0669,4.2166,44.9308,8.4599,0.1425,44.9308,13.3932,5.9537,44.9308,12.0669,4.2166,44.9308,8.4599,4.2166,44.9308,8.4599,0.1425,44.9308,13.3932,0.1425,44.9308,9.3897,-5.6687,44.9308,12.0669,0.1425,44.9308,13.3932,0.1425,44.9308,9.3897,0.1425,44.9308,9.3897,-5.6687,44.9308,12.0669,-3.9316,44.9308,8.4599,-10.3288,44.9308,8.3506,-5.6687,44.9308,12.0669,-3.9316,44.9308,8.4599,-3.9316,44.9308,8.4599,-10.3288,44.9308,8.3506,-7.1987,44.9308,5.8544,-12.9149,44.9308,2.9803,-10.3288,44.9308,8.3506,-7.1987,44.9308,5.8544,-7.1987,44.9308,5.8544,-12.9149,44.9308,2.9803,-9.0118,44.9308,2.0894,-12.9149,44.9308,-2.9803,-12.9149,44.9308,2.9803,-9.0118,44.9308,2.0894,-9.0118,44.9308,2.0894,-12.9149,44.9308,-2.9803,-9.0118,44.9308,-2.0894,-10.3288,44.9308,-8.3506,-12.9149,44.9308,-2.9803,-9.0118,44.9308,-2.0894,-9.0118,44.9308,-2.0894,-10.3288,44.9308,-8.3506,-7.1987,44.9308,-5.8544,-5.6686,44.9308,-12.0669,-10.3288,44.9308,-8.3506,-7.1987,44.9308,-5.8544,-7.1987,44.9308,-5.8544,-5.6686,44.9308,-12.0669,-3.9316,44.9308,-8.4599,-3.9316,-37.6971,-8.4599,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-13.3932,0.1425,-37.6971,-13.3932,-3.9316,-37.6971,-8.4599,-5.6686,-37.6971,-12.0669,0.1425,-37.6971,-9.3897,4.2166,-37.6971,-8.4599,5.9536,-37.6971,-12.0669,5.9536,-37.6971,-12.0669,0.1425,-37.6971,-9.3897,0.1425,-37.6971,-13.3932,4.2166,-37.6971,-8.4599,7.4837,-37.6971,-5.8544,10.6138,-37.6971,-8.3506,10.6138,-37.6971,-8.3506,4.2166,-37.6971,-8.4599,5.9536,-37.6971,-12.0669,7.4837,-37.6971,-5.8544,9.2968,-37.6971,-2.0894,13.2,-37.6971,-2.9803,13.2,-37.6971,-2.9803,7.4837,-37.6971,-5.8544,10.6138,-37.6971,-8.3506,9.2968,-37.6971,-2.0894,9.2968,-37.6971,2.0894,13.2,-37.6971,2.9802,13.2,-37.6971,2.9802,9.2968,-37.6971,-2.0894,13.2,-37.6971,-2.9803,9.2968,-37.6971,2.0894,7.4837,-37.6971,5.8544,10.6138,-37.6971,8.3505,10.6138,-37.6971,8.3505,9.2968,-37.6971,2.0894,13.2,-37.6971,2.9802,7.4837,-37.6971,5.8544,4.2166,-37.6971,8.4599,5.9535,-37.6971,12.0669,5.9535,-37.6971,12.0669,7.4837,-37.6971,5.8544,10.6138,-37.6971,8.3505,4.2166,-37.6971,8.4599,0.1425,-37.6971,9.3897,0.1425,-37.6971,13.3932,0.1425,-37.6971,13.3932,4.2166,-37.6971,8.4599,5.9535,-37.6971,12.0669,0.1425,-37.6971,9.3897,-3.9316,-37.6971,8.4599,-5.6686,-37.6971,12.0669,-5.6686,-37.6971,12.0669,0.1425,-37.6971,9.3897,0.1425,-37.6971,13.3932,-3.9316,-37.6971,8.4599,-7.1987,-37.6971,5.8544,-10.3288,-37.6971,8.3506,-10.3288,-37.6971,8.3506,-3.9316,-37.6971,8.4599,-5.6686,-37.6971,12.0669,-7.1987,-37.6971,5.8544,-9.0118,-37.6971,2.0894,-12.9149,-37.6971,2.9803,-12.9149,-37.6971,2.9803,-7.1987,-37.6971,5.8544,-10.3288,-37.6971,8.3506,-9.0118,-37.6971,2.0894,-9.0118,-37.6971,-2.0894,-12.9149,-37.6971,-2.9803,-12.9149,-37.6971,-2.9803,-9.0118,-37.6971,2.0894,-12.9149,-37.6971,2.9803,-9.0118,-37.6971,-2.0894,-7.1987,-37.6971,-5.8544,-10.3288,-37.6971,-8.3506,-10.3288,-37.6971,-8.3506,-9.0118,-37.6971,-2.0894,-12.9149,-37.6971,-2.9803,-7.1987,-37.6971,-5.8544,-3.9316,-37.6971,-8.4599,-5.6686,-37.6971,-12.0669,-5.6686,-37.6971,-12.0669,-7.1987,-37.6971,-5.8544,-10.3288,-37.6971,-8.3506,0.1425,-44.7984,-9.3897,-3.9316,-44.7984,-8.4599,-5.6686,-44.7984,-12.0669,-5.6686,-44.7984,-12.0669,0.1425,-44.7984,-9.3897,0.1425,-44.7984,-13.3932,-3.9316,-44.7984,-8.4599,-7.1987,-44.7984,-5.8544,-10.3288,-44.7984,-8.3506,-10.3288,-44.7984,-8.3506,-3.9316,-44.7984,-8.4599,-5.6686,-44.7984,-12.0669,-7.1987,-44.7984,-5.8544,-9.0118,-44.7984,-2.0894,-12.9149,-44.7984,-2.9803,-12.9149,-44.7984,-2.9803,-7.1987,-44.7984,-5.8544,-10.3288,-44.7984,-8.3506,-9.0118,-44.7984,-2.0894,-9.0118,-44.7984,2.0894,-12.9149,-44.7984,2.9803,-12.9149,-44.7984,2.9803,-9.0118,-44.7984,-2.0894,-12.9149,-44.7984,-2.9803,-9.0118,-44.7984,2.0894,-7.1987,-44.7984,5.8544,-10.3288,-44.7984,8.3506,-10.3288,-44.7984,8.3506,-9.0118,-44.7984,2.0894,-12.9149,-44.7984,2.9803,-7.1987,-44.7984,5.8544,-3.9316,-44.7984,8.4599,-5.6686,-44.7984,12.0669,-5.6686,-44.7984,12.0669,-7.1987,-44.7984,5.8544,-10.3288,-44.7984,8.3506,-3.9316,-44.7984,8.4599,0.1425,-44.7984,9.3897,0.1425,-44.7984,13.3932,0.1425,-44.7984,13.3932,-3.9316,-44.7984,8.4599,-5.6686,-44.7984,12.0669,0.1425,-44.7984,9.3897,4.2166,-44.7984,8.4599,5.9535,-44.7984,12.0669,5.9535,-44.7984,12.0669,0.1425,-44.7984,9.3897,0.1425,-44.7984,13.3932,4.2166,-44.7984,8.4599,7.4837,-44.7984,5.8544,10.6138,-44.7984,8.3505,10.6138,-44.7984,8.3505,4.2166,-44.7984,8.4599,5.9535,-44.7984,12.0669,7.4837,-44.7984,5.8544,9.2968,-44.7984,2.0894,13.2,-44.7984,2.9802,13.2,-44.7984,2.9802,7.4837,-44.7984,5.8544,10.6138,-44.7984,8.3505,9.2968,-44.7984,2.0894,9.2968,-44.7984,-2.0894,13.2,-44.7984,-2.9803,13.2,-44.7984,-2.9803,9.2968,-44.7984,2.0894,13.2,-44.7984,2.9802,9.2968,-44.7984,-2.0894,7.4837,-44.7984,-5.8544,10.6138,-44.7984,-8.3506,10.6138,-44.7984,-8.3506,9.2968,-44.7984,-2.0894,13.2,-44.7984,-2.9803,7.4837,-44.7984,-5.8544,4.2166,-44.7984,-8.4599,5.9536,-44.7984,-12.0669,5.9536,-44.7984,-12.0669,7.4837,-44.7984,-5.8544,10.6138,-44.7984,-8.3506,4.2166,-44.7984,-8.4599,0.1425,-44.7984,-9.3897,0.1425,-44.7984,-13.3932,0.1425,-44.7984,-13.3932,4.2166,-44.7984,-8.4599,5.9536,-44.7984,-12.0669,0.1425,40.3208,-9.3897,0.1425,44.9308,-9.3897,-3.9316,44.9308,-8.4599,-3.9316,44.9308,-8.4599,0.1425,40.3208,-9.3897,-3.9316,40.3208,-8.4599,4.2166,40.3208,-8.4599,4.2166,44.9308,-8.4599,0.1425,44.9308,-9.3897,0.1425,44.9308,-9.3897,4.2166,40.3208,-8.4599,0.1425,40.3208,-9.3897,7.4837,40.3208,-5.8544,7.4837,44.9308,-5.8544,4.2166,44.9308,-8.4599,4.2166,44.9308,-8.4599,7.4837,40.3208,-5.8544,4.2166,40.3208,-8.4599,9.2968,40.3208,-2.0894,9.2968,44.9308,-2.0894,7.4837,44.9308,-5.8544,7.4837,44.9308,-5.8544,9.2968,40.3208,-2.0894,7.4837,40.3208,-5.8544,9.2968,40.3208,2.0894,9.2968,44.9308,2.0894,9.2968,44.9308,-2.0894,9.2968,44.9308,-2.0894,9.2968,40.3208,2.0894,9.2968,40.3208,-2.0894,7.4837,40.3208,5.8544,7.4837,44.9308,5.8544,9.2968,44.9308,2.0894,9.2968,44.9308,2.0894,7.4837,40.3208,5.8544,9.2968,40.3208,2.0894,4.2166,40.3208,8.4599,4.2166,44.9308,8.4599,7.4837,44.9308,5.8544,7.4837,44.9308,5.8544,4.2166,40.3208,8.4599,7.4837,40.3208,5.8544,0.1425,40.3208,9.3897,0.1425,44.9308,9.3897,4.2166,44.9308,8.4599,4.2166,44.9308,8.4599,0.1425,40.3208,9.3897,4.2166,40.3208,8.4599,-3.9316,40.3208,8.4599,-3.9316,44.9308,8.4599,0.1425,44.9308,9.3897,0.1425,44.9308,9.3897,-3.9316,40.3208,8.4599,0.1425,40.3208,9.3897,-7.1987,40.3208,5.8544,-7.1987,44.9308,5.8544,-3.9316,44.9308,8.4599,-3.9316,44.9308,8.4599,-7.1987,40.3208,5.8544,-3.9316,40.3208,8.4599,-9.0118,40.3208,2.0894,-9.0118,44.9308,2.0894,-7.1987,44.9308,5.8544,-7.1987,44.9308,5.8544,-9.0118,40.3208,2.0894,-7.1987,40.3208,5.8544,-9.0118,40.3208,-2.0894,-9.0118,44.9308,-2.0894,-9.0118,44.9308,2.0894,-9.0118,44.9308,2.0894,-9.0118,40.3208,-2.0894,-9.0118,40.3208,2.0894,-7.1987,40.3208,-5.8544,-7.1987,44.9308,-5.8544,-9.0118,44.9308,-2.0894,-9.0118,44.9308,-2.0894,-7.1987,40.3208,-5.8544,-9.0118,40.3208,-2.0894,-3.9316,40.3208,-8.4599,-3.9316,44.9308,-8.4599,-7.1987,44.9308,-5.8544,-7.1987,44.9308,-5.8544,-3.9316,40.3208,-8.4599,-7.1987,40.3208,-5.8544,-7.1987,40.3208,-5.8544,-9.0118,40.3208,-2.0894,-9.0118,40.3208,2.0894,-9.0118,40.3208,2.0894,-7.1987,40.3208,-5.8544,0.1425,40.3208,-9.3897,-9.0118,40.3208,2.0894,-7.1987,40.3208,5.8544,-3.9316,40.3208,8.4599,-3.9316,40.3208,8.4599,-9.0118,40.3208,2.0894,4.2166,40.3208,8.4599,-3.9316,40.3208,8.4599,0.1425,40.3208,9.3897,4.2166,40.3208,8.4599,4.2166,40.3208,8.4599,7.4837,40.3208,5.8544,9.2968,40.3208,2.0894,9.2968,40.3208,2.0894,4.2166,40.3208,8.4599,7.4837,40.3208,-5.8544,9.2968,40.3208,2.0894,9.2968,40.3208,-2.0894,7.4837,40.3208,-5.8544,7.4837,40.3208,-5.8544,4.2166,40.3208,-8.4599,0.1425,40.3208,-9.3897,0.1425,40.3208,-9.3897,7.4837,40.3208,-5.8544,4.2166,40.3208,8.4599,-9.0118,40.3208,2.0894,4.2166,40.3208,8.4599,0.1425,40.3208,-9.3897,-3.9316,40.3208,-8.4599,-7.1987,40.3208,-5.8544,0.1425,40.3208,-9.3897,10.7801,46.4706,0.25,11.0675,43.9792,0.25,11.1468,43.9792,0,11.1468,43.9792,0,10.7801,46.4706,0.25,10.8572,46.4887,0,10.5692,46.4214,0.433,10.851,43.9792,0.433,11.0675,43.9792,0.25,11.0675,43.9792,0.25,10.5692,46.4214,0.433,10.7801,46.4706,0.25,10.2812,46.3542,0.5,10.5553,43.9792,0.5,10.851,43.9792,0.433,10.851,43.9792,0.433,10.2812,46.3542,0.5,10.5692,46.4214,0.433,9.9932,46.287,0.433,10.2595,43.9792,0.433,10.5553,43.9792,0.5,10.5553,43.9792,0.5,9.9932,46.287,0.433,10.2812,46.3542,0.5,9.7824,46.2378,0.25,10.043,43.9792,0.25,10.2595,43.9792,0.433,10.2595,43.9792,0.433,9.7824,46.2378,0.25,9.9932,46.287,0.433,9.7052,46.2198,0,9.9637,43.9792,0,10.043,43.9792,0.25,10.043,43.9792,0.25,9.7052,46.2198,0,9.7824,46.2378,0.25,9.7824,46.2378,-0.25,10.043,43.9792,-0.25,9.9637,43.9792,0,9.9637,43.9792,0,9.7824,46.2378,-0.25,9.7052,46.2198,0,9.9932,46.287,-0.433,10.2595,43.9792,-0.433,10.043,43.9792,-0.25,10.043,43.9792,-0.25,9.9932,46.287,-0.433,9.7824,46.2378,-0.25,10.2812,46.3542,-0.5,10.5553,43.9792,-0.5,10.2595,43.9792,-0.433,10.2595,43.9792,-0.433,10.2812,46.3542,-0.5,9.9932,46.287,-0.433,10.5692,46.4214,-0.433,10.851,43.9792,-0.433,10.5553,43.9792,-0.5,10.5553,43.9792,-0.5,10.5692,46.4214,-0.433,10.2812,46.3542,-0.5,10.7801,46.4706,-0.25,11.0675,43.9792,-0.25,10.851,43.9792,-0.433,10.851,43.9792,-0.433,10.7801,46.4706,-0.25,10.5692,46.4214,-0.433,10.8572,46.4887,0,11.1468,43.9792,0,11.0675,43.9792,-0.25,11.0675,43.9792,-0.25,10.8572,46.4887,0,10.7801,46.4706,-0.25,9.9608,48.7604,0.25,10.7801,46.4706,0.25,10.8572,46.4887,0,10.8572,46.4887,0,9.9608,48.7604,0.25,10.0319,48.7952,0,9.7664,48.6653,0.433,10.5692,46.4214,0.433,10.7801,46.4706,0.25,10.7801,46.4706,0.25,9.7664,48.6653,0.433,9.9608,48.7604,0.25,9.5008,48.5353,0.5,10.2812,46.3542,0.5,10.5692,46.4214,0.433,10.5692,46.4214,0.433,9.5008,48.5353,0.5,9.7664,48.6653,0.433,9.2353,48.4054,0.433,9.9932,46.287,0.433,10.2812,46.3542,0.5,10.2812,46.3542,0.5,9.2353,48.4054,0.433,9.5008,48.5353,0.5,9.0409,48.3102,0.25,9.7824,46.2378,0.25,9.9932,46.287,0.433,9.9932,46.287,0.433,9.0409,48.3102,0.25,9.2353,48.4054,0.433,8.9697,48.2754,0,9.7052,46.2198,0,9.7824,46.2378,0.25,9.7824,46.2378,0.25,8.9697,48.2754,0,9.0409,48.3102,0.25,9.0409,48.3102,-0.25,9.7824,46.2378,-0.25,9.7052,46.2198,0,9.7052,46.2198,0,9.0409,48.3102,-0.25,8.9697,48.2754,0,9.2353,48.4054,-0.433,9.9932,46.287,-0.433,9.7824,46.2378,-0.25,9.7824,46.2378,-0.25,9.2353,48.4054,-0.433,9.0409,48.3102,-0.25,9.5008,48.5353,-0.5,10.2812,46.3542,-0.5,9.9932,46.287,-0.433,9.9932,46.287,-0.433,9.5008,48.5353,-0.5,9.2353,48.4054,-0.433,9.7664,48.6653,-0.433,10.5692,46.4214,-0.433,10.2812,46.3542,-0.5,10.2812,46.3542,-0.5,9.7664,48.6653,-0.433,9.5008,48.5353,-0.5,9.9608,48.7604,-0.25,10.7801,46.4706,-0.25,10.5692,46.4214,-0.433,10.5692,46.4214,-0.433,9.9608,48.7604,-0.25,9.7664,48.6653,-0.433,10.0319,48.7952,0,10.8572,46.4887,0,10.7801,46.4706,-0.25,10.7801,46.4706,-0.25,10.0319,48.7952,0,9.9608,48.7604,-0.25,8.6759,50.7803,0.25,9.9608,48.7604,0.25,10.0319,48.7952,0,10.0319,48.7952,0,8.6759,50.7803,0.25,8.7377,50.8298,0,8.5071,50.6449,0.433,9.7664,48.6653,0.433,9.9608,48.7604,0.25,9.9608,48.7604,0.25,8.5071,50.6449,0.433,8.6759,50.7803,0.25,8.2765,50.46,0.5,9.5008,48.5353,0.5,9.7664,48.6653,0.433,9.7664,48.6653,0.433,8.2765,50.46,0.5,8.5071,50.6449,0.433,8.0459,50.275,0.433,9.2353,48.4054,0.433,9.5008,48.5353,0.5,9.5008,48.5353,0.5,8.0459,50.275,0.433,8.2765,50.46,0.5,7.8771,50.1396,0.25,9.0409,48.3102,0.25,9.2353,48.4054,0.433,9.2353,48.4054,0.433,7.8771,50.1396,0.25,8.0459,50.275,0.433,7.8154,50.0901,0,8.9697,48.2754,0,9.0409,48.3102,0.25,9.0409,48.3102,0.25,7.8154,50.0901,0,7.8771,50.1396,0.25,7.8771,50.1396,-0.25,9.0409,48.3102,-0.25,8.9697,48.2754,0,8.9697,48.2754,0,7.8771,50.1396,-0.25,7.8154,50.0901,0,8.0459,50.275,-0.433,9.2353,48.4054,-0.433,9.0409,48.3102,-0.25,9.0409,48.3102,-0.25,8.0459,50.275,-0.433,7.8771,50.1396,-0.25,8.2765,50.46,-0.5,9.5008,48.5353,-0.5,9.2353,48.4054,-0.433,9.2353,48.4054,-0.433,8.2765,50.46,-0.5,8.0459,50.275,-0.433,8.5071,50.6449,-0.433,9.7664,48.6653,-0.433,9.5008,48.5353,-0.5,9.5008,48.5353,-0.5,8.5071,50.6449,-0.433,8.2765,50.46,-0.5,8.6759,50.7803,-0.25,9.9608,48.7604,-0.25,9.7664,48.6653,-0.433,9.7664,48.6653,-0.433,8.6759,50.7803,-0.25,8.5071,50.6449,-0.433,8.7377,50.8298,0,10.0319,48.7952,0,9.9608,48.7604,-0.25,9.9608,48.7604,-0.25,8.7377,50.8298,0,8.6759,50.7803,-0.25,6.9911,52.465,0.25,8.6759,50.7803,0.25,8.7377,50.8298,0,8.7377,50.8298,0,6.9911,52.465,0.25,7.0407,52.5268,0,6.8557,52.2962,0.433,8.5071,50.6449,0.433,8.6759,50.7803,0.25,8.6759,50.7803,0.25,6.8557,52.2962,0.433,6.9911,52.465,0.25,6.6708,52.0657,0.5,8.2765,50.46,0.5,8.5071,50.6449,0.433,8.5071,50.6449,0.433,6.6708,52.0657,0.5,6.8557,52.2962,0.433,6.4859,51.8351,0.433,8.0459,50.275,0.433,8.2765,50.46,0.5,8.2765,50.46,0.5,6.4859,51.8351,0.433,6.6708,52.0657,0.5,6.3505,51.6663,0.25,7.8771,50.1396,0.25,8.0459,50.275,0.433,8.0459,50.275,0.433,6.3505,51.6663,0.25,6.4859,51.8351,0.433,6.3009,51.6045,0,7.8154,50.0901,0,7.8771,50.1396,0.25,7.8771,50.1396,0.25,6.3009,51.6045,0,6.3505,51.6663,0.25,6.3505,51.6663,-0.25,7.8771,50.1396,-0.25,7.8154,50.0901,0,7.8154,50.0901,0,6.3505,51.6663,-0.25,6.3009,51.6045,0,6.4859,51.8351,-0.433,8.0459,50.275,-0.433,7.8771,50.1396,-0.25,7.8771,50.1396,-0.25,6.4859,51.8351,-0.433,6.3505,51.6663,-0.25,6.6708,52.0657,-0.5,8.2765,50.46,-0.5,8.0459,50.275,-0.433,8.0459,50.275,-0.433,6.6708,52.0657,-0.5,6.4859,51.8351,-0.433,6.8557,52.2962,-0.433,8.5071,50.6449,-0.433,8.2765,50.46,-0.5,8.2765,50.46,-0.5,6.8557,52.2962,-0.433,6.6708,52.0657,-0.5,6.9911,52.465,-0.25,8.6759,50.7803,-0.25,8.5071,50.6449,-0.433,8.5071,50.6449,-0.433,6.9911,52.465,-0.25,6.8557,52.2962,-0.433,7.0407,52.5268,0,8.7377,50.8298,0,8.6759,50.7803,-0.25,8.6759,50.7803,-0.25,7.0407,52.5268,0,6.9911,52.465,-0.25,4.9712,53.7499,0.25,6.9911,52.465,0.25,7.0407,52.5268,0,7.0407,52.5268,0,4.9712,53.7499,0.25,5.006,53.8211,0,4.8761,53.5555,0.433,6.8557,52.2962,0.433,6.9911,52.465,0.25,6.9911,52.465,0.25,4.8761,53.5555,0.433,4.9712,53.7499,0.25,4.7462,53.29,0.5,6.6708,52.0657,0.5,6.8557,52.2962,0.433,6.8557,52.2962,0.433,4.7462,53.29,0.5,4.8761,53.5555,0.433,4.6162,53.0244,0.433,6.4859,51.8351,0.433,6.6708,52.0657,0.5,6.6708,52.0657,0.5,4.6162,53.0244,0.433,4.7462,53.29,0.5,4.5211,52.83,0.25,6.3505,51.6663,0.25,6.4859,51.8351,0.433,6.4859,51.8351,0.433,4.5211,52.83,0.25,4.6162,53.0244,0.433,4.4863,52.7589,0,6.3009,51.6045,0,6.3505,51.6663,0.25,6.3505,51.6663,0.25,4.4863,52.7589,0,4.5211,52.83,0.25,4.5211,52.83,-0.25,6.3505,51.6663,-0.25,6.3009,51.6045,0,6.3009,51.6045,0,4.5211,52.83,-0.25,4.4863,52.7589,0,4.6162,53.0244,-0.433,6.4859,51.8351,-0.433,6.3505,51.6663,-0.25,6.3505,51.6663,-0.25,4.6162,53.0244,-0.433,4.5211,52.83,-0.25,4.7462,53.29,-0.5,6.6708,52.0657,-0.5,6.4859,51.8351,-0.433,6.4859,51.8351,-0.433,4.7462,53.29,-0.5,4.6162,53.0244,-0.433,4.8761,53.5555,-0.433,6.8557,52.2962,-0.433,6.6708,52.0657,-0.5,6.6708,52.0657,-0.5,4.8761,53.5555,-0.433,4.7462,53.29,-0.5,4.9712,53.7499,-0.25,6.9911,52.465,-0.25,6.8557,52.2962,-0.433,6.8557,52.2962,-0.433,4.9712,53.7499,-0.25,4.8761,53.5555,-0.433,5.006,53.8211,0,7.0407,52.5268,0,6.9911,52.465,-0.25,6.9911,52.465,-0.25,5.006,53.8211,0,4.9712,53.7499,-0.25,2.6815,54.5692,0.25,4.9712,53.7499,0.25,5.006,53.8211,0,5.006,53.8211,0,2.6815,54.5692,0.25,2.6995,54.6464,0,2.6323,54.3584,0.433,4.8761,53.5555,0.433,4.9712,53.7499,0.25,4.9712,53.7499,0.25,2.6323,54.3584,0.433,2.6815,54.5692,0.25,2.5651,54.0704,0.5,4.7462,53.29,0.5,4.8761,53.5555,0.433,4.8761,53.5555,0.433,2.5651,54.0704,0.5,2.6323,54.3584,0.433,2.4979,53.7824,0.433,4.6162,53.0244,0.433,4.7462,53.29,0.5,4.7462,53.29,0.5,2.4979,53.7824,0.433,2.5651,54.0704,0.5,2.4487,53.5716,0.25,4.5211,52.83,0.25,4.6162,53.0244,0.433,4.6162,53.0244,0.433,2.4487,53.5716,0.25,2.4979,53.7824,0.433,2.4307,53.4944,0,4.4863,52.7589,0,4.5211,52.83,0.25,4.5211,52.83,0.25,2.4307,53.4944,0,2.4487,53.5716,0.25,2.4487,53.5716,-0.25,4.5211,52.83,-0.25,4.4863,52.7589,0,4.4863,52.7589,0,2.4487,53.5716,-0.25,2.4307,53.4944,0,2.4979,53.7824,-0.433,4.6162,53.0244,-0.433,4.5211,52.83,-0.25,4.5211,52.83,-0.25,2.4979,53.7824,-0.433,2.4487,53.5716,-0.25,2.5651,54.0704,-0.5,4.7462,53.29,-0.5,4.6162,53.0244,-0.433,4.6162,53.0244,-0.433,2.5651,54.0704,-0.5,2.4979,53.7824,-0.433,2.6323,54.3584,-0.433,4.8761,53.5555,-0.433,4.7462,53.29,-0.5,4.7462,53.29,-0.5,2.6323,54.3584,-0.433,2.5651,54.0704,-0.5,2.6815,54.5692,-0.25,4.9712,53.7499,-0.25,4.8761,53.5555,-0.433,4.8761,53.5555,-0.433,2.6815,54.5692,-0.25,2.6323,54.3584,-0.433,2.6995,54.6464,0,5.006,53.8211,0,4.9712,53.7499,-0.25,4.9712,53.7499,-0.25,2.6995,54.6464,0,2.6815,54.5692,-0.25,0.1901,54.8567,0.25,2.6815,54.5692,0.25,2.6995,54.6464,0,2.6995,54.6464,0,0.1901,54.8567,0.25,0.1901,54.9359,0,0.1901,54.6402,0.433,2.6323,54.3584,0.433,2.6815,54.5692,0.25,2.6815,54.5692,0.25,0.1901,54.6402,0.433,0.1901,54.8567,0.25,0.1901,54.3444,0.5,2.5651,54.0704,0.5,2.6323,54.3584,0.433,2.6323,54.3584,0.433,0.1901,54.3444,0.5,0.1901,54.6402,0.433,0.1901,54.0487,0.433,2.4979,53.7824,0.433,2.5651,54.0704,0.5,2.5651,54.0704,0.5,0.1901,54.0487,0.433,0.1901,54.3444,0.5,0.1901,53.8321,0.25,2.4487,53.5716,0.25,2.4979,53.7824,0.433,2.4979,53.7824,0.433,0.1901,53.8321,0.25,0.1901,54.0487,0.433,0.1901,53.7529,0,2.4307,53.4944,0,2.4487,53.5716,0.25,2.4487,53.5716,0.25,0.1901,53.7529,0,0.1901,53.8321,0.25,0.1901,53.8321,-0.25,2.4487,53.5716,-0.25,2.4307,53.4944,0,2.4307,53.4944,0,0.1901,53.8321,-0.25,0.1901,53.7529,0,0.1901,54.0487,-0.433,2.4979,53.7824,-0.433,2.4487,53.5716,-0.25,2.4487,53.5716,-0.25,0.1901,54.0487,-0.433,0.1901,53.8321,-0.25,0.1901,54.3444,-0.5,2.5651,54.0704,-0.5,2.4979,53.7824,-0.433,2.4979,53.7824,-0.433,0.1901,54.3444,-0.5,0.1901,54.0487,-0.433,0.1901,54.6402,-0.433,2.6323,54.3584,-0.433,2.5651,54.0704,-0.5,2.5651,54.0704,-0.5,0.1901,54.6402,-0.433,0.1901,54.3444,-0.5,0.1901,54.8567,-0.25,2.6815,54.5692,-0.25,2.6323,54.3584,-0.433,2.6323,54.3584,-0.433,0.1901,54.8567,-0.25,0.1901,54.6402,-0.433,0.1901,54.9359,0,2.6995,54.6464,0,2.6815,54.5692,-0.25,2.6815,54.5692,-0.25,0.1901,54.9359,0,0.1901,54.8567,-0.25,-2.3013,54.5692,0.25,0.1901,54.8567,0.25,0.1901,54.9359,0,0.1901,54.9359,0,-2.3013,54.5692,0.25,-2.3193,54.6464,0,-2.2521,54.3584,0.433,0.1901,54.6402,0.433,0.1901,54.8567,0.25,0.1901,54.8567,0.25,-2.2521,54.3584,0.433,-2.3013,54.5692,0.25,-2.1849,54.0704,0.5,0.1901,54.3444,0.5,0.1901,54.6402,0.433,0.1901,54.6402,0.433,-2.1849,54.0704,0.5,-2.2521,54.3584,0.433,-2.1177,53.7824,0.433,0.1901,54.0487,0.433,0.1901,54.3444,0.5,0.1901,54.3444,0.5,-2.1177,53.7824,0.433,-2.1849,54.0704,0.5,-2.0685,53.5716,0.25,0.1901,53.8321,0.25,0.1901,54.0487,0.433,0.1901,54.0487,0.433,-2.0685,53.5716,0.25,-2.1177,53.7824,0.433,-2.0505,53.4944,0,0.1901,53.7529,0,0.1901,53.8321,0.25,0.1901,53.8321,0.25,-2.0505,53.4944,0,-2.0685,53.5716,0.25,-2.0685,53.5716,-0.25,0.1901,53.8321,-0.25,0.1901,53.7529,0,0.1901,53.7529,0,-2.0685,53.5716,-0.25,-2.0505,53.4944,0,-2.1177,53.7824,-0.433,0.1901,54.0487,-0.433,0.1901,53.8321,-0.25,0.1901,53.8321,-0.25,-2.1177,53.7824,-0.433,-2.0685,53.5716,-0.25,-2.1849,54.0704,-0.5,0.1901,54.3444,-0.5,0.1901,54.0487,-0.433,0.1901,54.0487,-0.433,-2.1849,54.0704,-0.5,-2.1177,53.7824,-0.433,-2.2521,54.3584,-0.433,0.1901,54.6402,-0.433,0.1901,54.3444,-0.5,0.1901,54.3444,-0.5,-2.2521,54.3584,-0.433,-2.1849,54.0704,-0.5,-2.3013,54.5692,-0.25,0.1901,54.8567,-0.25,0.1901,54.6402,-0.433,0.1901,54.6402,-0.433,-2.3013,54.5692,-0.25,-2.2521,54.3584,-0.433,-2.3193,54.6464,0,0.1901,54.9359,0,0.1901,54.8567,-0.25,0.1901,54.8567,-0.25,-2.3193,54.6464,0,-2.3013,54.5692,-0.25,-4.5911,53.7499,0.25,-2.3013,54.5692,0.25,-2.3193,54.6464,0,-2.3193,54.6464,0,-4.5911,53.7499,0.25,-4.6259,53.8211,0,-4.496,53.5555,0.433,-2.2521,54.3584,0.433,-2.3013,54.5692,0.25,-2.3013,54.5692,0.25,-4.496,53.5555,0.433,-4.5911,53.7499,0.25,-4.366,53.29,0.5,-2.1849,54.0704,0.5,-2.2521,54.3584,0.433,-2.2521,54.3584,0.433,-4.366,53.29,0.5,-4.496,53.5555,0.433,-4.2361,53.0244,0.433,-2.1177,53.7824,0.433,-2.1849,54.0704,0.5,-2.1849,54.0704,0.5,-4.2361,53.0244,0.433,-4.366,53.29,0.5,-4.1409,52.83,0.25,-2.0685,53.5716,0.25,-2.1177,53.7824,0.433,-2.1177,53.7824,0.433,-4.1409,52.83,0.25,-4.2361,53.0244,0.433,-4.1061,52.7589,0,-2.0505,53.4944,0,-2.0685,53.5716,0.25,-2.0685,53.5716,0.25,-4.1061,52.7589,0,-4.1409,52.83,0.25,-4.1409,52.83,-0.25,-2.0685,53.5716,-0.25,-2.0505,53.4944,0,-2.0505,53.4944,0,-4.1409,52.83,-0.25,-4.1061,52.7589,0,-4.2361,53.0244,-0.433,-2.1177,53.7824,-0.433,-2.0685,53.5716,-0.25,-2.0685,53.5716,-0.25,-4.2361,53.0244,-0.433,-4.1409,52.83,-0.25,-4.366,53.29,-0.5,-2.1849,54.0704,-0.5,-2.1177,53.7824,-0.433,-2.1177,53.7824,-0.433,-4.366,53.29,-0.5,-4.2361,53.0244,-0.433,-4.496,53.5555,-0.433,-2.2521,54.3584,-0.433,-2.1849,54.0704,-0.5,-2.1849,54.0704,-0.5,-4.496,53.5555,-0.433,-4.366,53.29,-0.5,-4.5911,53.7499,-0.25,-2.3013,54.5692,-0.25,-2.2521,54.3584,-0.433,-2.2521,54.3584,-0.433,-4.5911,53.7499,-0.25,-4.496,53.5555,-0.433,-4.6259,53.8211,0,-2.3193,54.6464,0,-2.3013,54.5692,-0.25,-2.3013,54.5692,-0.25,-4.6259,53.8211,0,-4.5911,53.7499,-0.25,-6.611,52.465,0.25,-4.5911,53.7499,0.25,-4.6259,53.8211,0,-4.6259,53.8211,0,-6.611,52.465,0.25,-6.6605,52.5268,0,-6.4756,52.2962,0.433,-4.496,53.5555,0.433,-4.5911,53.7499,0.25,-4.5911,53.7499,0.25,-6.4756,52.2962,0.433,-6.611,52.465,0.25,-6.2906,52.0657,0.5,-4.366,53.29,0.5,-4.496,53.5555,0.433,-4.496,53.5555,0.433,-6.2906,52.0657,0.5,-6.4756,52.2962,0.433,-6.1057,51.8351,0.433,-4.2361,53.0244,0.433,-4.366,53.29,0.5,-4.366,53.29,0.5,-6.1057,51.8351,0.433,-6.2906,52.0657,0.5,-5.9703,51.6663,0.25,-4.1409,52.83,0.25,-4.2361,53.0244,0.433,-4.2361,53.0244,0.433,-5.9703,51.6663,0.25,-6.1057,51.8351,0.433,-5.9208,51.6045,0,-4.1061,52.7589,0,-4.1409,52.83,0.25,-4.1409,52.83,0.25,-5.9208,51.6045,0,-5.9703,51.6663,0.25,-5.9703,51.6663,-0.25,-4.1409,52.83,-0.25,-4.1061,52.7589,0,-4.1061,52.7589,0,-5.9703,51.6663,-0.25,-5.9208,51.6045,0,-6.1057,51.8351,-0.433,-4.2361,53.0244,-0.433,-4.1409,52.83,-0.25,-4.1409,52.83,-0.25,-6.1057,51.8351,-0.433,-5.9703,51.6663,-0.25,-6.2906,52.0657,-0.5,-4.366,53.29,-0.5,-4.2361,53.0244,-0.433,-4.2361,53.0244,-0.433,-6.2906,52.0657,-0.5,-6.1057,51.8351,-0.433,-6.4756,52.2962,-0.433,-4.496,53.5555,-0.433,-4.366,53.29,-0.5,-4.366,53.29,-0.5,-6.4756,52.2962,-0.433,-6.2906,52.0657,-0.5,-6.611,52.465,-0.25,-4.5911,53.7499,-0.25,-4.496,53.5555,-0.433,-4.496,53.5555,-0.433,-6.611,52.465,-0.25,-6.4756,52.2962,-0.433,-6.6605,52.5268,0,-4.6259,53.8211,0,-4.5911,53.7499,-0.25,-4.5911,53.7499,-0.25,-6.6605,52.5268,0,-6.611,52.465,-0.25,-8.2957,50.7803,0.25,-6.611,52.465,0.25,-6.6605,52.5268,0,-6.6605,52.5268,0,-8.2957,50.7803,0.25,-8.3575,50.8298,0,-8.1269,50.6449,0.433,-6.4756,52.2962,0.433,-6.611,52.465,0.25,-6.611,52.465,0.25,-8.1269,50.6449,0.433,-8.2957,50.7803,0.25,-7.8964,50.4599,0.5,-6.2906,52.0657,0.5,-6.4756,52.2962,0.433,-6.4756,52.2962,0.433,-7.8964,50.4599,0.5,-8.1269,50.6449,0.433,-7.6658,50.275,0.433,-6.1057,51.8351,0.433,-6.2906,52.0657,0.5,-6.2906,52.0657,0.5,-7.6658,50.275,0.433,-7.8964,50.4599,0.5,-7.497,50.1396,0.25,-5.9703,51.6663,0.25,-6.1057,51.8351,0.433,-6.1057,51.8351,0.433,-7.497,50.1396,0.25,-7.6658,50.275,0.433,-7.4352,50.0901,0,-5.9208,51.6045,0,-5.9703,51.6663,0.25,-5.9703,51.6663,0.25,-7.4352,50.0901,0,-7.497,50.1396,0.25,-7.497,50.1396,-0.25,-5.9703,51.6663,-0.25,-5.9208,51.6045,0,-5.9208,51.6045,0,-7.497,50.1396,-0.25,-7.4352,50.0901,0,-7.6658,50.275,-0.433,-6.1057,51.8351,-0.433,-5.9703,51.6663,-0.25,-5.9703,51.6663,-0.25,-7.6658,50.275,-0.433,-7.497,50.1396,-0.25,-7.8964,50.4599,-0.5,-6.2906,52.0657,-0.5,-6.1057,51.8351,-0.433,-6.1057,51.8351,-0.433,-7.8964,50.4599,-0.5,-7.6658,50.275,-0.433,-8.1269,50.6449,-0.433,-6.4756,52.2962,-0.433,-6.2906,52.0657,-0.5,-6.2906,52.0657,-0.5,-8.1269,50.6449,-0.433,-7.8964,50.4599,-0.5,-8.2957,50.7803,-0.25,-6.611,52.465,-0.25,-6.4756,52.2962,-0.433,-6.4756,52.2962,-0.433,-8.2957,50.7803,-0.25,-8.1269,50.6449,-0.433,-8.3575,50.8298,0,-6.6605,52.5268,0,-6.611,52.465,-0.25,-6.611,52.465,-0.25,-8.3575,50.8298,0,-8.2957,50.7803,-0.25,-9.5806,48.7604,0.25,-8.2957,50.7803,0.25,-8.3575,50.8298,0,-8.3575,50.8298,0,-9.5806,48.7604,0.25,-9.6518,48.7952,0,-9.3862,48.6653,0.433,-8.1269,50.6449,0.433,-8.2957,50.7803,0.25,-8.2957,50.7803,0.25,-9.3862,48.6653,0.433,-9.5806,48.7604,0.25,-9.1207,48.5353,0.5,-7.8964,50.4599,0.5,-8.1269,50.6449,0.433,-8.1269,50.6449,0.433,-9.1207,48.5353,0.5,-9.3862,48.6653,0.433,-8.8551,48.4054,0.433,-7.6658,50.275,0.433,-7.8964,50.4599,0.5,-7.8964,50.4599,0.5,-8.8551,48.4054,0.433,-9.1207,48.5353,0.5,-8.6607,48.3102,0.25,-7.497,50.1396,0.25,-7.6658,50.275,0.433,-7.6658,50.275,0.433,-8.6607,48.3102,0.25,-8.8551,48.4054,0.433,-8.5896,48.2754,0,-7.4352,50.0901,0,-7.497,50.1396,0.25,-7.497,50.1396,0.25,-8.5896,48.2754,0,-8.6607,48.3102,0.25,-8.6607,48.3102,-0.25,-7.497,50.1396,-0.25,-7.4352,50.0901,0,-7.4352,50.0901,0,-8.6607,48.3102,-0.25,-8.5896,48.2754,0,-8.8551,48.4054,-0.433,-7.6658,50.275,-0.433,-7.497,50.1396,-0.25,-7.497,50.1396,-0.25,-8.8551,48.4054,-0.433,-8.6607,48.3102,-0.25,-9.1207,48.5353,-0.5,-7.8964,50.4599,-0.5,-7.6658,50.275,-0.433,-7.6658,50.275,-0.433,-9.1207,48.5353,-0.5,-8.8551,48.4054,-0.433,-9.3862,48.6653,-0.433,-8.1269,50.6449,-0.433,-7.8964,50.4599,-0.5,-7.8964,50.4599,-0.5,-9.3862,48.6653,-0.433,-9.1207,48.5353,-0.5,-9.5806,48.7604,-0.25,-8.2957,50.7803,-0.25,-8.1269,50.6449,-0.433,-8.1269,50.6449,-0.433,-9.5806,48.7604,-0.25,-9.3862,48.6653,-0.433,-9.6518,48.7952,0,-8.3575,50.8298,0,-8.2957,50.7803,-0.25,-8.2957,50.7803,-0.25,-9.6518,48.7952,0,-9.5806,48.7604,-0.25,-10.3999,46.4706,0.25,-9.5806,48.7604,0.25,-9.6518,48.7952,0,-9.6518,48.7952,0,-10.3999,46.4706,0.25,-10.4771,46.4886,0,-10.1891,46.4214,0.433,-9.3862,48.6653,0.433,-9.5806,48.7604,0.25,-9.5806,48.7604,0.25,-10.1891,46.4214,0.433,-10.3999,46.4706,0.25,-9.9011,46.3542,0.5,-9.1207,48.5353,0.5,-9.3862,48.6653,0.433,-9.3862,48.6653,0.433,-9.9011,46.3542,0.5,-10.1891,46.4214,0.433,-9.6131,46.287,0.433,-8.8551,48.4054,0.433,-9.1207,48.5353,0.5,-9.1207,48.5353,0.5,-9.6131,46.287,0.433,-9.9011,46.3542,0.5,-9.4022,46.2378,0.25,-8.6607,48.3102,0.25,-8.8551,48.4054,0.433,-8.8551,48.4054,0.433,-9.4022,46.2378,0.25,-9.6131,46.287,0.433,-9.3251,46.2198,0,-8.5896,48.2754,0,-8.6607,48.3102,0.25,-8.6607,48.3102,0.25,-9.3251,46.2198,0,-9.4022,46.2378,0.25,-9.4022,46.2378,-0.25,-8.6607,48.3102,-0.25,-8.5896,48.2754,0,-8.5896,48.2754,0,-9.4022,46.2378,-0.25,-9.3251,46.2198,0,-9.6131,46.287,-0.433,-8.8551,48.4054,-0.433,-8.6607,48.3102,-0.25,-8.6607,48.3102,-0.25,-9.6131,46.287,-0.433,-9.4022,46.2378,-0.25,-9.9011,46.3542,-0.5,-9.1207,48.5353,-0.5,-8.8551,48.4054,-0.433,-8.8551,48.4054,-0.433,-9.9011,46.3542,-0.5,-9.6131,46.287,-0.433,-10.1891,46.4214,-0.433,-9.3862,48.6653,-0.433,-9.1207,48.5353,-0.5,-9.1207,48.5353,-0.5,-10.1891,46.4214,-0.433,-9.9011,46.3542,-0.5,-10.3999,46.4706,-0.25,-9.5806,48.7604,-0.25,-9.3862,48.6653,-0.433,-9.3862,48.6653,-0.433,-10.3999,46.4706,-0.25,-10.1891,46.4214,-0.433,-10.4771,46.4886,0,-9.6518,48.7952,0,-9.5806,48.7604,-0.25,-9.5806,48.7604,-0.25,-10.4771,46.4886,0,-10.3999,46.4706,-0.25,-10.6874,43.9792,0.25,-10.3999,46.4706,0.25,-10.4771,46.4886,0,-10.4771,46.4886,0,-10.6874,43.9792,0.25,-10.7666,43.9792,0,-10.4709,43.9792,0.433,-10.1891,46.4214,0.433,-10.3999,46.4706,0.25,-10.3999,46.4706,0.25,-10.4709,43.9792,0.433,-10.6874,43.9792,0.25,-10.1751,43.9792,0.5,-9.9011,46.3542,0.5,-10.1891,46.4214,0.433,-10.1891,46.4214,0.433,-10.1751,43.9792,0.5,-10.4709,43.9792,0.433,-9.8793,43.9792,0.433,-9.6131,46.287,0.433,-9.9011,46.3542,0.5,-9.9011,46.3542,0.5,-9.8793,43.9792,0.433,-10.1751,43.9792,0.5,-9.6628,43.9792,0.25,-9.4022,46.2378,0.25,-9.6131,46.287,0.433,-9.6131,46.287,0.433,-9.6628,43.9792,0.25,-9.8793,43.9792,0.433,-9.5836,43.9792,0,-9.3251,46.2198,0,-9.4022,46.2378,0.25,-9.4022,46.2378,0.25,-9.5836,43.9792,0,-9.6628,43.9792,0.25,-9.6628,43.9792,-0.25,-9.4022,46.2378,-0.25,-9.3251,46.2198,0,-9.3251,46.2198,0,-9.6628,43.9792,-0.25,-9.5836,43.9792,0,-9.8793,43.9792,-0.433,-9.6131,46.287,-0.433,-9.4022,46.2378,-0.25,-9.4022,46.2378,-0.25,-9.8793,43.9792,-0.433,-9.6628,43.9792,-0.25,-10.1751,43.9792,-0.5,-9.9011,46.3542,-0.5,-9.6131,46.287,-0.433,-9.6131,46.287,-0.433,-10.1751,43.9792,-0.5,-9.8793,43.9792,-0.433,-10.4709,43.9792,-0.433,-10.1891,46.4214,-0.433,-9.9011,46.3542,-0.5,-9.9011,46.3542,-0.5,-10.4709,43.9792,-0.433,-10.1751,43.9792,-0.5,-10.6874,43.9792,-0.25,-10.3999,46.4706,-0.25,-10.1891,46.4214,-0.433,-10.1891,46.4214,-0.433,-10.6874,43.9792,-0.25,-10.4709,43.9792,-0.433,-10.7666,43.9792,0,-10.4771,46.4886,0,-10.3999,46.4706,-0.25,-10.3999,46.4706,-0.25,-10.7666,43.9792,0,-10.6874,43.9792,-0.25] )
   , make_vbo(NORMAL, [-0.3443,0.6085,-0.715,-0.3853,0.4599,-0.8,0,0.4599,-0.888,0,0.4599,-0.888,-0.3443,0.6085,-0.715,0,0.6085,-0.7935,-0.6204,0.6085,-0.4948,-0.6942,0.4599,-0.5536,-0.3853,0.4599,-0.8,-0.3853,0.4599,-0.8,-0.6204,0.6085,-0.4948,-0.3443,0.6085,-0.715,-0.7737,0.6085,-0.1766,-0.8657,0.4599,-0.1976,-0.6942,0.4599,-0.5536,-0.6942,0.4599,-0.5536,-0.7737,0.6085,-0.1766,-0.6204,0.6085,-0.4948,-0.7737,0.6085,0.1766,-0.8657,0.4599,0.1976,-0.8657,0.4599,-0.1976,-0.8657,0.4599,-0.1976,-0.7737,0.6085,0.1766,-0.7737,0.6085,-0.1766,-0.6204,0.6085,0.4948,-0.6942,0.4599,0.5536,-0.8657,0.4599,0.1976,-0.8657,0.4599,0.1976,-0.6204,0.6085,0.4948,-0.7737,0.6085,0.1766,-0.3443,0.6085,0.715,-0.3853,0.4599,0.8,-0.6942,0.4599,0.5536,-0.6942,0.4599,0.5536,-0.3443,0.6085,0.715,-0.6204,0.6085,0.4948,0,0.6085,0.7935,0,0.4599,0.888,-0.3853,0.4599,0.8,-0.3853,0.4599,0.8,0,0.6085,0.7935,-0.3443,0.6085,0.715,0.3443,0.6085,0.715,0.3853,0.4599,0.8,0,0.4599,0.888,0,0.4599,0.888,0.3443,0.6085,0.715,0,0.6085,0.7935,0.6204,0.6085,0.4948,0.6942,0.4599,0.5536,0.3853,0.4599,0.8,0.3853,0.4599,0.8,0.6204,0.6085,0.4948,0.3443,0.6085,0.715,0.7737,0.6085,0.1766,0.8657,0.4599,0.1976,0.6942,0.4599,0.5536,0.6942,0.4599,0.5536,0.7737,0.6085,0.1766,0.6204,0.6085,0.4948,0.7737,0.6085,-0.1766,0.8657,0.4599,-0.1976,0.8657,0.4599,0.1976,0.8657,0.4599,0.1976,0.7737,0.6085,-0.1766,0.7737,0.6085,0.1766,0.6204,0.6085,-0.4948,0.6942,0.4599,-0.5536,0.8657,0.4599,-0.1976,0.8657,0.4599,-0.1976,0.6204,0.6085,-0.4948,0.7737,0.6085,-0.1766,0.3443,0.6085,-0.715,0.3853,0.4599,-0.8,0.6942,0.4599,-0.5536,0.6942,0.4599,-0.5536,0.3443,0.6085,-0.715,0.6204,0.6085,-0.4948,0,0.6085,-0.7935,0,0.4599,-0.888,0.3853,0.4599,-0.8,0.3853,0.4599,-0.8,0,0.6085,-0.7935,0.3443,0.6085,-0.715,-0.4236,0.216,-0.8797,-0.3443,0.6085,-0.715,0,0.6085,-0.7935,0,0.6085,-0.7935,-0.4236,0.216,-0.8797,0,0.216,-0.9764,-0.7634,0.216,-0.6088,-0.6204,0.6085,-0.4948,-0.3443,0.6085,-0.715,-0.3443,0.6085,-0.715,-0.7634,0.216,-0.6088,-0.4236,0.216,-0.8797,-0.9519,0.216,-0.2173,-0.7737,0.6085,-0.1766,-0.6204,0.6085,-0.4948,-0.6204,0.6085,-0.4948,-0.9519,0.216,-0.2173,-0.7634,0.216,-0.6088,-0.9519,0.216,0.2173,-0.7737,0.6085,0.1766,-0.7737,0.6085,-0.1766,-0.7737,0.6085,-0.1766,-0.9519,0.216,0.2173,-0.9519,0.216,-0.2173,-0.7634,0.216,0.6088,-0.6204,0.6085,0.4948,-0.7737,0.6085,0.1766,-0.7737,0.6085,0.1766,-0.7634,0.216,0.6088,-0.9519,0.216,0.2173,-0.4236,0.216,0.8797,-0.3443,0.6085,0.715,-0.6204,0.6085,0.4948,-0.6204,0.6085,0.4948,-0.4236,0.216,0.8797,-0.7634,0.216,0.6088,0,0.216,0.9764,0,0.6085,0.7935,-0.3443,0.6085,0.715,-0.3443,0.6085,0.715,0,0.216,0.9764,-0.4236,0.216,0.8797,0.4236,0.216,0.8797,0.3443,0.6085,0.715,0,0.6085,0.7935,0,0.6085,0.7935,0.4236,0.216,0.8797,0,0.216,0.9764,0.7634,0.216,0.6088,0.6204,0.6085,0.4948,0.3443,0.6085,0.715,0.3443,0.6085,0.715,0.7634,0.216,0.6088,0.4236,0.216,0.8797,0.9519,0.216,0.2173,0.7737,0.6085,0.1766,0.6204,0.6085,0.4948,0.6204,0.6085,0.4948,0.9519,0.216,0.2173,0.7634,0.216,0.6088,0.9519,0.216,-0.2173,0.7737,0.6085,-0.1766,0.7737,0.6085,0.1766,0.7737,0.6085,0.1766,0.9519,0.216,-0.2173,0.9519,0.216,0.2173,0.7634,0.216,-0.6088,0.6204,0.6085,-0.4948,0.7737,0.6085,-0.1766,0.7737,0.6085,-0.1766,0.7634,0.216,-0.6088,0.9519,0.216,-0.2173,0.4236,0.216,-0.8797,0.3443,0.6085,-0.715,0.6204,0.6085,-0.4948,0.6204,0.6085,-0.4948,0.4236,0.216,-0.8797,0.7634,0.216,-0.6088,0,0.216,-0.9764,0,0.6085,-0.7935,0.3443,0.6085,-0.715,0.3443,0.6085,-0.715,0,0.216,-0.9764,0.4236,0.216,-0.8797,-0.4236,-0.216,-0.8797,-0.4236,0.216,-0.8797,0,0.216,-0.9764,0,0.216,-0.9764,-0.4236,-0.216,-0.8797,0,-0.216,-0.9764,-0.7634,-0.216,-0.6088,-0.7634,0.216,-0.6088,-0.4236,0.216,-0.8797,-0.4236,0.216,-0.8797,-0.7634,-0.216,-0.6088,-0.4236,-0.216,-0.8797,-0.9519,-0.216,-0.2173,-0.9519,0.216,-0.2173,-0.7634,0.216,-0.6088,-0.7634,0.216,-0.6088,-0.9519,-0.216,-0.2173,-0.7634,-0.216,-0.6088,-0.9519,-0.216,0.2173,-0.9519,0.216,0.2173,-0.9519,0.216,-0.2173,-0.9519,0.216,-0.2173,-0.9519,-0.216,0.2173,-0.9519,-0.216,-0.2173,-0.7634,-0.216,0.6088,-0.7634,0.216,0.6088,-0.9519,0.216,0.2173,-0.9519,0.216,0.2173,-0.7634,-0.216,0.6088,-0.9519,-0.216,0.2173,-0.4236,-0.216,0.8797,-0.4236,0.216,0.8797,-0.7634,0.216,0.6088,-0.7634,0.216,0.6088,-0.4236,-0.216,0.8797,-0.7634,-0.216,0.6088,0,-0.216,0.9764,0,0.216,0.9764,-0.4236,0.216,0.8797,-0.4236,0.216,0.8797,0,-0.216,0.9764,-0.4236,-0.216,0.8797,0.4236,-0.216,0.8797,0.4236,0.216,0.8797,0,0.216,0.9764,0,0.216,0.9764,0.4236,-0.216,0.8797,0,-0.216,0.9764,0.7634,-0.216,0.6088,0.7634,0.216,0.6088,0.4236,0.216,0.8797,0.4236,0.216,0.8797,0.7634,-0.216,0.6088,0.4236,-0.216,0.8797,0.9519,-0.216,0.2173,0.9519,0.216,0.2173,0.7634,0.216,0.6088,0.7634,0.216,0.6088,0.9519,-0.216,0.2173,0.7634,-0.216,0.6088,0.9519,-0.216,-0.2173,0.9519,0.216,-0.2173,0.9519,0.216,0.2173,0.9519,0.216,0.2173,0.9519,-0.216,-0.2173,0.9519,-0.216,0.2173,0.7634,-0.216,-0.6088,0.7634,0.216,-0.6088,0.9519,0.216,-0.2173,0.9519,0.216,-0.2173,0.7634,-0.216,-0.6088,0.9519,-0.216,-0.2173,0.4236,-0.216,-0.8797,0.4236,0.216,-0.8797,0.7634,0.216,-0.6088,0.7634,0.216,-0.6088,0.4236,-0.216,-0.8797,0.7634,-0.216,-0.6088,0,-0.216,-0.9764,0,0.216,-0.9764,0.4236,0.216,-0.8797,0.4236,0.216,-0.8797,0,-0.216,-0.9764,0.4236,-0.216,-0.8797,-0.3443,-0.6085,-0.715,-0.4236,-0.216,-0.8797,0,-0.216,-0.9764,0,-0.216,-0.9764,-0.3443,-0.6085,-0.715,0,-0.6085,-0.7935,-0.6204,-0.6085,-0.4948,-0.7634,-0.216,-0.6088,-0.4236,-0.216,-0.8797,-0.4236,-0.216,-0.8797,-0.6204,-0.6085,-0.4948,-0.3443,-0.6085,-0.715,-0.7737,-0.6085,-0.1766,-0.9519,-0.216,-0.2173,-0.7634,-0.216,-0.6088,-0.7634,-0.216,-0.6088,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,-0.4948,-0.7737,-0.6085,0.1766,-0.9519,-0.216,0.2173,-0.9519,-0.216,-0.2173,-0.9519,-0.216,-0.2173,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,0.4948,-0.7634,-0.216,0.6088,-0.9519,-0.216,0.2173,-0.9519,-0.216,0.2173,-0.6204,-0.6085,0.4948,-0.7737,-0.6085,0.1766,-0.3443,-0.6085,0.715,-0.4236,-0.216,0.8797,-0.7634,-0.216,0.6088,-0.7634,-0.216,0.6088,-0.3443,-0.6085,0.715,-0.6204,-0.6085,0.4948,0,-0.6085,0.7935,0,-0.216,0.9764,-0.4236,-0.216,0.8797,-0.4236,-0.216,0.8797,0,-0.6085,0.7935,-0.3443,-0.6085,0.715,0.3443,-0.6085,0.715,0.4236,-0.216,0.8797,0,-0.216,0.9764,0,-0.216,0.9764,0.3443,-0.6085,0.715,0,-0.6085,0.7935,0.6204,-0.6085,0.4948,0.7634,-0.216,0.6088,0.4236,-0.216,0.8797,0.4236,-0.216,0.8797,0.6204,-0.6085,0.4948,0.3443,-0.6085,0.715,0.7737,-0.6085,0.1766,0.9519,-0.216,0.2173,0.7634,-0.216,0.6088,0.7634,-0.216,0.6088,0.7737,-0.6085,0.1766,0.6204,-0.6085,0.4948,0.7737,-0.6085,-0.1766,0.9519,-0.216,-0.2173,0.9519,-0.216,0.2173,0.9519,-0.216,0.2173,0.7737,-0.6085,-0.1766,0.7737,-0.6085,0.1766,0.6204,-0.6085,-0.4948,0.7634,-0.216,-0.6088,0.9519,-0.216,-0.2173,0.9519,-0.216,-0.2173,0.6204,-0.6085,-0.4948,0.7737,-0.6085,-0.1766,0.3443,-0.6085,-0.715,0.4236,-0.216,-0.8797,0.7634,-0.216,-0.6088,0.7634,-0.216,-0.6088,0.3443,-0.6085,-0.715,0.6204,-0.6085,-0.4948,0,-0.6085,-0.7935,0,-0.216,-0.9764,0.4236,-0.216,-0.8797,0.4236,-0.216,-0.8797,0,-0.6085,-0.7935,0.3443,-0.6085,-0.715,-0.3853,-0.4599,-0.8,-0.3443,-0.6085,-0.715,0,-0.6085,-0.7935,0,-0.6085,-0.7935,-0.3853,-0.4599,-0.8,0,-0.4599,-0.888,-0.6942,-0.4599,-0.5536,-0.6204,-0.6085,-0.4948,-0.3443,-0.6085,-0.715,-0.3443,-0.6085,-0.715,-0.6942,-0.4599,-0.5536,-0.3853,-0.4599,-0.8,-0.8657,-0.4599,-0.1976,-0.7737,-0.6085,-0.1766,-0.6204,-0.6085,-0.4948,-0.6204,-0.6085,-0.4948,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,-0.5536,-0.8657,-0.4599,0.1976,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,-0.1766,-0.7737,-0.6085,-0.1766,-0.8657,-0.4599,0.1976,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,0.5536,-0.6204,-0.6085,0.4948,-0.7737,-0.6085,0.1766,-0.7737,-0.6085,0.1766,-0.6942,-0.4599,0.5536,-0.8657,-0.4599,0.1976,-0.3853,-0.4599,0.8,-0.3443,-0.6085,0.715,-0.6204,-0.6085,0.4948,-0.6204,-0.6085,0.4948,-0.3853,-0.4599,0.8,-0.6942,-0.4599,0.5536,0,-0.4599,0.888,0,-0.6085,0.7935,-0.3443,-0.6085,0.715,-0.3443,-0.6085,0.715,0,-0.4599,0.888,-0.3853,-0.4599,0.8,0.3853,-0.4599,0.8,0.3443,-0.6085,0.715,0,-0.6085,0.7935,0,-0.6085,0.7935,0.3853,-0.4599,0.8,0,-0.4599,0.888,0.6942,-0.4599,0.5536,0.6204,-0.6085,0.4948,0.3443,-0.6085,0.715,0.3443,-0.6085,0.715,0.6942,-0.4599,0.5536,0.3853,-0.4599,0.8,0.767,-0.5895,0.2533,0.7737,-0.6085,0.1766,0.6204,-0.6085,0.4948,0.6204,-0.6085,0.4948,0.767,-0.5895,0.2533,0.6942,-0.4599,0.5536,0.767,-0.5895,-0.2533,0.7737,-0.6085,-0.1766,0.7737,-0.6085,0.1766,0.7737,-0.6085,0.1766,0.767,-0.5895,-0.2533,0.767,-0.5895,0.2533,0.6942,-0.4599,-0.5536,0.6204,-0.6085,-0.4948,0.7737,-0.6085,-0.1766,0.7737,-0.6085,-0.1766,0.6942,-0.4599,-0.5536,0.767,-0.5895,-0.2533,0.3853,-0.4599,-0.8,0.3443,-0.6085,-0.715,0.6204,-0.6085,-0.4948,0.6204,-0.6085,-0.4948,0.3853,-0.4599,-0.8,0.6942,-0.4599,-0.5536,0,-0.4599,-0.888,0,-0.6085,-0.7935,0.3443,-0.6085,-0.715,0.3443,-0.6085,-0.715,0,-0.4599,-0.888,0.3853,-0.4599,-0.8,0,-1,0,0.0005,-1,0.001,0,-1,0.0011,0,-1,0,0.0008,-1,0.0007,0.0005,-1,0.001,0,-1,0,0.001,-1,0.0002,0.0008,-1,0.0007,0,-1,0,0.001,-1,-0.0002,0.001,-1,0.0002,0,-1,0,0.0008,-1,-0.0007,0.001,-1,-0.0002,0,-1,0,0.0005,-1,-0.001,0.0008,-1,-0.0007,0,-1,0,0,-1,-0.0011,0.0005,-1,-0.001,0,-1,0,-0.0005,-1,-0.001,0,-1,-0.0011,0,-1,0,-0.0008,-1,-0.0007,-0.0005,-1,-0.001,0,-1,0,-0.001,-1,-0.0002,-0.0008,-1,-0.0007,0,-1,0,-0.001,-1,0.0002,-0.001,-1,-0.0002,0,-1,0,-0.0008,-1,0.0007,-0.001,-1,0.0002,0,-1,0,-0.0005,-1,0.001,-0.0008,-1,0.0007,0,-1,0,0,-1,0.0011,-0.0005,-1,0.001,0,0.4599,-0.888,-0.3853,0.4599,-0.8,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0.4599,-0.888,0,0,-1,-0.3853,0.4599,-0.8,-0.6942,0.4599,-0.5536,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.3853,0.4599,-0.8,-0.4339,0,-0.901,-0.6942,0.4599,-0.5536,-0.8657,0.4599,-0.1976,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.6942,0.4599,-0.5536,-0.7818,0,-0.6235,-0.8657,0.4599,-0.1976,-0.8657,0.4599,0.1976,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.8657,0.4599,-0.1976,-0.9749,0,-0.2225,-0.8657,0.4599,0.1976,-0.6942,0.4599,0.5536,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.8657,0.4599,0.1976,-0.9749,0,0.2225,-0.6942,0.4599,0.5536,-0.3853,0.4599,0.8,-0.4339,0,0.901,-0.4339,0,0.901,-0.6942,0.4599,0.5536,-0.7818,0,0.6235,-0.3853,0.4599,0.8,0,0.4599,0.888,0,0,1,0,0,1,-0.3853,0.4599,0.8,-0.4339,0,0.901,0,0.4599,0.888,0.3853,0.4599,0.8,0.4339,0,0.901,0.4339,0,0.901,0,0.4599,0.888,0,0,1,0.3853,0.4599,0.8,0.6942,0.4599,0.5536,0.7818,0,0.6235,0.7818,0,0.6235,0.3853,0.4599,0.8,0.4339,0,0.901,0.6942,0.4599,0.5536,0.8657,0.4599,0.1976,0.9749,0,0.2225,0.9749,0,0.2225,0.6942,0.4599,0.5536,0.7818,0,0.6235,0.8657,0.4599,0.1976,0.8657,0.4599,-0.1976,0.9749,0,-0.2225,0.9749,0,-0.2225,0.8657,0.4599,0.1976,0.9749,0,0.2225,0.8657,0.4599,-0.1976,0.6942,0.4599,-0.5536,0.7818,0,-0.6235,0.7818,0,-0.6235,0.8657,0.4599,-0.1976,0.9749,0,-0.2225,0.6942,0.4599,-0.5536,0.3853,0.4599,-0.8,0.4339,0,-0.901,0.4339,0,-0.901,0.6942,0.4599,-0.5536,0.7818,0,-0.6235,0.3853,0.4599,-0.8,0,0.4599,-0.888,0,0,-1,0,0,-1,0.3853,0.4599,-0.8,0.4339,0,-0.901,-0.3853,-0.4599,-0.8,0,-0.4599,-0.888,0,0,-1,0,0,-1,-0.3853,-0.4599,-0.8,-0.4339,0,-0.901,0,-0.4599,-0.888,0.3853,-0.4599,-0.8,0.4339,0,-0.901,0.4339,0,-0.901,0,-0.4599,-0.888,0,0,-1,0.3853,-0.4599,-0.8,0.6942,-0.4599,-0.5536,0.7818,0,-0.6235,0.7818,0,-0.6235,0.3853,-0.4599,-0.8,0.4339,0,-0.901,0.6942,-0.4599,-0.5536,0.767,-0.5895,-0.2533,0.901,0,-0.4339,0.901,0,-0.4339,0.6942,-0.4599,-0.5536,0.7818,0,-0.6235,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0.767,-0.5895,0.2533,0.6942,-0.4599,0.5536,0.7818,0,0.6235,0.7818,0,0.6235,0.767,-0.5895,0.2533,0.901,0,0.4339,0.6942,-0.4599,0.5536,0.3853,-0.4599,0.8,0.4339,0,0.901,0.4339,0,0.901,0.6942,-0.4599,0.5536,0.7818,0,0.6235,0.3853,-0.4599,0.8,0,-0.4599,0.888,0,0,1,0,0,1,0.3853,-0.4599,0.8,0.4339,0,0.901,0,-0.4599,0.888,-0.3853,-0.4599,0.8,-0.4339,0,0.901,-0.4339,0,0.901,0,-0.4599,0.888,0,0,1,-0.3853,-0.4599,0.8,-0.6942,-0.4599,0.5536,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.3853,-0.4599,0.8,-0.4339,0,0.901,-0.6942,-0.4599,0.5536,-0.8657,-0.4599,0.1976,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.6942,-0.4599,0.5536,-0.7818,0,0.6235,-0.8657,-0.4599,0.1976,-0.8657,-0.4599,-0.1976,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.8657,-0.4599,0.1976,-0.9749,0,0.2225,-0.8657,-0.4599,-0.1976,-0.6942,-0.4599,-0.5536,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.8657,-0.4599,-0.1976,-0.9749,0,-0.2225,-0.6942,-0.4599,-0.5536,-0.3853,-0.4599,-0.8,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.6942,-0.4599,-0.5536,-0.7818,0,-0.6235,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,0,0,1,0,0,1,0,0,1,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0.4339,0,0.901,0,0,1,0,0,1,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0,0,-1,0,0,-1,0,0,-1,0.4339,0,-0.901,0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0,0,-1,0,0,-1,0.4339,0,-0.901,0.7818,0,-0.6235,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.4339,0,0.901,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0,0,1,0,0,1,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0,0,1,-0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0,0,1,-0.4339,0,0.901,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,0,1,0,0,1,0.4339,0,0.901,0.4339,0,0.901,0,0,1,0.4339,0,0.901,-0.4339,0,0.901,-0.4339,0,0.901,0,0,1,0,0,1,-0.4339,0,0.901,0,0,1,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.4339,0,0.901,-0.4339,0,0.901,-0.7818,0,0.6235,-0.4339,0,0.901,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.7818,0,0.6235,-0.9749,0,0.2225,-0.7818,0,0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.9749,0,0.2225,-0.9749,0,-0.2225,-0.9749,0,0.2225,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.9749,0,-0.2225,-0.7818,0,-0.6235,-0.9749,0,-0.2225,-0.4339,0,-0.901,-0.4339,0,-0.901,-0.7818,0,-0.6235,-0.7818,0,-0.6235,-0.4339,0,-0.901,-0.7818,0,-0.6235,0,0,-1,0,0,-1,-0.4339,0,-0.901,-0.4339,0,-0.901,0,0,-1,-0.4339,0,-0.901,0.4339,0,-0.901,0.4339,0,-0.901,0,0,-1,0,0,-1,0.4339,0,-0.901,0,0,-1,0.7818,0,-0.6235,0.7818,0,-0.6235,0.4339,0,-0.901,0.4339,0,-0.901,0.7818,0,-0.6235,0.4339,0,-0.901,0.9749,0,-0.2225,0.9749,0,-0.2225,0.7818,0,-0.6235,0.7818,0,-0.6235,0.9749,0,-0.2225,0.7818,0,-0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.9749,0,-0.2225,0.9749,0,0.2225,0.9749,0,-0.2225,0.7818,0,0.6235,0.7818,0,0.6235,0.9749,0,0.2225,0.9749,0,0.2225,0.7818,0,0.6235,0.9749,0,0.2225,0.4339,0,0.901,0.4339,0,0.901,0.7818,0,0.6235,0.7818,0,0.6235,0.4339,0,0.901,0.7818,0,0.6235,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0.8467,0.1976,0.4941,0.8651,0.0998,0.4916,0.9934,0.1146,0,0.9934,0.1146,0,0.8467,0.1976,0.4941,0.9738,0.2272,0,0.4983,0.1163,0.8591,0.5109,0.0589,0.8576,0.8651,0.0998,0.4916,0.8651,0.0998,0.4916,0.4983,0.1163,0.8591,0.8467,0.1976,0.4941,0.0182,0.0043,0.9998,0.0188,0.0022,0.9998,0.5109,0.0589,0.8576,0.5109,0.0589,0.8576,0.0182,0.0043,0.9998,0.4983,0.1163,0.8591,-0.4708,-0.1099,0.8754,-0.4827,-0.0557,0.874,0.0188,0.0022,0.9998,0.0188,0.0022,0.9998,-0.4708,-0.1099,0.8754,0.0182,0.0043,0.9998,-0.8374,-0.1954,0.5105,-0.8556,-0.0987,0.5081,-0.4827,-0.0557,0.874,-0.4827,-0.0557,0.874,-0.8374,-0.1954,0.5105,-0.4708,-0.1099,0.8754,-0.9738,-0.2272,0,-0.9934,-0.1146,0,-0.8556,-0.0987,0.5081,-0.8556,-0.0987,0.5081,-0.9738,-0.2272,0,-0.8374,-0.1954,0.5105,-0.8374,-0.1954,-0.5105,-0.8556,-0.0987,-0.5081,-0.9934,-0.1146,0,-0.9934,-0.1146,0,-0.8374,-0.1954,-0.5105,-0.9738,-0.2272,0,-0.4708,-0.1099,-0.8754,-0.4827,-0.0557,-0.874,-0.8556,-0.0987,-0.5081,-0.8556,-0.0987,-0.5081,-0.4708,-0.1099,-0.8754,-0.8374,-0.1954,-0.5105,0.0182,0.0043,-0.9998,0.0188,0.0022,-0.9998,-0.4827,-0.0557,-0.874,-0.4827,-0.0557,-0.874,0.0182,0.0043,-0.9998,-0.4708,-0.1099,-0.8754,0.4983,0.1163,-0.8591,0.5109,0.0589,-0.8576,0.0188,0.0022,-0.9998,0.0188,0.0022,-0.9998,0.4983,0.1163,-0.8591,0.0182,0.0043,-0.9998,0.8467,0.1976,-0.4941,0.8651,0.0998,-0.4916,0.5109,0.0589,-0.8576,0.5109,0.0589,-0.8576,0.8467,0.1976,-0.4941,0.4983,0.1163,-0.8591,0.9738,0.2272,0,0.9934,0.1146,0,0.8651,0.0998,-0.4916,0.8651,0.0998,-0.4916,0.9738,0.2272,0,0.8467,0.1976,-0.4941,0.7809,0.3821,0.4942,0.8467,0.1976,0.4941,0.9738,0.2272,0,0.9738,0.2272,0,0.7809,0.3821,0.4942,0.8982,0.4395,0,0.4594,0.2248,0.8593,0.4983,0.1163,0.8591,0.8467,0.1976,0.4941,0.8467,0.1976,0.4941,0.4594,0.2248,0.8593,0.7809,0.3821,0.4942,0.0164,0.008,0.9998,0.0182,0.0043,0.9998,0.4983,0.1163,0.8591,0.4983,0.1163,0.8591,0.0164,0.008,0.9998,0.4594,0.2248,0.8593,-0.4347,-0.2127,0.8751,-0.4708,-0.1099,0.8754,0.0182,0.0043,0.9998,0.0182,0.0043,0.9998,-0.4347,-0.2127,0.8751,0.0164,0.008,0.9998,-0.7725,-0.378,0.5102,-0.8374,-0.1954,0.5105,-0.4708,-0.1099,0.8754,-0.4708,-0.1099,0.8754,-0.7725,-0.378,0.5102,-0.4347,-0.2127,0.8751,-0.8982,-0.4395,0,-0.9738,-0.2272,0,-0.8374,-0.1954,0.5105,-0.8374,-0.1954,0.5105,-0.8982,-0.4395,0,-0.7725,-0.378,0.5102,-0.7725,-0.378,-0.5102,-0.8374,-0.1954,-0.5105,-0.9738,-0.2272,0,-0.9738,-0.2272,0,-0.7725,-0.378,-0.5102,-0.8982,-0.4395,0,-0.4347,-0.2127,-0.8751,-0.4708,-0.1099,-0.8754,-0.8374,-0.1954,-0.5105,-0.8374,-0.1954,-0.5105,-0.4347,-0.2127,-0.8751,-0.7725,-0.378,-0.5102,0.0164,0.008,-0.9998,0.0182,0.0043,-0.9998,-0.4708,-0.1099,-0.8754,-0.4708,-0.1099,-0.8754,0.0164,0.008,-0.9998,-0.4347,-0.2127,-0.8751,0.4594,0.2248,-0.8593,0.4983,0.1163,-0.8591,0.0182,0.0043,-0.9998,0.0182,0.0043,-0.9998,0.4594,0.2248,-0.8593,0.0164,0.008,-0.9998,0.7809,0.3821,-0.4942,0.8467,0.1976,-0.4941,0.4983,0.1163,-0.8591,0.4983,0.1163,-0.8591,0.7809,0.3821,-0.4942,0.4594,0.2248,-0.8593,0.8982,0.4395,0,0.9738,0.2272,0,0.8467,0.1976,-0.4941,0.8467,0.1976,-0.4941,0.8982,0.4395,0,0.7809,0.3821,-0.4942,0.6781,0.5439,0.4943,0.7809,0.3821,0.4942,0.8982,0.4395,0,0.8982,0.4395,0,0.6781,0.5439,0.4943,0.7801,0.6257,0,0.3989,0.3199,0.8594,0.4594,0.2248,0.8593,0.7809,0.3821,0.4942,0.7809,0.3821,0.4942,0.3989,0.3199,0.8594,0.6781,0.5439,0.4943,0.014,0.0112,0.9998,0.0164,0.008,0.9998,0.4594,0.2248,0.8593,0.4594,0.2248,0.8593,0.014,0.0112,0.9998,0.3989,0.3199,0.8594,-0.3778,-0.303,0.8749,-0.4347,-0.2127,0.8751,0.0164,0.008,0.9998,0.0164,0.008,0.9998,-0.3778,-0.303,0.8749,0.014,0.0112,0.9998,-0.671,-0.5382,0.5099,-0.7725,-0.378,0.5102,-0.4347,-0.2127,0.8751,-0.4347,-0.2127,0.8751,-0.671,-0.5382,0.5099,-0.3778,-0.303,0.8749,-0.7801,-0.6257,0,-0.8982,-0.4395,0,-0.7725,-0.378,0.5102,-0.7725,-0.378,0.5102,-0.7801,-0.6257,0,-0.671,-0.5382,0.5099,-0.671,-0.5382,-0.5099,-0.7725,-0.378,-0.5102,-0.8982,-0.4395,0,-0.8982,-0.4395,0,-0.671,-0.5382,-0.5099,-0.7801,-0.6257,0,-0.3778,-0.303,-0.8749,-0.4347,-0.2127,-0.8751,-0.7725,-0.378,-0.5102,-0.7725,-0.378,-0.5102,-0.3778,-0.303,-0.8749,-0.671,-0.5382,-0.5099,0.014,0.0112,-0.9998,0.0164,0.008,-0.9998,-0.4347,-0.2127,-0.8751,-0.4347,-0.2127,-0.8751,0.014,0.0112,-0.9998,-0.3778,-0.303,-0.8749,0.3989,0.3199,-0.8594,0.4594,0.2248,-0.8593,0.0164,0.008,-0.9998,0.0164,0.008,-0.9998,0.3989,0.3199,-0.8594,0.014,0.0112,-0.9998,0.6781,0.5439,-0.4943,0.7809,0.3821,-0.4942,0.4594,0.2248,-0.8593,0.4594,0.2248,-0.8593,0.6781,0.5439,-0.4943,0.3989,0.3199,-0.8594,0.7801,0.6257,0,0.8982,0.4395,0,0.7809,0.3821,-0.4942,0.7809,0.3821,-0.4942,0.7801,0.6257,0,0.6781,0.5439,-0.4943,0.5439,0.6781,0.4943,0.6781,0.5439,0.4943,0.7801,0.6257,0,0.7801,0.6257,0,0.5439,0.6781,0.4943,0.6257,0.7801,0,0.3199,0.3989,0.8594,0.3989,0.3199,0.8594,0.6781,0.5439,0.4943,0.6781,0.5439,0.4943,0.3199,0.3989,0.8594,0.5439,0.6781,0.4943,0.0112,0.014,0.9998,0.014,0.0112,0.9998,0.3989,0.3199,0.8594,0.3989,0.3199,0.8594,0.0112,0.014,0.9998,0.3199,0.3989,0.8594,-0.303,-0.3778,0.8749,-0.3778,-0.303,0.8749,0.014,0.0112,0.9998,0.014,0.0112,0.9998,-0.303,-0.3778,0.8749,0.0112,0.014,0.9998,-0.5382,-0.671,0.5099,-0.671,-0.5382,0.5099,-0.3778,-0.303,0.8749,-0.3778,-0.303,0.8749,-0.5382,-0.671,0.5099,-0.303,-0.3778,0.8749,-0.6257,-0.7801,0,-0.7801,-0.6257,0,-0.671,-0.5382,0.5099,-0.671,-0.5382,0.5099,-0.6257,-0.7801,0,-0.5382,-0.671,0.5099,-0.5382,-0.671,-0.5099,-0.671,-0.5382,-0.5099,-0.7801,-0.6257,0,-0.7801,-0.6257,0,-0.5382,-0.671,-0.5099,-0.6257,-0.7801,0,-0.303,-0.3778,-0.8749,-0.3778,-0.303,-0.8749,-0.671,-0.5382,-0.5099,-0.671,-0.5382,-0.5099,-0.303,-0.3778,-0.8749,-0.5382,-0.671,-0.5099,0.0112,0.014,-0.9998,0.014,0.0112,-0.9998,-0.3778,-0.303,-0.8749,-0.3778,-0.303,-0.8749,0.0112,0.014,-0.9998,-0.303,-0.3778,-0.8749,0.3199,0.3989,-0.8594,0.3989,0.3199,-0.8594,0.014,0.0112,-0.9998,0.014,0.0112,-0.9998,0.3199,0.3989,-0.8594,0.0112,0.014,-0.9998,0.5439,0.6781,-0.4943,0.6781,0.5439,-0.4943,0.3989,0.3199,-0.8594,0.3989,0.3199,-0.8594,0.5439,0.6781,-0.4943,0.3199,0.3989,-0.8594,0.6257,0.7801,0,0.7801,0.6257,0,0.6781,0.5439,-0.4943,0.6781,0.5439,-0.4943,0.6257,0.7801,0,0.5439,0.6781,-0.4943,0.3821,0.7809,0.4942,0.5439,0.6781,0.4943,0.6257,0.7801,0,0.6257,0.7801,0,0.3821,0.7809,0.4942,0.4395,0.8982,0,0.2248,0.4594,0.8593,0.3199,0.3989,0.8594,0.5439,0.6781,0.4943,0.5439,0.6781,0.4943,0.2248,0.4594,0.8593,0.3821,0.7809,0.4942,0.008,0.0164,0.9998,0.0112,0.014,0.9998,0.3199,0.3989,0.8594,0.3199,0.3989,0.8594,0.008,0.0164,0.9998,0.2248,0.4594,0.8593,-0.2127,-0.4347,0.8751,-0.303,-0.3778,0.8749,0.0112,0.014,0.9998,0.0112,0.014,0.9998,-0.2127,-0.4347,0.8751,0.008,0.0164,0.9998,-0.378,-0.7725,0.5102,-0.5382,-0.671,0.5099,-0.303,-0.3778,0.8749,-0.303,-0.3778,0.8749,-0.378,-0.7725,0.5102,-0.2127,-0.4347,0.8751,-0.4395,-0.8982,0,-0.6257,-0.7801,0,-0.5382,-0.671,0.5099,-0.5382,-0.671,0.5099,-0.4395,-0.8982,0,-0.378,-0.7725,0.5102,-0.378,-0.7725,-0.5102,-0.5382,-0.671,-0.5099,-0.6257,-0.7801,0,-0.6257,-0.7801,0,-0.378,-0.7725,-0.5102,-0.4395,-0.8982,0,-0.2127,-0.4347,-0.8751,-0.303,-0.3778,-0.8749,-0.5382,-0.671,-0.5099,-0.5382,-0.671,-0.5099,-0.2127,-0.4347,-0.8751,-0.378,-0.7725,-0.5102,0.008,0.0164,-0.9998,0.0112,0.014,-0.9998,-0.303,-0.3778,-0.8749,-0.303,-0.3778,-0.8749,0.008,0.0164,-0.9998,-0.2127,-0.4347,-0.8751,0.2248,0.4594,-0.8593,0.3199,0.3989,-0.8594,0.0112,0.014,-0.9998,0.0112,0.014,-0.9998,0.2248,0.4594,-0.8593,0.008,0.0164,-0.9998,0.3821,0.7809,-0.4942,0.5439,0.6781,-0.4943,0.3199,0.3989,-0.8594,0.3199,0.3989,-0.8594,0.3821,0.7809,-0.4942,0.2248,0.4594,-0.8593,0.4395,0.8982,0,0.6257,0.7801,0,0.5439,0.6781,-0.4943,0.5439,0.6781,-0.4943,0.4395,0.8982,0,0.3821,0.7809,-0.4942,0.1976,0.8467,0.4941,0.3821,0.7809,0.4942,0.4395,0.8982,0,0.4395,0.8982,0,0.1976,0.8467,0.4941,0.2272,0.9738,0,0.1163,0.4983,0.8591,0.2248,0.4594,0.8593,0.3821,0.7809,0.4942,0.3821,0.7809,0.4942,0.1163,0.4983,0.8591,0.1976,0.8467,0.4941,0.0043,0.0182,0.9998,0.008,0.0164,0.9998,0.2248,0.4594,0.8593,0.2248,0.4594,0.8593,0.0043,0.0182,0.9998,0.1163,0.4983,0.8591,-0.1099,-0.4708,0.8754,-0.2127,-0.4347,0.8751,0.008,0.0164,0.9998,0.008,0.0164,0.9998,-0.1099,-0.4708,0.8754,0.0043,0.0182,0.9998,-0.1954,-0.8374,0.5105,-0.378,-0.7725,0.5102,-0.2127,-0.4347,0.8751,-0.2127,-0.4347,0.8751,-0.1954,-0.8374,0.5105,-0.1099,-0.4708,0.8754,-0.2272,-0.9738,0,-0.4395,-0.8982,0,-0.378,-0.7725,0.5102,-0.378,-0.7725,0.5102,-0.2272,-0.9738,0,-0.1954,-0.8374,0.5105,-0.1954,-0.8374,-0.5105,-0.378,-0.7725,-0.5102,-0.4395,-0.8982,0,-0.4395,-0.8982,0,-0.1954,-0.8374,-0.5105,-0.2272,-0.9738,0,-0.1099,-0.4708,-0.8754,-0.2127,-0.4347,-0.8751,-0.378,-0.7725,-0.5102,-0.378,-0.7725,-0.5102,-0.1099,-0.4708,-0.8754,-0.1954,-0.8374,-0.5105,0.0043,0.0182,-0.9998,0.008,0.0164,-0.9998,-0.2127,-0.4347,-0.8751,-0.2127,-0.4347,-0.8751,0.0043,0.0182,-0.9998,-0.1099,-0.4708,-0.8754,0.1163,0.4983,-0.8591,0.2248,0.4594,-0.8593,0.008,0.0164,-0.9998,0.008,0.0164,-0.9998,0.1163,0.4983,-0.8591,0.0043,0.0182,-0.9998,0.1976,0.8467,-0.4941,0.3821,0.7809,-0.4942,0.2248,0.4594,-0.8593,0.2248,0.4594,-0.8593,0.1976,0.8467,-0.4941,0.1163,0.4983,-0.8591,0.2272,0.9738,0,0.4395,0.8982,0,0.3821,0.7809,-0.4942,0.3821,0.7809,-0.4942,0.2272,0.9738,0,0.1976,0.8467,-0.4941,0,0.8694,0.4941,0.1976,0.8467,0.4941,0.2272,0.9738,0,0.2272,0.9738,0,0,0.8694,0.4941,0,1,0,0,0.5118,0.8591,0.1163,0.4983,0.8591,0.1976,0.8467,0.4941,0.1976,0.8467,0.4941,0,0.5118,0.8591,0,0.8694,0.4941,0,0.0188,0.9998,0.0043,0.0182,0.9998,0.1163,0.4983,0.8591,0.1163,0.4983,0.8591,0,0.0188,0.9998,0,0.5118,0.8591,0,-0.4834,0.8754,-0.1099,-0.4708,0.8754,0.0043,0.0182,0.9998,0.0043,0.0182,0.9998,0,-0.4834,0.8754,0,0.0188,0.9998,0,-0.8598,0.5106,-0.1954,-0.8374,0.5105,-0.1099,-0.4708,0.8754,-0.1099,-0.4708,0.8754,0,-0.8598,0.5106,0,-0.4834,0.8754,0,-1,0,-0.2272,-0.9738,0,-0.1954,-0.8374,0.5105,-0.1954,-0.8374,0.5105,0,-1,0,0,-0.8598,0.5106,0,-0.8598,-0.5106,-0.1954,-0.8374,-0.5105,-0.2272,-0.9738,0,-0.2272,-0.9738,0,0,-0.8598,-0.5106,0,-1,0,0,-0.4834,-0.8754,-0.1099,-0.4708,-0.8754,-0.1954,-0.8374,-0.5105,-0.1954,-0.8374,-0.5105,0,-0.4834,-0.8754,0,-0.8598,-0.5106,0,0.0188,-0.9998,0.0043,0.0182,-0.9998,-0.1099,-0.4708,-0.8754,-0.1099,-0.4708,-0.8754,0,0.0188,-0.9998,0,-0.4834,-0.8754,0,0.5118,-0.8591,0.1163,0.4983,-0.8591,0.0043,0.0182,-0.9998,0.0043,0.0182,-0.9998,0,0.5118,-0.8591,0,0.0188,-0.9998,0,0.8694,-0.4941,0.1976,0.8467,-0.4941,0.1163,0.4983,-0.8591,0.1163,0.4983,-0.8591,0,0.8694,-0.4941,0,0.5118,-0.8591,0,1,0,0.2272,0.9738,0,0.1976,0.8467,-0.4941,0.1976,0.8467,-0.4941,0,1,0,0,0.8694,-0.4941,-0.1976,0.8467,0.4941,0,0.8694,0.4941,0,1,0,0,1,0,-0.1976,0.8467,0.4941,-0.2272,0.9738,0,-0.1163,0.4983,0.8591,0,0.5118,0.8591,0,0.8694,0.4941,0,0.8694,0.4941,-0.1163,0.4983,0.8591,-0.1976,0.8467,0.4941,-0.0043,0.0182,0.9998,0,0.0188,0.9998,0,0.5118,0.8591,0,0.5118,0.8591,-0.0043,0.0182,0.9998,-0.1163,0.4983,0.8591,0.1099,-0.4708,0.8754,0,-0.4834,0.8754,0,0.0188,0.9998,0,0.0188,0.9998,0.1099,-0.4708,0.8754,-0.0043,0.0182,0.9998,0.1954,-0.8374,0.5105,0,-0.8598,0.5106,0,-0.4834,0.8754,0,-0.4834,0.8754,0.1954,-0.8374,0.5105,0.1099,-0.4708,0.8754,0.2272,-0.9738,0,0,-1,0,0,-0.8598,0.5106,0,-0.8598,0.5106,0.2272,-0.9738,0,0.1954,-0.8374,0.5105,0.1954,-0.8374,-0.5105,0,-0.8598,-0.5106,0,-1,0,0,-1,0,0.1954,-0.8374,-0.5105,0.2272,-0.9738,0,0.1099,-0.4708,-0.8754,0,-0.4834,-0.8754,0,-0.8598,-0.5106,0,-0.8598,-0.5106,0.1099,-0.4708,-0.8754,0.1954,-0.8374,-0.5105,-0.0043,0.0182,-0.9998,0,0.0188,-0.9998,0,-0.4834,-0.8754,0,-0.4834,-0.8754,-0.0043,0.0182,-0.9998,0.1099,-0.4708,-0.8754,-0.1163,0.4983,-0.8591,0,0.5118,-0.8591,0,0.0188,-0.9998,0,0.0188,-0.9998,-0.1163,0.4983,-0.8591,-0.0043,0.0182,-0.9998,-0.1976,0.8467,-0.4941,0,0.8694,-0.4941,0,0.5118,-0.8591,0,0.5118,-0.8591,-0.1976,0.8467,-0.4941,-0.1163,0.4983,-0.8591,-0.2272,0.9738,0,0,1,0,0,0.8694,-0.4941,0,0.8694,-0.4941,-0.2272,0.9738,0,-0.1976,0.8467,-0.4941,-0.3821,0.7809,0.4942,-0.1976,0.8467,0.4941,-0.2272,0.9738,0,-0.2272,0.9738,0,-0.3821,0.7809,0.4942,-0.4395,0.8982,0,-0.2248,0.4594,0.8593,-0.1163,0.4983,0.8591,-0.1976,0.8467,0.4941,-0.1976,0.8467,0.4941,-0.2248,0.4594,0.8593,-0.3821,0.7809,0.4942,-0.008,0.0164,0.9998,-0.0043,0.0182,0.9998,-0.1163,0.4983,0.8591,-0.1163,0.4983,0.8591,-0.008,0.0164,0.9998,-0.2248,0.4594,0.8593,0.2127,-0.4347,0.8751,0.1099,-0.4708,0.8754,-0.0043,0.0182,0.9998,-0.0043,0.0182,0.9998,0.2127,-0.4347,0.8751,-0.008,0.0164,0.9998,0.378,-0.7725,0.5102,0.1954,-0.8374,0.5105,0.1099,-0.4708,0.8754,0.1099,-0.4708,0.8754,0.378,-0.7725,0.5102,0.2127,-0.4347,0.8751,0.4395,-0.8982,0,0.2272,-0.9738,0,0.1954,-0.8374,0.5105,0.1954,-0.8374,0.5105,0.4395,-0.8982,0,0.378,-0.7725,0.5102,0.378,-0.7725,-0.5102,0.1954,-0.8374,-0.5105,0.2272,-0.9738,0,0.2272,-0.9738,0,0.378,-0.7725,-0.5102,0.4395,-0.8982,0,0.2127,-0.4347,-0.8751,0.1099,-0.4708,-0.8754,0.1954,-0.8374,-0.5105,0.1954,-0.8374,-0.5105,0.2127,-0.4347,-0.8751,0.378,-0.7725,-0.5102,-0.008,0.0164,-0.9998,-0.0043,0.0182,-0.9998,0.1099,-0.4708,-0.8754,0.1099,-0.4708,-0.8754,-0.008,0.0164,-0.9998,0.2127,-0.4347,-0.8751,-0.2248,0.4594,-0.8593,-0.1163,0.4983,-0.8591,-0.0043,0.0182,-0.9998,-0.0043,0.0182,-0.9998,-0.2248,0.4594,-0.8593,-0.008,0.0164,-0.9998,-0.3821,0.7809,-0.4942,-0.1976,0.8467,-0.4941,-0.1163,0.4983,-0.8591,-0.1163,0.4983,-0.8591,-0.3821,0.7809,-0.4942,-0.2248,0.4594,-0.8593,-0.4395,0.8982,0,-0.2272,0.9738,0,-0.1976,0.8467,-0.4941,-0.1976,0.8467,-0.4941,-0.4395,0.8982,0,-0.3821,0.7809,-0.4942,-0.5439,0.6781,0.4943,-0.3821,0.7809,0.4942,-0.4395,0.8982,0,-0.4395,0.8982,0,-0.5439,0.6781,0.4943,-0.6257,0.7801,0,-0.3199,0.3989,0.8594,-0.2248,0.4594,0.8593,-0.3821,0.7809,0.4942,-0.3821,0.7809,0.4942,-0.3199,0.3989,0.8594,-0.5439,0.6781,0.4943,-0.0112,0.014,0.9998,-0.008,0.0164,0.9998,-0.2248,0.4594,0.8593,-0.2248,0.4594,0.8593,-0.0112,0.014,0.9998,-0.3199,0.3989,0.8594,0.303,-0.3778,0.8749,0.2127,-0.4347,0.8751,-0.008,0.0164,0.9998,-0.008,0.0164,0.9998,0.303,-0.3778,0.8749,-0.0112,0.014,0.9998,0.5382,-0.671,0.5099,0.378,-0.7725,0.5102,0.2127,-0.4347,0.8751,0.2127,-0.4347,0.8751,0.5382,-0.671,0.5099,0.303,-0.3778,0.8749,0.6257,-0.7801,0,0.4395,-0.8982,0,0.378,-0.7725,0.5102,0.378,-0.7725,0.5102,0.6257,-0.7801,0,0.5382,-0.671,0.5099,0.5382,-0.671,-0.5099,0.378,-0.7725,-0.5102,0.4395,-0.8982,0,0.4395,-0.8982,0,0.5382,-0.671,-0.5099,0.6257,-0.7801,0,0.303,-0.3778,-0.8749,0.2127,-0.4347,-0.8751,0.378,-0.7725,-0.5102,0.378,-0.7725,-0.5102,0.303,-0.3778,-0.8749,0.5382,-0.671,-0.5099,-0.0112,0.014,-0.9998,-0.008,0.0164,-0.9998,0.2127,-0.4347,-0.8751,0.2127,-0.4347,-0.8751,-0.0112,0.014,-0.9998,0.303,-0.3778,-0.8749,-0.3199,0.3989,-0.8594,-0.2248,0.4594,-0.8593,-0.008,0.0164,-0.9998,-0.008,0.0164,-0.9998,-0.3199,0.3989,-0.8594,-0.0112,0.014,-0.9998,-0.5439,0.6781,-0.4943,-0.3821,0.7809,-0.4942,-0.2248,0.4594,-0.8593,-0.2248,0.4594,-0.8593,-0.5439,0.6781,-0.4943,-0.3199,0.3989,-0.8594,-0.6257,0.7801,0,-0.4395,0.8982,0,-0.3821,0.7809,-0.4942,-0.3821,0.7809,-0.4942,-0.6257,0.7801,0,-0.5439,0.6781,-0.4943,-0.6781,0.5439,0.4943,-0.5439,0.6781,0.4943,-0.6257,0.7801,0,-0.6257,0.7801,0,-0.6781,0.5439,0.4943,-0.7801,0.6257,0,-0.3989,0.3199,0.8594,-0.3199,0.3989,0.8594,-0.5439,0.6781,0.4943,-0.5439,0.6781,0.4943,-0.3989,0.3199,0.8594,-0.6781,0.5439,0.4943,-0.014,0.0112,0.9998,-0.0112,0.014,0.9998,-0.3199,0.3989,0.8594,-0.3199,0.3989,0.8594,-0.014,0.0112,0.9998,-0.3989,0.3199,0.8594,0.3778,-0.303,0.8749,0.303,-0.3778,0.8749,-0.0112,0.014,0.9998,-0.0112,0.014,0.9998,0.3778,-0.303,0.8749,-0.014,0.0112,0.9998,0.671,-0.5382,0.5099,0.5382,-0.671,0.5099,0.303,-0.3778,0.8749,0.303,-0.3778,0.8749,0.671,-0.5382,0.5099,0.3778,-0.303,0.8749,0.7801,-0.6257,0,0.6257,-0.7801,0,0.5382,-0.671,0.5099,0.5382,-0.671,0.5099,0.7801,-0.6257,0,0.671,-0.5382,0.5099,0.671,-0.5382,-0.5099,0.5382,-0.671,-0.5099,0.6257,-0.7801,0,0.6257,-0.7801,0,0.671,-0.5382,-0.5099,0.7801,-0.6257,0,0.3778,-0.303,-0.8749,0.303,-0.3778,-0.8749,0.5382,-0.671,-0.5099,0.5382,-0.671,-0.5099,0.3778,-0.303,-0.8749,0.671,-0.5382,-0.5099,-0.014,0.0112,-0.9998,-0.0112,0.014,-0.9998,0.303,-0.3778,-0.8749,0.303,-0.3778,-0.8749,-0.014,0.0112,-0.9998,0.3778,-0.303,-0.8749,-0.3989,0.3199,-0.8594,-0.3199,0.3989,-0.8594,-0.0112,0.014,-0.9998,-0.0112,0.014,-0.9998,-0.3989,0.3199,-0.8594,-0.014,0.0112,-0.9998,-0.6781,0.5439,-0.4943,-0.5439,0.6781,-0.4943,-0.3199,0.3989,-0.8594,-0.3199,0.3989,-0.8594,-0.6781,0.5439,-0.4943,-0.3989,0.3199,-0.8594,-0.7801,0.6257,0,-0.6257,0.7801,0,-0.5439,0.6781,-0.4943,-0.5439,0.6781,-0.4943,-0.7801,0.6257,0,-0.6781,0.5439,-0.4943,-0.7809,0.3821,0.4942,-0.6781,0.5439,0.4943,-0.7801,0.6257,0,-0.7801,0.6257,0,-0.7809,0.3821,0.4942,-0.8982,0.4395,0,-0.4594,0.2248,0.8593,-0.3989,0.3199,0.8594,-0.6781,0.5439,0.4943,-0.6781,0.5439,0.4943,-0.4594,0.2248,0.8593,-0.7809,0.3821,0.4942,-0.0164,0.008,0.9998,-0.014,0.0112,0.9998,-0.3989,0.3199,0.8594,-0.3989,0.3199,0.8594,-0.0164,0.008,0.9998,-0.4594,0.2248,0.8593,0.4347,-0.2127,0.8751,0.3778,-0.303,0.8749,-0.014,0.0112,0.9998,-0.014,0.0112,0.9998,0.4347,-0.2127,0.8751,-0.0164,0.008,0.9998,0.7725,-0.378,0.5102,0.671,-0.5382,0.5099,0.3778,-0.303,0.8749,0.3778,-0.303,0.8749,0.7725,-0.378,0.5102,0.4347,-0.2127,0.8751,0.8982,-0.4395,0,0.7801,-0.6257,0,0.671,-0.5382,0.5099,0.671,-0.5382,0.5099,0.8982,-0.4395,0,0.7725,-0.378,0.5102,0.7725,-0.378,-0.5102,0.671,-0.5382,-0.5099,0.7801,-0.6257,0,0.7801,-0.6257,0,0.7725,-0.378,-0.5102,0.8982,-0.4395,0,0.4347,-0.2127,-0.8751,0.3778,-0.303,-0.8749,0.671,-0.5382,-0.5099,0.671,-0.5382,-0.5099,0.4347,-0.2127,-0.8751,0.7725,-0.378,-0.5102,-0.0164,0.008,-0.9998,-0.014,0.0112,-0.9998,0.3778,-0.303,-0.8749,0.3778,-0.303,-0.8749,-0.0164,0.008,-0.9998,0.4347,-0.2127,-0.8751,-0.4594,0.2248,-0.8593,-0.3989,0.3199,-0.8594,-0.014,0.0112,-0.9998,-0.014,0.0112,-0.9998,-0.4594,0.2248,-0.8593,-0.0164,0.008,-0.9998,-0.7809,0.3821,-0.4942,-0.6781,0.5439,-0.4943,-0.3989,0.3199,-0.8594,-0.3989,0.3199,-0.8594,-0.7809,0.3821,-0.4942,-0.4594,0.2248,-0.8593,-0.8982,0.4395,0,-0.7801,0.6257,0,-0.6781,0.5439,-0.4943,-0.6781,0.5439,-0.4943,-0.8982,0.4395,0,-0.7809,0.3821,-0.4942,-0.8467,0.1976,0.4941,-0.7809,0.3821,0.4942,-0.8982,0.4395,0,-0.8982,0.4395,0,-0.8467,0.1976,0.4941,-0.9738,0.2272,0,-0.4983,0.1163,0.8591,-0.4594,0.2248,0.8593,-0.7809,0.3821,0.4942,-0.7809,0.3821,0.4942,-0.4983,0.1163,0.8591,-0.8467,0.1976,0.4941,-0.0182,0.0043,0.9998,-0.0164,0.008,0.9998,-0.4594,0.2248,0.8593,-0.4594,0.2248,0.8593,-0.0182,0.0043,0.9998,-0.4983,0.1163,0.8591,0.4708,-0.1099,0.8754,0.4347,-0.2127,0.8751,-0.0164,0.008,0.9998,-0.0164,0.008,0.9998,0.4708,-0.1099,0.8754,-0.0182,0.0043,0.9998,0.8374,-0.1954,0.5105,0.7725,-0.378,0.5102,0.4347,-0.2127,0.8751,0.4347,-0.2127,0.8751,0.8374,-0.1954,0.5105,0.4708,-0.1099,0.8754,0.9738,-0.2272,0,0.8982,-0.4395,0,0.7725,-0.378,0.5102,0.7725,-0.378,0.5102,0.9738,-0.2272,0,0.8374,-0.1954,0.5105,0.8374,-0.1954,-0.5105,0.7725,-0.378,-0.5102,0.8982,-0.4395,0,0.8982,-0.4395,0,0.8374,-0.1954,-0.5105,0.9738,-0.2272,0,0.4708,-0.1099,-0.8754,0.4347,-0.2127,-0.8751,0.7725,-0.378,-0.5102,0.7725,-0.378,-0.5102,0.4708,-0.1099,-0.8754,0.8374,-0.1954,-0.5105,-0.0182,0.0043,-0.9998,-0.0164,0.008,-0.9998,0.4347,-0.2127,-0.8751,0.4347,-0.2127,-0.8751,-0.0182,0.0043,-0.9998,0.4708,-0.1099,-0.8754,-0.4983,0.1163,-0.8591,-0.4594,0.2248,-0.8593,-0.0164,0.008,-0.9998,-0.0164,0.008,-0.9998,-0.4983,0.1163,-0.8591,-0.0182,0.0043,-0.9998,-0.8467,0.1976,-0.4941,-0.7809,0.3821,-0.4942,-0.4594,0.2248,-0.8593,-0.4594,0.2248,-0.8593,-0.8467,0.1976,-0.4941,-0.4983,0.1163,-0.8591,-0.9738,0.2272,0,-0.8982,0.4395,0,-0.7809,0.3821,-0.4942,-0.7809,0.3821,-0.4942,-0.9738,0.2272,0,-0.8467,0.1976,-0.4941,-0.8651,0.0998,0.4916,-0.8467,0.1976,0.4941,-0.9738,0.2272,0,-0.9738,0.2272,0,-0.8651,0.0998,0.4916,-0.9934,0.1146,0,-0.5109,0.0589,0.8576,-0.4983,0.1163,0.8591,-0.8467,0.1976,0.4941,-0.8467,0.1976,0.4941,-0.5109,0.0589,0.8576,-0.8651,0.0998,0.4916,-0.0188,0.0022,0.9998,-0.0182,0.0043,0.9998,-0.4983,0.1163,0.8591,-0.4983,0.1163,0.8591,-0.0188,0.0022,0.9998,-0.5109,0.0589,0.8576,0.4827,-0.0557,0.874,0.4708,-0.1099,0.8754,-0.0182,0.0043,0.9998,-0.0182,0.0043,0.9998,0.4827,-0.0557,0.874,-0.0188,0.0022,0.9998,0.8556,-0.0987,0.5081,0.8374,-0.1954,0.5105,0.4708,-0.1099,0.8754,0.4708,-0.1099,0.8754,0.8556,-0.0987,0.5081,0.4827,-0.0557,0.874,0.9934,-0.1146,0,0.9738,-0.2272,0,0.8374,-0.1954,0.5105,0.8374,-0.1954,0.5105,0.9934,-0.1146,0,0.8556,-0.0987,0.5081,0.8556,-0.0987,-0.5081,0.8374,-0.1954,-0.5105,0.9738,-0.2272,0,0.9738,-0.2272,0,0.8556,-0.0987,-0.5081,0.9934,-0.1146,0,0.4827,-0.0557,-0.874,0.4708,-0.1099,-0.8754,0.8374,-0.1954,-0.5105,0.8374,-0.1954,-0.5105,0.4827,-0.0557,-0.874,0.8556,-0.0987,-0.5081,-0.0188,0.0022,-0.9998,-0.0182,0.0043,-0.9998,0.4708,-0.1099,-0.8754,0.4708,-0.1099,-0.8754,-0.0188,0.0022,-0.9998,0.4827,-0.0557,-0.874,-0.5109,0.0589,-0.8576,-0.4983,0.1163,-0.8591,-0.0182,0.0043,-0.9998,-0.0182,0.0043,-0.9998,-0.5109,0.0589,-0.8576,-0.0188,0.0022,-0.9998,-0.8651,0.0998,-0.4916,-0.8467,0.1976,-0.4941,-0.4983,0.1163,-0.8591,-0.4983,0.1163,-0.8591,-0.8651,0.0998,-0.4916,-0.5109,0.0589,-0.8576,-0.9934,0.1146,0,-0.9738,0.2272,0,-0.8467,0.1976,-0.4941,-0.8467,0.1976,-0.4941,-0.9934,0.1146,0,-0.8651,0.0998,-0.4916] )
   , make_vbo(COLOR, [0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0.9686,1,0.6235,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882,0.5882] )
  ],
  mode: gl.TRIANGLES ,
  vertex_count: 2262
};
var generator =  function () {
        var vertices = []
        var colors = []
        for (var i = 0; i < 400; i++)
        {
            var spread = 100.0
            vertices.push((Math.random() - 0.5) * spread, Math.random() * spread, (Math.random() - 0.5) * spread)
            //colors.push(1.0, 0.6, 0.7)
            colors.push(Math.random(), Math.random(), Math.random())
        }
        
        //console.log(vertices)
        //console.log(normals)
      return {
        buffers: [
          make_vbo(POS, vertices),
          make_vbo(NORMAL, colors)
        ],
        mode: gl.POINTS,
        vertex_count: vertices.length / 3
      }
    }
geometries["particles"] = generator();
geometries["plane"] = {
  buffers: [
     make_vbo(POS, [-1,-1,0,-1,1,0,1,-1,0,1,1,0] )
   , make_vbo(NORMAL, [0,0,1,0,0,1,0,0,1,0,0,1] )
  ],
  mode: gl.TRIANGLE_STRIP ,
  vertex_count: 4
};
geometries["porte-tori"] = {
  buffers: [
     make_vbo(POS, [52.8361,0.0102,-0.0077,53.9545,0.0102,6.3644,53.9545,5.6433,6.3644,53.9545,5.6433,6.3644,52.8361,0.0102,-0.0077,52.8361,5.6433,-0.0077,53.9545,0.0102,6.3644,57.1848,0.0102,11.9697,57.1848,5.6433,11.9697,57.1848,5.6433,11.9697,53.9545,0.0102,6.3644,53.9545,5.6433,6.3644,57.1848,0.0102,11.9697,62.1374,0.0102,16.1321,62.1374,5.6433,16.1321,62.1374,5.6433,16.1321,57.1848,0.0102,11.9697,57.1848,5.6433,11.9697,62.1374,0.0102,16.1321,68.215,0.0102,18.3497,68.215,5.6433,18.3497,68.215,5.6433,18.3497,62.1374,0.0102,16.1321,62.1374,5.6433,16.1321,68.215,0.0102,18.3497,74.6844,0.0102,18.3548,74.6844,5.6433,18.3548,74.6844,5.6433,18.3548,68.215,0.0102,18.3497,68.215,5.6433,18.3497,74.6844,0.0102,18.3548,80.7655,0.0102,16.1469,80.7655,5.6433,16.1469,80.7655,5.6433,16.1469,74.6844,0.0102,18.3548,74.6844,5.6433,18.3548,80.7655,0.0102,16.1469,85.7247,0.0102,11.9923,85.7247,5.6433,11.9923,85.7247,5.6433,11.9923,80.7655,0.0102,16.1469,80.7655,5.6433,16.1469,85.7247,0.0102,11.9923,88.9639,0.0102,6.3922,88.9639,5.6433,6.3922,88.9639,5.6433,6.3922,85.7247,0.0102,11.9923,85.7247,5.6433,11.9923,88.9639,0.0102,6.3922,90.0923,0.0102,0.0219,90.0923,5.6433,0.0219,90.0923,5.6433,0.0219,88.9639,0.0102,6.3922,88.9639,5.6433,6.3922,90.0923,0.0102,0.0219,88.974,0.0102,-6.3502,88.974,5.6433,-6.3502,88.974,5.6433,-6.3502,90.0923,0.0102,0.0219,90.0923,5.6433,0.0219,88.974,0.0102,-6.3502,85.7437,0.0102,-11.9555,85.7437,5.6433,-11.9555,85.7437,5.6433,-11.9555,88.974,0.0102,-6.3502,88.974,5.6433,-6.3502,85.7437,0.0102,-11.9555,80.7911,0.0102,-16.1179,80.7911,5.6433,-16.1179,80.7911,5.6433,-16.1179,85.7437,0.0102,-11.9555,85.7437,5.6433,-11.9555,80.7911,0.0102,-16.1179,74.7135,0.0102,-18.3354,74.7135,5.6433,-18.3354,74.7135,5.6433,-18.3354,80.7911,0.0102,-16.1179,80.7911,5.6433,-16.1179,74.7135,0.0102,-18.3354,68.244,0.0102,-18.3406,68.244,5.6433,-18.3406,68.244,5.6433,-18.3406,74.7135,0.0102,-18.3354,74.7135,5.6433,-18.3354,68.244,0.0102,-18.3406,62.163,0.0102,-16.1327,62.163,5.6433,-16.1327,62.163,5.6433,-16.1327,68.244,0.0102,-18.3406,68.244,5.6433,-18.3406,62.163,0.0102,-16.1327,57.2038,0.0102,-11.9781,57.2038,5.6433,-11.9781,57.2038,5.6433,-11.9781,62.163,0.0102,-16.1327,62.163,5.6433,-16.1327,57.2038,0.0102,-11.9781,53.9646,0.0102,-6.378,53.9646,5.6433,-6.378,53.9646,5.6433,-6.378,57.2038,0.0102,-11.9781,57.2038,5.6433,-11.9781,53.9646,0.0102,-6.378,52.8361,0.0102,-0.0077,52.8361,5.6433,-0.0077,52.8361,5.6433,-0.0077,53.9646,0.0102,-6.378,53.9646,5.6433,-6.378,53.9545,5.6433,6.3644,55.0565,8.2313,5.9643,54.0085,8.2313,-0.0067,54.0085,8.2313,-0.0067,53.9545,5.6433,6.3644,52.8361,5.6433,-0.0077,57.1848,5.6433,11.9697,58.0835,8.2313,11.2168,55.0565,8.2313,5.9643,55.0565,8.2313,5.9643,57.1848,5.6433,11.9697,53.9545,5.6433,6.3644,62.1374,5.6433,16.1321,62.7244,8.2313,15.1173,58.0835,8.2313,11.2168,58.0835,8.2313,11.2168,62.1374,5.6433,16.1321,57.1848,5.6433,11.9697,68.215,5.6433,18.3497,68.4194,8.2313,17.1953,62.7244,8.2313,15.1173,62.7244,8.2313,15.1173,68.215,5.6433,18.3497,62.1374,5.6433,16.1321,74.6844,5.6433,18.3548,74.4818,8.2313,17.2001,68.4194,8.2313,17.1953,68.4194,8.2313,17.1953,74.6844,5.6433,18.3548,68.215,5.6433,18.3497,80.7655,5.6433,16.1469,80.1801,8.2313,15.1311,74.4818,8.2313,17.2001,74.4818,8.2313,17.2001,80.7655,5.6433,16.1469,74.6844,5.6433,18.3548,85.7247,5.6433,11.9923,84.8272,8.2313,11.238,80.1801,8.2313,15.1311,80.1801,8.2313,15.1311,85.7247,5.6433,11.9923,80.7655,5.6433,16.1469,88.9639,5.6433,6.3922,87.8625,8.2313,5.9903,84.8272,8.2313,11.238,84.8272,8.2313,11.238,88.9639,5.6433,6.3922,85.7247,5.6433,11.9923,90.0923,5.6433,0.0219,88.92,8.2313,0.0209,87.8625,8.2313,5.9903,87.8625,8.2313,5.9903,90.0923,5.6433,0.0219,88.9639,5.6433,6.3922,88.974,5.6433,-6.3502,87.872,8.2313,-5.9501,88.92,8.2313,0.0209,88.92,8.2313,0.0209,88.974,5.6433,-6.3502,90.0923,5.6433,0.0219,85.7437,5.6433,-11.9555,84.845,8.2313,-11.2026,87.872,8.2313,-5.9501,87.872,8.2313,-5.9501,85.7437,5.6433,-11.9555,88.974,5.6433,-6.3502,80.7911,5.6433,-16.1179,80.2041,8.2313,-15.1031,84.845,8.2313,-11.2026,84.845,8.2313,-11.2026,80.7911,5.6433,-16.1179,85.7437,5.6433,-11.9555,74.7135,5.6433,-18.3354,74.509,8.2313,-17.181,80.2041,8.2313,-15.1031,80.2041,8.2313,-15.1031,74.7135,5.6433,-18.3354,80.7911,5.6433,-16.1179,68.244,5.6433,-18.3406,68.4467,8.2313,-17.1859,74.509,8.2313,-17.181,74.509,8.2313,-17.181,68.244,5.6433,-18.3406,74.7135,5.6433,-18.3354,62.163,5.6433,-16.1327,62.7483,8.2313,-15.1169,68.4467,8.2313,-17.1859,68.4467,8.2313,-17.1859,62.163,5.6433,-16.1327,68.244,5.6433,-18.3406,57.2038,5.6433,-11.9781,58.1012,8.2313,-11.2238,62.7483,8.2313,-15.1169,62.7483,8.2313,-15.1169,57.2038,5.6433,-11.9781,62.163,5.6433,-16.1327,53.9646,5.6433,-6.378,55.0659,8.2313,-5.9761,58.1012,8.2313,-11.2238,58.1012,8.2313,-11.2238,53.9646,5.6433,-6.378,57.2038,5.6433,-11.9781,52.8361,5.6433,-0.0077,54.0085,8.2313,-0.0067,55.0659,8.2313,-5.9761,55.0659,8.2313,-5.9761,52.8361,5.6433,-0.0077,53.9646,5.6433,-6.378,56.4642,10.0171,-0.0048,57.3648,10.0171,5.1262,57.8901,49.0637,4.9355,57.8901,49.0637,4.9355,56.4642,10.0171,-0.0048,57.0231,49.0637,-0.0043,57.3648,10.0171,5.1262,59.9659,10.0171,9.6398,60.3943,49.0637,9.2809,60.3943,49.0637,9.2809,57.3648,10.0171,5.1262,57.8901,49.0637,4.9355,59.9659,10.0171,9.6398,63.9539,10.0171,12.9915,64.2338,49.0637,12.5078,64.2338,49.0637,12.5078,59.9659,10.0171,9.6398,60.3943,49.0637,9.2809,63.9539,10.0171,12.9915,68.8478,10.0171,14.7772,68.9453,49.0637,14.2269,68.9453,49.0637,14.2269,63.9539,10.0171,12.9915,64.2338,49.0637,12.5078,68.8478,10.0171,14.7772,74.0573,10.0171,14.7813,73.9606,49.0637,14.2308,73.9606,49.0637,14.2308,68.8478,10.0171,14.7772,68.9453,49.0637,14.2269,74.0573,10.0171,14.7813,78.9539,10.0171,13.0034,78.6749,49.0637,12.5192,78.6749,49.0637,12.5192,74.0573,10.0171,14.7813,73.9606,49.0637,14.2308,78.9539,10.0171,13.0034,82.9473,10.0171,9.658,82.5194,49.0637,9.2984,82.5194,49.0637,9.2984,78.9539,10.0171,13.0034,78.6749,49.0637,12.5192,82.9473,10.0171,9.658,85.5555,10.0171,5.1486,85.0305,49.0637,4.957,85.0305,49.0637,4.957,82.9473,10.0171,9.658,82.5194,49.0637,9.2984,85.5555,10.0171,5.1486,86.4642,10.0171,0.019,85.9054,49.0637,0.0186,85.9054,49.0637,0.0186,85.5555,10.0171,5.1486,85.0305,49.0637,4.957,86.4642,10.0171,0.019,85.5637,10.0171,-5.112,85.0384,49.0637,-4.9213,85.0384,49.0637,-4.9213,86.4642,10.0171,0.019,85.9054,49.0637,0.0186,85.5637,10.0171,-5.112,82.9625,10.0171,-9.6256,82.5341,49.0637,-9.2667,82.5341,49.0637,-9.2667,85.5637,10.0171,-5.112,85.0384,49.0637,-4.9213,82.9625,10.0171,-9.6256,78.9745,10.0171,-12.9773,78.6947,49.0637,-12.4935,78.6947,49.0637,-12.4935,82.9625,10.0171,-9.6256,82.5341,49.0637,-9.2667,78.9745,10.0171,-12.9773,74.0807,10.0171,-14.7629,73.9832,49.0637,-14.2126,73.9832,49.0637,-14.2126,78.9745,10.0171,-12.9773,78.6947,49.0637,-12.4935,74.0807,10.0171,-14.7629,68.8712,10.0171,-14.7671,68.9678,49.0637,-14.2166,68.9678,49.0637,-14.2166,74.0807,10.0171,-14.7629,73.9832,49.0637,-14.2126,68.8712,10.0171,-14.7671,63.9745,10.0171,-12.9892,64.2536,49.0637,-12.505,64.2536,49.0637,-12.505,68.8712,10.0171,-14.7671,68.9678,49.0637,-14.2166,63.9745,10.0171,-12.9892,59.9812,10.0171,-9.6438,60.4091,49.0637,-9.2842,60.4091,49.0637,-9.2842,63.9745,10.0171,-12.9892,64.2536,49.0637,-12.505,59.9812,10.0171,-9.6438,57.3729,10.0171,-5.1344,57.8979,49.0637,-4.9428,57.8979,49.0637,-4.9428,59.9812,10.0171,-9.6438,60.4091,49.0637,-9.2842,57.3729,10.0171,-5.1344,56.4642,10.0171,-0.0048,57.0231,49.0637,-0.0043,57.0231,49.0637,-0.0043,57.3729,10.0171,-5.1344,57.8979,49.0637,-4.9428,57.8901,49.0637,4.9355,58.8556,51.7017,4.5849,58.0503,51.7017,-0.0035,58.0503,51.7017,-0.0035,57.8901,49.0637,4.9355,57.0231,49.0637,-0.0043,60.3943,49.0637,9.2809,61.1817,51.7017,8.6213,58.8556,51.7017,4.5849,58.8556,51.7017,4.5849,60.3943,49.0637,9.2809,57.8901,49.0637,4.9355,64.2338,49.0637,12.5078,64.7481,51.7017,11.6186,61.1817,51.7017,8.6213,61.1817,51.7017,8.6213,64.2338,49.0637,12.5078,60.3943,49.0637,9.2809,68.9453,49.0637,14.2269,69.1245,51.7017,13.2154,64.7481,51.7017,11.6186,64.7481,51.7017,11.6186,68.9453,49.0637,14.2269,64.2338,49.0637,12.5078,73.9606,49.0637,14.2308,73.7831,51.7017,13.2191,69.1245,51.7017,13.2154,69.1245,51.7017,13.2154,73.9606,49.0637,14.2308,68.9453,49.0637,14.2269,78.6749,49.0637,12.5192,78.162,51.7017,11.6292,73.7831,51.7017,13.2191,73.7831,51.7017,13.2191,78.6749,49.0637,12.5192,73.9606,49.0637,14.2308,82.5194,49.0637,9.2984,81.7331,51.7017,8.6375,78.162,51.7017,11.6292,78.162,51.7017,11.6292,82.5194,49.0637,9.2984,78.6749,49.0637,12.5192,85.0305,49.0637,4.957,84.0656,51.7017,4.6049,81.7331,51.7017,8.6375,81.7331,51.7017,8.6375,85.0305,49.0637,4.957,82.5194,49.0637,9.2984,85.9054,49.0637,0.0186,84.8781,51.7017,0.0177,84.0656,51.7017,4.6049,84.0656,51.7017,4.6049,85.9054,49.0637,0.0186,85.0305,49.0637,4.957,85.0384,49.0637,-4.9213,84.0728,51.7017,-4.5707,84.8781,51.7017,0.0177,84.8781,51.7017,0.0177,85.0384,49.0637,-4.9213,85.9054,49.0637,0.0186,82.5341,49.0637,-9.2667,81.7467,51.7017,-8.607,84.0728,51.7017,-4.5707,84.0728,51.7017,-4.5707,82.5341,49.0637,-9.2667,85.0384,49.0637,-4.9213,78.6947,49.0637,-12.4935,78.1804,51.7017,-11.6044,81.7467,51.7017,-8.607,81.7467,51.7017,-8.607,78.6947,49.0637,-12.4935,82.5341,49.0637,-9.2667,73.9832,49.0637,-14.2126,73.804,51.7017,-13.2012,78.1804,51.7017,-11.6044,78.1804,51.7017,-11.6044,73.9832,49.0637,-14.2126,78.6947,49.0637,-12.4935,68.9678,49.0637,-14.2166,69.1454,51.7017,-13.2049,73.804,51.7017,-13.2012,73.804,51.7017,-13.2012,68.9678,49.0637,-14.2166,73.9832,49.0637,-14.2126,64.2536,49.0637,-12.505,64.7665,51.7017,-11.615,69.1454,51.7017,-13.2049,69.1454,51.7017,-13.2049,64.2536,49.0637,-12.505,68.9678,49.0637,-14.2166,60.4091,49.0637,-9.2842,61.1954,51.7017,-8.6233,64.7665,51.7017,-11.615,64.7665,51.7017,-11.615,60.4091,49.0637,-9.2842,64.2536,49.0637,-12.505,57.8979,49.0637,-4.9428,58.8629,51.7017,-4.5907,61.1954,51.7017,-8.6233,61.1954,51.7017,-8.6233,57.8979,49.0637,-4.9428,60.4091,49.0637,-9.2842,57.0231,49.0637,-0.0043,58.0503,51.7017,-0.0035,58.8629,51.7017,-4.5907,58.8629,51.7017,-4.5907,57.0231,49.0637,-0.0043,57.8979,49.0637,-4.9428,60.5748,53.5917,-0.0015,61.2285,53.5917,3.7234,61.2285,151.0146,3.7234,61.2285,151.0146,3.7234,60.5748,53.5917,-0.0015,60.5748,151.0146,-0.0015,61.2285,53.5917,3.7234,63.1169,53.5917,7.0001,63.1169,151.0146,7.0001,63.1169,151.0146,7.0001,61.2285,53.5917,3.7234,61.2285,151.0146,3.7234,63.1169,53.5917,7.0001,66.012,53.5917,9.4333,66.012,151.0146,9.4333,66.012,151.0146,9.4333,63.1169,53.5917,7.0001,63.1169,151.0146,7.0001,66.012,53.5917,9.4333,69.5648,53.5917,10.7296,69.5648,151.0146,10.7296,69.5648,151.0146,10.7296,66.012,53.5917,9.4333,66.012,151.0146,9.4333,69.5648,53.5917,10.7296,73.3467,53.5917,10.7326,73.3467,151.0146,10.7326,73.3467,151.0146,10.7326,69.5648,53.5917,10.7296,69.5648,151.0146,10.7296,73.3467,53.5917,10.7326,76.9015,53.5917,9.442,76.9015,151.0146,9.442,76.9015,151.0146,9.442,73.3467,53.5917,10.7326,73.3467,151.0146,10.7326,76.9015,53.5917,9.442,79.8005,53.5917,7.0133,79.8005,151.0146,7.0133,79.8005,151.0146,7.0133,76.9015,53.5917,9.442,76.9015,151.0146,9.442,79.8005,53.5917,7.0133,81.694,53.5917,3.7396,81.694,151.0146,3.7396,81.694,151.0146,3.7396,79.8005,53.5917,7.0133,79.8005,151.0146,7.0133,81.694,53.5917,3.7396,82.3537,53.5917,0.0157,82.3537,151.0146,0.0157,82.3537,151.0146,0.0157,81.694,53.5917,3.7396,81.694,151.0146,3.7396,82.3537,53.5917,0.0157,81.6999,53.5917,-3.7092,81.6999,151.0146,-3.7092,81.6999,151.0146,-3.7092,82.3537,53.5917,0.0157,82.3537,151.0146,0.0157,81.6999,53.5917,-3.7092,79.8116,53.5917,-6.9859,79.8116,151.0146,-6.9859,79.8116,151.0146,-6.9859,81.6999,53.5917,-3.7092,81.6999,151.0146,-3.7092,79.8116,53.5917,-6.9859,76.9164,53.5917,-9.4191,76.9164,151.0146,-9.4191,76.9164,151.0146,-9.4191,79.8116,53.5917,-6.9859,79.8116,151.0146,-6.9859,76.9164,53.5917,-9.4191,73.3637,53.5917,-10.7154,73.3637,151.0146,-10.7154,73.3637,151.0146,-10.7154,76.9164,53.5917,-9.4191,76.9164,151.0146,-9.4191,73.3637,53.5917,-10.7154,69.5818,53.5917,-10.7184,69.5818,151.0146,-10.7184,69.5818,151.0146,-10.7184,73.3637,53.5917,-10.7154,73.3637,151.0146,-10.7154,69.5818,53.5917,-10.7184,66.027,53.5917,-9.4278,66.027,151.0146,-9.4278,66.027,151.0146,-9.4278,69.5818,53.5917,-10.7184,69.5818,151.0146,-10.7184,66.027,53.5917,-9.4278,63.128,53.5917,-6.9991,63.128,151.0146,-6.9991,63.128,151.0146,-6.9991,66.027,53.5917,-9.4278,66.027,151.0146,-9.4278,63.128,53.5917,-6.9991,61.2344,53.5917,-3.7254,61.2344,151.0146,-3.7254,61.2344,151.0146,-3.7254,63.128,53.5917,-6.9991,63.128,151.0146,-6.9991,61.2344,53.5917,-3.7254,60.5748,53.5917,-0.0015,60.5748,151.0146,-0.0015,60.5748,151.0146,-0.0015,61.2344,53.5917,-3.7254,61.2344,151.0146,-3.7254,60.5748,151.0146,-0.0015,61.2285,151.0146,3.7234,61.2285,165.5339,3.7234,61.2285,165.5339,3.7234,60.5748,151.0146,-0.0015,60.5748,165.5339,-0.0015,61.2285,151.0146,3.7234,63.1169,151.0146,7.0001,63.1169,165.5339,7.0001,63.1169,165.5339,7.0001,61.2285,151.0146,3.7234,61.2285,165.5339,3.7234,63.1169,151.0146,7.0001,66.012,151.0146,9.4333,66.012,165.5339,9.4333,66.012,165.5339,9.4333,63.1169,151.0146,7.0001,63.1169,165.5339,7.0001,66.012,151.0146,9.4333,69.5648,151.0146,10.7296,69.5648,165.5339,10.7296,69.5648,165.5339,10.7296,66.012,151.0146,9.4333,66.012,165.5339,9.4333,69.5648,151.0146,10.7296,73.3467,151.0146,10.7326,73.3467,165.5339,10.7326,73.3467,165.5339,10.7326,69.5648,151.0146,10.7296,69.5648,165.5339,10.7296,73.3467,151.0146,10.7326,76.9015,151.0146,9.442,76.9015,165.5339,9.442,76.9015,165.5339,9.442,73.3467,151.0146,10.7326,73.3467,165.5339,10.7326,76.9015,151.0146,9.442,79.8005,151.0146,7.0133,79.8005,165.5339,7.0133,79.8005,165.5339,7.0133,76.9015,151.0146,9.442,76.9015,165.5339,9.442,79.8005,151.0146,7.0133,81.694,151.0146,3.7396,81.694,165.5339,3.7396,81.694,165.5339,3.7396,79.8005,151.0146,7.0133,79.8005,165.5339,7.0133,81.694,151.0146,3.7396,82.3537,151.0146,0.0157,82.3537,165.5339,0.0157,82.3537,165.5339,0.0157,81.694,151.0146,3.7396,81.694,165.5339,3.7396,82.3537,151.0146,0.0157,81.6999,151.0146,-3.7092,81.6999,165.5339,-3.7092,81.6999,165.5339,-3.7092,82.3537,151.0146,0.0157,82.3537,165.5339,0.0157,81.6999,151.0146,-3.7092,79.8116,151.0146,-6.9859,79.8116,165.5339,-6.9859,79.8116,165.5339,-6.9859,81.6999,151.0146,-3.7092,81.6999,165.5339,-3.7092,79.8116,151.0146,-6.9859,76.9164,151.0146,-9.4191,76.9164,165.5339,-9.4191,76.9164,165.5339,-9.4191,79.8116,151.0146,-6.9859,79.8116,165.5339,-6.9859,76.9164,151.0146,-9.4191,73.3637,151.0146,-10.7154,73.3637,165.5339,-10.7154,73.3637,165.5339,-10.7154,76.9164,151.0146,-9.4191,76.9164,165.5339,-9.4191,73.3637,151.0146,-10.7154,69.5818,151.0146,-10.7184,69.5818,165.5339,-10.7184,69.5818,165.5339,-10.7184,73.3637,151.0146,-10.7154,73.3637,165.5339,-10.7154,69.5818,151.0146,-10.7184,66.027,151.0146,-9.4278,66.027,165.5339,-9.4278,66.027,165.5339,-9.4278,69.5818,151.0146,-10.7184,69.5818,165.5339,-10.7184,66.027,151.0146,-9.4278,63.128,151.0146,-6.9991,63.128,165.5339,-6.9991,63.128,165.5339,-6.9991,66.027,151.0146,-9.4278,66.027,165.5339,-9.4278,63.128,151.0146,-6.9991,61.2344,151.0146,-3.7254,61.2344,165.5339,-3.7254,61.2344,165.5339,-3.7254,63.128,151.0146,-6.9991,63.128,165.5339,-6.9991,61.2344,151.0146,-3.7254,60.5748,151.0146,-0.0015,60.5748,165.5339,-0.0015,60.5748,165.5339,-0.0015,61.2344,151.0146,-3.7254,61.2344,165.5339,-3.7254,61.2285,165.5339,3.7234,63.1169,165.5339,7.0001,63.1169,180.0531,7.0001,63.1169,180.0531,7.0001,61.2285,165.5339,3.7234,61.2285,180.0531,3.7234,63.1169,165.5339,7.0001,66.012,165.5339,9.4333,66.012,180.0531,9.4333,66.012,180.0531,9.4333,63.1169,165.5339,7.0001,63.1169,180.0531,7.0001,66.012,165.5339,9.4333,69.5648,165.5339,10.7296,69.5648,180.0531,10.7296,69.5648,180.0531,10.7296,66.012,165.5339,9.4333,66.012,180.0531,9.4333,69.5648,165.5339,10.7296,73.3467,165.5339,10.7326,73.3467,180.0531,10.7326,73.3467,180.0531,10.7326,69.5648,165.5339,10.7296,69.5648,180.0531,10.7296,73.3467,165.5339,10.7326,76.9015,165.5339,9.442,76.9015,180.0531,9.442,76.9015,180.0531,9.442,73.3467,165.5339,10.7326,73.3467,180.0531,10.7326,76.9015,165.5339,9.442,79.8005,165.5339,7.0133,79.8005,180.0531,7.0133,79.8005,180.0531,7.0133,76.9015,165.5339,9.442,76.9015,180.0531,9.442,79.8005,165.5339,7.0133,81.694,165.5339,3.7396,81.694,180.0531,3.7396,81.694,180.0531,3.7396,79.8005,165.5339,7.0133,79.8005,180.0531,7.0133,94.368,165.5339,3.7476,94.3643,165.5339,0.0231,98.4253,180.0531,0.0231,98.4253,180.0531,0.0231,94.368,165.5339,3.7476,98.4292,180.0531,3.7476,94.3643,165.5339,0.0231,94.3741,165.5339,-3.7013,98.4355,180.0531,-3.7013,98.4355,180.0531,-3.7013,94.3643,165.5339,0.0231,98.4253,180.0531,0.0231,81.6999,165.5339,-3.7092,79.8116,165.5339,-6.9859,79.8116,180.0531,-6.9859,79.8116,180.0531,-6.9859,81.6999,165.5339,-3.7092,81.6999,180.0531,-3.7092,79.8116,165.5339,-6.9859,76.9164,165.5339,-9.4191,76.9164,180.0531,-9.4191,76.9164,180.0531,-9.4191,79.8116,165.5339,-6.9859,79.8116,180.0531,-6.9859,76.9164,165.5339,-9.4191,73.3637,165.5339,-10.7154,73.3637,180.0531,-10.7154,73.3637,180.0531,-10.7154,76.9164,165.5339,-9.4191,76.9164,180.0531,-9.4191,73.3637,165.5339,-10.7154,69.5818,165.5339,-10.7184,69.5818,180.0531,-10.7184,69.5818,180.0531,-10.7184,73.3637,165.5339,-10.7154,73.3637,180.0531,-10.7154,69.5818,165.5339,-10.7184,66.027,165.5339,-9.4278,66.027,180.0531,-9.4278,66.027,180.0531,-9.4278,69.5818,165.5339,-10.7184,69.5818,180.0531,-10.7184,66.027,165.5339,-9.4278,63.128,165.5339,-6.9991,63.128,180.0531,-6.9991,63.128,180.0531,-6.9991,66.027,165.5339,-9.4278,66.027,180.0531,-9.4278,63.128,165.5339,-6.9991,61.2344,165.5339,-3.7254,61.2344,180.0531,-3.7254,61.2344,180.0531,-3.7254,63.128,165.5339,-6.9991,63.128,180.0531,-6.9991,60.5748,180.0531,-0.0015,61.2285,180.0531,3.7234,61.2285,194.5724,3.7234,61.2285,194.5724,3.7234,60.5748,180.0531,-0.0015,60.5748,194.5724,-0.0015,61.2285,180.0531,3.7234,63.1169,180.0531,7.0001,63.1169,194.5724,7.0001,63.1169,194.5724,7.0001,61.2285,180.0531,3.7234,61.2285,194.5724,3.7234,63.1169,180.0531,7.0001,66.012,180.0531,9.4333,66.012,194.5724,9.4333,66.012,194.5724,9.4333,63.1169,180.0531,7.0001,63.1169,194.5724,7.0001,66.012,180.0531,9.4333,69.5648,180.0531,10.7296,69.5648,194.5724,10.7296,69.5648,194.5724,10.7296,66.012,180.0531,9.4333,66.012,194.5724,9.4333,69.5648,180.0531,10.7296,73.3467,180.0531,10.7326,73.3467,194.5724,10.7326,73.3467,194.5724,10.7326,69.5648,180.0531,10.7296,69.5648,194.5724,10.7296,73.3467,180.0531,10.7326,76.9015,180.0531,9.442,76.9015,194.5724,9.442,76.9015,194.5724,9.442,73.3467,180.0531,10.7326,73.3467,194.5724,10.7326,76.9015,180.0531,9.442,79.8005,180.0531,7.0133,79.8005,194.5724,7.0133,79.8005,194.5724,7.0133,76.9015,180.0531,9.442,76.9015,194.5724,9.442,79.8005,180.0531,7.0133,81.694,180.0531,3.7396,81.694,194.5724,3.7396,81.694,194.5724,3.7396,79.8005,180.0531,7.0133,79.8005,194.5724,7.0133,81.694,180.0531,3.7396,82.3537,180.0531,0.0157,82.3537,194.5724,0.0157,82.3537,194.5724,0.0157,81.694,180.0531,3.7396,81.694,194.5724,3.7396,82.3537,180.0531,0.0157,81.6999,180.0531,-3.7092,81.6999,194.5724,-3.7092,81.6999,194.5724,-3.7092,82.3537,180.0531,0.0157,82.3537,194.5724,0.0157,81.6999,180.0531,-3.7092,79.8116,180.0531,-6.9859,79.8116,194.5724,-6.9859,79.8116,194.5724,-6.9859,81.6999,180.0531,-3.7092,81.6999,194.5724,-3.7092,79.8116,180.0531,-6.9859,76.9164,180.0531,-9.4191,76.9164,194.5724,-9.4191,76.9164,194.5724,-9.4191,79.8116,180.0531,-6.9859,79.8116,194.5724,-6.9859,76.9164,180.0531,-9.4191,73.3637,180.0531,-10.7154,73.3637,194.5724,-10.7154,73.3637,194.5724,-10.7154,76.9164,180.0531,-9.4191,76.9164,194.5724,-9.4191,73.3637,180.0531,-10.7154,69.5818,180.0531,-10.7184,69.5818,194.5724,-10.7184,69.5818,194.5724,-10.7184,73.3637,180.0531,-10.7154,73.3637,194.5724,-10.7154,69.5818,180.0531,-10.7184,66.027,180.0531,-9.4278,66.027,194.5724,-9.4278,66.027,194.5724,-9.4278,69.5818,180.0531,-10.7184,69.5818,194.5724,-10.7184,66.027,180.0531,-9.4278,63.128,180.0531,-6.9991,63.128,194.5724,-6.9991,63.128,194.5724,-6.9991,66.027,180.0531,-9.4278,66.027,194.5724,-9.4278,63.128,180.0531,-6.9991,61.2344,180.0531,-3.7254,61.2344,194.5724,-3.7254,61.2344,194.5724,-3.7254,63.128,180.0531,-6.9991,63.128,194.5724,-6.9991,61.2344,180.0531,-3.7254,60.5748,180.0531,-0.0015,60.5748,194.5724,-0.0015,60.5748,194.5724,-0.0015,61.2344,180.0531,-3.7254,61.2344,194.5724,-3.7254,57.2038,0.0102,-11.9781,62.163,0.0102,-16.1327,68.244,0.0102,-18.3406,68.244,0.0102,-18.3406,57.2038,0.0102,-11.9781,52.8361,0.0102,-0.0077,68.244,0.0102,-18.3406,74.7135,0.0102,-18.3354,80.7911,0.0102,-16.1179,80.7911,0.0102,-16.1179,68.244,0.0102,-18.3406,88.974,0.0102,-6.3502,80.7911,0.0102,-16.1179,85.7437,0.0102,-11.9555,88.974,0.0102,-6.3502,88.974,0.0102,-6.3502,90.0923,0.0102,0.0219,88.9639,0.0102,6.3922,88.9639,0.0102,6.3922,88.974,0.0102,-6.3502,80.7655,0.0102,16.1469,88.9639,0.0102,6.3922,85.7247,0.0102,11.9923,80.7655,0.0102,16.1469,80.7655,0.0102,16.1469,74.6844,0.0102,18.3548,68.215,0.0102,18.3497,68.215,0.0102,18.3497,80.7655,0.0102,16.1469,57.1848,0.0102,11.9697,68.215,0.0102,18.3497,62.1374,0.0102,16.1321,57.1848,0.0102,11.9697,88.974,0.0102,-6.3502,80.7655,0.0102,16.1469,57.1848,0.0102,11.9697,57.1848,0.0102,11.9697,88.974,0.0102,-6.3502,68.244,0.0102,-18.3406,57.1848,0.0102,11.9697,53.9545,0.0102,6.3644,52.8361,0.0102,-0.0077,52.8361,0.0102,-0.0077,57.1848,0.0102,11.9697,68.244,0.0102,-18.3406,53.9646,0.0102,-6.378,57.2038,0.0102,-11.9781,52.8361,0.0102,-0.0077,61.2285,53.5917,3.7234,60.5748,53.5917,-0.0015,58.0503,51.7017,-0.0035,58.0503,51.7017,-0.0035,61.2285,53.5917,3.7234,58.8556,51.7017,4.5849,63.1169,53.5917,7.0001,61.2285,53.5917,3.7234,58.8556,51.7017,4.5849,58.8556,51.7017,4.5849,63.1169,53.5917,7.0001,61.1817,51.7017,8.6213,66.012,53.5917,9.4333,63.1169,53.5917,7.0001,61.1817,51.7017,8.6213,61.1817,51.7017,8.6213,66.012,53.5917,9.4333,64.7481,51.7017,11.6186,69.5648,53.5917,10.7296,66.012,53.5917,9.4333,64.7481,51.7017,11.6186,64.7481,51.7017,11.6186,69.5648,53.5917,10.7296,69.1245,51.7017,13.2154,73.3467,53.5917,10.7326,69.5648,53.5917,10.7296,69.1245,51.7017,13.2154,69.1245,51.7017,13.2154,73.3467,53.5917,10.7326,73.7831,51.7017,13.2191,76.9015,53.5917,9.442,73.3467,53.5917,10.7326,73.7831,51.7017,13.2191,73.7831,51.7017,13.2191,76.9015,53.5917,9.442,78.162,51.7017,11.6292,79.8005,53.5917,7.0133,76.9015,53.5917,9.442,78.162,51.7017,11.6292,78.162,51.7017,11.6292,79.8005,53.5917,7.0133,81.7331,51.7017,8.6375,81.694,53.5917,3.7396,79.8005,53.5917,7.0133,81.7331,51.7017,8.6375,81.7331,51.7017,8.6375,81.694,53.5917,3.7396,84.0656,51.7017,4.6049,82.3537,53.5917,0.0157,81.694,53.5917,3.7396,84.0656,51.7017,4.6049,84.0656,51.7017,4.6049,82.3537,53.5917,0.0157,84.8781,51.7017,0.0177,81.6999,53.5917,-3.7092,82.3537,53.5917,0.0157,84.8781,51.7017,0.0177,84.8781,51.7017,0.0177,81.6999,53.5917,-3.7092,84.0728,51.7017,-4.5707,79.8116,53.5917,-6.9859,81.6999,53.5917,-3.7092,84.0728,51.7017,-4.5707,84.0728,51.7017,-4.5707,79.8116,53.5917,-6.9859,81.7467,51.7017,-8.607,76.9164,53.5917,-9.4191,79.8116,53.5917,-6.9859,81.7467,51.7017,-8.607,81.7467,51.7017,-8.607,76.9164,53.5917,-9.4191,78.1804,51.7017,-11.6044,73.3637,53.5917,-10.7154,76.9164,53.5917,-9.4191,78.1804,51.7017,-11.6044,78.1804,51.7017,-11.6044,73.3637,53.5917,-10.7154,73.804,51.7017,-13.2012,69.5818,53.5917,-10.7184,73.3637,53.5917,-10.7154,73.804,51.7017,-13.2012,73.804,51.7017,-13.2012,69.5818,53.5917,-10.7184,69.1454,51.7017,-13.2049,66.027,53.5917,-9.4278,69.5818,53.5917,-10.7184,69.1454,51.7017,-13.2049,69.1454,51.7017,-13.2049,66.027,53.5917,-9.4278,64.7665,51.7017,-11.615,63.128,53.5917,-6.9991,66.027,53.5917,-9.4278,64.7665,51.7017,-11.615,64.7665,51.7017,-11.615,63.128,53.5917,-6.9991,61.1954,51.7017,-8.6233,61.2344,53.5917,-3.7254,63.128,53.5917,-6.9991,61.1954,51.7017,-8.6233,61.1954,51.7017,-8.6233,61.2344,53.5917,-3.7254,58.8629,51.7017,-4.5907,60.5748,53.5917,-0.0015,61.2344,53.5917,-3.7254,58.8629,51.7017,-4.5907,58.8629,51.7017,-4.5907,60.5748,53.5917,-0.0015,58.0503,51.7017,-0.0035,57.3648,10.0171,5.1262,56.4642,10.0171,-0.0048,54.0085,8.2313,-0.0067,54.0085,8.2313,-0.0067,57.3648,10.0171,5.1262,55.0565,8.2313,5.9643,59.9659,10.0171,9.6398,57.3648,10.0171,5.1262,55.0565,8.2313,5.9643,55.0565,8.2313,5.9643,59.9659,10.0171,9.6398,58.0835,8.2313,11.2168,63.9539,10.0171,12.9915,59.9659,10.0171,9.6398,58.0835,8.2313,11.2168,58.0835,8.2313,11.2168,63.9539,10.0171,12.9915,62.7244,8.2313,15.1173,68.8478,10.0171,14.7772,63.9539,10.0171,12.9915,62.7244,8.2313,15.1173,62.7244,8.2313,15.1173,68.8478,10.0171,14.7772,68.4194,8.2313,17.1953,74.0573,10.0171,14.7813,68.8478,10.0171,14.7772,68.4194,8.2313,17.1953,68.4194,8.2313,17.1953,74.0573,10.0171,14.7813,74.4818,8.2313,17.2001,78.9539,10.0171,13.0034,74.0573,10.0171,14.7813,74.4818,8.2313,17.2001,74.4818,8.2313,17.2001,78.9539,10.0171,13.0034,80.1801,8.2313,15.1311,82.9473,10.0171,9.658,78.9539,10.0171,13.0034,80.1801,8.2313,15.1311,80.1801,8.2313,15.1311,82.9473,10.0171,9.658,84.8272,8.2313,11.238,85.5555,10.0171,5.1486,82.9473,10.0171,9.658,84.8272,8.2313,11.238,84.8272,8.2313,11.238,85.5555,10.0171,5.1486,87.8625,8.2313,5.9903,86.4642,10.0171,0.019,85.5555,10.0171,5.1486,87.8625,8.2313,5.9903,87.8625,8.2313,5.9903,86.4642,10.0171,0.019,88.92,8.2313,0.0209,85.5637,10.0171,-5.112,86.4642,10.0171,0.019,88.92,8.2313,0.0209,88.92,8.2313,0.0209,85.5637,10.0171,-5.112,87.872,8.2313,-5.9501,82.9625,10.0171,-9.6256,85.5637,10.0171,-5.112,87.872,8.2313,-5.9501,87.872,8.2313,-5.9501,82.9625,10.0171,-9.6256,84.845,8.2313,-11.2026,78.9745,10.0171,-12.9773,82.9625,10.0171,-9.6256,84.845,8.2313,-11.2026,84.845,8.2313,-11.2026,78.9745,10.0171,-12.9773,80.2041,8.2313,-15.1031,74.0807,10.0171,-14.7629,78.9745,10.0171,-12.9773,80.2041,8.2313,-15.1031,80.2041,8.2313,-15.1031,74.0807,10.0171,-14.7629,74.509,8.2313,-17.181,68.8712,10.0171,-14.7671,74.0807,10.0171,-14.7629,74.509,8.2313,-17.181,74.509,8.2313,-17.181,68.8712,10.0171,-14.7671,68.4467,8.2313,-17.1859,63.9745,10.0171,-12.9892,68.8712,10.0171,-14.7671,68.4467,8.2313,-17.1859,68.4467,8.2313,-17.1859,63.9745,10.0171,-12.9892,62.7483,8.2313,-15.1169,59.9812,10.0171,-9.6438,63.9745,10.0171,-12.9892,62.7483,8.2313,-15.1169,62.7483,8.2313,-15.1169,59.9812,10.0171,-9.6438,58.1012,8.2313,-11.2238,57.3729,10.0171,-5.1344,59.9812,10.0171,-9.6438,58.1012,8.2313,-11.2238,58.1012,8.2313,-11.2238,57.3729,10.0171,-5.1344,55.0659,8.2313,-5.9761,56.4642,10.0171,-0.0048,57.3729,10.0171,-5.1344,55.0659,8.2313,-5.9761,55.0659,8.2313,-5.9761,56.4642,10.0171,-0.0048,54.0085,8.2313,-0.0067,81.694,165.5339,3.7396,82.3537,165.5339,0.0157,94.3643,165.5339,0.0231,94.3643,165.5339,0.0231,81.694,165.5339,3.7396,94.368,165.5339,3.7476,82.3537,165.5339,0.0157,81.6999,165.5339,-3.7092,94.3741,165.5339,-3.7013,94.3741,165.5339,-3.7013,82.3537,165.5339,0.0157,94.3643,165.5339,0.0231,81.6999,165.5339,-3.7092,81.6999,180.0531,-3.7092,98.4355,180.0531,-3.7013,98.4355,180.0531,-3.7013,81.6999,165.5339,-3.7092,94.3741,165.5339,-3.7013,81.6999,180.0531,-3.7092,82.3537,180.0531,0.0157,98.4253,180.0531,0.0231,98.4253,180.0531,0.0231,81.6999,180.0531,-3.7092,98.4355,180.0531,-3.7013,82.3537,180.0531,0.0157,81.694,180.0531,3.7396,98.4292,180.0531,3.7476,98.4292,180.0531,3.7476,82.3537,180.0531,0.0157,98.4253,180.0531,0.0231,81.694,180.0531,3.7396,81.694,165.5339,3.7396,94.368,165.5339,3.7476,94.368,165.5339,3.7476,81.694,180.0531,3.7396,98.4292,180.0531,3.7476,-57.7008,0.0102,0.0071,-58.8243,0.0102,-6.3641,-58.8243,5.6433,-6.3641,-58.8243,5.6433,-6.3641,-57.7008,0.0102,0.0071,-57.7008,5.6433,0.0071,-58.8243,0.0102,-6.3641,-62.059,0.0102,-11.9668,-62.059,5.6433,-11.9668,-62.059,5.6433,-11.9668,-58.8243,0.0102,-6.3641,-58.8243,5.6433,-6.3641,-62.059,0.0102,-11.9668,-67.0149,0.0102,-16.1253,-67.0149,5.6433,-16.1253,-67.0149,5.6433,-16.1253,-62.059,0.0102,-11.9668,-62.059,5.6433,-11.9668,-67.0149,0.0102,-16.1253,-73.0942,0.0102,-18.338,-73.0942,5.6433,-18.338,-73.0942,5.6433,-18.338,-67.0149,0.0102,-16.1253,-67.0149,5.6433,-16.1253,-73.0942,0.0102,-18.338,-79.5637,0.0102,-18.338,-79.5637,5.6433,-18.338,-79.5637,5.6433,-18.338,-73.0942,0.0102,-18.338,-73.0942,5.6433,-18.338,-79.5637,0.0102,-18.338,-85.643,0.0102,-16.1253,-85.643,5.6433,-16.1253,-85.643,5.6433,-16.1253,-79.5637,0.0102,-18.338,-79.5637,5.6433,-18.338,-85.643,0.0102,-16.1253,-90.599,0.0102,-11.9668,-90.599,5.6433,-11.9668,-90.599,5.6433,-11.9668,-85.643,0.0102,-16.1253,-85.643,5.6433,-16.1253,-90.599,0.0102,-11.9668,-93.8337,0.0102,-6.3641,-93.8337,5.6433,-6.3641,-93.8337,5.6433,-6.3641,-90.599,0.0102,-11.9668,-90.599,5.6433,-11.9668,-93.8337,0.0102,-6.3641,-94.9571,0.0102,0.0071,-94.9571,5.6433,0.0071,-94.9571,5.6433,0.0071,-93.8337,0.0102,-6.3641,-93.8337,5.6433,-6.3641,-94.9571,0.0102,0.0071,-93.8337,0.0102,6.3783,-93.8337,5.6433,6.3783,-93.8337,5.6433,6.3783,-94.9571,0.0102,0.0071,-94.9571,5.6433,0.0071,-93.8337,0.0102,6.3783,-90.599,0.0102,11.981,-90.599,5.6433,11.981,-90.599,5.6433,11.981,-93.8337,0.0102,6.3783,-93.8337,5.6433,6.3783,-90.599,0.0102,11.981,-85.6431,0.0102,16.1395,-85.6431,5.6433,16.1395,-85.6431,5.6433,16.1395,-90.599,0.0102,11.981,-90.599,5.6433,11.981,-85.6431,0.0102,16.1395,-79.5637,0.0102,18.3522,-79.5637,5.6433,18.3522,-79.5637,5.6433,18.3522,-85.6431,0.0102,16.1395,-85.6431,5.6433,16.1395,-79.5637,0.0102,18.3522,-73.0943,0.0102,18.3522,-73.0943,5.6433,18.3522,-73.0943,5.6433,18.3522,-79.5637,0.0102,18.3522,-79.5637,5.6433,18.3522,-73.0943,0.0102,18.3522,-67.0149,0.0102,16.1395,-67.0149,5.6433,16.1395,-67.0149,5.6433,16.1395,-73.0943,0.0102,18.3522,-73.0943,5.6433,18.3522,-67.0149,0.0102,16.1395,-62.059,0.0102,11.981,-62.059,5.6433,11.981,-62.059,5.6433,11.981,-67.0149,0.0102,16.1395,-67.0149,5.6433,16.1395,-62.059,0.0102,11.981,-58.8243,0.0102,6.3783,-58.8243,5.6433,6.3783,-58.8243,5.6433,6.3783,-62.059,0.0102,11.981,-62.059,5.6433,11.981,-58.8243,0.0102,6.3783,-57.7008,0.0102,0.0071,-57.7008,5.6433,0.0071,-57.7008,5.6433,0.0071,-58.8243,0.0102,6.3783,-58.8243,5.6433,6.3783,-58.8243,5.6433,-6.3641,-59.9259,8.2313,-5.9631,-58.8732,8.2313,0.0071,-58.8732,8.2313,0.0071,-58.8243,5.6433,-6.3641,-57.7008,5.6433,0.0071,-62.059,5.6433,-11.9668,-62.9571,8.2313,-11.2132,-59.9259,8.2313,-5.9631,-59.9259,8.2313,-5.9631,-62.059,5.6433,-11.9668,-58.8243,5.6433,-6.3641,-67.0149,5.6433,-16.1253,-67.6011,8.2313,-15.11,-62.9571,8.2313,-11.2132,-62.9571,8.2313,-11.2132,-67.0149,5.6433,-16.1253,-62.059,5.6433,-11.9668,-73.0942,5.6433,-18.338,-73.2978,8.2313,-17.1835,-67.6011,8.2313,-15.11,-67.6011,8.2313,-15.11,-73.0942,5.6433,-18.338,-67.0149,5.6433,-16.1253,-79.5637,5.6433,-18.338,-79.3601,8.2313,-17.1835,-73.2978,8.2313,-17.1835,-73.2978,8.2313,-17.1835,-79.5637,5.6433,-18.338,-73.0942,5.6433,-18.338,-85.643,5.6433,-16.1253,-85.0569,8.2313,-15.11,-79.3601,8.2313,-17.1835,-79.3601,8.2313,-17.1835,-85.643,5.6433,-16.1253,-79.5637,5.6433,-18.338,-90.599,5.6433,-11.9668,-89.7009,8.2313,-11.2132,-85.0569,8.2313,-15.11,-85.0569,8.2313,-15.11,-90.599,5.6433,-11.9668,-85.643,5.6433,-16.1253,-93.8337,5.6433,-6.3641,-92.732,8.2313,-5.9631,-89.7009,8.2313,-11.2132,-89.7009,8.2313,-11.2132,-93.8337,5.6433,-6.3641,-90.599,5.6433,-11.9668,-94.9571,5.6433,0.0071,-93.7847,8.2313,0.0071,-92.732,8.2313,-5.9631,-92.732,8.2313,-5.9631,-94.9571,5.6433,0.0071,-93.8337,5.6433,-6.3641,-93.8337,5.6433,6.3783,-92.732,8.2313,5.9773,-93.7847,8.2313,0.0071,-93.7847,8.2313,0.0071,-93.8337,5.6433,6.3783,-94.9571,5.6433,0.0071,-90.599,5.6433,11.981,-89.7009,8.2313,11.2274,-92.732,8.2313,5.9773,-92.732,8.2313,5.9773,-90.599,5.6433,11.981,-93.8337,5.6433,6.3783,-85.6431,5.6433,16.1395,-85.0569,8.2313,15.1242,-89.7009,8.2313,11.2274,-89.7009,8.2313,11.2274,-85.6431,5.6433,16.1395,-90.599,5.6433,11.981,-79.5637,5.6433,18.3522,-79.3602,8.2313,17.1977,-85.0569,8.2313,15.1242,-85.0569,8.2313,15.1242,-79.5637,5.6433,18.3522,-85.6431,5.6433,16.1395,-73.0943,5.6433,18.3522,-73.2978,8.2313,17.1977,-79.3602,8.2313,17.1977,-79.3602,8.2313,17.1977,-73.0943,5.6433,18.3522,-79.5637,5.6433,18.3522,-67.0149,5.6433,16.1395,-67.6011,8.2313,15.1242,-73.2978,8.2313,17.1977,-73.2978,8.2313,17.1977,-67.0149,5.6433,16.1395,-73.0943,5.6433,18.3522,-62.059,5.6433,11.981,-62.9571,8.2313,11.2275,-67.6011,8.2313,15.1242,-67.6011,8.2313,15.1242,-62.059,5.6433,11.981,-67.0149,5.6433,16.1395,-58.8243,5.6433,6.3783,-59.9259,8.2313,5.9773,-62.9571,8.2313,11.2275,-62.9571,8.2313,11.2275,-58.8243,5.6433,6.3783,-62.059,5.6433,11.981,-57.7008,5.6433,0.0071,-58.8732,8.2313,0.0071,-59.9259,8.2313,5.9773,-59.9259,8.2313,5.9773,-57.7008,5.6433,0.0071,-58.8243,5.6433,6.3783,-61.329,10.0171,0.0071,-62.2336,10.0171,-5.1232,-62.7588,49.0637,-4.9321,-62.7588,49.0637,-4.9321,-61.329,10.0171,0.0071,-61.8878,49.0637,0.0071,-62.2336,10.0171,-5.1232,-64.8383,10.0171,-9.6347,-65.2664,49.0637,-9.2755,-65.2664,49.0637,-9.2755,-62.2336,10.0171,-5.1232,-62.7588,49.0637,-4.9321,-64.8383,10.0171,-9.6347,-68.829,10.0171,-12.9833,-69.1084,49.0637,-12.4993,-69.1084,49.0637,-12.4993,-64.8383,10.0171,-9.6347,-65.2664,49.0637,-9.2755,-68.829,10.0171,-12.9833,-73.7243,10.0171,-14.765,-73.8213,49.0637,-14.2146,-73.8213,49.0637,-14.2146,-68.829,10.0171,-12.9833,-69.1084,49.0637,-12.4993,-73.7243,10.0171,-14.765,-78.9337,10.0171,-14.765,-78.8367,49.0637,-14.2146,-78.8367,49.0637,-14.2146,-73.7243,10.0171,-14.765,-73.8213,49.0637,-14.2146,-78.9337,10.0171,-14.765,-83.829,10.0171,-12.9833,-83.5495,49.0637,-12.4993,-83.5495,49.0637,-12.4993,-78.9337,10.0171,-14.765,-78.8367,49.0637,-14.2146,-83.829,10.0171,-12.9833,-87.8197,10.0171,-9.6347,-87.3915,49.0637,-9.2755,-87.3915,49.0637,-9.2755,-83.829,10.0171,-12.9833,-83.5495,49.0637,-12.4993,-87.8197,10.0171,-9.6347,-90.4244,10.0171,-5.1232,-89.8992,49.0637,-4.9321,-89.8992,49.0637,-4.9321,-87.8197,10.0171,-9.6347,-87.3915,49.0637,-9.2755,-90.4244,10.0171,-5.1232,-91.329,10.0171,0.0071,-90.7701,49.0637,0.0071,-90.7701,49.0637,0.0071,-90.4244,10.0171,-5.1232,-89.8992,49.0637,-4.9321,-91.329,10.0171,0.0071,-90.4244,10.0171,5.1374,-89.8992,49.0637,4.9463,-89.8992,49.0637,4.9463,-91.329,10.0171,0.0071,-90.7701,49.0637,0.0071,-90.4244,10.0171,5.1374,-87.8196,10.0171,9.6489,-87.3915,49.0637,9.2897,-87.3915,49.0637,9.2897,-90.4244,10.0171,5.1374,-89.8992,49.0637,4.9463,-87.8196,10.0171,9.6489,-83.829,10.0171,12.9975,-83.5495,49.0637,12.5135,-83.5495,49.0637,12.5135,-87.8196,10.0171,9.6489,-87.3915,49.0637,9.2897,-83.829,10.0171,12.9975,-78.9337,10.0171,14.7792,-78.8367,49.0637,14.2288,-78.8367,49.0637,14.2288,-83.829,10.0171,12.9975,-83.5495,49.0637,12.5135,-78.9337,10.0171,14.7792,-73.7243,10.0171,14.7792,-73.8213,49.0637,14.2288,-73.8213,49.0637,14.2288,-78.9337,10.0171,14.7792,-78.8367,49.0637,14.2288,-73.7243,10.0171,14.7792,-68.829,10.0171,12.9975,-69.1084,49.0637,12.5135,-69.1084,49.0637,12.5135,-73.7243,10.0171,14.7792,-73.8213,49.0637,14.2288,-68.829,10.0171,12.9975,-64.8383,10.0171,9.6489,-65.2664,49.0637,9.2897,-65.2664,49.0637,9.2897,-68.829,10.0171,12.9975,-69.1084,49.0637,12.5135,-64.8383,10.0171,9.6489,-62.2336,10.0171,5.1374,-62.7588,49.0637,4.9463,-62.7588,49.0637,4.9463,-64.8383,10.0171,9.6489,-65.2664,49.0637,9.2897,-62.2336,10.0171,5.1374,-61.329,10.0171,0.0071,-61.8878,49.0637,0.0071,-61.8878,49.0637,0.0071,-62.2336,10.0171,5.1374,-62.7588,49.0637,4.9463,-62.7588,49.0637,-4.9321,-63.724,51.7017,-4.5807,-62.9151,51.7017,0.0071,-62.9151,51.7017,0.0071,-62.7588,49.0637,-4.9321,-61.8878,49.0637,0.0071,-65.2664,49.0637,-9.2755,-66.0533,51.7017,-8.6152,-63.724,51.7017,-4.5807,-63.724,51.7017,-4.5807,-65.2664,49.0637,-9.2755,-62.7588,49.0637,-4.9321,-69.1084,49.0637,-12.4993,-69.622,51.7017,-11.6097,-66.0533,51.7017,-8.6152,-66.0533,51.7017,-8.6152,-69.1084,49.0637,-12.4993,-65.2664,49.0637,-9.2755,-73.8213,49.0637,-14.2146,-73.9997,51.7017,-13.203,-69.622,51.7017,-11.6097,-69.622,51.7017,-11.6097,-73.8213,49.0637,-14.2146,-69.1084,49.0637,-12.4993,-78.8367,49.0637,-14.2146,-78.6583,51.7017,-13.203,-73.9997,51.7017,-13.203,-73.9997,51.7017,-13.203,-78.8367,49.0637,-14.2146,-73.8213,49.0637,-14.2146,-83.5495,49.0637,-12.4993,-83.0359,51.7017,-11.6097,-78.6583,51.7017,-13.203,-78.6583,51.7017,-13.203,-83.5495,49.0637,-12.4993,-78.8367,49.0637,-14.2146,-87.3915,49.0637,-9.2755,-86.6046,51.7017,-8.6152,-83.0359,51.7017,-11.6097,-83.0359,51.7017,-11.6097,-87.3915,49.0637,-9.2755,-83.5495,49.0637,-12.4993,-89.8992,49.0637,-4.9321,-88.9339,51.7017,-4.5807,-86.6046,51.7017,-8.6152,-86.6046,51.7017,-8.6152,-89.8992,49.0637,-4.9321,-87.3915,49.0637,-9.2755,-90.7701,49.0637,0.0071,-89.7429,51.7017,0.0071,-88.9339,51.7017,-4.5807,-88.9339,51.7017,-4.5807,-90.7701,49.0637,0.0071,-89.8992,49.0637,-4.9321,-89.8992,49.0637,4.9463,-88.9339,51.7017,4.5949,-89.7429,51.7017,0.0071,-89.7429,51.7017,0.0071,-89.8992,49.0637,4.9463,-90.7701,49.0637,0.0071,-87.3915,49.0637,9.2897,-86.6046,51.7017,8.6294,-88.9339,51.7017,4.5949,-88.9339,51.7017,4.5949,-87.3915,49.0637,9.2897,-89.8992,49.0637,4.9463,-83.5495,49.0637,12.5135,-83.0359,51.7017,11.6239,-86.6046,51.7017,8.6294,-86.6046,51.7017,8.6294,-83.5495,49.0637,12.5135,-87.3915,49.0637,9.2897,-78.8367,49.0637,14.2288,-78.6583,51.7017,13.2172,-83.0359,51.7017,11.6239,-83.0359,51.7017,11.6239,-78.8367,49.0637,14.2288,-83.5495,49.0637,12.5135,-73.8213,49.0637,14.2288,-73.9997,51.7017,13.2172,-78.6583,51.7017,13.2172,-78.6583,51.7017,13.2172,-73.8213,49.0637,14.2288,-78.8367,49.0637,14.2288,-69.1084,49.0637,12.5135,-69.622,51.7017,11.6239,-73.9997,51.7017,13.2172,-73.9997,51.7017,13.2172,-69.1084,49.0637,12.5135,-73.8213,49.0637,14.2288,-65.2664,49.0637,9.2897,-66.0533,51.7017,8.6294,-69.622,51.7017,11.6239,-69.622,51.7017,11.6239,-65.2664,49.0637,9.2897,-69.1084,49.0637,12.5135,-62.7588,49.0637,4.9463,-63.724,51.7017,4.5949,-66.0533,51.7017,8.6294,-66.0533,51.7017,8.6294,-62.7588,49.0637,4.9463,-65.2664,49.0637,9.2897,-61.8878,49.0637,0.0071,-62.9151,51.7017,0.0071,-63.724,51.7017,4.5949,-63.724,51.7017,4.5949,-61.8878,49.0637,0.0071,-62.7588,49.0637,4.9463,-65.4395,53.5917,0.0071,-66.0962,53.5917,-3.7173,-66.0962,151.0146,-3.7173,-66.0962,151.0146,-3.7173,-65.4395,53.5917,0.0071,-65.4395,151.0146,0.0071,-66.0962,53.5917,-3.7173,-67.9872,53.5917,-6.9925,-67.9872,151.0146,-6.9925,-67.9872,151.0146,-6.9925,-66.0962,53.5917,-3.7173,-66.0962,151.0146,-3.7173,-67.9872,53.5917,-6.9925,-70.8842,53.5917,-9.4234,-70.8842,151.0146,-9.4234,-70.8842,151.0146,-9.4234,-67.9872,53.5917,-6.9925,-67.9872,151.0146,-6.9925,-70.8842,53.5917,-9.4234,-74.438,53.5917,-10.7169,-74.438,151.0146,-10.7169,-74.438,151.0146,-10.7169,-70.8842,53.5917,-9.4234,-70.8842,151.0146,-9.4234,-74.438,53.5917,-10.7169,-78.2199,53.5917,-10.7169,-78.2199,151.0146,-10.7169,-78.2199,151.0146,-10.7169,-74.438,53.5917,-10.7169,-74.438,151.0146,-10.7169,-78.2199,53.5917,-10.7169,-81.7737,53.5917,-9.4234,-81.7737,151.0146,-9.4234,-81.7737,151.0146,-9.4234,-78.2199,53.5917,-10.7169,-78.2199,151.0146,-10.7169,-81.7737,53.5917,-9.4234,-84.6708,53.5917,-6.9925,-84.6708,151.0146,-6.9925,-84.6708,151.0146,-6.9925,-81.7737,53.5917,-9.4234,-81.7737,151.0146,-9.4234,-84.6708,53.5917,-6.9925,-86.5617,53.5917,-3.7173,-86.5617,151.0146,-3.7173,-86.5617,151.0146,-3.7173,-84.6708,53.5917,-6.9925,-84.6708,151.0146,-6.9925,-86.5617,53.5917,-3.7173,-87.2185,53.5917,0.0071,-87.2185,151.0146,0.0071,-87.2185,151.0146,0.0071,-86.5617,53.5917,-3.7173,-86.5617,151.0146,-3.7173,-87.2185,53.5917,0.0071,-86.5617,53.5917,3.7315,-86.5617,151.0146,3.7315,-86.5617,151.0146,3.7315,-87.2185,53.5917,0.0071,-87.2185,151.0146,0.0071,-86.5617,53.5917,3.7315,-84.6708,53.5917,7.0067,-84.6708,151.0146,7.0067,-84.6708,151.0146,7.0067,-86.5617,53.5917,3.7315,-86.5617,151.0146,3.7315,-84.6708,53.5917,7.0067,-81.7737,53.5917,9.4377,-81.7737,151.0146,9.4377,-81.7737,151.0146,9.4377,-84.6708,53.5917,7.0067,-84.6708,151.0146,7.0067,-81.7737,53.5917,9.4377,-78.2199,53.5917,10.7311,-78.2199,151.0146,10.7311,-78.2199,151.0146,10.7311,-81.7737,53.5917,9.4377,-81.7737,151.0146,9.4377,-78.2199,53.5917,10.7311,-74.438,53.5917,10.7311,-74.438,151.0146,10.7311,-74.438,151.0146,10.7311,-78.2199,53.5917,10.7311,-78.2199,151.0146,10.7311,-74.438,53.5917,10.7311,-70.8843,53.5917,9.4377,-70.8843,151.0146,9.4377,-70.8843,151.0146,9.4377,-74.438,53.5917,10.7311,-74.438,151.0146,10.7311,-70.8843,53.5917,9.4377,-67.9872,53.5917,7.0067,-67.9872,151.0146,7.0067,-67.9872,151.0146,7.0067,-70.8843,53.5917,9.4377,-70.8843,151.0146,9.4377,-67.9872,53.5917,7.0067,-66.0962,53.5917,3.7315,-66.0962,151.0146,3.7315,-66.0962,151.0146,3.7315,-67.9872,53.5917,7.0067,-67.9872,151.0146,7.0067,-66.0962,53.5917,3.7315,-65.4395,53.5917,0.0071,-65.4395,151.0146,0.0071,-65.4395,151.0146,0.0071,-66.0962,53.5917,3.7315,-66.0962,151.0146,3.7315,-65.4395,151.0146,0.0071,-66.0962,151.0146,-3.7173,-66.0962,165.5339,-3.7173,-66.0962,165.5339,-3.7173,-65.4395,151.0146,0.0071,-65.4395,165.5339,0.0071,-66.0962,151.0146,-3.7173,-67.9872,151.0146,-6.9925,-67.9872,165.5339,-6.9925,-67.9872,165.5339,-6.9925,-66.0962,151.0146,-3.7173,-66.0962,165.5339,-3.7173,-67.9872,151.0146,-6.9925,-70.8842,151.0146,-9.4234,-70.8842,165.5339,-9.4234,-70.8842,165.5339,-9.4234,-67.9872,151.0146,-6.9925,-67.9872,165.5339,-6.9925,-70.8842,151.0146,-9.4234,-74.438,151.0146,-10.7169,-74.438,165.5339,-10.7169,-74.438,165.5339,-10.7169,-70.8842,151.0146,-9.4234,-70.8842,165.5339,-9.4234,-74.438,151.0146,-10.7169,-78.2199,151.0146,-10.7169,-78.2199,165.5339,-10.7169,-78.2199,165.5339,-10.7169,-74.438,151.0146,-10.7169,-74.438,165.5339,-10.7169,-78.2199,151.0146,-10.7169,-81.7737,151.0146,-9.4234,-81.7737,165.5339,-9.4234,-81.7737,165.5339,-9.4234,-78.2199,151.0146,-10.7169,-78.2199,165.5339,-10.7169,-81.7737,151.0146,-9.4234,-84.6708,151.0146,-6.9925,-84.6708,165.5339,-6.9925,-84.6708,165.5339,-6.9925,-81.7737,151.0146,-9.4234,-81.7737,165.5339,-9.4234,-84.6708,151.0146,-6.9925,-86.5617,151.0146,-3.7173,-86.5617,165.5339,-3.7173,-86.5617,165.5339,-3.7173,-84.6708,151.0146,-6.9925,-84.6708,165.5339,-6.9925,-86.5617,151.0146,-3.7173,-87.2185,151.0146,0.0071,-87.2185,165.5339,0.0071,-87.2185,165.5339,0.0071,-86.5617,151.0146,-3.7173,-86.5617,165.5339,-3.7173,-87.2185,151.0146,0.0071,-86.5617,151.0146,3.7315,-86.5617,165.5339,3.7315,-86.5617,165.5339,3.7315,-87.2185,151.0146,0.0071,-87.2185,165.5339,0.0071,-86.5617,151.0146,3.7315,-84.6708,151.0146,7.0067,-84.6708,165.5339,7.0067,-84.6708,165.5339,7.0067,-86.5617,151.0146,3.7315,-86.5617,165.5339,3.7315,-84.6708,151.0146,7.0067,-81.7737,151.0146,9.4377,-81.7737,165.5339,9.4377,-81.7737,165.5339,9.4377,-84.6708,151.0146,7.0067,-84.6708,165.5339,7.0067,-81.7737,151.0146,9.4377,-78.2199,151.0146,10.7311,-78.2199,165.5339,10.7311,-78.2199,165.5339,10.7311,-81.7737,151.0146,9.4377,-81.7737,165.5339,9.4377,-78.2199,151.0146,10.7311,-74.438,151.0146,10.7311,-74.438,165.5339,10.7311,-74.438,165.5339,10.7311,-78.2199,151.0146,10.7311,-78.2199,165.5339,10.7311,-74.438,151.0146,10.7311,-70.8843,151.0146,9.4377,-70.8843,165.5339,9.4377,-70.8843,165.5339,9.4377,-74.438,151.0146,10.7311,-74.438,165.5339,10.7311,-70.8843,151.0146,9.4377,-67.9872,151.0146,7.0067,-67.9872,165.5339,7.0067,-67.9872,165.5339,7.0067,-70.8843,151.0146,9.4377,-70.8843,165.5339,9.4377,-67.9872,151.0146,7.0067,-66.0962,151.0146,3.7315,-66.0962,165.5339,3.7315,-66.0962,165.5339,3.7315,-67.9872,151.0146,7.0067,-67.9872,165.5339,7.0067,-66.0962,151.0146,3.7315,-65.4395,151.0146,0.0071,-65.4395,165.5339,0.0071,-65.4395,165.5339,0.0071,-66.0962,151.0146,3.7315,-66.0962,165.5339,3.7315,-66.0962,165.5339,-3.7173,-67.9872,165.5339,-6.9925,-67.9872,180.0531,-6.9925,-67.9872,180.0531,-6.9925,-66.0962,165.5339,-3.7173,-66.0962,180.0531,-3.7173,-67.9872,165.5339,-6.9925,-70.8842,165.5339,-9.4234,-70.8842,180.0531,-9.4234,-70.8842,180.0531,-9.4234,-67.9872,165.5339,-6.9925,-67.9872,180.0531,-6.9925,-70.8842,165.5339,-9.4234,-74.438,165.5339,-10.7169,-74.438,180.0531,-10.7169,-74.438,180.0531,-10.7169,-70.8842,165.5339,-9.4234,-70.8842,180.0531,-9.4234,-74.438,165.5339,-10.7169,-78.2199,165.5339,-10.7169,-78.2199,180.0531,-10.7169,-78.2199,180.0531,-10.7169,-74.438,165.5339,-10.7169,-74.438,180.0531,-10.7169,-78.2199,165.5339,-10.7169,-81.7737,165.5339,-9.4234,-81.7737,180.0531,-9.4234,-81.7737,180.0531,-9.4234,-78.2199,165.5339,-10.7169,-78.2199,180.0531,-10.7169,-81.7737,165.5339,-9.4234,-84.6708,165.5339,-6.9925,-84.6708,180.0531,-6.9925,-84.6708,180.0531,-6.9925,-81.7737,165.5339,-9.4234,-81.7737,180.0531,-9.4234,-84.6708,165.5339,-6.9925,-86.5617,165.5339,-3.7173,-86.5617,180.0531,-3.7173,-86.5617,180.0531,-3.7173,-84.6708,165.5339,-6.9925,-84.6708,180.0531,-6.9925,-99.2358,165.5339,-3.7173,-99.2291,165.5339,0.0071,-103.2901,180.0531,0.0071,-103.2901,180.0531,0.0071,-99.2358,165.5339,-3.7173,-103.2971,180.0531,-3.7173,-99.2291,165.5339,0.0071,-99.2358,165.5339,3.7315,-103.2971,180.0531,3.7315,-103.2971,180.0531,3.7315,-99.2291,165.5339,0.0071,-103.2901,180.0531,0.0071,-86.5617,165.5339,3.7315,-84.6708,165.5339,7.0067,-84.6708,180.0531,7.0067,-84.6708,180.0531,7.0067,-86.5617,165.5339,3.7315,-86.5617,180.0531,3.7315,-84.6708,165.5339,7.0067,-81.7737,165.5339,9.4377,-81.7737,180.0531,9.4377,-81.7737,180.0531,9.4377,-84.6708,165.5339,7.0067,-84.6708,180.0531,7.0067,-81.7737,165.5339,9.4377,-78.2199,165.5339,10.7311,-78.2199,180.0531,10.7311,-78.2199,180.0531,10.7311,-81.7737,165.5339,9.4377,-81.7737,180.0531,9.4377,-78.2199,165.5339,10.7311,-74.438,165.5339,10.7311,-74.438,180.0531,10.7311,-74.438,180.0531,10.7311,-78.2199,165.5339,10.7311,-78.2199,180.0531,10.7311,-74.438,165.5339,10.7311,-70.8843,165.5339,9.4377,-70.8843,180.0531,9.4377,-70.8843,180.0531,9.4377,-74.438,165.5339,10.7311,-74.438,180.0531,10.7311,-70.8843,165.5339,9.4377,-67.9872,165.5339,7.0067,-67.9872,180.0531,7.0067,-67.9872,180.0531,7.0067,-70.8843,165.5339,9.4377,-70.8843,180.0531,9.4377,-67.9872,165.5339,7.0067,-66.0962,165.5339,3.7315,-66.0962,180.0531,3.7315,-66.0962,180.0531,3.7315,-67.9872,165.5339,7.0067,-67.9872,180.0531,7.0067,-65.4395,180.0531,0.0071,-66.0962,180.0531,-3.7173,-66.0962,194.5724,-3.7173,-66.0962,194.5724,-3.7173,-65.4395,180.0531,0.0071,-65.4395,194.5724,0.0071,-66.0962,180.0531,-3.7173,-67.9872,180.0531,-6.9925,-67.9872,194.5724,-6.9925,-67.9872,194.5724,-6.9925,-66.0962,180.0531,-3.7173,-66.0962,194.5724,-3.7173,-67.9872,180.0531,-6.9925,-70.8842,180.0531,-9.4234,-70.8842,194.5724,-9.4234,-70.8842,194.5724,-9.4234,-67.9872,180.0531,-6.9925,-67.9872,194.5724,-6.9925,-70.8842,180.0531,-9.4234,-74.438,180.0531,-10.7169,-74.438,194.5724,-10.7169,-74.438,194.5724,-10.7169,-70.8842,180.0531,-9.4234,-70.8842,194.5724,-9.4234,-74.438,180.0531,-10.7169,-78.2199,180.0531,-10.7169,-78.2199,194.5724,-10.7169,-78.2199,194.5724,-10.7169,-74.438,180.0531,-10.7169,-74.438,194.5724,-10.7169,-78.2199,180.0531,-10.7169,-81.7737,180.0531,-9.4234,-81.7737,194.5724,-9.4234,-81.7737,194.5724,-9.4234,-78.2199,180.0531,-10.7169,-78.2199,194.5724,-10.7169,-81.7737,180.0531,-9.4234,-84.6708,180.0531,-6.9925,-84.6708,194.5724,-6.9925,-84.6708,194.5724,-6.9925,-81.7737,180.0531,-9.4234,-81.7737,194.5724,-9.4234,-84.6708,180.0531,-6.9925,-86.5617,180.0531,-3.7173,-86.5617,194.5724,-3.7173,-86.5617,194.5724,-3.7173,-84.6708,180.0531,-6.9925,-84.6708,194.5724,-6.9925,-86.5617,180.0531,-3.7173,-87.2185,180.0531,0.0071,-87.2185,194.5724,0.0071,-87.2185,194.5724,0.0071,-86.5617,180.0531,-3.7173,-86.5617,194.5724,-3.7173,-87.2185,180.0531,0.0071,-86.5617,180.0531,3.7315,-86.5617,194.5724,3.7315,-86.5617,194.5724,3.7315,-87.2185,180.0531,0.0071,-87.2185,194.5724,0.0071,-86.5617,180.0531,3.7315,-84.6708,180.0531,7.0067,-84.6708,194.5724,7.0067,-84.6708,194.5724,7.0067,-86.5617,180.0531,3.7315,-86.5617,194.5724,3.7315,-84.6708,180.0531,7.0067,-81.7737,180.0531,9.4377,-81.7737,194.5724,9.4377,-81.7737,194.5724,9.4377,-84.6708,180.0531,7.0067,-84.6708,194.5724,7.0067,-81.7737,180.0531,9.4377,-78.2199,180.0531,10.7311,-78.2199,194.5724,10.7311,-78.2199,194.5724,10.7311,-81.7737,180.0531,9.4377,-81.7737,194.5724,9.4377,-78.2199,180.0531,10.7311,-74.438,180.0531,10.7311,-74.438,194.5724,10.7311,-74.438,194.5724,10.7311,-78.2199,180.0531,10.7311,-78.2199,194.5724,10.7311,-74.438,180.0531,10.7311,-70.8843,180.0531,9.4377,-70.8843,194.5724,9.4377,-70.8843,194.5724,9.4377,-74.438,180.0531,10.7311,-74.438,194.5724,10.7311,-70.8843,180.0531,9.4377,-67.9872,180.0531,7.0067,-67.9872,194.5724,7.0067,-67.9872,194.5724,7.0067,-70.8843,180.0531,9.4377,-70.8843,194.5724,9.4377,-67.9872,180.0531,7.0067,-66.0962,180.0531,3.7315,-66.0962,194.5724,3.7315,-66.0962,194.5724,3.7315,-67.9872,180.0531,7.0067,-67.9872,194.5724,7.0067,-66.0962,180.0531,3.7315,-65.4395,180.0531,0.0071,-65.4395,194.5724,0.0071,-65.4395,194.5724,0.0071,-66.0962,180.0531,3.7315,-66.0962,194.5724,3.7315,-62.059,0.0102,11.981,-67.0149,0.0102,16.1395,-73.0943,0.0102,18.3522,-73.0943,0.0102,18.3522,-62.059,0.0102,11.981,-57.7008,0.0102,0.0071,-73.0943,0.0102,18.3522,-79.5637,0.0102,18.3522,-85.6431,0.0102,16.1395,-85.6431,0.0102,16.1395,-73.0943,0.0102,18.3522,-93.8337,0.0102,6.3783,-85.6431,0.0102,16.1395,-90.599,0.0102,11.981,-93.8337,0.0102,6.3783,-93.8337,0.0102,6.3783,-94.9571,0.0102,0.0071,-93.8337,0.0102,-6.3641,-93.8337,0.0102,-6.3641,-93.8337,0.0102,6.3783,-85.643,0.0102,-16.1253,-93.8337,0.0102,-6.3641,-90.599,0.0102,-11.9668,-85.643,0.0102,-16.1253,-85.643,0.0102,-16.1253,-79.5637,0.0102,-18.338,-73.0942,0.0102,-18.338,-73.0942,0.0102,-18.338,-85.643,0.0102,-16.1253,-62.059,0.0102,-11.9668,-73.0942,0.0102,-18.338,-67.0149,0.0102,-16.1253,-62.059,0.0102,-11.9668,-93.8337,0.0102,6.3783,-85.643,0.0102,-16.1253,-62.059,0.0102,-11.9668,-62.059,0.0102,-11.9668,-93.8337,0.0102,6.3783,-73.0943,0.0102,18.3522,-62.059,0.0102,-11.9668,-58.8243,0.0102,-6.3641,-57.7008,0.0102,0.0071,-57.7008,0.0102,0.0071,-62.059,0.0102,-11.9668,-73.0943,0.0102,18.3522,-58.8243,0.0102,6.3783,-62.059,0.0102,11.981,-57.7008,0.0102,0.0071,-66.0962,53.5917,-3.7173,-65.4395,53.5917,0.0071,-62.9151,51.7017,0.0071,-62.9151,51.7017,0.0071,-66.0962,53.5917,-3.7173,-63.724,51.7017,-4.5807,-67.9872,53.5917,-6.9925,-66.0962,53.5917,-3.7173,-63.724,51.7017,-4.5807,-63.724,51.7017,-4.5807,-67.9872,53.5917,-6.9925,-66.0533,51.7017,-8.6152,-70.8842,53.5917,-9.4234,-67.9872,53.5917,-6.9925,-66.0533,51.7017,-8.6152,-66.0533,51.7017,-8.6152,-70.8842,53.5917,-9.4234,-69.622,51.7017,-11.6097,-74.438,53.5917,-10.7169,-70.8842,53.5917,-9.4234,-69.622,51.7017,-11.6097,-69.622,51.7017,-11.6097,-74.438,53.5917,-10.7169,-73.9997,51.7017,-13.203,-78.2199,53.5917,-10.7169,-74.438,53.5917,-10.7169,-73.9997,51.7017,-13.203,-73.9997,51.7017,-13.203,-78.2199,53.5917,-10.7169,-78.6583,51.7017,-13.203,-81.7737,53.5917,-9.4234,-78.2199,53.5917,-10.7169,-78.6583,51.7017,-13.203,-78.6583,51.7017,-13.203,-81.7737,53.5917,-9.4234,-83.0359,51.7017,-11.6097,-84.6708,53.5917,-6.9925,-81.7737,53.5917,-9.4234,-83.0359,51.7017,-11.6097,-83.0359,51.7017,-11.6097,-84.6708,53.5917,-6.9925,-86.6046,51.7017,-8.6152,-86.5617,53.5917,-3.7173,-84.6708,53.5917,-6.9925,-86.6046,51.7017,-8.6152,-86.6046,51.7017,-8.6152,-86.5617,53.5917,-3.7173,-88.9339,51.7017,-4.5807,-87.2185,53.5917,0.0071,-86.5617,53.5917,-3.7173,-88.9339,51.7017,-4.5807,-88.9339,51.7017,-4.5807,-87.2185,53.5917,0.0071,-89.7429,51.7017,0.0071,-86.5617,53.5917,3.7315,-87.2185,53.5917,0.0071,-89.7429,51.7017,0.0071,-89.7429,51.7017,0.0071,-86.5617,53.5917,3.7315,-88.9339,51.7017,4.5949,-84.6708,53.5917,7.0067,-86.5617,53.5917,3.7315,-88.9339,51.7017,4.5949,-88.9339,51.7017,4.5949,-84.6708,53.5917,7.0067,-86.6046,51.7017,8.6294,-81.7737,53.5917,9.4377,-84.6708,53.5917,7.0067,-86.6046,51.7017,8.6294,-86.6046,51.7017,8.6294,-81.7737,53.5917,9.4377,-83.0359,51.7017,11.6239,-78.2199,53.5917,10.7311,-81.7737,53.5917,9.4377,-83.0359,51.7017,11.6239,-83.0359,51.7017,11.6239,-78.2199,53.5917,10.7311,-78.6583,51.7017,13.2172,-74.438,53.5917,10.7311,-78.2199,53.5917,10.7311,-78.6583,51.7017,13.2172,-78.6583,51.7017,13.2172,-74.438,53.5917,10.7311,-73.9997,51.7017,13.2172,-70.8843,53.5917,9.4377,-74.438,53.5917,10.7311,-73.9997,51.7017,13.2172,-73.9997,51.7017,13.2172,-70.8843,53.5917,9.4377,-69.622,51.7017,11.6239,-67.9872,53.5917,7.0067,-70.8843,53.5917,9.4377,-69.622,51.7017,11.6239,-69.622,51.7017,11.6239,-67.9872,53.5917,7.0067,-66.0533,51.7017,8.6294,-66.0962,53.5917,3.7315,-67.9872,53.5917,7.0067,-66.0533,51.7017,8.6294,-66.0533,51.7017,8.6294,-66.0962,53.5917,3.7315,-63.724,51.7017,4.5949,-65.4395,53.5917,0.0071,-66.0962,53.5917,3.7315,-63.724,51.7017,4.5949,-63.724,51.7017,4.5949,-65.4395,53.5917,0.0071,-62.9151,51.7017,0.0071,-62.2336,10.0171,-5.1232,-61.329,10.0171,0.0071,-58.8732,8.2313,0.0071,-58.8732,8.2313,0.0071,-62.2336,10.0171,-5.1232,-59.9259,8.2313,-5.9631,-64.8383,10.0171,-9.6347,-62.2336,10.0171,-5.1232,-59.9259,8.2313,-5.9631,-59.9259,8.2313,-5.9631,-64.8383,10.0171,-9.6347,-62.9571,8.2313,-11.2132,-68.829,10.0171,-12.9833,-64.8383,10.0171,-9.6347,-62.9571,8.2313,-11.2132,-62.9571,8.2313,-11.2132,-68.829,10.0171,-12.9833,-67.6011,8.2313,-15.11,-73.7243,10.0171,-14.765,-68.829,10.0171,-12.9833,-67.6011,8.2313,-15.11,-67.6011,8.2313,-15.11,-73.7243,10.0171,-14.765,-73.2978,8.2313,-17.1835,-78.9337,10.0171,-14.765,-73.7243,10.0171,-14.765,-73.2978,8.2313,-17.1835,-73.2978,8.2313,-17.1835,-78.9337,10.0171,-14.765,-79.3601,8.2313,-17.1835,-83.829,10.0171,-12.9833,-78.9337,10.0171,-14.765,-79.3601,8.2313,-17.1835,-79.3601,8.2313,-17.1835,-83.829,10.0171,-12.9833,-85.0569,8.2313,-15.11,-87.8197,10.0171,-9.6347,-83.829,10.0171,-12.9833,-85.0569,8.2313,-15.11,-85.0569,8.2313,-15.11,-87.8197,10.0171,-9.6347,-89.7009,8.2313,-11.2132,-90.4244,10.0171,-5.1232,-87.8197,10.0171,-9.6347,-89.7009,8.2313,-11.2132,-89.7009,8.2313,-11.2132,-90.4244,10.0171,-5.1232,-92.732,8.2313,-5.9631,-91.329,10.0171,0.0071,-90.4244,10.0171,-5.1232,-92.732,8.2313,-5.9631,-92.732,8.2313,-5.9631,-91.329,10.0171,0.0071,-93.7847,8.2313,0.0071,-90.4244,10.0171,5.1374,-91.329,10.0171,0.0071,-93.7847,8.2313,0.0071,-93.7847,8.2313,0.0071,-90.4244,10.0171,5.1374,-92.732,8.2313,5.9773,-87.8196,10.0171,9.6489,-90.4244,10.0171,5.1374,-92.732,8.2313,5.9773,-92.732,8.2313,5.9773,-87.8196,10.0171,9.6489,-89.7009,8.2313,11.2274,-83.829,10.0171,12.9975,-87.8196,10.0171,9.6489,-89.7009,8.2313,11.2274,-89.7009,8.2313,11.2274,-83.829,10.0171,12.9975,-85.0569,8.2313,15.1242,-78.9337,10.0171,14.7792,-83.829,10.0171,12.9975,-85.0569,8.2313,15.1242,-85.0569,8.2313,15.1242,-78.9337,10.0171,14.7792,-79.3602,8.2313,17.1977,-73.7243,10.0171,14.7792,-78.9337,10.0171,14.7792,-79.3602,8.2313,17.1977,-79.3602,8.2313,17.1977,-73.7243,10.0171,14.7792,-73.2978,8.2313,17.1977,-68.829,10.0171,12.9975,-73.7243,10.0171,14.7792,-73.2978,8.2313,17.1977,-73.2978,8.2313,17.1977,-68.829,10.0171,12.9975,-67.6011,8.2313,15.1242,-64.8383,10.0171,9.6489,-68.829,10.0171,12.9975,-67.6011,8.2313,15.1242,-67.6011,8.2313,15.1242,-64.8383,10.0171,9.6489,-62.9571,8.2313,11.2275,-62.2336,10.0171,5.1374,-64.8383,10.0171,9.6489,-62.9571,8.2313,11.2275,-62.9571,8.2313,11.2275,-62.2336,10.0171,5.1374,-59.9259,8.2313,5.9773,-61.329,10.0171,0.0071,-62.2336,10.0171,5.1374,-59.9259,8.2313,5.9773,-59.9259,8.2313,5.9773,-61.329,10.0171,0.0071,-58.8732,8.2313,0.0071,-86.5617,165.5339,-3.7173,-87.2185,165.5339,0.0071,-99.2291,165.5339,0.0071,-99.2291,165.5339,0.0071,-86.5617,165.5339,-3.7173,-99.2358,165.5339,-3.7173,-87.2185,165.5339,0.0071,-86.5617,165.5339,3.7315,-99.2358,165.5339,3.7315,-99.2358,165.5339,3.7315,-87.2185,165.5339,0.0071,-99.2291,165.5339,0.0071,-86.5617,165.5339,3.7315,-86.5617,180.0531,3.7315,-103.2971,180.0531,3.7315,-103.2971,180.0531,3.7315,-86.5617,165.5339,3.7315,-99.2358,165.5339,3.7315,-86.5617,180.0531,3.7315,-87.2185,180.0531,0.0071,-103.2901,180.0531,0.0071,-103.2901,180.0531,0.0071,-86.5617,180.0531,3.7315,-103.2971,180.0531,3.7315,-87.2185,180.0531,0.0071,-86.5617,180.0531,-3.7173,-103.2971,180.0531,-3.7173,-103.2971,180.0531,-3.7173,-87.2185,180.0531,0.0071,-103.2901,180.0531,0.0071,-86.5617,180.0531,-3.7173,-86.5617,165.5339,-3.7173,-99.2358,165.5339,-3.7173,-99.2358,165.5339,-3.7173,-86.5617,180.0531,-3.7173,-103.2971,180.0531,-3.7173,-12.3875,165.5339,3.7281,-12.2835,165.5339,0.0035,7.4188,165.5339,0.0021,7.4188,165.5339,0.0021,-12.3875,165.5339,3.7281,7.5198,165.5339,3.7268,-12.3875,180.0531,3.7281,-12.3875,165.5339,3.7281,7.5198,165.5339,3.7268,7.5198,165.5339,3.7268,-12.3875,180.0531,3.7281,7.5198,180.0531,3.7268,-12.385,180.0531,-3.7207,-12.3343,180.0531,-1.8586,7.471,180.0531,-1.8599,7.471,180.0531,-1.8599,-12.385,180.0531,-3.7207,7.5232,180.0531,-3.722,-12.385,165.5339,-3.7207,-12.385,180.0531,-3.7207,7.5232,180.0531,-3.722,7.5232,180.0531,-3.722,-12.385,165.5339,-3.7207,7.5232,165.5339,-3.722,-12.2835,165.5339,0.0035,-12.385,165.5339,-3.7207,7.5232,165.5339,-3.722,7.5232,165.5339,-3.722,-12.2835,165.5339,0.0035,7.4188,165.5339,0.0021,103.7982,194.5724,3.7507,103.7778,194.5724,0.0268,108.2601,208.5724,0.0268,108.2601,208.5724,0.0268,103.7982,194.5724,3.7507,108.255,208.5724,3.7507,103.7778,194.5724,0.0268,103.8046,194.5724,-3.6981,108.2617,208.5724,-3.6981,108.2617,208.5724,-3.6981,103.7778,194.5724,0.0268,108.2601,208.5724,0.0268,-108.6662,194.5724,-3.7173,-108.6099,194.5724,0.0071,-113.0967,208.5724,0.0071,-113.0967,208.5724,0.0071,-108.6662,194.5724,-3.7173,-113.1231,208.5724,-3.7173,-108.6099,194.5724,0.0071,-108.6662,194.5724,3.7315,-113.1231,208.5724,3.7315,-113.1231,208.5724,3.7315,-108.6099,194.5724,0.0071,-113.0967,208.5724,0.0071,-11.6601,194.5724,3.7281,-11.6119,194.5724,1.8658,6.7459,194.5724,1.8645,6.7459,194.5724,1.8645,-11.6601,194.5724,3.7281,6.7926,194.5724,3.7268,-12.3875,208.5724,3.7281,-11.6601,194.5724,3.7281,6.7926,194.5724,3.7268,6.7926,194.5724,3.7268,-12.3875,208.5724,3.7281,7.5198,208.5724,3.7268,-12.2835,208.5724,0.0035,-12.3875,208.5724,3.7281,7.5198,208.5724,3.7268,7.5198,208.5724,3.7268,-12.2835,208.5724,0.0035,7.4188,208.5724,0.0021,-12.385,208.5724,-3.7207,-12.2835,208.5724,0.0035,7.4188,208.5724,0.0021,7.4188,208.5724,0.0021,-12.385,208.5724,-3.7207,7.5232,208.5724,-3.722,-11.6578,194.5724,-3.7207,-12.385,208.5724,-3.7207,7.5232,208.5724,-3.722,7.5232,208.5724,-3.722,-11.6578,194.5724,-3.7207,6.7958,194.5724,-3.722,81.694,194.5724,3.7396,82.3537,194.5724,0.0157,103.7778,194.5724,0.0268,103.7778,194.5724,0.0268,81.694,194.5724,3.7396,103.7982,194.5724,3.7507,82.3537,194.5724,0.0157,81.6999,194.5724,-3.7092,103.8046,194.5724,-3.6981,103.8046,194.5724,-3.6981,82.3537,194.5724,0.0157,103.7778,194.5724,0.0268,81.6999,194.5724,-3.7092,81.6999,208.5724,-3.7092,108.2617,208.5724,-3.6981,108.2617,208.5724,-3.6981,81.6999,194.5724,-3.7092,103.8046,194.5724,-3.6981,81.6999,208.5724,-3.7092,82.3537,208.5724,0.0157,108.2601,208.5724,0.0268,108.2601,208.5724,0.0268,81.6999,208.5724,-3.7092,108.2617,208.5724,-3.6981,82.3537,208.5724,0.0157,81.694,208.5724,3.7396,108.255,208.5724,3.7507,108.255,208.5724,3.7507,82.3537,208.5724,0.0157,108.2601,208.5724,0.0268,81.694,208.5724,3.7396,81.694,194.5724,3.7396,103.7982,194.5724,3.7507,103.7982,194.5724,3.7507,81.694,208.5724,3.7396,108.255,208.5724,3.7507,-86.5617,194.5724,-3.7173,-87.2185,194.5724,0.0071,-108.6099,194.5724,0.0071,-108.6099,194.5724,0.0071,-86.5617,194.5724,-3.7173,-108.6662,194.5724,-3.7173,-87.2185,194.5724,0.0071,-86.5617,194.5724,3.7315,-108.6662,194.5724,3.7315,-108.6662,194.5724,3.7315,-87.2185,194.5724,0.0071,-108.6099,194.5724,0.0071,-86.5617,194.5724,3.7315,-86.5617,208.5724,3.7315,-113.1231,208.5724,3.7315,-113.1231,208.5724,3.7315,-86.5617,194.5724,3.7315,-108.6662,194.5724,3.7315,-86.5617,208.5724,3.7315,-87.2185,208.5724,0.0071,-113.0967,208.5724,0.0071,-113.0967,208.5724,0.0071,-86.5617,208.5724,3.7315,-113.1231,208.5724,3.7315,-87.2185,208.5724,0.0071,-86.5617,208.5724,-3.7173,-113.1231,208.5724,-3.7173,-113.1231,208.5724,-3.7173,-87.2185,208.5724,0.0071,-113.0967,208.5724,0.0071,-86.5617,208.5724,-3.7173,-86.5617,194.5724,-3.7173,-108.6662,194.5724,-3.7173,-108.6662,194.5724,-3.7173,-86.5617,208.5724,-3.7173,-113.1231,208.5724,-3.7173,-12.3875,165.5339,3.7281,-66.0962,165.5339,3.7315,-65.4395,165.5339,0.0071,-65.4395,165.5339,0.0071,-12.3875,165.5339,3.7281,-12.2835,165.5339,0.0035,60.5748,165.5339,-0.0015,61.2285,165.5339,3.7234,7.5198,165.5339,3.7268,7.5198,165.5339,3.7268,60.5748,165.5339,-0.0015,7.4188,165.5339,0.0021,-66.0962,180.0531,3.7315,-66.0962,165.5339,3.7315,-12.3875,165.5339,3.7281,-12.3875,165.5339,3.7281,-66.0962,180.0531,3.7315,-12.3875,180.0531,3.7281,61.2285,165.5339,3.7234,61.2285,180.0531,3.7234,7.5198,180.0531,3.7268,7.5198,180.0531,3.7268,61.2285,165.5339,3.7234,7.5198,165.5339,3.7268,-65.4395,180.0531,0.0071,-66.0962,180.0531,3.7315,-12.3875,180.0531,3.7281,-12.3875,180.0531,3.7281,-65.4395,180.0531,0.0071,-12.3355,180.0531,1.8658,-12.2835,180.0531,0.0035,-65.4395,180.0531,0.0071,-12.3355,180.0531,1.8658,7.5198,180.0531,3.7268,61.2285,180.0531,3.7234,60.5748,180.0531,-0.0015,60.5748,180.0531,-0.0015,7.5198,180.0531,3.7268,7.4693,180.0531,1.8645,60.5748,180.0531,-0.0015,7.4188,180.0531,0.0021,7.4693,180.0531,1.8645,-12.385,180.0531,-3.7207,-66.0962,180.0531,-3.7173,-65.4395,180.0531,0.0071,-65.4395,180.0531,0.0071,-12.385,180.0531,-3.7207,-12.3343,180.0531,-1.8586,-65.4395,180.0531,0.0071,-12.2835,180.0531,0.0035,-12.3343,180.0531,-1.8586,60.5748,180.0531,-0.0015,61.2344,180.0531,-3.7254,7.5232,180.0531,-3.722,7.5232,180.0531,-3.722,60.5748,180.0531,-0.0015,7.471,180.0531,-1.8599,7.4188,180.0531,0.0021,60.5748,180.0531,-0.0015,7.471,180.0531,-1.8599,-66.0962,165.5339,-3.7173,-66.0962,180.0531,-3.7173,-12.385,180.0531,-3.7207,-12.385,180.0531,-3.7207,-66.0962,165.5339,-3.7173,-12.385,165.5339,-3.7207,61.2344,180.0531,-3.7254,61.2344,165.5339,-3.7254,7.5232,165.5339,-3.722,7.5232,165.5339,-3.722,61.2344,180.0531,-3.7254,7.5232,180.0531,-3.722,-65.4395,165.5339,0.0071,-66.0962,165.5339,-3.7173,-12.385,165.5339,-3.7207,-12.385,165.5339,-3.7207,-65.4395,165.5339,0.0071,-12.2835,165.5339,0.0035,7.5232,165.5339,-3.722,61.2344,165.5339,-3.7254,60.5748,165.5339,-0.0015,60.5748,165.5339,-0.0015,7.5232,165.5339,-3.722,7.4188,165.5339,0.0021,-11.6601,194.5724,3.7281,-66.0962,194.5724,3.7315,-65.4395,194.5724,0.0071,-65.4395,194.5724,0.0071,-11.6601,194.5724,3.7281,-11.6119,194.5724,1.8658,-65.4395,194.5724,0.0071,-11.5638,194.5724,0.0035,-11.6119,194.5724,1.8658,60.5748,194.5724,-0.0015,61.2285,194.5724,3.7234,6.7926,194.5724,3.7268,6.7926,194.5724,3.7268,60.5748,194.5724,-0.0015,6.7459,194.5724,1.8645,6.6991,194.5724,0.0021,60.5748,194.5724,-0.0015,6.7459,194.5724,1.8645,-66.0962,208.5724,3.7315,-66.0962,194.5724,3.7315,-11.6601,194.5724,3.7281,-11.6601,194.5724,3.7281,-66.0962,208.5724,3.7315,-12.3875,208.5724,3.7281,61.2285,194.5724,3.7234,61.2285,208.5724,3.7234,7.5198,208.5724,3.7268,7.5198,208.5724,3.7268,61.2285,194.5724,3.7234,6.7926,194.5724,3.7268,-65.4395,208.5724,0.0071,-66.0962,208.5724,3.7315,-12.3875,208.5724,3.7281,-12.3875,208.5724,3.7281,-65.4395,208.5724,0.0071,-12.2835,208.5724,0.0035,7.5198,208.5724,3.7268,61.2285,208.5724,3.7234,60.5748,208.5724,-0.0015,60.5748,208.5724,-0.0015,7.5198,208.5724,3.7268,7.4188,208.5724,0.0021,-12.385,208.5724,-3.7207,-66.0962,208.5724,-3.7173,-65.4395,208.5724,0.0071,-65.4395,208.5724,0.0071,-12.385,208.5724,-3.7207,-12.2835,208.5724,0.0035,60.5748,208.5724,-0.0015,61.2344,208.5724,-3.7254,7.5232,208.5724,-3.722,7.5232,208.5724,-3.722,60.5748,208.5724,-0.0015,7.4188,208.5724,0.0021,-66.0962,194.5724,-3.7173,-66.0962,208.5724,-3.7173,-12.385,208.5724,-3.7207,-12.385,208.5724,-3.7207,-66.0962,194.5724,-3.7173,-11.6578,194.5724,-3.7207,61.2344,208.5724,-3.7254,61.2344,194.5724,-3.7254,6.7958,194.5724,-3.722,6.7958,194.5724,-3.722,61.2344,208.5724,-3.7254,7.5232,208.5724,-3.722,-65.4395,194.5724,0.0071,-66.0962,194.5724,-3.7173,-11.6578,194.5724,-3.7207,-11.6578,194.5724,-3.7207,-65.4395,194.5724,0.0071,-11.6108,194.5724,-1.8586,-11.5638,194.5724,0.0035,-65.4395,194.5724,0.0071,-11.6108,194.5724,-1.8586,6.7958,194.5724,-3.722,61.2344,194.5724,-3.7254,60.5748,194.5724,-0.0015,60.5748,194.5724,-0.0015,6.7958,194.5724,-3.722,6.7474,194.5724,-1.8599,60.5748,194.5724,-0.0015,6.6991,194.5724,0.0021,6.7474,194.5724,-1.8599,-12.3875,180.0531,3.7281,7.5198,180.0531,3.7268,7.4693,180.0531,1.8645,7.4693,180.0531,1.8645,-12.3875,180.0531,3.7281,-12.3355,180.0531,1.8658,-11.6578,194.5724,-3.7207,6.7958,194.5724,-3.722,6.7474,194.5724,-1.8599,6.7474,194.5724,-1.8599,-11.6578,194.5724,-3.7207,-11.6108,194.5724,-1.8586,7.4188,180.0531,0.0021,7.471,180.0531,-1.8599,6.7474,194.5724,-1.8599,6.7474,194.5724,-1.8599,7.4188,180.0531,0.0021,6.6991,194.5724,0.0021,7.471,180.0531,-1.8599,-12.3343,180.0531,-1.8586,-11.6108,194.5724,-1.8586,-11.6108,194.5724,-1.8586,7.471,180.0531,-1.8599,6.7474,194.5724,-1.8599,-12.3343,180.0531,-1.8586,-12.2835,180.0531,0.0035,-11.5638,194.5724,0.0035,-11.5638,194.5724,0.0035,-12.3343,180.0531,-1.8586,-11.6108,194.5724,-1.8586,-12.2835,180.0531,0.0035,-12.3355,180.0531,1.8658,-11.6119,194.5724,1.8658,-11.6119,194.5724,1.8658,-12.2835,180.0531,0.0035,-11.5638,194.5724,0.0035,-12.3355,180.0531,1.8658,7.4693,180.0531,1.8645,6.7459,194.5724,1.8645,6.7459,194.5724,1.8645,-12.3355,180.0531,1.8658,-11.6119,194.5724,1.8658,7.4693,180.0531,1.8645,7.4188,180.0531,0.0021,6.6991,194.5724,0.0021,6.6991,194.5724,0.0021,7.4693,180.0531,1.8645,6.7459,194.5724,1.8645,-86.5617,194.5724,-3.7173,-86.5617,208.5724,-3.7173,-66.0962,208.5724,-3.7173,-66.0962,208.5724,-3.7173,-86.5617,194.5724,-3.7173,-66.0962,194.5724,-3.7173,-67.9872,194.5724,-6.9925,-70.8842,194.5724,-9.4234,-74.438,194.5724,-10.7169,-74.438,194.5724,-10.7169,-67.9872,194.5724,-6.9925,-86.5617,194.5724,-3.7173,-74.438,194.5724,-10.7169,-78.2199,194.5724,-10.7169,-81.7737,194.5724,-9.4234,-81.7737,194.5724,-9.4234,-74.438,194.5724,-10.7169,-86.5617,194.5724,-3.7173,-81.7737,194.5724,-9.4234,-84.6708,194.5724,-6.9925,-86.5617,194.5724,-3.7173,-66.0962,194.5724,-3.7173,-67.9872,194.5724,-6.9925,-86.5617,194.5724,-3.7173,-86.5617,194.5724,3.7315,-66.0962,194.5724,3.7315,-66.0962,208.5724,3.7315,-66.0962,208.5724,3.7315,-86.5617,194.5724,3.7315,-86.5617,208.5724,3.7315,-66.0962,208.5724,-3.7173,-86.5617,208.5724,-3.7173,-87.2185,208.5724,0.0071,-87.2185,208.5724,0.0071,-66.0962,208.5724,-3.7173,-65.4395,208.5724,0.0071,-86.5617,208.5724,3.7315,-66.0962,208.5724,3.7315,-65.4395,208.5724,0.0071,-65.4395,208.5724,0.0071,-86.5617,208.5724,3.7315,-87.2185,208.5724,0.0071,-84.6708,194.5724,7.0067,-81.7737,194.5724,9.4377,-78.2199,194.5724,10.7311,-78.2199,194.5724,10.7311,-84.6708,194.5724,7.0067,-66.0962,194.5724,3.7315,-78.2199,194.5724,10.7311,-74.438,194.5724,10.7311,-70.8843,194.5724,9.4377,-70.8843,194.5724,9.4377,-78.2199,194.5724,10.7311,-66.0962,194.5724,3.7315,-70.8843,194.5724,9.4377,-67.9872,194.5724,7.0067,-66.0962,194.5724,3.7315,-86.5617,194.5724,3.7315,-84.6708,194.5724,7.0067,-66.0962,194.5724,3.7315,61.2285,194.5724,3.7234,81.694,194.5724,3.7396,81.694,208.5724,3.7396,81.694,208.5724,3.7396,61.2285,194.5724,3.7234,61.2285,208.5724,3.7234,61.2344,194.5724,-3.7254,61.2344,208.5724,-3.7254,81.6999,208.5724,-3.7092,81.6999,208.5724,-3.7092,61.2344,194.5724,-3.7254,81.6999,194.5724,-3.7092,81.694,208.5724,3.7396,82.3537,208.5724,0.0157,60.5748,208.5724,-0.0015,60.5748,208.5724,-0.0015,81.694,208.5724,3.7396,61.2285,208.5724,3.7234,81.6999,208.5724,-3.7092,61.2344,208.5724,-3.7254,60.5748,208.5724,-0.0015,60.5748,208.5724,-0.0015,81.6999,208.5724,-3.7092,82.3537,208.5724,0.0157,79.8116,194.5724,-6.9859,76.9164,194.5724,-9.4191,73.3637,194.5724,-10.7154,73.3637,194.5724,-10.7154,79.8116,194.5724,-6.9859,61.2344,194.5724,-3.7254,73.3637,194.5724,-10.7154,69.5818,194.5724,-10.7184,66.027,194.5724,-9.4278,66.027,194.5724,-9.4278,73.3637,194.5724,-10.7154,61.2344,194.5724,-3.7254,66.027,194.5724,-9.4278,63.128,194.5724,-6.9991,61.2344,194.5724,-3.7254,81.6999,194.5724,-3.7092,79.8116,194.5724,-6.9859,61.2344,194.5724,-3.7254,63.1169,194.5724,7.0001,66.012,194.5724,9.4333,69.5648,194.5724,10.7296,69.5648,194.5724,10.7296,63.1169,194.5724,7.0001,81.694,194.5724,3.7396,69.5648,194.5724,10.7296,73.3467,194.5724,10.7326,76.9015,194.5724,9.442,76.9015,194.5724,9.442,69.5648,194.5724,10.7296,81.694,194.5724,3.7396,76.9015,194.5724,9.442,79.8005,194.5724,7.0133,81.694,194.5724,3.7396,61.2285,194.5724,3.7234,63.1169,194.5724,7.0001,81.694,194.5724,3.7396,-114.9192,206.4684,4.5155,-114.9192,206.4684,-4.4845,-99.9052,206.4684,-4.4845,-99.9052,206.4684,-4.4845,-114.9192,206.4684,4.5155,-99.9052,206.4684,4.5155,-99.9052,206.4684,4.5155,-99.9052,206.4684,-4.4845,-84.8912,206.4684,-4.4845,-84.8912,206.4684,-4.4845,-99.9052,206.4684,4.5155,-84.8912,206.4684,4.5155,-84.8912,206.4684,4.5155,-84.8912,206.4684,-4.4845,-69.8772,206.4684,-4.4845,-69.8772,206.4684,-4.4845,-84.8912,206.4684,4.5155,-69.8772,206.4684,4.5155,-69.8772,206.4684,4.5155,-69.8772,206.4684,-4.4845,-54.8633,206.4684,-4.4845,-54.8633,206.4684,-4.4845,-69.8772,206.4684,4.5155,-54.8633,206.4684,4.5155,-54.8633,206.4684,4.5155,-54.8633,206.4684,-4.4845,-39.8493,206.4684,-4.4845,-39.8493,206.4684,-4.4845,-54.8633,206.4684,4.5155,-39.8493,206.4684,4.5155,-39.8493,206.4684,4.5155,-39.8493,206.4684,-4.4845,-24.8353,206.4684,-4.4845,-24.8353,206.4684,-4.4845,-39.8493,206.4684,4.5155,-24.8353,206.4684,4.5155,-24.8353,206.4684,4.5155,-24.8353,206.4684,-4.4845,-9.8213,206.4684,-4.4845,-9.8213,206.4684,-4.4845,-24.8353,206.4684,4.5155,-9.8213,206.4684,4.5155,-9.8213,206.4684,4.5155,-9.8213,206.4684,-4.4845,5.1926,206.4684,-4.4845,5.1926,206.4684,-4.4845,-9.8213,206.4684,4.5155,5.1926,206.4684,4.5155,5.1926,206.4684,4.5155,5.1926,206.4684,-4.4845,20.2066,206.4684,-4.4845,20.2066,206.4684,-4.4845,5.1926,206.4684,4.5155,20.2066,206.4684,4.5155,20.2066,206.4684,4.5155,20.2066,206.4684,-4.4845,35.2206,206.4684,-4.4845,35.2206,206.4684,-4.4845,20.2066,206.4684,4.5155,35.2206,206.4684,4.5155,35.2206,206.4684,4.5155,35.2206,206.4684,-4.4845,50.2346,206.4684,-4.4845,50.2346,206.4684,-4.4845,35.2206,206.4684,4.5155,50.2346,206.4684,4.5155,50.2346,206.4684,4.5155,50.2346,206.4684,-4.4845,65.2485,206.4684,-4.4845,65.2485,206.4684,-4.4845,50.2346,206.4684,4.5155,65.2485,206.4684,4.5155,65.2485,206.4684,4.5155,65.2485,206.4684,-4.4845,80.2625,206.4684,-4.4845,80.2625,206.4684,-4.4845,65.2485,206.4684,4.5155,80.2625,206.4684,4.5155,80.2625,206.4684,4.5155,80.2625,206.4684,-4.4845,95.2765,206.4684,-4.4845,95.2765,206.4684,-4.4845,80.2625,206.4684,4.5155,95.2765,206.4684,4.5155,95.2765,206.4684,4.5155,95.2765,206.4684,-4.4845,110.2905,206.4684,-4.4845,110.2905,206.4684,-4.4845,95.2765,206.4684,4.5155,110.2905,206.4684,4.5155,-120.1107,224.2791,4.5155,-106.4787,220.1112,4.5155,-106.4787,220.1112,-4.4845,-106.4787,220.1112,-4.4845,-120.1107,224.2791,4.5155,-120.1107,224.2791,-4.4845,-106.4787,220.1112,4.5155,-86.8882,216.6506,4.5155,-86.8881,216.6506,-4.4845,-86.8881,216.6506,-4.4845,-106.4787,220.1112,4.5155,-106.4787,220.1112,-4.4845,-86.8882,216.6506,4.5155,-69.8772,214.8485,4.5155,-69.8772,214.8485,-4.4845,-69.8772,214.8485,-4.4845,-86.8882,216.6506,4.5155,-86.8881,216.6506,-4.4845,-69.8772,214.8485,4.5155,-54.8633,213.7381,4.5155,-54.8633,213.7381,-4.4845,-54.8633,213.7381,-4.4845,-69.8772,214.8485,4.5155,-69.8772,214.8485,-4.4845,-54.8633,213.7381,4.5155,-39.8493,212.8604,4.5155,-39.8493,212.8604,-4.4845,-39.8493,212.8604,-4.4845,-54.8633,213.7381,4.5155,-54.8633,213.7381,-4.4845,-39.8493,212.8604,4.5155,-24.8353,212.8604,4.5155,-24.8353,212.8604,-4.4845,-24.8353,212.8604,-4.4845,-39.8493,212.8604,4.5155,-39.8493,212.8604,-4.4845,-24.8353,212.8604,4.5155,-9.8213,212.8604,4.5155,-9.8213,212.8604,-4.4845,-9.8213,212.8604,-4.4845,-24.8353,212.8604,4.5155,-24.8353,212.8604,-4.4845,-9.8213,212.8604,4.5155,5.1926,212.8604,4.5155,5.1926,212.8604,-4.4845,5.1926,212.8604,-4.4845,-9.8213,212.8604,4.5155,-9.8213,212.8604,-4.4845,5.1926,212.8604,4.5155,20.2066,212.8604,4.5155,20.2066,212.8604,-4.4845,20.2066,212.8604,-4.4845,5.1926,212.8604,4.5155,5.1926,212.8604,-4.4845,20.2066,212.8604,4.5155,35.2206,212.8604,4.5155,35.2206,212.8604,-4.4845,35.2206,212.8604,-4.4845,20.2066,212.8604,4.5155,20.2066,212.8604,-4.4845,35.2206,212.8604,4.5155,50.2346,213.7381,4.5155,50.2346,213.7381,-4.4845,50.2346,213.7381,-4.4845,35.2206,212.8604,4.5155,35.2206,212.8604,-4.4845,50.2346,213.7381,4.5155,65.2485,214.8352,4.5155,65.2485,214.8352,-4.4845,65.2485,214.8352,-4.4845,50.2346,213.7381,4.5155,50.2346,213.7381,-4.4845,65.2485,214.8352,4.5155,82.2594,216.6639,4.5155,82.2594,216.6639,-4.4845,82.2594,216.6639,-4.4845,65.2485,214.8352,4.5155,65.2485,214.8352,-4.4845,82.2594,216.6639,4.5155,101.85,220.1112,4.5155,101.85,220.1634,-4.4845,101.85,220.1634,-4.4845,82.2594,216.6639,4.5155,82.2594,216.6639,-4.4845,101.85,220.1112,4.5155,120.6734,224.2791,4.5155,120.5673,224.2034,-4.4845,120.5673,224.2034,-4.4845,101.85,220.1112,4.5155,101.85,220.1634,-4.4845,-114.9192,206.4684,4.5155,-99.9052,206.4684,4.5155,-106.4787,220.1112,4.5155,-106.4787,220.1112,4.5155,-114.9192,206.4684,4.5155,-120.1107,224.2791,4.5155,-99.9052,206.4684,4.5155,-84.8912,206.4684,4.5155,-86.8882,216.6506,4.5155,-86.8882,216.6506,4.5155,-99.9052,206.4684,4.5155,-106.4787,220.1112,4.5155,-84.8912,206.4684,4.5155,-69.8772,206.4684,4.5155,-69.8772,214.8485,4.5155,-69.8772,214.8485,4.5155,-84.8912,206.4684,4.5155,-86.8882,216.6506,4.5155,-69.8772,206.4684,4.5155,-54.8633,206.4684,4.5155,-54.8633,213.7381,4.5155,-54.8633,213.7381,4.5155,-69.8772,206.4684,4.5155,-69.8772,214.8485,4.5155,-54.8633,206.4684,4.5155,-39.8493,206.4684,4.5155,-39.8493,212.8604,4.5155,-39.8493,212.8604,4.5155,-54.8633,206.4684,4.5155,-54.8633,213.7381,4.5155,-39.8493,206.4684,4.5155,-24.8353,206.4684,4.5155,-24.8353,212.8604,4.5155,-24.8353,212.8604,4.5155,-39.8493,206.4684,4.5155,-39.8493,212.8604,4.5155,-24.8353,206.4684,4.5155,-9.8213,206.4684,4.5155,-9.8213,212.8604,4.5155,-9.8213,212.8604,4.5155,-24.8353,206.4684,4.5155,-24.8353,212.8604,4.5155,-9.8213,206.4684,4.5155,5.1926,206.4684,4.5155,5.1926,212.8604,4.5155,5.1926,212.8604,4.5155,-9.8213,206.4684,4.5155,-9.8213,212.8604,4.5155,5.1926,206.4684,4.5155,20.2066,206.4684,4.5155,20.2066,212.8604,4.5155,20.2066,212.8604,4.5155,5.1926,206.4684,4.5155,5.1926,212.8604,4.5155,20.2066,206.4684,4.5155,35.2206,206.4684,4.5155,35.2206,212.8604,4.5155,35.2206,212.8604,4.5155,20.2066,206.4684,4.5155,20.2066,212.8604,4.5155,35.2206,206.4684,4.5155,50.2346,206.4684,4.5155,50.2346,213.7381,4.5155,50.2346,213.7381,4.5155,35.2206,206.4684,4.5155,35.2206,212.8604,4.5155,50.2346,206.4684,4.5155,65.2485,206.4684,4.5155,65.2485,214.8352,4.5155,65.2485,214.8352,4.5155,50.2346,206.4684,4.5155,50.2346,213.7381,4.5155,65.2485,206.4684,4.5155,80.2625,206.4684,4.5155,82.2594,216.6639,4.5155,82.2594,216.6639,4.5155,65.2485,206.4684,4.5155,65.2485,214.8352,4.5155,80.2625,206.4684,4.5155,95.2765,206.4684,4.5155,101.85,220.1112,4.5155,101.85,220.1112,4.5155,80.2625,206.4684,4.5155,82.2594,216.6639,4.5155,95.2765,206.4684,4.5155,110.2905,206.4684,4.5155,120.6734,224.2791,4.5155,120.6734,224.2791,4.5155,95.2765,206.4684,4.5155,101.85,220.1112,4.5155,110.2905,206.4684,4.5155,110.2905,206.4684,-4.4845,120.5673,224.2034,-4.4845,120.5673,224.2034,-4.4845,110.2905,206.4684,4.5155,120.6734,224.2791,4.5155,110.2905,206.4684,-4.4845,95.2765,206.4684,-4.4845,101.85,220.1634,-4.4845,101.85,220.1634,-4.4845,110.2905,206.4684,-4.4845,120.5673,224.2034,-4.4845,95.2765,206.4684,-4.4845,80.2625,206.4684,-4.4845,82.2594,216.6639,-4.4845,82.2594,216.6639,-4.4845,95.2765,206.4684,-4.4845,101.85,220.1634,-4.4845,80.2625,206.4684,-4.4845,65.2485,206.4684,-4.4845,65.2485,214.8352,-4.4845,65.2485,214.8352,-4.4845,80.2625,206.4684,-4.4845,82.2594,216.6639,-4.4845,65.2485,206.4684,-4.4845,50.2346,206.4684,-4.4845,50.2346,213.7381,-4.4845,50.2346,213.7381,-4.4845,65.2485,206.4684,-4.4845,65.2485,214.8352,-4.4845,50.2346,206.4684,-4.4845,35.2206,206.4684,-4.4845,35.2206,212.8604,-4.4845,35.2206,212.8604,-4.4845,50.2346,206.4684,-4.4845,50.2346,213.7381,-4.4845,35.2206,206.4684,-4.4845,20.2066,206.4684,-4.4845,20.2066,212.8604,-4.4845,20.2066,212.8604,-4.4845,35.2206,206.4684,-4.4845,35.2206,212.8604,-4.4845,20.2066,206.4684,-4.4845,5.1926,206.4684,-4.4845,5.1926,212.8604,-4.4845,5.1926,212.8604,-4.4845,20.2066,206.4684,-4.4845,20.2066,212.8604,-4.4845,5.1926,206.4684,-4.4845,-9.8213,206.4684,-4.4845,-9.8213,212.8604,-4.4845,-9.8213,212.8604,-4.4845,5.1926,206.4684,-4.4845,5.1926,212.8604,-4.4845,-9.8213,206.4684,-4.4845,-24.8353,206.4684,-4.4845,-24.8353,212.8604,-4.4845,-24.8353,212.8604,-4.4845,-9.8213,206.4684,-4.4845,-9.8213,212.8604,-4.4845,-24.8353,206.4684,-4.4845,-39.8493,206.4684,-4.4845,-39.8493,212.8604,-4.4845,-39.8493,212.8604,-4.4845,-24.8353,206.4684,-4.4845,-24.8353,212.8604,-4.4845,-39.8493,206.4684,-4.4845,-54.8633,206.4684,-4.4845,-54.8633,213.7381,-4.4845,-54.8633,213.7381,-4.4845,-39.8493,206.4684,-4.4845,-39.8493,212.8604,-4.4845,-54.8633,206.4684,-4.4845,-69.8772,206.4684,-4.4845,-69.8772,214.8485,-4.4845,-69.8772,214.8485,-4.4845,-54.8633,206.4684,-4.4845,-54.8633,213.7381,-4.4845,-69.8772,206.4684,-4.4845,-84.8912,206.4684,-4.4845,-86.8881,216.6506,-4.4845,-86.8881,216.6506,-4.4845,-69.8772,206.4684,-4.4845,-69.8772,214.8485,-4.4845,-84.8912,206.4684,-4.4845,-99.9052,206.4684,-4.4845,-106.4787,220.1112,-4.4845,-106.4787,220.1112,-4.4845,-84.8912,206.4684,-4.4845,-86.8881,216.6506,-4.4845,-99.9052,206.4684,-4.4845,-114.9192,206.4684,-4.4845,-120.1107,224.2791,-4.4845,-120.1107,224.2791,-4.4845,-99.9052,206.4684,-4.4845,-106.4787,220.1112,-4.4845,-114.9192,206.4684,-4.4845,-114.9192,206.4684,4.5155,-120.1107,224.2791,4.5155,-120.1107,224.2791,4.5155,-114.9192,206.4684,-4.4845,-120.1107,224.2791,-4.4845] )
   , make_vbo(NORMAL, [-1,0,-0.0008,-0.94,0,0.3413,-0.9197,0.2064,0.3339,-0.9197,0.2064,0.3339,-1,0,-0.0008,-0.9785,0.2064,-0.0008,-0.94,0,0.3413,-0.7666,0,0.6422,-0.75,0.2064,0.6284,-0.75,0.2064,0.6284,-0.94,0,0.3413,-0.9197,0.2064,0.3339,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.4899,0.2064,0.847,-0.4899,0.2064,0.847,-0.7666,0,0.6422,-0.75,0.2064,0.6284,-0.5007,0,0.8656,-0.1744,0,0.9847,-0.1707,0.2064,0.9635,-0.1707,0.2064,0.9635,-0.5007,0,0.8656,-0.4899,0.2064,0.847,-0.1744,0,0.9847,0.1729,0,0.9849,0.1691,0.2064,0.9637,0.1691,0.2064,0.9637,-0.1744,0,0.9847,-0.1707,0.2064,0.9635,0.1729,0,0.9849,0.4993,0,0.8664,0.4886,0.2064,0.8478,0.4886,0.2064,0.8478,0.1729,0,0.9849,0.1691,0.2064,0.9637,0.4993,0,0.8664,0.7655,0,0.6434,0.749,0.2064,0.6295,0.749,0.2064,0.6295,0.4993,0,0.8664,0.4886,0.2064,0.8478,0.7655,0,0.6434,0.9394,0,0.3428,0.9192,0.2064,0.3354,0.9192,0.2064,0.3354,0.7655,0,0.6434,0.749,0.2064,0.6295,0.9394,0,0.3428,1,0,0.0008,0.9785,0.2064,0.0008,0.9785,0.2064,0.0008,0.9394,0,0.3428,0.9192,0.2064,0.3354,1,0,0.0008,0.94,0,-0.3413,0.9197,0.2064,-0.3339,0.9197,0.2064,-0.3339,1,0,0.0008,0.9785,0.2064,0.0008,0.94,0,-0.3413,0.7666,0,-0.6422,0.75,0.2064,-0.6284,0.75,0.2064,-0.6284,0.94,0,-0.3413,0.9197,0.2064,-0.3339,0.7666,0,-0.6422,0.5007,0,-0.8656,0.4899,0.2064,-0.847,0.4899,0.2064,-0.847,0.7666,0,-0.6422,0.75,0.2064,-0.6284,0.5007,0,-0.8656,0.1744,0,-0.9847,0.1707,0.2064,-0.9635,0.1707,0.2064,-0.9635,0.5007,0,-0.8656,0.4899,0.2064,-0.847,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.1691,0.2064,-0.9637,-0.1691,0.2064,-0.9637,0.1744,0,-0.9847,0.1707,0.2064,-0.9635,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.4886,0.2064,-0.8478,-0.4886,0.2064,-0.8478,-0.1729,0,-0.9849,-0.1691,0.2064,-0.9637,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.749,0.2064,-0.6295,-0.749,0.2064,-0.6295,-0.4993,0,-0.8664,-0.4886,0.2064,-0.8478,-0.7655,0,-0.6434,-0.9394,0,-0.3428,-0.9192,0.2064,-0.3354,-0.9192,0.2064,-0.3354,-0.7655,0,-0.6434,-0.749,0.2064,-0.6295,-0.9394,0,-0.3428,-1,0,-0.0008,-0.9785,0.2064,-0.0008,-0.9785,0.2064,-0.0008,-0.9394,0,-0.3428,-0.9192,0.2064,-0.3354,-0.9197,0.2064,0.3339,-0.7389,0.6181,0.2683,-0.7861,0.6181,-0.0006,-0.7861,0.6181,-0.0006,-0.9197,0.2064,0.3339,-0.9785,0.2064,-0.0008,-0.75,0.2064,0.6284,-0.6026,0.6181,0.5048,-0.7389,0.6181,0.2683,-0.7389,0.6181,0.2683,-0.75,0.2064,0.6284,-0.9197,0.2064,0.3339,-0.4899,0.2064,0.847,-0.3936,0.6181,0.6804,-0.6026,0.6181,0.5048,-0.6026,0.6181,0.5048,-0.4899,0.2064,0.847,-0.75,0.2064,0.6284,-0.1707,0.2064,0.9635,-0.1371,0.6181,0.774,-0.3936,0.6181,0.6804,-0.3936,0.6181,0.6804,-0.1707,0.2064,0.9635,-0.4899,0.2064,0.847,0.1691,0.2064,0.9637,0.1359,0.6181,0.7742,-0.1371,0.6181,0.774,-0.1371,0.6181,0.774,0.1691,0.2064,0.9637,-0.1707,0.2064,0.9635,0.4886,0.2064,0.8478,0.3925,0.6181,0.6811,0.1359,0.6181,0.7742,0.1359,0.6181,0.7742,0.4886,0.2064,0.8478,0.1691,0.2064,0.9637,0.749,0.2064,0.6295,0.6018,0.6181,0.5057,0.3925,0.6181,0.6811,0.3925,0.6181,0.6811,0.749,0.2064,0.6295,0.4886,0.2064,0.8478,0.9192,0.2064,0.3354,0.7384,0.6181,0.2694,0.6018,0.6181,0.5057,0.6018,0.6181,0.5057,0.9192,0.2064,0.3354,0.749,0.2064,0.6295,0.9785,0.2064,0.0008,0.7861,0.6181,0.0006,0.7384,0.6181,0.2694,0.7384,0.6181,0.2694,0.9785,0.2064,0.0008,0.9192,0.2064,0.3354,0.9197,0.2064,-0.3339,0.7389,0.6181,-0.2683,0.7861,0.6181,0.0006,0.7861,0.6181,0.0006,0.9197,0.2064,-0.3339,0.9785,0.2064,0.0008,0.75,0.2064,-0.6284,0.6026,0.6181,-0.5048,0.7389,0.6181,-0.2683,0.7389,0.6181,-0.2683,0.75,0.2064,-0.6284,0.9197,0.2064,-0.3339,0.4899,0.2064,-0.847,0.3936,0.6181,-0.6804,0.6026,0.6181,-0.5048,0.6026,0.6181,-0.5048,0.4899,0.2064,-0.847,0.75,0.2064,-0.6284,0.1707,0.2064,-0.9635,0.1371,0.6181,-0.774,0.3936,0.6181,-0.6804,0.3936,0.6181,-0.6804,0.1707,0.2064,-0.9635,0.4899,0.2064,-0.847,-0.1691,0.2064,-0.9637,-0.1359,0.6181,-0.7742,0.1371,0.6181,-0.774,0.1371,0.6181,-0.774,-0.1691,0.2064,-0.9637,0.1707,0.2064,-0.9635,-0.4886,0.2064,-0.8478,-0.3925,0.6181,-0.6811,-0.1359,0.6181,-0.7742,-0.1359,0.6181,-0.7742,-0.4886,0.2064,-0.8478,-0.1691,0.2064,-0.9637,-0.749,0.2064,-0.6295,-0.6018,0.6181,-0.5057,-0.3925,0.6181,-0.6811,-0.3925,0.6181,-0.6811,-0.749,0.2064,-0.6295,-0.4886,0.2064,-0.8478,-0.9192,0.2064,-0.3354,-0.7384,0.6181,-0.2694,-0.6018,0.6181,-0.5057,-0.6018,0.6181,-0.5057,-0.9192,0.2064,-0.3354,-0.749,0.2064,-0.6295,-0.9785,0.2064,-0.0008,-0.7861,0.6181,-0.0006,-0.7384,0.6181,-0.2694,-0.7384,0.6181,-0.2694,-0.9785,0.2064,-0.0008,-0.9192,0.2064,-0.3354,-0.8764,0.4816,-0.0007,-0.8238,0.4816,0.2991,-0.9232,0.188,0.3352,-0.9232,0.188,0.3352,-0.8764,0.4816,-0.0007,-0.9822,0.188,-0.0008,-0.8238,0.4816,0.2991,-0.6718,0.4816,0.5628,-0.7529,0.188,0.6307,-0.7529,0.188,0.6307,-0.8238,0.4816,0.2991,-0.9232,0.188,0.3352,-0.6718,0.4816,0.5628,-0.4388,0.4816,0.7586,-0.4918,0.188,0.8502,-0.4918,0.188,0.8502,-0.6718,0.4816,0.5628,-0.7529,0.188,0.6307,-0.4388,0.4816,0.7586,-0.1529,0.4816,0.8629,-0.1713,0.188,0.9671,-0.1713,0.188,0.9671,-0.4388,0.4816,0.7586,-0.4918,0.188,0.8502,-0.1529,0.4816,0.8629,0.1515,0.4816,0.8632,0.1698,0.188,0.9674,0.1698,0.188,0.9674,-0.1529,0.4816,0.8629,-0.1713,0.188,0.9671,0.1515,0.4816,0.8632,0.4376,0.4816,0.7593,0.4904,0.188,0.851,0.4904,0.188,0.851,0.1515,0.4816,0.8632,0.1698,0.188,0.9674,0.4376,0.4816,0.7593,0.6709,0.4816,0.5638,0.7519,0.188,0.6319,0.7519,0.188,0.6319,0.4376,0.4816,0.7593,0.4904,0.188,0.851,0.6709,0.4816,0.5638,0.8233,0.4816,0.3004,0.9227,0.188,0.3367,0.9227,0.188,0.3367,0.6709,0.4816,0.5638,0.7519,0.188,0.6319,0.8233,0.4816,0.3004,0.8764,0.4816,0.0007,0.9822,0.188,0.0008,0.9822,0.188,0.0008,0.8233,0.4816,0.3004,0.9227,0.188,0.3367,0.8764,0.4816,0.0007,0.8238,0.4816,-0.2991,0.9232,0.188,-0.3352,0.9232,0.188,-0.3352,0.8764,0.4816,0.0007,0.9822,0.188,0.0008,0.8238,0.4816,-0.2991,0.6718,0.4816,-0.5628,0.7529,0.188,-0.6307,0.7529,0.188,-0.6307,0.8238,0.4816,-0.2991,0.9232,0.188,-0.3352,0.6718,0.4816,-0.5628,0.4388,0.4816,-0.7586,0.4918,0.188,-0.8502,0.4918,0.188,-0.8502,0.6718,0.4816,-0.5628,0.7529,0.188,-0.6307,0.4388,0.4816,-0.7586,0.1529,0.4816,-0.8629,0.1713,0.188,-0.9671,0.1713,0.188,-0.9671,0.4388,0.4816,-0.7586,0.4918,0.188,-0.8502,0.1529,0.4816,-0.8629,-0.1515,0.4816,-0.8632,-0.1698,0.188,-0.9674,-0.1698,0.188,-0.9674,0.1529,0.4816,-0.8629,0.1713,0.188,-0.9671,-0.1515,0.4816,-0.8632,-0.4376,0.4816,-0.7593,-0.4904,0.188,-0.851,-0.4904,0.188,-0.851,-0.1515,0.4816,-0.8632,-0.1698,0.188,-0.9674,-0.4376,0.4816,-0.7593,-0.6709,0.4816,-0.5639,-0.7519,0.188,-0.6319,-0.7519,0.188,-0.6319,-0.4376,0.4816,-0.7593,-0.4904,0.188,-0.851,-0.6709,0.4816,-0.5639,-0.8233,0.4816,-0.3004,-0.9227,0.188,-0.3367,-0.9227,0.188,-0.3367,-0.6709,0.4816,-0.5639,-0.7519,0.188,-0.6319,-0.8233,0.4816,-0.3004,-0.8764,0.4816,-0.0007,-0.9822,0.188,-0.0008,-0.9822,0.188,-0.0008,-0.8233,0.4816,-0.3004,-0.9227,0.188,-0.3367,-0.9232,0.188,0.3352,-0.7584,0.5907,0.2754,-0.8069,0.5907,-0.0006,-0.8069,0.5907,-0.0006,-0.9232,0.188,0.3352,-0.9822,0.188,-0.0008,-0.7529,0.188,0.6307,-0.6185,0.5907,0.5182,-0.7584,0.5907,0.2754,-0.7584,0.5907,0.2754,-0.7529,0.188,0.6307,-0.9232,0.188,0.3352,-0.4918,0.188,0.8502,-0.404,0.5907,0.6985,-0.6185,0.5907,0.5182,-0.6185,0.5907,0.5182,-0.4918,0.188,0.8502,-0.7529,0.188,0.6307,-0.1713,0.188,0.9671,-0.1407,0.5907,0.7945,-0.404,0.5907,0.6985,-0.404,0.5907,0.6985,-0.1713,0.188,0.9671,-0.4918,0.188,0.8502,0.1698,0.188,0.9674,0.1395,0.5907,0.7947,-0.1407,0.5907,0.7945,-0.1407,0.5907,0.7945,0.1698,0.188,0.9674,-0.1713,0.188,0.9671,0.4904,0.188,0.851,0.4029,0.5907,0.6991,0.1395,0.5907,0.7947,0.1395,0.5907,0.7947,0.4904,0.188,0.851,0.1698,0.188,0.9674,0.7519,0.188,0.6319,0.6177,0.5907,0.5191,0.4029,0.5907,0.6991,0.4029,0.5907,0.6991,0.7519,0.188,0.6319,0.4904,0.188,0.851,0.9227,0.188,0.3367,0.758,0.5907,0.2766,0.6177,0.5907,0.5191,0.6177,0.5907,0.5191,0.9227,0.188,0.3367,0.7519,0.188,0.6319,0.9822,0.188,0.0008,0.8069,0.5907,0.0006,0.758,0.5907,0.2766,0.758,0.5907,0.2766,0.9822,0.188,0.0008,0.9227,0.188,0.3367,0.9232,0.188,-0.3352,0.7584,0.5907,-0.2754,0.8069,0.5907,0.0006,0.8069,0.5907,0.0006,0.9232,0.188,-0.3352,0.9822,0.188,0.0008,0.7529,0.188,-0.6307,0.6185,0.5907,-0.5182,0.7584,0.5907,-0.2754,0.7584,0.5907,-0.2754,0.7529,0.188,-0.6307,0.9232,0.188,-0.3352,0.4918,0.188,-0.8502,0.404,0.5907,-0.6985,0.6185,0.5907,-0.5182,0.6185,0.5907,-0.5182,0.4918,0.188,-0.8502,0.7529,0.188,-0.6307,0.1713,0.188,-0.9671,0.1407,0.5907,-0.7945,0.404,0.5907,-0.6985,0.404,0.5907,-0.6985,0.1713,0.188,-0.9671,0.4918,0.188,-0.8502,-0.1698,0.188,-0.9674,-0.1395,0.5907,-0.7947,0.1407,0.5907,-0.7945,0.1407,0.5907,-0.7945,-0.1698,0.188,-0.9674,0.1713,0.188,-0.9671,-0.4904,0.188,-0.851,-0.4029,0.5907,-0.6991,-0.1395,0.5907,-0.7947,-0.1395,0.5907,-0.7947,-0.4904,0.188,-0.851,-0.1698,0.188,-0.9674,-0.7519,0.188,-0.6319,-0.6177,0.5907,-0.5191,-0.4029,0.5907,-0.6991,-0.4029,0.5907,-0.6991,-0.7519,0.188,-0.6319,-0.4904,0.188,-0.851,-0.9227,0.188,-0.3367,-0.758,0.5907,-0.2766,-0.6177,0.5907,-0.5191,-0.6177,0.5907,-0.5191,-0.9227,0.188,-0.3367,-0.7519,0.188,-0.6319,-0.9822,0.188,-0.0008,-0.8069,0.5907,-0.0006,-0.758,0.5907,-0.2766,-0.758,0.5907,-0.2766,-0.9822,0.188,-0.0008,-0.9227,0.188,-0.3367,-0.8834,0.4687,-0.0007,-0.8303,0.4687,0.3015,-0.94,0,0.3413,-0.94,0,0.3413,-0.8834,0.4687,-0.0007,-1,0,-0.0008,-0.8303,0.4687,0.3015,-0.6772,0.4687,0.5673,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.8303,0.4687,0.3015,-0.94,0,0.3413,-0.6772,0.4687,0.5673,-0.4423,0.4687,0.7647,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.6772,0.4687,0.5673,-0.7666,0,0.6422,-0.4423,0.4687,0.7647,-0.1541,0.4687,0.8698,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.4423,0.4687,0.7647,-0.5007,0,0.8656,-0.1541,0.4687,0.8698,0.1527,0.4687,0.8701,0.1729,0,0.9849,0.1729,0,0.9849,-0.1541,0.4687,0.8698,-0.1744,0,0.9847,0.1527,0.4687,0.8701,0.4411,0.4687,0.7654,0.4993,0,0.8664,0.4993,0,0.8664,0.1527,0.4687,0.8701,0.1729,0,0.9849,0.4411,0.4687,0.7654,0.6763,0.4687,0.5684,0.7655,0,0.6434,0.7655,0,0.6434,0.4411,0.4687,0.7654,0.4993,0,0.8664,0.6763,0.4687,0.5684,0.8299,0.4687,0.3028,0.9394,0,0.3428,0.9394,0,0.3428,0.6763,0.4687,0.5684,0.7655,0,0.6434,0.8299,0.4687,0.3028,0.8834,0.4687,0.0007,1,0,0.0008,1,0,0.0008,0.8299,0.4687,0.3028,0.9394,0,0.3428,0.8834,0.4687,0.0007,0.8303,0.4687,-0.3015,0.94,0,-0.3413,0.94,0,-0.3413,0.8834,0.4687,0.0007,1,0,0.0008,0.8303,0.4687,-0.3015,0.6772,0.4687,-0.5673,0.7666,0,-0.6422,0.7666,0,-0.6422,0.8303,0.4687,-0.3015,0.94,0,-0.3413,0.6772,0.4687,-0.5673,0.4423,0.4687,-0.7647,0.5007,0,-0.8656,0.5007,0,-0.8656,0.6772,0.4687,-0.5673,0.7666,0,-0.6422,0.4423,0.4687,-0.7647,0.1541,0.4687,-0.8698,0.1744,0,-0.9847,0.1744,0,-0.9847,0.4423,0.4687,-0.7647,0.5007,0,-0.8656,0.1541,0.4687,-0.8698,-0.1527,0.4687,-0.8701,-0.1729,0,-0.9849,-0.1729,0,-0.9849,0.1541,0.4687,-0.8698,0.1744,0,-0.9847,-0.1527,0.4687,-0.8701,-0.4411,0.4687,-0.7654,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.1527,0.4687,-0.8701,-0.1729,0,-0.9849,-0.4411,0.4687,-0.7654,-0.6763,0.4687,-0.5684,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.4411,0.4687,-0.7654,-0.4993,0,-0.8664,-0.6763,0.4687,-0.5684,-0.8299,0.4687,-0.3028,-0.9394,0,-0.3428,-0.9394,0,-0.3428,-0.6763,0.4687,-0.5684,-0.7655,0,-0.6434,-0.8299,0.4687,-0.3028,-0.8834,0.4687,-0.0007,-1,0,-0.0008,-1,0,-0.0008,-0.8299,0.4687,-0.3028,-0.9394,0,-0.3428,-1,0,-0.0008,-0.94,0,0.3413,-0.9183,0,0.3958,-0.9183,0,0.3958,-1,0,-0.0008,-1,0,-0.0008,-0.94,0,0.3413,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.94,0,0.3413,-0.9183,0,0.3958,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.1744,0,0.9847,0.1729,0,0.9849,0.1729,0,0.9849,0.1729,0,0.9849,-0.1744,0,0.9847,-0.1744,0,0.9847,0.1729,0,0.9849,0.4993,0,0.8664,0.4993,0,0.8664,0.4993,0,0.8664,0.1729,0,0.9849,0.1729,0,0.9849,0.4993,0,0.8664,0.7655,0,0.6434,0.7655,0,0.6434,0.7655,0,0.6434,0.4993,0,0.8664,0.4993,0,0.8664,0.7655,0,0.6434,0.9394,0,0.3428,0.9177,0,0.3973,0.9177,0,0.3973,0.7655,0,0.6434,0.7655,0,0.6434,0.9394,0,0.3428,1,0,0.0008,1,0,0.0008,1,0,0.0008,0.9394,0,0.3428,0.9177,0,0.3973,1,0,0.0008,0.94,0,-0.3413,0.9183,0,-0.3958,0.9183,0,-0.3958,1,0,0.0008,1,0,0.0008,0.94,0,-0.3413,0.7666,0,-0.6422,0.7666,0,-0.6422,0.7666,0,-0.6422,0.94,0,-0.3413,0.9183,0,-0.3958,0.7666,0,-0.6422,0.5007,0,-0.8656,0.5007,0,-0.8656,0.5007,0,-0.8656,0.7666,0,-0.6422,0.7666,0,-0.6422,0.5007,0,-0.8656,0.1744,0,-0.9847,0.1744,0,-0.9847,0.1744,0,-0.9847,0.5007,0,-0.8656,0.5007,0,-0.8656,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.1729,0,-0.9849,0.1744,0,-0.9847,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.9394,0,-0.3428,-0.9177,0,-0.3973,-0.9177,0,-0.3973,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.9394,0,-0.3428,-1,0,-0.0008,-1,0,-0.0008,-1,0,-0.0008,-0.9394,0,-0.3428,-0.9177,0,-0.3973,-0.9183,0,0.3958,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.9183,0,0.3958,-0.9183,0,0.3958,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.1744,0,0.9847,0.1729,0,0.9849,0.1729,0,0.9849,0.1729,0,0.9849,-0.1744,0,0.9847,-0.1744,0,0.9847,0.1729,0,0.9849,0.4993,0,0.8664,0.4993,0,0.8664,0.4993,0,0.8664,0.1729,0,0.9849,0.1729,0,0.9849,0.4993,0,0.8664,0.7655,0,0.6434,0.7655,0,0.6434,0.7655,0,0.6434,0.4993,0,0.8664,0.4993,0,0.8664,0.7655,0,0.6434,0.9177,0,0.3973,0.9177,0,0.3973,0.9177,0,0.3973,0.7655,0,0.6434,0.7655,0,0.6434,0.963,-0.2694,-0.001,0.963,-0.2694,0.0008,0.963,-0.2694,0.0008,0.963,-0.2694,0.0008,0.963,-0.2694,-0.001,0.963,-0.2694,-0.001,0.963,-0.2694,0.0008,0.963,-0.2694,0.0025,0.963,-0.2694,0.0026,0.963,-0.2694,0.0026,0.963,-0.2694,0.0008,0.963,-0.2694,0.0008,0.9183,0,-0.3958,0.7666,0,-0.6422,0.7666,0,-0.6422,0.7666,0,-0.6422,0.9183,0,-0.3958,0.9183,0,-0.3958,0.7666,0,-0.6422,0.5007,0,-0.8656,0.5007,0,-0.8656,0.5007,0,-0.8656,0.7666,0,-0.6422,0.7666,0,-0.6422,0.5007,0,-0.8656,0.1744,0,-0.9847,0.1744,0,-0.9847,0.1744,0,-0.9847,0.5007,0,-0.8656,0.5007,0,-0.8656,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.1729,0,-0.9849,0.1744,0,-0.9847,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.9177,0,-0.3973,-0.9177,0,-0.3973,-0.9177,0,-0.3973,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-1,0,-0.0008,-0.9183,0,0.3958,-0.94,0,0.3413,-0.94,0,0.3413,-1,0,-0.0008,-1,0,-0.0008,-0.9183,0,0.3958,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.9183,0,0.3958,-0.94,0,0.3413,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.7666,0,0.6422,-0.7666,0,0.6422,-0.5007,0,0.8656,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.1744,0,0.9847,-0.5007,0,0.8656,-0.5007,0,0.8656,-0.1744,0,0.9847,0.1729,0,0.9849,0.1729,0,0.9849,0.1729,0,0.9849,-0.1744,0,0.9847,-0.1744,0,0.9847,0.1729,0,0.9849,0.4993,0,0.8664,0.4993,0,0.8664,0.4993,0,0.8664,0.1729,0,0.9849,0.1729,0,0.9849,0.4993,0,0.8664,0.7655,0,0.6434,0.7655,0,0.6434,0.7655,0,0.6434,0.4993,0,0.8664,0.4993,0,0.8664,0.7655,0,0.6434,0.9177,0,0.3973,0.9394,0,0.3428,0.9394,0,0.3428,0.7655,0,0.6434,0.7655,0,0.6434,0.9177,0,0.3973,1,0,0.0008,1,0,0.0008,1,0,0.0008,0.9177,0,0.3973,0.9394,0,0.3428,1,0,0.0008,0.9183,0,-0.3958,0.94,0,-0.3413,0.94,0,-0.3413,1,0,0.0008,1,0,0.0008,0.9183,0,-0.3958,0.7666,0,-0.6422,0.7666,0,-0.6422,0.7666,0,-0.6422,0.9183,0,-0.3958,0.94,0,-0.3413,0.7666,0,-0.6422,0.5007,0,-0.8656,0.5007,0,-0.8656,0.5007,0,-0.8656,0.7666,0,-0.6422,0.7666,0,-0.6422,0.5007,0,-0.8656,0.1744,0,-0.9847,0.1744,0,-0.9847,0.1744,0,-0.9847,0.5007,0,-0.8656,0.5007,0,-0.8656,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.1729,0,-0.9849,0.1744,0,-0.9847,0.1744,0,-0.9847,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.1729,0,-0.9849,-0.1729,0,-0.9849,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.4993,0,-0.8664,-0.4993,0,-0.8664,-0.7655,0,-0.6434,-0.9177,0,-0.3973,-0.9394,0,-0.3428,-0.9394,0,-0.3428,-0.7655,0,-0.6434,-0.7655,0,-0.6434,-0.9177,0,-0.3973,-1,0,-0.0008,-1,0,-0.0008,-1,0,-0.0008,-0.9177,0,-0.3973,-0.9394,0,-0.3428,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,-0.8303,0.4687,0.3015,-0.8834,0.4687,-0.0007,-0.8069,0.5907,-0.0006,-0.8069,0.5907,-0.0006,-0.8303,0.4687,0.3015,-0.7584,0.5907,0.2754,-0.6772,0.4687,0.5673,-0.8303,0.4687,0.3015,-0.7584,0.5907,0.2754,-0.7584,0.5907,0.2754,-0.6772,0.4687,0.5673,-0.6185,0.5907,0.5182,-0.4423,0.4687,0.7647,-0.6772,0.4687,0.5673,-0.6185,0.5907,0.5182,-0.6185,0.5907,0.5182,-0.4423,0.4687,0.7647,-0.404,0.5907,0.6985,-0.1541,0.4687,0.8698,-0.4423,0.4687,0.7647,-0.404,0.5907,0.6985,-0.404,0.5907,0.6985,-0.1541,0.4687,0.8698,-0.1407,0.5907,0.7945,0.1527,0.4687,0.8701,-0.1541,0.4687,0.8698,-0.1407,0.5907,0.7945,-0.1407,0.5907,0.7945,0.1527,0.4687,0.8701,0.1395,0.5907,0.7947,0.4411,0.4687,0.7654,0.1527,0.4687,0.8701,0.1395,0.5907,0.7947,0.1395,0.5907,0.7947,0.4411,0.4687,0.7654,0.4029,0.5907,0.6991,0.6763,0.4687,0.5684,0.4411,0.4687,0.7654,0.4029,0.5907,0.6991,0.4029,0.5907,0.6991,0.6763,0.4687,0.5684,0.6177,0.5907,0.5191,0.8299,0.4687,0.3028,0.6763,0.4687,0.5684,0.6177,0.5907,0.5191,0.6177,0.5907,0.5191,0.8299,0.4687,0.3028,0.758,0.5907,0.2766,0.8834,0.4687,0.0007,0.8299,0.4687,0.3028,0.758,0.5907,0.2766,0.758,0.5907,0.2766,0.8834,0.4687,0.0007,0.8069,0.5907,0.0006,0.8303,0.4687,-0.3015,0.8834,0.4687,0.0007,0.8069,0.5907,0.0006,0.8069,0.5907,0.0006,0.8303,0.4687,-0.3015,0.7584,0.5907,-0.2754,0.6772,0.4687,-0.5673,0.8303,0.4687,-0.3015,0.7584,0.5907,-0.2754,0.7584,0.5907,-0.2754,0.6772,0.4687,-0.5673,0.6185,0.5907,-0.5182,0.4423,0.4687,-0.7647,0.6772,0.4687,-0.5673,0.6185,0.5907,-0.5182,0.6185,0.5907,-0.5182,0.4423,0.4687,-0.7647,0.404,0.5907,-0.6985,0.1541,0.4687,-0.8698,0.4423,0.4687,-0.7647,0.404,0.5907,-0.6985,0.404,0.5907,-0.6985,0.1541,0.4687,-0.8698,0.1407,0.5907,-0.7945,-0.1527,0.4687,-0.8701,0.1541,0.4687,-0.8698,0.1407,0.5907,-0.7945,0.1407,0.5907,-0.7945,-0.1527,0.4687,-0.8701,-0.1395,0.5907,-0.7947,-0.4411,0.4687,-0.7654,-0.1527,0.4687,-0.8701,-0.1395,0.5907,-0.7947,-0.1395,0.5907,-0.7947,-0.4411,0.4687,-0.7654,-0.4029,0.5907,-0.6991,-0.6763,0.4687,-0.5684,-0.4411,0.4687,-0.7654,-0.4029,0.5907,-0.6991,-0.4029,0.5907,-0.6991,-0.6763,0.4687,-0.5684,-0.6177,0.5907,-0.5191,-0.8299,0.4687,-0.3028,-0.6763,0.4687,-0.5684,-0.6177,0.5907,-0.5191,-0.6177,0.5907,-0.5191,-0.8299,0.4687,-0.3028,-0.758,0.5907,-0.2766,-0.8834,0.4687,-0.0007,-0.8299,0.4687,-0.3028,-0.758,0.5907,-0.2766,-0.758,0.5907,-0.2766,-0.8834,0.4687,-0.0007,-0.8069,0.5907,-0.0006,-0.8238,0.4816,0.2991,-0.8764,0.4816,-0.0007,-0.7861,0.6181,-0.0006,-0.7861,0.6181,-0.0006,-0.8238,0.4816,0.2991,-0.7389,0.6181,0.2683,-0.6718,0.4816,0.5628,-0.8238,0.4816,0.2991,-0.7389,0.6181,0.2683,-0.7389,0.6181,0.2683,-0.6718,0.4816,0.5628,-0.6026,0.6181,0.5048,-0.4388,0.4816,0.7586,-0.6718,0.4816,0.5628,-0.6026,0.6181,0.5048,-0.6026,0.6181,0.5048,-0.4388,0.4816,0.7586,-0.3936,0.6181,0.6804,-0.1529,0.4816,0.8629,-0.4388,0.4816,0.7586,-0.3936,0.6181,0.6804,-0.3936,0.6181,0.6804,-0.1529,0.4816,0.8629,-0.1371,0.6181,0.774,0.1515,0.4816,0.8632,-0.1529,0.4816,0.8629,-0.1371,0.6181,0.774,-0.1371,0.6181,0.774,0.1515,0.4816,0.8632,0.1359,0.6181,0.7742,0.4376,0.4816,0.7593,0.1515,0.4816,0.8632,0.1359,0.6181,0.7742,0.1359,0.6181,0.7742,0.4376,0.4816,0.7593,0.3925,0.6181,0.6811,0.6709,0.4816,0.5638,0.4376,0.4816,0.7593,0.3925,0.6181,0.6811,0.3925,0.6181,0.6811,0.6709,0.4816,0.5638,0.6018,0.6181,0.5057,0.8233,0.4816,0.3004,0.6709,0.4816,0.5638,0.6018,0.6181,0.5057,0.6018,0.6181,0.5057,0.8233,0.4816,0.3004,0.7384,0.6181,0.2694,0.8764,0.4816,0.0007,0.8233,0.4816,0.3004,0.7384,0.6181,0.2694,0.7384,0.6181,0.2694,0.8764,0.4816,0.0007,0.7861,0.6181,0.0006,0.8238,0.4816,-0.2991,0.8764,0.4816,0.0007,0.7861,0.6181,0.0006,0.7861,0.6181,0.0006,0.8238,0.4816,-0.2991,0.7389,0.6181,-0.2683,0.6718,0.4816,-0.5628,0.8238,0.4816,-0.2991,0.7389,0.6181,-0.2683,0.7389,0.6181,-0.2683,0.6718,0.4816,-0.5628,0.6026,0.6181,-0.5048,0.4388,0.4816,-0.7586,0.6718,0.4816,-0.5628,0.6026,0.6181,-0.5048,0.6026,0.6181,-0.5048,0.4388,0.4816,-0.7586,0.3936,0.6181,-0.6804,0.1529,0.4816,-0.8629,0.4388,0.4816,-0.7586,0.3936,0.6181,-0.6804,0.3936,0.6181,-0.6804,0.1529,0.4816,-0.8629,0.1371,0.6181,-0.774,-0.1515,0.4816,-0.8632,0.1529,0.4816,-0.8629,0.1371,0.6181,-0.774,0.1371,0.6181,-0.774,-0.1515,0.4816,-0.8632,-0.1359,0.6181,-0.7742,-0.4376,0.4816,-0.7593,-0.1515,0.4816,-0.8632,-0.1359,0.6181,-0.7742,-0.1359,0.6181,-0.7742,-0.4376,0.4816,-0.7593,-0.3925,0.6181,-0.6811,-0.6709,0.4816,-0.5639,-0.4376,0.4816,-0.7593,-0.3925,0.6181,-0.6811,-0.3925,0.6181,-0.6811,-0.6709,0.4816,-0.5639,-0.6018,0.6181,-0.5057,-0.8233,0.4816,-0.3004,-0.6709,0.4816,-0.5639,-0.6018,0.6181,-0.5057,-0.6018,0.6181,-0.5057,-0.8233,0.4816,-0.3004,-0.7384,0.6181,-0.2694,-0.8764,0.4816,-0.0007,-0.8233,0.4816,-0.3004,-0.7384,0.6181,-0.2694,-0.7384,0.6181,-0.2694,-0.8764,0.4816,-0.0007,-0.7861,0.6181,-0.0006,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,1,0,0,0.9397,0,-0.342,0.9195,0.2064,-0.3347,0.9195,0.2064,-0.3347,1,0,0,0.9785,0.2064,0,0.9397,0,-0.342,0.766,0,-0.6428,0.7495,0.2064,-0.6289,0.7495,0.2064,-0.6289,0.9397,0,-0.342,0.9195,0.2064,-0.3347,0.766,0,-0.6428,0.5,0,-0.866,0.4892,0.2064,-0.8474,0.4892,0.2064,-0.8474,0.766,0,-0.6428,0.7495,0.2064,-0.6289,0.5,0,-0.866,0.1736,0,-0.9848,0.1699,0.2064,-0.9636,0.1699,0.2064,-0.9636,0.5,0,-0.866,0.4892,0.2064,-0.8474,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1699,0.2064,-0.9636,-0.1699,0.2064,-0.9636,0.1736,0,-0.9848,0.1699,0.2064,-0.9636,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.4892,0.2064,-0.8474,-0.4892,0.2064,-0.8474,-0.1736,0,-0.9848,-0.1699,0.2064,-0.9636,-0.5,0,-0.866,-0.766,0,-0.6428,-0.7495,0.2064,-0.6289,-0.7495,0.2064,-0.6289,-0.5,0,-0.866,-0.4892,0.2064,-0.8474,-0.766,0,-0.6428,-0.9397,0,-0.342,-0.9195,0.2064,-0.3347,-0.9195,0.2064,-0.3347,-0.766,0,-0.6428,-0.7495,0.2064,-0.6289,-0.9397,0,-0.342,-1,0,0,-0.9785,0.2064,0,-0.9785,0.2064,0,-0.9397,0,-0.342,-0.9195,0.2064,-0.3347,-1,0,0,-0.9397,0,0.342,-0.9195,0.2064,0.3347,-0.9195,0.2064,0.3347,-1,0,0,-0.9785,0.2064,0,-0.9397,0,0.342,-0.766,0,0.6428,-0.7495,0.2064,0.6289,-0.7495,0.2064,0.6289,-0.9397,0,0.342,-0.9195,0.2064,0.3347,-0.766,0,0.6428,-0.5,0,0.866,-0.4892,0.2064,0.8474,-0.4892,0.2064,0.8474,-0.766,0,0.6428,-0.7495,0.2064,0.6289,-0.5,0,0.866,-0.1736,0,0.9848,-0.1699,0.2064,0.9636,-0.1699,0.2064,0.9636,-0.5,0,0.866,-0.4892,0.2064,0.8474,-0.1736,0,0.9848,0.1736,0,0.9848,0.1699,0.2064,0.9636,0.1699,0.2064,0.9636,-0.1736,0,0.9848,-0.1699,0.2064,0.9636,0.1736,0,0.9848,0.5,0,0.866,0.4892,0.2064,0.8474,0.4892,0.2064,0.8474,0.1736,0,0.9848,0.1699,0.2064,0.9636,0.5,0,0.866,0.766,0,0.6428,0.7495,0.2064,0.6289,0.7495,0.2064,0.6289,0.5,0,0.866,0.4892,0.2064,0.8474,0.766,0,0.6428,0.9397,0,0.342,0.9195,0.2064,0.3347,0.9195,0.2064,0.3347,0.766,0,0.6428,0.7495,0.2064,0.6289,0.9397,0,0.342,1,0,0,0.9785,0.2064,0,0.9785,0.2064,0,0.9397,0,0.342,0.9195,0.2064,0.3347,0.9195,0.2064,-0.3347,0.7387,0.6181,-0.2688,0.7861,0.6181,0,0.7861,0.6181,0,0.9195,0.2064,-0.3347,0.9785,0.2064,0,0.7495,0.2064,-0.6289,0.6022,0.6181,-0.5053,0.7387,0.6181,-0.2688,0.7387,0.6181,-0.2688,0.7495,0.2064,-0.6289,0.9195,0.2064,-0.3347,0.4892,0.2064,-0.8474,0.393,0.6181,-0.6808,0.6022,0.6181,-0.5053,0.6022,0.6181,-0.5053,0.4892,0.2064,-0.8474,0.7495,0.2064,-0.6289,0.1699,0.2064,-0.9636,0.1365,0.6181,-0.7741,0.393,0.6181,-0.6808,0.393,0.6181,-0.6808,0.1699,0.2064,-0.9636,0.4892,0.2064,-0.8474,-0.1699,0.2064,-0.9636,-0.1365,0.6181,-0.7741,0.1365,0.6181,-0.7741,0.1365,0.6181,-0.7741,-0.1699,0.2064,-0.9636,0.1699,0.2064,-0.9636,-0.4892,0.2064,-0.8474,-0.393,0.6181,-0.6808,-0.1365,0.6181,-0.7741,-0.1365,0.6181,-0.7741,-0.4892,0.2064,-0.8474,-0.1699,0.2064,-0.9636,-0.7495,0.2064,-0.6289,-0.6022,0.6181,-0.5053,-0.393,0.6181,-0.6808,-0.393,0.6181,-0.6808,-0.7495,0.2064,-0.6289,-0.4892,0.2064,-0.8474,-0.9195,0.2064,-0.3347,-0.7387,0.6181,-0.2688,-0.6022,0.6181,-0.5053,-0.6022,0.6181,-0.5053,-0.9195,0.2064,-0.3347,-0.7495,0.2064,-0.6289,-0.9785,0.2064,0,-0.7861,0.6181,0,-0.7387,0.6181,-0.2688,-0.7387,0.6181,-0.2688,-0.9785,0.2064,0,-0.9195,0.2064,-0.3347,-0.9195,0.2064,0.3347,-0.7387,0.6181,0.2688,-0.7861,0.6181,0,-0.7861,0.6181,0,-0.9195,0.2064,0.3347,-0.9785,0.2064,0,-0.7495,0.2064,0.6289,-0.6022,0.6181,0.5053,-0.7387,0.6181,0.2688,-0.7387,0.6181,0.2688,-0.7495,0.2064,0.6289,-0.9195,0.2064,0.3347,-0.4892,0.2064,0.8474,-0.393,0.6181,0.6808,-0.6022,0.6181,0.5053,-0.6022,0.6181,0.5053,-0.4892,0.2064,0.8474,-0.7495,0.2064,0.6289,-0.1699,0.2064,0.9636,-0.1365,0.6181,0.7741,-0.393,0.6181,0.6808,-0.393,0.6181,0.6808,-0.1699,0.2064,0.9636,-0.4892,0.2064,0.8474,0.1699,0.2064,0.9636,0.1365,0.6181,0.7741,-0.1365,0.6181,0.7741,-0.1365,0.6181,0.7741,0.1699,0.2064,0.9636,-0.1699,0.2064,0.9636,0.4892,0.2064,0.8474,0.393,0.6181,0.6808,0.1365,0.6181,0.7741,0.1365,0.6181,0.7741,0.4892,0.2064,0.8474,0.1699,0.2064,0.9636,0.7495,0.2064,0.6289,0.6022,0.6181,0.5053,0.393,0.6181,0.6808,0.393,0.6181,0.6808,0.7495,0.2064,0.6289,0.4892,0.2064,0.8474,0.9195,0.2064,0.3347,0.7387,0.6181,0.2689,0.6022,0.6181,0.5053,0.6022,0.6181,0.5053,0.9195,0.2064,0.3347,0.7495,0.2064,0.6289,0.9785,0.2064,0,0.7861,0.6181,0,0.7387,0.6181,0.2689,0.7387,0.6181,0.2689,0.9785,0.2064,0,0.9195,0.2064,0.3347,0.8764,0.4816,0,0.8235,0.4816,-0.2997,0.9229,0.188,-0.3359,0.9229,0.188,-0.3359,0.8764,0.4816,0,0.9822,0.188,0,0.8235,0.4816,-0.2997,0.6713,0.4816,-0.5633,0.7524,0.188,-0.6313,0.7524,0.188,-0.6313,0.8235,0.4816,-0.2997,0.9229,0.188,-0.3359,0.6713,0.4816,-0.5633,0.4382,0.4816,-0.759,0.4911,0.188,-0.8506,0.4911,0.188,-0.8506,0.6713,0.4816,-0.5633,0.7524,0.188,-0.6313,0.4382,0.4816,-0.759,0.1522,0.4816,-0.8631,0.1706,0.188,-0.9672,0.1706,0.188,-0.9672,0.4382,0.4816,-0.759,0.4911,0.188,-0.8506,0.1522,0.4816,-0.8631,-0.1522,0.4816,-0.8631,-0.1706,0.188,-0.9672,-0.1706,0.188,-0.9672,0.1522,0.4816,-0.8631,0.1706,0.188,-0.9672,-0.1522,0.4816,-0.8631,-0.4382,0.4816,-0.759,-0.4911,0.188,-0.8506,-0.4911,0.188,-0.8506,-0.1522,0.4816,-0.8631,-0.1706,0.188,-0.9672,-0.4382,0.4816,-0.759,-0.6713,0.4816,-0.5633,-0.7524,0.188,-0.6313,-0.7524,0.188,-0.6313,-0.4382,0.4816,-0.759,-0.4911,0.188,-0.8506,-0.6713,0.4816,-0.5633,-0.8235,0.4816,-0.2997,-0.9229,0.188,-0.3359,-0.9229,0.188,-0.3359,-0.6713,0.4816,-0.5633,-0.7524,0.188,-0.6313,-0.8235,0.4816,-0.2997,-0.8764,0.4816,0,-0.9822,0.188,0,-0.9822,0.188,0,-0.8235,0.4816,-0.2997,-0.9229,0.188,-0.3359,-0.8764,0.4816,0,-0.8235,0.4816,0.2997,-0.9229,0.188,0.3359,-0.9229,0.188,0.3359,-0.8764,0.4816,0,-0.9822,0.188,0,-0.8235,0.4816,0.2997,-0.6713,0.4816,0.5633,-0.7524,0.188,0.6313,-0.7524,0.188,0.6313,-0.8235,0.4816,0.2997,-0.9229,0.188,0.3359,-0.6713,0.4816,0.5633,-0.4382,0.4816,0.759,-0.4911,0.188,0.8506,-0.4911,0.188,0.8506,-0.6713,0.4816,0.5633,-0.7524,0.188,0.6313,-0.4382,0.4816,0.759,-0.1522,0.4816,0.8631,-0.1706,0.188,0.9672,-0.1706,0.188,0.9672,-0.4382,0.4816,0.759,-0.4911,0.188,0.8506,-0.1522,0.4816,0.8631,0.1522,0.4816,0.8631,0.1706,0.188,0.9672,0.1706,0.188,0.9672,-0.1522,0.4816,0.8631,-0.1706,0.188,0.9672,0.1522,0.4816,0.8631,0.4382,0.4816,0.759,0.4911,0.188,0.8506,0.4911,0.188,0.8506,0.1522,0.4816,0.8631,0.1706,0.188,0.9672,0.4382,0.4816,0.759,0.6713,0.4816,0.5633,0.7524,0.188,0.6313,0.7524,0.188,0.6313,0.4382,0.4816,0.759,0.4911,0.188,0.8506,0.6713,0.4816,0.5633,0.8235,0.4816,0.2997,0.9229,0.188,0.3359,0.9229,0.188,0.3359,0.6713,0.4816,0.5633,0.7524,0.188,0.6313,0.8235,0.4816,0.2997,0.8764,0.4816,0,0.9822,0.188,0,0.9822,0.188,0,0.8235,0.4816,0.2997,0.9229,0.188,0.3359,0.9229,0.188,-0.3359,0.7582,0.5907,-0.276,0.8069,0.5907,0,0.8069,0.5907,0,0.9229,0.188,-0.3359,0.9822,0.188,0,0.7524,0.188,-0.6313,0.6181,0.5907,-0.5186,0.7582,0.5907,-0.276,0.7582,0.5907,-0.276,0.7524,0.188,-0.6313,0.9229,0.188,-0.3359,0.4911,0.188,-0.8506,0.4034,0.5907,-0.6988,0.6181,0.5907,-0.5186,0.6181,0.5907,-0.5186,0.4911,0.188,-0.8506,0.7524,0.188,-0.6313,0.1706,0.188,-0.9672,0.1401,0.5907,-0.7946,0.4034,0.5907,-0.6988,0.4034,0.5907,-0.6988,0.1706,0.188,-0.9672,0.4911,0.188,-0.8506,-0.1706,0.188,-0.9672,-0.1401,0.5907,-0.7946,0.1401,0.5907,-0.7946,0.1401,0.5907,-0.7946,-0.1706,0.188,-0.9672,0.1706,0.188,-0.9672,-0.4911,0.188,-0.8506,-0.4034,0.5907,-0.6988,-0.1401,0.5907,-0.7946,-0.1401,0.5907,-0.7946,-0.4911,0.188,-0.8506,-0.1706,0.188,-0.9672,-0.7524,0.188,-0.6313,-0.6181,0.5907,-0.5186,-0.4034,0.5907,-0.6988,-0.4034,0.5907,-0.6988,-0.7524,0.188,-0.6313,-0.4911,0.188,-0.8506,-0.9229,0.188,-0.3359,-0.7582,0.5907,-0.276,-0.6181,0.5907,-0.5186,-0.6181,0.5907,-0.5186,-0.9229,0.188,-0.3359,-0.7524,0.188,-0.6313,-0.9822,0.188,0,-0.8069,0.5907,0,-0.7582,0.5907,-0.276,-0.7582,0.5907,-0.276,-0.9822,0.188,0,-0.9229,0.188,-0.3359,-0.9229,0.188,0.3359,-0.7582,0.5907,0.276,-0.8069,0.5907,0,-0.8069,0.5907,0,-0.9229,0.188,0.3359,-0.9822,0.188,0,-0.7524,0.188,0.6313,-0.6181,0.5907,0.5186,-0.7582,0.5907,0.276,-0.7582,0.5907,0.276,-0.7524,0.188,0.6313,-0.9229,0.188,0.3359,-0.4911,0.188,0.8506,-0.4034,0.5907,0.6988,-0.6181,0.5907,0.5186,-0.6181,0.5907,0.5186,-0.4911,0.188,0.8506,-0.7524,0.188,0.6313,-0.1706,0.188,0.9672,-0.1401,0.5907,0.7946,-0.4034,0.5907,0.6988,-0.4034,0.5907,0.6988,-0.1706,0.188,0.9672,-0.4911,0.188,0.8506,0.1706,0.188,0.9672,0.1401,0.5907,0.7946,-0.1401,0.5907,0.7946,-0.1401,0.5907,0.7946,0.1706,0.188,0.9672,-0.1706,0.188,0.9672,0.4911,0.188,0.8506,0.4034,0.5907,0.6988,0.1401,0.5907,0.7946,0.1401,0.5907,0.7946,0.4911,0.188,0.8506,0.1706,0.188,0.9672,0.7524,0.188,0.6313,0.6181,0.5907,0.5186,0.4034,0.5907,0.6988,0.4034,0.5907,0.6988,0.7524,0.188,0.6313,0.4911,0.188,0.8506,0.9229,0.188,0.3359,0.7582,0.5907,0.276,0.6181,0.5907,0.5186,0.6181,0.5907,0.5186,0.9229,0.188,0.3359,0.7524,0.188,0.6313,0.9822,0.188,0,0.8069,0.5907,0,0.7582,0.5907,0.276,0.7582,0.5907,0.276,0.9822,0.188,0,0.9229,0.188,0.3359,0.8834,0.4687,0,0.8301,0.4687,-0.3021,0.9397,0,-0.342,0.9397,0,-0.342,0.8834,0.4687,0,1,0,0,0.8301,0.4687,-0.3021,0.6767,0.4687,-0.5678,0.766,0,-0.6428,0.766,0,-0.6428,0.8301,0.4687,-0.3021,0.9397,0,-0.342,0.6767,0.4687,-0.5678,0.4417,0.4687,-0.765,0.5,0,-0.866,0.5,0,-0.866,0.6767,0.4687,-0.5678,0.766,0,-0.6428,0.4417,0.4687,-0.765,0.1534,0.4687,-0.87,0.1736,0,-0.9848,0.1736,0,-0.9848,0.4417,0.4687,-0.765,0.5,0,-0.866,0.1534,0.4687,-0.87,-0.1534,0.4687,-0.87,-0.1736,0,-0.9848,-0.1736,0,-0.9848,0.1534,0.4687,-0.87,0.1736,0,-0.9848,-0.1534,0.4687,-0.87,-0.4417,0.4687,-0.765,-0.5,0,-0.866,-0.5,0,-0.866,-0.1534,0.4687,-0.87,-0.1736,0,-0.9848,-0.4417,0.4687,-0.765,-0.6767,0.4687,-0.5678,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.4417,0.4687,-0.765,-0.5,0,-0.866,-0.6767,0.4687,-0.5678,-0.8301,0.4687,-0.3021,-0.9397,0,-0.342,-0.9397,0,-0.342,-0.6767,0.4687,-0.5678,-0.766,0,-0.6428,-0.8301,0.4687,-0.3021,-0.8834,0.4687,0,-1,0,0,-1,0,0,-0.8301,0.4687,-0.3021,-0.9397,0,-0.342,-0.8834,0.4687,0,-0.8301,0.4687,0.3021,-0.9397,0,0.342,-0.9397,0,0.342,-0.8834,0.4687,0,-1,0,0,-0.8301,0.4687,0.3021,-0.6767,0.4687,0.5678,-0.766,0,0.6428,-0.766,0,0.6428,-0.8301,0.4687,0.3021,-0.9397,0,0.342,-0.6767,0.4687,0.5678,-0.4417,0.4687,0.765,-0.5,0,0.866,-0.5,0,0.866,-0.6767,0.4687,0.5678,-0.766,0,0.6428,-0.4417,0.4687,0.765,-0.1534,0.4687,0.87,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.4417,0.4687,0.765,-0.5,0,0.866,-0.1534,0.4687,0.87,0.1534,0.4687,0.87,0.1736,0,0.9848,0.1736,0,0.9848,-0.1534,0.4687,0.87,-0.1736,0,0.9848,0.1534,0.4687,0.87,0.4417,0.4687,0.765,0.5,0,0.866,0.5,0,0.866,0.1534,0.4687,0.87,0.1736,0,0.9848,0.4417,0.4687,0.765,0.6767,0.4687,0.5678,0.766,0,0.6428,0.766,0,0.6428,0.4417,0.4687,0.765,0.5,0,0.866,0.6767,0.4687,0.5678,0.8301,0.4687,0.3021,0.9397,0,0.342,0.9397,0,0.342,0.6767,0.4687,0.5678,0.766,0,0.6428,0.8301,0.4687,0.3021,0.8834,0.4687,0,1,0,0,1,0,0,0.8301,0.4687,0.3021,0.9397,0,0.342,1,0,0,0.9397,0,-0.342,0.918,0,-0.3966,0.918,0,-0.3966,1,0,0,1,0,0,0.9397,0,-0.342,0.766,0,-0.6428,0.766,0,-0.6428,0.766,0,-0.6428,0.9397,0,-0.342,0.918,0,-0.3966,0.766,0,-0.6428,0.5,0,-0.866,0.5,0,-0.866,0.5,0,-0.866,0.766,0,-0.6428,0.766,0,-0.6428,0.5,0,-0.866,0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,0.5,0,-0.866,0.5,0,-0.866,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.5,0,-0.866,-0.5,0,-0.866,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.5,0,-0.866,-0.5,0,-0.866,-0.766,0,-0.6428,-0.9397,0,-0.342,-0.918,0,-0.3966,-0.918,0,-0.3966,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.9397,0,-0.342,-1,0,0,-1,0,0,-1,0,0,-0.9397,0,-0.342,-0.918,0,-0.3966,-1,0,0,-0.9397,0,0.342,-0.918,0,0.3966,-0.918,0,0.3966,-1,0,0,-1,0,0,-0.9397,0,0.342,-0.766,0,0.6428,-0.766,0,0.6428,-0.766,0,0.6428,-0.9397,0,0.342,-0.918,0,0.3966,-0.766,0,0.6428,-0.5,0,0.866,-0.5,0,0.866,-0.5,0,0.866,-0.766,0,0.6428,-0.766,0,0.6428,-0.5,0,0.866,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.5,0,0.866,-0.5,0,0.866,-0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.5,0,0.866,0.5,0,0.866,0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.766,0,0.6428,0.766,0,0.6428,0.766,0,0.6428,0.5,0,0.866,0.5,0,0.866,0.766,0,0.6428,0.9397,0,0.342,0.918,0,0.3966,0.918,0,0.3966,0.766,0,0.6428,0.766,0,0.6428,0.9397,0,0.342,1,0,0,1,0,0,1,0,0,0.9397,0,0.342,0.918,0,0.3966,0.918,0,-0.3966,0.766,0,-0.6428,0.766,0,-0.6428,0.766,0,-0.6428,0.918,0,-0.3966,0.918,0,-0.3966,0.766,0,-0.6428,0.5,0,-0.866,0.5,0,-0.866,0.5,0,-0.866,0.766,0,-0.6428,0.766,0,-0.6428,0.5,0,-0.866,0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,0.5,0,-0.866,0.5,0,-0.866,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.5,0,-0.866,-0.5,0,-0.866,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.5,0,-0.866,-0.5,0,-0.866,-0.766,0,-0.6428,-0.918,0,-0.3966,-0.918,0,-0.3966,-0.918,0,-0.3966,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.963,-0.2694,0.0018,-0.963,-0.2694,0,-0.963,-0.2694,0,-0.963,-0.2694,0,-0.963,-0.2694,0.0018,-0.963,-0.2694,0.0018,-0.963,-0.2694,0,-0.963,-0.2694,-0.0017,-0.963,-0.2694,-0.0018,-0.963,-0.2694,-0.0018,-0.963,-0.2694,0,-0.963,-0.2694,0,-0.918,0,0.3966,-0.766,0,0.6428,-0.766,0,0.6428,-0.766,0,0.6428,-0.918,0,0.3966,-0.918,0,0.3966,-0.766,0,0.6428,-0.5,0,0.866,-0.5,0,0.866,-0.5,0,0.866,-0.766,0,0.6428,-0.766,0,0.6428,-0.5,0,0.866,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.5,0,0.866,-0.5,0,0.866,-0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.5,0,0.866,0.5,0,0.866,0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.766,0,0.6428,0.766,0,0.6428,0.766,0,0.6428,0.5,0,0.866,0.5,0,0.866,0.766,0,0.6428,0.918,0,0.3966,0.918,0,0.3966,0.918,0,0.3966,0.766,0,0.6428,0.766,0,0.6428,1,0,0,0.918,0,-0.3966,0.9397,0,-0.342,0.9397,0,-0.342,1,0,0,1,0,0,0.918,0,-0.3966,0.766,0,-0.6428,0.766,0,-0.6428,0.766,0,-0.6428,0.918,0,-0.3966,0.9397,0,-0.342,0.766,0,-0.6428,0.5,0,-0.866,0.5,0,-0.866,0.5,0,-0.866,0.766,0,-0.6428,0.766,0,-0.6428,0.5,0,-0.866,0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,0.5,0,-0.866,0.5,0,-0.866,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.1736,0,-0.9848,0.1736,0,-0.9848,0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.5,0,-0.866,-0.5,0,-0.866,-0.1736,0,-0.9848,-0.1736,0,-0.9848,-0.5,0,-0.866,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.5,0,-0.866,-0.5,0,-0.866,-0.766,0,-0.6428,-0.918,0,-0.3966,-0.9397,0,-0.342,-0.9397,0,-0.342,-0.766,0,-0.6428,-0.766,0,-0.6428,-0.918,0,-0.3966,-1,0,0,-1,0,0,-1,0,0,-0.918,0,-0.3966,-0.9397,0,-0.342,-1,0,0,-0.918,0,0.3966,-0.9397,0,0.342,-0.9397,0,0.342,-1,0,0,-1,0,0,-0.918,0,0.3966,-0.766,0,0.6428,-0.766,0,0.6428,-0.766,0,0.6428,-0.918,0,0.3966,-0.9397,0,0.342,-0.766,0,0.6428,-0.5,0,0.866,-0.5,0,0.866,-0.5,0,0.866,-0.766,0,0.6428,-0.766,0,0.6428,-0.5,0,0.866,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,-0.5,0,0.866,-0.5,0,0.866,-0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,0.1736,0,0.9848,-0.1736,0,0.9848,-0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.5,0,0.866,0.5,0,0.866,0.1736,0,0.9848,0.1736,0,0.9848,0.5,0,0.866,0.766,0,0.6428,0.766,0,0.6428,0.766,0,0.6428,0.5,0,0.866,0.5,0,0.866,0.766,0,0.6428,0.918,0,0.3966,0.9397,0,0.342,0.9397,0,0.342,0.766,0,0.6428,0.766,0,0.6428,0.918,0,0.3966,1,0,0,1,0,0,1,0,0,0.918,0,0.3966,0.9397,0,0.342,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.8301,0.4687,-0.3021,0.8834,0.4687,0,0.8069,0.5907,0,0.8069,0.5907,0,0.8301,0.4687,-0.3021,0.7582,0.5907,-0.276,0.6767,0.4687,-0.5678,0.8301,0.4687,-0.3021,0.7582,0.5907,-0.276,0.7582,0.5907,-0.276,0.6767,0.4687,-0.5678,0.6181,0.5907,-0.5186,0.4417,0.4687,-0.765,0.6767,0.4687,-0.5678,0.6181,0.5907,-0.5186,0.6181,0.5907,-0.5186,0.4417,0.4687,-0.765,0.4034,0.5907,-0.6988,0.1534,0.4687,-0.87,0.4417,0.4687,-0.765,0.4034,0.5907,-0.6988,0.4034,0.5907,-0.6988,0.1534,0.4687,-0.87,0.1401,0.5907,-0.7946,-0.1534,0.4687,-0.87,0.1534,0.4687,-0.87,0.1401,0.5907,-0.7946,0.1401,0.5907,-0.7946,-0.1534,0.4687,-0.87,-0.1401,0.5907,-0.7946,-0.4417,0.4687,-0.765,-0.1534,0.4687,-0.87,-0.1401,0.5907,-0.7946,-0.1401,0.5907,-0.7946,-0.4417,0.4687,-0.765,-0.4034,0.5907,-0.6988,-0.6767,0.4687,-0.5678,-0.4417,0.4687,-0.765,-0.4034,0.5907,-0.6988,-0.4034,0.5907,-0.6988,-0.6767,0.4687,-0.5678,-0.6181,0.5907,-0.5186,-0.8301,0.4687,-0.3021,-0.6767,0.4687,-0.5678,-0.6181,0.5907,-0.5186,-0.6181,0.5907,-0.5186,-0.8301,0.4687,-0.3021,-0.7582,0.5907,-0.276,-0.8834,0.4687,0,-0.8301,0.4687,-0.3021,-0.7582,0.5907,-0.276,-0.7582,0.5907,-0.276,-0.8834,0.4687,0,-0.8069,0.5907,0,-0.8301,0.4687,0.3021,-0.8834,0.4687,0,-0.8069,0.5907,0,-0.8069,0.5907,0,-0.8301,0.4687,0.3021,-0.7582,0.5907,0.276,-0.6767,0.4687,0.5678,-0.8301,0.4687,0.3021,-0.7582,0.5907,0.276,-0.7582,0.5907,0.276,-0.6767,0.4687,0.5678,-0.6181,0.5907,0.5186,-0.4417,0.4687,0.765,-0.6767,0.4687,0.5678,-0.6181,0.5907,0.5186,-0.6181,0.5907,0.5186,-0.4417,0.4687,0.765,-0.4034,0.5907,0.6988,-0.1534,0.4687,0.87,-0.4417,0.4687,0.765,-0.4034,0.5907,0.6988,-0.4034,0.5907,0.6988,-0.1534,0.4687,0.87,-0.1401,0.5907,0.7946,0.1534,0.4687,0.87,-0.1534,0.4687,0.87,-0.1401,0.5907,0.7946,-0.1401,0.5907,0.7946,0.1534,0.4687,0.87,0.1401,0.5907,0.7946,0.4417,0.4687,0.765,0.1534,0.4687,0.87,0.1401,0.5907,0.7946,0.1401,0.5907,0.7946,0.4417,0.4687,0.765,0.4034,0.5907,0.6988,0.6767,0.4687,0.5678,0.4417,0.4687,0.765,0.4034,0.5907,0.6988,0.4034,0.5907,0.6988,0.6767,0.4687,0.5678,0.6181,0.5907,0.5186,0.8301,0.4687,0.3021,0.6767,0.4687,0.5678,0.6181,0.5907,0.5186,0.6181,0.5907,0.5186,0.8301,0.4687,0.3021,0.7582,0.5907,0.276,0.8834,0.4687,0,0.8301,0.4687,0.3021,0.7582,0.5907,0.276,0.7582,0.5907,0.276,0.8834,0.4687,0,0.8069,0.5907,0,0.8235,0.4816,-0.2997,0.8764,0.4816,0,0.7861,0.6181,0,0.7861,0.6181,0,0.8235,0.4816,-0.2997,0.7387,0.6181,-0.2688,0.6713,0.4816,-0.5633,0.8235,0.4816,-0.2997,0.7387,0.6181,-0.2688,0.7387,0.6181,-0.2688,0.6713,0.4816,-0.5633,0.6022,0.6181,-0.5053,0.4382,0.4816,-0.759,0.6713,0.4816,-0.5633,0.6022,0.6181,-0.5053,0.6022,0.6181,-0.5053,0.4382,0.4816,-0.759,0.393,0.6181,-0.6808,0.1522,0.4816,-0.8631,0.4382,0.4816,-0.759,0.393,0.6181,-0.6808,0.393,0.6181,-0.6808,0.1522,0.4816,-0.8631,0.1365,0.6181,-0.7741,-0.1522,0.4816,-0.8631,0.1522,0.4816,-0.8631,0.1365,0.6181,-0.7741,0.1365,0.6181,-0.7741,-0.1522,0.4816,-0.8631,-0.1365,0.6181,-0.7741,-0.4382,0.4816,-0.759,-0.1522,0.4816,-0.8631,-0.1365,0.6181,-0.7741,-0.1365,0.6181,-0.7741,-0.4382,0.4816,-0.759,-0.393,0.6181,-0.6808,-0.6713,0.4816,-0.5633,-0.4382,0.4816,-0.759,-0.393,0.6181,-0.6808,-0.393,0.6181,-0.6808,-0.6713,0.4816,-0.5633,-0.6022,0.6181,-0.5053,-0.8235,0.4816,-0.2997,-0.6713,0.4816,-0.5633,-0.6022,0.6181,-0.5053,-0.6022,0.6181,-0.5053,-0.8235,0.4816,-0.2997,-0.7387,0.6181,-0.2688,-0.8764,0.4816,0,-0.8235,0.4816,-0.2997,-0.7387,0.6181,-0.2688,-0.7387,0.6181,-0.2688,-0.8764,0.4816,0,-0.7861,0.6181,0,-0.8235,0.4816,0.2997,-0.8764,0.4816,0,-0.7861,0.6181,0,-0.7861,0.6181,0,-0.8235,0.4816,0.2997,-0.7387,0.6181,0.2688,-0.6713,0.4816,0.5633,-0.8235,0.4816,0.2997,-0.7387,0.6181,0.2688,-0.7387,0.6181,0.2688,-0.6713,0.4816,0.5633,-0.6022,0.6181,0.5053,-0.4382,0.4816,0.759,-0.6713,0.4816,0.5633,-0.6022,0.6181,0.5053,-0.6022,0.6181,0.5053,-0.4382,0.4816,0.759,-0.393,0.6181,0.6808,-0.1522,0.4816,0.8631,-0.4382,0.4816,0.759,-0.393,0.6181,0.6808,-0.393,0.6181,0.6808,-0.1522,0.4816,0.8631,-0.1365,0.6181,0.7741,0.1522,0.4816,0.8631,-0.1522,0.4816,0.8631,-0.1365,0.6181,0.7741,-0.1365,0.6181,0.7741,0.1522,0.4816,0.8631,0.1365,0.6181,0.7741,0.4382,0.4816,0.759,0.1522,0.4816,0.8631,0.1365,0.6181,0.7741,0.1365,0.6181,0.7741,0.4382,0.4816,0.759,0.393,0.6181,0.6808,0.6713,0.4816,0.5633,0.4382,0.4816,0.759,0.393,0.6181,0.6808,0.393,0.6181,0.6808,0.6713,0.4816,0.5633,0.6022,0.6181,0.5053,0.8235,0.4816,0.2997,0.6713,0.4816,0.5633,0.6022,0.6181,0.5053,0.6022,0.6181,0.5053,0.8235,0.4816,0.2997,0.7387,0.6181,0.2689,0.8764,0.4816,0,0.8235,0.4816,0.2997,0.7387,0.6181,0.2689,0.7387,0.6181,0.2689,0.8764,0.4816,0,0.7861,0.6181,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.9526,-0.3041,-0.002,0.9526,-0.3041,-0.002,0.9526,-0.3041,-0.002,0.9526,-0.3041,-0.002,0.9526,-0.3041,-0.002,0.9526,-0.3041,-0.002,0.9526,-0.3041,0.0036,0.9526,-0.3041,0.0036,0.9526,-0.3041,0.0036,0.9526,-0.3041,0.0036,0.9526,-0.3041,0.0036,0.9526,-0.3041,0.0036,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,0.0106,-0.9525,-0.3043,-0.0106,-0.9525,-0.3043,-0.0106,-0.9525,-0.3043,-0.0106,-0.9525,-0.3043,-0.0106,-0.9525,-0.3043,-0.0106,-0.9525,-0.3043,-0.0106,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0.0005,-0.0001,-1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,-0.0005,0.0001,1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.9984,0.0496,0.027,0.9984,0.0496,0.027,0.9984,0.0496,0.027,0.9984,0.0496,0.027,0.9984,0.0496,0.027,0.9984,0.0496,0.027,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.0001,0,-1,-0.9984,0.0496,0.0262,-0.9984,0.0496,0.0262,-0.9984,0.0496,0.0262,-0.9984,0.0496,0.0262,-0.9984,0.0496,0.0262,-0.9984,0.0496,0.0262,-0.9984,0.0496,-0.0268,-0.9984,0.0496,-0.0268,-0.9984,0.0496,-0.0268,-0.9984,0.0496,-0.0268,-0.9984,0.0496,-0.0268,-0.9984,0.0496,-0.0268,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.0001,0,1,0.9984,0.0496,-0.0261,0.9984,0.0496,-0.0261,0.9984,0.0496,-0.0261,0.9984,0.0496,-0.0261,0.9984,0.0496,-0.0261,0.9984,0.0496,-0.0261,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0008,0,1,-0.0008,0,1,-0.0008,0,1,-0.0008,0,1,-0.0008,0,1,-0.0008,0,1,0.0008,0,-1,0.0008,0,-1,0.0008,0,-1,0.0008,0,-1,0.0008,0,-1,0.0008,0,-1,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0.2924,0.9563,0,0.2336,0.9723,0,0.2336,0.9723,0,0.2336,0.9723,0,0.2924,0.9563,0,0.2924,0.9563,0,0.2336,0.9723,0,0.1397,0.9902,0,0.1397,0.9902,0,0.1397,0.9902,0,0.2336,0.9723,0,0.2336,0.9723,0,0.1397,0.9902,0,0.0896,0.996,0,0.0896,0.996,0,0.0896,0.996,0,0.1397,0.9902,0,0.1397,0.9902,0,0.0896,0.996,0,0.0661,0.9978,0,0.0661,0.9978,0,0.0661,0.9978,0,0.0896,0.996,0,0.0896,0.996,0,0.0661,0.9978,0,0.0292,0.9996,0,0.0292,0.9996,0,0.0292,0.9996,0,0.0661,0.9978,0,0.0661,0.9978,0,0.0292,0.9996,0,0,1,0,0,1,0,0,1,0,0.0292,0.9996,0,0.0292,0.9996,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-0.0292,0.9996,0,-0.0292,0.9996,0,-0.0292,0.9996,0,0,1,0,0,1,0,-0.0292,0.9996,0,-0.0656,0.9978,0,-0.0656,0.9978,0,-0.0656,0.9978,0,-0.0292,0.9996,0,-0.0292,0.9996,0,-0.0656,0.9978,0,-0.0899,0.996,0,-0.0899,0.996,0,-0.0899,0.996,0,-0.0656,0.9978,0,-0.0656,0.9978,0,-0.0899,0.996,0,-0.1411,0.99,0.0008,-0.1415,0.9899,0,-0.1415,0.9899,0,-0.0899,0.996,0,-0.0899,0.996,0,-0.1411,0.99,0.0008,-0.1929,0.9812,0.0041,-0.1925,0.9813,0.0049,-0.1925,0.9813,0.0049,-0.1411,0.99,0.0008,-0.1415,0.9899,0,-0.1929,0.9812,0.0041,-0.2162,0.9763,-0.0057,-0.2147,0.9767,-0.0025,-0.2147,0.9767,-0.0025,-0.1929,0.9812,0.0041,-0.1925,0.9813,0.0049,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0.8649,-0.502,-0.0016,0.8652,-0.5014,0,0.8643,-0.503,-0.0044,0.8643,-0.503,-0.0044,0.8649,-0.502,-0.0016,0.8639,-0.5036,-0.006,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,-0.96,-0.2798,0,-0.96,-0.2798,0,-0.96,-0.2798,0,-0.96,-0.2798,0,-0.96,-0.2798,0,-0.96,-0.2798,0] )
   , make_vbo(COLOR, [0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.0824,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0.8118,0.1373,0.1373,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0] )
  ],
  mode: gl.TRIANGLES ,
  vertex_count: 3168
};
geometries["quad"] = {
  buffers: [
     make_vbo(POS, [-1,-1,-1,1,1,-1,1,1] )
  ],
  mode: gl.TRIANGLE_STRIP ,
  vertex_count: 4
};
var generator =  function () {
    var positions = make_sphere(10, 5);
    var normals = map_triangles(positions, flat_normals);
    var triangle_ids = map_triangles(positions, triangle_index);
    return {
      buffers: [
        make_vbo(POS, positions),
        make_vbo(NORMAL, normals),
        make_vbo(TRIANGLE_ID, triangle_ids)
      ],
      mode: gl.TRIANGLES,
      vertex_count: positions.length / 3
    };
  }
geometries["sphere"] = generator();
geometries["stone"] = {
  buffers: [
     make_vbo(POS, [-47.8253,0.2152,26.4127,-47.8253,0.2152,-26.9503,48.1043,0.2152,-26.9503,48.1043,0.2152,-26.9503,-47.8253,0.2152,26.4127,48.1043,0.2152,26.4126,6.1122,185.9584,-1.6797,8.9466,185.9584,-1.6797,8.9466,185.9584,-4.3796,8.9466,185.9584,-4.3796,6.1122,185.9584,-1.6797,6.1122,185.9584,-4.3796,-17.9224,33.7343,7.5075,18.2048,33.7343,7.5075,18.2048,155.0181,7.5075,18.2048,155.0181,7.5075,-17.9224,33.7343,7.5075,-17.9224,155.0181,7.5075,36.841,24.2855,15.3576,36.841,24.2855,-20.1958,27.6519,164.4669,-20.1958,27.6519,164.4669,-20.1958,36.841,24.2855,15.3576,27.6519,164.4669,15.3576,36.841,24.2855,-20.1958,-36.5621,24.2855,-20.1958,-27.3729,164.4669,-20.1958,-27.3729,164.4669,-20.1958,36.841,24.2855,-20.1958,27.6519,164.4669,-20.1958,-36.5621,24.2855,-20.1958,-36.5621,24.2855,15.3576,-27.3729,164.4669,15.3576,-27.3729,164.4669,15.3576,-36.5621,24.2855,-20.1958,-27.3729,164.4669,-20.1958,-36.5621,24.2855,15.3576,36.841,24.2855,15.3576,18.2031,33.7343,15.3576,18.2031,33.7343,15.3576,-36.5621,24.2855,15.3576,-17.9241,33.7343,15.3576,36.841,24.2855,15.3576,27.6519,164.4669,15.3576,18.2031,155.0181,15.3576,18.2031,155.0181,15.3576,36.841,24.2855,15.3576,18.2031,33.7343,15.3576,27.6519,164.4669,15.3576,-27.3729,164.4669,15.3576,-17.9241,155.0181,15.3576,-17.9241,155.0181,15.3576,27.6519,164.4669,15.3576,18.2031,155.0181,15.3576,-27.3729,164.4669,15.3576,-36.5621,24.2855,15.3576,-17.9241,33.7343,15.3576,-17.9241,33.7343,15.3576,-27.3729,164.4669,15.3576,-17.9241,155.0181,15.3576,-27.3729,164.4669,15.3576,27.6519,164.4669,15.3576,8.9466,185.9584,-1.6797,8.9466,185.9584,-1.6797,-27.3729,164.4669,15.3576,6.1122,185.9584,-1.6797,27.6519,164.4669,15.3576,27.6519,164.4669,-20.1958,8.9466,185.9584,-4.3796,8.9466,185.9584,-4.3796,27.6519,164.4669,15.3576,8.9466,185.9584,-1.6797,27.6519,164.4669,-20.1958,-27.3729,164.4669,-20.1958,6.1122,185.9584,-4.3796,6.1122,185.9584,-4.3796,27.6519,164.4669,-20.1958,8.9466,185.9584,-4.3796,-27.3729,164.4669,-20.1958,-27.3729,164.4669,15.3576,6.1122,185.9584,-1.6797,6.1122,185.9584,-1.6797,-27.3729,164.4669,-20.1958,6.1122,185.9584,-4.3796,-17.9241,33.7343,15.3576,18.2031,33.7343,15.3576,18.2048,33.7343,7.5075,18.2048,33.7343,7.5075,-17.9241,33.7343,15.3576,-17.9224,33.7343,7.5075,18.2031,33.7343,15.3576,18.2031,155.0181,15.3576,18.2048,155.0181,7.5075,18.2048,155.0181,7.5075,18.2031,33.7343,15.3576,18.2048,33.7343,7.5075,18.2031,155.0181,15.3576,-17.9241,155.0181,15.3576,-17.9224,155.0181,7.5075,-17.9224,155.0181,7.5075,18.2031,155.0181,15.3576,18.2048,155.0181,7.5075,-17.9241,155.0181,15.3576,-17.9241,33.7343,15.3576,-17.9224,33.7343,7.5075,-17.9224,33.7343,7.5075,-17.9241,155.0181,15.3576,-17.9224,155.0181,7.5075,-36.5621,24.2855,15.3576,-36.5621,24.2855,-20.1958,-47.8253,19.4152,-26.8189,-47.8253,19.4152,-26.8189,-36.5621,24.2855,15.3576,-47.8253,19.4152,26.5441,-36.5621,24.2855,-20.1958,36.841,24.2855,-20.1958,48.1043,19.4152,-26.8189,48.1043,19.4152,-26.8189,-36.5621,24.2855,-20.1958,-47.8253,19.4152,-26.8189,36.841,24.2855,-20.1958,36.841,24.2855,15.3576,48.1043,19.4152,26.5441,48.1043,19.4152,26.5441,36.841,24.2855,-20.1958,48.1043,19.4152,-26.8189,36.841,24.2855,15.3576,-36.5621,24.2855,15.3576,-47.8253,19.4152,26.5441,-47.8253,19.4152,26.5441,36.841,24.2855,15.3576,48.1043,19.4152,26.5441,-47.8253,19.4152,26.5441,-47.8253,19.4152,-26.8189,-47.8253,0.2152,-26.9503,-47.8253,0.2152,-26.9503,-47.8253,19.4152,26.5441,-47.8253,0.2152,26.4127,-47.8253,19.4152,-26.8189,48.1043,19.4152,-26.8189,48.1043,0.2152,-26.9503,48.1043,0.2152,-26.9503,-47.8253,19.4152,-26.8189,-47.8253,0.2152,-26.9503,48.1043,19.4152,-26.8189,48.1043,19.4152,26.5441,48.1043,0.2152,26.4126,48.1043,0.2152,26.4126,48.1043,19.4152,-26.8189,48.1043,0.2152,-26.9503,48.1043,19.4152,26.5441,-47.8253,19.4152,26.5441,-47.8253,0.2152,26.4127,-47.8253,0.2152,26.4127,48.1043,19.4152,26.5441,48.1043,0.2152,26.4126] )
   , make_vbo(NORMAL, [0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0.9979,0.0654,0,0.9979,0.0654,0,0.9979,0.0654,0,0.9979,0.0654,0,0.9979,0.0654,0,0.9979,0.0654,0,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,-0.9979,0.0654,0,-0.9979,0.0654,0,-0.9979,0.0654,0,-0.9979,0.0654,0,-0.9979,0.0654,0,-0.9979,0.0654,0,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0.6212,0.7836,0,0.6212,0.7836,0,0.6212,0.7836,0,0.6212,0.7836,0,0.6212,0.7836,0,0.6212,0.7836,0.7543,0.6565,0,0.7543,0.6565,0,0.7543,0.6565,0,0.7543,0.6565,0,0.7543,0.6565,0,0.7543,0.6565,0,0,0.5927,-0.8054,0,0.5927,-0.8054,0,0.5927,-0.8054,0,0.5927,-0.8054,0,0.5927,-0.8054,0,0.5927,-0.8054,-0.5401,0.8416,0,-0.5401,0.8416,0,-0.5401,0.8416,0,-0.5401,0.8416,0,-0.5401,0.8416,0,-0.5401,0.8416,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,-1,0,-0.0002,-1,0,-0.0002,-1,0,-0.0002,-1,0,-0.0002,-1,0,-0.0002,-1,0,-0.0002,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,1,0,0.0002,1,0,0.0002,1,0,0.0002,1,0,0.0002,1,0,0.0002,1,0,0.0002,-0.3969,0.9179,0,-0.3969,0.9179,0,-0.3969,0.9179,0,-0.3969,0.9179,0,-0.3969,0.9179,0,-0.3969,0.9179,0,0,0.8056,-0.5924,0,0.8056,-0.5924,0,0.8056,-0.5924,0,0.8056,-0.5924,0,0.8056,-0.5924,0,0.8056,-0.5924,0.3969,0.9179,0,0.3969,0.9179,0,0.3969,0.9179,0,0.3969,0.9179,0,0.3969,0.9179,0,0.3969,0.9179,0,0,0.9169,0.3992,0,0.9169,0.3992,0,0.9169,0.3992,0,0.9169,0.3992,0,0.9169,0.3992,0,0.9169,0.3992,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,0,0.0068,-1,0,0.0068,-1,0,0.0068,-1,0,0.0068,-1,0,0.0068,-1,0,0.0068,-1,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,0,-0.0068,1,0,-0.0068,1,0,-0.0068,1,0,-0.0068,1,0,-0.0068,1,0,-0.0068,1] )
   , make_vbo(COLOR, [0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549,0.5882,0.5882,0.549] )
  ],
  mode: gl.TRIANGLES ,
  vertex_count: 156
};
}
function bezier(p0, p1, p2, p3, t) {
    var ct = 1 - t;
    var tmp = new GLMAT_ARRAY_TYPE(3);
    var position = new GLMAT_ARRAY_TYPE(3);
    var derivative = new GLMAT_ARRAY_TYPE(3);
    
    vec3.scale(position, p0, ct * ct * ct);
    
    vec3.scale(tmp, p1, 3 * ct * ct * t);
    vec3.add(position, position, tmp);
    
    vec3.scale(tmp, p2, 3 * ct * t * t);
    vec3.add(position, position, tmp);
    
    vec3.scale(tmp, p3, t * t * t);
    vec3.add(position, position, tmp);
    
    vec3.sub(tmp, p1, p0);
    vec3.scale(derivative, tmp, 3 * ct * ct);
    
    vec3.sub(tmp, p2, p1);
    vec3.scale(tmp, tmp, 6 * ct * t);
    vec3.add(derivative, derivative, tmp);
    
    vec3.sub(tmp, p3, p2);
    vec3.scale(tmp, tmp, 3 * t * t);
    vec3.add(derivative, derivative, tmp);
    
    vec3.normalize(derivative, derivative);
    
    return [position, derivative];
}

function calc_bezier(spline, index) {
    var spline_index = Math.floor(index) * 2;
    var p0 = spline[spline_index];
    var d0 = spline[spline_index + 1];
    var p3 = spline[spline_index + 2];
    var d3 = spline[spline_index + 3];
    
    var p1 = new GLMAT_ARRAY_TYPE(3);
    vec3.add(p1, p0, d0);
    
    var p2 = new GLMAT_ARRAY_TYPE(3);
    vec3.sub(p2, p3, d3);
    
    return bezier(p0, p1, p2, p3, index % 1);
}

function intro_title() {
    var spline = [
        [0, 0, 100], [0, 0, -10],
        [0, 0, 50], [10, 0,-30],
        [15, 15, 15], [-5, -5, 5],
        [15, -18, 20], [0, -5, -5],
        [5, 5, -5], [-5, -5, 5]
    ];

    var passes = [
        {
          clear: [.05, .05, .05, 1],
          depth_test: true,
          render_to: { color: "texture_1", depth: "texture_depth" },
          update: function(time) {
            var explode_param = Math.max(0, time.scene_norm * 1.2 - 0.2);
            
            uniforms["u_color1"] = [0.5, -0.1, 0.0, 1.0];
            uniforms["u_color2"] = [1, .7, .7, 1];
            uniforms["u_fade"] = explode_param;
            uniforms["u_param"] = Math.sin(explode_param * 4) * 5 + explode_param * 10;
            
            var calc = calc_bezier(spline, time.scene_norm * (spline.length / 2 - 1));
            uniforms["cam_pos"] = calc[0];
            uniforms["cam_target"] = [0,0,0];
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [1, [5]]
              ], time.scene_norm);
          },
          geometry: "sphere",
          program: "explode"
        },
        {
          depth_test: true,
          render_to: { color: "texture_1", depth: "texture_depth" },
          update: function(time) {
            var explode_param = Math.max(0, time.scene_norm * 1.2 - 0.2);
            
            uniforms["u_color1"] = [1,0.7,0.7,1];
            uniforms["u_color2"] = [3, 2, 2, 1.0];
            uniforms["u_fade"] = explode_param;
            uniforms["u_scale"] = 0.9;
          },
          geometry: "sphere",
          program: "scale"
        },
        {
          texture_inputs: ["texture_1"],
          render_to: { color: "texture_2" },
          update: function(time) {
            uniforms["u_threshold"] = 2.5;
          },
          geometry: "quad",
          program: "high_pass"
        },
        directional_blur("texture_2", "texture_3", [3,0]),
        //directional_blur("texture_3", "texture_2", [4,0]),
        //directional_blur("texture_2", "texture_3", [0,4]),
        directional_blur("texture_3", "texture_2", [0,3]),
        {
          texture_inputs: ["texture_1", "texture_2"],
          geometry: "quad",
          program: "add"
        }
    ];
    
    return {
        duration: 32,
        passes: passes
    };
}

function gates(spline) {
    var gates_count = 56;
    var gates_per_spline_segment = 20;
    
    var passes = [
        {
            clear: [1,1,1,1],
            update: function(time) {
                var pos_calc = calc_bezier(spline, time.scene * 0.1 + 0.5);
                var target_calc = calc_bezier(spline, time.scene * 0.1 + 0.6);
                
                pos_calc[0][1] += 100;
                target_calc[0][1] += 100;
                
                uniforms["cam_pos"] = pos_calc[0];
                uniforms["cam_target"] = target_calc[0];
                uniforms["cam_tilt"] = 0;
                uniforms["focus"] = 10;
            }
        }
    ];
    
    function update_gate(i) {
        var calc = calc_bezier(spline, i / gates_per_spline_segment + 1);
        var right = new GLMAT_ARRAY_TYPE(3);
        vec3.cross(right, calc[1], [0, 1, 0]);
        var world_mat = mat4.create();
        mat4.copy(world_mat, [
            calc[1][0], calc[1][1], calc[1][2], 0,
            0, 1, 0, 0,
            right[0], right[1], right[2], 0,
            calc[0][0], calc[0][1], calc[0][2], 1
        ]);
        mat4.rotateY(world_mat, world_mat, Math.PI / 2)
        
        return function() {
            uniforms["world_mat"] = world_mat;
        }
    }
    
    for (var i = 0; i < gates_count; ++i) {
        passes.push({
          depth_test: true,
          update: update_gate(i),
          geometry: "porte-tori",
          program: "model"
        
        });
    }
    
    passes.push({
        depth_test: false,
        blend: [gl.DST_COLOR, gl.ZERO],
        update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [10, 70, -50]],
                [128, [-20, 40, -30]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 50, 0]],
                [128, [0, 50, 0]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [128, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [2]],
                [128, [50]]
              ], time.scene);
        },
        geometry: "particles",
        program: "particles"
    })
    
    return {
        duration: 32,
        passes: passes
    };
}

function directional_blur(src, dst, dir) {
    return {
      texture_inputs: [src],
      render_to: {color: dst},
      geometry: "quad",
      program: "directional_blur",
      update: function(time) {
          uniforms["u_direction"] = dir;
      }
    };
}

function fit_height(height, texture) {
    return [height * textures[texture].width / textures[texture].height, height, 1];
}

function fit_width(width, texture) {
    return [width, width * textures[texture].height / textures[texture].width, 1];
}

function greets(duration, texture, camera) {
    return {
        duration: duration,
        passes: [
            {
                clear: [1,1,1,1],
                depth_test: true,
                update: function(time) {
                    uniforms["cam_pos"] = animate(camera(), time.scene_norm);
                    uniforms["cam_target"] = [0, 100, 0];
                    uniforms["cam_tilt"] = 0;
                    uniforms["focus"] = 10;
                    uniforms["world_mat"] = mat4.create();
                },
                geometry: "stone",
                program: "model"
            },
            {
                depth_test: true,
                blend: [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA],
                texture_inputs: [texture],
                geometry: "plane",
                program: "text",
                update: function(time) {
                    uniforms["u_light"] = [1,0,0];
                    
					var world_mat = mat4.create();
					mat4.translate(world_mat, world_mat, [0, 90, 10]);
					mat4.scale(world_mat, world_mat, fit_height(70, texture));
                    uniforms["world_mat"] = world_mat;
                }
            },
            {
                depth_test: false,
                blend: [gl.DST_COLOR, gl.ZERO],
                update: function(time) {
                    uniforms["cam_pos"] = animate([
                        [0, [10, 70, -50]],
                        [128, [-20, 40, -30]]
                      ], time.scene);
                    uniforms["cam_target"] = animate([
                        [0, [0, 50, 0]],
                        [128, [0, 50, 0]]
                      ], time.scene);
                    uniforms["cam_tilt"] = animate([
                        [0, [0]],
                        [128, [1]]
                      ], time.scene);
                    uniforms["focus"] = animate([
                        [0, [2]],
                        [128, [50]]
                      ], time.scene);
                },
                geometry: "particles",
                program: "particles"
            }
        ]
    }
}

function stone_aligned(x, z) {
    return {
      depth_test: true,
      update: function(time) {
          var world_mat = mat4.create();
          mat4.translate(world_mat, world_mat, [x, 0, z]);
          mat4.rotateY(world_mat, world_mat, x > 0 ? -Math.PI / 2 : Math.PI / 2);
          uniforms["world_mat"] = world_mat;
      },
      geometry: "stone",
      program: "model"
    }
}

function lantern2_aligned(x, z) {
    return {
      depth_test: true,
      update: function(time) {
          var world_mat = mat4.create();
          mat4.translate(world_mat, world_mat, [x, 400, z]);
          mat4.scale(world_mat, world_mat, [5, 5, 5]);
          uniforms["world_mat"] = world_mat;
      },
      geometry: "lantern2",
      program: "model"
    }
}

function demo_init() {
    uniforms["u_light"] = [0.707, 0.707, 0];
    uniforms["focus"] = 100;

    uniforms["cam_pos"] = [20, 20, 20]
    uniforms["cam_target"] = [0, 0, 0];
    uniforms["cam_tilt"] = 0

    var white = [1, 1, 1, 1];
    var black = [0, 0, 0, 1];
    var blue = [0, 0.3, 0.6, 1];

  scenes = [
    intro_title(),
    {
      duration: 32,
      passes: [
        {
          clear: [1,1,1,1],
          depth_test: false,
          blend: [gl.DST_COLOR, gl.ZERO],
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [10, 70, -50]],
                [128, [-20, 40, -30]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 50, 0]],
                [128, [0, 50, 0]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [128, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [1]],
                [32, [100]]
              ], time.scene);
          },
          geometry: "particles",
          program: "particles"
        }
      ]
    },
    gates([
        [-500, 0, 500], [500, 0, 0],
        [0, 0, 0], [0, 0, -800],
        [-1000, 100, -1000], [-600, 0, -400],
        [-1000, 0, -2000], [-200, -50, -600],
        [400, -150, -3000], [-200, 0, -400],
        [800, 0, -3600], [200, 0, -600]
    ]),
    {
      duration: 32,
      passes: [
        {
          clear: [1,1,1,1],
          depth_test: true,
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [0, 100, 600]],
                [16, [0, 100, 0]],
                [32, [0, 100, -600]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 300, 0]],
                [32, [0, 100, -2000]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [32, [-0.4]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [2]],
                [32, [50]]
              ], time.scene);
              
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [0, 400, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        stone_aligned(400, -500),
        stone_aligned(-400, -500),
        stone_aligned(400, -1000),
        stone_aligned(-400, -1000),
        stone_aligned(400, -1500),
        stone_aligned(-400, -1500),
        stone_aligned(400, -2000),
        stone_aligned(-400, -2000),
        stone_aligned(400, -2500),
        stone_aligned(-400, -2500),
        stone_aligned(400, -3000),
        stone_aligned(-400, -3000),
        stone_aligned(400, -3500),
        stone_aligned(-400, -3500),
        stone_aligned(400, -4000),
        stone_aligned(-400, -4000),
        {
          depth_test: false,
          blend: [gl.DST_COLOR, gl.ZERO],
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [10, 70, -50]],
                [128, [-20, 40, -30]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 50, 0]],
                [128, [0, 50, 0]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [128, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [2]],
                [128, [50]]
              ], time.scene);
          },
          geometry: "particles",
          program: "particles"
        }
      ]
    },
    greets(8, "greets_ctrlalttest", function() { return [
        [0, [-100, 100, 400]],
        [1, [100, 150, 100]]
      ] }),
    greets(8, "greets_cocoon", function() { return [
        [0, [100, 80, 130]],
        [1, [-100, 110, 90]]
      ] }),
    greets(8, "greets_vitalmotion", function() { return [
        [0, [-80, 150, 120]],
        [1, [120, 130, 100]]
      ] }),
    greets(8, "greets_poobrain", function() { return [
        [0, [60, 90, 80]],
        [1, [-110, 120, 260]]
      ] }),
    gates([
        [0, 500, 500], [0, 0, 400],
        [0, 0, 0], [0, 0, -400],
        [100, -100, -1000], [200, 0, -400],
        [1000, 0, -1400], [600, 100, 0],
        [800, 200, -2600], [400, 0, -400],
        [800, 0, -3600], [200, 0, 600]
    ]),
    greets(8, "greets_fairlight", function() { return [
        [0, [-100, 150, 240]],
        [1, [100, 120, 90]]
      ] }),
    greets(8, "greets_systemk", function() { return [
        [0, [100, 80, 130]],
        [1, [-100, 110, 90]]
      ] }),
    greets(8, "greets_farbrausch", function() { return [
        [0, [-80, 140, 120]],
        [1, [120, 100, 100]]
      ] }),
    greets(8, "greets_trsi", function() { return [
        [0, [100, 130, 130]],
        [1, [-100, 80, 340]]
      ] }),
    {
      duration: 64,
      passes: [
        {
          clear: [1,1,1,1],
          depth_test: true,
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [0, 300, 600]],
                [16, [0, 100, 0]],
                [32, [0, 300, -600]],
                [64, [0, 400, -1200]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 400, 0]],
                [32, [0, 400, -2000]],
                [64, [0, 100, -4000]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [32, [0]],
                [64, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [2]],
                [32, [50]]
              ], time.scene);
              
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [0, 400, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: true,
          update: function(time) {
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [400, 400, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: true,
          update: function(time) {
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [-400, 400, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        lantern2_aligned(400, -500),
        lantern2_aligned(-400, -500),
        lantern2_aligned(400, -1000),
        lantern2_aligned(-400, -1000),
        lantern2_aligned(400, -1500),
        lantern2_aligned(-400, -1500),
        lantern2_aligned(400, -2000),
        lantern2_aligned(-400, -2000),
        lantern2_aligned(400, -2500),
        lantern2_aligned(-400, -2500),
        lantern2_aligned(400, -3000),
        lantern2_aligned(-400, -3000),
        lantern2_aligned(400, -3500),
        lantern2_aligned(-400, -3500),
        lantern2_aligned(400, -4000),
        lantern2_aligned(-400, -4000),
        {
          depth_test: false,
          blend: [gl.DST_COLOR, gl.ZERO],
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [10, 70, -50]],
                [128, [-20, 40, -30]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 50, 0]],
                [128, [0, 50, 0]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [128, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [1]],
                [32, [50]]
              ], time.scene);
          },
          geometry: "particles",
          program: "particles"
        }
      ]
    },
    {
      duration: 64,
      passes: [
        {
          clear: [1,1,1,1],
          depth_test: true,
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [-200, 0, 600]],
                [1, [2300, 0, 600]]
              ], time.scene_norm);
            uniforms["cam_target"] = animate([
                [0, [0, 0, 0]],
                [1, [2100, 0, 0]]
              ], time.scene_norm);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [1, [-Math.PI * 2]]
              ], time.scene_norm);;
            uniforms["focus"] = 50;
              
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [0, 0, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              mat4.rotateZ(world_mat, world_mat, time.scene * Math.PI / 8);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: true,
          update: function(time) {
              var world_mat = mat4.create();
              mat4.translate(world_mat, world_mat, [700, 0, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              mat4.rotateZ(world_mat, world_mat, (time.scene + 1) * Math.PI / 8);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: true,
          update: function(time) {
              var world_mat = mat4.create(); 
              mat4.translate(world_mat, world_mat, [1400, 0, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              mat4.rotateZ(world_mat, world_mat, (time.scene + 2) * Math.PI / 8);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: true,
          update: function(time) {
              var world_mat = mat4.create(); 
              mat4.translate(world_mat, world_mat, [2100, 0, 0]);
              mat4.scale(world_mat, world_mat, [5, 5, 5]);
              mat4.rotateZ(world_mat, world_mat, (time.scene + 3) * Math.PI / 8);
              uniforms["world_mat"] = world_mat;
          },
          geometry: "lantern1",
          program: "model"
        },
        {
          depth_test: false,
          blend: [gl.DST_COLOR, gl.ZERO],
          update: function(time) {
            uniforms["cam_pos"] = animate([
                [0, [10, 70, -50]],
                [128, [-20, 40, -30]]
              ], time.scene);
            uniforms["cam_target"] = animate([
                [0, [0, 50, 0]],
                [128, [0, 50, 0]]
              ], time.scene);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [128, [1]]
              ], time.scene);
            uniforms["focus"] = animate([
                [0, [2]],
                [128, [50]]
              ], time.scene);
          },
          geometry: "particles",
          program: "particles"
        }
      ]
    },
    {
        duration: 32,
        passes: [
        {
          clear: [.05, .05, .05, 1],
          depth_test: true,
          render_to: { color: "texture_1", depth: "texture_depth" },
          update: function(time) {
            var explode_param = Math.max(0, 1 - time.scene_norm * 1.5);
            
            uniforms["u_color1"] = [0.5, -0.1, 0.0, 1.0];
            uniforms["u_color2"] = [1, .7, .7, 1];
            uniforms["u_fade"] = explode_param;
            uniforms["u_param"] = 20 * explode_param;
            
            uniforms["cam_pos"] = 
            uniforms["cam_pos"] = animate([
                [0, [0, 0, 5]],
                [0.8, [0, -25, 100]]
              ], time.scene_norm);
            uniforms["cam_target"] = animate([
                [0, [0, 0, 0]],
                [0.7, [0, 0, 0]],
                [0.8, [0, -25, 0]]
              ], time.scene_norm);
            uniforms["cam_tilt"] = animate([
                [0, [0]],
                [0.7, [Math.PI * 2]]
              ], time.scene_norm);
          },
          geometry: "sphere",
          program: "explode"
        },
        {
          depth_test: true,
          render_to: { color: "texture_1", depth: "texture_depth" },
          update: function(time) {
            var explode_param = Math.max(0, 1 - time.scene_norm * 1.5);
            
            uniforms["u_color1"] = [1,0.7,0.7,1];
            uniforms["u_color2"] = [3, 2, 2, 1.0];
            uniforms["u_fade"] = explode_param;
            uniforms["u_scale"] = 0.9;
          },
          geometry: "sphere",
          program: "scale"
        },
        {
          texture_inputs: ["texture_1"],
          render_to: { color: "texture_2" },
          update: function(time) {
            uniforms["u_threshold"] = 2.5;
          },
          geometry: "quad",
          program: "high_pass"
        },
        directional_blur("texture_2", "texture_3", [3,0]),
        //directional_blur("texture_3", "texture_2", [4,0]),
        //directional_blur("texture_2", "texture_3", [0,4]),
        directional_blur("texture_3", "texture_2", [0,3]),
        {
          texture_inputs: ["texture_1", "texture_2"],
          geometry: "quad",
          program: "add",
          update: function() {
              uniforms["u_threshold"] = 1.5;
          }
        },
            {
                depth_test: true,
                blend: [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA],
                texture_inputs: ["text_travel_to"],
                geometry: "plane",
                program: "text",
                update: function(time) {
                    /*
                    uniforms["cam_pos"] = [0, 0, 100];
                    uniforms["cam_target"] = [0, 0, 0];
                    uniforms["cam_tilt"] = 0;
                    uniforms["focus"] = 10;
                    */
                    uniforms["u_light"] = [1,0,0];
                    
					var world_mat = mat4.create();
					mat4.translate(world_mat, world_mat, [0, 30, 0]);
					mat4.scale(world_mat, world_mat, fit_width(30, "text_travel_to"));
                    uniforms["world_mat"] = world_mat;
                }
            },
            {
                depth_test: true,
                blend: [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA],
                texture_inputs: ["text_japan"],
                geometry: "plane",
                program: "text",
                update: function(time) {
					var world_mat = mat4.create();
					mat4.translate(world_mat, world_mat, [0, -35, 0]);
					mat4.scale(world_mat, world_mat, fit_width(50, "text_japan"));
                    uniforms["world_mat"] = world_mat;
                }
            },
            {
                depth_test: true,
                blend: [gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA],
                texture_inputs: ["text_credits"],
                geometry: "plane",
                program: "text",
                update: function(time) {
					var world_mat = mat4.create();
					mat4.translate(world_mat, world_mat, animate([
                        [0.7, [0, -140, 0]],
                        [0.8, [0, -80, 0]]
                    ], time.scene_norm));
					mat4.scale(world_mat, world_mat, fit_width(40, "text_credits"));
                    uniforms["world_mat"] = world_mat;
                }
            }
          ]
        }
  ];
}
onload=main;
