#include <QDebug>

#include "Player.h"
#include "Tile.h"
#include "TileStack.h"
#include "PathFinder.h"
#include "StepCounter.h"
#include "TileHandler.h"
#include "TileCollection.h"

Player::Player(int id, const QColor &color, PathFinder *pathFinder, TileHandler *tileHandler)
	: m_connected(false)
	, m_dead(false)
	, m_id(id)
	, m_reserved(-1)
	, m_tile(nullptr)
	, m_pathFinder(pathFinder)
	, m_tileHandler(tileHandler)
	, m_response(nullptr)
	, m_color(color)
{

}

int Player::id() const
{
	return m_id;
}

QColor Player::color() const
{
	return m_color;
}

QList<int> &Player::hand()
{
	return m_hand;
}

Tile *Player::tile() const
{
	return m_tile;
}

void Player::setTile(Tile *tile)
{
	m_tile = tile;

	emit tileChanged();
}

QString Player::name() const
{
	return m_name;
}

void Player::setName(const QString &name)
{
	m_name = name;

	emit nameChanged();
}

bool Player::connected() const
{
	return m_connected;
}

void Player::setConnected(bool connected)
{
	m_connected = connected;

	emit connectedChanged();
}

bool Player::dead() const
{
	return m_dead;
}

void Player::setDead(bool dead)
{
	m_dead = dead;

	if (dead)
	{
		m_tile = nullptr;
	}

	emit deadChanged();
}

QHttpResponse *Player::response()
{
	return m_response;
}

void Player::setResponse(QHttpResponse *response)
{
	m_response = response;
}

void Player::populateHand(TileStack &tileStack)
{
	if (m_reserved > -1)
	{
		m_hand.removeAt(m_reserved);
		m_reserved = -1;
	}

	const int count = HAND_COUNT - m_hand.count();

	for (int i = 0; i < count; i++)
	{
		m_hand << tileStack.take();
	}
}

void Player::move(Tile *target)
{
	StepCounter stepCounter;
	TileCollection tileCollection;

	if (m_pathFinder->walkThePath(target, this, stepCounter, tileCollection))
	{
		for (Tile *tile : tileCollection)
		{
			process(tile);
		}
	}
	else
	{
		process(target);
		setDead(true);
	}
}

Reservation Player::reserve(int index)
{
	if (index == m_reserved)
	{
		const int value = m_hand[index];
		const int rotated = Entrance::rotate(value);

		m_hand[index] = rotated;

		return Reservation(rotated, ActionVisualizer::Rotate);
	}
	else
	{
		m_reserved = index;

		const int value = m_hand[index];

		return Reservation(value, ActionVisualizer::Select);
	}
}

Player *Player::none()
{
	static Player *player = new Player(0, "#374548", nullptr, nullptr);

	return player;
}

void Player::kill()
{
	setDead(true);
}

Player::operator int() const
{
	return m_id;
}

void Player::process(Tile *target)
{
	const QList<Tile *> &tiles = m_tileHandler->neighbours(target);

	for (Tile *tile : tiles)
	{
		Player *owner = tile->owner();

		if (owner &&
			owner != this &&
			owner->tile() == tile)
		{
			owner->setDead(true);
		}

		tile->setOwner(this);
	}

	setTile(target);
}
